/**************************************************************************************
 * @file        : SXMClockService.h
 * @addtogroup  : AppHmi_Sxm
 * @brief       : SXM Clock utility class required to interact with MIDW Clock Fi.
 * @copyright   : (c) 2019-2019 Robert Bosch Car Multimedia GmbH
 *                The reproduction, distribution and utilization of this file as
 *                well as the communication of its contents to others without express
 *                authorization is prohibited. Offenders will be held liable for the
 *                payment of damages. All rights reserved in the event of the grant
 *                of a patent, utility model or design.
 **************************************************************************************/
#ifndef _SXMCLOCKSERVICE_H_
#define _SXMCLOCKSERVICE_H_

#include "AppBase/ServiceAvailableIF.h"
#include "clock_main_fiProxy.h"
#include "App/Core/SxmUtils/SxmUtilsSingleton.h"
#include "App/Core/SxmUtils/SxmCommonTypes.h"
#include "ISxmClockService.h"
#include "AppHmi_SxmTypes.h"

namespace App {
namespace Core {
/**
 *  SXM Clock utility class required to interact with MIDW Clock Fi.
 * @class : SXMClockService
 */
class SXMClockService
   : public SxmUtilsSingleton<SXMClockService>
   , public hmibase::ServiceAvailableIF
   , public StartupSync::PropertyRegistrationIF
   , public clock_main_fi::NewDateFormatCallbackIF
   , public clock_main_fi::TimeFormatCallbackIF
   , public clock_main_fi::LocalTimeDate_MinuteUpdateCallbackIF
   , public clock_main_fi::TimeZoneCallbackIF
   , public clock_main_fi::DaylightSavingTimeCallbackIF
   , public clock_main_fi::LocalTimeOffsetCallbackIF
   , public clock_main_fi::GPS_AutoSyncCallbackIF
{
   public:
      /**
       * SXMClockService Class constructor
       */
      SXMClockService();
      /**
       * SXMClockService Class destructor
       */
      virtual ~SXMClockService();
      /**
       * Utility function that registers for notifications updates from clockservice.
       */
      void registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& /*proxy*/,
                              const asf::core::ServiceStateChange& /*stateChange*/);
      /**
       * Utility function that deregisters for notifications updates from clockservice.
       */
      void deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& /*proxy*/,
                                const asf::core::ServiceStateChange& /*stateChange*/);
      /**
      * Utility function that registers for notifications updates from clockservice.
      */
      void onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy,
                       const asf::core::ServiceStateChange& /*stateChange*/);
      /**
      * Utility function that deregisters for notifications updates from clockservice.
      */
      void onUnavailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy,
                         const asf::core::ServiceStateChange& /*stateChange*/);
      /**
       * Error handling function for Time Street property.
       */
      void onTimeFormatError(const ::boost::shared_ptr< clock_main_fi::Clock_main_fiProxy >& proxy,
                             const ::boost::shared_ptr< clock_main_fi::TimeFormatError >& error);
      /**
       * Update handling function for Time Format property.
       */
      void onTimeFormatStatus(const ::boost::shared_ptr< clock_main_fi::Clock_main_fiProxy >& proxy,
                              const ::boost::shared_ptr< clock_main_fi::TimeFormatStatus >& status);
      /**
       * Error handling function for Date Street property.
       */
      void onNewDateFormatError(const ::boost::shared_ptr< clock_main_fi::Clock_main_fiProxy >& proxy,
                                const ::boost::shared_ptr< clock_main_fi::NewDateFormatError >& error);
      /**
        * Update handling function for DateFormat property.
        */
      void onNewDateFormatStatus(const ::boost::shared_ptr< clock_main_fi::Clock_main_fiProxy >& proxy,
                                 const ::boost::shared_ptr< clock_main_fi::NewDateFormatStatus >& status);
      /**
       * Error handling function for TimeZone property.
       */
      void onTimeZoneError(const ::boost::shared_ptr< clock_main_fi::Clock_main_fiProxy >& proxy,
                           const ::boost::shared_ptr< clock_main_fi::TimeZoneError >& error);
      /**
       * Update handling function for TimeZone property.
       */
      void onTimeZoneStatus(const ::boost::shared_ptr< clock_main_fi::Clock_main_fiProxy >& proxy,
                            const ::boost::shared_ptr< clock_main_fi::TimeZoneStatus >& status);
      /**
       * Error handling function for DaylightSavingTime property.
       */
      void onDaylightSavingTimeError(const ::boost::shared_ptr< clock_main_fi::Clock_main_fiProxy >& proxy,
                                     const ::boost::shared_ptr< clock_main_fi::DaylightSavingTimeError >& error);
      /**
       * Update handling function for DaylightSavingTime property.
       */
      void onDaylightSavingTimeStatus(const ::boost::shared_ptr< clock_main_fi::Clock_main_fiProxy >& proxy,
                                      const ::boost::shared_ptr< clock_main_fi::DaylightSavingTimeStatus >& status);
      /**
      * Error handling function for LocalTimeDate property.
      */
      void onLocalTimeDate_MinuteUpdateError(const ::boost::shared_ptr< clock_main_fi::Clock_main_fiProxy >& proxy,
                                             const ::boost::shared_ptr< clock_main_fi::LocalTimeDate_MinuteUpdateError >& error);
      /**
      * Update handling function for LocalTimeDate property.
      */
      void onLocalTimeDate_MinuteUpdateStatus(const ::boost::shared_ptr< clock_main_fi::Clock_main_fiProxy >& proxy,
                                              const ::boost::shared_ptr< clock_main_fi::LocalTimeDate_MinuteUpdateStatus >& status);
      /**
      * Error handling function for LocalTimeOffset property.
      */
      void onLocalTimeOffsetError(const ::boost::shared_ptr< clock_main_fi::Clock_main_fiProxy >& proxy,
                                  const ::boost::shared_ptr< clock_main_fi::LocalTimeOffsetError >& error);
      /**
      * Update handling function for LocalTimeOffset property.
      */
      void onLocalTimeOffsetStatus(const ::boost::shared_ptr< clock_main_fi::Clock_main_fiProxy >& proxy,
                                   const ::boost::shared_ptr< clock_main_fi::LocalTimeOffsetStatus >& status);
      /**
      * Error handling function for GPS_AutoSync property.
      */
      void onGPS_AutoSyncError(const ::boost::shared_ptr< clock_main_fi::Clock_main_fiProxy >& proxy,
                               const ::boost::shared_ptr< clock_main_fi::GPS_AutoSyncError >& error);
      /**
      * Update handling function for GPS_AutoSync property.
      */
      void onGPS_AutoSyncStatus(const ::boost::shared_ptr< clock_main_fi::Clock_main_fiProxy >& proxy,
                                const ::boost::shared_ptr< clock_main_fi::GPS_AutoSyncStatus >& status);
      /**
       * Helper function to get Time format from  MIDW Clock fi.
       */
      ::clock_main_fi_types::VDCLK_TEN_TimeFormat GetCurrentTimeFormat()const;
      /**
       * Helper function to get date format from  MIDW Clock fi.
       */
      ::clock_main_fi_types::VDCLK_TEN_NewDateFormat GetCurrentDateFormat()const;
      /**
       * Helper function to get LocalTimedate  from  MIDW Clock fi.
       */
      LocalDateTimeType GetLocalTimeDate()const;
      /**
       * Helper function to get current Time Zone.
       */
      std::string GetTimeZone()const;

      void registerTimeStatusNotification(ISxmClockService* _obsptr, enSXMServiceType enServiceType);
      /**
       * De-Register call back function to stop receiving updates.
       */
      void deregisterTimeStatusNotification(ISxmClockService* _obsptr, enSXMServiceType enServiceType);

   private:
      friend class  SxmUtilsSingleton<SXMClockService>;
      ::boost::shared_ptr< ::clock_main_fi::Clock_main_fiProxy > _mPtrClockproxy;
      std::map <enSXMServiceType, ISxmClockService*> _stateObsList;
      clock_main_fi_types::VDCLK_TEN_TimeFormat _entimeFormat;
      clock_main_fi_types::VDCLK_TEN_NewDateFormat _endateFormat;
      bool _autoClockMode;
      std::string _offset;
      bool _dstFlag;
      bool _dstAutoModeFlag;

      /**
       * Courier Message Handler
       */
      COURIER_MSG_MAP_BEGIN(TR_CLASS_APPHMI_SXM_COURIER_PAYLOAD_MODEL_COMP)
      COURIER_CASE_DUMMY_ENTRY()
      COURIER_MSG_MAP_DELEGATE_START()
      COURIER_MSG_MAP_DELEGATE_DEF_END()
      COURIER_MSG_MAP_DELEGATE_END()

   protected:
};


}  // namespace Core
}  // namespace App

#endif
