/**************************************************************************************
 * @file        : SXMClockService.cpp
 * @addtogroup  : AppHmi_Sxm
 * @brief       : SXM Clock utility class required to interact with MIDW Clock Fi.
 * @copyright   : (c) 2019-2019 Robert Bosch Car Multimedia GmbH
 *                The reproduction, distribution and utilization of this file as
 *                well as the communication of its contents to others without express
 *                authorization is prohibited. Offenders will be held liable for the
 *                payment of damages. All rights reserved in the event of the grant
 *                of a patent, utility model or design.
 **************************************************************************************/
#include "hmi_trace_if.h"
#include "CgiExtensions/CourierMessageMapper.h"
//#include "hall_std_if.h"  //when other headers also needed, include hall_std_if and remove above two
#include "SXMClockService.h"
#include "AppHmi_SxmMessages.h"
#include "App/Core/SxmUtils/SxmUtils.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS       TR_CLASS_APPHMI_SXM_HALL
#define ETG_I_TRACE_CHANNEL           TR_TTFIS_APPHMI_SXM
#define ETG_I_TTFIS_CMD_PREFIX        "APPHMI_SXM_"
#define ETG_I_FILE_PREFIX             App::Core::SXMClockService::
#include "trcGenProj/Header/SXMClockService.cpp.trc.h"
#endif
namespace App {
namespace Core {
using namespace clock_main_fi;

/**
 * SXMClockService Class constructor
 */
SXMClockService::SXMClockService()
   : _mPtrClockproxy(Clock_main_fiProxy::createProxy("SxmClockFiPort", *this))
{
   ETG_I_REGISTER_FILE();
   ETG_TRACE_USR4(("SXMClockService() Constructor"));
   SXM_UTILS_REGISTER_PROPERTY_CALLBACKS(StartupSync::getInstance(), this, _mPtrClockproxy);
   _stateObsList.clear();
   _entimeFormat = clock_main_fi_types::VDCLK_TEN_TimeFormat__VDCLK_EN_TF_Mode12;
   _endateFormat = clock_main_fi_types::VDCLK_TEN_NewDateFormat__VDCLK_EN_DF_dd_mm_yyyy_Dash;
   _autoClockMode = false;
   _offset = "";
   _dstFlag = false;
   _dstAutoModeFlag = false;
}


/**
 * SXMClockService Class destructor
 */
SXMClockService::~SXMClockService()
{
   ETG_TRACE_USR4(("~SXMClockService() Destructor"));
   _mPtrClockproxy.reset();
   ETG_I_UNREGISTER_FILE();
}


/**
 * Utility function that registers for notifications updates from clock service.
 * @param [in] : proxy - clock service class instance
 * @param [in] : stateChange - Current ASF state
 */
void SXMClockService::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& /*proxy*/,
      const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("SXMClockService registerProperties"));
}


/**
 * Utility function that deregisters for notifications updates from clock service.
 * @param [in] : proxy - clock service class instance
 * @param [in] : stateChange - Current ASF state
 */
void SXMClockService::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& /*proxy*/,
      const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("SXMClockService deregisterProperties"));
}


/**
 * Utility function that registers for notifications updates from clock service.
 * @param [in] : proxy - clock service class instance
 * @param [in] : stateChange - Current ASF state
 */
void SXMClockService::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy,
                                  const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("SXMClockService onAvailable"));
   if ((proxy == _mPtrClockproxy) && (_mPtrClockproxy))
   {
      _mPtrClockproxy->sendTimeFormatUpReg(*this);
      _mPtrClockproxy->sendNewDateFormatUpReg(*this);
      _mPtrClockproxy->sendLocalTimeDate_MinuteUpdateUpReg(*this);
      _mPtrClockproxy->sendTimeZoneUpReg(*this);
      _mPtrClockproxy->sendDaylightSavingTimeUpReg(*this);
      _mPtrClockproxy->sendLocalTimeOffsetUpReg(*this);
      _mPtrClockproxy->sendGPS_AutoSyncUpReg(*this);
   }
}


/**
 * Utility function that deregisters for notifications updates from clock service.
 * @param [in] : proxy - clock service class instance
 * @param [in] : stateChange - Current ASF state
 */
void SXMClockService::onUnavailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy,
                                    const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("SXMClockService onUnavailable"));
   if ((proxy == _mPtrClockproxy) && (_mPtrClockproxy))
   {
      _mPtrClockproxy->sendTimeFormatRelUpRegAll();
      _mPtrClockproxy->sendNewDateFormatRelUpRegAll();
      _mPtrClockproxy->sendLocalTimeDate_MinuteUpdateRelUpRegAll();
      _mPtrClockproxy->sendTimeZoneRelUpRegAll();
      _mPtrClockproxy->sendDaylightSavingTimeRelUpRegAll();
      _mPtrClockproxy->sendLocalTimeOffsetRelUpRegAll();
      _mPtrClockproxy->sendGPS_AutoSyncRelUpRegAll();
   }
}


/**
 * Error handling function for Current Street property.
 * @param [in] : proxy - MIDW Clock Fi class instance
 * @param [in] : error - Error message of "CurrentStreet".
 */
void SXMClockService::onTimeFormatError(const ::boost::shared_ptr< Clock_main_fiProxy >& /*proxy*/,
                                        const ::boost::shared_ptr< TimeFormatError >& /*error*/)
{
   ETG_TRACE_USR4(("SXMClockService onTimeFormatError"));
}


/**
 * Update handling function for Time Format property.
 * @param [in] : proxy - MIDW Clock Fi class instance
 * @param [in] : status - Update message of "TimeFormat".
 */
void SXMClockService::onTimeFormatStatus(const ::boost::shared_ptr< Clock_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< TimeFormatStatus >& status)
{
   _entimeFormat = status->getEnTimeFormat();
   ETG_TRACE_USR4(("SXMClockService onTimeFormatStatus %d", _entimeFormat));
   std::map <enSXMServiceType, ISxmClockService*>::const_iterator it;
   for (it = _stateObsList.begin(); it != _stateObsList.end(); ++it)
   {
      ETG_TRACE_USR4(("SXMClockService onTimeFormatStatus, Service:%d", (it->first)));
      it->second->vNotifyTimeStatus(_entimeFormat);
   }
}


/**
 * Error handling function for Current Street property.
 * @param [in] : proxy - MIDW Clock Fi class instance
 * @param [in] : error - Error message of "CurrentStreet".
 */
void SXMClockService::onNewDateFormatError(const ::boost::shared_ptr< Clock_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< NewDateFormatError >& /*error*/)
{
   ETG_TRACE_USR4(("SXMClockService onNewDateFormatError"));
}


/**
 * Update handling function for DateFormat property.
 * @param [in] : proxy - MIDW Clock Fi class instance
 * @param [in] : status - Update message of "DateFormat".
 */
void SXMClockService::onNewDateFormatStatus(const ::boost::shared_ptr< Clock_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< NewDateFormatStatus >& status)
{
   _endateFormat = status->getEnDateFormat();
   ETG_TRACE_USR4(("SXMClockService NewDateFormatStatus %d", _endateFormat));
   std::map <enSXMServiceType, ISxmClockService*>::const_iterator it;
   for (it = _stateObsList.begin(); it != _stateObsList.end(); ++it)
   {
      ETG_TRACE_USR4(("SXMClockService NewDateFormatStatus, Service:%d", (it->first)));
      it->second->vNotifyDateStatus(_endateFormat);
   }
}


/**
 * Error handling function for TimeZone property.
 * @param [in] : proxy - MIDW Clock Fi class instance
 * @param [in] : error - Error message of "TimeZone".
 */
void SXMClockService::onTimeZoneError(const ::boost::shared_ptr< Clock_main_fiProxy >& /*proxy*/,
                                      const ::boost::shared_ptr< TimeZoneError >& /*error*/)
{
   ETG_TRACE_USR4(("SXMClockService onTimeZoneError"));
}


/**
 * Update handling function for TimeZone property.
 * @param [in] : proxy - MIDW Clock Fi class instance
 * @param [in] : status - Update message of "TimeZone".
 */
void SXMClockService::onTimeZoneStatus(const ::boost::shared_ptr< Clock_main_fiProxy >& /*proxy*/,
                                       const ::boost::shared_ptr< TimeZoneStatus >& status)
{
   ETG_TRACE_USR4(("SXMClockService onTimeZoneStatus %s", status->getSzDescription().c_str()));
}


/**
 * Error handling function for DaylightSavingTime property.
 * @param [in] : proxy - MIDW Clock Fi class instance
 * @param [in] : error - Error message of "DaylightSavingTime".
 */
void SXMClockService::onDaylightSavingTimeError(const ::boost::shared_ptr< Clock_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< DaylightSavingTimeError >& /*error*/)
{
   ETG_TRACE_USR4(("SXMClockService onDaylightSavingTimeError"));
}


/**
 * Update handling function for DaylightSavingTime property.
 * @param [in] : proxy - MIDW Clock Fi class instance
 * @param [in] : status - Update message of "DaylightSavingTime".
 */
void SXMClockService::onDaylightSavingTimeStatus(const ::boost::shared_ptr< Clock_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< DaylightSavingTimeStatus >& status)
{
   _dstFlag = status->getBDST_Status();
   if ((status->hasBDST_Status()) && !_autoClockMode)
   {
      POST_MSG((COURIER_MESSAGE_NEW(SXMAudioPostTimeZoneChangeReqMsg)(_offset.c_str(), _dstFlag)));
   }
}


/**
 * Error handling function for LocalTimeDate_MinuteUpdate property.
 * @param [in] : proxy - MIDW Clock Fi class instance
 * @param [in] : error - Error message of "LocalTimeDate_MinuteUpdate".
 */
void SXMClockService::onLocalTimeDate_MinuteUpdateError(const ::boost::shared_ptr< Clock_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< LocalTimeDate_MinuteUpdateError >& /*error*/)
{
   ETG_TRACE_USR4(("SXMClockService onLocalTimeDate_MinuteUpdateError"));
}


/**
 * Update handling function for LocalTimeDate_MinuteUpdate property.
 * @param [in] : proxy - MIDW Clock Fi class instance
 * @param [in] : status - Update message of "LocalTimeDate_MinuteUpdate".
 */
void SXMClockService::onLocalTimeDate_MinuteUpdateStatus(const ::boost::shared_ptr< Clock_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< LocalTimeDate_MinuteUpdateStatus >& status)
{
   ETG_TRACE_USR4(("SXMClockService onLocalTimeDate_MinuteUpdateStatus hour:%u, Min:%u, Weekday:%u", status->getU8Hours(),
                   status->getU8Minutes(), status->getU8Weekday()));
   if (_autoClockMode)
   {
      POST_MSG((COURIER_MESSAGE_NEW(SXMAudioPostLocalTimeChangeReqMsg)(_offset.c_str(), _dstAutoModeFlag)));
   }
   else
   {
      POST_MSG((COURIER_MESSAGE_NEW(SXMAudioPostLocalTimeChangeReqMsg)(_offset.c_str(), _dstFlag)));
   }
}


/**
 * Error handling function for LocalTimeOffset property.
 * @param [in] : proxy - MIDW Clock Fi class instance
 * @param [in] : error - Error message of "LocalTimeOffset".
 */
void SXMClockService::onLocalTimeOffsetError(const ::boost::shared_ptr< Clock_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< LocalTimeOffsetError >& /*error*/)
{
   ETG_TRACE_USR4(("SXMClockService onLocalTimeOffsetError"));
}


/**
 * Update handling function for LocalTimeOffset property.
 * @param [in] : proxy - MIDW Clock Fi class instance
 * @param [in] : status - Update message of "LocalTimeOffset".
 */
void SXMClockService::onLocalTimeOffsetStatus(const ::boost::shared_ptr< Clock_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< LocalTimeOffsetStatus >& status)
{
   ETG_TRACE_USR4(("SXMClockService onLocalTimeOffsetValue: %d", status->getS64Offset_ScalarValue()));
   int64 temp = status->getS64Offset_ScalarValue();
   if (temp < 0)
   {
      temp = temp * (-1); // To convert negative to positive value
      _offset = SxmUtils::conCatData("si", 2, "-", temp);
   }
   else
   {
      _offset =  SxmUtils::conCatData("i", 1, temp);
   }
   ETG_TRACE_USR4(("SXMClockService onLocalTimeOffsetStatus offset  : %s", _offset.c_str()));
   if (_autoClockMode)
   {
      POST_MSG((COURIER_MESSAGE_NEW(SXMAudioPostTimeZoneChangeReqMsg)(_offset.c_str(), _dstAutoModeFlag)));
   }
   else
   {
      POST_MSG((COURIER_MESSAGE_NEW(SXMAudioPostTimeZoneChangeReqMsg)(_offset.c_str(), _dstFlag)));
   }
}


/**
 * Error handling function for GPS_AutoSync property.
 * @param [in] : proxy - MIDW Clock Fi class instance
 * @param [in] : status - Update message of "GPS_AutoSync".
 */
void SXMClockService::onGPS_AutoSyncError(const ::boost::shared_ptr< Clock_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< GPS_AutoSyncError >& /*error*/)
{
   ETG_TRACE_USR4(("SXMClockService onLocalTimeOffsetError"));
}


/**
 * Update handling function for GPS_AutoSync property.
 * @param [in] : proxy - MIDW Clock Fi class instance
 * @param [in] : status - Update message of "GPS_AutoSync".
 */
void SXMClockService::onGPS_AutoSyncStatus(const ::boost::shared_ptr< Clock_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< GPS_AutoSyncStatus >& status)
{
   ETG_TRACE_USR4(("SXMClockServiceInterface onGPS_AutoSyncStatus getBEnabled : %d", status->getBEnabled()));
   ETG_TRACE_USR4(("SXMClockServiceInterface onGPS_AutoSyncStatus getBPositionBased : %d", status->getBPositionBased()));
   /*enabled =1 and position = 0 then timezone
   enabled  = 0 and position = 0 then manual
   enabled =1 and postion = 1 then auto*/
   _autoClockMode = (status->getBEnabled()) && (status->getBPositionBased());
   ETG_TRACE_USR4(("SXMClockServiceInterface onGPS_AutoSyncStatus _autoClockMode : %d", _autoClockMode));
   if (_autoClockMode)
   {
      POST_MSG((COURIER_MESSAGE_NEW(SXMAudioPostTimeZoneChangeReqMsg)(_offset.c_str(), _dstAutoModeFlag)));
   }
}


/**
 * Helper function to get Time format from  MIDW Clock fi.
 * @return : returns time format
 */
clock_main_fi_types::VDCLK_TEN_TimeFormat SXMClockService::GetCurrentTimeFormat()const
{
   ETG_TRACE_USR4(("SXMClockServiceInterface GetCurrentTimeFormat %d", _entimeFormat));
   return _entimeFormat;
}


/**
 * Helper function to get Date format from  MIDW Clock fi.
 * @return : returns Date format
 */
clock_main_fi_types::VDCLK_TEN_NewDateFormat SXMClockService::GetCurrentDateFormat()const
{
   ETG_TRACE_USR4(("SXMClockServiceInterface GetCurrentDateFormat %d", _endateFormat));
   return _endateFormat;
}


/**
 * Helper function to get LocalTimedate  from  MIDW Clock fi.
 * @return : returns local date time
 */
LocalDateTimeType SXMClockService::GetLocalTimeDate()const
{
   ETG_TRACE_USR4(("SXMClockServiceInterface GetLocalTimeDate"));
   LocalDateTimeType llocaldatetime;
   if (_mPtrClockproxy && _mPtrClockproxy->hasLocalTimeDate_MinuteUpdate())
   {
      llocaldatetime.s16Year = _mPtrClockproxy->getLocalTimeDate_MinuteUpdate().getS16Year();
      llocaldatetime.u8Month = _mPtrClockproxy->getLocalTimeDate_MinuteUpdate().getU8Month();
      llocaldatetime.u8Day =  _mPtrClockproxy->getLocalTimeDate_MinuteUpdate().getU8Day();
      llocaldatetime.u8Hours = _mPtrClockproxy->getLocalTimeDate_MinuteUpdate().getU8Hours();
      llocaldatetime.u8Minutes = _mPtrClockproxy->getLocalTimeDate_MinuteUpdate().getU8Minutes();
      llocaldatetime.u8Weekday = _mPtrClockproxy->getLocalTimeDate_MinuteUpdate().getU8Weekday();
      llocaldatetime.u8Seconds = _mPtrClockproxy->getLocalTimeDate_MinuteUpdate().getU8Seconds();
   }
   return llocaldatetime;
}


/**
 * Helper function to get current Time Zone.
 * @return : returns time zone.
 */
std::string SXMClockService::GetTimeZone()const
{
   ETG_TRACE_USR4(("SXMClockService GetTimeZone"));
   std::string sTimeZone = "";
   if (_mPtrClockproxy && _mPtrClockproxy->hasTimeZone())
   {
      sTimeZone = _mPtrClockproxy->getTimeZone().getSzDescription();
   }
   return sTimeZone;
}


/**
 * Register call back function to receive updates.
 * @param [in] : _obsptr - object reference of observer.
 * @param [in] : enSXMServiceType - data Service type
 */
void SXMClockService::registerTimeStatusNotification(ISxmClockService* _obsptr, enSXMServiceType enServiceType)
{
   ETG_TRACE_USR4(("SXMClockService::registerTimeStatusNotification"));
   if (_stateObsList.find(enServiceType) != _stateObsList.end())
   {
      ETG_TRACE_ERR(("Service is already registered"));
   }
   else
   {
      _stateObsList.insert(std::make_pair(enServiceType, _obsptr));
   }
}


/**
 * De-Register call back function to receive updates.
 * @param [in] : _obsptr - object reference of observer.
 * @param [in] : enSXMServiceType - data Service type
 */
void SXMClockService::deregisterTimeStatusNotification(ISxmClockService* _obsptr, enSXMServiceType enServiceType)
{
   if (_stateObsList.find(enServiceType) != _stateObsList.end())
   {
      _stateObsList.erase(enServiceType);
   }
   else
   {
      ETG_TRACE_ERR(("Service is not registered"));
   }
}


}  // namespace Core
}  // namespace App
