/**************************************************************************************
 * @file        : ApplicationSwitchClientHandler.cpp
 * @addtogroup  : AppHmi_Sxm
 * @brief       : DBus client for Application switch interface.
 * @copyright   : (c) 2019-2020 Robert Bosch Car Multimedia GmbH
 *                The reproduction, distribution and utilization of this file as
 *                well as the communication of its contents to others without express
 *                authorization is prohibited. Offenders will be held liable for the
 *                payment of damages. All rights reserved in the event of the grant
 *                of a patent, utility model or design.
 **************************************************************************************/

/*****************************************************************
| includes
|----------------------------------------------------------------*/
#include "hmi_trace_if.h"
#include "../SourceSwitch/SXMSourceSwitchInterface.h"
#include "AppHmi_MasterBase/AudioInterface/AudioDefines.h"
#include "App/Core/SxmUtils/SxmUtils.h"
#include "App/Core/SxmCommon/SXMCommonInterface.h"
#include "AppHmi_SxmStateMachineData.h"
#include "App/Core/ClientHandler/HmiInfoService/SxmHmiInfoService.h"
#include "ApplicationSwitchClientHandler.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS       TR_CLASS_APPHMI_SXMCOMMON
#define ETG_I_TRACE_CHANNEL           TR_TTFIS_APPHMI_SXM
#define ETG_I_TTFIS_CMD_PREFIX        "APPHMI_SXM_"
#define ETG_I_FILE_PREFIX             App::Core::ApplicationSwitchClientHandler::
#include "trcGenProj/Header/ApplicationSwitchClientHandler.cpp.trc.h"
#endif

using namespace ::bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch;

namespace App {
namespace Core {
/*
 * Constructor
 */
ApplicationSwitchClientHandler::ApplicationSwitchClientHandler()
   : _sxmApplicationSwitchClientProxy(ApplicationSwitchProxy::createProxy("applicationSwitchPort", *this))
   , _activeRegion(Region__REG_COCKPIT)
   , _isInternalRequestContext(false)
{
   if (_sxmApplicationSwitchClientProxy.get())
   {
      ETG_TRACE_USR4(("ApplicationSwitchClientHandler _sxmApplicationSwitchClientProxy.get is TRUE"));
   }
   _applicationSwitchClientComponent = ApplicationSwitchClientComponent::poGetInstance();
   SXM_UTILS_REGISTER_PROPERTY_CALLBACKS(StartupSync::getInstance(), this, _sxmApplicationSwitchClientProxy);
}


/*
 * Destructor
 */
ApplicationSwitchClientHandler::~ApplicationSwitchClientHandler()
{
   _sxmApplicationSwitchClientProxy.reset();
   _isInternalRequestContext = false;
   if (_applicationSwitchClientComponent != NULL)
   {
      _applicationSwitchClientComponent = NULL;
   }
}


/**
 * Utility function that registers for notifications updates from AudioSourceChange.
 * @param [in] : proxy - AudioSourceChange class instance
 * @param [in] : stateChange - Current ASF state
 */
void ApplicationSwitchClientHandler::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& /*proxy*/,
      const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("SXMSourceSwitch registerProperties"));
}


/**
 * Utility function that deregisters for notifications updates from AudioSourceChange.
 * @param [in] : proxy - AudioSourceChange class instance
 * @param [in] : stateChange - Current ASF state
 */
void ApplicationSwitchClientHandler::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& /*proxy*/,
      const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("SXMSourceSwitch registerProperties"));
}


/**
 * Utility function that registers for notifications updates from Application switch
 * @param [in] : proxy - Application switch class instance
 * @param [in] : stateChange - Current ASF state
 */
void ApplicationSwitchClientHandler::onAvailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy,
      const ::asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onAvailable"));
   if (_sxmApplicationSwitchClientProxy && _sxmApplicationSwitchClientProxy == proxy)
   {
      _sxmApplicationSwitchClientProxy->sendSig_ActivateContextRegister(*this);
      //_sxmApplicationSwitchClientProxy->sendActiveRegionIdGet(*this);
      _sxmApplicationSwitchClientProxy->sendActiveRegionIdRegister(*this);
   }
}


/**
 * Utility function that de-registers for notifications updates from Application switch
 * @param [in] : proxy - Application switch class instance
 * @param [in] : stateChange - Current ASF state
 */
void ApplicationSwitchClientHandler::onUnavailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy,
      const ::asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onUnavailable"));
   if (_sxmApplicationSwitchClientProxy && _sxmApplicationSwitchClientProxy == proxy)
   {
      _sxmApplicationSwitchClientProxy->sendSig_ActivateContextDeregisterAll();
      _sxmApplicationSwitchClientProxy->sendActiveRegionIdDeregisterAll();
   }
}


/**
 * function implemented to get update of ActiveRegionId Error
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'AudioSourceChangeProxy'
 * @param[in]      : error : The error message of 'ActiveRegionId'
 */
void ApplicationSwitchClientHandler::onActiveRegionIdError(const ::boost::shared_ptr< ApplicationSwitchProxy >& /*proxy*/,
      const ::boost::shared_ptr< ActiveRegionIdError >& /*error*/)
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onActiveRegionIdError"));
}


/**
 * function implemented to get update of ActiveRegionId Update
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'AudioSourceChangeProxy'
 * @param[in]      : Update : The response message of 'ActiveRegionId'
 */
void ApplicationSwitchClientHandler::onActiveRegionIdUpdate(const ::boost::shared_ptr< ApplicationSwitchProxy >& /*proxy*/,
      const ::boost::shared_ptr< ActiveRegionIdUpdate >& update)
{
   if (update && update->hasActiveRegionId())
   {
      _activeRegion = update->getActiveRegionId();
      ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onActiveRegionIdUpdate, activeRegion:%d", _activeRegion));
      SXMSourceSwitchInterface::setConnectionState(_activeRegion);
   }
}


/**
 * function implemented to get update of onSig_ActivateContext Error
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'ApplicationSwitchProxy'
 * @param[in]      : error : The error message
 */
void ApplicationSwitchClientHandler::onSig_ActivateContextError(const ::boost::shared_ptr< ApplicationSwitchProxy >& /*proxy*/,
      const ::boost::shared_ptr< Sig_ActivateContextError >& /*error*/)
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onSig_ActivateContextError"));
}


/**
 * function implemented to get update of onSig_ActivateContext Signal
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'ApplicationSwitchProxy'
 * @param[in]      : Update : The response message
 */
void ApplicationSwitchClientHandler::onSig_ActivateContextSignal(const ::boost::shared_ptr< ApplicationSwitchProxy >& /*proxy*/,
      const ::boost::shared_ptr< Sig_ActivateContextSignal >& signal)
{
   if (signal && signal->hasTargetactivityId())
   {
      ETG_TRACE_USR1(("ApplicationSwitchClientHandler::onSig_ActivateContextSignal %d", signal->getTargetactivityId()));
      enActivityIDs targetActivityID = signal->getTargetactivityId();
      displaySXMContextScene(targetActivityID);
   }
}


/**
 * function to handle activate context
 * @param[in]      : activity ID
 */
void ApplicationSwitchClientHandler::displaySXMContextScene(enActivityIDs targetActivityID)
{
   ETG_TRACE_USR1(("ApplicationSwitchClientHandler::displaySXMContextScene activityID:%d, _isInternalRequestContext=%d", targetActivityID, _isInternalRequestContext));
   switch (targetActivityID)
   {
      case enActivityIDs__eActivityID_SXM:
      {
         break;//handled in Tuner to show "Source selection" screen
      }

      case enActivityIDs__eActivityID_SXM_MAIN:
      {
         //This activityID informed to Master for AppFlyin handling (internal requestContext)
         //If called externally, Src change & Screen change to be handled
         if (!_isInternalRequestContext)
         {
            uint8 sxmNewsSourceStatus = SXMSourceSwitchInterface::getSxmNewsSourceStatus();
            uint8 sxmSourceStatues = SXMSourceSwitchInterface::getSxmSourceStatus();
            ETG_TRACE_USR1(("ApplicationSwitchClientHandler::displaySXMContextScene sxmNewsSourceStatus=%d", sxmNewsSourceStatus));
            ETG_TRACE_USR1(("ApplicationSwitchClientHandler::displaySXMContextScene sxmSourceStatues=%d", sxmSourceStatues));
            if (!(sxmNewsSourceStatus == connected || sxmNewsSourceStatus == suspended || sxmSourceStatues == connected || sxmSourceStatues == suspended))
            {
               SXMSourceSwitchInterface::handleSourceChangeRequest(SRC_TUNER_XM, true, true, false);
            }
            POST_MSG((COURIER_MESSAGE_NEW(SXMAudioGotoAudioMainMsg)()));
         }
         break;
      }

      case enActivityIDs__eActivityID_SXM_SETTINGS_VIEW:
      {
         POST_MSG((COURIER_MESSAGE_NEW(SXMAudioGotoAudioSettingsMsg)()));
         if (SXMSourceSwitchInterface::getSxmAudioScanStatus())//if scan active
         {
            POST_MSG((COURIER_MESSAGE_NEW(SXMAudioScanOpertationReqMsg)(SCAN_ABORT)));
         }
         uint32 _popUpID = SXMCommonInterface::getActiveViewName(Enum_SURFACEID_TOP_POPUP_SURFACE_SXM);
         ETG_TRACE_USR2(("ApplicationSwitchClientHandler::displaySXMContextScene PopupID:%d", _popUpID));
         if ((_popUpID == Sxm_SxmPopups_SXM_RADIO__POPUP_REMOVE_ALL_ARTISTS_USERCONF)
               || (_popUpID == Sxm_SxmPopups_SXM_RADIO__ICPOP_SF_FAVTEAM_GAME_AVAIL)
               || (_popUpID == Sxm_SxmPopups_SXM_RADIO__ICPOP_SF_NOGAME_FAVTEAM)
               || (_popUpID == Sxm_SxmPopups_SXM_RADIO__ICPOP_SF_REPLAY)
               || (_popUpID == Sxm_SxmPopups_SXM_RADIO__ICPOP_SPORTS_DEFINE_TEAM)
               || (_popUpID == Sxm_SxmPopups_SXM_RADIO__ICPOP_TWN_DEFINE_CITY)
               || (_popUpID == Sxm_SxmPopups_SXM_RADIO__MPOP_SPORTSFLASH_MAX_FAV)
               || (_popUpID == Sxm_SxmPopups_SXM_RADIO__POPUP_ALERT_ARTIST_SONG_SYSDLG))
         {
            POST_MSG((COURIER_MESSAGE_NEW(SXMDeActivatePopUpMsg)(_popUpID)));
         }
         break;
      }

      case enActivityIDs__eActivityID_SXM_SETTINGS_TESTMODE:
      {
         POST_MSG((COURIER_MESSAGE_NEW(SXMAudioGotoAudioSXMDTMSettingsMsg)()));
         break;
      }

      case enActivityIDs__eActivityID_SXM_MONITOR_TESTMODE:
      {
         POST_MSG((COURIER_MESSAGE_NEW(SXMAudioGotoAudioSXMMonitorMsg)()));
         break;
      }
      case enActivityIDs__eActivityID_HOME:
      {
         uint32 _popUpID = SXMCommonInterface::getActiveViewName(Enum_SURFACEID_TOP_POPUP_SURFACE_SXM);
         ETG_TRACE_USR4(("SXMCommonInterface::getActiveViewName::PopupID %u : ", _popUpID));
         if (_popUpID == Sxm_SxmPopups_SXM_RADIO__POPUP_REMOVE_ALL_ARTISTS_USERCONF)
         {
            POST_MSG((COURIER_MESSAGE_NEW(SXMDeActivatePopUpMsg)(_popUpID)));
         }
         break;
      }
      case enActivityIDs__eActivityID_SXM_AUDIO_UTILITY:
      {
         break;//Used by Master for AppFlyin
      }
      case enActivityIDs__eActivityID_SXM_SELF_CONTEXT:
      {
         POST_MSG((COURIER_MESSAGE_NEW(SXMAudioGotoAudioMainMsg)()));
         break;
      }
      default:
      {
         //do Nothing
         break;
      }
   }
   _isInternalRequestContext = false;
}


/**
 * Callback to handle Courier Message SxmContextSwitchScreenReqMsg
 * @param [in] : Reference of Courier::SxmContextSwitchScreenReqMsg
 * @return     : True-When Message is processed.
 */
bool ApplicationSwitchClientHandler::onCourierMessage(const SxmContextSwitchScreenReqMsg& oMsg)
{
   ETG_TRACE_USR1(("ApplicationSwitchClientHandler::SxmContextSwitchScreenReqMsg contextID: %d", oMsg.GetContextId()));
   switch (oMsg.GetContextId())
   {
      //Context switch request from sxm radio main scene to tuner source slection screen after short press on source button
      case SXM_BACK_BUTTON_RADIO_MAIN:
         if (_applicationSwitchClientComponent)
         {
            _applicationSwitchClientComponent->requestContextSwitchApplication(_activeRegion, APPID_APPHMI_SXM, enActivityIDs__eActivityID_SXM,
                  APPID_APPHMI_TUNER, enActivityIDs__eActivityID_TUNER_SOURCE_SELECTION);
         }
         break;

      //Context switch request from sxm testmode monitor scene to tuner testmode source slection screen after pressing back button
      case SXM_BACK_BUTTON_TEST_MODE_MONITOR:
         if (_applicationSwitchClientComponent)
         {
            _applicationSwitchClientComponent->requestContextSwitchApplication(_activeRegion, APPID_APPHMI_SXM, enActivityIDs__eActivityID_SXM,
                  APPID_APPHMI_TUNER, enActivityIDs__eActivityID_TUNER_TESTMODESCREEN);
         }
         break;

      //Context switch request from sxm testmode settings scene to tuner testmode source slection screen after pressing back button.
      case SXM_BACK_BUTTON_DTM_MAIN:
         if (_applicationSwitchClientComponent)
         {
            _applicationSwitchClientComponent->requestContextSwitchApplication(_activeRegion, APPID_APPHMI_SXM, enActivityIDs__eActivityID_SXM,
                  APPID_APPHMI_TUNER, enActivityIDs__eActivityID_TUNER_TESTMODESCREEN);
         }
         break;

      //Context switch request from Master home screen to tuner source slection screen after pressing source button.
      case SXM_SOURCE_BUTTON_HOME:
         if (_applicationSwitchClientComponent)
         {
            _applicationSwitchClientComponent->requestContextSwitchApplication(_activeRegion, APPID_APPHMI_MASTER, enActivityIDs__eActivityID_HOME,
                  APPID_APPHMI_TUNER, enActivityIDs__eActivityID_TUNER_SOURCE_SELECTION);
         }
         break;

      //Context switch request from Master home screen(Touch area in sxm gadget screen) to sxm radio main screen.
      case SXM_GADGET_TOUCH_AREA_HOME:
         if (_applicationSwitchClientComponent)
         {
            _applicationSwitchClientComponent->requestContextSwitchApplication(_activeRegion, APPID_APPHMI_MASTER, enActivityIDs__eActivityID_HOME,
                  APPID_APPHMI_SXM, enActivityIDs__eActivityID_SXM_MAIN);
         }
         break;

      //Context switch request from Home screen to Media screen
      case SXM_GADGET_BUTTON_TOGGLE:
         SxmHmiInfoService::instance()->sendToggleRequest(_activeRegion);
         break;

      default:
         // Do Nothing
         break;
   }
   return true;
}


/**
 * Function to request context
 * @param [in] : Source AppID
 * @param [in] : Source ActivityId
 */
void ApplicationSwitchClientHandler::requestSendContext(uint8 sourceAppId, enActivityIDs sourceActivityId)
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::requestSendContext"));
   if (_applicationSwitchClientComponent)
   {
      _applicationSwitchClientComponent->requestSendContext(_activeRegion, sourceAppId, sourceActivityId);
   }
   _isInternalRequestContext = true;
}


/**
 * Function to request Self context
 */
void ApplicationSwitchClientHandler::requestSxmSelfContext()
{
   ETG_TRACE_USR1(("ApplicationSwitchClientHandler::requestSxmSelfContext"));
   if (_applicationSwitchClientComponent /*&& (SXMSourceSwitchInterface::getPrevSource() != SRC_TUNER_XM)*/)
   {
      _applicationSwitchClientComponent->requestContextSwitchApplication(_activeRegion, APPID_APPHMI_UNKNOWN, enActivityIDs__eActivityID_NONE,
            APPID_APPHMI_SXM, enActivityIDs__eActivityID_SXM_SELF_CONTEXT);
   }
}


/*
 * Function for context switch request return to previous source
 */
void ApplicationSwitchClientHandler::returnToPreviousSource()
{
   ETG_TRACE_USR1(("ApplicationSwitchClientHandler::returnToPreviousSource"));
   if (_applicationSwitchClientComponent && (SXMSourceSwitchInterface::getPrevSource() != SRC_TUNER_XM))//if previous source is SXM, no need to do context switch
   {
      _applicationSwitchClientComponent->requestContextSwitchApplication(_activeRegion, APPID_APPHMI_SXM, enActivityIDs__eActivityID_SXM_NEWS_RETURN,
            APPID_APPHMI_UNKNOWN, enActivityIDs__eActivityID_NONE);
   }
}


/**
 * Function to get active region
 * @return     : True-When Message is processed.
 */
uint8 ApplicationSwitchClientHandler::getActiveRegion()
{
   return _activeRegion;
}


}
}
