/**************************************************************************************
 * @file         : VehicleDataSpiClientHandler.cpp
 * @author       : Bruce Mario Savio Netto (RBEI/ECH2)
 * @addtogroup   : AppHmi_SPI
 * @brief        :
 * @copyright    : (c) 2019-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 *09.01.19 Rev. 0.1 Bruce Mario Savio Netto (RBEI/ECH2)
 * 		  			Initial Revision.
 **************************************************************************************/

#include "hall_std_if.h"
#include "Core/VehicleDataSpiClientHandler/VehicleDataSpiClientHandler.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_SPI_HALL
#define ETG_I_TRACE_CHANNEL TR_TTFIS_APPHMI_SPI
#define ETG_I_TTFIS_CMD_PREFIX "APPHMI_SPI_"
#define ETG_I_FILE_PREFIX App::Core::VehicleDataSpiClientHandler::
#include "trcGenProj/Header/VehicleDataSpiClientHandler.cpp.trc.h"
#endif

using namespace ::asf::core;
using namespace ::vehicle_main_fi_types;
using namespace ::VEHICLE_MAIN_FI;
using namespace ::midw_smartphoneint_fi;
using namespace ::midw_smartphoneint_fi_types;
using namespace ::bosch::cm::ai::nissan::hmi::hmidataservice::HmiData;

namespace App {
namespace Core {
/*****************************************************************
| static data implementation
|----------------------------------------------------------------*/
VehicleDataSpiClientHandler* VehicleDataSpiClientHandler::_vehicleDataSpiClientHandler = NULL;

/************************************************************************
*FUNCTION:		VehicleDataSpiClientHandler
*DESCRIPTION:	Constructor
*PARAMETER:     None
*RETURNVALUE:	NA
*SWFL:
*HISTORY:
*0.1     09.01.19      Bruce Netto      Initial Version
************************************************************************/
VehicleDataSpiClientHandler::VehicleDataSpiClientHandler()
   : _vehicleProxy(::VEHICLE_MAIN_FIProxy::createProxy("vehicleMainFiPort", *this))
   , _smartphoneFIProxy(Midw_smartphoneint_fiProxy::createProxy("midwspiPort", *this))
   , _hmiDataProxyClient(HmiDataProxy::createProxy("hmiDataServicePort", *this))
   , _connectionClientHandler()
{
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::VehicleDataSpiClientHandler()"));
   if (_vehicleProxy.get())
   {
      ETG_TRACE_USR4(("VehicleDataSpiClientHandler::VehicleDataSpiClientHandler StartupSync _vehicleProxy"));
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _vehicleProxy->getPortName());
   }
   if (_smartphoneFIProxy.get())
   {
      ETG_TRACE_USR4(("VehicleDataSpiClientHandler::VehicleDataSpiClientHandler StartupSync _smartphoneFIProxy"));
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _smartphoneFIProxy->getPortName());
   }
   if (_hmiDataProxyClient.get())
   {
      ETG_TRACE_USR4(("VehicleDataSpiClientHandler::VehicleDataSpiClientHandler StartupSync _hmiDataProxyClient"));
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _hmiDataProxyClient->getPortName());
   }
   uint8 languageIndexSystem = _dpVDLanguageIndexSystem.tGetData();
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onCourierMessage GuiStartupFinishedUpdMsg languageIndexSystem = %d", languageIndexSystem));
   onChangeLanguage(languageIndexSystem);
   setVehiclespeed(m_speedvalue);
   ETG_I_REGISTER_FILE();
   vInitialize();
}


/************************************************************************
*FUNCTION:		~VehicleDataSpiClientHandler
*DESCRIPTION:	Destructor
*PARAMETER:     None
*RETURNVALUE:	NA
*SWFL:
*HISTORY:
*0.1     09.01.19      Bruce Netto      Initial Version
************************************************************************/
VehicleDataSpiClientHandler::~VehicleDataSpiClientHandler()
{
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::~VehicleDataSpiClientHandler()"));
   if (_vehicleDataSpiClientHandler)
   {
      delete _vehicleDataSpiClientHandler;
   }
   _vehicleDataSpiClientHandler = NULL;
   _smartphoneFIProxy.reset();
   ETG_I_UNREGISTER_FILE();
}


/************************************************************************
*FUNCTION:		deregisterProperties
*DESCRIPTION:	Trigger property deregistration
*PARAMETER:     proxy, stateChange
*RETURNVALUE:	NA
*SWFL:
*HISTORY:
*0.1     09.01.19      Bruce Netto      Initial Version
************************************************************************/
void VehicleDataSpiClientHandler::deregisterProperties(const ::boost::shared_ptr<asf::core::Proxy>& proxy,
      const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::deregisterProperties()"));
   if ((proxy == _vehicleProxy) && (_vehicleProxy.get()))
   {
      _vehicleProxy->sendSpeedRelUpRegAll();
   }
}


/************************************************************************
*FUNCTION:		registerProperties
*DESCRIPTION:	Trigger property registration
*PARAMETER:     proxy, stateChange
*RETURNVALUE:	NA
*SWFL:
*HISTORY:
*0.1     09.01.19      Bruce Netto      Initial Version
************************************************************************/
void VehicleDataSpiClientHandler::registerProperties(const ::boost::shared_ptr<asf::core::Proxy>& proxy,
      const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::registerProperties()"));
   if ((proxy == _vehicleProxy) && (_vehicleProxy.get()))
   {
      _vehicleProxy->sendSpeedUpReg(*this);
   }
}


/************************************************************************
*FUNCTION:		onUnavailable
*DESCRIPTION:	Method called when SPI service is unavailable
*PARAMETER:     proxy, stateChange
*RETURNVALUE:	NA
*SWFL:
*HISTORY:
*0.1     09.01.19      Bruce Netto      Initial Version
************************************************************************/
void VehicleDataSpiClientHandler::onUnavailable(const ::boost::shared_ptr<asf::core::Proxy>& proxy,
      const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onUnavailable()"));
   StartupSync::getInstance().onUnavailable(proxy, stateChange);
   if ((proxy == _vehicleProxy) && (_vehicleProxy.get()))
   {
      _vehicleProxy->sendLanguageRelUpRegAll();
      _vehicleProxy->sendSpeedRelUpRegAll();
   }
   if ((proxy == _hmiDataProxyClient) && (_hmiDataProxyClient.get()))
   {
      _hmiDataProxyClient->sendDimmingModeDeregisterAll();
   }
}


/************************************************************************
*FUNCTION:		onAvailable
*DESCRIPTION:	Method called when SPI service is available
*PARAMETER:     proxy, stateChange
*RETURNVALUE:	NA
*SWFL:
*HISTORY:
*0.1     09.01.19      Bruce Netto      Initial Version
************************************************************************/
void VehicleDataSpiClientHandler::onAvailable(const ::boost::shared_ptr<asf::core::Proxy>& proxy,
      const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onAvailable()"));
   StartupSync::getInstance().onAvailable(proxy, stateChange);
   if ((proxy == _vehicleProxy) && (_vehicleProxy.get()))
   {
      _vehicleProxy->sendLanguageUpReg(*this);
      _vehicleProxy->sendLanguageSyncSourceUpReg(*this);
      _vehicleProxy->sendSpeedUpReg(*this);
   }
   if ((proxy == _hmiDataProxyClient) && (_hmiDataProxyClient.get()))
   {
      _hmiDataProxyClient->sendDimmingModeRegister(*this);
   }
}


/************************************************************************
*FUNCTION:		onCourierMessage
*DESCRIPTION:	Function to process courier message
*PARAMETER:     StartupMsg
*RETURNVALUE:	bool
*SWFL:          NA
*HISTORY:
*0.1     09.01.19      Bruce Netto      Initial Version
************************************************************************/
bool VehicleDataSpiClientHandler::onCourierMessage(const Courier::StartupMsg& oMsg)
{
   return true;
}


/************************************************************************
*FUNCTION:		vInitialize
*DESCRIPTION:	Initialization of member variables
*PARAMETER:     None
*RETURNVALUE:	NA
*SWFL:
*HISTORY:
*0.1     09.01.19      Bruce Netto      Initial Version
************************************************************************/
void VehicleDataSpiClientHandler::vInitialize()
{
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::vInitialize()"));
   m_DimmingStatusInfo = T_e8_Vehicle_Configuration__DAY_MODE;
   m_parkModeRestrictionInfo = AAP_DONTHIDE;
   m_carplayDriveModeRestriction = AAP_DONTHIDE;
   m_GeneralRestriction = AAP_DONTHIDE;
   m_speedvalue = 0;
   m_gearvalue = 0;
   m_parkmode_decimal =  AAP_C_UNRESTRICTED ;
   m_drivemode_decimal =  AAP_C_UNRESTRICTED ;
   m_GeneralRestriction = AAP_DontHide ;
   m_ParkBrakeInfo = ::midw_smartphoneint_fi_types::T_e8_ParkBrake__PARK_BRAKE_NOT_ENGAGED;
   m_GearStateInfo = ::midw_smartphoneint_fi_types::T_e8_GearState__GEAR_INFO_UNAVAILABLE;
   m_VehicleStateInfo = ::midw_smartphoneint_fi_types::T_e8_VehicleState__PARK_MODE;
}


/************************************************************************
*FUNCTION:		onSetLanguageResult
*DESCRIPTION:	Function to get the Set Language Result
*PARAMETER:     StartupMsg
*RETURNVALUE:	bool
*SWFL:          Task 449426 SWFL-2727
*HISTORY:
*0.1     09.01.19      Bruce Netto      Initial Version
************************************************************************/
void VehicleDataSpiClientHandler::onSetLanguageResult(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& /*proxy*/,  const ::boost::shared_ptr< SetLanguageResult >& result)
{
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onSetLanguageResult received :  %d", result->getResult()));
}


/************************************************************************
*FUNCTION:		onSetLanguageError
*DESCRIPTION:	Function to get the Set Language Error
*PARAMETER:     StartupMsg
*RETURNVALUE:	bool
*SWFL:          Task 449426 SWFL-2727
*HISTORY:
*0.1     09.01.19      Bruce Netto      Initial Version
************************************************************************/
void VehicleDataSpiClientHandler::onSetLanguageError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& /*proxy*/, const ::boost::shared_ptr< SetLanguageError >& /*error*/)
{
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onSetLanguageError"));
}


/************************************************************************
*FUNCTION:		onLanguageStatus
*DESCRIPTION:	Function to get the Language Status
*PARAMETER:     StartupMsg
*RETURNVALUE:	bool
*SWFL:          Task 449426 SWFL-2727
*HISTORY:
*0.1     09.01.19      Bruce Netto      Initial Version
* todo :check As of now, after setLanguage, onLanguageStatus() is received twice from the VD_Vehicle data.
* First update represents, previously configured Language ID and the second one represents currently configured ID.
************************************************************************/
void VehicleDataSpiClientHandler::onLanguageStatus(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& /*proxy*/,  const ::boost::shared_ptr< LanguageStatus >& status)
{
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onLanguageStatus"));
   uint8 l_LanguageIndex = status->getLanguage();
   //Important to trigger language change it has no other use throughout this code scope, the below vector
   std::vector<T_Language_SourceTable> l_langTable = status->getLangTable();
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onLanguageStatus l_LanguageIndex = %d", l_LanguageIndex));
   //Persistency Storage
   _dpVDLanguageIndexSystem.vSetData((uint8)l_LanguageIndex);
   std::string l_textid = getNewLanguageID(l_LanguageIndex);
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onLanguageStatus l_textid = %s", l_textid.c_str()));
   //COURIER_MESSAGE_NEW(Courier::SetCultureReqMsg)(Courier::ItemId(l_textid.c_str()))->Post();
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onLanguageStatus Courier Posted"));
}


/************************************************************************
*FUNCTION:		getNewLanguageID
*DESCRIPTION:	Function to check the LanguageTable and assign
*PARAMETER:     StartupMsg
*RETURNVALUE:	bool
*SWFL:          Task 449426 SWFL-2727
*HISTORY:
*0.1     09.01.19      Bruce Netto      Initial Version
************************************************************************/
std::string VehicleDataSpiClientHandler::getNewLanguageID(int index)
{
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::getNewLanguageID"));
   std::string l_langId;
   for (int i = 0; i < LANGNO; i++)
   {
      if (LanguageMappingTable[i].LangIndex == index)
      {
         l_langId = LanguageMappingTable[i].LangString.c_str();
         ETG_TRACE_USR4(("VehicleDataSpiClientHandler::getNewLanguageID Looper %s", l_langId));
         break;
      }
   }
   return l_langId;
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::getNewLanguageID l_langId returned %s", l_langId));
}


/************************************************************************
*FUNCTION:		onLanguageError
*DESCRIPTION:	Function to check if an Error Occurred in the Language callback
*PARAMETER:     StartupMsg
*RETURNVALUE:	bool
*SWFL:          Task 449426
*HISTORY:
*0.1     09.01.19      Bruce Netto      Initial Version
************************************************************************/
void VehicleDataSpiClientHandler::onLanguageError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& /*proxy*/, const ::boost::shared_ptr< LanguageError >& /*error*/)
{
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onLanguageError"));
}


/************************************************************************
*FUNCTION:		onCourierMessage GuiStartupFinishedUpdMsg
*DESCRIPTION:	Function to check if the system is started up
*PARAMETER:     StartupMsg
*RETURNVALUE:	bool
*SWFL:          Task 449426 SWFL-2727
*HISTORY:
*0.1     18.01.19      Bruce Netto      Initial Version
************************************************************************/
bool VehicleDataSpiClientHandler::onCourierMessage(const GuiStartupFinishedUpdMsg& /*msg*/)
{
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onCourierMessage(const GuiStartupFinishedUpdMsg()"));
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onCourierMessage(const GuiStartupFinishedUpdMsg() Finished"));
   return false;
}


/************************************************************************
*FUNCTION:		onChangeLanguage
*DESCRIPTION:	Function to check if the language index has changed
*PARAMETER:     l_LanguageIndex
*RETURNVALUE:	void
*SWFL:          Task 449426, SWFL-2727
*HISTORY:
*0.1     18.01.19      Bruce Netto      Initial Version
************************************************************************/
void VehicleDataSpiClientHandler::onChangeLanguage(int l_LanguageIndex)
{
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onChangeLanguage()"));
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onChangeLanguage l_LanguageIndex = %d", l_LanguageIndex));
   std::string l_textid = getNewLanguageID(l_LanguageIndex);
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onChangeLanguage l_textid = %s", l_textid.c_str()));
   //COURIER_MESSAGE_NEW(Courier::SetCultureReqMsg)(Courier::ItemId(l_textid.c_str()))->Post();
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onChangeLanguage Courier Posted"));
}


/************************************************************************
*FUNCTION:		onSpeedStatus
*DESCRIPTION:	Function to receive the vehicles speed status
*PARAMETER:     result
*RETURNVALUE:	void
*SWFL:          SWFL-2599
*HISTORY:
*0.1     26.03.19      Bruce Netto      Initial Version
************************************************************************/
void VehicleDataSpiClientHandler::onSpeedStatus(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
      const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::SpeedStatus >& status)
{
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onSpeedStatus speedvalue = %d", status->getSpeedValue()));
   m_ActiveDeviceCategory = _connectionClientHandler->tGetActiveDeviceCategory();
   if (status->getSpeedValue() <= 0)
   {
      ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onSpeedStatus PARK_MODE sent"));
      getVehicleStateInfo(PARK_MODE);
      if (m_ActiveDeviceCategory == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_ANDROIDAUTO)
      {
         setParkBrakeInfo(PARK_BRAKE_ENGAGED);
      }
      else if (m_ActiveDeviceCategory == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_MYSPIN)
      {
         setParkBrakeInfo(PARK_BRAKE_INFO_UNAVAILABLE);
      }
      getGearStateInfo();
   }
   else
   {
      ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onSpeedStatus speedvalue DRIVE_MODE sent"));
      getVehicleStateInfo(DRIVE_MODE);
      if (m_ActiveDeviceCategory == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_ANDROIDAUTO)
      {
         setParkBrakeInfo(PARK_BRAKE_NOT_ENGAGED);
      }
      else if (m_ActiveDeviceCategory == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_MYSPIN)
      {
         setParkBrakeInfo(PARK_BRAKE_INFO_UNAVAILABLE);
      }
      getGearStateInfo();
   }

   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onSpeedStatus setVehicleMovementState sent m_ParkBrakeInfo %d// m_GearStateInfo %d// m_VehicleStateInfo %d//", m_ParkBrakeInfo, m_GearStateInfo, m_VehicleStateInfo));
   setVehicleMovementState(m_ParkBrakeInfo, m_GearStateInfo, m_VehicleStateInfo);
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::ActiveDeviceCategory= %d" , m_ActiveDeviceCategory));
   if (m_ActiveDeviceCategory == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_DIPO)
   {
      ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onSpeedStatus speedvalue = %d", status->getSpeedValue()));
      m_speedvalue = int32((status->getSpeedValue() / 100));
      setVehiclespeed(m_speedvalue);
      setFeatureRestrictions_Carplay();
   }
   else if (m_ActiveDeviceCategory == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_ANDROIDAUTO)
   {
      ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onSpeedStatus speedvalue = %d", status->getSpeedValue()));
      m_speedvalue = int32((status->getSpeedValue() / 100));
      setVehiclespeed(m_speedvalue);
      getVehicleCondition();
   }
}


/************************************************************************
*FUNCTION:		onSpeedError
*DESCRIPTION:	Function to receive the vehicles speed status error message
*PARAMETER:     result
*RETURNVALUE:	void
*SWFL:          SWFL-2599
*HISTORY:
*0.1     26.03.19      Bruce Netto      Initial Version
************************************************************************/
void VehicleDataSpiClientHandler::onSpeedError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
      const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::SpeedError >& error)
{
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onSpeedError "));
}


/************************************************************************
*FUNCTION:		onGearStatus
*DESCRIPTION:	Function to receive the vehicles gear status
*PARAMETER:     result
*RETURNVALUE:	void
*SWFL:
*HISTORY:
*0.1     26.03.19      Bruce Netto      Initial Version
************************************************************************/
// void VehicleDataSpiClientHandler::onGearStatus(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
// const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::GearStatus >& status)
// {
// ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onGearStatus gearvalue = %d", status->getGearValue()));
// m_gearvalue = status->getGearValue();
// }


/************************************************************************
*FUNCTION:		onGearError
*DESCRIPTION:	Function to receive the vehicles gear status error message
*PARAMETER:     result
*RETURNVALUE:	void
*SWFL:
*HISTORY:
*0.1     26.03.19      Bruce Netto      Initial Version
************************************************************************/
// void VehicleDataSpiClientHandler::onGearError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
// const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::GearError >& error)
// {
// ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onGearError "));
// }


/************************************************************************
*FUNCTION:		setParkBrakeInfo
*DESCRIPTION:	Function to send the park type of the vehicle
*PARAMETER:     result
*RETURNVALUE:	void
*SWFL:          SWFL-2599
*HISTORY:
*0.1     25.03.19      Bruce Netto      Initial Version
************************************************************************/
void VehicleDataSpiClientHandler::setParkBrakeInfo(uint8 State)
{
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::setParkBrakeInfo"));

   switch (State)
   {
      case PARK_BRAKE_INFO_UNAVAILABLE :
      {
         ETG_TRACE_USR4(("VehicleDataSpiClientHandler::getParkBrakeInfo case PARK_BRAKE_INFO_UNAVAILABLE"));
         m_ParkBrakeInfo = T_e8_ParkBrake__PARK_BRAKE_INFO_UNAVAILABLE;
         break;
      }
      case PARK_BRAKE_NOT_ENGAGED :
      {
         ETG_TRACE_USR4(("VehicleDataSpiClientHandler::getParkBrakeInfo case PARK_BRAKE_NOT_ENGAGED"));
         m_ParkBrakeInfo = T_e8_ParkBrake__PARK_BRAKE_NOT_ENGAGED;
         break;
      }
      case PARK_BRAKE_ENGAGED :
      {
         ETG_TRACE_USR4(("VehicleDataSpiClientHandler::getParkBrakeInfo case PARK_BRAKE_ENGAGED"));
         m_ParkBrakeInfo = T_e8_ParkBrake__PARK_BRAKE_ENGAGED;
         POST_MSG((COURIER_MESSAGE_NEW(SPI_SPI_ParkedCondition_Popup_Close)()));
         break;
      }
      default:
         break;
   }
   //setVehicleMovementState(m_ParkBrakeInfo,getGearStateInfo(),m_VehicleStateInfo);
}


//*FUNCTION:		getParkBrakeInfo

uint32 VehicleDataSpiClientHandler::getParkBrakeInfo()
{
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::getParkBrakeInfo"));
   return m_ParkBrakeInfo;
}


/************************************************************************
*FUNCTION:		getGearStateInfo
*DESCRIPTION:	Function to send the gear state of the vehicle
*PARAMETER:     result
*RETURNVALUE:	void
*SWFL:          SWFL-2599
*HISTORY:
*0.1     25.03.19      Bruce Netto      Initial Version
************************************************************************/
midw_smartphoneint_fi_types::T_e8_GearState VehicleDataSpiClientHandler::getGearStateInfo()
{
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::getGearStateInfo"));
   m_GearStateInfo = T_e8_GearState__GEAR_INFO_UNAVAILABLE;
   if (m_gearvalue == GEAR_NEUTRAL)
   {
      m_GearStateInfo = T_e8_GearState__GEAR_NEUTRAL;
   }
   else if (m_gearvalue == GEAR_1)
   {
      m_GearStateInfo = T_e8_GearState__GEAR_1;
   }
   else if (m_gearvalue == GEAR_2)
   {
      m_GearStateInfo = T_e8_GearState__GEAR_2;
   }
   else if (m_gearvalue == GEAR_3)
   {
      m_GearStateInfo = T_e8_GearState__GEAR_3;
   }
   else if (m_gearvalue == GEAR_4)
   {
      m_GearStateInfo = T_e8_GearState__GEAR_4;
   }
   else if (m_gearvalue == GEAR_5)
   {
      m_GearStateInfo = T_e8_GearState__GEAR_5;
   }
   else if (m_gearvalue == GEAR_6)
   {
      m_GearStateInfo = T_e8_GearState__GEAR_6;
   }
   else if (m_gearvalue == GEAR_LOW)
   {
      m_GearStateInfo = T_e8_GearState__GEAR_LOW;
   }
   else if (m_gearvalue == GEAR_DRIVE)
   {
      m_GearStateInfo = T_e8_GearState__GEAR_DRIVE;
   }
   else if (m_gearvalue == GEAR_PARK)
   {
      m_GearStateInfo = T_e8_GearState__GEAR_PARK;
   }
   else if (m_gearvalue == GEAR_REVERSE)
   {
      m_GearStateInfo = T_e8_GearState__GEAR_REVERSE;
   }
   else if (m_gearvalue == GEAR_INFO_UNAVAILABLE)
   {
      m_GearStateInfo = T_e8_GearState__GEAR_INFO_UNAVAILABLE;
   }
   // if (_mParkBrakeInfo)
   // {
   // m_GearStateInfo = T_e8_GearState__GEAR_REVERSE;
   // }
   // else if (_mParkBrakeInfo == T_e8_ParkBrake__PARK_BRAKE_ENGAGED)
   // {
   // m_GearStateInfo = T_e8_GearState__GEAR_PARK;
   // }
   // else if (_mVehicleSpeed == 0)
   // {
   // m_GearStateInfo = T_e8_GearState__GEAR_NEUTRAL;
   // }
   // else if (_mVehicleSpeed > 0)
   // {
   // m_GearStateInfo = T_e8_GearState__GEAR_DRIVE;
   // }
   return m_GearStateInfo;
}


/************************************************************************
*FUNCTION:		getVehicleStateInfo
*DESCRIPTION:	Function to send the vehicle's current state
*PARAMETER:     result
*RETURNVALUE:	void
*SWFL:          SWFL-2599
*HISTORY:
*0.1     25.03.19      Bruce Netto      Initial Version
************************************************************************/
void VehicleDataSpiClientHandler::getVehicleStateInfo(uint8 State)
{
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::getVehicleStateInfo"));
   m_ActiveDeviceCategory = _connectionClientHandler->tGetActiveDeviceCategory();
   switch (State)
   {
      case PARK_MODE :
      {
         m_VehicleStateInfo = ::midw_smartphoneint_fi_types::T_e8_VehicleState__PARK_MODE;

         break;
      }
      case DRIVE_MODE :
      {
         m_VehicleStateInfo =  ::midw_smartphoneint_fi_types::T_e8_VehicleState__DRIVE_MODE;

         break;
      }
      default:
         break;
   }
   if (m_ActiveDeviceCategory == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_ANDROIDAUTO)
   {
      //setFeatureRestrictionsAAP();
   }
   //setVehicleMovementState(m_ParkBrakeInfo,getGearStateInfo(),m_VehicleStateInfo);
}


/************************************************************************
*FUNCTION:		setVehicleMovementState
*DESCRIPTION:	Function to set the vehicle movement state as park or drive
*PARAMETER:     result
*RETURNVALUE:	void
*SWFL:          SWFL-2599
*HISTORY:
*0.1     21.03.19      Bruce Netto      Initial Version
************************************************************************/
void VehicleDataSpiClientHandler::setVehicleMovementState(::midw_smartphoneint_fi_types::T_e8_ParkBrake ParkBreakInfo,
      ::midw_smartphoneint_fi_types::T_e8_GearState GearStateInfo,
      ::midw_smartphoneint_fi_types::T_e8_VehicleState VehicleStateInfo)
{
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::setVehicleMovementState Received m_ParkBrakeInfo %d// m_GearStateInfo %d// m_VehicleStateInfo %d//", m_ParkBrakeInfo, m_GearStateInfo, m_VehicleStateInfo));
   _smartphoneFIProxy->sendSetVehicleMovementStateStart(*this,
         m_ParkBrakeInfo,
         m_GearStateInfo,
         m_VehicleStateInfo);
}


//FUNCTION:		getVehicleCondition
void VehicleDataSpiClientHandler::getVehicleCondition()
{
   if (m_speedvalue == SPEED_STATIONARY)
   {
      setParkBrakeInfo(PARK_BRAKE_ENGAGED);
      ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onSpeedStatus PARK_MODE sent"));
      getVehicleStateInfo(PARK_MODE);
   }
   else
   {
      setParkBrakeInfo(PARK_BRAKE_NOT_ENGAGED);

      if (m_speedvalue < SPEED_TRESHOLD)
      {
         ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onSpeedStatus PARK_MODE sent"));
         getVehicleStateInfo(PARK_MODE);
      }
      else
      {
         ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onSpeedStatus speedvalue DRIVE_MODE sent"));
         getVehicleStateInfo(DRIVE_MODE);
      }
   }
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onSpeedStatus setVehicleMovementState sent m_ParkBrakeInfo %d// m_GearStateInfo %d// m_VehicleStateInfo %d//", m_ParkBrakeInfo, m_GearStateInfo, m_VehicleStateInfo));
   setVehicleMovementState(m_ParkBrakeInfo, m_GearStateInfo, m_VehicleStateInfo);
}


/************************************************************************
*FUNCTION:		onSetVehicleMovementStateResult
*DESCRIPTION:	Function to set the vehicle movement state as park or drive
*PARAMETER:     result
*RETURNVALUE:	void
*SWFL:          SWFL-2599
*HISTORY:
*0.1     21.03.19      Bruce Netto      Initial Version
************************************************************************/
void VehicleDataSpiClientHandler::onSetVehicleMovementStateResult(const ::boost::shared_ptr< midw_smartphoneint_fi::Midw_smartphoneint_fiProxy >& proxy,
      const ::boost::shared_ptr< midw_smartphoneint_fi::SetVehicleMovementStateResult >& result)
{
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onSetVehicleMovementStateResult received"));
}


/************************************************************************
*FUNCTION:		onSetVehicleMovementStateError
*DESCRIPTION:	Function to get the Set Language Error
*PARAMETER:     error message
*RETURNVALUE:	void
*SWFL:          SWFL-2599
*HISTORY:
*0.1     21.03.19      Bruce Netto      Initial Version
************************************************************************/
void VehicleDataSpiClientHandler::onSetVehicleMovementStateError(const ::boost::shared_ptr< midw_smartphoneint_fi::Midw_smartphoneint_fiProxy >& proxy,
      const ::boost::shared_ptr< midw_smartphoneint_fi::SetVehicleMovementStateError >& error)
{
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onSetVehicleMovementStateError received"));
}


/************************************************************************
*FUNCTION:		onDimmingModeError
*DESCRIPTION:	Function to send the vehicle's day night mode
*PARAMETER:     result
*RETURNVALUE:	void
*SWFL:          SWFL-2599
*HISTORY:
*0.1     28.03.19      Bruce Netto      Initial Version
************************************************************************/
void VehicleDataSpiClientHandler::onDimmingModeError(const ::boost::shared_ptr< ::bosch::cm::ai::nissan::hmi::hmidataservice::HmiData::HmiDataProxy >& proxy,
      const ::boost::shared_ptr< ::bosch::cm::ai::nissan::hmi::hmidataservice::HmiData::DimmingModeError >& error)
{
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onDimmingModeError"));
}


/************************************************************************
*FUNCTION:		onDimmingModeUpdate
*DESCRIPTION:	Function to send the vehicle's day night mode
*PARAMETER:     result
*RETURNVALUE:	void
*SWFL:          SWFL-2599
*HISTORY:
*0.1     28.03.19      Bruce Netto      Initial Version
************************************************************************/
void VehicleDataSpiClientHandler::onDimmingModeUpdate(const ::boost::shared_ptr< ::bosch::cm::ai::nissan::hmi::hmidataservice::HmiData::HmiDataProxy >& proxy,
      const ::boost::shared_ptr< ::bosch::cm::ai::nissan::hmi::hmidataservice::HmiData::DimmingModeUpdate >& update)
{
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onDimmingModeUpdate"));
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onDimmingModeUpdate(%d)", update->getDimmingMode()));
   uint8 State = update->getDimmingMode();
   getDimmingStatusInfo(State);
}


/************************************************************************
*FUNCTION:		setVehicleConfiguration
*DESCRIPTION:	Function to set the vehicle mode as Day or Night
*PARAMETER:     result
*RETURNVALUE:	void
*SWFL:          SWFL-2599
*HISTORY:
*0.1     22.03.19      Bruce Netto      Initial Version
************************************************************************/
void VehicleDataSpiClientHandler::setVehicleConfiguration(::midw_smartphoneint_fi_types::T_e8_Vehicle_Configuration)
{
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::setVehicleConfiguration"));
   _smartphoneFIProxy->sendSetVehicleConfigurationStart(*this,
         m_DimmingStatusInfo,
         true);
   //post to SPI middleware
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::setVehicleConfiguration Dimming Mode Sent to SPI MW: %d", m_DimmingStatusInfo));
}


/************************************************************************
*FUNCTION:		getDimmingStatusInfo
*DESCRIPTION:	Function to send the vehicle's day night mode
*PARAMETER:     result
*RETURNVALUE:	void
*SWFL:          SWFL-2599
*HISTORY:
*0.1     25.03.19      Bruce Netto      Initial Version
************************************************************************/
void VehicleDataSpiClientHandler::getDimmingStatusInfo(uint8 State)
{
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::getDimmingStatusInfo"));
   switch (State)
   {
      case DAY_MODE:
      case AUTO_DAY_MODE:
      {
         m_DimmingStatusInfo = T_e8_Vehicle_Configuration__DAY_MODE;
         ETG_TRACE_USR4(("VehicleDataSpiClientHandler::getDimmingStatusInfo : DAY MODE SET"));
         break;
      }
      case NIGHT_MODE:
      case AUTO_NIGHT_MODE:
      {
         m_DimmingStatusInfo = T_e8_Vehicle_Configuration__NIGHT_MODE;
         ETG_TRACE_USR4(("VehicleDataSpiClientHandler::getDimmingStatusInfo : NIGHT MODE SET"));
         break;
      }
      default:
         break;
   }
   setVehicleConfiguration(m_DimmingStatusInfo);
}


/************************************************************************
*FUNCTION:		onSetVehicleConfigurationResult
*DESCRIPTION:	Function to get the result of the sent mode
*               [Day/Night] to middleware
*PARAMETER:     result
*RETURNVALUE:	void
*SWFL:          SWFL-2599
*HISTORY:
*0.1     22.03.19      Bruce Netto      Initial Version
************************************************************************/
void VehicleDataSpiClientHandler::onSetVehicleConfigurationResult(const ::boost::shared_ptr< midw_smartphoneint_fi::Midw_smartphoneint_fiProxy >& proxy,
      const ::boost::shared_ptr< midw_smartphoneint_fi::SetVehicleConfigurationResult >& result)
{
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onSetVehicleConfigurationResult received"));
}


/************************************************************************
*FUNCTION:		onSetVehicleConfigurationError
*DESCRIPTION:	Function to get the Error of dimming
*PARAMETER:     error message
*RETURNVALUE:	void
*SWFL:          SWFL-2599
*HISTORY:
*0.1     22.03.19      Bruce Netto      Initial Version
************************************************************************/
void VehicleDataSpiClientHandler::onSetVehicleConfigurationError(const ::boost::shared_ptr< midw_smartphoneint_fi::Midw_smartphoneint_fiProxy >& proxy,
      const ::boost::shared_ptr< midw_smartphoneint_fi::SetVehicleConfigurationError >& error)
{
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onSetVehicleConfigurationError received"));
}


/************************************************************************
*FUNCTION:		setFeatureRestrictions_Carplay
*DESCRIPTION:	Function to set the Feature Restriction for Carplay
*PARAMETER:     NA
*RETURNVALUE:	void
************************************************************************/
void VehicleDataSpiClientHandler::setFeatureRestrictions_Carplay()
{
   ETG_TRACE_USR4(("*******************VehicleDataSpiClientHandler::setFeatureRestrictions_Carplay Speed = %d" , m_speedvalue));
   if (SPEED_TRESHOLD < m_speedvalue && m_ActiveDeviceCategory == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_DIPO) //Bug = 579258
   {
      ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onSpeedStatus Speed is greater than Threshold"));
      m_carplayDriveModeRestriction = CARPLAY_DRIVERESTRICTED_SPEED;
      _smartphoneFIProxy->sendSetFeatureRestrictionsStart(*this, ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_DIPO,
            (::midw_smartphoneint_fi_types::T_FeatureRestriction)m_parkModeRestrictionInfo,
            (::midw_smartphoneint_fi_types::T_FeatureRestriction) m_carplayDriveModeRestriction, m_GeneralRestriction);
   }
   else
   {
      ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onSpeedStatus "));
   }
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onSpeedStatus device category = %d", m_ActiveDeviceCategory));
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onSpeedStatus parkModeRestriction value = %d", m_parkModeRestrictionInfo));
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onSpeedStatus DriveModeRestriction value = %d", m_carplayDriveModeRestriction));
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onSpeedStatus GeneralRestriction value = %d", m_GeneralRestriction));
}


//FUNCTION:		setFeatureRestrictionsAAP

void VehicleDataSpiClientHandler::setFeatureRestrictionsAAP()
{
   // ConnectionClientHandler::getInstance()->tGetActiveDeviceCategory();
   // ETG_TRACE_USR4(("ConnectionClientHandler::setFeatureRestrictionsAAP Device Category = %d", m_ActiveDeviceCategory));
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::setFeatureRestrictionsAAP - Validation"));

   // Set Park Mode Restrictions
   // if(m_ActiveDeviceCategory == 3)
   // {
   // ETG_TRACE_USR4(("VehicleDataSpiClientHandler::setFeatureRestrictionsAAP - IF condition"));
   // m_parkmode_decimal = AAP_DUMMY;
   // m_VehicleStateInfo = ::midw_smartphoneint_fi_types::T_e8_VehicleState__PARK_MODE;
   // }
   if ((m_ParkBrakeInfo == T_e8_ParkBrake__PARK_BRAKE_ENGAGED) && (m_speedvalue == SPEED_STATIONARY))
   {
      ETG_TRACE_USR4(("VehicleDataSpiClientHandler::setFeatureRestrictionsAAP() - Brake Connected & Speed is %d : PARK MODE Restriction Set -> 00000", m_speedvalue));
      m_parkmode_decimal = AAP_C_UNRESTRICTED;
   }
   else if ((m_ParkBrakeInfo == T_e8_ParkBrake__PARK_BRAKE_NOT_ENGAGED) && (m_speedvalue == SPEED_STATIONARY))
   {
      ETG_TRACE_USR4(("VehicleDataSpiClientHandler::setFeatureRestrictionsAAP() - Brake Not Connected & Speed is %d : PARK MODE Restriction Set -> 10010", m_speedvalue));
      m_parkmode_decimal =  AAP_C_VHSTATIONARY;
   }
   else if (m_speedvalue <= SPEED_TRESHOLD)
   {
      ETG_TRACE_USR4(("VehicleDataSpiClientHandler::setFeatureRestrictionsAAP() - Speed Lock Is Disabled & Speed is Less Than 8 : PARK MODE Restriction Set -> 10011"));
      m_parkmode_decimal = AAP_C_DRIVERESTRICTED_SLOFF;
   }
   else if ((m_ParkBrakeInfo == T_e8_ParkBrake__PARK_BRAKE_NOT_ENGAGED) && (m_speedvalue <= SPEED_TRESHOLD))
   {
      ETG_TRACE_USR4(("VehicleDataSpiClientHandler::setFeatureRestrictionsAAP() -  Brake Not Connected & Speed is Less Than 8 : PARK MODE Restriction Set -> 11011"));
      m_parkmode_decimal = AAP_C_DRIVERESTRICTED_SLON;
   }
   else
   {
      ETG_TRACE_USR4(("VehicleDataSpiClientHandler::setFeatureRestrictionsAAP() - Do Not Set Any PARK MODE Restrictions"));
   }
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::setFeatureRestrictionsAAP() - Decimal value is %d", m_parkmode_decimal));

   //Set Drive Mode Restrictions

   // if((m_speedlockvalue = SPEEDLOCK_OFF)&&( m_speedvalue  >= SPEED_TRESHOLD))
   // {
   // ETG_TRACE_USR4(("VehicleDataSpiClientHandler::setFeatureRestrictionsAAP() - Speed Lock Is Disabled & Speed is Greater Than 8 : DRIVE MODE Restriction Set -> 10011"));
   // m_drivemode_decimal = AAP_C_DRIVERESTRICTED_SLOFF;
   // }
   //else
   //{
   m_drivemode_decimal =  AAP_C_DRIVERESTRICTED_SLON;
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::setFeatureRestrictionsAAP()  - DRIVE MODE Restriction Set -> 11011"));
   //}
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::setFeatureRestrictionsAAP() Park FeatureLockout = %d , Drive FeatureLockout = %d", m_parkmode_decimal , m_drivemode_decimal));

// _smartphoneFIProxy->sendSetFeatureRestrictionsStart(*this,(::midw_smartphoneint_fi_types::T_e8_DeviceCategory)m_ActiveDeviceCategory,
   // (::midw_smartphoneint_fi_ types::T_FeatureRestriction)m_drivemode_decimal ,
   // (::midw_smartphoneint_fi_types::T_FeatureRestriction)m_parkmode_decimal );
   ::midw_smartphoneint_fi_types::T_FeatureRestriction parkModeRestrictionInfo ;
   parkModeRestrictionInfo.setFeatureLockout(m_parkmode_decimal);
   ::midw_smartphoneint_fi_types::T_FeatureRestriction driveModeRestrictionInfo ;
   driveModeRestrictionInfo.setFeatureLockout(m_drivemode_decimal);
   _smartphoneFIProxy->sendSetFeatureRestrictionsStart(*this, ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_ANDROIDAUTO, parkModeRestrictionInfo, driveModeRestrictionInfo, m_GeneralRestriction);
}


/************************************************************************
*FUNCTION:		onSetFeatureRestrictionsResult
*DESCRIPTION:	Function to set the Feature Restriction
*PARAMETER:     result
*RETURNVALUE:	void
************************************************************************/
void VehicleDataSpiClientHandler::onSetFeatureRestrictionsResult(const ::boost::shared_ptr<midw_smartphoneint_fi::Midw_smartphoneint_fiProxy>& proxy,
      const ::boost::shared_ptr<midw_smartphoneint_fi::SetFeatureRestrictionsResult>& result)
{
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onSetFeatureRestrictionsResult"));
}


/************************************************************************
*FUNCTION:		onSetFeatureRestrictionsError
*DESCRIPTION:	Function to get the Set Feature Restriction Error
*PARAMETER:     result
*RETURNVALUE:	void
************************************************************************/
void VehicleDataSpiClientHandler::onSetFeatureRestrictionsError(const ::boost::shared_ptr<midw_smartphoneint_fi::Midw_smartphoneint_fiProxy>& proxy,
      const ::boost::shared_ptr<midw_smartphoneint_fi::SetFeatureRestrictionsError>& error)
{
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onSetFeatureRestrictionsError"));
}


/************************************************************************
*FUNCTION:		setVehiclespeed
*DESCRIPTION:
*PARAMETER:     result
*RETURNVALUE:	void
************************************************************************/
void VehicleDataSpiClientHandler::setVehiclespeed(int32 m_speedvalue)
{
   ETG_TRACE_USR4((" onSetVehicleMovementStateResult...Received"));
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler:sendVehicleMechanicalSpeedStart speed in cm/sec = %d ", m_speedvalue));
   _smartphoneFIProxy->sendVehicleMechanicalSpeedStart(*this, m_speedvalue);
}


/************************************************************************
*FUNCTION:		onVehicleMechanicalSpeedResult
*DESCRIPTION:
*PARAMETER:     result
*RETURNVALUE:	void
************************************************************************/
void VehicleDataSpiClientHandler::onVehicleMechanicalSpeedResult(const ::boost::shared_ptr<midw_smartphoneint_fi::Midw_smartphoneint_fiProxy>& proxy,
      const ::boost::shared_ptr<midw_smartphoneint_fi::VehicleMechanicalSpeedResult>& result)
{
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onVehicleMechanicalSpeedResult"));
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onVehicleMechanicalSpeedResult ResponseCode = %d ", result->getResponseCode()));
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onVehicleMechanicalSpeedResult ErrorCode = %d ", result->getErrorCode()));
}


/************************************************************************
*FUNCTION:		onVehicleMechanicalSpeedError
*DESCRIPTION:
*PARAMETER:     result
*RETURNVALUE:	void
************************************************************************/
void VehicleDataSpiClientHandler::onVehicleMechanicalSpeedError(const ::boost::shared_ptr<midw_smartphoneint_fi::Midw_smartphoneint_fiProxy>& proxy,
      const ::boost::shared_ptr<midw_smartphoneint_fi::VehicleMechanicalSpeedError>& error)
{
   ETG_TRACE_USR4(("VehicleDataSpiClientHandler::onVehicleMechanicalSpeedError received"));
}


}
}//EOF
