/*********************************************************************************************************************************
 * @file         : SystemPowerModeClientHandler.cpp
 * @author       : Sundeep Parvathikar (RBEI/ECH2)
 * @addtogroup   : AppHmi_SPI
 * @brief        :
 * @copyright    : (c) 2019-2019 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 *28.05.2019 Rev. 0.1 Sundeep Parvathikar (RBEI/ECH2)  Initial Revision to handle System Power Mode.
 *********************************************************************************************************************************/

#include "hall_std_if.h"
#include "Core/SystemPowerModeClientHandler/SystemPowerModeClientHandler.h"
#include "Core/ConnectionClientHandler/ConnectionClientHandler.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_SPI_HALL
#define ETG_I_TRACE_CHANNEL TR_TTFIS_APPHMI_SPI
#define ETG_I_TTFIS_CMD_PREFIX "APPHMI_SPI_"
#define ETG_I_FILE_PREFIX App::Core::SystemPowerModeClientHandler::
#include "trcGenProj/Header/SystemPowerModeClientHandler.cpp.trc.h"
#endif

namespace App {
namespace Core {

/*************************************
| static data members initialization
*************************************/
SystemPowerModeClientHandler* SystemPowerModeClientHandler::_SystemPowerModeClientHandler = NULL;


/************************************************************************
*FUNCTION:		getInstance
*DESCRIPTION:	class instance create/return
*PARAMETER:     None
*RETURNVALUE:	pointer to the class object
*SWFL:          NA
*HISTORY:
*28.05.2019 Rev. 0.1 Sundeep Parvathikar (RBEI/ECH2)      Initial Version
************************************************************************/
SystemPowerModeClientHandler* SystemPowerModeClientHandler::getInstance()
{
   if (!_SystemPowerModeClientHandler)
   {
      _SystemPowerModeClientHandler = new SystemPowerModeClientHandler();
   }

   ETG_TRACE_USR4(("SystemPowerModeClientHandler::getInstance()"));
   return _SystemPowerModeClientHandler;
}


/************************************************************************
*FUNCTION:		SystemPowerModeClientHandler
*DESCRIPTION:	Constructor
*PARAMETER:     None
*RETURNVALUE:	NA
*SWFL:          NA
*HISTORY:
*28.05.2019 Rev. 0.1 Sundeep Parvathikar (RBEI/ECH2)      Initial Version
************************************************************************/
SystemPowerModeClientHandler::SystemPowerModeClientHandler() : _SPMSetproxy(::SPM_CORE_FI::SPM_CORE_FIProxy::createProxy("spmFiPort", *this))
{
   ETG_I_REGISTER_FILE();

   if (_SPMSetproxy.get())
   {
      ETG_TRACE_USR4(("SystemPowerModeClientHandler::SystemPowerModeClientHandler StartupSync _SPMSetproxy"));
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _SPMSetproxy->getPortName());
   }

   m_bStandByContextFlag = false;
   m_bSleepContextFlag = false;

   ETG_TRACE_USR4(("SystemPowerModeClientHandler::SystemPowerModeClientHandler()"));
}


/************************************************************************
*FUNCTION:		~SystemPowerModeClientHandler
*DESCRIPTION:	Destructor
*PARAMETER:     None
*RETURNVALUE:	NA
*SWFL:          NA
*HISTORY:
*28.05.2019 Rev. 0.1 Sundeep Parvathikar (RBEI/ECH2)      Initial Version
************************************************************************/
SystemPowerModeClientHandler::~SystemPowerModeClientHandler()
{
   ETG_TRACE_USR4(("SystemPowerModeClientHandler::~SystemPowerModeClientHandler()"));
   if (!_SystemPowerModeClientHandler)
   {
      delete _SystemPowerModeClientHandler;
   }
   _SystemPowerModeClientHandler = NULL;
   ETG_I_UNREGISTER_FILE();
}


/************************************************************************
*FUNCTION:		onUnavailable
*DESCRIPTION:	Method called when SPI service is unavailable
*PARAMETER:     proxy, stateChange
*RETURNVALUE:	NA
*SWFL:
*HISTORY:
*28.05.2019 Rev. 0.1 Sundeep Parvathikar (RBEI/ECH2)      Initial Version
************************************************************************/
void SystemPowerModeClientHandler::onUnavailable(const ::boost::shared_ptr<asf::core::Proxy>& proxy,
      const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("SystemPowerModeClientHandler::onUnavailable()"));
   if ((proxy == _SPMSetproxy) && (_SPMSetproxy.get()))
   {
      ETG_TRACE_USR4(("SystemPowerModeClientHandler::onUnavailable() if block"));
      _SPMSetproxy->sendHmiStateRelUpRegAll();
   }
   else
   {
      ETG_TRACE_USR4(("SystemPowerModeClientHandler::onUnavailable() else block"));
   }
}


/************************************************************************
*FUNCTION:		onAvailable
*DESCRIPTION:	Method called when SPI service is available
*PARAMETER:     proxy, stateChange
*RETURNVALUE:	NA
*SWFL:
*HISTORY:
*28.05.2019 Rev. 0.1 Sundeep Parvathikar (RBEI/ECH2)      Initial Version
************************************************************************/
void SystemPowerModeClientHandler::onAvailable(const ::boost::shared_ptr<asf::core::Proxy>& proxy,
      const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("SystemPowerModeClientHandler::onAvailable()"));
   if (proxy && (proxy == _SPMSetproxy) && (_SPMSetproxy.get()))
   {
      ETG_TRACE_USR4(("SystemPowerModeClientHandler::onAvailable() if block"));
      _SPMSetproxy->sendHmiStateUpReg(*this);
   }
}


/************************************************************************
*FUNCTION:      SystemPowerModeClientHandler - registerProperties
*DESCRIPTION:   SPM service property registration is handled.
*PARAMETER:     proxy,stateChange
*RETURNVALUE:   None
*HISTORY:
*28.05.2019 Rev. 0.1 Sundeep Parvathikar (RBEI/ECH2)      Initial Version
************************************************************************/
void SystemPowerModeClientHandler :: registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy,
      const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("SystemPowerModeClientHandler::registerProperties"));
   if (_SPMSetproxy && (proxy == _SPMSetproxy))
   {
      //_SPMSetproxy->sendHmiStateUpReg(*this);
      ETG_TRACE_USR4(("SystemPowerModeClientHandler::registerProperties() if block"));
   }
   else
   {
      ETG_TRACE_USR4(("ERROR: _SPMSetproxy is not received"));
   }
}


/************************************************************************
*FUNCTION:      SystemPowerModeClientHandler - deregisterProperties
*DESCRIPTION:   SPM service property deregistration is handled.
*PARAMETER:     proxy,stateChange
*RETURNVALUE:   None
*HISTORY:
*28.05.2019 Rev. 0.1 Sundeep Parvathikar (RBEI/ECH2)      Initial Version
************************************************************************/
void SystemPowerModeClientHandler :: deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy,
      const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("SystemPowerModeClientHandler::deregisterProperties"));
   if (_SPMSetproxy && (proxy == _SPMSetproxy))
   {
      // _SPMSetproxy->sendHmiStateRelUpRegAll();
      ETG_TRACE_USR4(("SystemPowerModeClientHandler::deregisterProperties() if block"));
   }
   else
   {
      ETG_TRACE_USR4(("ERROR: _SPMSetproxy is not received"));
   }
}


/************************************************************************
*FUNCTION:		onCourierMessage
*DESCRIPTION:	Function to process courier message
*PARAMETER:     StartupMsg
*RETURNVALUE:	bool
*SWFL:          NA
*HISTORY:
*28.05.2019 Rev. 0.1 Sundeep Parvathikar (RBEI/ECH2)      Initial Version
************************************************************************/
bool SystemPowerModeClientHandler::onCourierMessage(const Courier::StartupMsg& oMsg)
{
   ETG_TRACE_USR4(("SystemPowerModeClientHandler::onCourierMessage(const Courier::StartupMsg& oMsg)"));
   return true;
}


/************************************************************************
*FUNCTION:      onHmiStateError
*DESCRIPTION:   onHmiStateError method to intimate that the service has error
*PARAMETER:     proxy,error
*RETURNVALUE:   None
*28.05.2019 Rev. 0.1 Sundeep Parvathikar (RBEI/ECH2)      Initial Version
************************************************************************/
void SystemPowerModeClientHandler::onHmiStateError(const ::boost::shared_ptr< SPM_CORE_FI::SPM_CORE_FIProxy >& proxy,
      const boost::shared_ptr< ::SPM_CORE_FI::HmiStateError >& error)
{
   ETG_TRACE_ERR(("SystemPowerModeClientHandler:: onHmiStateError"));
}


/************************************************************************
*FUNCTION:      onHmiStateStatus
*DESCRIPTION:   onHmiStateStatus method to intimate that the service is available.
*PARAMETER:     proxy,status
*RETURNVALUE:   None
*31.05.2019 Rev. 0.1 Sundeep Parvathikar (RBEI/ECH2)      Initial Version
************************************************************************/
void SystemPowerModeClientHandler::onHmiStateStatus(const ::boost::shared_ptr< ::SPM_CORE_FI::SPM_CORE_FIProxy >& proxy,
      const boost::shared_ptr< ::SPM_CORE_FI::HmiStateStatus >& status)
{
   ETG_TRACE_USR4(("SystemPowerModeClientHandler::onHmiStateStatus() status->getHmiState() = %d", status->getHmiState()));

   /*
   	=============================== HMI State Mapping ===============================
   	*BACKGROUND / STANDBY / PREPARE_SHUTDOWN : T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_OFF
   	*IGNITION(User OFF)						 : T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_STANDBY
   	*ON										 : T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_HMI
   	*STATE_PROFILE (Hour Logic )			 : T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_HMI_HOUR_MODE
   	=============================== HMI State Mapping ===============================
   */

   tU8 u8currentAppId = DisplaySourceClientHandler::getInstance(ConnectionClientHandler::getInstance())->getCurrentAppID();
   switch (status->getHmiState())
   {
      case ::spm_core_fi_types::T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_OFF :
      {
         ETG_TRACE_USR4(("SystemPowerModeClientHandler::onHmiStateStatus() HMISTATE_SHOW_OFF"));

         if (m_bSleepContextFlag) // To unborrow sleep context before borrowing for StandBy Context
         {
            ETG_TRACE_USR4(("SystemPowerModeClientHandler::onHmiStateStatus() HMISTATE_SHOW_OFF if(m_bSleepContextFlag)"));
            AudioSourceClientHandler::getInstance(ConnectionClientHandler::getInstance())->vSetAccessoryAudioContext(false, ::midw_smartphoneint_fi_types::T_e8_AudioContext__SPI_AUDIO_SLEEP_MODE);
            DisplaySourceClientHandler::getInstance(ConnectionClientHandler::getInstance())->vsendCurrentAppID(::midw_smartphoneint_fi_types::T_e8_DisplayContext__DISPLAY_CONTEXT_SLEEP_MODE, false);

            m_bSleepContextFlag = false;
         }
         if (!m_bStandByContextFlag)
         {
            ETG_TRACE_USR4(("SystemPowerModeClientHandler::onHmiStateStatus() HMISTATE_SHOW_OFF if(!m_bStandByContextFlag)"));
            AudioSourceClientHandler::getInstance(ConnectionClientHandler::getInstance())->vSetAccessoryAudioContext(true, ::midw_smartphoneint_fi_types::T_e8_AudioContext__SPI_AUDIO_STANDBY_MODE);
            DisplaySourceClientHandler::getInstance(ConnectionClientHandler::getInstance())->vsendCurrentAppID(::midw_smartphoneint_fi_types::T_e8_DisplayContext__DISPLAY_CONTEXT_STANDBY_MODE, true);

            m_bStandByContextFlag = true;
         }
         break;
      }
      case ::spm_core_fi_types::T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_STANDBY :
      {
         ETG_TRACE_USR4(("SystemPowerModeClientHandler::onHmiStateStatus() HMISTATE_SHOW_STANDBY"));

         if (!m_bSleepContextFlag)
         {
            ETG_TRACE_USR4(("SystemPowerModeClientHandler::onHmiStateStatus() HMISTATE_SHOW_STANDBY if(!m_bSleepContextFlag)"));
            //Setting accessory display context for carplay device
            AudioSourceClientHandler::getInstance(ConnectionClientHandler::getInstance())->vSetAccessoryAudioContext(true, ::midw_smartphoneint_fi_types::T_e8_AudioContext__SPI_AUDIO_SLEEP_MODE);
            DisplaySourceClientHandler::getInstance(ConnectionClientHandler::getInstance())->vsendCurrentAppID(::midw_smartphoneint_fi_types::T_e8_DisplayContext__DISPLAY_CONTEXT_SLEEP_MODE, true);

            m_bSleepContextFlag = true; // To avoid sending false display context/ unborrow request on system startup.
         }
         break;
      }
      case ::spm_core_fi_types::T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_HMI :
      case ::spm_core_fi_types::T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_HMI_HOUR_MODE :
      {
         ETG_TRACE_USR4(("SystemPowerModeClientHandler::onHmiStateStatus() HMISTATE_SHOW_HMI or HMISTATE_SHOW_HMI_HOUR_MODE"));

         if (m_bSleepContextFlag)
         {
            ETG_TRACE_USR4(("SystemPowerModeClientHandler::onHmiStateStatus() HMISTATE_SHOW_HMI_HOUR_MODE if(m_bSleepContextFlag)"));
            AudioSourceClientHandler::getInstance(ConnectionClientHandler::getInstance())->vSetAccessoryAudioContext(false, ::midw_smartphoneint_fi_types::T_e8_AudioContext__SPI_AUDIO_SLEEP_MODE);
            DisplaySourceClientHandler::getInstance(ConnectionClientHandler::getInstance())->vsendCurrentAppID(::midw_smartphoneint_fi_types::T_e8_DisplayContext__DISPLAY_CONTEXT_SLEEP_MODE, false);

            m_bSleepContextFlag = false;
            if ((ConnectionClientHandler::getInstance())->tGetActiveDeviceCategory() == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_MYSPIN && u8currentAppId == SPI_APP_ID)
            {
               ConnectionClientHandler::getInstance()->AAPMySPinStandby();
            }
            else if ((ConnectionClientHandler::getInstance())->tGetActiveDeviceCategory() == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_ANDROIDAUTO)
            {
               ConnectionClientHandler::getInstance()->AAPMySPinStandby();
            }
         }
         if (m_bStandByContextFlag)
         {
            ETG_TRACE_USR4(("SystemPowerModeClientHandler::onHmiStateStatus() HMISTATE_SHOW_HMI_HOUR_MODE if(m_bStandByContextFlag)"));
            AudioSourceClientHandler::getInstance(ConnectionClientHandler::getInstance())->vSetAccessoryAudioContext(false, ::midw_smartphoneint_fi_types::T_e8_AudioContext__SPI_AUDIO_STANDBY_MODE);
            DisplaySourceClientHandler::getInstance(ConnectionClientHandler::getInstance())->vsendCurrentAppID(::midw_smartphoneint_fi_types::T_e8_DisplayContext__DISPLAY_CONTEXT_STANDBY_MODE, false);

            m_bStandByContextFlag = false;
         }
         break;
      }
      default :
      {
         ETG_TRACE_USR4(("SystemPowerModeClientHandler::onHmiStateStatus() default"));
         break;
      }
   };
   getSleepContextFlag();
}


//*FUNCTION:		getSleepContextFlag

bool SystemPowerModeClientHandler::getSleepContextFlag()
{
   ETG_TRACE_USR4(("SystemPowerModeClientHandler::getSleepContextFlag"));
   return m_bSleepContextFlag;
}


} // namespace Core
} // namespace App
