/**************************************************************************************
 * @file         : ContextSwitchClientHandler.cpp
 * @author       : Sundeep Parvathikar (RBEI/ECH2)
 * @addtogroup   : AppHmi_SPI
 * @brief        : Singleton class to handle Context Switch functionalities similar to ApplicationSwitchClientComponent with added functionalities
 * @copyright    : (c) 2019-2019 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 *14.03.19 Rev. 0.1 Sundeep Parvathikar (RBEI/ECH2)
 * 		  			Initial Revision.
 **************************************************************************************/


#include "ContextSwitchClientHandler.h"
#include "ProjectBaseExchangingTextures.h"
#include "hmi_trace_if.h"
#include "hall_std_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS         TR_CLASS_APPHMI_SPI_HALL
#define ETG_I_TRACE_CHANNEL             TR_TTFIS_APPHMI_SPI
#define ETG_I_TTFIS_CMD_PREFIX          "APPHMI_SPI_"
#define ETG_I_FILE_PREFIX               App::Core::ContextSwitchClientHandler::
#include "trcGenProj/Header/ContextSwitchClientHandler.cpp.trc.h"
#endif

/*****************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------*/
#define PORTNAME "applicationSwitchPort"

namespace App {
namespace Core {
ContextSwitchClientHandler* ContextSwitchClientHandler::_contextSwitch = NULL;

/****************************************************************
	private default constructor ContextSwitchClientHandler()
****************************************************************/
ContextSwitchClientHandler::ContextSwitchClientHandler()
   : m_poApplicationSwitchProxy(ApplicationSwitchProxy::createProxy(PORTNAME, *this))
{
   ETG_TRACE_USR4(("ContextSwitchClientHandler::ContextSwitchClientHandler()"));
}


/************************************************************************************************
	destructor  ~ContextSwitchClientHandler()
************************************************************************************************/
ContextSwitchClientHandler::~ContextSwitchClientHandler()
{
   ETG_TRACE_USR4(("ContextSwitchClientHandler::~ContextSwitchClientHandler()"));
   if (NULL != _contextSwitch)
   {
      delete _contextSwitch;
      _contextSwitch = NULL;
   }
}


/************************************************************************************************
	returns pointer to the instance of the ContextSwitchClientHandler class if memory is allocated
************************************************************************************************/
ContextSwitchClientHandler* ContextSwitchClientHandler::poGetInstance()
{
   /* No more than one instance of ContextSwitchClientHandler should be created */
   ETG_TRACE_USR4(("ContextSwitchClientHandler::poGetInstance()"));
   if (NULL == _contextSwitch)
   {
      _contextSwitch =  new ContextSwitchClientHandler();
   }
   return _contextSwitch;
}


void ContextSwitchClientHandler::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("ContextSwitchClientHandler::onAvailable()"));
   if (m_poApplicationSwitchProxy && m_poApplicationSwitchProxy == proxy)
   {
      ETG_TRACE_USR4(("ContextSwitchClientHandler::onAvailable() IF"));
      m_poApplicationSwitchProxy->sendActiveRegionIdRegister(*this);
      m_poApplicationSwitchProxy->sendSig_ActivateContextRegister(*this);
   }
}


void ContextSwitchClientHandler::onUnavailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("ContextSwitchClientHandler::onUnavailable()"));
   if (m_poApplicationSwitchProxy && m_poApplicationSwitchProxy == proxy)
   {
      ETG_TRACE_USR4(("ContextSwitchClientHandler::onUnavailable() IF"));
      m_poApplicationSwitchProxy->sendActiveRegionIdDeregisterAll();
      m_poApplicationSwitchProxy->sendSig_ActivateContextDeregisterAll();
   }
}


/*
* Description     : Virtual function implemented to get update of onSig_ActivateContextError Error
*
* @param[in]      : proxy: the client side representation of the ContextSwitch IF
* @param[in]      : error : The error message of 'onSig_ActivateContextError'
* @return         : void
*/
void ContextSwitchClientHandler::onSig_ActivateContextError(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< Sig_ActivateContextError >& error)
{
   ETG_TRACE_USR4(("ContextSwitchClientHandler::onSig_ActivateContextError"));
}


void ContextSwitchClientHandler::onRequestContextSwitchError(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< RequestContextSwitchError >& error)
{
   ETG_TRACE_USR4(("ContextSwitchClientHandler::onRequestContextSwitchError"));
}


void ContextSwitchClientHandler::onRequestContextSwitchResponse(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< RequestContextSwitchResponse >& response)
{
   ETG_TRACE_USR4(("ContextSwitchClientHandler::onRequestContextSwitchResponse()"));
}


void ContextSwitchClientHandler::onSendContextError(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< SendContextError >& error)
{
   ETG_TRACE_USR4(("ContextSwitchClientHandler::onSendContextError()"));
}


void ContextSwitchClientHandler::onSendContextResponse(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< SendContextResponse >& response)
{
   ETG_TRACE_USR4(("ContextSwitchClientHandler::onSendContextResponse()"));
}


/************************************************************************
*FUNCTION		: onSig_ActivateContextSignal
*DESCRIPTION	:
*PARAMETER		: NA
*PARAMETER		: NA
*RETURNVALUE	: void
*SWFL			:
************************************************************************/
void ContextSwitchClientHandler::onSig_ActivateContextSignal(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< Sig_ActivateContextSignal >& signal)
{
   ETG_TRACE_USR4(("ContextSwitchClientHandler::onSig_ActivateContextSignal %d", signal->getTargetactivityId()));
}


bool ContextSwitchClientHandler::onCourierMessage(const Courier::StartupMsg& oMsg)
{
   ETG_TRACE_USR4(("ContextSwitchClientHandler::onCourierMessage(const Courier::StartupMsg& oMsg)"));
   return true;
}


void ContextSwitchClientHandler::onActiveRegionIdError(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy,
      const ::boost::shared_ptr< ActiveRegionIdError >& error)
{
   ETG_TRACE_USR4(("ContextSwitchClientHandler::onActiveRegionIdError()"));
}


void ContextSwitchClientHandler::onActiveRegionIdUpdate(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy,
      const ::boost::shared_ptr< ActiveRegionIdUpdate >& update)
{
   ETG_TRACE_USR4(("ContextSwitchClientHandler::onActiveRegionIdUpdate, update->getActiveRegionId(): %d", update->getActiveRegionId()));
}


/************************************************************************
*FUNCTION:		onCourierMessage
*DESCRIPTION:	Entry to the SPI Screen from Home Screen
*PARAMETER:		SPIEntryMsg&
*RETURNVALUE:	bool
*SWFL:			SWFL-2604, SWFL-2603, SWFL-2602, SWFL-2575 SWFL-2574, SWFL-2572
*HISTORY:
*0.1     14.03.19      Sundeep Parvathikar   Initial Version
************************************************************************/
bool ContextSwitchClientHandler::onCourierMessage(const SPIEntryMsg& oMsg)
{
   ETG_TRACE_USR4(("ContextSwitchClientHandler::SPIEntryMsg()"));

   APPLICATIONSWITCH_TYPES::enActivityIDs targetActivityId = enActivityIDs__eActivityID_SPI_NOPHONECONNECTED;
   if (ConnectionClientHandler::getInstance() != NULL)
   {
      ETG_TRACE_USR4(("ContextSwitchClientHandler::SPIEntryMsg() ConnectionClientHandler instance available"));
      if (
         (ConnectionClientHandler::getInstance()->tGetDeviceConnectionStatus() == ::midw_smartphoneint_fi_types::T_e8_DeviceConnectionStatus__DEV_CONNECTED) && //value="1"
         ((ConnectionClientHandler::getInstance()->tGetActiveDeviceCategory() == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_MYSPIN)	||
          (ConnectionClientHandler::getInstance()->tGetActiveDeviceCategory() == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_DIPO) ||
          (ConnectionClientHandler::getInstance()->tGetActiveDeviceCategory() == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_ANDROIDAUTO))		 && //value="4"
         (ConnectionClientHandler::getInstance()->tGetSessionStatus() == ::midw_smartphoneint_fi_types::T_e8_SessionStatus__SESSION_ACTIVE)					   //value="2"
      )//Entry into mySPIN_AppView SWFL-2581, SWFL-2604
      {
         ETG_TRACE_USR4(("ContextSwitchClientHandler::SPIEntryMsg() mySPIN_AppView"));
         targetActivityId = enActivityIDs__eActivityID_SPI_FULL_SCREEN;
      }
      else//Entry into Phone_not_connected	SWFL-2572
      {
         ETG_TRACE_USR4(("ContextSwitchClientHandler::SPIEntryMsg() Phone_not_connected"));
         targetActivityId = enActivityIDs__eActivityID_SPI_NOPHONECONNECTED;
      }
   }
   else
   {
      ETG_TRACE_USR4(("ContextSwitchClientHandler::SPIEntryMsg() ConnectionClientHandler instance un-available"));
   }
   //Switch to SPI's screen request
   requestContextSwitchApplication(
      (uint8)2 //regionId
      , (uint8)APPID_APPHMI_MASTER //sourceAppId
      , (APPLICATIONSWITCH_TYPES::enActivityIDs)APPMGR_C_ACTIVITY_ID_HOME_DEFAULT //sourceActivityId
      , (uint8)APPID_APPHMI_SPI //targetAppId
      , targetActivityId
   );
   return true;
}


/************************************************************************
*FUNCTION:		requestContextSwitchApplication
*DESCRIPTION:	When switch in context required to other HMI applications
*PARAMETER:		uint8, uint8, APPLICATIONSWITCH_TYPES::enActivityIDs, uint8, APPLICATIONSWITCH_TYPES::enActivityIDs
*RETURNVALUE:	void
*SWFL:
*HISTORY:
*0.1     14.03.19      Sundeep Parvathikar   Initial Version
************************************************************************/
void ContextSwitchClientHandler::requestContextSwitchApplication(
   uint8  regionId,
   uint8  sourceAppId,
   APPLICATIONSWITCH_TYPES::enActivityIDs  sourceActivityId,
   uint8  targetAppId,
   APPLICATIONSWITCH_TYPES::enActivityIDs  targetActivityId)
{
   ETG_TRACE_USR4(("ContextSwitchClientHandler::requestContextSwitchApplication"));
   if (m_poApplicationSwitchProxy)
   {
      m_poApplicationSwitchProxy->sendRequestContextSwitchRequest(*this, regionId, sourceAppId, sourceActivityId, targetAppId, targetActivityId);
   }
}


/************************************************************************
*FUNCTION:			requestSendContext
*DESCRIPTION:
*PARAMETER:
*RETURNVALUE:
*SWFL:
*HISTORY:
*0.1     14.03.19      Sundeep Parvathikar   Initial Version
************************************************************************/
void ContextSwitchClientHandler::requestSendContext(uint8  regionId, uint8  sourceAppId, APPLICATIONSWITCH_TYPES::enActivityIDs  sourceActivityId)
{
   ETG_TRACE_USR4(("ContextSwitchClientHandler::requestSendContext"));
   if (m_poApplicationSwitchProxy)
   {
      m_poApplicationSwitchProxy->sendSendContextRequest(*this, regionId, sourceAppId, sourceActivityId);
   }
}


/************************************************************************
*FUNCTION:		onCourierMessage
*DESCRIPTION:	Updates the master with current activity ID
*PARAMETER:		ViewSPIActivityId&
*RETURNVALUE:	bool
*SWFL:
*HISTORY:
*0.1     27.03.19      Sundeep Parvathikar   Initial Version
************************************************************************/
bool ContextSwitchClientHandler::onCourierMessage(const ViewSPIActivityId& oMsg)
{
   ETG_TRACE_USR4(("ContextSwitchClientHandler::ViewSPIActivityId() m_CurrentView = %d", m_CurrentView));

   requestSendContext((uint8)2 , (uint8)APPID_APPHMI_SPI, ContextSwitchClientHandler::poGetInstance()->getActivityIDforCurrentView());
   return true;
}


/************************************************************************
*FUNCTION:		getActivityIDforCurrentView
*DESCRIPTION:	Updates the master with current activity ID
*PARAMETER:		void
*RETURNVALUE:	bool
*SWFL:			General
*HISTORY:
*0.1     27.03.19      Sundeep Parvathikar   Initial Version
************************************************************************/
APPLICATIONSWITCH_TYPES::enActivityIDs ContextSwitchClientHandler::getActivityIDforCurrentView(void)
{
   APPLICATIONSWITCH_TYPES::enActivityIDs  activityID = (APPLICATIONSWITCH_TYPES::enActivityIDs)enActivityIDs__eActivityID_SPI_NOPHONECONNECTED;
   switch (m_CurrentView)
   {
      case VIEW_MYSPIN_APPVIEW:
      case VIEW_S_FULLSCREEN:
      {
         activityID = (APPLICATIONSWITCH_TYPES::enActivityIDs)enActivityIDs__eActivityID_SPI_FULL_SCREEN;
         ETG_TRACE_USR4(("ContextSwitchClientHandler::getActivityIDforCurrentView() activityID = %d L = %d", activityID, __LINE__));
         break;
      }
      case VIEW_PHONE_NOT_CONNECTED:
      {
         activityID = (APPLICATIONSWITCH_TYPES::enActivityIDs)enActivityIDs__eActivityID_SPI_NOPHONECONNECTED;
         ETG_TRACE_USR4(("ContextSwitchClientHandler::getActivityIDforCurrentView() activityID = %d L = %d", activityID, __LINE__));
         break;
      }
      default :
      {
         ETG_TRACE_USR4(("ContextSwitchClientHandler::getActivityIDforCurrentView() default case entered"));
         break;
      }
   }

   return activityID;
}


}//namespace Core
}//namespace App
