/**************************************************************************************
 * @file         : AudioSourceClientHandler.cpp
 * @author       : Athira K Sreekumar
 * @addtogroup   : AppHmi_SPI
 * @brief        :
 * @copyright    : (c) 2019-2019 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 *20.04.18 Rev. 0.1 Athira K Sreekumar (RBEI/ECG2)
 * 		  			Initial Revision.
 *12.02.19 Rev. 0.2 Bruce Netto (RBEI/ECH2)
 *                  Optimizations and error fixes to the client handler file
 **************************************************************************************/
#include "hall_std_if.h"
#include "Core/AudioSourceClientHandler/AudioSourceClientHandler.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_SPI_HALL
#define ETG_I_TRACE_CHANNEL TR_TTFIS_APPHMI_SPI
#define ETG_I_TTFIS_CMD_PREFIX "APPHMI_SPI_"
#define ETG_I_FILE_PREFIX App::Core::AudioSourceClientHandler::
#include "trcGenProj/Header/AudioSourceClientHandler.cpp.trc.h"
#endif

using namespace ::asf::core;

using namespace MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange;
using namespace MASTERAUDIOSERVICE_INTERFACE::SoundProperties;
using namespace ::midw_smartphoneint_fi;

namespace App {
namespace Core {
/*****************************************************************
| static data implementation
|----------------------------------------------------------------*/
AudioSourceClientHandler* AudioSourceClientHandler::_audioSourceClientHandler = NULL;

/************************************************************************
*FUNCTION:		AudioSourceClientHandler
*DESCRIPTION:	Constructor
*PARAMETER:     None
*RETURNVALUE:	NA
*SWFL:
*HISTORY:
*0.1     04.06.18      Athira Sreekumar       Initial Version
************************************************************************/
AudioSourceClientHandler::AudioSourceClientHandler(ConnectionClientHandler* pConnectionClientHandler)
   : _smartphoneFIProxy(Midw_smartphoneint_fiProxy::createProxy("midwspiPort", *this))
   , _audioSourceChangeProxy(AudioSourceChangeProxy::createProxy("audioSourceChangePort", *this))
   , _soundPropertiesProxy(SoundPropertiesProxy::createProxy("soundPropertiesPort", *this))
   , _connectionClientHandler(pConnectionClientHandler)
{
   ETG_TRACE_USR4(("AudioSourceClientHandler::AudioSourceClientHandler()"));
   if (_smartphoneFIProxy.get())
   {
      ETG_TRACE_USR4(("AudioSourceClientHandler::AudioSourceClientHandler StartupSync _smartphoneFIProxy"));
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _smartphoneFIProxy->getPortName());
   }

   if (_audioSourceChangeProxy.get())
   {
      ETG_TRACE_USR4(("AudioSourceClientHandler::AudioSourceClientHandler StartupSync _audioSourceChangeProxy"));
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _audioSourceChangeProxy->getPortName());
   }

   if (_soundPropertiesProxy.get())
   {
      ETG_TRACE_USR4(("AudioSourceClientHandler::AudioSourceClientHandler StartupSync _soundPropertiesProxy"));
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _soundPropertiesProxy->getPortName());
   }
   ETG_I_REGISTER_FILE();
   vInitialize();
}


/************************************************************************
*FUNCTION:		~AudioSourceClientHandler
*DESCRIPTION:	Destructor
*PARAMETER:     None
*RETURNVALUE:	NA
*SWFL:
*HISTORY:
*0.1     04.06.18      Athira Sreekumar       Initial Version
************************************************************************/
AudioSourceClientHandler::~AudioSourceClientHandler()
{
   ETG_TRACE_USR4(("AudioSourceClientHandler::~AudioSourceClientHandler()"));
   if (_audioSourceClientHandler)
   {
      delete _audioSourceClientHandler;
   }
   _audioSourceClientHandler = NULL;
   ETG_I_UNREGISTER_FILE();
}


/************************************************************************
*FUNCTION:		onCourierMessage
*DESCRIPTION:	Function to process courier message
*PARAMETER:     StartupMsg
*RETURNVALUE:	bool
*SWFL:          NA
*HISTORY:
*0.1     12.02.19      Bruce Netto       Initial Version
************************************************************************/
bool AudioSourceClientHandler::onCourierMessage(const Courier::StartupMsg& oMsg)
{
   return true;
}


/************************************************************************
*FUNCTION:		deregisterProperties
*DESCRIPTION:	Trigger property deregistration
*PARAMETER:     proxy, stateChange
*RETURNVALUE:	NA
*SWFL:
*HISTORY:
*0.1     04.06.18      Athira Sreekumar       Initial Version
************************************************************************/
void AudioSourceClientHandler::deregisterProperties(const ::boost::shared_ptr<asf::core::Proxy>& proxy,
      const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("AudioSourceClientHandler::deregisterProperties()"));
}


/************************************************************************
*FUNCTION:		registerProperties
*DESCRIPTION:	Trigger property registration
*PARAMETER:     proxy, stateChange
*RETURNVALUE:	NA
*SWFL:
*HISTORY:
*0.1     04.06.18      Athira Sreekumar       Initial Version
*0.2	 17.12.2019    Shivani Priya          Audio Issue : Bug 579243, Bug 615993
************************************************************************/
void AudioSourceClientHandler::registerProperties(const ::boost::shared_ptr<asf::core::Proxy>& proxy,
      const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("AudioSourceClientHandler::registerProperties()"));
   if (_connectionClientHandler != NULL)
   {
      if (_connectionClientHandler->ismySPINEnabled()) //KDS CHECK
      {
         ETG_TRACE_USR4(("AudioSourceClientHandler::registerProperties() : Sending AccessoryAudioContext value at startup"));
         vSetAccessoryAudioContext(true,
                                   ::midw_smartphoneint_fi_types::T_e8_AudioContext__SPI_AUDIO_MAIN);
      }
   }
}


/************************************************************************
*FUNCTION:		onUnavailable
*DESCRIPTION:	Method called when SPI service is unavailable
*PARAMETER:     proxy, stateChange
*RETURNVALUE:	NA
*SWFL:
*HISTORY:
*0.1     04.06.18      Athira Sreekumar       Initial Version
************************************************************************/
void AudioSourceClientHandler::onUnavailable(const ::boost::shared_ptr<asf::core::Proxy>& proxy,
      const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("AudioSourceClientHandler::onUnavailable()"));
   if ((proxy == _smartphoneFIProxy) && (_smartphoneFIProxy.get()))
   {
      _smartphoneFIProxy->sendDeviceAudioContextRelUpRegAll();
   }

   else if ((proxy == _audioSourceChangeProxy) && (_audioSourceChangeProxy.get()))
   {
      _audioSourceChangeProxy->sendDeregisterAll();
   }

   else if ((proxy == _soundPropertiesProxy) && (_soundPropertiesProxy.get()))
   {
      //_soundPropertiesProxy->sendVolumeDeregisterAll();
      _soundPropertiesProxy->sendMuteStateDeregisterAll();
   }
}


/************************************************************************
*FUNCTION:		onAvailable
*DESCRIPTION:	Method called when SPI service is available
*PARAMETER:     proxy, stateChange
*RETURNVALUE:	NA
*SWFL:
*HISTORY:
*0.1     04.06.18      Athira Sreekumar       Initial Version
************************************************************************/
void AudioSourceClientHandler::onAvailable(const ::boost::shared_ptr<asf::core::Proxy>& proxy,
      const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("AudioSourceClientHandler::onAvailable()"));
   if ((proxy == _smartphoneFIProxy) && (_smartphoneFIProxy.get()))
   {
      _smartphoneFIProxy->sendDeviceAudioContextUpReg(*this);
   }

   else if ((proxy == _audioSourceChangeProxy) && (_audioSourceChangeProxy.get()))
   {
      _audioSourceChangeProxy->sendActiveSourceRegister(*this);
   }

   else if ((proxy == _soundPropertiesProxy) && (_soundPropertiesProxy.get()))
   {
      //_soundPropertiesProxy->sendVolumeRegister(*this);
      _soundPropertiesProxy->sendMuteStateRegister(*this);
   }
}


/************************************************************************
*FUNCTION:		vInitialize
*DESCRIPTION:	Initialization of member variables
*PARAMETER:     None
*RETURNVALUE:	NA
*SWFL:
*HISTORY:
*0.1     04.06.18      Athira Sreekumar       Initial Version
************************************************************************/
void AudioSourceClientHandler::vInitialize()
{
   ETG_TRACE_USR4(("AudioSourceClientHandler::vInitialize()"));
   m_CurrentSource = DEFAULT_SOURCE;
   m_CurrentAudioContext = ::midw_smartphoneint_fi_types::T_e8_AudioContext__SPI_AUDIO_MAIN;
   m_CurrentAudioFlag = false;
   m_MuteState = false;
}


/*********************************************************************************************
                                GETTERS AND SETTERS
*********************************************************************************************/
/************************************************************************
*FUNCTION:		vSetAudioFlag
*DESCRIPTION:	Function to set audio flag
*PARAMETER:     audioFlag
*RETURNVALUE:	NA
*SWFL:
*HISTORY:
*0.1     15.06.18      Athira Sreekumar       Initial Version
************************************************************************/
void AudioSourceClientHandler::vSetAudioFlag(bool audioFlag)
{
   ETG_TRACE_USR4(("AudioSourceClientHandler::vSetAudioFlag()"));
   ETG_TRACE_USR4(("AudioSourceClientHandler::vSetAudioFlag m_CurrentAudioFlag = %d", m_CurrentAudioFlag));
   m_CurrentAudioFlag = audioFlag;
   ETG_TRACE_USR4(("AudioSourceClientHandler::vSetAudioFlag m_CurrentAudioFlag = %d", m_CurrentAudioFlag));
}


/************************************************************************
*FUNCTION:		bGetAudioFlag
*DESCRIPTION:	Function to get audio flag
*PARAMETER:     NA
*RETURNVALUE:	bool
*SWFL:
*HISTORY:
*0.1     15.06.18      Athira Sreekumar       Initial Version
************************************************************************/
bool AudioSourceClientHandler::bGetAudioFlag()
{
   ETG_TRACE_USR4(("AudioSourceClientHandler::tGetAudioFlag()"));
   ETG_TRACE_USR4(("AudioSourceClientHandler::tGetAudioFlag m_CurrentAudioFlag = %d", m_CurrentAudioFlag));
   return m_CurrentAudioFlag;
}


/************************************************************************
*FUNCTION:		vSetAudioContext
*DESCRIPTION:	Function to set audio context
*PARAMETER:     audioFlag, audioContext
*RETURNVALUE:	NA
*SWFL:
*HISTORY:
*0.1     15.06.18      Athira Sreekumar       Initial Version
************************************************************************/
void AudioSourceClientHandler::vSetAudioContext(bool audioFlag,
      ::midw_smartphoneint_fi_types::T_e8_AudioContext audioContext)
{
   ETG_TRACE_USR4(("AudioSourceClientHandler::vSetAudioContext m_CurrentAudioContext = %d", m_CurrentAudioContext));
   if (audioFlag)
   {
      m_CurrentAudioContext = audioContext;
   }
   else
   {
      m_CurrentAudioContext = ::midw_smartphoneint_fi_types::T_e8_AudioContext__SPI_AUDIO_MAIN;
   }
   ETG_TRACE_USR4(("AudioSourceClientHandler::vSetAudioContext m_CurrentAudioContext = %d", m_CurrentAudioContext));
}


/************************************************************************
*FUNCTION:		tGetAudioContext
*DESCRIPTION:	Function to get audio context
*PARAMETER:     NA
*RETURNVALUE:	midw_smartphoneint_fi_types::T_e8_AudioContext
*SWFL:
*HISTORY:
*0.1     15.06.18      Athira Sreekumar       Initial Version
************************************************************************/
midw_smartphoneint_fi_types::T_e8_AudioContext AudioSourceClientHandler::tGetAudioContext()
{
   ETG_TRACE_USR4(("AudioSourceClientHandler::tGetAudioContext()"));
   ETG_TRACE_USR4(("AudioSourceClientHandler::tGetAudioContext m_CurrentAudioContext = %d", m_CurrentAudioContext));
   return m_CurrentAudioContext;
}


/************************************************************************
*FUNCTION:		vSetCurrentSource
*DESCRIPTION:	Function to set audio source
*PARAMETER:     audioSource
*RETURNVALUE:	NA
*SWFL:
*HISTORY:
*0.1     15.06.18      Athira Sreekumar       Initial Version
************************************************************************/
void AudioSourceClientHandler::vSetCurrentSource(uint32 audioSource)
{
   ETG_TRACE_USR4(("AudioSourceClientHandler::vSetCurrentSource()"));
   ETG_TRACE_USR4(("AudioSourceClientHandler::vSetCurrentSource m_CurrentSource = %d", m_CurrentSource));
   m_CurrentSource = audioSource;
   ETG_TRACE_USR4(("AudioSourceClientHandler::vSetCurrentSource m_CurrentSource = %d", m_CurrentSource));
}


/************************************************************************
*FUNCTION:		tGetCurrentSource
*DESCRIPTION:	Function to get audio source
*PARAMETER:     NA
*RETURNVALUE:	uint32
*SWFL:
*HISTORY:
*0.1     15.06.18      Athira Sreekumar       Initial Version
************************************************************************/
uint32 AudioSourceClientHandler::tGetCurrentSource()
{
   ETG_TRACE_USR4(("AudioSourceClientHandler::tGetCurrentSource()"));
   ETG_TRACE_USR4(("AudioSourceClientHandler::tGetCurrentSource m_CurrentSource = %d", m_CurrentSource));
   return m_CurrentSource;
}


/************************************************************************
*FUNCTION:		vSetMuteState
*DESCRIPTION:	Function to set mute state
*PARAMETER:     muteState
*RETURNVALUE:	NA
*SWFL:
*HISTORY:
*0.1     15.06.18      Athira Sreekumar       Initial Version
************************************************************************/
void AudioSourceClientHandler::vSetMuteState(bool muteState)
{
   ETG_TRACE_USR4(("AudioSourceClientHandler::vSetMuteState()"));
   ETG_TRACE_USR4(("AudioSourceClientHandler::vSetMuteState m_MuteState = %d", m_MuteState));
   m_MuteState = muteState;
   ETG_TRACE_USR4(("AudioSourceClientHandler::vSetMuteState m_MuteState = %d", m_MuteState));
}


/************************************************************************
*FUNCTION:		bGetMuteState
*DESCRIPTION:	Function to get mute state
*PARAMETER:     NA
*RETURNVALUE:	bool
*SWFL:
*HISTORY:
*0.1     15.06.18      Athira Sreekumar       Initial Version
************************************************************************/
bool AudioSourceClientHandler::bGetMuteState()
{
   ETG_TRACE_USR4(("AudioSourceClientHandler::bGetMuteState()"));
   ETG_TRACE_USR4(("AudioSourceClientHandler::bGetMuteState m_MuteState = %d", m_MuteState));
   return m_MuteState;
}


/*********************************************************************************************
                                Property: DeviceAudioContext
*********************************************************************************************/
/************************************************************************
*FUNCTION:		onDeviceAudioContextStatus
*DESCRIPTION:	Function to get status of DeviceDisplayContext
*PARAMETER:     proxy, status
*RETURNVALUE:	NA
*SWFL:          SWFL-2590/ SWFL-2596
*HISTORY:
*0.1     05.06.18      Athira Sreekumar       Initial Version
************************************************************************/
void AudioSourceClientHandler::onDeviceAudioContextStatus(const ::boost::shared_ptr<midw_smartphoneint_fi::Midw_smartphoneint_fiProxy>& proxy,
      const ::boost::shared_ptr<midw_smartphoneint_fi::DeviceAudioContextStatus>& status)
{
   ETG_TRACE_USR4(("AudioSourceClientHandler::onDeviceAudioContextStatus()"));
   if (proxy == _smartphoneFIProxy)
   {
      ETG_TRACE_USR4(("AudioSourceClientHandler::onDeviceAudioContextStatus DeviceHandle = %d", status->getDeviceHandle()));
      ETG_TRACE_USR4(("AudioSourceClientHandler::onDeviceAudioContextStatus AudioFlag = %d", status->getAudioFlag()));
      ETG_TRACE_USR4(("AudioSourceClientHandler::onDeviceAudioContextStatus AudioContext = %d", status->getAudioContext()));
   }
}


/************************************************************************
*FUNCTION:		onDeviceAudioContextError
*DESCRIPTION:	Function to get error from DeviceDisplayContext
*PARAMETER:     proxy, error
*RETURNVALUE:	NA
*SWFL:          SWFL-2590/ SWFL-2596
*HISTORY:
*0.1     05.06.18      Athira Sreekumar       Initial Version
************************************************************************/
void AudioSourceClientHandler::onDeviceAudioContextError(const ::boost::shared_ptr<midw_smartphoneint_fi::Midw_smartphoneint_fiProxy>& proxy,
      const ::boost::shared_ptr<midw_smartphoneint_fi::DeviceAudioContextError>& error)
{
   ETG_TRACE_USR4(("AudioSourceClientHandler::onDeviceAudioContextError()"));
}


/*********************************************************************************************
                                Method: AccessoryAudioContext
*********************************************************************************************/
/************************************************************************
*FUNCTION:		onAccessoryAudioContextResult
*DESCRIPTION:	Function to get result of AccessoryAudioContext
*PARAMETER:     proxy, result
*RETURNVALUE:	NA
*SWFL:          SWFL-2590/ SWFL-2596
*HISTORY:
*0.1     05.06.18      Athira Sreekumar       Initial Version
************************************************************************/
//TODO: Deprecated. Use SetAccessoryAudioContext
void AudioSourceClientHandler::onAccessoryAudioContextResult(const ::boost::shared_ptr<midw_smartphoneint_fi::Midw_smartphoneint_fiProxy>& proxy,
      const ::boost::shared_ptr<midw_smartphoneint_fi::AccessoryAudioContextResult>& result)
{
   ETG_TRACE_USR4(("AudioSourceClientHandler::onAccessoryAudioContextResult()"));
   // ETG_TRACE_USR4(("AudioSourceClientHandler::onAccessoryAudioContextResult DeviceContext = %d", result->getAudioContext()));
   ETG_TRACE_USR4(("AudioSourceClientHandler::onAccessoryAudioContextResult ResponseCode = %d", result->getResponseCode()));
   ETG_TRACE_USR4(("AudioSourceClientHandler::onAccessoryAudioContextResult ErrorCode = %d", result->getErrorCode()));
}


/************************************************************************
*FUNCTION:		onAccessoryAudioContextError
*DESCRIPTION:	Function to get error from AccessoryAudioContext
*PARAMETER:     proxy, error
*RETURNVALUE:	NA
*SWFL:          SWFL-2590/ SWFL-2596
*HISTORY:
*0.1     05.06.18      Athira Sreekumar       Initial Version
************************************************************************/
void AudioSourceClientHandler::onAccessoryAudioContextError(const ::boost::shared_ptr<midw_smartphoneint_fi::Midw_smartphoneint_fiProxy>& proxy,
      const ::boost::shared_ptr<midw_smartphoneint_fi::AccessoryAudioContextError>& error)
{
   ETG_TRACE_USR4(("AudioSourceClientHandler::onAccessoryAudioContextError()"));
}


/************************************************************************
*FUNCTION:		vSetAccessoryAudioContext
*DESCRIPTION:	Function to set accessory audio context
*PARAMETER:     audioFlag, audioContext
*RETURNVALUE:	NA
*SWFL:          SWFL-2590/ SWFL-2596
*HISTORY:
*0.1     05.06.18      Athira Sreekumar       Initial Version
************************************************************************/
void AudioSourceClientHandler::vSetAccessoryAudioContext(bool audioFlag,
      ::midw_smartphoneint_fi_types::T_e8_AudioContext audioContext)
{
   ETG_TRACE_USR4(("AudioSourceClientHandler::vSetAccessoryAudioContext()"));
   ETG_TRACE_USR4(("AudioSourceClientHandler::vSetAccessoryAudioContext audioFlag = %d", audioFlag));
   ETG_TRACE_USR4(("AudioSourceClientHandler::vSetAccessoryAudioContext audioContext = %d", audioContext));

   if (_connectionClientHandler != NULL)
   {
      _smartphoneFIProxy->sendAccessoryAudioContextStart(*this,
            _connectionClientHandler->tGetActiveDeviceHandle(),
            audioFlag,
            audioContext);
   }
}


/*********************************************************************************************
                                Property: ActiveSource
*********************************************************************************************/
/************************************************************************
*FUNCTION:		onActiveSourceUpdate
*DESCRIPTION:	Function to get result of AccessoryAudioContext
*PARAMETER:     proxy, update
*RETURNVALUE:	NA
*SWFL:          SWFL-2590/ SWFL-2596
*HISTORY:
*0.1     05.06.18      Athira Sreekumar       Initial Version
************************************************************************/
void AudioSourceClientHandler::onActiveSourceUpdate(const ::boost::shared_ptr<MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::AudioSourceChangeProxy>& proxy,
      const ::boost::shared_ptr<MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::ActiveSourceUpdate>& update)
{
   ETG_TRACE_USR4(("AudioSourceClientHandler::onActiveSourceUpdate()"));
   uint32 l_ActiveSource = update->getActiveSource().getSrcId();
   ETG_TRACE_USR4(("AudioSourceClientHandler::onActiveSourceUpdate ActiveSource = %d", l_ActiveSource));

   if (bIsChangeAudioSource(l_ActiveSource))
   {
      bool l_AudioFlag = true;
      ::midw_smartphoneint_fi_types::T_e8_AudioContext l_AudioContext = ::midw_smartphoneint_fi_types::T_e8_AudioContext__SPI_AUDIO_MAIN;
      vSetCurrentSource(update->getActiveSource().getSrcId());
      vFindAccessoryAudioContext(l_AudioFlag, l_AudioContext); // preapration of audio context that has to be sent to SPI MeW
      vUpdateAccessoryAudioContext(l_AudioFlag, l_AudioContext); // proces
   }
}


/************************************************************************
*FUNCTION:		onAccessoryAudioContextError
*DESCRIPTION:	Function to get error from AccessoryAudioContext
*PARAMETER:     proxy, error
*RETURNVALUE:	NA
*SWFL:          SWFL-2590/ SWFL-2596
*HISTORY:
*0.1     05.06.18      Athira Sreekumar       Initial Version
************************************************************************/
void AudioSourceClientHandler::onActiveSourceError(const ::boost::shared_ptr<MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::AudioSourceChangeProxy>& proxy,
      const ::boost::shared_ptr<MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::ActiveSourceError>& error)
{
   ETG_TRACE_USR4(("AudioSourceClientHandler::onActiveSourceError()"));
}


/************************************************************************
*FUNCTION:		bIsChangeAudioSource
*DESCRIPTION:	Function to check if the source can be changed
*PARAMETER:     audioSource
*RETURNVALUE:	bool
*SWFL:          SWFL-2590/ SWFL-2596
*HISTORY:
*0.1     05.06.18      Athira Sreekumar       Initial Version
************************************************************************/
bool AudioSourceClientHandler::bIsChangeAudioSource(uint32 audioSource)
{
   ETG_TRACE_USR4(("AudioSourceClientHandler::bIsChangeAudioSource()"));

   bool l_IsChangeAudioSource = false;

   if ((audioSource != tGetCurrentSource())
         && (false == bIsMixedSourceActive(audioSource))
         && (false == bIsSPITempSourceActive(audioSource)))
   {
      l_IsChangeAudioSource = true;

      ETG_TRACE_USR4(("AudioSourceClientHandler::bIsChangeAudioSource SourceChanged = %d", l_IsChangeAudioSource));
   }

   return l_IsChangeAudioSource;
}


/************************************************************************
*FUNCTION:		bIsMixedSourceActive
*DESCRIPTION:	Function to mixed audio source is active
*PARAMETER:     audioSource
*RETURNVALUE:	bool
*SWFL:          SWFL-2590/ SWFL-2596
*HISTORY:
*0.1     05.06.18      Athira Sreekumar       Initial Version
************************************************************************/
bool AudioSourceClientHandler::bIsMixedSourceActive(uint32 audioSource)
{
   ETG_TRACE_USR4(("AudioSourceClientHandler::bIsMixedSourceActive()"));

   bool l_bMixedSourceActive = false;
   switch (audioSource) //Example Traffic Announcements or Navi Instructions, lowers main sound channel and increases Message Volume
   {
      case SRC_NAVI_SPEECH:
      case SRC_SPI_INFO:
      {
         l_bMixedSourceActive = true;
         break;
      }

      default:
      {
         l_bMixedSourceActive = false;
         break;
      }
   }
   ETG_TRACE_USR4(("AudioSourceClientHandler::bIsMixedSourceActive l_bMixedSourceActive = %d", l_bMixedSourceActive));
   return l_bMixedSourceActive;
}


/************************************************************************
*FUNCTION:		bIsSPITempSourceActive
*DESCRIPTION:	Function to SPI source is active
*PARAMETER:     audioSource
*RETURNVALUE:	bool
*SWFL:          SWFL-2590/ SWFL-2596
*HISTORY:
*0.1     05.06.18      Athira Sreekumar       Initial Version
************************************************************************/
bool AudioSourceClientHandler::bIsSPITempSourceActive(uint32 audioSource)
{
   ETG_TRACE_USR4(("AudioSourceClientHandler::bIsSPITempSourceActive()"));

   bool l_bIsSPITempSourceActive = false;

   switch (audioSource) //Example like an SMS alert from the SPI Phone side
   {
      case SRC_SPI_PHONE:
      case SRC_SPI_SPEECHRECOGNITION:
      case SRC_SPI_ALERT:
      {
         l_bIsSPITempSourceActive = true;
         break;
      }

      default:
      {
         l_bIsSPITempSourceActive = false;
         break;
      }
   }
   ETG_TRACE_USR4(("AudioSourceClientHandler::bIsSPITempSourceActive l_bIsSPITempSourceActive = %d", l_bIsSPITempSourceActive));
   return l_bIsSPITempSourceActive;
}


/************************************************************************
*FUNCTION:		vFindAccessoryAudioContext
*DESCRIPTION:	Function to set audio flag and audio context based on
*               current active audio source
*PARAMETER:     audioFlag, audioContext
*RETURNVALUE:	NA
*SWFL:          SWFL-2590/ SWFL-2596
*HISTORY:
*0.1     15.06.18      Athira Sreekumar       Initial Version
************************************************************************/
void AudioSourceClientHandler::vFindAccessoryAudioContext(bool& audioFlag, ::midw_smartphoneint_fi_types::T_e8_AudioContext& audioContext)
{
   ETG_TRACE_USR4(("AudioSourceClientHandler::vFindAccessoryAudioContext()"));

   uint32 l_CurrentSource = tGetCurrentSource(); //return m_CurrentSource
   ETG_TRACE_USR4(("AudioSourceClientHandler::vFindAccessoryAudioContext audioFlag = %d", audioFlag));
   ETG_TRACE_USR4(("AudioSourceClientHandler::vFindAccessoryAudioContext audioContext = %d", audioContext));
   ETG_TRACE_USR4(("AudioSourceClientHandler::vFindAccessoryAudioContext m_CurrentSource = %d", l_CurrentSource));

   switch (l_CurrentSource)
   {
      case SRC_SPI_ENTERTAIN:
      case SRC_SPI_PHONE:
      case SRC_SPI_SPEECHRECOGNITION:
      case SRC_SPI_ALERT:
      {
         audioFlag = false;
         audioContext = ::midw_smartphoneint_fi_types::T_e8_AudioContext__SPI_AUDIO_MAIN;
         break;
      }

      case SRC_TUNER_TA_FM:// XM SIRIUS
      case SRC_TUNER_TA_DAB:
      {
         audioFlag = true;
         audioContext = ::midw_smartphoneint_fi_types::T_e8_AudioContext__SPI_AUDIO_TRAFFIC;
         break;
      }

      case SRC_PHONE: //Actual phone call
      case SRC_PHONE_OUTBAND_RING:
         //case SRC_PHONE_INBAND_RING:
         //case SRC_TCU_SERVICE_CALL:
      {
         audioFlag = true;
         audioContext = ::midw_smartphoneint_fi_types::T_e8_AudioContext__SPI_AUDIO_PHONE;
         break;
      }

      case SRC_TCU_ECALL:
      {
         audioFlag = true;
         audioContext = ::midw_smartphoneint_fi_types::T_e8_AudioContext__SPI_AUDIO_EMER_PHONE;
         break;
      }

      case SRC_PHONE_SPEECHRECOGNITION:
      case SRC_SMS_SPEECH:
      case SRC_SDS_SPEECHRECOGNITION:
      {
         audioFlag = true;
         audioContext = ::midw_smartphoneint_fi_types::T_e8_AudioContext__SPI_AUDIO_SPEECH_REC;
         break;
      }

      case SRC_TUNER_FM://Future Sources for tuner: SRC_TUNER_WX WX Weather
      case SRC_TUNER_AM:
      case SRC_DAB:
      case SRC_TUNER_XM:
      case SRC_MEDIA_AUX:
      case SRC_MEDIA_PLAYER:
      //case SRC_MEDIA_CDDA:
      case SRC_PHONE_BTAUDIO: // BT music play
      {
         audioFlag = true;
         audioContext = ::midw_smartphoneint_fi_types::T_e8_AudioContext__SPI_AUDIO_MAIN;
         break;
      }

      case SRC_MEDIA_NO_SRC:
      case SRC_INVALID:
      default:
      {
         ETG_TRACE_USR4(("AudioSourceClientHandler::vFindAccessoryAudioContext Set default audioContext"));
         break;
      }
   }
   ETG_TRACE_USR4(("AudioSourceClientHandler::vFindAccessoryAudioContext audioFlag = %d", audioFlag));
   ETG_TRACE_USR4(("AudioSourceClientHandler::vFindAccessoryAudioContext audioContext = %d", audioContext));
}


/************************************************************************
*FUNCTION:		vUpdateAccessoryAudioContext
*DESCRIPTION:	Function to check if audio source has changed and
*               update audio context accordingly
*PARAMETER:     audioFlag, audioContext
*RETURNVALUE:	NA
*SWFL:          SWFL-2590/ SWFL-2596
*HISTORY:
*0.1     15.06.18      Athira Sreekumar       Initial Version
************************************************************************/
void AudioSourceClientHandler::vUpdateAccessoryAudioContext(bool audioFlag, ::midw_smartphoneint_fi_types::T_e8_AudioContext audioContext)
{
   //We Reveice the latest flag and contest and compare them with the previously stored flag and context, if they are the different the new flag
   //and the new context are stored.
   ETG_TRACE_USR4(("AudioSourceClientHandler::vUpdateAccessoryAudioContext"));
   ::midw_smartphoneint_fi_types::T_e8_AudioContext l_CurrentAudioContext = tGetAudioContext();//return m_CurrentAudioContext GEtting prev Context
   bool l_CurrentAudioFlag = bGetAudioFlag();//return m_CurrentAudioFlag getting prev flag

   ETG_TRACE_USR4(("AudioSourceClientHandler::vUpdateAccessoryAudioContext m_CurrentAudioContext = %d", l_CurrentAudioContext));
   ETG_TRACE_USR4(("AudioSourceClientHandler::vUpdateAccessoryAudioContext audioContext = %d", audioContext));

   ETG_TRACE_USR4(("AudioSourceClientHandler::vUpdateAccessoryAudioContext m_CurrentAudioFlag = %d", l_CurrentAudioFlag));
   ETG_TRACE_USR4(("AudioSourceClientHandler::vUpdateAccessoryAudioContext audioFlag = %d", audioFlag));

   if (audioContext != l_CurrentAudioContext)//audioContext -> New Source //l_CurrentAudioContext -> Prev Source
   {
      //The Previous Context would be unborrowed and the new Context will be borrowed
      //Hence below if conditions would be run to change contexts simultaneously
      //Unborrow previous audio source
      //e.g., CarPlay music to native speech. Unborrow audio for CarPlay music
      if (bIsAudioBorrowContext(l_CurrentAudioContext))
      {
         ETG_TRACE_USR4(("AudioSourceClientHandler::vUpdateAccessoryAudioContext Unborrow current source"));
         vSetAccessoryAudioContext(false, l_CurrentAudioContext);// for borrowed Source, we need to unborrow it, this is re-sent for the prev source to free it
      }

      //audioContext is of borrow type. Send context
      //e.g., CarPlay music to native speech. Borrow audio for native speech
      if (bIsAudioBorrowContext(audioContext))
      {
         ETG_TRACE_USR4(("AudioSourceClientHandler::vUpdateAccessoryAudioContext audioContext is of borrow type"));
         vSetAccessoryAudioContext(audioFlag, audioContext);
      }

      //Previous audio source is  of 'take' type
      //e.g., CarPlay music to FM. Take audio for FM
      else if (!bIsAudioBorrowContext(l_CurrentAudioContext))
      {
         ETG_TRACE_USR4(("AudioSourceClientHandler::vUpdateAccessoryAudioContext m_CurrentAudioContext is of take type"));
         vSetAccessoryAudioContext(audioFlag, audioContext);
      }
   }

   //Same audio context. Unborrow.
   //e.g., Triggering and terminating CarPlay Siri when native FM is playing
   else if (audioFlag != l_CurrentAudioFlag)
   {
      ETG_TRACE_USR4(("AudioSourceClientHandler::vUpdateAccessoryAudioContext m_CurrentAudioContext == audioContext"))
      vSetAccessoryAudioContext(audioFlag, audioContext);
   }

   vSetAudioFlag(audioFlag);//setting the new flag
   vSetAudioContext(audioFlag, audioContext);//setting the new context
}


/************************************************************************
*FUNCTION:		bIsAudioBorrowContext
*DESCRIPTION:	Function to check if audio context is of borrow type
*PARAMETER:     audioContext
*RETURNVALUE:	NA
*SWFL:          SWFL-2590/ SWFL-2596
*HISTORY:
*0.1     15.06.18      Athira Sreekumar       Initial Version
************************************************************************/
bool AudioSourceClientHandler::bIsAudioBorrowContext(::midw_smartphoneint_fi_types::T_e8_AudioContext audioContext)
{
   ETG_TRACE_USR4(("AudioSourceClientHandler::bIsAudioBorrowContext()"));
   ETG_TRACE_USR4(("AudioSourceClientHandler::bIsAudioBorrowContext audioContext = %d", audioContext));
   bool l_AudioBorrowContext = false;

   switch (audioContext)
   {
      case ::midw_smartphoneint_fi_types::T_e8_AudioContext__SPI_AUDIO_SPEECH_REC:
      case ::midw_smartphoneint_fi_types::T_e8_AudioContext__SPI_AUDIO_ADVISOR_PHONE:
      case ::midw_smartphoneint_fi_types::T_e8_AudioContext__SPI_AUDIO_EMER_PHONE:
      case ::midw_smartphoneint_fi_types::T_e8_AudioContext__SPI_AUDIO_PHONE:
      case ::midw_smartphoneint_fi_types::T_e8_AudioContext__SPI_AUDIO_INCOM_TONE:
      case ::midw_smartphoneint_fi_types::T_e8_AudioContext__SPI_AUDIO_TRAFFIC:
      case ::midw_smartphoneint_fi_types::T_e8_AudioContext__SPI_AUDIO_EMERGENCY_MSG:
      case ::midw_smartphoneint_fi_types::T_e8_AudioContext__SPI_AUDIO_SYNC_MSG:
      case ::midw_smartphoneint_fi_types::T_e8_AudioContext__SPI_AUDIO_STANDBY_MODE:
      {
         l_AudioBorrowContext = true; //borrow type
         break;
      }

      default:
         l_AudioBorrowContext = false; //take type
         break;
   }

   ETG_TRACE_USR4(("AudioSourceClientHandler::bIsAudioBorrowContext l_AudioBorrowContext = %d", l_AudioBorrowContext));
   return l_AudioBorrowContext;
}


/*********************************************************************************************
                                Property: VolumeUpdate
*********************************************************************************************/
/************************************************************************
*FUNCTION:		onVolumeUpdate
*DESCRIPTION:	Function to get result of VolumeUpdate
*PARAMETER:     proxy, update
*RETURNVALUE:	NA
*SWFL:
*HISTORY:
*0.1     15.06.18      Athira Sreekumar       Initial Version
*0.2     12.02.19      Bruce Netto            Commented for future use if necessary
************************************************************************/
// void AudioSourceClientHandler::onVolumeUpdate(const ::boost::shared_ptr<MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SoundPropertiesProxy> &proxy,
//                                               const ::boost::shared_ptr<MASTERAUDIOSERVICE_INTERFACE::SoundProperties::VolumeUpdate> &update)
// {
//     ETG_TRACE_USR4(("AudioSourceClientHandler::onVolumeUpdate()"));
//     ETG_TRACE_USR4(("AudioSourceClientHandler::onVolumeUpdate Volume = %d", update->getVolume()));
//     ::midw_smartphoneint_fi_types::T_e8_AudioContext audioContext = ::midw_smartphoneint_fi_types::T_e8_AudioContext__SPI_AUDIO_MUTE;

//     if (0 == update->getVolume())
//     {

//     }

//     else
//     {

//     }
// }

/************************************************************************
*FUNCTION:		onVolumeError
*DESCRIPTION:	Function to get error from VolumeUpdate
*PARAMETER:     proxy, error
*RETURNVALUE:	NA
*SWFL:
*HISTORY:
*0.1     15.06.18      Athira Sreekumar       Initial Version
*0.2     12.02.19      Bruce Netto            Commented for future use if necessary
************************************************************************/
// void AudioSourceClientHandler::onVolumeError(const ::boost::shared_ptr<MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SoundPropertiesProxy> &proxy,
//                                              const ::boost::shared_ptr<MASTERAUDIOSERVICE_INTERFACE::SoundProperties::VolumeError> &error)
// {
//     ETG_TRACE_USR4(("AudioSourceClientHandler::onVolumeError()"));
// }

/*********************************************************************************************
                                Property: MuteStateUpdate
*********************************************************************************************/
/************************************************************************
*FUNCTION:		onMuteStateUpdate
*DESCRIPTION:	Function to get result of MuteUpdate
*PARAMETER:     proxy, update
*RETURNVALUE:	NA
*SWFL:          SWFL-2595
*HISTORY:
*0.1     15.06.18      Athira Sreekumar       Initial Version
************************************************************************/
void AudioSourceClientHandler::onMuteStateUpdate(const ::boost::shared_ptr<MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SoundPropertiesProxy>& proxy,
      const ::boost::shared_ptr<MASTERAUDIOSERVICE_INTERFACE::SoundProperties::MuteStateUpdate>& update)
{
   ETG_TRACE_USR4(("AudioSourceClientHandler::onMuteStateUpdate()"));

   //::midw_smartphoneint_fi_types::T_e8_AudioContext l_AudioContext = ::midw_smartphoneint_fi_types::T_e8_AudioContext__SPI_AUDIO_MUTE;

   bool l_MuteState = update->getMuteState();
   ETG_TRACE_USR4(("AudioSourceClientHandler::onMuteStateUpdate l_MuteState = %d", l_MuteState));

   if (l_MuteState)
   {
      if (false == bGetMuteState())
      {
         ETG_TRACE_USR4(("AudioSourceClientHandler::onMuteStateUpdate Mute"));
         vSetMuteState(true);
         vSetAccessoryAudioContext(true, ::midw_smartphoneint_fi_types::T_e8_AudioContext__SPI_AUDIO_MUTE);
      }
   }

   else
   {
      if (true == bGetMuteState())
      {
         ETG_TRACE_USR4(("AudioSourceClientHandler::onMuteStateUpdate Unmute"));
         vSetMuteState(false);
         vSetAccessoryAudioContext(false, ::midw_smartphoneint_fi_types::T_e8_AudioContext__SPI_AUDIO_MUTE);
      }
   }
}


/************************************************************************
*FUNCTION:		onMuteStateError
*DESCRIPTION:	Function to get error from MuteState
*PARAMETER:     proxy, error
*RETURNVALUE:	NA
*SWFL:          SWFL-2595
*HISTORY:
*0.1     15.06.18      Athira Sreekumar       Initial Version
************************************************************************/
void AudioSourceClientHandler::onMuteStateError(const ::boost::shared_ptr<MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SoundPropertiesProxy>& proxy,
      const ::boost::shared_ptr<MASTERAUDIOSERVICE_INTERFACE::SoundProperties::MuteStateError>& error)
{
   ETG_TRACE_USR4(("AudioSourceClientHandler::onMuteStateError()"));
}


} // namespace Core
} // namespace App
