/************************************************************************
*FILE:         AccessoryAppStateHandler.cpp
*SW-COMPONENT:    SPI Hall
*DESCRIPTION:  AccessoryAppStateHandler checks for the native app states
        and updates to CarPlay.
*AUTHORS:      ECG-vav6kor
*COPYRIGHT:    * Copyright (c) 2019-2019 Robert Bosch Car Multimedia GmbH
        * duplication and disclosure to third parties is prohibited.
*HISTORY:
*-----------------------------------------------------------------------
*03.10.16      Rev. 1.0 ECG-vav6kor
        * Initial Draft
*09.05.2017    Rev 1.1 Pratik Bhagwat
        * Send PPT SubState to SPM once Siri is activated and
          HMI is in user OFF mode.
*24.05.17      Rev. 1.2 Pratik Bhagwat
        * Implementation of Setter and Getter functions for
          Speech/Phone/Navigation Status
        * Implementation for AccessoryAppState to Update the
          app states to CarPlay whenever there is a change in
          Speech/Phone/Navigation Status.
*14.12.17      Rev. 1.3 Pratik Bhagwat
        * Implementation of ConflictsDetected Property from BTSettings
             to update BT service to proceed with CP in case of conflict
************************************************************************/

#include "Core/AccessoryAppStateHandler/AccessoryAppStateHandler.h"
#include "Core/DisplaySourceClientHandler/DisplaySourceClientHandler.h"
#include "hall_std_if.h"
//#include <string>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_SPI_HALL
#define ETG_I_TRACE_CHANNEL TR_TTFIS_APPHMI_SPI
#define ETG_I_TTFIS_CMD_PREFIX "APPHMI_SPI_"
#define ETG_I_FILE_PREFIX App::Core::AccessoryAppStateHandler::
#include "trcGenProj/Header/AccessoryAppStateHandler.cpp.trc.h"
#endif

using namespace ::asf::core;
using namespace ::midw_smartphoneint_fi;

namespace App {
namespace Core {
using namespace std;
AccessoryAppStateHandler* AccessoryAppStateHandler::_accessoryAppStateHandler = NULL;

/************************************************************************
*FUNCTION:		AccessoryAppStateHandler
*DESCRIPTION:	Constructor
*PARAMETER:     None
*RETURNVALUE:	NA
************************************************************************/
AccessoryAppStateHandler::AccessoryAppStateHandler(ConnectionClientHandler* pConnectionClientHandler)
   : _smartphoneFIProxy(Midw_smartphoneint_fiProxy::createProxy("midwspiPort", *this))
   , _spiTelephoneProxy(::MOST_Tel_FI::MOST_Tel_FIProxy::createProxy("btTelFiPort", *this))
   , _spiBTSetProxy(::MOST_BTSet_FI::MOST_BTSet_FIProxy::createProxy("btSetFiPort", *this))
   , _connectionClientHandler(pConnectionClientHandler)
{
   ETG_TRACE_USR4(("AccessoryAppStateHandler::AccessoryAppStateHandler()"));
   if (_smartphoneFIProxy.get())
   {
      ETG_TRACE_USR4(("AccessoryAppStateHandler::AccessoryAppStateHandler StartupSync _smartphoneFIProxy"));
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _smartphoneFIProxy->getPortName());
      StartupSync::getInstance().registerPropertyRegistrationIF(this, "btTelFiPort");
      ETG_TRACE_USR4(("Spi AccessoryAppStateHandler constructor"));
      setAccessoryAppState();
   }
   ETG_I_REGISTER_FILE();
   vInitialize();
}


/************************************************************************
*FUNCTION:		AccessoryAppStateHandler
*DESCRIPTION:	Destructor
*PARAMETER:     None
*RETURNVALUE:	NA
************************************************************************/
AccessoryAppStateHandler::~AccessoryAppStateHandler()
{
   ETG_TRACE_USR4(("Spi AccessoryAppStateHandler destructor"));
   _smartphoneFIProxy.reset();
   _spiTelephoneProxy.reset();
   _accessoryAppStateHandler = NULL;
   ETG_I_UNREGISTER_FILE();
}


/************************************************************************
*FUNCTION:		registerProperties
*DESCRIPTION:	Trigger property registration
*PARAMETER:     proxy, stateChange
*RETURNVALUE:	NA
************************************************************************/
void AccessoryAppStateHandler::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("AccessoryAppStateHandler::registerProperties()"));
}


/************************************************************************
*FUNCTION:		deregisterProperties
*DESCRIPTION:	Trigger property registration
*PARAMETER:     proxy, stateChange
*RETURNVALUE:	NA
************************************************************************/
void AccessoryAppStateHandler::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("AccessoryAppStateHandler::deregisterProperties()"));
}


/************************************************************************
*FUNCTION:		onAvailable
*DESCRIPTION:	Method called when SPI service is available
*PARAMETER:     proxy, stateChange
*RETURNVALUE:	NA
************************************************************************/
void AccessoryAppStateHandler::onAvailable(const ::boost::shared_ptr<asf::core::Proxy>& proxy,
      const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("AccessoryAppStateHandler::onAvailable()"));
   if ((proxy == _smartphoneFIProxy) && (_smartphoneFIProxy.get()))
   {
      _smartphoneFIProxy->sendDiPOAppStatusInfoUpReg(*this);
   }
   if (_spiTelephoneProxy && (proxy == _spiTelephoneProxy))
   {
      _spiTelephoneProxy->sendCallStatusNoticeUpReg(*this);
   }
   if (_spiBTSetProxy && (proxy == _spiBTSetProxy))
   {
      _spiBTSetProxy->sendConflictsDetectedUpReg(*this);
   }
}


/************************************************************************
*FUNCTION:		onUnavailable
*DESCRIPTION:	Method called when SPI service is unavailable
*PARAMETER:     proxy, stateChange
*RETURNVALUE:	NA
************************************************************************/
void AccessoryAppStateHandler::onUnavailable(const ::boost::shared_ptr<asf::core::Proxy>& proxy,
      const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("AccessoryAppStateHandler::onUnavailable()"));
   if ((proxy == _smartphoneFIProxy) && (_smartphoneFIProxy.get()))
   {
      _smartphoneFIProxy->sendDiPOAppStatusInfoRelUpRegAll();
   }
   if (_spiTelephoneProxy && (proxy == _spiTelephoneProxy))
   {
      _spiTelephoneProxy->sendCallStatusNoticeRelUpRegAll();
   }
   if (_spiBTSetProxy && (proxy == _spiBTSetProxy))
   {
      _spiBTSetProxy->sendConflictsDetectedRelUpRegAll();
   }
}


/************************************************************************
*FUNCTION:		onCourierMessage
*DESCRIPTION:	Function to process courier message
*PARAMETER:     StartupMsg
*RETURNVALUE:	bool
************************************************************************/
bool AccessoryAppStateHandler::onCourierMessage(const Courier::StartupMsg& oMsg)
{
   return true;
}


/************************************************************************
*FUNCTION:		vInitialize
*DESCRIPTION:	Initialization of member variables
*PARAMETER:     None
*RETURNVALUE:	void
************************************************************************/
void AccessoryAppStateHandler::vInitialize()
{
   ETG_TRACE_USR4(("AccessoryAppStateHandler::vInitialize()"));
   //DIPO_APP_STATE = {DIPO_APP_PHONE_DISABLE, DIPO_APP_SPEECH_DISABLE, 0};
   DIPO_APP_STATE.DipoAppPhoneState_Flag = DIPO_APP_PHONE_DISABLE;
   DIPO_APP_STATE.DipoAppSpeechState_Flag = DIPO_APP_SPEECH_DISABLE;
   DIPO_APP_STATE.isSPIState = 0;
   _mePhoneAppState = PhoneAppState__SPI_APP_STATE_PHONE_UNKNOWN;
   _meSpeechAppState = SpeechAppState__SPI_APP_STATE_SPEECH_UNKNOWN;
   _meNavigationAppState = NavigationAppState__SPI_APP_STATE_NAV_UNKNOWN;
   _meDIPOPhoneAppState = PhoneAppState__SPI_APP_STATE_PHONE_UNKNOWN;
   _meDIPOSpeechAppState = SpeechAppState__SPI_APP_STATE_SPEECH_UNKNOWN;
   _meDIPONaviAppState = NavigationAppState__SPI_APP_STATE_NAV_UNKNOWN;
   m_tU32ConflictTrigger = 0;
}


/************************************************************************
*FUNCTION:		getDIPOSpeechStatus
*DESCRIPTION:	This function to get the Current Siri speech status
*PARAMETER:     none
*RETURNVALUE:	_meDIPOSpeechAppState
************************************************************************/
enSpeechAppState AccessoryAppStateHandler::getDIPOSpeechStatus()
{
   return _meDIPOSpeechAppState;
}


/************************************************************************
*FUNCTION:		getDIPOPhoneStatus
*DESCRIPTION:	This function to get the Current DIPO phone status
*PARAMETER:     none
*RETURNVALUE:	_meDIPOPhoneAppState
************************************************************************/
enPhoneAppState AccessoryAppStateHandler::getDIPOPhoneStatus()
{
   return _meDIPOPhoneAppState;
}


/************************************************************************
*FUNCTION:		getDIPOSpeechStatus
*DESCRIPTION:	This function to get the Current DIPO navi status
*PARAMETER:     none
*RETURNVALUE:	_meDIPONaviAppState
************************************************************************/
enNavigationAppState AccessoryAppStateHandler::getDIPONaviStatus()
{
   return _meDIPONaviAppState;
}


/************************************************************************
*FUNCTION:		onDiPOAppStatusInfoStatus
*DESCRIPTION:	This function is called when the property update for DiPOAppStatusInfo Received
*PARAMETER:     proxy, status
*RETURNVALUE:	void


************************************************************************/
void AccessoryAppStateHandler::onDiPOAppStatusInfoStatus(const ::boost::shared_ptr< ::midw_smartphoneint_fi::Midw_smartphoneint_fiProxy >& proxy,
      const ::boost::shared_ptr< ::midw_smartphoneint_fi::DiPOAppStatusInfoStatus >& status)
{
   ETG_TRACE_USR4(("AccessoryAppStateHandler: onDiPOAppStatusInfoStatus Received ..."));

   if (proxy == _smartphoneFIProxy)
   {
      ETG_TRACE_USR4(("AccessoryAppStateHandler: CarplayphoneStatus Received  %d", _meDIPOPhoneAppState));
      if (_meDIPOPhoneAppState != enPhoneAppState(status->getAppStatePhone()))
      {
         _meDIPOPhoneAppState = enPhoneAppState(status->getAppStatePhone());
         if (_meDIPOPhoneAppState == PhoneAppState__SPI_APP_STATE_PHONE_ACTIVE)
         {
            ETG_TRACE_USR4(("AccessoryAppStateHandler: CarplayphoneStatus Received  when Active"));
            DIPO_APP_STATE.DipoAppPhoneState_Flag = DIPO_APP_PHONE_ENABLE;
         }
         else if (_meDIPOPhoneAppState == PhoneAppState__SPI_APP_STATE_PHONE_NOTACTIVE)
         {
            ETG_TRACE_USR4(("AccessoryAppStateHandler: CarplayphoneStatus Received when NotActive"));
            DIPO_APP_STATE.DipoAppPhoneState_Flag = DIPO_APP_PHONE_DISABLE;
         }
         else
         {
            ETG_TRACE_USR4(("AccessoryAppStateHandler: CarplayphoneStatus not Received "));
            DIPO_APP_STATE.DipoAppPhoneState_Flag = DIPO_APP_PHONE_DISABLE;
         }
      }
      if (_meDIPOSpeechAppState != enSpeechAppState(status->getAppStateSpeech()))
      {
         _meDIPOSpeechAppState = enSpeechAppState(status->getAppStateSpeech());
         ETG_TRACE_USR4(("AccessoryAppStateHandler: CarplaySpeechStatus Received  %d", _meDIPOSpeechAppState));

         if ((_meDIPOSpeechAppState == SpeechAppState__SPI_APP_STATE_SPEECH_SPEAKING) || (_meDIPOSpeechAppState == SpeechAppState__SPI_APP_STATE_SPEECH_RECOGNISING))
         {
            ETG_TRACE_USR4(("AccessoryAppStateHandler: CarplaySpeechStatus when siri is active "));
            ETG_TRACE_USR4(("AccessoryAppStateHandler: CarplaySpeechStatus when siri is active _meDIPOSpeechAppState = %d " , _meDIPOSpeechAppState));
            DIPO_APP_STATE.DipoAppSpeechState_Flag = DIPO_APP_SPEECH_ENABLE;
         }
         else if ((_meDIPOSpeechAppState == SpeechAppState__SPI_APP_STATE_SPEECH_END))
         {
            ETG_TRACE_USR4(("AccessoryAppStateHandler: CarplaySpeechStatus when siri is not active "));
            ETG_TRACE_USR4(("AccessoryAppStateHandler: CarplaySpeechStatus when siri is not active _meDIPOSpeechAppState = %d " , _meDIPOSpeechAppState));
            DIPO_APP_STATE.DipoAppSpeechState_Flag = DIPO_APP_SPEECH_DISABLE;
         }
         else
         {
            ETG_TRACE_USR4(("AccessoryAppStateHandler: CarplaySpeechStatus when siri is active "));
            DIPO_APP_STATE.DipoAppSpeechState_Flag = DIPO_APP_SPEECH_DISABLE;
         }
      }
      if ((DIPO_APP_STATE.isSPIState == false) &&
            ((DIPO_APP_STATE.DipoAppPhoneState_Flag == DIPO_APP_PHONE_ENABLE) ||
             (DIPO_APP_STATE.DipoAppSpeechState_Flag == DIPO_APP_SPEECH_ENABLE)))
      {
         ETG_TRACE_USR4(("AccessoryAppStateHandler: CarplaySpeechStatus requestContextSwitchApplication if "));
         ContextSwitchClientHandler::poGetInstance()->requestContextSwitchApplication((uint8)2
               , 0
               , (APPLICATIONSWITCH_TYPES::enActivityIDs)0
               , (uint8)APPID_APPHMI_SPI
               , (APPLICATIONSWITCH_TYPES::enActivityIDs)enActivityIDs__eActivityID_SPI_FULL_SCREEN);
         DIPO_APP_STATE.isSPIState = true; //Target is in Projection
      }
      else if ((DIPO_APP_STATE.isSPIState == true) &&
               ((DIPO_APP_STATE.DipoAppPhoneState_Flag == DIPO_APP_PHONE_DISABLE) ||
                (DIPO_APP_STATE.DipoAppSpeechState_Flag == DIPO_APP_SPEECH_DISABLE)))
      {
         ETG_TRACE_USR4(("AccessoryAppStateHandler: CarplaySpeechStatus requestContextSwitchApplication else "));
         ContextSwitchClientHandler::poGetInstance()->requestContextSwitchApplication((uint8)2
               , (uint8)APPID_APPHMI_SPI
               , (APPLICATIONSWITCH_TYPES::enActivityIDs)enActivityIDs__eActivityID_SPI_FULL_SCREEN
               , 0
               , (APPLICATIONSWITCH_TYPES::enActivityIDs)0);
         DIPO_APP_STATE.isSPIState = false;
      }

      if (_meDIPONaviAppState != enNavigationAppState(status->getAppStateNavigation()))
      {
         _meDIPONaviAppState = enNavigationAppState(status->getAppStateNavigation());
         ETG_TRACE_USR4(("AccessoryAppStateHandler: CarplayNaviStatus Received  %d", _meDIPONaviAppState));
      }
   }
}


/************************************************************************
*FUNCTION:		onDiPOAppStatusInfoError
*DESCRIPTION:	This function is called if DipoAppStatus property is updated with error
*PARAMETER:     proxy, status
*RETURNVALUE:	void
************************************************************************/
void AccessoryAppStateHandler::onDiPOAppStatusInfoError(const ::boost::shared_ptr< ::midw_smartphoneint_fi::Midw_smartphoneint_fiProxy >& proxy,
      const ::boost::shared_ptr< ::midw_smartphoneint_fi::DiPOAppStatusInfoError >& error)
{
   ETG_TRACE_USR4(("AccessoryAppStateHandler: onDiPOAppStatusInfoError Received ..."));
}


/************************************************************************
*FUNCTION:      setNativePhoneAppState
*DESCRIPTION:   Sets Native phone Status
*PARAMETER:     phoneAppState
*RETURNVALUE:   void
************************************************************************/
void AccessoryAppStateHandler::setNativePhoneAppState(enPhoneAppState phoneAppState)
{
   ETG_TRACE_USR4(("AccessoryAppStateHandler:setNativePhoneAppState Received phoneAppState = %d",  ETG_CENUM(enPhoneAppState, phoneAppState)));
   if (_mePhoneAppState != phoneAppState)
   {
      _mePhoneAppState = phoneAppState;
      setAccessoryAppState();
   }
}


/************************************************************************
*FUNCTION:      setNativeSpeechAppState
*DESCRIPTION:   Sets Native Speech Status
*PARAMETER:     speechAppState
*RETURNVALUE:   void
************************************************************************/
void AccessoryAppStateHandler::setNativeSpeechAppState(enSpeechAppState speechAppState)
{
   ETG_TRACE_USR4(("AccessoryAppStateHandler:setNativeSpeechAppState Received speechAppState = %d",  ETG_CENUM(enSpeechAppState, speechAppState)));
   if (_meSpeechAppState != speechAppState)
   {
      _meSpeechAppState = speechAppState;
      setAccessoryAppState();
   }
}


/************************************************************************
*FUNCTION:      setNativeNaviAppState
*DESCRIPTION:   Sets Native Navigation Status
*PARAMETER:     naviAppState
*RETURNVALUE:   void
************************************************************************/
void AccessoryAppStateHandler::setNativeNaviAppState(enNavigationAppState naviAppState)
{
   ETG_TRACE_USR4(("AccessoryAppStateHandler:setNativeNaviAppState Received naviAppState = %d",  ETG_CENUM(enNavigationAppState, naviAppState)));
   if (_meNavigationAppState != naviAppState)
   {
      _meNavigationAppState = naviAppState;
      setAccessoryAppState();
   }
}


/************************************************************************
*FUNCTION:      getNativePhoneAppState
*DESCRIPTION:   Returns current Native Phone Status
*PARAMETER:     none
*RETURNVALUE:   _mePhoneAppState
************************************************************************/
enPhoneAppState AccessoryAppStateHandler::getNativePhoneAppState()
{
   return _mePhoneAppState;
}


/************************************************************************
*FUNCTION:      getNativeSpeechAppState
*DESCRIPTION:   Returns current Native Speech Status
*PARAMETER:     none
*RETURNVALUE:   _meSpeechAppState
************************************************************************/
enSpeechAppState AccessoryAppStateHandler::getNativeSpeechAppState()
{
   return _meSpeechAppState;
}


/************************************************************************
*FUNCTION:      getNativeNaviAppState
*DESCRIPTION:   Returns current Native Navigation Status
*PARAMETER:     none
*RETURNVALUE:   _meNavigationAppState
************************************************************************/
enNavigationAppState AccessoryAppStateHandler::getNativeNaviAppState()
{
   return _meNavigationAppState;
}


/************************************************************************
*FUNCTION:      setAccessoryAppState
*DESCRIPTION:   Send the native app states to CarPlay
*PARAMETER:     none
*RETURNVALUE:   void
************************************************************************/
void AccessoryAppStateHandler::setAccessoryAppState()
{
   ETG_TRACE_USR4(("AccessoryAppStateHandler:sendAccessoryAppStateStart called eSpeechAppState = %d ePhoneAppState = %d eNavigationAppState = %d",
                   ETG_CENUM(enSpeechAppState, _meSpeechAppState), ETG_CENUM(enPhoneAppState, _mePhoneAppState), ETG_CENUM(enNavigationAppState, _meNavigationAppState)));

   _smartphoneFIProxy->sendAccessoryAppStateStart(*this, (::midw_smartphoneint_fi_types::T_e8_SpeechAppState)_meSpeechAppState,
         (::midw_smartphoneint_fi_types::T_e8_PhoneAppState)_mePhoneAppState,
         (::midw_smartphoneint_fi_types::T_e8_NavigationAppState)_meNavigationAppState);
}


/************************************************************************
*FUNCTION:		onAccessoryAppStateResult
*DESCRIPTION:
*PARAMETER:     proxy, result
*RETURNVALUE:	void
************************************************************************/
void AccessoryAppStateHandler::onAccessoryAppStateResult(const ::boost::shared_ptr< ::midw_smartphoneint_fi::Midw_smartphoneint_fiProxy >& proxy,
      const ::boost::shared_ptr< ::midw_smartphoneint_fi::AccessoryAppStateResult >& result)
{
   ETG_TRACE_USR4(("AccessoryAppStateHandler: onAccessoryAppStateResult Received ..."));
}


/************************************************************************
*FUNCTION:		onAccessoryAppStateError
*DESCRIPTION:
*PARAMETER:     proxy, error
*RETURNVALUE:	void
************************************************************************/
void AccessoryAppStateHandler::onAccessoryAppStateError(const ::boost::shared_ptr< ::midw_smartphoneint_fi::Midw_smartphoneint_fiProxy >& proxy,
      const ::boost::shared_ptr< ::midw_smartphoneint_fi::AccessoryAppStateError >& error)
{
   ETG_TRACE_USR4(("AccessoryAppStateHandler: onAccessoryAppStateError Received ..."));
}


/************************************************************************
*FUNCTION:		onCallStatusNoticeStatus
*DESCRIPTION:	This function receives status update for CallStatusNotice property
*PARAMETER:     proxy, status
*RETURNVALUE:	none
************************************************************************/
void AccessoryAppStateHandler::onCallStatusNoticeStatus(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy,
      const ::boost::shared_ptr< ::MOST_Tel_FI::CallStatusNoticeStatus >& status)
{
   ETG_TRACE_USR4(("AccessoryAppStateHandler: onCallStatusNoticeStatus Received ..."));
   if (proxy == _spiTelephoneProxy)
   {
      enPhoneAppState enNewCallStatus = evaluateBTTelCallStatus(status);
      ETG_TRACE_USR4(("AccessoryAppStateHandler: onCallStatusNoticeStatus  ePhoneAppState = %d  NewCallState = %d",  ETG_CENUM(enPhoneAppState, _mePhoneAppState), ETG_CENUM(enPhoneAppState, enNewCallStatus)));
      setNativePhoneAppState(enNewCallStatus);
   }
}


/************************************************************************
*FUNCTION:		onCallStatusNoticeError
*DESCRIPTION:	This function to get error from CallStatusNotice
*PARAMETER:     proxy, error
*RETURNVALUE:	none
************************************************************************/
void AccessoryAppStateHandler::onCallStatusNoticeError(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy,
      const ::boost::shared_ptr< ::MOST_Tel_FI::CallStatusNoticeError >& error)
{
   ETG_TRACE_USR4(("AccessoryAppStateHandler: onCallStatusNoticeError Received ..."));
}


/************************************************************************
*FUNCTION:		evaluateBTTelCallStatus
*DESCRIPTION:	This is a helper function to evaluate the BT call status according to both call instances
*PARAMETER:     status
*RETURNVALUE:	eCallInstance
************************************************************************/
enPhoneAppState AccessoryAppStateHandler:: evaluateBTTelCallStatus(const boost::shared_ptr< ::MOST_Tel_FI::CallStatusNoticeStatus >& status)
{
   enPhoneAppState eCallInstance = PhoneAppState__SPI_APP_STATE_PHONE_NOTACTIVE;
   uint8 u8CallState1 = status->getOCallStatusNoticeStream()[0].getE8CallStatus();
   uint8 u8CallState2 = status->getOCallStatusNoticeStream()[1].getE8CallStatus();
   enPhoneAppState eCallInstance1 = evaluateCallInstance1(u8CallState1);
   enPhoneAppState eCallInstance2 = evaluateCallInstance2(u8CallState2);
   ETG_TRACE_USR4(("AccessoryAppStateHandler: evaluateBTTelCallStatus called eCallInstance1 = %d eCallInstance2 = %d", ETG_CENUM(enPhoneAppState, eCallInstance1), ETG_CENUM(enPhoneAppState, eCallInstance1)));
   if ((PhoneAppState__SPI_APP_STATE_PHONE_ACTIVE == eCallInstance1) || (PhoneAppState__SPI_APP_STATE_PHONE_ACTIVE == eCallInstance2))
   {
      eCallInstance = PhoneAppState__SPI_APP_STATE_PHONE_ACTIVE;
   }
   return eCallInstance;
}


/************************************************************************
*FUNCTION:		evaluateCallInstance1
*DESCRIPTION:	This is a helper function to evaluate the BT call status for call instance1
*PARAMETER:     u8CallState1
*RETURNVALUE:	enPhoneAppState
************************************************************************/
enPhoneAppState AccessoryAppStateHandler::evaluateCallInstance1(uint8 u8CallState1)
{
   ETG_TRACE_USR4(("AccessoryAppStateHandler: evaluateCallInstance1 called %d ", u8CallState1));
   enPhoneAppState eCallInstance1 = PhoneAppState__SPI_APP_STATE_PHONE_UNKNOWN;
   switch (u8CallState1)
   {
      case ::most_Tel_fi_types::T_e8_TelCallStatus__e8DIALING:
      case ::most_Tel_fi_types::T_e8_TelCallStatus__e8ACTIVE:
      case ::most_Tel_fi_types::T_e8_TelCallStatus__e8ON_HOLD:
      case ::most_Tel_fi_types::T_e8_TelCallStatus__e8DISCONNECTING:
      case ::most_Tel_fi_types::T_e8_TelCallStatus__e8BUSY:
      case ::most_Tel_fi_types::T_e8_TelCallStatus__e8CONFERENCE:
      case ::most_Tel_fi_types::T_e8_TelCallStatus__e8IN_VOICEMAIL:
      case ::most_Tel_fi_types::T_e8_TelCallStatus__e8RINGTONE:
      {
         eCallInstance1 = PhoneAppState__SPI_APP_STATE_PHONE_ACTIVE;
         break;
      }
      case ::most_Tel_fi_types::T_e8_TelCallStatus__e8IDLE:
      default:
      {
         eCallInstance1 = PhoneAppState__SPI_APP_STATE_PHONE_NOTACTIVE;
         break;
      }
   }
   return eCallInstance1;
}


/************************************************************************
*FUNCTION:		evaluateCallInstance2
*DESCRIPTION:	This is a helper function to evaluate the BT call status for call instance2
*PARAMETER:     u8CallState2
*RETURNVALUE:	enPhoneAppState
************************************************************************/
enPhoneAppState AccessoryAppStateHandler::evaluateCallInstance2(uint8 u8CallState2)
{
   ETG_TRACE_USR4(("AccessoryAppStateHandler: evaluateCallInstance2 called %d", u8CallState2));
   enPhoneAppState eCallInstance2 = PhoneAppState__SPI_APP_STATE_PHONE_UNKNOWN;
   switch (u8CallState2)
   {
      case ::most_Tel_fi_types::T_e8_TelCallStatus__e8DIALING:
      case ::most_Tel_fi_types::T_e8_TelCallStatus__e8ACTIVE:
      case ::most_Tel_fi_types::T_e8_TelCallStatus__e8ON_HOLD:
      case ::most_Tel_fi_types::T_e8_TelCallStatus__e8DISCONNECTING:
      case ::most_Tel_fi_types::T_e8_TelCallStatus__e8BUSY:
      case ::most_Tel_fi_types::T_e8_TelCallStatus__e8CONFERENCE:
      case ::most_Tel_fi_types::T_e8_TelCallStatus__e8IN_VOICEMAIL:
      case ::most_Tel_fi_types::T_e8_TelCallStatus__e8RINGTONE:
      {
         eCallInstance2 = PhoneAppState__SPI_APP_STATE_PHONE_ACTIVE;
         break;
      }
      case ::most_Tel_fi_types::T_e8_TelCallStatus__e8IDLE:
      default:
      {
         eCallInstance2 = PhoneAppState__SPI_APP_STATE_PHONE_NOTACTIVE;
         break;
      }
   }
   return eCallInstance2;
}


/************************************************************************
*FUNCTION:		onConflictsDetectedStatus
*DESCRIPTION:	This function is to detect the conflicts
*PARAMETER:     proxy, status
*RETURNVALUE:	void
************************************************************************/
void AccessoryAppStateHandler::onConflictsDetectedStatus(const ::boost::shared_ptr< ::MOST_BTSet_FI::MOST_BTSet_FIProxy >& proxy,
      const ::boost::shared_ptr< ::MOST_BTSet_FI::ConflictsDetectedStatus >& status)
{
   ETG_TRACE_USR4(("AccessoryAppStateHandler: onConflictsDetectedStatus Received ..."));
   uint32 l_ActiveDeviceCategory = _connectionClientHandler->tGetActiveDeviceCategory();
   ETG_TRACE_USR4(("AccessoryAppStateHandler: onConflictsDetectedStatus Device category= %d", l_ActiveDeviceCategory));
   if (_spiBTSetProxy && (_spiBTSetProxy == proxy))
   {
      const ::most_BTSet_fi_types_Extended::T_e8_BTConflictTrigger& m_tU32ConflictTrigger = status->getE8ConflictTrigger();
      const ::most_BTSet_fi_types_Extended::T_BTConflictInfoList& conflictInfoList = status->getOBTConflictInfoList();
      ::most_BTSet_fi_types_Extended::T_BTConflictInfoList::const_iterator it;
      for (it = conflictInfoList.begin(); it != conflictInfoList.end(); it++)
      {
         ETG_TRACE_USR4(("onConflictsDetectedStatus ConflictState:%d  BTProfileUsageType:%d  BTProfile:%d  ConflictType:%d",
                         it->getE8ConflictState(), it->getE8BTProfileUsageType(), it->getE8BTProfile(), it->getE8ConflictType()));

         if ((it->getE8ConflictState() == ::most_BTSet_fi_types_Extended::T_e8_BTConflictState__e8CONFLICT_STATE_NEW) &&
               (it->getE8BTProfileUsageType() == ::most_BTSet_fi_types_Extended::T_e8_BTProfileUsageType__e8PROFILE_USAGE_NONE))
         {
            if (l_ActiveDeviceCategory == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_DIPO)
            {
               ETG_TRACE_USR4(("AccessoryAppStateHandler:onConflictsDetectedStatus DIPO Connected"));
               _spiBTSetProxy->sendResolveConflictsStart(*this, ::most_BTSet_fi_types_Extended::T_e8_BTConflictTrigger__e8CP_USB,
                     ::most_BTSet_fi_types_Extended::T_e8_BTResolutionType__e8RESOLUTION_TYPE_PROCEED);
            }
            else if (l_ActiveDeviceCategory == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_ANDROIDAUTO)
            {
               ETG_TRACE_USR4(("AccessoryAppStateHandler:onConflictsDetectedStatus  Android Auto Connected"));
               _spiBTSetProxy->sendResolveConflictsStart(*this, ::most_BTSet_fi_types_Extended::T_e8_BTConflictTrigger__e8AAP_USB,
                     ::most_BTSet_fi_types_Extended::T_e8_BTResolutionType__e8RESOLUTION_TYPE_PROCEED);
            }
         }
         if ((it->getE8ConflictState() == ::most_BTSet_fi_types_Extended::T_e8_BTConflictState__e8CONFLICT_STATE_NEW) &&
               (it->getE8BTProfileUsageType() == ::most_BTSet_fi_types_Extended::T_e8_BTProfileUsageType__e8PROFILE_USAGE_ACTIVE_CALL))
         {
            ETG_TRACE_USR4(("AccessoryAppStateHandler:onConflictsDetectedStatus entered when BT Call is Active"));
            if (l_ActiveDeviceCategory == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_DIPO)
            {
               ETG_TRACE_USR4(("AccessoryAppStateHandler:onConflictsDetectedStatus DIPO Connected when BT Call is Active"));
               POST_MSG((COURIER_MESSAGE_NEW(SPI_PFO_2ndDevice_Connect_confirmation)()));    //show popup here
               (*_SPI_PFO_2ndDev_Conf_Popup).mDevice1Name = false;
               _SPI_PFO_2ndDev_Conf_Popup.MarkItemModified(ItemKey::SPI_PFO_2ndDev_Conf_Popup::Device1NameItem);
               _SPI_PFO_2ndDev_Conf_Popup.SendUpdate();
               ETG_TRACE_USR4(("AccessoryAppStateHandler:onConflictsDetectedStatus DIPO Connected 2nd Device Connect Confirmation popup"));
            }
            else if (l_ActiveDeviceCategory == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_ANDROIDAUTO)
            {
               ETG_TRACE_USR4(("AccessoryAppStateHandler:onConflictsDetectedStatus  Android Auto Connected when BT call is active"));
               POST_MSG((COURIER_MESSAGE_NEW(SPI_PFO_2ndDevice_Connect_confirmation)()));     //show popup here
               (*_SPI_PFO_2ndDev_Conf_Popup).mDevice1Name = true;
               _SPI_PFO_2ndDev_Conf_Popup.MarkItemModified(ItemKey::SPI_PFO_2ndDev_Conf_Popup::Device1NameItem);
               _SPI_PFO_2ndDev_Conf_Popup.SendUpdate();
               ETG_TRACE_USR4(("AccessoryAppStateHandler:onConflictsDetectedStatus Android Auto Connected 2nd Device Connect Confirmation popup"));
            }
         }
      }
   }
}


/************************************************************************
*FUNCTION:		onConflictsDetectedError
*DESCRIPTION:	This function is to get the conflict detected error
*PARAMETER:     proxy, status
*RETURNVALUE:	void
************************************************************************/
void AccessoryAppStateHandler::onConflictsDetectedError(const ::boost::shared_ptr< ::MOST_BTSet_FI::MOST_BTSet_FIProxy >& proxy,
      const ::boost::shared_ptr< ::MOST_BTSet_FI::ConflictsDetectedError >& error)
{
   ETG_TRACE_USR4(("AccessoryAppStateHandler: onConflictsDetectedError Received ..."));
}


/************************************************************************
*FUNCTION:		onCourierMessage
*DESCRIPTION:	Function to resolve the conflict when BT is on active phone call
*PARAMETER:     oMsg
*RETURNVALUE:	bool
************************************************************************/
bool AccessoryAppStateHandler::onCourierMessage(const SecondDeviceConnectConfirmationYesMsg& oMsg)
{
   ETG_TRACE_USR4(("AccessoryAppStateHandler::SecondDeviceConnectConfirmationYesMsg() STARTS"));
   if (m_ActiveDeviceCategory == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_DIPO)
   {
      ETG_TRACE_USR4(("AccessoryAppStateHandler:SecondDeviceConnectConfirmationYesMsg DIPO Connected when BT Call is Active"));
      _spiBTSetProxy->sendResolveConflictsStart(*this, ::most_BTSet_fi_types_Extended::T_e8_BTConflictTrigger__e8CP_USB,
            ::most_BTSet_fi_types_Extended::T_e8_BTResolutionType__e8RESOLUTION_TYPE_PROCEED);
   }
   if (m_ActiveDeviceCategory == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_ANDROIDAUTO)
   {
      ETG_TRACE_USR4(("AccessoryAppStateHandler:SecondDeviceConnectConfirmationYesMsg Android Auto when BT Call is Active"));
      _spiBTSetProxy->sendResolveConflictsStart(*this, ::most_BTSet_fi_types_Extended::T_e8_BTConflictTrigger__e8AAP_USB,
            ::most_BTSet_fi_types_Extended::T_e8_BTResolutionType__e8RESOLUTION_TYPE_PROCEED);
   }
   return true;
}


/************************************************************************
*FUNCTION:		onCourierMessage
*DESCRIPTION:	Function to disconnect the device connected via USB
*PARAMETER:     oMsg
*RETURNVALUE:	bool
************************************************************************/
bool AccessoryAppStateHandler::onCourierMessage(const SecondDeviceConnectConfirmationNOMsg& oMsg)
{
   ETG_TRACE_USR4(("AccessoryAppStateHandler::SecondDeviceConnectConfirmationNoMsg() STARTS"));

   if (m_ActiveDeviceCategory == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_DIPO ||
         m_ActiveDeviceCategory == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_ANDROIDAUTO)
   {
      if (m_SessionStatus ==::midw_smartphoneint_fi_types::T_e8_SessionStatus__SESSION_ACTIVE)
      {
         ETG_TRACE_USR4(("AccessoryAppStateHandler::DeviceConnectionMsg()::Session Active Message"));
         _connectionClientHandler->vSendSelectDeviceStart();
         ETG_TRACE_USR4(("AccessoryAppStateHandler::vSendSelectDeviceStart()"));
      }
   }
   return true;
}


/************************************************************************
*FUNCTION:		onResolveConflictsResult
*DESCRIPTION:	Function to get the result of Resolve Conflict
*PARAMETER:     proxy, result
*RETURNVALUE:	void
************************************************************************/
void AccessoryAppStateHandler::onResolveConflictsResult(const ::boost::shared_ptr< ::MOST_BTSet_FI::MOST_BTSet_FIProxy >& proxy,
      const ::boost::shared_ptr< ::MOST_BTSet_FI::ResolveConflictsResult >& result)
{
   ETG_TRACE_USR4(("AccessoryAppStateHandler: onResolveConflictsResult Received ..."));
}


/************************************************************************
*FUNCTION:		onResolveConflictsError
*DESCRIPTION:	Function to get the error of Resolve Conflict
*PARAMETER:     proxy, error
*RETURNVALUE:	void
************************************************************************/
void AccessoryAppStateHandler::onResolveConflictsError(const ::boost::shared_ptr< ::MOST_BTSet_FI::MOST_BTSet_FIProxy >& proxy,
      const ::boost::shared_ptr< ::MOST_BTSet_FI::ResolveConflictsError >& error)
{
   ETG_TRACE_USR4(("AccessoryAppStateHandler: onResolveConflictsError Received ..."));
}


} //namespace Core
} //namespace app
