/* ***************************************************************************************
* FILE:          SpellerModel.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  SpellerModel.cpp is part of HMI-Base reference/demo/test applications
*    COPYRIGHT:  (c) 2018-2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"

#include "util/StrUtf8.h"
#include "AppHmi_PhoneStateMachine.h"
#include "AppHmi_PhoneConstants.h"
#include "SpellerModel.h"
#include "../PhoneBook/ServicePhonebook.h"
#include "hmi_trace_if.h"

#ifdef WINDOWS_SIMULATION
#include <iostream>
#include "AppHmi_PhoneStateMachineSEMLibB.h"
using namespace std;
#endif

#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_PHONE_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_I_FILE_PREFIX            App::Core::PhoneSpellerModel::
#include "trcGenProj/Header/SpellerModel.cpp.trc.h"
#endif

namespace App {
namespace Core {


SpellerModel* SpellerModel::m_poSpellerModel = NULL;
/************************************************************************
*FUNCTION: 	   SpellerModel
*DESCRIPTION:  Constructor
*PARAMETER: NA
*
*RETURNVALUE: 	NA
*
*HISTORY:
15/09/2017 Shwetha K
Initial Revision

************************************************************************/
SpellerModel::SpellerModel()

{
   ListRegistry::s_getInstance().addListImplementation(LISTID_DIALPADMATCHLIST, this);
   u8MatchListSize = 0;
}


/************************************************************************
*FUNCTION: 	   ~SpellerModel
*DESCRIPTION:  Destructor
*PARAMETER: NA
*
*RETURNVALUE: 	NA
*
*HISTORY:
15/09/2017 Shwetha K
Initial Revision

************************************************************************/
SpellerModel::~SpellerModel()
{
   ListRegistry::s_getInstance().removeListImplementation(LISTID_DIALPADMATCHLIST);
}


bool SpellerModel::Init()
{
   return true;
}


/**
* @brief   : Incomming message for HMI data model a message component
* @param   :
//*/
bool SpellerModel::onCourierMessage(const Courier::StartupMsg& /*oMsg*/)
{
   ETG_TRACE_USR4(("SpellerModel::onCourierMessage(StartupMsg)"));
   (*oListDataSize).mMatchListSize = BTPHN_C_VALUE_ZERO;
   (*oListID).mMatchListId = LISTID_DIALPADMATCHLIST;
   oListID.MarkAllItemsModified();
   oListID.SendUpdate(true);
   oListDataSize.MarkItemModified(ItemKey::ListDataSize::MatchListSizeItem);
   oListDataSize.SendUpdate(true);
   return true;
}


/************************************************************************
*FUNCTION: 	   OnUpdEntryText
*DESCRIPTION:  Updates entry text into edit field widget
*PARAMETER: sEntrytext: pressed key received from GUI
*
*RETURNVALUE: 	NA
*
*HISTORY:
15/09/2017 Shwetha K
*System Requirements : (SWFL - 278, SWFL - 279, SWFL - 285, SWFL - 292, SWFL - 1953, SWFL - 1961, SWFL - 270, SWFL - 283, SWFL - 299)
Initial Revision

************************************************************************/
void SpellerModel::OnUpdEntryText(std::string sEntrytext)
{
   ETG_TRACE_USR4(("Entered value strEditInfo: %s", sEntrytext.c_str()));
   (*oEntryText).mText = sEntrytext.c_str();
   oEntryText.MarkAllItemsModified();
   oEntryText.SendUpdate(true);
}


/************************************************************************
*FUNCTION: 	  strGetEntryText
*DESCRIPTION:  NA
*PARAMETER: NA
*
*RETURNVALUE: 	returns Pressed key string from GUI
*
*HISTORY:
15/09/2017 Shwetha K
*System Requirements : (SWFL - 278, SWFL - 279, SWFL - 285, SWFL - 292, SWFL - 1953, SWFL - 1961, SWFL - 270, SWFL - 283, SWFL - 299)
Initial Revision

************************************************************************/
std::string SpellerModel::strGetEntryText()
{
   return (*oEntryText).mText.GetCString();
}


/************************************************************************
*FUNCTION: 	   OnUpdValidCharSet
*DESCRIPTION:  Updates valid character set to Speller widget in GUI
*PARAMETER: sCharset: character set received from client
*
*RETURNVALUE: 	NA
*
*HISTORY:
15/09/2017 Shwetha K
*System Requirements : (SWFL - 278, SWFL - 279, SWFL - 285, SWFL - 292, SWFL - 1953, SWFL - 1961, SWFL - 270, SWFL - 283, SWFL - 299)
Initial Revision

************************************************************************/
void SpellerModel::OnUpdValidCharSet(Candera::String sCharset)
{
   (*oValidCharset).mCharacterSet = sCharset.GetCString();
   oValidCharset.MarkAllItemsModified();
   oValidCharset.SendUpdate(true);
}


/************************************************************************
*FUNCTION: 	   OnUpdSuggestionText
*DESCRIPTION:  updates suggestion text if there is valid match found to GUI
*PARAMETER: sSuggestionText: matched suggestion string
*
*RETURNVALUE: 	NA
*
*HISTORY:
15/09/2017 Shwetha K
*System Requirements : (SWFL - 278, SWFL - 279, SWFL - 285, SWFL - 292, SWFL - 1953, SWFL - 1961, SWFL - 270, SWFL - 283, SWFL - 299)
Initial Revision

************************************************************************/
void SpellerModel::OnUpdSuggestionText(std::string sSuggestionText)
{
   ETG_TRACE_USR4(("Suggestion Text Sent: %s", sSuggestionText.c_str()));
   (*oSuggesstionText).mText = sSuggestionText.c_str();
   oSuggesstionText.MarkAllItemsModified();
   bool retval = oSuggesstionText.SendUpdate(true);

   if (true == retval)
   {
      ETG_TRACE_USR4(("oSuggesstionText updated successfully "));
   }
   else
   {
      ETG_TRACE_USR4(("oSuggesstionText Not updated  "));
   }
}


/************************************************************************
*FUNCTION: 	   OnUpdDialButton
*DESCRIPTION:  Updates CAll button enable/disable in GUI
*PARAMETER: sEditInfo gives length of pressed key string.
			If 0, call button disabled, else call button is enabled

*
*RETURNVALUE: 	NA
*
*HISTORY:
*System Requirements : (SWFL - 278, SWFL - 279, SWFL - 285, SWFL - 292, SWFL - 1953, SWFL - 1961, SWFL - 270, SWFL - 283, SWFL - 299)
15/09/2017 Shwetha K
Initial Revision

************************************************************************/
void SpellerModel::OnUpdDialButton(std::string sEditInfo)
{
   if (BTPHN_C_VALUE_ZERO < sEditInfo.length())
   {
      (*oDialButton).mEnabled = BTPHN_C_BUTTON_ENABLED;
      ETG_TRACE_USR4(("Dial Call Button Enabled "));
   }
   else
   {
      (*oDialButton).mEnabled = BTPHN_C_BUTTON_DISABLED;
      ETG_TRACE_USR4(("Dial Call Button Disabled "));
   }
   ETG_TRACE_USR4(("Character Length: %d", sEditInfo.length()));
   oDialButton.MarkAllItemsModified();
   oDialButton.SendUpdate(true);
}


/************************************************************************
*FUNCTION: OnUpdMatchList
*DESCRIPTION: Invoked when match is found and match list data has to be updated
*
*PARAMETER:
*		u8Listsize : list size of match list as received from client
		u8WindowSize : max window size
*RETURNVALUE: 	NA
*HISTORY:
*System Requirements : (SWFL - 278, SWFL - 279, SWFL - 285, SWFL - 292, SWFL - 1953, SWFL - 1961, SWFL - 270, SWFL - 283, SWFL - 299)
*15/09/2017 Shwetha K
Initial Revision
************************************************************************/
void SpellerModel::OnUpdMatchList(tU16 u8Listsize, tU8 u8WindowSize)
{
   ETG_TRACE_USR4(("SpellerModel : OnUpdMatchList()"));
   bool bretval = FALSE;
   char strListSize[BTPHN_C_VALUE_TWO];
   /* Getting Call Status */
   (*oListDataSize).mMatchListSize = u8Listsize;
   /*(*oWindowDataSize).mMatchListWindowSize = u8WindowSize;
   	copoMatchListResp->GetWindowSize();*/

   sprintf(strListSize, "%d", u8Listsize);
   if (BTPHN_C_VALUE_ZERO < u8Listsize)
   {
      (*oMatchedItems).mCount = (Candera::String) strListSize;
      ETG_TRACE_USR4(("Matched Items %s", (*oMatchedItems).mCount.GetCString()));
      (*oMatchList).mEnabled = BTPHN_C_BUTTON_ENABLED;
   }
   else
   {
      (*oMatchedItems).mCount = (Candera::String)BTPHN_C_BT_SPACE;
      ETG_TRACE_USR4(("Matched Items %s", (*oMatchedItems).mCount.GetCString()));
      (*oMatchList).mEnabled = BTPHN_C_BUTTON_DISABLED;
   }

   oListDataSize.MarkItemModified(ItemKey::ListDataSize::MatchListSizeItem);
   bretval = oListDataSize.SendUpdate(true);

   oMatchedItems.MarkAllItemsModified();
   bretval = bretval && oMatchedItems.SendUpdate(true);
   oMatchList.MarkAllItemsModified();
   bretval = bretval && oMatchList.SendUpdate(true);

   if (true == bretval)
   {
      ETG_TRACE_USR4(("Match List Updated"));
   }
   else
   {
      ETG_TRACE_ERR(("Match List Update Failed"));
   }
}


/************************************************************************
*FUNCTION: 	   OnUpdDialMatchList
*DESCRIPTION:  it calls function to update suggestion text , match list , matched item count
*PARAMETER: bAutocompleteMatch: if match found or not
			copcAutocompleteMatchNum : matched number string
			u8MatchNumber: number of matches found
			arMatchList: vector containing matched number info
*
*RETURNVALUE: 	NA
*
*HISTORY:
*System Requirements : (SWFL - 278, SWFL - 279, SWFL - 285, SWFL - 292, SWFL - 1953, SWFL - 1961, SWFL - 270, SWFL - 283, SWFL - 299)
15/09/2017 Shwetha K
Initial Revision

************************************************************************/
void SpellerModel::OnUpdDialMatchList(tBool bAutocompleteMatch, tCString copcAutocompleteMatchNum, tU8 u8MatchNumber, std::vector<btphn_trPhnBkMatchListEntry>& arMatchList)
{
   ETG_TRACE_USR4(("SpellerModel : OnUpdDialMatchList()"));
   ETG_TRACE_USR4(("Autocomplete match found = %d\t Number of matches = %d", bAutocompleteMatch, u8MatchNumber));
   ETG_TRACE_USR4(("Autocomplete match number = %s", copcAutocompleteMatchNum));

   /* Sends the autocomplete matched contact number to HMI Model */
   OnUpdSuggestionText(copcAutocompleteMatchNum);

   /* Limiting the size of match list to BTPHN_C_MATCHLIST_WINDOW_SIZE */
   u8MatchListSize = (BTPHN_C_MATCHLIST_WINDOW_SIZE < u8MatchNumber) ?	BTPHN_C_MATCHLIST_WINDOW_SIZE : u8MatchNumber;

   /* Creates the match list and sends it to HMI */
   OnUpdMatchList(u8MatchListSize, BTPHN_C_MATCHLIST_WINDOW_SIZE);
   vSetMatchListElements(u8MatchListSize, arMatchList);
}


/************************************************************************
*FUNCTION: 	   vSetMatchListElements
*DESCRIPTION:  Updates  dialpad match list
*PARAMETER: u8ListSize : list size to be updated upto
			arMatchList: vector containing matched number info
*
*RETURNVALUE: 	NA
*
*HISTORY:
*System Requirements : (SWFL - 278, SWFL - 279, SWFL - 285, SWFL - 292, SWFL - 1953, SWFL - 1961, SWFL - 270, SWFL - 283, SWFL - 299)
15/09/2017 Shwetha K
Initial Revision

************************************************************************/
void SpellerModel::vSetMatchListElements(tU8 u8ListSize, std::vector<btphn_trPhnBkMatchListEntry>& arMatchList)
{
   ETG_TRACE_USR4(("SpellerModel : vSetMatchListElements u8ListSize is %d", u8ListSize));
   tU8 rowVal = 0;
   std::string sFirst_name;
   std::string sLast_name;
   std::string sName;
   ListDataProviderBuilder listBuilder(LISTID_DIALPADMATCHLIST);
   MatchList_ListItemData ListData[BTPHN_C_DEFAULT_MATCH_LIST_SIZE];
   most_PhonBk_fi_types::T_e8_PhonBkPreferredPhoneBookSortOrder sortorder = T_e8_PhonBkPreferredPhoneBookSortOrder__e8PREFERRED_SORT_ORDER_FIRSTNAME;
   if (ServicePhoneBook::poGetInstance() != NULL)
   {
      sortorder = ServicePhoneBook::poGetInstance()->enGetSortOrder(); //temporary untill P_OPT scene implementation
   }
   if (u8ListSize != 0)
   {
      /*loop from 0 to Matched list list size to update each row of Matched list*/
      for (rowVal = BTPHN_C_DEFAULT_VALUE_ZERO; rowVal < u8ListSize;  rowVal++)
      {
         /* Populating Matched numbers to the list*/
         // New addition of populating matched contact names along with numbers wrt Diaplpad New PSD CRQ
         ListData[rowVal].mMatchedNumber = arMatchList[rowVal].copcPhoneNumber.c_str();
         sFirst_name = arMatchList[rowVal].copcFirstName.c_str();
         sLast_name = arMatchList[rowVal].copcLastName.c_str();
         sName = AppHmiPhone_Helper::GetContactname(sFirst_name, sLast_name, sortorder);
         ListData[rowVal].mMatchedContactName = sName.c_str();
         /*adding each row into matched list */
         listBuilder.AddItem(rowVal, 0, "Layer_MatchListEntry_1").AddDataBindingUpdater<MatchList_ListItemDataBindingSource>(ListData[rowVal]);
      }//end of for loop
   }// end of if (u8ListSize!=0)

   /*Courier message to base class to update the list*/
   tSharedPtrDataProvider dataProvider = listBuilder.CreateDataProvider();
   POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
}


/************************************************************************
*FUNCTION: SpellerModel::OnUpdClearEditField
*DESCRIPTION: Invoked when ClearEditFieldRequestReq message called by IAR
*			  Updating EditField bindable property
*PARAMETER:
*		corfoMsg:	(I)
* 					Courier message from the Message Router
*RETURNVALUE: 	Bool
*HISTORY:
*System Requirements : (SWFL - 278, SWFL - 279, SWFL - 285, SWFL - 292, SWFL - 1953, SWFL - 1961, SWFL - 270, SWFL - 283, SWFL - 299)
15/09/2017 Shwetha K
Initial Revision

************************************************************************/
bool SpellerModel::OnUpdClearEditField()
{
   ETG_TRACE_USR4(("SpellerModel : OnUpdClearEditField()"));
   (*oDialButton).mEnabled = BTPHN_C_BUTTON_DISABLED;
   ETG_TRACE_USR4(("Dial Call Button Disabled "));
   oDialButton.MarkAllItemsModified();
   bool bRetVal = oDialButton.SendUpdate(true);

   (*oEntryText).mText = BTPHN_C_BT_NULL;
   oEntryText.MarkAllItemsModified();
   bRetVal = bRetVal && oEntryText.SendUpdate(true);

   (*oMatchedItems).mCount = (Candera::String)BTPHN_C_BT_SPACE;
   oMatchedItems.MarkAllItemsModified();
   bRetVal = bRetVal && oMatchedItems.SendUpdate(true);

   (*oMatchList).mEnabled = BTPHN_C_BUTTON_DISABLED;
   oMatchList.MarkAllItemsModified();
   bRetVal = bRetVal && oMatchList.SendUpdate(true);
   return bRetVal;
}


}
}
