/* ***************************************************************************************
* FILE:          HMIModelComponent.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  HMIModelComponent.cpp is part of HMI-Base reference/demo/test applications
*    COPYRIGHT:  (c) 2019-2020 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"

#include "util/StrUtf8.h"
#include "AppHmi_PhoneStateMachine.h"
#include "AppHmi_PhoneConstants.h"
#include "PhonebookModelComponent.h"
#include "App/Core/HmiTranslation_TextIds.h"
#include "../PhoneBook/ServicePhonebook.h"
#include "hmi_trace_if.h"
#include <iostream>
#include <sstream>
#ifdef WINDOWS_SIMULATION
#include <iostream>
#include "AppHmi_PhoneStateMachineSEMLibB.h"
#endif

#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_PHONE_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_I_FILE_PREFIX            App::Core::PhonebookModelComponent::
#include "trcGenProj/Header/PhonebookModelComponent.cpp.trc.h"
#endif

using namespace std;
namespace App {
namespace Core {
PhonebookModelComponent* PhonebookModelComponent::m_poPhonebookModelComponent = NULL;

/************************************************************************
		*FUNCTION: 	   PhonebookModelComponent
		*DESCRIPTION:  Constructor
		*PARAMETER: NA
		* Vke6kor
************************************************************************/
PhonebookModelComponent::PhonebookModelComponent() :
   m_rotationEndValue_Phonebook(360),
   m_rotationStartValue_Phonebook(0),
   m_rotationEndValue_Calllist(360),
   m_rotationStartValue_Calllist(0)
{
   ETG_TRACE_USR4(("PhonebookModelComponent : PhonebookModelComponent()"));
   ListRegistry::s_getInstance().addListImplementation(LISTID_PHONECONTACTS, this);
   ListRegistry::s_getInstance().addListImplementation(LISTID_CONTACTCALLLIST, this);
   ListRegistry::s_getInstance().addListImplementation(LISTID_PSETTING, this);
   ListRegistry::s_getInstance().addListImplementation(LISTID_SORTORDER, this);
   _currentPhoneValOption = FirstName;
   m_isMessageSettingEnabled = true;
   m_isRingtoneSettingsEnabled = true;
   m_isSortOrderAvailable = true;

   Init();
}


FeatStd::TextId id49("0x0535");
FeatStd::TextId id50("0x0536");
Candera::String PhoneModeStrings[3] = { Candera::String(TextId_0x0535), Candera::String(TextId_0x0536), Candera::String(TextId_0x0534) };
static const char* const PHONESETTING_DATA_TEXT_BUTTON_TOGGLE_LIST_ITEM = "Layer_Checkbox";
static const char* const PHONESETTING_DATA_TEXT_BUTTON_TOGGLE_LIST_ITEM_TONES = "Layer_Checkbox_Tones";
Candera::String PhoneSettingString[2] = { Candera::String(TextId_0x06A7), Candera::String(TextId_0x06A5) };
//std::string PhoneSettingString[2] = {"Show SmartPhone Messages","Use PhoneRingtone"};
/************************************************************************
		*FUNCTION: 	   ~PhonebookModelComponent
		*DESCRIPTION:  Destructor
		*PARAMETER: NA
		* Vke6kor
************************************************************************/
PhonebookModelComponent::~PhonebookModelComponent()
{
   ETG_TRACE_USR4(("PhonebookModelComponent : ~PhonebookModelComponent()"));

   ListRegistry::s_getInstance().removeListImplementation(LISTID_PHONECONTACTS);//For phonebook
   ListRegistry::s_getInstance().removeListImplementation(LISTID_CONTACTCALLLIST);
   ListRegistry::s_getInstance().removeListImplementation(LISTID_PSETTING);
   ListRegistry::s_getInstance().removeListImplementation(LISTID_SORTORDER);
}


bool PhonebookModelComponent::Init()
{
   ETG_TRACE_USR4(("PhonebookModelComponent :Init retrieve last opt setting from datapool "));

   uint16 u16VehicleType = vReadVehicleVariant();

   // MSG services are available only when vehicle variant is other than Smart
   if (VEHICLETYPE_SMART != u16VehicleType)
   {
      vSetShowIncomingMessage(PhoneDataPoolConfig::getInstance()->getMessageSettings());
   }
   vSetUsePhoneRingtone(PhoneDataPoolConfig::getInstance()->getDpPhoneRingToneStatus());

   if (DP_FIRST_NAME == PhoneDataPoolConfig::getInstance()->getDpPhoneSortOrderValue())
   {
      vSetSortOrder(BTPHN_C_SORT_FIRST_NAME, false);
   }
   else
   {
      vSetSortOrder(BTPHN_C_SORT_LAST_NAME, true);
   }

   return true;
}


/**
 * @brief   : Incomming message for HMI data model a message component
 * @param   :
 */
bool PhonebookModelComponent::onCourierMessage(const Courier::StartupMsg& /*oMsg*/)
{
   ETG_TRACE_USR4(("PhonebookModelComponent : StartupMsg"));
//	For Phone Book
   (*m_ListDataSize).mContactListSize = BTPHN_C_VALUE_ZERO;
   (*m_ListID).mPhoneContactListId = LISTID_PHONECONTACTS;
   m_ListID.MarkAllItemsModified();
   m_ListID.SendUpdate(true);
   m_ListDataSize.MarkItemModified(ItemKey::ListDataSize::ContactListSizeItem);
   m_ListDataSize.SendUpdate(true);

   (*m_ListID).mContactCallListId = LISTID_CONTACTCALLLIST;
   m_ListID.MarkAllItemsModified();
   m_ListID.SendUpdate(true);

   return true;
}


/**
* This function creates ListDataProviderBuilder and fills Phonebook list items to it.
* Each row in Phonebook list includes first name, last name, contact name, contact count in the same order.
* Once all the data is filled to list builder object, it posts ListDateProviderResMsg courier message by passing a shared pointer to lister builder object.
* This courier message is handled by framework to update the list
* @param[in] - arContactListSlice - Phonebook list item
*
* @param[out]- returns void
* System Requirements : (SWFL - 706, SWFL - 1944)
initial rev:Vidya
*/
void PhonebookModelComponent::OnUpdContactListSlice(std::vector<btphn_trContactInfo>& arContactListSlice,
      tU16 u16PBStartIndex, tU16 lastItemIndex_PBList, tU16 u16TotalListSize, tBool bLoadPersistent, tU32 m_U8ViewPortStartIndex)
{
   ETG_TRACE_USR4(("PhonebookModelComponent : OnUpdContactListSlice u16TotalListSize= %d ", u16TotalListSize));
   tU8 u8Windowsize = arContactListSlice.size();
   ETG_TRACE_USR4(("OnUpdContactListSlice u8Windowsize = %d , start index is %d, ", u8Windowsize, u16PBStartIndex));
   if (u8Windowsize == 0)
   {
      ETG_TRACE_FATAL(("PhonebookModelComponent  arContactListSlice.size() is 0"));
      return;
   }

   ListDataProviderBuilder listBuilder(LISTID_PHONECONTACTS);
#ifdef WINDOWS_SIMULATION
   ContactList_ListItemData ListData[25];
#else
   ContactList_ListItemData ListData[lastItemIndex_PBList + 1];
#endif

   std::string sFirst_name;
   std::string sLast_name;
   std::string sName;
   most_PhonBk_fi_types::T_e8_PhonBkPreferredPhoneBookSortOrder sortorder = T_e8_PhonBkPreferredPhoneBookSortOrder__e8PREFERRED_SORT_ORDER_FIRSTNAME;

   if (ServicePhoneBook::poGetInstance() != NULL)
   {
      sortorder = ServicePhoneBook::poGetInstance()->enGetSortOrder(); //temporary untill P_OPT scene implementation
   }

   /*Extract data from array arContactListSlice and fill the data to the list starting from start index to last index */
   for (tU32 u8Index = BTPHN_C_DEFAULT_VALUE_ZERO, dataIndex = u16PBStartIndex;
         ((u8Index < u8Windowsize) && (dataIndex <= lastItemIndex_PBList));
         u8Index++, dataIndex++)
   {
      sFirst_name = arContactListSlice[u8Index].szFirstName.c_str();
      sLast_name = arContactListSlice[u8Index].szLastName.c_str();
      /*invoking GetContactname fuction to obtain contact name  */
      sName = AppHmiPhone_Helper::GetContactname(sFirst_name, sLast_name, sortorder);
      ListData[dataIndex].mContactName = sName.c_str();
      //conversion from integer to string type
      stringstream ss_PhoneNumberCount_str;
      ss_PhoneNumberCount_str << arContactListSlice[u8Index].u8PhoneNumberCount;
      string PhoneNumberCount_str = ss_PhoneNumberCount_str.str();
      ListData[dataIndex].mPhoneNumberCount = PhoneNumberCount_str.c_str();
      /*add each row items to the list*/
      ETG_TRACE_USR4(("OnUpdContactListSlice row[%d] sName =%s ", dataIndex, ListData[dataIndex].mContactName.GetCString()));
      listBuilder.AddItem(dataIndex, 0, "Layer_ContactListEntry_1").AddDataBindingUpdater<ContactList_ListItemDataBindingSource>(ListData[dataIndex]);
   }
   if (bLoadPersistent)
   {
      ETG_TRACE_USR4(("PhonebookModelComponent : OnUpdContactListSlice : bLoadPersistent is %d", bLoadPersistent));
      POST_MSG(COURIER_MESSAGE_NEW(ListChangeMsg)(LISTID_PHONECONTACTS, ListChangeSet, m_U8ViewPortStartIndex));

      u16PBStartIndex = m_U8ViewPortStartIndex;
      // Setting this entry to false as we have update the list widget with the ListChangeMsg. If not done, we will be repeatedly sending this mag again and again
      bLoadPersistent = false;
   }
   /*Courier message to base class to update the list*/
   tSharedPtrDataProvider dataProvider = listBuilder.CreateDataProvider(u16PBStartIndex, u16TotalListSize);
   POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));

   //TODO: Vidya:To post the ListDateProviderResMsg courier message outside the if(arCallListSlice.size() != 0) check once the no Contact list is implemented
}


/**
* This function updates phonebook list size and is list empty data binding values based on the list size received from client
* @param[in] - corfu16TotalListSize- phonebook list size
* @param[out]- returns void
* System Requirements : SWFL-706
initial rev:Vidya
*/
void PhonebookModelComponent::OnUpdContactListSizeToModel(tU16 corfu16TotalListSize)
{
   ETG_TRACE_USR4(("PhonebookModelComponent : OnUpdContactListSizeToModel corfu16TotalListSize: %d ", corfu16TotalListSize));
   /*update phonebook list size*/
   (*m_ListDataSize).mContactListSize = corfu16TotalListSize;
   m_ListDataSize.MarkItemModified(ItemKey::ListDataSize::ContactListSizeItem);
   m_ListDataSize.SendUpdate(true);
   /*update list empty status */
   if (BTPHN_C_VALUE_ZERO == corfu16TotalListSize)
   {
      (*oIsListNotempty).mContactList = BTPHN_C_VALUE_ONE;
   }
   else
   {
      (*oIsListNotempty).mContactList = BTPHN_C_VALUE_ZERO;
   }
   oIsListNotempty.MarkItemModified(ItemKey::IsListNotempty::ContactListItem);
   oIsListNotempty.SendUpdate(true);
}


/**
* This function creates ListDataProviderBuilder and fills detailed contacts list items to it.
* Each row in Phonebook list includes contact number and the contact type icon..
* Once all the data is filled to list builder object, it posts ListDateProviderResMsg courier message by passing a shared pointer to lister builder object.
* This courier message is handled by framework to update the list
* @param[in] - arContactListSlice - copcFirstName, copcLastName, acopcContactNumbers, ContactDetailType
** System Requirements : (SWFL - 1946)
* @param[out]- returns void

initial rev:Vidya
*/
//void PhonebookModelComponent::OnUpdGetContactDetails(std::string copcFirstName, std::string copcLastName, std::string acopcContactNumbers[], tU8 ContactDetailType[])
//{
//	ETG_TRACE_USR4(("PhonebookModelComponent : OnUpdGetContactDetails"));
//	vUpdateHeaderContactName(copcFirstName, copcLastName);
//	ListDataProviderBuilder listBuilder(LISTID_CONTACTCALLLIST);
//
//	ContactCall_ListItemData ListData[BTPHN_C_CONTACT_DETAILS_MAX_NUMBER];
//
//	for (int index = 0; index < BTPHN_C_CONTACT_DETAILS_MAX_NUMBER; index++)
//	{
//		ETG_TRACE_USR4(("PhonebookModelComponent : OnUpdGetContactDetails : acopcContactNumbers[index]:: %s", acopcContactNumbers[index].c_str()));
//		ListData[index].mContactPhoneNumber = acopcContactNumbers[index].c_str();
//		ListData[index].mContactInfoIcon = ContactDetailType[index];
//		listBuilder.AddItem(index, 0, "Layer_CallListEntry_1").AddDataBindingUpdater<ContactCall_ListItemDataBindingSource>(ListData[index]);
//
//
//		tSharedPtrDataProvider dataProvider = listBuilder.CreateDataProvider();
//		POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
//
//	}
//}
void PhonebookModelComponent::OnUpdGetContactDetails(std::string copcFirstName, std::string copcLastName, std::vector<btphn_trContactCallDetail>& arDetailedContactNum)
{
   ETG_TRACE_USR4(("PhonebookModelComponent : OnUpdGetContactDetails"));
   vUpdateHeaderContactName(copcFirstName, copcLastName);
   tU32 u8Listsize = arDetailedContactNum.size();
   if (u8Listsize == 0)
   {
      ETG_TRACE_ERR((" OnUpdGetContactDetails: arDetailedContactNum size = 0 "));
   }

   ListDataProviderBuilder listBuilder(LISTID_CONTACTCALLLIST);
   ContactCall_ListItemData ListData[BTPHN_C_CONTACT_DETAILS_MAX_NUMBER];

   for (int index = 0; ((index < BTPHN_C_CONTACT_DETAILS_MAX_NUMBER) && (index < u8Listsize)); index++)
   {
      ETG_TRACE_USR4(("PhonebookModelComponent : OnUpdGetContactDetails : acopcContactNumbers[%d]:: %s", index, arDetailedContactNum[index].szContactNumber.c_str()));
      ServicePhoneBook::poGetInstance()->ContactNumbersList[index] = arDetailedContactNum[index].szContactNumber.c_str(); //680019
      ListData[index].mContactPhoneNumber = arDetailedContactNum[index].szContactNumber.c_str();
      ListData[index].mContactInfoIcon = arDetailedContactNum[index].u8ContactNumType;
      ListData[index].mPrimaryIcon = arDetailedContactNum[index].bPrefferedNum;
      listBuilder.AddItem(index, 0, "Layer_CallListEntry_1").AddDataBindingUpdater<ContactCall_ListItemDataBindingSource>(ListData[index]);
   }
   tSharedPtrDataProvider dataProvider = listBuilder.CreateDataProvider();
   POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
}


/**
* This function updates the header contact name in the detailed contact scene.
* @param[in] - arContactListSlice - copcFirstName, copcLastName
*
* @param[out]- returns void
* System Requirements : (SWFL - 706, SWFL - 1944, SYSFL-1779, SYSFL-1777, SYSFL-1766, SYSLFL-1765, SYSFL-1855)
initial rev:Vidya
*/
void PhonebookModelComponent::vUpdateHeaderContactName(std::string copcFirstName, std::string copcLastName)
{
   std::string sName;
   most_PhonBk_fi_types::T_e8_PhonBkPreferredPhoneBookSortOrder sortorder = T_e8_PhonBkPreferredPhoneBookSortOrder__e8PREFERRED_SORT_ORDER_FIRSTNAME;

   if (ServicePhoneBook::poGetInstance() != NULL)
   {
      sortorder = ServicePhoneBook::poGetInstance()->enGetSortOrder(); //temporary untill P_OPT scene implementation
   }
   /*invoking GetContactname fuction to obtain contact name  */
   sName = AppHmiPhone_Helper::GetContactname(copcFirstName, copcLastName, sortorder);
   POST_MSG((COURIER_MESSAGE_NEW(ContactDetailsResp)((Candera::String)sName.c_str(), 10, BTPHN_C_CONTACTDETAILSLIST_WINDOW_SIZE)));
   /*updating the contact name of header to the GUI*/
   (*oContactCallName).mCallerName = (Candera::String)sName.c_str();
   oContactCallName.MarkAllItemsModified();
   oContactCallName.SendUpdate(true);
}


void PhonebookModelComponent::OnUpdSettingsList()
{
   // Always take latest from datapool
   Init();

   ETG_TRACE_USR4(("PhonebookModelComponent : OnUpdSettingsList LISTID_PSETTING"));
   ListDataProviderBuilder listBuilder(LISTID_PSETTING);

   ETG_TRACE_USR4((" OnUpdSettingsList:SortOrder: %s ", strGetSortOrder().c_str()));
   uint8 rowindexValue = 0; //If its CMC then message settings are present so ringtone should be in index1, sort order index2
   uint16 u16VehicleType = vReadVehicleVariant();
   // MSG services are available only when vehicle variant is other than Smart
   if (VEHICLETYPE_SMART != u16VehicleType)
   {
      listBuilder.AddItem(
         rowindexValue, 	//  - identifies the row
         0UL,   //  - column
         PHONESETTING_DATA_TEXT_BUTTON_TOGGLE_LIST_ITEM)  //  - overwrites the default data context
      .AddData(PhoneSettingString[0])//1 - Text of the button, should be obtained from Candera.
      .AddData(bGetShowIncomingMessage())//2 - Toggle On/Off
      .AddData(m_isMessageSettingEnabled);//3 - Enable/Disable

      rowindexValue = 1;//If its CMC then message settings are present so ringtone should be in index1,sort order index2
   }

   listBuilder.AddItem(
      rowindexValue, 	//  - identifies the row
      0UL,   //  - column
      PHONESETTING_DATA_TEXT_BUTTON_TOGGLE_LIST_ITEM)  //  - overwrites the default data context
   .AddData(PhoneSettingString[1])//1 - Text of the button, should be obtained from Candera.
   .AddData(u8GetUsePhoneRingtone())//2 - Toggle On/Off
   .AddData(m_isRingtoneSettingsEnabled);//3 - Enable/Disable

   rowindexValue++;// increment the row value

   if (strGetSortOrder() == "First Name")
   {
      listBuilder.AddExpandableItem(static_cast<int>(rowindexValue), LISTID_SORTORDER, "Combobox_ListItem")
      .AddData(PhoneModeStrings[2])
      .AddData(LISTID_SORTORDER)
      .AddData(Candera::String(id49));
   }
   else
   {
      listBuilder.AddExpandableItem(static_cast<int>(rowindexValue), LISTID_SORTORDER, "Combobox_ListItem")
      .AddData(PhoneModeStrings[2])
      .AddData(LISTID_SORTORDER)
      .AddData(Candera::String(id50));
   }

   tSharedPtrDataProvider dataProvider = listBuilder.CreateDataProvider();
   POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));

   //
   //listBuilder.AddExpandableItem(static_cast<int>(e_MODE),            //   - identifies the row
   // LIST_ID_SYSTEM_DISPLAYSETTING_DAYNIGHTAUTOMODE_LIST,                          //   - used to automatically toggle the visibility of the combobox dropdown list when the combox button is clicked
   // "Combobox_ListItem")         //   - overwrites the default data context
   // .AddData(DisplaySettingText[1])  //1 - Text of the combobox button.
   // .AddData(LIST_ID_SYSTEM_DISPLAYSETTING_DAYNIGHTAUTOMODE_LIST)				 //2 - ListId of the combobox dropdown list.
   // .AddData(_currentDispModeOptionStr);

   //
}


void PhonebookModelComponent::OnUpdSortOrder()
{
   ETG_TRACE_USR4(("PhonebookModelComponent : OnUpdSortOrder LISTID_SORTORDER"));

   //here giving the second argument is must
   ListDataProviderBuilder listBuilder(LISTID_SORTORDER, "DropdownList");
   SettingsData ListData;
   ListData.mSortPhoneBookTypeText = Candera::String(TextId_0x0535);
   listBuilder.AddItem(static_cast<int>(0)).AddData(Candera::String(TextId_0x0535));
   ETG_TRACE_USR4((" OnUpdSortOrder:sort name: %s ", ListData.mSortPhoneBookTypeText));
   ListData.mSortPhoneBookTypeText = Candera::String(TextId_0x0536);
   listBuilder.AddItem(static_cast<int>(1)).AddData(Candera::String(TextId_0x0536));
   ETG_TRACE_USR4((" OnUpdSortOrder:sort name: %s ", ListData.mSortPhoneBookTypeText));

   tSharedPtrDataProvider dataProvider = listBuilder.CreateDataProvider();
   POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
}


/* void PhonebookModelComponent::OnUpdChangeDropdown(Candera::String Sort)
{
   (*oSettings).mSortPhoneBookTypeText = Sort;
   oSettings.MarkItemModified
   (ItemKey::Settings::SortPhoneBookTypeTextItem);
   oSettings.SendUpdate(true);
} */


void PhonebookModelComponent::OnUpdChangeRingtoneStatus(tU16 corfu16RingtoneID)
{
   if (m_isRingtoneSettingsEnabled) // If bEnable is true,that means hfp connection is active,
      //only then the middleware updates to be taken.
   {
      ETG_TRACE_USR4(("PhonebookModelComponent : OnUpdChangeRingtoneStatus()"));

      if (BTPHN_C_VALUE_ONE == corfu16RingtoneID)
      {
         (*oSettings).mUsePhoneRingtone_IsSelected = BTPHN_C_CHECKED;
         ETG_TRACE_USR4(("PhonebookModelComponent : OnUpdChangeRingtoneStatus():BTPHN_C_CHECKED"));
      }
      else
      {
         (*oSettings).mUsePhoneRingtone_IsSelected = BTPHN_C_UNCHECKED;
         ETG_TRACE_USR4(("PhonebookModelComponent : OnUpdChangeRingtoneStatus():BTPHN_C_UNCHECKED"));
      }

      vSetUsePhoneRingtone((*oSettings).mUsePhoneRingtone_IsSelected);

      ETG_TRACE_USR4(("PhonebookModelComponent :RingTone Status: %d", (*oSettings).mUsePhoneRingtone_IsSelected));
      PhonebookModelComponent::poGetInstance()->OnUpdSettingsList();
   }
}


void PhonebookModelComponent::vClearContactList()
{
   ETG_TRACE_USR4(("PhonebookModelComponent : vClearContactList "));
   ListDataProviderBuilder listBuilder(LISTID_PHONECONTACTS);
   tSharedPtrDataProvider dataProvider = listBuilder.CreateDataProvider();
   POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
}


/**
* This function updates the databinding property for clockwiserotation of the busy animation widget in clockwise
* direction for calllist loading scene
* initial rev:Vidya
*/

void PhonebookModelComponent::OnUpdChannel1_CalllistLoading()
{
   ETG_TRACE_USR4(("PhonebookModelComponent : bUpdChannel1_CalllistLoading "));
   tBool bretval;
   char buffer[40];
   SNPRINTF(buffer, sizeof(buffer), "%u;%u", m_rotationStartValue_Calllist, m_rotationEndValue_Calllist);
   (*oLoadingAnimation).mChannel1_Calllist = buffer;
   oLoadingAnimation.MarkItemModified(ItemKey::LoadingAnimationInfo::Channel1_CalllistItem);
   bretval = oLoadingAnimation.SendUpdate(true);

   if (false == bretval)
   {
      ETG_TRACE_USR4(("bUpdChannel1_CalllistLoading, Data updation failed"));
   }
   else
   {
      ETG_TRACE_USR4(("bUpdChannel1_CalllistLoading, Data updation Successfully"));
   }
}


/**
* This function updates the databinding property for clockwiserotation of the busy animation widget in clockwise
* direction for contacts loading scene
* initial rev:Vidya
*/
void PhonebookModelComponent::OnUpdChannel1_PhonebookLoading()
{
   ETG_TRACE_USR4(("PhonebookModelComponent : bUpdChannel1_PhonebookLoading "));
   tBool bretval;
   char buffer[40];
   SNPRINTF(buffer, sizeof(buffer), "%u;%u", m_rotationStartValue_Phonebook, m_rotationEndValue_Phonebook);
   (*oLoadingAnimation).mChannel1_Phonebook = buffer;
   oLoadingAnimation.MarkItemModified(ItemKey::LoadingAnimationInfo::Channel1_PhonebookItem);
   bretval = oLoadingAnimation.SendUpdate(true);

   if (false == bretval)
   {
      ETG_TRACE_USR4(("bUpdChannel1_PhonebookLoading, Data updation failed"));
   }
   else
   {
      ETG_TRACE_USR4(("bUpdChannel1_PhonebookLoading, Data updation Successfully"));
   }
}


/* setter function for UsePhoneRingtone option*/
void PhonebookModelComponent::vSetUsePhoneRingtone(tU8 CheckBoxStatus)
{
   if (!m_isRingtoneSettingsEnabled) // If bEnable is false, then checkbox should be either checkeddisable/uncheckeddisable
   {
      if (BTPHN_C_CHECKED == CheckBoxStatus)
      {
         CheckBoxStatus = BTPHN_C_CHECKEDDISABLE;
         ETG_TRACE_USR4(("BTPHN_C_CHECKEDDISABLE : ringtone"));
      }
      else
      {
         ETG_TRACE_USR4(("BTPHN_C_UNCHECKEDDISABLE : ringtone"));
         CheckBoxStatus = BTPHN_C_UNCHECKEDDISABLE;
      }
   }

   (*oSettings).mUsePhoneRingtone_IsSelected = CheckBoxStatus;
   oSettings.MarkItemModified(ItemKey::Settings::UsePhoneRingtone_IsSelectedItem);
   oSettings.SendUpdate(true);

   // Scn: Change in Ringtone status is coming from Service BT. Hence update datapool
   ETG_TRACE_USR4(("PhonebookModelComponent update change in ringtone status"));
   if (m_isRingtoneSettingsEnabled) // set to Dp only if the list is enabled, i.e active hfp connection
   {
      PhoneDataPoolConfig::getInstance()->setDpPhoneRingToneStatus(CheckBoxStatus);
   }

   // Update settings list
   // OnUpdSettingsList();
}


/*setter for sort order*/
void PhonebookModelComponent::vSetSortOrder(std::string  strSortOrder, bool isLastName)
{
   if (0 != strSortOrder.compare(""))
   {
      (*oSettings).mSortPhoneBookTypeText = strSortOrder.c_str();
      oSettings.MarkItemModified(ItemKey::Settings::SortPhoneBookTypeTextItem);
      oSettings.SendUpdate(true);
   }

   //this data binding either enables or disables the dropdown list
   (*oSettings).mEnableDisableSortOrder = m_isSortOrderAvailable;
   oSettings.MarkItemModified(ItemKey::Settings::EnableDisableSortOrderItem);
   oSettings.SendUpdate(true);

   // Scn: Change in sort order status is coming from Service BT. Hence update datapool
   ETG_TRACE_USR4(("PhonebookModelComponent update change in sort order status "));

   // optimization logic:
   //    for first name order : isLastName will be false (0). 0th index of list is first name
   // 	  for last name order: isLastName will be true (1). 1st index of list is last name
   // 	  Hence we update the bool as it is to datapool. 0-for first and 1- for last name order
   if (m_isSortOrderAvailable) //set to dp only if the list is enabled, i.e active hfp connection
   {
      PhoneDataPoolConfig::getInstance()->setDpSortOrderValue(isLastName);
   }
}


void PhonebookModelComponent::vChangeActivationOfMessageSetting(bool bEnable)
{
   // ETG_TRACE_USR4(("PhonebookModelComponent::vChangeActivationOfMessageSetting"));
   ETG_TRACE_USR4(("bEnable,value:%d", bEnable));
   m_isMessageSettingEnabled = bEnable;
}


void PhonebookModelComponent::vChangeActivationOfSortOrderSetting(bool bEnable)
{
   //ETG_TRACE_USR4(("PhonebookModelComponent::vChangeActivationOfSortOrderSetting"));
   ETG_TRACE_USR4(("bEnable,value:%d", bEnable));

   m_isSortOrderAvailable = bEnable;//set the toggle option to enable or disable
}


void PhonebookModelComponent::vChangeActivationOfRingtoneSetting(bool bEnable)
{
   //ETG_TRACE_USR4(("PhonebookModelComponent::vChangeActivationOfRingtoneSetting"));
   ETG_TRACE_USR4(("bEnableDisable,value:%d", bEnable));

   m_isRingtoneSettingsEnabled = bEnable; //set the toggle option to enable or disable
}


uint16 PhonebookModelComponent::vReadVehicleVariant()
{
   uint16 u16VehicleType = VEHICLETYPE_TRUCK;
   //Read VehicleType from KDS
   if (PhoneDataPoolConfig::getInstance())
   {
      ETG_TRACE_USR4(("Retrive vehicleType information from KDS"));
      u16VehicleType = PhoneDataPoolConfig::getInstance()->getKDSVehicleType();
      ETG_TRACE_USR4(("getKDSVehicleType, Vehicle type : %d", u16VehicleType));
   }
   else
   {
      ETG_TRACE_USR4(("Error in getting PhoneDataPoolConfig instance"));
   }

   return u16VehicleType;
}


}
}
