/* ***************************************************************************************
* FILE:          DeviceListModel.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  DeviceListModel.cpp is part of HMI-Base reference/demo/test applications
*    COPYRIGHT:  (c) 2017-2020 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"

#include "util/StrUtf8.h"
#include "AppHmi_PhoneStateMachine.h"
#include "AppHmi_PhoneConstants.h"
#include "DeviceListModel.h"
#include "App/Core/SmartPhone/SmartPhoneClient.h"
#include "hmi_trace_if.h"
#ifdef WINDOWS_SIMULATION
#include <iostream>
#include "AppHmi_PhoneStateMachineSEMLibB.h"
#endif
#include "CallHistoryModel.h"
#include "App/datapool/PhoneDataPoolConfig.h"
#include "Common/VariantHandling/VariantHandling.h"

#define MAX_LENGTH_BTNAME 10
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_PHONE_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_I_FILE_PREFIX            App::Core::DeviceListModel::
#include "trcGenProj/Header/DeviceListModel.cpp.trc.h"
#endif

namespace App {
namespace Core {


DeviceListModel* DeviceListModel::m_poDeviceListModel = NULL;

//bool DeviceListModel::bIsDeviceConnectionTimerRunning=FALSE; TODO :will be handled at the end of pairing and unpairing feature completion

/**
*  DeviceListModel::getInstance - function to get singleton instance
*  @return Singleton Instance
*/
DeviceListModel* DeviceListModel::poGetInstance()
{
   if (m_poDeviceListModel == NULL)
   {
      m_poDeviceListModel = new DeviceListModel();
   }

   assert(m_poDeviceListModel);
   return m_poDeviceListModel;
}


/**
*  MediaDatabinding::removeInstance - function to delete the singleton instance
*  @return Singleton Instance
*/
void DeviceListModel::vDeleteInstance()
{
   if (m_poDeviceListModel)
   {
      delete m_poDeviceListModel;
      m_poDeviceListModel = 0;
   }
}


/************************************************************************
*FUNCTION: 	   DeviceListModel
*DESCRIPTION:  Constructor
*PARAMETER: NA
*
*RETURNVALUE: 	NA
*
*HISTORY:
15/09/2017 Shwetha K
Initial Revision

************************************************************************/
DeviceListModel::DeviceListModel()
{
   ETG_TRACE_USR4(("DeviceListModel : DeviceListModel()"));
   ListRegistry::s_getInstance().addListImplementation(LISTID_PHONESELECTION, this);

   m_bSLCStatus = 0;
   u8BTLinkLost = BTPHN_C_HEXA_ZERO;

   if (VariantHandling::getInstance() != NULL)
   {
      _isCabinOnly = (VariantHandling::getInstance()->getVariantType() == VARIANTTYPE_CC);
   }
   else
   {
      _isCabinOnly = false;
   }
}


/************************************************************************
*FUNCTION: 	   ~DeviceListModel
*DESCRIPTION:  Destructor
*PARAMETER: NA
*
*RETURNVALUE: 	NA
*
*HISTORY:
15/09/2017 Shwetha K
Initial Revision

************************************************************************/
DeviceListModel::~DeviceListModel()
{
   ETG_TRACE_USR4(("BTDeviceListModel : ~DeviceListModel()"));
   ListRegistry::s_getInstance().removeListImplementation(LISTID_PHONESELECTION);
}


bool DeviceListModel::Init()
{
   return true;
}


/**
 * @brief   : Incomming message for HMI data model a message component
 * @param   :
 */
bool DeviceListModel::onCourierMessage(const Courier::StartupMsg& /*oMsg*/)
{
#ifdef WINDOWS_SIMULATION
   popupcheckflag = false;
   ActivateApplicationMsg* poActivateMsg = COURIER_MESSAGE_NEW(ActivateApplicationMsg)();
   if (NULL != poActivateMsg)
   {
      /* Posting Courier ActivateApplicationMsg message */
      bool bRetVal = poActivateMsg->Post();
      if (true == bRetVal)
      {
         ETG_TRACE_USR4(("Posted ActivateApplicationMsg"));
      }
      else
      {
         ETG_TRACE_USR4(("Error in Posting ActivateApplicationMsg"));
      }
      IsDeviceListOperationProgress = 0;
      IsParingActive = 0;
      //DeviceListFullResp *poDeviceListFullResp = COURIER_MESSAGE_NEW(DeviceListFullResp)();
      //IsDeviceReseted = 1;
      DeviceTypeConnected = 1;
   }

#endif
   (*m_ListDataSize).mPairedDeviceListSize = BTPHN_C_VALUE_ZERO;
   (*m_ListID).mPairedDeviceListId = LISTID_PHONESELECTION;
   m_ListID.MarkAllItemsModified();
   m_ListID.SendUpdate(true);
   m_ListDataSize.MarkItemModified(ItemKey::ListDataSize::PairedDeviceListSizeItem);
   m_ListDataSize.SendUpdate(true);

   return true;
}


/************************************************************************
*FUNCTION: 	   CreateBTDeviceList
*DESCRIPTION:  Handles the list message and posts courier message ListDateProviderResMsg
which updates the list to HMI
* Each row in device list contains DEvice connected statuus, device name, device profile type , delete device icon
*PARAMETER:		 u8DeviceListSize -> device list size
arPairedDeviceInfo -> vector of btphn_trBTPairedDeviceInfo type
arDeviceSupportInfo -> vector of btphn_trBTDeviceSupportInfo type
*RETURNVALUE: NA

Device type value: 0->Icon_DeviceTypePhone_active#1
1-> Icon_DeviceTypePhone#1
2->Icon_DeviceTypeAudio_active#1
3->Icon_DeviceTypeAudio#1
4->Icon_DeviceTypeBT_active#1
5->Icon_DeviceTypeBT#1

Profile type value : 0->Icon_ProfileBTOff#1
1->Icon_ProfilePhoneAudio#1
2->Icon_ProfilePhone#1
3->Icon_ProfileAudio#1
*HISTORY:
15/09/2017 Shwetha K
Initial Revision

************************************************************************/
void DeviceListModel::OnUpdCreateBTDeviceList
(tU8& u8DeviceListSize, std::vector<DeviceListInformation>& m_arDeviceListInfo)
{
   ETG_TRACE_USR4(("BTDeviceListModel : OnUpdCreateBTDeviceList: List Size %d", u8DeviceListSize));
   ETG_TRACE_USR4(("BTDeviceListModel : OnUpdCreateBTDeviceList: _isCabinOnly %d", _isCabinOnly));
   tU8 rowVal = 0, u8SupportInfoIndex = 0;
   ListDataProviderBuilder listBuilder(LISTID_PHONESELECTION);
   PairedDevice_ListItemData ListData[BTPHN_C_DEFAULT_DEVICE_LIST_SIZE];
   (*m_ListDataSize).mPairedDeviceListSize = u8DeviceListSize;
   m_ListDataSize.MarkItemModified(ItemKey::ListDataSize::PairedDeviceListSizeItem);
   m_ListDataSize.SendUpdate(true);

   (*_isCabinOnlyIcon).mIsCabinonlyIcon = !_isCabinOnly;
   _isCabinOnlyIcon.MarkAllItemsModified();
   _isCabinOnlyIcon.SendUpdate(true);

   (*_isTouchableOnlyIcon).mIsIconTouchable = !_isCabinOnly;
   _isTouchableOnlyIcon.MarkAllItemsModified();
   _isTouchableOnlyIcon.SendUpdate(true);

   if ((u8DeviceListSize != 0) /*&& (arDeviceSupportInfo.size() != 0)*/)
   {
      for (rowVal = BTPHN_C_DEFAULT_VALUE_ZERO; (rowVal < u8DeviceListSize); rowVal++)
      {
         ETG_TRACE_USR4(("OnUpdCreateBTDeviceList: updating device type index for handle : %d", m_arDeviceListInfo[rowVal].GetDeviceHandle()));
         /* Populating the profile information in the device list for
         the entry corresponding to the device handle */
         if (TRUE == m_arDeviceListInfo[rowVal].GetDeviceConnectedStatus())
         {
            ListData[rowVal].mDeviceTypeIndex = 0;
         }
         else
         {
            ListData[rowVal].mDeviceTypeIndex = 1;
         }
         ListData[rowVal].mProfilePhone = false;
         ListData[rowVal].mProfileAudio = false;
         /* Update Profile status to true only when android auto session is not active */
         if ((m_arDeviceListInfo[rowVal].GetHFPStatus() == most_BTSet_fi_types_Extended::T_e8_BTConnectionStatus__e8STATUS_CONNECTED) &&
               (g_SM_SpiSessionStatus != SPI_SESSION_ANDROID_AUTO_ACTIVE) && (g_SM_SpiSessionStatus != SPI_SESSION_CARPLAY_ACTIVE))
         {
            ListData[rowVal].mProfilePhone = true;
         }

         /* Update AVP status to true only when android auto session is not active or device handle is other than android auto device handle when android auto session is active */
         if (m_arDeviceListInfo[rowVal].GetAVPStatus() == most_BTSet_fi_types_Extended::T_e8_BTConnectionStatus__e8STATUS_CONNECTED)
         {
            ListData[rowVal].mProfileAudio = true;
         }
         /*if ((m_arDeviceListInfo[rowVal].GetAVPStatus() == most_BTSet_fi_types_Extended::T_e8_BTConnectionStatus__e8STATUS_CONNECTED) &&
          (!((SmartPhoneClient::getInstance()->sGetAndroidAutoDevBtAddress().compare(m_arDeviceListInfo[rowVal].GetDeviceBtAddress()) == 0) && (SpiSessionStatus == SPI_SESSION_ANDROID_AUTO_ACTIVE))))
           {
              ListData[rowVal].mProfileAudio = true;
           }*/
         /*Setting the BT Icon as Active/Inactive based on the connected Devices of the list*/
         if (TRUE == m_arDeviceListInfo[rowVal].GetDeviceConnectedStatus())
         {
            ETG_TRACE_USR4(("ServiceBT: OnUpdCreateBTDeviceList: Device is connected so BT icon is Green"));
            ListData[rowVal].mBTIcon = true;
            if (_isCabinOnly)
            {
               ETG_TRACE_USR4(("ServiceBT: OnUpdCreateBTDeviceList: Current ActivePhoneDevice = %d %d", m_arDeviceListInfo[rowVal].GetDeviceHandle(), ServiceTel::poGetInstance()->u8GetActiveDev()));
               //Green
               Candera::Color DeviceNameTxtColor(0.62f, 0.85f, 0.03f, 1.0f);
               ListData[rowVal].mDeviceNameTxtColor = DeviceNameTxtColor;
            }
            else
            {
               if (m_arDeviceListInfo[rowVal].GetDeviceHandle() == ServiceTel::poGetInstance()->u8GetActiveDev())
               {
                  ETG_TRACE_USR4(("ServiceBT: OnUpdCreateBTDeviceList: Current ActivePhoneDevice = %d %d", m_arDeviceListInfo[rowVal].GetDeviceHandle(), ServiceTel::poGetInstance()->u8GetActiveDev()));
                  //Green
                  Candera::Color DeviceNameTxtColor(0.62f, 0.85f, 0.03f, 1.0f);
                  ListData[rowVal].mDeviceNameTxtColor = DeviceNameTxtColor;
               }
               else
               {
                  ETG_TRACE_USR4(("ServiceBT: OnUpdCreateBTDeviceList: InActivePhoneDevice = %d %d", m_arDeviceListInfo[rowVal].GetDeviceHandle(), ServiceTel::poGetInstance()->u8GetActiveDev()));
                  //White
                  Candera::Color DeviceNameTxtColor(1.0f, 1.0f, 1.0f, 1.0f);
                  ListData[rowVal].mDeviceNameTxtColor = DeviceNameTxtColor;
               }
            }
         }
         else
         {
            ETG_TRACE_USR4(("ServiceBT: OnUpdCreateBTDeviceList: Device is not connected so BT icon is White"));
            ListData[rowVal].mBTIcon = false;
            Candera::Color DeviceNameTxtColor(1.0f, 1.0f, 1.0f, 1.0f);
            ListData[rowVal].mDeviceNameTxtColor = DeviceNameTxtColor;
         }
         /* Populating device name information to the list*/
         ListData[rowVal].mDeviceName = m_arDeviceListInfo[rowVal].GetDeviceName().c_str();
         ETG_TRACE_USR4(("OnUpdCreateBTDeviceList: GUI update[%d] device index=%d device name =%s", rowVal, ListData[rowVal].mDeviceTypeIndex, m_arDeviceListInfo[rowVal].GetDeviceName().c_str()));
         /*adding each row into device list */
         listBuilder.AddItem(rowVal, 0, "Layer_PhoneSelectionListEntry_1").AddId(rowVal, 1). AddId(rowVal, 2).AddId(rowVal, 3).AddId(rowVal, 4).AddDataBindingUpdater<PairedDevice_ListItemDataBindingSource>(ListData[rowVal]);
      }
   }// end of if ((u8DeviceListSize != 0) && (arDeviceSupportInfo.size() != 0))

   /*Courier message to base class to update the list*/
   tSharedPtrDataProvider dataProvider = listBuilder.CreateDataProvider();
   POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
}


tU8 DeviceListModel::u8FindSupportInfoIndex(tU8 u8PairedDeviceHandle, std::vector<btphn_trBTDeviceSupportInfo>& arDeviceSupportInfo)
{
   ETG_TRACE_USR4(("DeviceListModel::u8FindSupportInfoIndex(), u8PairedDeviceHandle is %d", u8PairedDeviceHandle));
   tU8 i;
   for (i = 0; i < arDeviceSupportInfo.size(); i++)
   {
      ETG_TRACE_USR4(("arDeviceSupportInfo[i].u8DeviceHandle is %d", arDeviceSupportInfo[i].u8DeviceHandle));
      if (u8PairedDeviceHandle == arDeviceSupportInfo[i].u8DeviceHandle)
      {
         ETG_TRACE_USR4(("u8FindSupportInfoIndex:returning u8SupportInfoIndex %d, handle is = %d ", i, u8PairedDeviceHandle));
         return i;
      }
   }
   ETG_TRACE_FATAL(("No Support Info match found for u8PairedDeviceHandle %d", u8PairedDeviceHandle));
   return 0;
}


/************************************************************************
*FUNCTION: 	   u16GetDeviceTypeIndex
*DESCRIPTION:  Updates device type icon  based on device connected status and device class of the phone
*0-> devicetypePhone active icon
 1->devicetypePhone phone icon
 2->devicetypePhone audio active icon
 3->devicetypePhone audio icon
 4->devicetypePhone bt active icon
 5->devicetypePhone bt icon

*PARAMETER:		u8nMajorDeviceClass -> gives the class of device
*				bConnectedStatus -> connected status of the device
*RETURNVALUE: 	 u16DeviceTypeIndex -> device type value to be updted to icon
*
*HISTORY:
15/09/2017 Shwetha K
Initial Revision

************************************************************************/
UInt16 DeviceListModel::u16GetDeviceTypeIndex(tU8 u8nMajorDeviceClass, bool bConnectedStatus)
{
   UInt16 u16DeviceTypeIndex = 0;
   ETG_TRACE_USR4(("DeviceListModel::u8nMajorDeviceClass = %d bConnectedStatus =%d", u8nMajorDeviceClass, bConnectedStatus));
   if (BTPHN_EN_MAJOR_DEVICE_CLASS_PHONE == u8nMajorDeviceClass)
   {
      ETG_TRACE_USR4(("DeviceListModel::u8nMajorDeviceClass = BTPHN_EN_MAJOR_DEVICE_CLASS_PHONE"));
      u16DeviceTypeIndex = (TRUE == bConnectedStatus) ? 0 : 1;
   }
   else if (BTPHN_EN_MAJOR_DEVICE_CLASS_AUDIOVIDEO == u8nMajorDeviceClass)
   {
      ETG_TRACE_USR4(("DeviceListModel::u8nMajorDeviceClass = BTPHN_EN_MAJOR_DEVICE_CLASS_AUDIOVIDEO"));
      u16DeviceTypeIndex = (TRUE == bConnectedStatus) ? 2 : 1;
   }
   else
   {
      ETG_TRACE_USR4(("DeviceListModel::u8nMajorDeviceClass = default"));
      u16DeviceTypeIndex = (TRUE == bConnectedStatus) ? 4 : 5;
   }
   ETG_TRACE_USR4(("DeviceListModel::u16DeviceTypeIndex = %d", u16DeviceTypeIndex));
   return u16DeviceTypeIndex;
}


/************************************************************************
*FUNCTION: 	   u16GetIconProfileIndex
*DESCRIPTION:  Updates profile icon  based on audio streaming support of the device and slc status of the device
*PARAMETER:		bAudioStreamingSupport -> gives the audio sreaming support of device
*
*RETURNVALUE: 	 u16IconProfileIndex -> index value of pfofile icon to be updated to gui
*
*HISTORY:
15/09/2017 Shwetha K
Initial Revision

************************************************************************/
UInt16 DeviceListModel::u16GetIconProfileIndex(bool bAudioStreamingSupport)
{
   ETG_TRACE_USR4(("DeviceListModel::bAudioStreamingSupport= %d", bAudioStreamingSupport));
   UInt16 u16IconProfileIndex = 0;
   if ((TRUE == bGetSLCStatus()) && (TRUE == bAudioStreamingSupport))
   {
      u16IconProfileIndex = 1;//Phone audio icon
   }
   else if (TRUE == bGetSLCStatus())
   {
      u16IconProfileIndex = 2;//Phone icon
   }
   else if (TRUE == bAudioStreamingSupport)
   {
      u16IconProfileIndex = 3;//Audio icon
   }
   else
   {
      u16IconProfileIndex = 0;//BT off icon
   }
   ETG_TRACE_USR4(("DeviceListModel::u16IconProfileIndex= %d", u16IconProfileIndex));
   return u16IconProfileIndex;
}


/************************************************************************
*FUNCTION: 	   UpdateSLCStatus
*DESCRIPTION:  saves SLCstatus to model
*PARAMETER:
*
*RETURNVALUE: 	 bCommandStatus -> SCLStatus received from client handler
*
*HISTORY:
15/09/2017 Shwetha K
Initial Revision

************************************************************************/
void DeviceListModel::OnUpdSLCStatus(tBool& bCommandStatus)
{
   m_bSLCStatus = bCommandStatus;
   ETG_TRACE_USR4(("DeviceListModel::OnUpdSLCStatus,m_bSLCStatus = %d", ETG_CENUM(enBool, (enBool)m_bSLCStatus)));
}


/************************************************************************
*FUNCTION: 	   vEnablePhoneSettings
*DESCRIPTION:  updates the model to enable phone settings
*PARAMETER:
*
*
*RETURNVALUE:
*
************************************************************************/
void DeviceListModel::vEnablePhoneSettings()
{
   uint16 u16VehicleType = vReadVehicleVariant();
   if (PhonebookModelComponent::poGetInstance() != NULL)
   {
      ETG_TRACE_USR4(("vEnablePhoneSettings - call to PhonebookModelComponent for Settings update. "));
      // MSG services are available only when vehicle variant is other than Smart
      if (VEHICLETYPE_SMART != u16VehicleType)
      {
         // 1st paramaeter :Enable/disable ,2nd parameter: value
         PhonebookModelComponent::poGetInstance()->vChangeActivationOfMessageSetting(true);//message settings enabled.
      }
      PhonebookModelComponent::poGetInstance()->vChangeActivationOfSortOrderSetting(true); //Ringtone settings to be enabled. Default :First Name
      PhonebookModelComponent::poGetInstance()->vChangeActivationOfRingtoneSetting(true);//Sort order settings to be enabled. Default :ON

      PhonebookModelComponent::poGetInstance()->OnUpdSettingsList(); //updating the list builder for phonesettings
   }
   else
   {
      ETG_TRACE_USR4(("vEnablePhoneSettings - Failed to get PhoneBookModelComponent Instance. "));
   }
}


/************************************************************************
*FUNCTION: 	   vDisablePhoneSettings
*DESCRIPTION:  updates the model to disable phone settings
*PARAMETER:
*
*
*RETURNVALUE:
*
************************************************************************/
void DeviceListModel::vDisablePhoneSettings()
{
   uint16 u16VehicleType = vReadVehicleVariant();
   if (PhonebookModelComponent::poGetInstance() != NULL)
   {
      ETG_TRACE_USR4(("vdisablePhoneSettings - call to PhonebookModelComponent for Settings update. "));
      if (VEHICLETYPE_SMART != u16VehicleType)
      {
         // 1st paramaeter :Enable/disable ,2nd parameter: value
         PhonebookModelComponent::poGetInstance()->vChangeActivationOfMessageSetting(false);//message settings to be disabled. Default: ON
      }
      PhonebookModelComponent::poGetInstance()->vChangeActivationOfSortOrderSetting(false); //Ringtone settings to be disabled. Default :First Name
      PhonebookModelComponent::poGetInstance()->vChangeActivationOfRingtoneSetting(false);//Sort order settings to be disabled. Default :ON
      PhonebookModelComponent::poGetInstance()->OnUpdSettingsList(); //updating the list builder for phonesettings
   }
   else
   {
      ETG_TRACE_USR4(("vDisablePhoneSettings - Failed to get PhoneBookModelComponent. "));
   }
}


/************************************************************************
*FUNCTION: 	   vReadVehicleVariant
*DESCRIPTION:  Reads the vehicle variant from DP
*
*RETURNVALUE: 	 u16VehicleType , vehicle type
*
************************************************************************/

uint16 DeviceListModel::vReadVehicleVariant()
{
   uint16 u16VehicleType = VEHICLETYPE_TRUCK;
   //Read VehicleType from KDS
   if (PhoneDataPoolConfig::getInstance())
   {
      ETG_TRACE_USR4(("Retrive vehicleType information from KDS"));
      u16VehicleType = PhoneDataPoolConfig::getInstance()->getKDSVehicleType();
      ETG_TRACE_USR4(("getKDSVehicleType, Vehicle type : %d", u16VehicleType));
   }
   else
   {
      ETG_TRACE_USR4(("Error in getting PhoneDataPoolConfig instance"));
   }

   return u16VehicleType;
}


/************************************************************************
*FUNCTION: 	   OnUpdDeviceConnected
*DESCRIPTION:  Updates data binding band supported status to GUI
* call bUpdateBTTelltaleUpdate whenever OnUpdDeviceConnected is being called from service to update BTTele databinding values
*PARAMETER:		szDeviceName	:connected device name to be updated to GUI
enDeviceStatus	: device connected status as received from service
*
*RETURNVALUE: 	 None
*
*HISTORY:
15/09/2017 Shwetha K
05/02/2019 Sai Prabha
Initial Revision v 0.1
28/02/2019 Sai Prabha
Initial Revision v 0.2 Changes : Bug 280289

************************************************************************/

void DeviceListModel::OnUpdDeviceConnected(const tU8& enDeviceStatus, const tU8& NoOfHfpDevices, const tU8& NoOfConnectedDevices)
{
   ETG_TRACE_USR4(("OnUpdDeviceConnected::enDeviceStatus %d", enDeviceStatus));
   ETG_TRACE_USR4(("OnUpdDeviceConnected::IsDeviceConnected %d", IsDeviceConnected));
   ETG_TRACE_USR4(("OnUpdDeviceConnected::NoOfHfpDevices :%d", NoOfHfpDevices));
   /* Condition check for device connection */
   switch (enDeviceStatus)
   {
      case BTPHN_EN_DEVICE_CONNECTED:

         u8BTLinkLost = BTPHN_C_HEXA_ZERO;//Update BTLinkLost Icon Status

         IsDeviceConnected = BTPHN_C_VALUE_ONE;

         ETG_TRACE_USR4(("OnUpdDeviceConnected::NoOfHfpDevices %d", NoOfHfpDevices));
         ETG_TRACE_USR4(("OnUpdDeviceConnected::NoOfConnectedDevices %d", NoOfConnectedDevices));
         if (NoOfHfpDevices > 0 && NoOfConnectedDevices > 0)
         {
            DeviceTypeConnected = BTPHN_C_VALUE_ONE;
            ETG_TRACE_USR4(("OnUpdDeviceConnected::DeviceTypeConnected %d", DeviceTypeConnected));
         }

         /*Error Handling done in PL*/
         IsDeviceListOperationProgress = BTPHN_C_VALUE_ZERO;
         break;

      case BTPHN_EN_DEVICE_CONNECTED_OSD :
         u8BTLinkLost = BTPHN_C_HEXA_ZERO;//Update BTLinkLost Icon Status

         IsDeviceConnected = BTPHN_C_VALUE_ONE;
         ETG_TRACE_USR4(("OnUpdDeviceConnected::IsDeviceConnected %d", IsDeviceConnected));

         ETG_TRACE_USR4(("OnUpdDeviceConnected::NoOfHfpDevices %d", NoOfHfpDevices));
         ETG_TRACE_USR4(("OnUpdDeviceConnected::NoOfConnectedDevices %d", NoOfConnectedDevices));
         if (NoOfHfpDevices > 0 && NoOfConnectedDevices > 0)
         {
            DeviceTypeConnected = BTPHN_C_VALUE_ONE;
            ETG_TRACE_USR4(("OnUpdDeviceConnected::DeviceTypeConnected %d", DeviceTypeConnected));
         }

         break;

      /* Condition check for device disconnection */
      case BTPHN_EN_DEVICE_DISCONNECTED:

         u8BTLinkLost = BTPHN_C_HEXA_0X01F;//Update BTLinkLost Icon Status
         // the entry into call list view is decided on the status of noofHfpdevices and noofconnecteddevices
         if (NoOfHfpDevices == 0 && NoOfConnectedDevices == 0)
         {
            ETG_TRACE_USR4(("OnUpdDeviceConnected::DeviceTypeConnected %d", DeviceTypeConnected));
            //DeviceTypeConnected- for both the profiles
            DeviceTypeConnected = BTPHN_C_VALUE_ZERO;
         }
         if (NoOfHfpDevices == 0)
         {
            ETG_TRACE_USR4(("OnUpdDeviceConnected::IsDeviceConnected %d", IsDeviceConnected));
            // IsDeviceConnected - only for hfp connection
            IsDeviceConnected = BTPHN_C_VALUE_ZERO;
         }

         /* Updating Inband Ringtone binding property for Check*/
         if (PhonebookModelComponent::poGetInstance() != NULL)
         {
            ETG_TRACE_USR4(("OnUpdDeviceConnected::vSetBandSupport"));
            PhonebookModelComponent::poGetInstance()->vSetBandSupport(BTPHN_C_VALUE_ONE);
            /* Fix for bug 651513 - User settings should not be overridden on device disconnection */
            //PhonebookModelComponent::poGetInstance()->vSetUsePhoneRingtone(BTPHN_C_VALUE_ONE);
         }

         break;
      // when a particular profile(hfp/avp) is disconnected, device-no change will be the status
      //DeviceTypeConnected is zero when no devices are connected, IsDeviceConnected will be zero only when no of hfp connected devices are zero
      // the entry into call list view is decided on the status of noofHfpdevices and noofconnecteddevices
      case BTPHN_EN_DEVICE_NO_CHANGE :

         if (NoOfHfpDevices > 0 && NoOfConnectedDevices > 0)
         {
            DeviceTypeConnected = BTPHN_C_VALUE_ONE;
            ETG_TRACE_USR4(("OnUpdDeviceConnected::DeviceTypeConnected %d", DeviceTypeConnected));
         }
         else
         {
            DeviceTypeConnected = BTPHN_C_VALUE_ZERO;
            ETG_TRACE_USR4(("OnUpdDeviceConnected::DeviceTypeConnected %d", DeviceTypeConnected));
         }
         if (NoOfHfpDevices > 0)
         {
            IsDeviceConnected = BTPHN_C_VALUE_ONE;
            ETG_TRACE_USR4(("OnUpdDeviceConnected::IsDeviceConnected %d", IsDeviceConnected));
         }
         else if (NoOfHfpDevices == 0)
         {
            IsDeviceConnected = BTPHN_C_VALUE_ZERO;
            ETG_TRACE_USR4(("OnUpdDeviceConnected::IsDeviceConnected %d", IsDeviceConnected));
         }

         break;

      default:

         /* To be taken care later in future releases */
         ETG_TRACE_USR4(("DeviceListModel : OnUpdDeviceConnected(), "
                         "DeviceStatusRespMessage"
                         ": Unknown Value "));

         break;
   }
   CallHistoryModel* poCallHistoryModel = CallHistoryModel::poGetInstance();
   if (NULL != poCallHistoryModel)
   {
      poCallHistoryModel->bUpdateBTTelltaleUpdate();
   }
}


/************************************************************************
*FUNCTION: 	   OnUpdConnectedDeviceName
*DESCRIPTION:  Called from client when there is device status changed or when Call list screen is loaded
				Updates data binding connected bt device nameto gui
*PARAMETER : szDeviceName -> connected bt device name
*RETURNVALUE: 	 None
*
*HISTORY:
15/09/2017 Shwetha K
Initial Revision

************************************************************************/
void DeviceListModel::OnUpdConnectedDeviceName(const std::string& szDeviceName)
{
   if (NULL == szDeviceName.c_str())
   {
      ETG_TRACE_USR4(("DeviceListModel : OnUpdConnectedDeviceName(NULL)"));
      return;
   }
   else
   {
      ETG_TRACE_USR4(("DeviceListModel : OnUpdConnectedDeviceName(%s)", szDeviceName.c_str()));
      (*oConnectedActiveBTDeviceName).mDeviceName = (Candera::String)szDeviceName.c_str();
      oConnectedActiveBTDeviceName.MarkAllItemsModified();
      oConnectedActiveBTDeviceName.SendUpdate(true);
   }
}


/************************************************************************
*FUNCTION: 	   OnUpdActPassiveDeviceName
*DESCRIPTION:  Called from client when there is device status changed or when Call list screen is loaded
				Updates data binding connected bt device nameto gui
*PARAMETER : szDeviceName -> connected bt device name
*RETURNVALUE: 	 None
*
*HISTORY:
28/02/2018 Sai Prabha
Initial Revision
************************************************************************/
void  DeviceListModel:: OnUpdActPassiveDeviceName(const std::string& szDeviceName)
{
   std::string l_str_devicenameconcat = "...";
   std::string l_str_devicename = szDeviceName;
   ETG_TRACE_USR4((" Connectedname Size : %d", l_str_devicename.size()));
   if (MAX_LENGTH_BTNAME <= l_str_devicename.size())
   {
      ETG_TRACE_USR4((" Connectedname > 10 : %d", l_str_devicename.size()));
      l_str_devicename = l_str_devicename.substr(0, MAX_LENGTH_BTNAME);
      ETG_TRACE_USR4((" Connectedname > 10 : %s", l_str_devicename.c_str()));
      l_str_devicename = l_str_devicename.append(l_str_devicenameconcat);
   }
   ETG_TRACE_USR4(("DeviceListModel : OnUpdActPassiveDeviceName((%s)", l_str_devicename.c_str()));
   (*oConnectedBTDeviceName).mDeviceName = (Candera::String)l_str_devicename.c_str();
   oConnectedBTDeviceName.MarkAllItemsModified();
   oConnectedBTDeviceName.SendUpdate(true);
}


/************************************************************************
*FUNCTION: 	   OnUpdChangeRingtoneSettingsReq
*RETURNVALUE: 	 None
*
************************************************************************/
tU16 DeviceListModel::OnUpdChangeRingtoneSettingsReq()
{
   Courier::UInt8 u8RingToneID = BTPHN_C_VALUE_ZERO;
   if (NULL != PhonebookModelComponent::poGetInstance())
   {
      if (BTPHN_C_CHECKED == PhonebookModelComponent::poGetInstance()->u8GetUsePhoneRingtone())
      {
         u8RingToneID = BTPHN_C_VALUE_TWO;
      }
      else
      {
         u8RingToneID = BTPHN_C_VALUE_ONE;
      }
   }
   POST_MSG((COURIER_MESSAGE_NEW(ChangeRingtoneReq)(u8RingToneID)));
   return u8RingToneID;
}


/************************************************************************
*FUNCTION: 	   OnUpdNewDeviceInfoResult
*DESCRIPTION:  Updates Device name and the pin of the pairing device to GUI
*13/11/2017 vke6kor
Initial Revision

************************************************************************/
void DeviceListModel::OnUpdNewDeviceInfoResult(std::string copcDeviceName, std::string copcPin, btphn_tenBTPairingType const& corfenPairingType)
{
   ETG_TRACE_USR4(("DeviceListModel : OnUpdNewDeviceInfoResult() corfenPairingType = %d", corfenPairingType));
   ETG_TRACE_USR4(("DeviceListModel : OnUpdNewDeviceInfoResult() copcPin = %s", copcPin.c_str()));
   ETG_TRACE_USR4(("DeviceListModel : OnUpdNewDeviceInfoResult() copcDeviceName = %s", copcDeviceName.c_str()));
   POST_MSG((COURIER_MESSAGE_NEW(AddNewDeviceResp)((Candera::String)copcDeviceName.c_str(), (Candera::String)copcPin.c_str(), corfenPairingType)));

   (*oPairingDeviceInfo).mPairingPIN = (Candera::String)copcPin.c_str();
   (*oPairingDeviceInfo).mDeviceName = (Candera::String)copcDeviceName.c_str();
   oPairingDeviceInfo.MarkAllItemsModified();
   oPairingDeviceInfo.SendUpdate(true);
#ifdef WINDOWS_SIMULATION
   /*std::cout << "OnUpdNewDeviceInfoResult\n";
   std::cout << copcPin.c_str() << "\n";
   std::cout << copcDeviceName.c_str() << "\n";*/
#endif
}


/*****************************************************************************
* @brief: Proprty Status Update from clienthandler for OnUpdVehicleBTFriendlyNameStatus
*18.09.2017: vke6kor
*Rev 0.1: Initial Version
*****************************************************************************/
void DeviceListModel::OnUpdVehicleBTFriendlyNameStatus(std::string copcVehicleBTName)
{
   ETG_TRACE_USR4(("DeviceListModel : OnUpdVehicleBTFriendlyNameStatus()"));
   ETG_TRACE_USR4(("DeviceListModel : OnUpdVehicleBTFriendlyNameStatus() copcVehicleBTName = %s", copcVehicleBTName.c_str()));
   POST_MSG((COURIER_MESSAGE_NEW(VehicleBTNameResp)((Candera::String)copcVehicleBTName.c_str())));

   (*oFriendlyName).mHUDName = (Candera::String)copcVehicleBTName.c_str();

   oFriendlyName.MarkAllItemsModified();
   tBool  bRetVal = oFriendlyName.SendUpdate(true);

   if (true == bRetVal)
   {
      ETG_TRACE_USR4(("DeviceListModel : OnUpdVehicleBTFriendlyName(),BTSettingsList - Vehicle Name Updated"));
   }
   else
   {
      ETG_TRACE_ERR(("bDeviceListModel : OnUpdVehicleBTFriendlyName(), BTSettingsList - Vehicle Name Update error"));
   }
#ifdef WINDOWS_SIMULATION
   //std::cout << "OnUpdVehicleBTFriendlyNameStatus/n";
#endif
}


/*****************************************************************************
* @brief: Proprty Status Update from clent handlerfor OnUpdPairingPinStatus
*18.09.2017: vke6kor
*Rev 0.1: Initial Version
*****************************************************************************/
void DeviceListModel::OnUpdPairingPinStatus(std::string copcPinNumber)
{
   POST_MSG((COURIER_MESSAGE_NEW(PairingPinResp)((Candera::String)copcPinNumber.c_str())));
   (*oPairingDeviceInfo).mPairingPIN = (Candera::String)copcPinNumber.c_str();
   oPairingDeviceInfo.MarkAllItemsModified();

   oFriendlyName.MarkAllItemsModified();
   oFriendlyName.SendUpdate(true);

   tBool bRetVal = oPairingDeviceInfo.SendUpdate(true);
   if (true == bRetVal)
   {
      ETG_TRACE_USR4(("DeviceListModel : OnUpdPairingPinStatus(),Updated Binding Property - Pairing PIN"));
   }
   else
   {
      ETG_TRACE_ERR(("DeviceListModel : OnUpdPairingPinStatus(),Error in Updating Binding Property - PIN Number"));
   }
#ifdef WINDOWS_SIMULATION


   //std::cout << "pairing initiate\n";

   //std::cout << "OnUpdNewDeviceInfoResult called\n";
   //IsHomeScreenActive = 0;
   //vUpdateDeviceConnected("Device_name", BTPHN_EN_DEVICE_CONNECTED);
   DeviceList_Stub* poHelperInstance = DeviceList_Stub::getInstance();
   if (!(poHelperInstance == NULL))
   {
      poHelperInstance->vPairingStub();
   }


#endif
}


/*****************************************************************************
* @brief: Proprty Status Update from clienthandler for OnUpdDeviceListfull
to check the Device list Full condition
*13.11.2017: vke6kor
*Rev 0.1: Initial Version
*****************************************************************************/
void DeviceListModel::OnUpdDeviceListfull()
{
   ETG_TRACE_USR4(("DeviceListModel : OnUpdDeviceListfull()\t"));

   POST_MSG((COURIER_MESSAGE_NEW(DeviceListFullResp)()));
}


/*****************************************************************************
* @brief: This function is been called when by service when device deletion is triggered fro device list
* It updates deviceinfo and pairingdevice info databinding values with deleted device name

*29.11.2017: kis8kor
*Rev 0.1: Initial Version
*****************************************************************************/
void DeviceListModel::OnUpdDeviceName(const std::string& szDeviceName)
{
   std::string l_str_devicename =  szDeviceName.c_str();

   std::string l_str_devicenameconcat = "...";
   ETG_TRACE_USR4((" Connectedname Size : %d", l_str_devicename.size()));

   if (MAX_LENGTH_BTNAME <= l_str_devicename.size())
   {
      ETG_TRACE_USR4((" Connectedname > 10 : %d", l_str_devicename.size()));
      l_str_devicename = l_str_devicename.substr(0, MAX_LENGTH_BTNAME);
      ETG_TRACE_USR4((" Connectedname > 10 : %s", l_str_devicename.c_str()));
      l_str_devicename = l_str_devicename.append(l_str_devicenameconcat);
   }
   ETG_TRACE_USR4(("DeviceListModel : OnUpdDeviceName() szDeviceName = %s", l_str_devicename.c_str()));
   /* Updating Deleting Device Name*/
   (*oDeviceInfo).mDeviceName = l_str_devicename.c_str();
   oDeviceInfo.MarkAllItemsModified();
   oDeviceInfo.SendUpdate(true);

   (*oPairingDeviceInfo).mDeviceName = l_str_devicename.c_str();
   oPairingDeviceInfo.MarkAllItemsModified();
   oPairingDeviceInfo.SendUpdate(true);
}


void DeviceListModel::OnUpdDisconnectedDeviceName(const std::string& szDeviceName)
{
   std::string l_str_devicename =  szDeviceName.c_str();
   std::string l_str_devicenameconcat = "...";
   ETG_TRACE_USR4((" Connectedname Size : %d", l_str_devicename.size()));
   if (MAX_LENGTH_BTNAME <= l_str_devicename.size())
   {
      ETG_TRACE_USR4((" Connectedname > 10 : %d", l_str_devicename.size()));
      l_str_devicename = l_str_devicename.substr(0, MAX_LENGTH_BTNAME);
      ETG_TRACE_USR4((" Connectedname > 10 : %s", l_str_devicename.c_str()));
      l_str_devicename = l_str_devicename.append(l_str_devicenameconcat);
   }
   ETG_TRACE_USR4(("DeviceListModel : OnUpdDeviceName() szDeviceName = %s", l_str_devicename.c_str()));
   /* Updating Deleting Device Name*/
   (*oDisconnectedBTDeviceName).mDeviceName = l_str_devicename.c_str();
   oDisconnectedBTDeviceName.MarkAllItemsModified();
   oDisconnectedBTDeviceName.SendUpdate(true);
}


void DeviceListModel::OnUpdIntercomStatus(bool ISupported, bool INotSupported)
{
   ETG_TRACE_USR4(("ServiceBT::OnUpdIntercomStatus"));
   (*oIntercomSupportstatus).mIntercom_Supported = ISupported ;
   (*oIntercomSupportstatus).mIntercom_NotSupported = INotSupported ;
   oIntercomSupportstatus.MarkAllItemsModified();
   oIntercomSupportstatus.SendUpdate(true);
}


#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
void DeviceListModel::OnUpdGadgetStatus(uint8 u8Active_Source)
{
   ETG_TRACE_USR4(("ServiceBT::OnUpdGadgetStatus u8Active_Source %d", u8Active_Source));

   if ((u8Active_Source == PHONEONE) || (u8Active_Source == PHONETWO))
   {
      ETG_TRACE_USR4(("OnUpdGadgetStatus Phone source"));
      /* If android auto session or carplay is active, update the value of 3 as to show phone not connected icon else show respective device handle icon */
      if ((g_SM_SpiSessionStatus == SPI_SESSION_ANDROID_AUTO_ACTIVE) || (g_SM_SpiSessionStatus == SPI_SESSION_CARPLAY_ACTIVE))
      {
         /* Update with value of 3 as to show phone not connected icon*/
         (*_phoneGadgetInfoData).mPhoneHandle = 3;
      }
      else
      {
         /* Update the respective device handle icon index*/
         (*_phoneGadgetInfoData).mPhoneHandle = 5;
      }
      (*_PhoneButtonText).mSource = Candera::String(TextId_0x13A5);
      (*_Intercomenable).mIntercomenable = true;
   }
   else if (u8Active_Source == INTERCOM)
   {
      ETG_TRACE_USR4(("OnUpdGadgetStatus Intercom source"));
      (*_phoneGadgetInfoData).mPhoneHandle = 4;
      (*_PhoneButtonText).mSource = Candera::String(TextId_0x14B0);

      (*_Intercomenable).mIntercomenable = false;
   }
   else
   {
      ETG_TRACE_USR4(("OnUpdGadgetStatus invalid source"));
   }

   _phoneGadgetInfoData.MarkAllItemsModified();
   bool bRetval = _phoneGadgetInfoData.SendUpdate(true);

   if (true == bRetval)
   {
      ETG_TRACE_USR4(("_phoneGadgetInfoData databinding update success"));
   }
   else
   {
      ETG_TRACE_USR4(("_phoneGadgetInfoData databinding update failed"));
   }
   _PhoneButtonText.MarkAllItemsModified();
   bRetval = _PhoneButtonText.SendUpdate(true);
   if (true == bRetval)
   {
      ETG_TRACE_USR4(("PhoneButtonText databinding update success"));
   }
   else
   {
      ETG_TRACE_USR4(("PhoneButtonText databinding update failed"));
   }

   _Intercomenable.MarkAllItemsModified();
   bRetval = _Intercomenable.SendUpdate(true);

   if (true == bRetval)
   {
      ETG_TRACE_USR4(("PhoneButtonText databinding update success"));
   }
   else
   {
      ETG_TRACE_USR4(("PhoneButtonText databinding update failed"));
   }
}


#endif

#ifdef VARIANT_S_FTR_ENABLE_BLUETOOTH_ON_OFF
void DeviceListModel::OnUpdBTONOFFStatus(bool BTstatus)
{
   ETG_TRACE_USR4(("ServiceBT::OnUpdBTONOFFStatus BTstatus = %d", BTstatus));
   (*oBTONOFF).mBTONOFFStatus = BTstatus;
   oBTONOFF.MarkAllItemsModified();
   oBTONOFF.SendUpdate(true);
}


#endif

#ifdef WINDOWS_SIMULATION
bool DeviceListModel::onCourierMessage(const AddNewDeviceReq& corfoMsg)
{
   DeviceList_Stub* poHelperInstance = DeviceList_Stub::getInstance();

   if (!(poHelperInstance == NULL))
   {
      poHelperInstance->vAddDeviceStub();
   }

   return true;
}


bool DeviceListModel::onCourierMessage(const DeviceListReq& oMsg)
{
   DeviceList_Stub* poHelperInstance = DeviceList_Stub::getInstance();
   if (!(poHelperInstance == NULL))
   {
      poHelperInstance->vDeviceListStub();
   }
   return true;
}


#endif
}


}
