/***************************************************************************
* Copyright(c) 2017-2020 Robert Bosch Car Multimedia GmbH
* This software is property of Robert Bosch GmbH.
***************************************************************************/
/*****************************************************************************
* (c) Robert Bosch Car Multimedia GmbH
*
* Add details here
****************************************************************************/

#include "hall_std_if.h"
#include "ServiceTel.h"
#include "ServiceBT.h"
#include "AppHmiPhone_Helper.h"

//#include "tclMostTelFiIf.h"
#include "../PhoneBook/ServicePhonebook.h"
#include "../VehicleSrvClient/VehicleSrvClient.h"  //CAN Imp
#include "Core/Microphone/MicClientHandler.h"
#include "Common/VariantHandling/VariantHandling.h"


#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_PHONE_HALL
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ServiceTel.cpp.trc.h"
#endif
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
#include "App/Core/Intercom/IntercomClienthandler.h"
#endif
using namespace asf::core;
using namespace ::MOST_Tel_FI;
using namespace ::most_Tel_fi_types;
using namespace ::most_Tel_fi_types_Extended;

namespace App {
namespace Core {

ServiceTel* ServiceTel::_poServiceTel = NULL;

//Add include files here

/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
ServiceTel::ServiceTel() :
   rfoCallHistoryModel(*CallHistoryModel::poGetInstance()),
   _most_tel_fi(MOST_Tel_FIProxy::createProxy("btTelFiPort", *this)),
   rfoDeviceListModel(*DeviceListModel::poGetInstance()),
   rfoPhoneCallModel(*PhoneCallModel::poGetInstance()),
   u8IncomingCallInstance(BTPHN_C_CALL_INSTANCE_INVALID),
   bIsDialCallTimerRunning(FALSE),
   u8NumSwapRequests(BTPHN_C_DEFAULT_VALUE_ZERO),
   bIsConference(FALSE),
   u8CurrentCallInstance(BTPHN_C_CALL_INSTANCE_INVALID),
   u8PreviousCallInstance(BTPHN_C_CALL_INSTANCE_INVALID),
   u8CurrentActiveCallInstance_dtmf(BTPHN_C_CALL_INSTANCE_INVALID),
   bIsCallRejected(FALSE),
   bIsEndCallTimerRunning(FALSE),
   copcUnknownText(BTPHN_C_CONTACT_UNKNOWN),
   bMicrophoneState(FALSE),
   u8NumActiveCalls(0),
   u32EndCallTimerDuration(BTPHN_C_END_CALL_TIMER_DURATION),
   hDialCallTimer(0),
   hEndCallTimer(0),
   m_bBTVoiceActive(FALSE),
   m_bBTRecSupported(FALSE),
   m_nActiveCall(0),
   m_u8BTVRSessionActive(0),
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
   m_bMissedBTCall(false),
#endif
   bCallReqFlag(FALSE),
   m_u8ActiveDev(0),
   m_index(0),
   m_u8PassiveDevHandle(0),
   bIsIncomingCallPopupActive(false),
   bIsOutgoingCallPopupActive(false),
   bIsOngoingCallActive(false),
   bIsAcceptCallRequestedWhenVREnds(false),
   bIsIdlestatus(false),
   u8ChargeStatus(6),
   u8NetworkStrength(6),
   strDeviceNameCurrent(" ")

{
   // Initialize _isCabinOnly based on the VariantType
   if (VariantHandling::getInstance() != NULL)
   {
      _isCabinOnly = (VariantHandling::getInstance()->getVariantType() == VARIANTTYPE_CC);
   }
   else
   {
      _isCabinOnly = false; // Default to false if VariantHandling is not available
   }

   if (_most_tel_fi.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _most_tel_fi->getPortName());
   }
   for (tU8 u8Index = BTPHN_C_DEFAULT_VALUE_ZERO; u8Index < BTPHN_C_NUM_CALLS_SUPPORTED; u8Index++)
   {
      arCallStatusInfo[u8Index].u16CallInstance = u8Index;
      arCallStatusInfo[u8Index].enPhoneType = BTPHN_EN_GENERIC;
   }
   for (tU8 u8Index = BTPHN_C_DEFAULT_VALUE_ZERO; u8Index < MAX_HFP_CONNECTION; u8Index++)
   {
      BtvrExtendedStatus[u8Index].u8DeviceHandle = 0;
      BtvrExtendedStatus[u8Index].enSiriAvailabilityState = most_Tel_fi_types_Extended::T_e8_TelSiriAvailabilityState__e8NOT_AVAILABLE;
   }
   if (OSAL_s32TimerCreate(vOnDialCallTimerExpiry, this, &hDialCallTimer) == OSAL_ERROR)
   {
      ETG_TRACE_FATAL(("hDialCallTimer not created"));
   }
   if (OSAL_s32TimerCreate(vOnEndCallTimerExpiry, this, &hEndCallTimer) == OSAL_ERROR)
   {
      ETG_TRACE_FATAL(("hEndCallTimer not created"));
   }
   vSet_BTVoiceRecognition(false);
   bBTVRViewShown = FALSE;
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
ServiceTel::~ServiceTel()
{
   //Add your code here
}


void ServiceTel::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR2(("ServiceTel registerProperties"));
}


void ServiceTel::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR2(("ServiceTel deregisterProperties"));
}


void ServiceTel::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR2(("ServiceTel onAvailable"));

   if (!_isCabinOnly)
   {
      if (proxy == _most_tel_fi && _most_tel_fi.get())
      {
         _most_tel_fi->sendSLCStatusUpReg(*this);
         _most_tel_fi->sendCellStateofChargeUpReg(*this);
         _most_tel_fi->sendCellSignalStrengthUpReg(*this);
         _most_tel_fi->sendCellNetworkStatusUpReg(*this);
         _most_tel_fi->sendInComingCallEventUpReg(*this);
#ifndef VARIANT_S_FTR_ENABLE_PHONECALLMANAGER_SUPPORT
         _most_tel_fi->sendCallStatusNoticeUpReg(*this);
#endif
#ifdef VARIANT_S_FTR_ENABLE_PHONECALLMANAGER_SUPPORT
         _most_tel_fi->sendCallStatusNoticeExtendedUpReg(*this);
#endif
         _most_tel_fi->sendMicrophoneMuteStateUpReg(*this);
         _most_tel_fi->sendActiveCallinHandsetUpReg(*this);
         _most_tel_fi->sendNetworkOperatorUpReg(*this);
         _most_tel_fi->sendVehicleAudioUpReg(*this);
         _most_tel_fi->sendBTDeviceVoiceRecognitionExtendedUpReg(*this);
         _most_tel_fi->sendActivePhoneDeviceUpReg(*this);
         _most_tel_fi->sendCellDeviceNameUpReg(*this);
#ifdef VARIANT_S_FTR_ENABLE_PHONECALLMANAGER_SUPPORT
         _most_tel_fi->sendSystemWideRingtonesListUpReg(*this);
         _most_tel_fi->sendDeviceRingtoneListUpReg(*this);
         _most_tel_fi->sendCallOperationFailedEventUpReg(*this);
         _most_tel_fi->sendSendDTMFFailedEventUpReg(*this);
         _most_tel_fi->sendAudioTransferFailedEventUpReg(*this);
#endif
      }
   }
}


void ServiceTel::onUnavailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR2(("ServiceTel onUnavailable"));
   if (!_isCabinOnly)
   {
      if (proxy == _most_tel_fi)
      {
         _most_tel_fi->sendSLCStatusRelUpRegAll();
         _most_tel_fi->sendCellStateofChargeRelUpRegAll();
         _most_tel_fi->sendCellSignalStrengthRelUpRegAll();
         _most_tel_fi->sendCellNetworkStatusRelUpRegAll();
         _most_tel_fi->sendInComingCallEventRelUpRegAll();
#ifndef VARIANT_S_FTR_ENABLE_PHONECALLMANAGER_SUPPORT
         _most_tel_fi->sendCallStatusNoticeRelUpRegAll();
#endif
#ifdef VARIANT_S_FTR_ENABLE_PHONECALLMANAGER_SUPPORT
         _most_tel_fi->sendCallStatusNoticeExtendedRelUpRegAll();
#endif
         _most_tel_fi->sendMicrophoneMuteStateRelUpRegAll();
         _most_tel_fi->sendActiveCallinHandsetRelUpRegAll();
         _most_tel_fi->sendNetworkOperatorRelUpRegAll();
         _most_tel_fi->sendVehicleAudioRelUpRegAll();
         _most_tel_fi->sendBTDeviceVoiceRecognitionExtendedRelUpRegAll();
         _most_tel_fi->sendActivePhoneDeviceRelUpRegAll();
         _most_tel_fi->sendCellDeviceNameRelUpRegAll();
#ifdef VARIANT_S_FTR_ENABLE_PHONECALLMANAGER_SUPPORT
         _most_tel_fi->sendSystemWideRingtonesListRelUpRegAll();
         _most_tel_fi->sendDeviceRingtoneListRelUpRegAll();
         _most_tel_fi->sendCallOperationFailedEventRelUpRegAll();
         _most_tel_fi->sendSendDTMFFailedEventRelUpRegAll();
         _most_tel_fi->sendAudioTransferFailedEventRelUpRegAll();
#endif
      }
   }
}


/*******************************Implementation of Manual code*****************************************/

/************************************************************************
*FUNCTION: 		 onAcceptCallError
*DESCRIPTION: 	 Handles Callback from Phone Service For AcceptCall Error
*
*PARAMETER:
* 					 proxy: MOST_Tel_FIProxy shared pointer
*					Error: AcceptCallError obj
*
*RETURNVALUE: 	 void
*
*
*HISTORY:
*revision 0.1	ess7kor	11/08/2017
*************************************************************************/
void ServiceTel::onAcceptCallError(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::AcceptCallError >& error)
{
   ETG_TRACE_USR4(("ServiceTel : onAcceptCallError()\t"));
   bCallReqFlag = TRUE;
}


/************************************************************************
*FUNCTION: 		 onAcceptCallResult
*DESCRIPTION: 	 Handles Callback from Phone Service For AcceptCall Result
*
*PARAMETER:
* 					 proxy: MOST_Tel_FIProxy shared pointer
*					result: AcceptCallResult obj
*
*RETURNVALUE: 	 void
*
*
*HISTORY:
*revision 0.1	ess7kor	11/08/2017
*13.08.2019 Rev 0.2, Palani K, SYSFL-5565
*************************************************************************/
void ServiceTel::onAcceptCallResult(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::AcceptCallResult >& result)
{
   ETG_TRACE_USR4(("ServiceTel : onAcceptCallResult()\t"));
   bCallReqFlag = FALSE;
   ETG_TRACE_USR4(("Call Instance = %d\t",	result->getU16CallInstance()));
   ETG_TRACE_USR4(("Call Status = %d\t",
                   ETG_CENUM(btphn_tenPhoneCallStatus, (btphn_tenPhoneCallStatus)result->getE8CallStatus())));
   ETG_TRACE_USR4(("Using Vehicle Audio = %d", result->getBUsingVehicleAudio()));

   if (m_u8PassiveDevHandle != 0)
   {
      /* On accepting passive incoming call, abort active call */
      if (_most_tel_fi && _most_tel_fi.get())
      {
         if (arCallStatusInfo[u8CurrentCallInstance].enCallStatus != BTPHN_EN_IDLE)
         {
            _most_tel_fi->sendHangUpCallStart(*this, u8PreviousCallInstance);
            ETG_TRACE_USR4(("ServiceTel::onAcceptCallResult sendHangUpCallStart()"));
            arCallStatusInfo[u8CurrentCallInstance].bIsContactInfoReq = FALSE;
            arCallStatusInfo[u8CurrentCallInstance].coarCallerId = "";
         }
      }
      m_u8PassiveDevHandle = 0;
   }
}


/************************************************************************
*FUNCTION: 		 onCancelOutgoingCallError
*DESCRIPTION: 	 Handles Callback from Phone Service For CancelOutgoingCall Result
*
*PARAMETER:
* 					 proxy: MOST_Tel_FIProxy shared pointer
*					result: AcceptCallResult obj
*
*RETURNVALUE: 	 void
*
*
*HISTORY:
*revision 0.1	ess7kor	11/08/2017
*************************************************************************/
void ServiceTel::onCancelOutgoingCallError(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::CancelOutgoingCallError >& error)
{
   ETG_TRACE_USR4(("ServiceTel : onCancelOutgoingCallError()\t"));
   bCallReqFlag = FALSE;
}


/************************************************************************
*FUNCTION:		vCancelDialCallResult
*DESCRIPTION:	Receives the MethodResult message from Phone service
*              for Cancel Dial call
*PARAMETER:
*              poMessage:   (->I)
*                 Message object
*
*RETURNVALUE:	tVoid
*
*HISTORY:
*16.01.2018		ess7kor
*Rev 0.1 Initial Revision
************************************************************************/
void ServiceTel::onCancelOutgoingCallResult(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::CancelOutgoingCallResult >& result)
{
   ETG_TRACE_USR4(("ServiceTel : onCancelOutgoingCallResult()"));

   bCallReqFlag = FALSE;
   bIsOutgoingCallPopupActive = false;
   ETG_TRACE_USR4(("CallReqFlag : %d", bCallReqFlag));
   ETG_TRACE_USR4(("Call Instance = %d\t", result->getU16CallInstance()));
   ETG_TRACE_USR4(("Call Status = %d\t",
                   ETG_CENUM(btphn_tenPhoneCallStatus, (btphn_tenPhoneCallStatus)result->getE8CallStatus())));
   ETG_TRACE_USR4(("Using Vehicle Audio = %d", result->getBUsingVehicleAudio()));
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void ServiceTel::onConferenceInError(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::ConferenceInError >& error)
{
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void ServiceTel::onConferenceInResult(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::ConferenceInResult >& result)
{
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void ServiceTel::onHangUpCallError(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::HangUpCallError >& error)
{
   ETG_TRACE_FATAL(("ServiceTel : onHangUpCallError()"));

   bCallReqFlag = FALSE;
}


/************************************************************************
*FUNCTION:		vHangUpCallResult
*DESCRIPTION:	Receives the MethodResult message from Phone service
*              for Hangup call/Terminate an ongoing call
*PARAMETER:
*              result:   (->I)
*                 result object
*
*RETURNVALUE:	tVoid
*
*HISTORY:
*01.16.2018 ess7kor
*Rev 0.1 Initial Revision
************************************************************************/
void ServiceTel::onHangUpCallResult(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::HangUpCallResult >& result)
{
   ETG_TRACE_USR4(("ServiceTel : vHangUpCallResult()"));

   bCallReqFlag = FALSE;
   ETG_TRACE_USR4(("CallReqFlag : %d", bCallReqFlag));

   ETG_TRACE_USR4(("Call Instance = %d\t", result->getU16CallInstance()));
   ETG_TRACE_USR4(("Call Status = %d\t",
                   ETG_CENUM(btphn_tenPhoneCallStatus, (btphn_tenPhoneCallStatus)result->getE8CallStatus())));
   ETG_TRACE_USR4(("Using Vehicle Audio = %d", result->getBUsingVehicleAudio()));
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void ServiceTel::onHoldCallError(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::HoldCallError >& error)
{
   ETG_TRACE_USR4(("ServiceTel : onHoldCallError()"));
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void ServiceTel::onHoldCallResult(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::HoldCallResult >& result)
{
}


/************************************************************************
*FUNCTION: 		 onIgnoreCallError
*DESCRIPTION: 	 Handles Callback from Phone Service For IgnoreCall Error
*
*PARAMETER:
* 					 proxy: MOST_Tel_FIProxy shared pointer
*					Error: IgnoreCallError obj
*
*RETURNVALUE: 	 void
*
*
*HISTORY:
*revision 0.1	ess7kor	11/08/2017
*************************************************************************/
void ServiceTel::onIgnoreCallError(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::IgnoreCallError >& error)
{
   ETG_TRACE_USR4(("ServiceTel : onIgnoreCallError()"));
   vResetCallFlag();
}


/************************************************************************
*FUNCTION: 		 onIgnoreCallResult
*DESCRIPTION: 	 Handles Callback from Phone Service For IgnoreCall Result
*
*PARAMETER:
* 					 proxy: MOST_Tel_FIProxy shared pointer
*					result: IgnoreCallResult obj
*
*RETURNVALUE: 	 void
*
*
*HISTORY:
*revision 0.1	ess7kor	11/08/2017
*************************************************************************/
void ServiceTel::onIgnoreCallResult(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::IgnoreCallResult >& result)
{
   ETG_TRACE_USR4(("ServiceTel : onIgnoreCallResult()"));
   vResetCallFlag();
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void ServiceTel::onRejectCallError(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::RejectCallError >& error)
{
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void ServiceTel::onRejectCallResult(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::RejectCallResult >& result)
{
   ETG_TRACE_USR4(("ServiceTel : onendCallResult()"));
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void ServiceTel::onResumeCallError(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::ResumeCallError >& error)
{
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void ServiceTel::onResumeCallResult(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::ResumeCallResult >& result)
{
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void ServiceTel::onSendDTMFError(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::SendDTMFError >& error)
{
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void ServiceTel::onSendDTMFResult(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::SendDTMFResult >& result)
{
   ETG_TRACE_USR4(("ServiceTel :: onSendDTMFResult call inst =%d, command status= %d ", result->getU16CallInstance(), result->getBCommandStatus()));
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void ServiceTel::onSpeedDialError(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::SpeedDialError >& error)
{
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void ServiceTel::onSpeedDialResult(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::SpeedDialResult >& result)
{
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void ServiceTel::onSplitCallError(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::SplitCallError >& error)
{
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void ServiceTel::onSplitCallResult(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::SplitCallResult >& result)
{
}


/************************************************************************
FUNCTION:     onSwapCallError
* Virtual function implemented to get if there is any error for the method start SwapCall
*
* @param[in]      : proxy: Client side representation of the CCA Functional Interface 'MOST_Tel_FI
* @param[in]      : error : The error message of 'SwapCall'
* @return         : void
*HISTORY:
*01.11.2018 ess7kor
*Rev 0.1 Initial Revision.
************************************************************************/
void ServiceTel::onSwapCallError(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::SwapCallError >& error)
{
   ETG_TRACE_FATAL(("ServiceTel: onSwapCallError"));
}


/************************************************************************
FUNCTION:     onSwapCallResult
* DESCRIPTION: Virtual function implemented to get the result of method start SwapCall
*
* @param[in]      : proxy: Client side representation of the CCA Functional Interface 'MOST_Tel_FI
* @param[in]      : result : The result message of the method "SwapCall"
* @return         : void
*HISTORY:
*01.11.2018 ess7kor
*Rev 0.1 Initial Revision.
************************************************************************/
void ServiceTel::onSwapCallResult(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::SwapCallResult >& result)
{
   ETG_TRACE_USR2(("SwapCallResult: Call instance : %d , Call status : %d", result->getU16CallInstance(), ETG_CENUM(btphn_tenPhoneCallStatus, (btphn_tenPhoneCallStatus)result->getE8CallStatus())));
}


/************************************************************************
FUNCTION:     onDialCallError
* Virtual function implemented to get if there is any error for the method start DIalCall
*
* @param[in]      : proxy: Client side representation of the CCA Functional Interface 'MOST_Tel_FI
* @param[in]      : error : The error message of 'DialCall'
* @return         : void
*HISTORY:
*01.11.2018 ess7kor
*Rev 0.1 Initial Revision.
************************************************************************/
//void ServiceTel::onDialError(const ::boost::shared_ptr< MOST_Tel_FIProxy >& proxy, const ::boost::shared_ptr< DialError >& error)
//{
//	ETG_TRACE_FATAL(("ServiceTel: onDialError"));
//}

/************************************************************************
*FUNCTION:     onDialCallResult
*DESCRIPTION:  Handles DialCall result from FC_Phone. When DialCall result is
*					received, a 2 second timer is started and if call status
*					is not received as dialing before the timer expires, then on
*					timer expiry, dial error information is sent to HMI.
*					Sometimes, when dial request is sent to
*					FC_Phone, dial result is received but the call status is not
*					updated as DIALING since dialing does not happen in phone. In
*				   such cases, after DIAL_START, on timer expiry, DIAL_ERROR is
*					updated to HMI
*PARAMETER:		None
*RETURNVALUE: 	tVoid
*
*HISTORY:
*01.11.2018 ess7kor
*Rev 0.1 Initial Revision.
************************************************************************/

//void ServiceTel::onDialResult(const ::boost::shared_ptr< MOST_Tel_FIProxy >& proxy, const ::boost::shared_ptr< DialResult >& result)
//{
//	ETG_TRACE_USR4(("ServiceTel : onDialResult()"));
//	vResetCallFlag();
//	vStartDialCallTimer();
//}

/************************************************************************
*FUNCTION: 		vOnDialCallTimerExpiry
*DESCRIPTION: 	Timer callback function which is invoked when the dial call timer
*					expires. This timer expires when dial request has been
*					acknowledged but the call status is not received as Dialing.
*					Dial error is updated here on timer expiry
*PARAMETER:		tPVoid
*RETURNVALUE: 	tVoid
*
*HISTORY:
*01.11.2018 ess7kor
*Rev 0.1 Initial Revision.
************************************************************************/

tVoid ServiceTel::vOnDialCallTimerExpiry(tPVoid pvObject)
{
   ETG_TRACE_USR4(("ServiceTel : vOnDialCallTimerExpiry()"));
   BTPHN_M_UNUSED(pvObject);

   if (NULL != ServiceTel::poGetInstance())
   {
      ServiceTel::poGetInstance()->bIsDialCallTimerRunning = FALSE;
      ServiceTel::poGetInstance()->vHandleDialError();
   }
}


/************************************************************************
*FUNCTION: 		vOnEndCallTimerExpiry
*DESCRIPTION: 	Timer callback function which is invoked when the end call timer
*					expires. This functions sends the current call status with Idle
*					to HMI
*PARAMETER:		tPVoid
*RETURNVALUE: 	tVoidF
*
*HISTORY:
*01.11.2018 ess7kor
*Rev 0.1 Initial Revision.
************************************************************************/

tVoid ServiceTel::vOnEndCallTimerExpiry(tPVoid pvObject)
{
   ETG_TRACE_USR4(("ServiceTel : vOnEndCallTimerExpiry()"));
   BTPHN_M_UNUSED(pvObject);

   if (NULL != ServiceTel::poGetInstance())
   {
      ServiceTel::poGetInstance()->bIsEndCallTimerRunning = FALSE;
      ServiceTel::poGetInstance()->vSendCallStatus(ServiceTel::poGetInstance()->arCallStatusInfo[ServiceTel::poGetInstance()->u8CurrentCallInstance]);
   }
}


/************************************************************************
*FUNCTION: 		vHandleDialError
*DESCRIPTION: 	Handles error that occurs while dialing a number
*				send Dial error information to HMI
*PARAMETER:		None
*RETURNVALUE: 	tVoid
*
*HISTORY:
*01.11.2018 ess7kor
*Rev 0.1 Initial Revision.
************************************************************************/

tVoid ServiceTel::vHandleDialError()
{
   ETG_TRACE_USR4(("ServiceTel : vHandleDialError()"));

   // as we have encountered error in Dialing, setting the flag to false
   bIsOutgoingCallPopupActive = false;

   for (tU8 u8Index = BTPHN_C_DEFAULT_VALUE_ZERO; u8Index < BTPHN_C_NUM_CALLS_SUPPORTED; u8Index++)
   {
      if (BTPHN_EN_DIAL_START == arCallStatusInfo[u8Index].enCallStatus)
      {
         arCallStatusInfo[u8Index].enCallStatus = BTPHN_EN_DIAL_ERROR;

         vSendCallStatus(arCallStatusInfo[u8Index]);

         arCallStatusInfo[u8Index].coarCallerId = "";
         arCallStatusInfo[u8Index].enCallStatus = BTPHN_EN_IDLE;
         arCallStatusInfo[u8Index].bIsContactInfoReq = FALSE;
         u8NumActiveCalls--;
         break;
      }
   }
}


#ifndef VARIANT_S_FTR_ENABLE_PHONECALLMANAGER_SUPPORT
void ServiceTel::onQueryTelephoneNumberError(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::QueryTelephoneNumberError >& error)
{
   ETG_TRACE_FATAL(("ServiceTel : onQueryTelephoneNumberError()"));
}


/************************************************************************
*FUNCTION: 		onQueryTelephoneNumberResult
*DESCRIPTION: 	Handles method result callback of querytelephonenumber
*PARAMETER:		None
*RETURNVALUE: 	tVoid
*
*HISTORY:
*01.11.2018 ess7kor
*Rev 0.1 Initial Revision.
************************************************************************/
void ServiceTel::onQueryTelephoneNumberResult(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& /*proxy*/, const boost::shared_ptr< ::MOST_Tel_FI::QueryTelephoneNumberResult >& result)
{
   ETG_TRACE_USR4(("ServiceTel : onQueryTelephoneNumberResult()"));

   std::string strTelphoneno = result->getSTelephoneNumber().c_str();
   tU16 u16CallInstance = result->getU16CallInstance();

   ETG_TRACE_USR4(("ServiceTel : Call Insatnce is %d, Telephoneno is %s", u16CallInstance, strTelphoneno.c_str()));

   if (BTPHN_C_CALL_INSTANCE_INVALID != u16CallInstance)
   {
      if ((u16CallInstance == 0) || (u16CallInstance == 2))
      {
         u16CallInstance = BTPHN_C_FIRST_CALL_INDEX;
      }
      else
      {
         u16CallInstance = BTPHN_C_SECOND_CALL_INDEX;
      }
      /* Check for non-empty telephone number */
      if (BTPHN_C_EQUAL != strTelphoneno.compare(""))
      {
         /* If the caller id is not already updated (for an incoming call,
         caller id is updated on receiving incoming call update from
         service), then store the telephone number */
         if (BTPHN_C_EQUAL == arCallStatusInfo[u16CallInstance].coarCallerId.compare(""))
         {
            arCallStatusInfo[u16CallInstance].coarCallerId = strTelphoneno.c_str();
         }
         if (NULL != ServicePhoneBook::poGetInstance())
         {
            tBool bret = ServicePhoneBook::poGetInstance()->vGetContactInfo(strTelphoneno);
            if (!bret)
            {
               ETG_TRACE_ERR(("ServiceTel : vGetContactInfo returned FALSE"));
               arCallStatusInfo[u16CallInstance].bIsCallerIdUpdated = TRUE;
            }
         }
      }
      else
      {
         /* The caller id is updated as "unknown" if no telephone number
         is available */

         arCallStatusInfo[u16CallInstance].coarCallerId = copcUnknownText.c_str();
         arCallStatusInfo[u16CallInstance].bIsCallerIdUpdated = TRUE;
      }
      /* After the telephone number is updated, the call status information
      is sent to HMI Model */
      vProcessCallStatus();
   }
}


#endif

/************************************************************************
*FUNCTION: 		vProcessAndUpdateTelephoneNumber
*DESCRIPTION: 	Process the telephone number string for the respective call instance
*PARAMETER:		None
*RETURNVALUE: 	tVoid
*
*HISTORY:
*
*Rev 0.1 Initial Revision.
************************************************************************/
#ifdef VARIANT_S_FTR_ENABLE_PHONECALLMANAGER_SUPPORT
tVoid ServiceTel:: vProcessAndUpdateTelephoneNumber(std::string strTelphoneno, tU16 u16CallInstance)
{
   ETG_TRACE_USR4(("ServiceTel : vProcessAndUpdateTelephoneNumber()"));

   ETG_TRACE_USR4(("ServiceTel : Call Insatnce is %d, Telephoneno is %s", u16CallInstance, strTelphoneno.c_str()));

   if (BTPHN_C_CALL_INSTANCE_INVALID != u16CallInstance)
   {
      if ((u16CallInstance == 0) || (u16CallInstance == 2))
      {
         u16CallInstance = BTPHN_C_FIRST_CALL_INDEX;
      }
      else
      {
         u16CallInstance = BTPHN_C_SECOND_CALL_INDEX;
      }
      ETG_TRACE_USR4(("ServiceTel : updated callInstance value is : %d", u16CallInstance));
      /* Check for non-empty telephone number */
      if (BTPHN_C_EQUAL != strTelphoneno.compare(""))
      {
         /* If the caller id is not already updated (for an incoming call,
         caller id is updated on receiving incoming call update from
         service), then store the telephone number */
         if (BTPHN_C_EQUAL == arCallStatusInfo[u16CallInstance].coarCallerId.compare(""))
         {
            arCallStatusInfo[u16CallInstance].coarCallerId = strTelphoneno.c_str();
            //LongDataSync
            m_strlongDataCallNum = strTelphoneno.c_str();
         }
         if (NULL != ServicePhoneBook::poGetInstance())
         {
            tBool bret = ServicePhoneBook::poGetInstance()->vGetContactInfo(strTelphoneno);
            if (!bret)
            {
               ETG_TRACE_ERR(("ServiceTel : vGetContactInfo returned FALSE"));
               arCallStatusInfo[u16CallInstance].bIsCallerIdUpdated = TRUE;
            }
            else
            {
               ETG_TRACE_USR4(("ServiceTel : vGetContactInfo returned TRUE"));
            }
         }
      }
      else
      {
         /* The caller id is updated as "unknown" if no telephone number
         is available */

         arCallStatusInfo[u16CallInstance].coarCallerId = copcUnknownText.c_str();
         arCallStatusInfo[u16CallInstance].bIsCallerIdUpdated = TRUE;
         //LongData Sync
         m_strlongDataCallNum = copcUnknownText.c_str();
      }
   }
//LongDataSync
   VehicleSrvClient::poGetInstance()->vUpdateCallInfo(LONGDATATYPE_CALLNUM, m_strlongDataCallNum); //CAN imp
}


#endif
/************************************************************************
*FUNCTION: 		 vUpdateContactInfo
*DESCRIPTION: 	 Receives the contact information namely the contact name and
*					 and contact detail type (with respect to a call) and updates the
*					 same to HMI Model.
*PARAMETER:
*					 corfu32ContactHandle:   (I)
*					    Contact handle
* 				    SFirstName:	(->I)
* 					    Contact's first name
* 				    SLastName:	(->I)
* 					    Contact's last name
* 				    enContactDetailType:	(I)
* 					    Contact detail type - Mobile/Home/Work/Other
*
*RETURNVALUE: 	tVoid
*
*HISTORY:
*01.15.2018 ess7kor
*Rev 0.1 Initial Revision.
************************************************************************/
tVoid ServiceTel::vUpdateContactInfo(tU32 u32ContactHandle, std::string SFirstName, std::string SLastName, btphn_tenContactDetailType enContactDetailType)
{
   ETG_TRACE_USR4(("ServiceTel : vUpdateContactInfo()"));
   ETG_TRACE_USR4(("Contact handle : %d", u32ContactHandle));
   ETG_TRACE_USR4(("First name : %s", SFirstName.c_str()));
   ETG_TRACE_USR4(("Last name : %s", SLastName.c_str()));
   ETG_TRACE_USR4(("Contact detail type : %d", enContactDetailType/*ETG_CENUM(btphn_tenContactDetailType, enContactDetailType)*/));

   btphn_tenPhoneType enPhoneType = BTPHN_EN_GENERIC;
   if ((BTPHN_EN_CELL_NUMBER_1 <= enContactDetailType) &&
         (BTPHN_EN_OTHER_NUMBER >= enContactDetailType))
   {
      /* Determining the contact number type icon to be displayed */
      enPhoneType = static_cast<btphn_tenPhoneType>((enContactDetailType / BTPHN_C_CONTACTTYPE_MAX_NUMBER) - 1);
   }

   most_PhonBk_fi_types::T_e8_PhonBkPreferredPhoneBookSortOrder sortorder = T_e8_PhonBkPreferredPhoneBookSortOrder__e8PREFERRED_SORT_ORDER_FIRSTNAME;

   if (ServicePhoneBook::poGetInstance() != NULL)
   {
      sortorder = ServicePhoneBook::poGetInstance()->enGetSortOrder(); //temporary untill P_OPT scene implementation
   }
   /* Invoking method to provide the contact full name */
   std::string strContactName = AppHmiPhone_Helper::GetContactname(SFirstName, SLastName, sortorder);

   /* Sending the contact name and phone type to HMI Model */
   vUpdateContactInfo(strContactName, enPhoneType);
}


/************************************************************************
*FUNCTION: 		vUpdateContactInfo
*DESCRIPTION: 	Receives the contact information for a particular telephone
*					dDisplay number / contact name of the
*											ongoing call)
*PARAMETER:
*					copcContactName:   (->I)
*						Contact name/unknown
*					corfenPhoneType:   (->I)
*						Phone type - Mobile/Home/Work/Other
*RETURNVALUE: 	tVoid
*
*HISTORY:
*01.15.2018 ess7kor
*Rev 0.1 Initial Revision
************************************************************************/
tVoid ServiceTel::vUpdateContactInfo(std::string copcContactName, const btphn_tenPhoneType& corfenPhoneType)
{
   ETG_TRACE_USR4(("ServiceTel : vUpdateContactInfo()"));
   ETG_TRACE_USR4(("Contact name = %s\t", copcContactName.c_str()));
   //   ETG_TRACE_USR4(("Phone type = %d", ETG_CENUM(btphn_tenPhoneType, corfenPhoneType)));

   for (tU8 u8Index = BTPHN_C_DEFAULT_VALUE_ZERO;
         u8Index < BTPHN_C_NUM_CALLS_SUPPORTED;
         u8Index++)
   {
      int nRetVal = arCallStatusInfo[u8Index].coarCallerId.compare("");

      /* Checks if the caller id is not null (i.e caller id is updated
      with the telephone number), call status is not IDLE and caller id
      is not updated for the call instance */
      if ((BTPHN_C_EQUAL != nRetVal) &&
            (FALSE == arCallStatusInfo[u8Index].bIsCallerIdUpdated) &&
            (BTPHN_EN_IDLE != arCallStatusInfo[u8Index].enCallStatus))
      {
         arCallStatusInfo[u8Index].bIsCallerIdUpdated = TRUE;
         nRetVal = copcContactName.compare(copcUnknownText);
         /* If the contact name received is not set to 'unknown', then no
         change is made to the caller id (which has the telephone number)
         */
         if (BTPHN_C_EQUAL != nRetVal)
         {
            arCallStatusInfo[u8Index].coarCallerId = copcContactName;
         }
         arCallStatusInfo[u8Index].enPhoneType = corfenPhoneType;
         break;
      }
   }
   /* After updating the caller id with the contact name, the call status
   information is sent to HMI Model */
   vProcessCallStatus();
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void ServiceTel::onTransferCallToHandsetError(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::TransferCallToHandsetError >& error)
{
   ETG_TRACE_FATAL(("ServiceTel : onTransferCallToHandsetError()"));
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void ServiceTel::onTransferCallToHandsetResult(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::TransferCallToHandsetResult >& result)
{
   ETG_TRACE_USR4(("ServiceTel : onTransferCallToHandsetError()"));
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
//void ServiceTel::onTransferCallToVehicleError(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::onTransferCallToVehicleError >& error)
void ServiceTel::onTransferCallToVehicleError(const ::boost::shared_ptr< MOST_Tel_FIProxy >& proxy, const ::boost::shared_ptr< TransferCallToVehicleError >& error)

{
   ETG_TRACE_FATAL(("ServiceTel : onTransferCallToVehicleError()"));
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
//void ServiceTel::onTransferCallToVehicleResult(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::onTransferCallToVehicleResult >& result)
void ServiceTel::onTransferCallToVehicleResult(const ::boost::shared_ptr< MOST_Tel_FIProxy >& proxy, const ::boost::shared_ptr< TransferCallToVehicleResult >& result)
{
   ETG_TRACE_USR4(("ServiceTel : onTransferCallToVehicleResult()"));
}


/************************************************************************
FUNCTION:     onActiveCallinHandsetError
* DESCRIPTION: Virtual function implemented to get the error status callback of ActiveCallinHandset
*
* @param[in]      : proxy: Client side representation of the CCA Functional Interface 'MOST_Tel_FI
* @param[in]      : error : The result message of the method "ActiveCallinHandset"
* @return         : void
*HISTORY:
*01.11.2018 ess7kor
*Rev 0.1 Initial Revision.
************************************************************************/
void ServiceTel::onActiveCallinHandsetError(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::ActiveCallinHandsetError >& error)
{
   ETG_TRACE_FATAL(("ServiceTel : onActiveCallinHandsetError()"));
}


/************************************************************************
FUNCTION:     onActiveCallinHandsetStatus
* DESCRIPTION: Virtual function implemented to get the status callback of ActiveCallinHandset
*
* @param[in]      : proxy: Client side representation of the CCA Functional Interface 'MOST_Tel_FI
* @param[in]      : status : The status message of the method "ActiveCallinHandset"
* @return         : void
*HISTORY:
*01.11.2018 ess7kor
*Rev 0.1 Initial Revision.
************************************************************************/
void ServiceTel::onActiveCallinHandsetStatus(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::ActiveCallinHandsetStatus >& status)
{
   ETG_TRACE_USR4(("ServiceTel : onActiveCallinHandsetStatus()"));
   //btphn_tenPhoneCallHandsetState enCallStatus =
   //	static_cast <btphn_tenPhoneCallHandsetState>(status->gete8HandsetState());

   //ETG_TRACE_USR4(("enCallStatus is %d", ETG_CENUM(btphn_tenPhoneCallHandsetState, enCallStatus));
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
#ifndef VARIANT_S_FTR_ENABLE_PHONECALLMANAGER_SUPPORT
void ServiceTel::onCallStatusNoticeError(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::CallStatusNoticeError >& error)
{
   ETG_TRACE_USR4(("ServiceTel::onCallStatusNoticeError"));
   //Update the Call status error status to the CAN Interface
   if (error->hasSystemErrorCode() == true)
   {
      ETG_TRACE_USR4(("onCallStatusNoticeError:SYSTEM Error code"));
      VehicleSrvClient::poGetInstance()->updateCallDurationError(PHONEMW_SNA); //CAN Imp
      VehicleSrvClient::poGetInstance()->updateCallStatusError(PHONEMW_SNA); //CAN Imp
   }
   else if (error->hasCcaErrorCode() == true)
   {
      ETG_TRACE_USR4(("onCallStatusNoticeError:CCA Error code"));
      VehicleSrvClient::poGetInstance()->updateCallDurationError(CAN_ERROR); //CAN Imp
      VehicleSrvClient::poGetInstance()->updateCallStatusError(CAN_ERROR); //CAN Imp
   }
   else
   {
      ETG_TRACE_USR4(("onCallStatusNoticeError:Unknown Error"));
      VehicleSrvClient::poGetInstance()->updateCallDurationError(PHONEMW_SNA); //CAN Imp
      VehicleSrvClient::poGetInstance()->updateCallStatusError(PHONEMW_SNA); //CAN Imp
   }
}


#endif
/************************************************************************
*FUNCTION:     covCallStatusNoticeStatus
*DESCRIPTION:  Receives the updated value of the property CallStatusNotice

*
*RETURNVALUE:	tVoid
*03.06.2019 ebg1kor
*Rev 0.1 Initial Revision.
************************************************************************/
#ifndef VARIANT_S_FTR_ENABLE_PHONECALLMANAGER_SUPPORT
void ServiceTel::onCallStatusNoticeStatus(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< CallStatusNoticeStatus >& status)
{
   ETG_TRACE_USR4(("ServiceTel : onCallStatusNoticeStatus()"));

   if (_most_tel_fi && _most_tel_fi.get())
   {
      ETG_TRACE_USR4(("No of Call Instances = %d", status->getOCallStatusNoticeStream().size()));
      std::vector<btphn_trCallStatus> vecCallStatusInfo;
      for (int i = 0; i < status->getOCallStatusNoticeStream().size(); i++)
      {
         btphn_trCallStatus rCallStatus;
         memset(&rCallStatus, '\0', sizeof(btphn_trCallStatus));

         arCallStatusInfo[i].u16CallInstance = status->getOCallStatusNoticeStream().at(i).getU16CallInstance();
         rCallStatus.u16CallInstance = status->getOCallStatusNoticeStream().at(i).getU16CallInstance();
         ETG_TRACE_USR4(("onCallStatusNoticeStatus : Call Instance = %d\t", rCallStatus.u16CallInstance));

         //most_Tel_fi_types::T_e8_TelCallStatus enCallStatusService = status->getOCallStatusNoticeStream().at(i).getE8CallStatus();
         rCallStatus.enCallStatus = (btphn_tenPhoneCallStatus)status->getOCallStatusNoticeStream().at(i).getE8CallStatus();
         //rCallStatus.enCallStatus = (btphn_tenPhoneCallStatus)(enCallStatusService);
         ETG_TRACE_USR4(("onCallStatusNoticeStatus: Call Status = %d", ETG_CENUM(btphn_tenPhoneCallStatus, (btphn_tenPhoneCallStatus)rCallStatus.enCallStatus)));

         rCallStatus.bUsingVehicleAudio = status->getOCallStatusNoticeStream().at(i).getBUsingVehicleAudio();
         ETG_TRACE_USR4(("Vehicle Audio = %d\t", rCallStatus.bUsingVehicleAudio));

         rCallStatus.u8CallDurationHr = status->getOCallStatusNoticeStream().at(i).getU8CallDurationHr();
         rCallStatus.u8CallDurationMin = status->getOCallStatusNoticeStream().at(i).getU8CallDurationMin();
         rCallStatus.u8CallDurationSec = status->getOCallStatusNoticeStream().at(i).getU8CallDurationSec();

         ETG_TRACE_USR4(("Call duration is %u:%u:%u\t",
                         rCallStatus.u8CallDurationHr,
                         rCallStatus.u8CallDurationMin,
                         rCallStatus.u8CallDurationSec));

         vecCallStatusInfo.push_back(rCallStatus);

         //Sending the call status for updating HMI
         vUpdateCallStatus(vecCallStatusInfo);
         vSendCallStatusToBTVR(vecCallStatusInfo);
      }
   }
   else
   {
      // Set to false as we got Error for DialCall request
      bIsOutgoingCallPopupActive = false;
      ETG_TRACE_FATAL(("ServiceTel : onCallStatusNoticeStatus() ERROR"));
   }
}


#endif

/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
#ifdef VARIANT_S_FTR_ENABLE_PHONECALLMANAGER_SUPPORT
void ServiceTel::onCallStatusNoticeExtendedError(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::CallStatusNoticeExtendedError >& error)
{
   if (error->hasSystemErrorCode() == true)
   {
      ETG_TRACE_USR4(("onCallStatusNoticeError:SYSTEM Error code"));
      VehicleSrvClient::poGetInstance()->updateCallDurationError(PHONEMW_SNA); //CAN Imp
      VehicleSrvClient::poGetInstance()->updateCallStatusError(PHONEMW_SNA); //CAN Imp
   }
   else if (error->hasCcaErrorCode() == true)
   {
      ETG_TRACE_USR4(("onCallStatusNoticeError:CCA Error code"));
      VehicleSrvClient::poGetInstance()->updateCallDurationError(CAN_ERROR); //CAN Imp
      VehicleSrvClient::poGetInstance()->updateCallStatusError(CAN_ERROR); //CAN Imp
   }
   else
   {
      ETG_TRACE_USR4(("onCallStatusNoticeError: Unknown Error"));
      VehicleSrvClient::poGetInstance()->updateCallDurationError(PHONEMW_SNA); //CAN Imp
      VehicleSrvClient::poGetInstance()->updateCallStatusError(PHONEMW_SNA); //CAN Imp
   }
}


#endif

/************************************************************************
*FUNCTION:     onCallStatusNoticeExtendedStatus
*DESCRIPTION:  Receives the updated value of the property CallStatusNoticeExtended

*
*RETURNVALUE:	tVoid
*
*Rev 0.1 Initial Revision.
************************************************************************/
#ifdef VARIANT_S_FTR_ENABLE_PHONECALLMANAGER_SUPPORT
void ServiceTel::onCallStatusNoticeExtendedStatus(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< CallStatusNoticeExtendedStatus >& status)
{
   ETG_TRACE_USR4(("ServiceTel : onCallStatusNoticeExtendedStatus()"));

   if (_most_tel_fi && _most_tel_fi.get())
   {
      ETG_TRACE_USR4(("No of Call Instances = %d", status->getOCallStatusNoticeExtendedStream().size()));
      std::vector<btphn_trCallStatus> vecCallStatusInfo;
      for (int i = 0, j = 0; i < status->getOCallStatusNoticeExtendedStream().size(); i++)
      {
         //Process the call status extended stream data only for active device call instances, since the HMI application shall support maximum of two call instances alone.
         if (status->getOCallStatusNoticeExtendedStream().at(i).getU8DeviceHandle() == m_u8ActiveDev)
         {
            btphn_trCallStatus rCallStatus;
            memset(&rCallStatus, '\0', sizeof(btphn_trCallStatus));

            arCallStatusInfo[j].u16CallInstance = status->getOCallStatusNoticeExtendedStream().at(i).getU16CallInstance();
            rCallStatus.u16CallInstance = status->getOCallStatusNoticeExtendedStream().at(i).getU16CallInstance();
            ETG_TRACE_USR4(("onCallStatusNoticeExtendedStatus : Call Instance = %d\t", rCallStatus.u16CallInstance));

            //String to store the telephone number from CallStatusNoticeExtended interface
            arCallStatusInfo[j].strCallStatusExTelNo = status->getOCallStatusNoticeExtendedStream().at(i).getSTelephoneNumber().c_str();
            ETG_TRACE_USR4(("onCallStatusNoticeExtendedStatus : TelephoneNumber = %s\t", arCallStatusInfo[j].strCallStatusExTelNo.c_str()));

            // Map the call status value through mapper function and store it
            rCallStatus.enCallStatus = CallStatusEnumMapper(status->getOCallStatusNoticeExtendedStream().at(i).getE8CallStatus());
            ETG_TRACE_USR4(("onCallStatusNoticeExtendedStatus:Call Status = %d", ETG_CENUM(btphn_tenPhoneCallStatus, (btphn_tenPhoneCallStatus)rCallStatus.enCallStatus)));

            rCallStatus.bUsingVehicleAudio = status->getOCallStatusNoticeExtendedStream().at(i).getBUsingVehicleAudio();
            ETG_TRACE_USR4(("Vehicle Audio = %d\t", rCallStatus.bUsingVehicleAudio));

            rCallStatus.u8CallDurationHr = status->getOCallStatusNoticeExtendedStream().at(i).getU8CallDurationHr();
            rCallStatus.u8CallDurationMin = status->getOCallStatusNoticeExtendedStream().at(i).getU8CallDurationMin();
            rCallStatus.u8CallDurationSec = status->getOCallStatusNoticeExtendedStream().at(i).getU8CallDurationSec();

            ETG_TRACE_USR4(("Call duration is %u:%u:%u\t",
                            rCallStatus.u8CallDurationHr,
                            rCallStatus.u8CallDurationMin,
                            rCallStatus.u8CallDurationSec));

            vecCallStatusInfo.push_back(rCallStatus);

            //Sending the call status for updating HMI
            vUpdateCallStatus(vecCallStatusInfo);
            vSendCallStatusToBTVR(vecCallStatusInfo);

            //Increment counter 'J" as to store call instance and telephone number for active device instances of maximum two count.
            j++;
         }
      }
   }
   else
   {
      // Set to false as we got Error for DialCall request
      bIsOutgoingCallPopupActive = false;
      ETG_TRACE_FATAL(("ServiceTel : onCallStatusNoticeStatus() ERROR"));
   }
}


#endif
/************************************************************************
*FUNCTION: 		CallStatusEnumMapper
*DESCRIPTION: 	The FI call status values shall be mapped against the local types.
*PARAMETER:     T_e8_TelCallStatus
*
*RETURNVALUE: 	btphn_tenPhoneCallStatus
*
*HISTORY:
*18.09.2020 Palani K
*Rev 0.1 Initial Revision.
************************************************************************/
btphn_tenPhoneCallStatus ServiceTel::CallStatusEnumMapper(most_Tel_fi_types::T_e8_TelCallStatus callStatus)
{
   ETG_TRACE_USR4(("ServiceTel : CallStatusEnumMapper()"));

   btphn_tenPhoneCallStatus temp_callStatus;

   switch (callStatus)
   {
      case T_e8_TelCallStatus__e8IDLE:
         temp_callStatus = BTPHN_EN_IDLE;
         break;

      case T_e8_TelCallStatus__e8DIALING:
         temp_callStatus = BTPHN_EN_DIALING;
         break;

      case T_e8_TelCallStatus__e8ACTIVE:
         temp_callStatus = BTPHN_EN_ACTIVE;
         break;

      case T_e8_TelCallStatus__e8ON_HOLD:
         temp_callStatus = BTPHN_EN_ON_HOLD;
         break;

      case T_e8_TelCallStatus__e8DISCONNECTING:
         temp_callStatus = BTPHN_EN_DISCONNECTING;
         break;

      case T_e8_TelCallStatus__e8BUSY:
         temp_callStatus = BTPHN_EN_BUSY;
         break;

      case T_e8_TelCallStatus__e8CONFERENCE:
         temp_callStatus = BTPHN_EN_CONFERENCE;
         break;

      case T_e8_TelCallStatus__e8IN_VOICEMAIL:
         temp_callStatus = BTPHN_EN_IN_VOICEMAIL;
         break;

      case T_e8_TelCallStatus__e8RINGTONE:
         temp_callStatus = BTPHN_EN_RINGTONE;
         break;

      default:
         temp_callStatus = BTPHN_EN_DIAL_ERROR;
         break;
   }

   return temp_callStatus;
}


/************************************************************************
*FUNCTION: 		vCheckCallStatus
*DESCRIPTION: 	It will update the status of current call status and update
the result to PhoneBookSevice.
*PARAMETER:
*
*RETURNVALUE: 	tBool
*					   TRUE, if the callstatus belongs to any of the checkedStates.
*					   FALSE, if the callstatus belongs to non of them.
*
*HISTORY:
*01.15.2018 ess7kor
*Rev 0.1 Initial Revision.
************************************************************************/
tBool ServiceTel::vCheckCallStatus()
{
   ETG_TRACE_USR4(("ServiceTel : vCheckCallStatus()"));

   for (tU8 u8Index = BTPHN_C_DEFAULT_VALUE_ZERO;
         u8Index < BTPHN_C_NUM_CALLS_SUPPORTED;
         u8Index++)
   {
      ETG_TRACE_USR4(("Call instance : %d", u8Index));
      ETG_TRACE_USR4(("Caller Id : %s", arCallStatusInfo[u8Index].coarCallerId.c_str()));

      if ((BTPHN_EN_DIALING == arCallStatusInfo[u8Index].enCallStatus) ||
            (BTPHN_EN_ACTIVE == arCallStatusInfo[u8Index].enCallStatus) ||
            (BTPHN_EN_ON_HOLD == arCallStatusInfo[u8Index].enCallStatus) ||
            (BTPHN_EN_BUSY == arCallStatusInfo[u8Index].enCallStatus) ||
            (BTPHN_EN_CONFERENCE == arCallStatusInfo[u8Index].enCallStatus) ||
            (BTPHN_EN_RINGTONE == arCallStatusInfo[u8Index].enCallStatus))
      {
         ETG_TRACE_USR4(("ServiceTel::SearchPhoneBookRequired"));
         return TRUE;
      }
   }
   ETG_TRACE_USR4(("ServiceTel::SearchPhoneBook NOT Required"));
   return FALSE;
}


/************************************************************************
*FUNCTION: 		vUpdateCallStatus
*DESCRIPTION: 	Receives the call status information from Phone service and sends
*					update message the same to HMI Model.
*PARAMETER:
*					corfarNewCallStatus:   (I)
*						Vector of structures containing call status information for
*					   each call
*RETURNVALUE: 	tVoid
*
*HISTORY:
*01.11.2018 ess7kor
*Rev 0.1 Initial Revision
*Rev 0.2 25.07.2019 pka5cob; SYSFL1839
************************************************************************/
tVoid ServiceTel::vUpdateCallStatus(const std::vector<btphn_trCallStatus>& corfarNewCallStatus)
{
   ETG_TRACE_USR4(("ServiceTel : vUpdateCallStatus()"));

   u8CurrentCallInstance = BTPHN_C_DEFAULT_VALUE_ZERO;
   u8NumActiveCalls = BTPHN_C_DEFAULT_VALUE_ZERO;
   ETG_TRACE_USR4(("vUpdateCallStatus : arCallStatusInfo[0].u16CallInstance = %d", arCallStatusInfo[BTPHN_C_FIRST_CALL_INDEX].u16CallInstance));
   ETG_TRACE_USR4(("vUpdateCallStatus : arCallStatusInfo[1].u16CallInstance = %d", arCallStatusInfo[BTPHN_C_SECOND_CALL_INDEX].u16CallInstance));
   if ((BTPHN_C_NUM_CALLS_SUPPORTED <= corfarNewCallStatus.size()) &&
         (arCallStatusInfo[BTPHN_C_FIRST_CALL_INDEX].u16CallInstance == corfarNewCallStatus[BTPHN_C_FIRST_CALL_INDEX].u16CallInstance) /*&&
																																		(arCallStatusInfo[BTPHN_C_SECOND_CALL_INDEX].u16CallInstance ==	corfarNewCallStatus[BTPHN_C_SECOND_CALL_INDEX].u16CallInstance)*/)
   {
      /* Check if call status should be updated to HMI. Intermediate call states
      are not updated to HMI */
      tBool bUpdateStatus = bIsCallStatusUpdateRequired(corfarNewCallStatus[BTPHN_C_FIRST_CALL_INDEX].enCallStatus, corfarNewCallStatus[BTPHN_C_SECOND_CALL_INDEX].enCallStatus);
      ETG_TRACE_USR4(("Call status update : %d", bUpdateStatus));

      /*Stores the active call instance of the call status when call status is ACTIVE for processing as part of DTMF*/
      if (corfarNewCallStatus[BTPHN_C_FIRST_CALL_INDEX].enCallStatus == BTPHN_EN_ACTIVE)
      {
         u8CurrentActiveCallInstance_dtmf = static_cast<tU8>(corfarNewCallStatus[BTPHN_C_FIRST_CALL_INDEX].u16CallInstance);
      }
      else if (corfarNewCallStatus[BTPHN_C_SECOND_CALL_INDEX].enCallStatus == BTPHN_EN_ACTIVE)
      {
         u8CurrentActiveCallInstance_dtmf = static_cast<tU8>(corfarNewCallStatus[BTPHN_C_SECOND_CALL_INDEX].u16CallInstance);
      }
      else
      {
      }
      ETG_TRACE_USR4((" Active Call instance when call status is ACTIVE: %d", u8CurrentActiveCallInstance_dtmf));

      for (tU8 u8Index = BTPHN_C_DEFAULT_VALUE_ZERO; u8Index < BTPHN_C_NUM_CALLS_SUPPORTED; u8Index++)
      {
         ETG_TRACE_USR4(("Call instance : %d", u8Index));
         ETG_TRACE_USR4(("Caller Id : %s", arCallStatusInfo[u8Index].coarCallerId.c_str()));

         /* Stores the call status information received from phone service */
         arCallStatusInfo[u8Index].bUsingVehicleAudio =	corfarNewCallStatus[u8Index].bUsingVehicleAudio;
         arCallStatusInfo[u8Index].u8CallDurationHr = corfarNewCallStatus[u8Index].u8CallDurationHr;
         arCallStatusInfo[u8Index].u8CallDurationMin = corfarNewCallStatus[u8Index].u8CallDurationMin;
         arCallStatusInfo[u8Index].u8CallDurationSec = corfarNewCallStatus[u8Index].u8CallDurationSec;

         if (TRUE == bUpdateStatus)
         {
            ETG_TRACE_USR4(("vUpdateCallStatus : TRUE == bUpdateStatus"));
            if (BTPHN_EN_DIAL_START == arCallStatusInfo[u8Index].enCallStatus)
            {
               if ((BTPHN_EN_DIALING == corfarNewCallStatus[u8Index].enCallStatus) ||
                     (BTPHN_EN_ACTIVE == corfarNewCallStatus[u8Index].enCallStatus) ||
                     (BTPHN_EN_IDLE ==  corfarNewCallStatus[u8Index].enCallStatus))
               {
                  ETG_TRACE_USR4(("vUpdateCallStatus : (corfarNewCallStatus.enCallStatus = %d) and (BTPHN_EN_DIAL_START == arCallStatusInfo)", corfarNewCallStatus[u8Index].enCallStatus));
                  /* The dial call timer is reset if it is running */
                  vStopDialCallTimer();

                  /* Flag and caller ID is reset so that the telephone number
                  is requested again. This is done because there might be a
                  scenario where dial is initiated to a number from hmi and
                  parallely dial to another number is initiated from phone.
                  If BL sets the status as DIAL_START and service updates the
                  status as DIALING for the same call instance but
                  corresponding to the number dialed from phone, then
                  telephone number has to be fetched again to display the
                  number being dialed from phone */
                  arCallStatusInfo[u8Index].bIsContactInfoReq = FALSE;
                  arCallStatusInfo[u8Index].coarCallerId = "";
                  arCallStatusInfo[u8Index].enCallStatus = corfarNewCallStatus[u8Index].enCallStatus;
               }
            }
            else
            {
               ETG_TRACE_USR4(("vUpdateCallStatus : BTPHN_EN_DIAL_START == arCallStatusInfo.enStatus"));
               if ((arCallStatusInfo[u8Index].enCallStatus != BTPHN_EN_IDLE) && (corfarNewCallStatus[u8Index].enCallStatus == BTPHN_EN_IDLE))
               {
                  m_index = u8Index;
                  ETG_TRACE_USR4(("vUpdateCallStatus : m_index = %d", m_index));
               }
               arCallStatusInfo[u8Index].enCallStatus = corfarNewCallStatus[u8Index].enCallStatus;
            }
         }
         /* Caller Id and phone type is reset if the call status is IDLE */
         if (BTPHN_EN_IDLE == corfarNewCallStatus[u8Index].enCallStatus)
         {
            ETG_TRACE_USR4(("vUpdateCallStatus :(BTPHN_EN_IDLE == corfarNewCallStatus[%d].enCallStatus = %d)OR(arCallStatusInfo[%d].enCallStatus(%d) != corfarNewCallStatus[%d].enCallStatus))", u8Index, corfarNewCallStatus[u8Index].enCallStatus, u8Index, arCallStatusInfo[u8Index].enCallStatus, u8Index));
            arCallStatusInfo[u8Index].bIsCallerIdUpdated = FALSE;
            arCallStatusInfo[u8Index].bIsContactInfoReq = FALSE;
            arCallStatusInfo[u8Index].coarCallerId = "";
            arCallStatusInfo[u8Index].enPhoneType = BTPHN_EN_GENERIC;
         }
         else if (arCallStatusInfo[u8Index].enCallStatus != corfarNewCallStatus[u8Index].enCallStatus) /*Bug 1071469*/
         {
            ETG_TRACE_USR4(("vUpdateCallStatus :(arCallStatusInfo[%d].enCallStatus(%d) != corfarNewCallStatus[%d].enCallStatus))", u8Index, arCallStatusInfo[u8Index].enCallStatus, u8Index));
            arCallStatusInfo[u8Index].bIsContactInfoReq = FALSE;
            arCallStatusInfo[u8Index].enPhoneType = BTPHN_EN_GENERIC;
            if (corfarNewCallStatus[u8Index].enCallStatus == BTPHN_EN_ACTIVE)
            {
               ETG_TRACE_USR4(("vUpdateCallStatus:if the call status is active get the phone number of active call and set bIsContactInfoReq TRUE"));
#ifndef VARIANT_S_FTR_ENABLE_PHONECALLMANAGER_SUPPORT
               if (NULL != _most_tel_fi)
               {
                  _most_tel_fi->sendQueryTelephoneNumberStart(*this, arCallStatusInfo[u8Index].u16CallInstance); //bIsCallerIdUpdated flag will be set in onQueryTelephoneNumberResult
               }
               else
               {
                  ETG_TRACE_ERR(("ServiceTel::vUpdateCallStatus _most_tel_fi() NULL Pointer"));
               }
#else
               // Process the received telephone number for the respective call instance
               vProcessAndUpdateTelephoneNumber(arCallStatusInfo[u8Index].strCallStatusExTelNo.c_str(), arCallStatusInfo[u8Index].u16CallInstance);
#endif
               arCallStatusInfo[u8Index].bIsContactInfoReq = TRUE;
            }
            else
            {
               ETG_TRACE_USR4(("vUpdateCallStatus:Callstatus is not active. Telephone number will not be fetched from the MW"));
            }
         }
         else
         {
            ETG_TRACE_USR4(("vUpdateCallStatus : u8NumActiveCalls++"));
            u8CurrentCallInstance = u8Index;
            u8NumActiveCalls++;
            /* If the caller Id is not updated, then the telephone number
            is fetched from phone service */
            if (FALSE == arCallStatusInfo[u8Index].bIsContactInfoReq)
            {
#ifndef VARIANT_S_FTR_ENABLE_PHONECALLMANAGER_SUPPORT
               if (NULL != _most_tel_fi)
               {
                  _most_tel_fi->sendQueryTelephoneNumberStart(*this, arCallStatusInfo[u8Index].u16CallInstance);
               }
#else
               // Process the received telephone number for the respective call instance
               vProcessAndUpdateTelephoneNumber(arCallStatusInfo[u8Index].strCallStatusExTelNo.c_str(), arCallStatusInfo[u8Index].u16CallInstance);
#endif
               arCallStatusInfo[u8Index].bIsContactInfoReq = TRUE;
            }
         }
      }
      /* After populating the call status information, sends the call
      status to HMI Model */
      vProcessCallStatus();
   }
}


/************************************************************************
*FUNCTION: 		vProcessCallStatus
*DESCRIPTION: 	Determines if there is a single call or multiple call and
*					sends the call status information to HMI accordingly.
*PARAMETER:		tVoid
*RETURNVALUE: 	tVoid
*01.11.2018 ess7kor
*Rev 0.1 Initial Revision
************************************************************************/

tVoid ServiceTel::vProcessCallStatus()
{
   ETG_TRACE_USR4(("ServiceTel : vProcessCallStatus()"));

   bIsConference = FALSE;
   /* Boundary Checking to avoid Lint warning */
   if (BTPHN_C_NUM_CALLS_SUPPORTED > u8CurrentCallInstance)
   {
      /* Condition check for multiparty/single call */
      if (BTPHN_C_NUM_CALLS_SUPPORTED == u8NumActiveCalls)
      {
         ETG_TRACE_USR4(("vProcessCallStatus : u8NumActiveCalls = %d", u8NumActiveCalls));
         /* Condition check for conference call */
         if ((arCallStatusInfo[BTPHN_C_FIRST_CALL_INDEX].enCallStatus == arCallStatusInfo[BTPHN_C_SECOND_CALL_INDEX].enCallStatus) &&
               ((BTPHN_EN_ACTIVE == arCallStatusInfo[BTPHN_C_FIRST_CALL_INDEX].enCallStatus) ||
                (BTPHN_EN_ON_HOLD == arCallStatusInfo[BTPHN_C_FIRST_CALL_INDEX].enCallStatus) ||
                (BTPHN_EN_CONFERENCE == arCallStatusInfo[BTPHN_C_FIRST_CALL_INDEX].enCallStatus))
            )
         {
            ETG_TRACE_USR4(("Conference call in progress"));
            bIsConference = TRUE;

            /* If conference call is initiated from phone before completion
            of swap operation */
            u8NumSwapRequests = BTPHN_C_DEFAULT_VALUE_ZERO;
         }
         if (FALSE == bIsConference)
         {
            ETG_TRACE_USR4(("vProcessCallStatus : bIsConference = %d", bIsConference));
            /* If there is a multiparty call, then the active call instance
            is maintained */
            for (tU8 u8Index = BTPHN_C_DEFAULT_VALUE_ZERO; u8Index < BTPHN_C_NUM_CALLS_SUPPORTED; u8Index++)
            {
               if ((BTPHN_EN_ACTIVE == arCallStatusInfo[u8Index].enCallStatus) ||
                     (BTPHN_EN_DIALING == arCallStatusInfo[u8Index].enCallStatus) ||
                     (BTPHN_EN_DIAL_START == arCallStatusInfo[u8Index].enCallStatus))
               {
                  u8CurrentCallInstance = u8Index;
                  ETG_TRACE_USR4(("Active call instance : %d", u8CurrentCallInstance));
                  break;
               }
            }
         }

         vSendMultipleCallStatus(arCallStatusInfo);
      }
      else
      {
         //bruce Long data location for call number and call id
         ETG_TRACE_USR4(("vProcessCallStatus : BTPHN_C_NUM_CALLS_SUPPORTED != u8NumActiveCalls "));
         /* If the single call is being disconnected, then end call timer is
         started to notify the disconnection to the user for a project specific
         time duration */
         VehicleSrvClient::poGetInstance()->vUpdateCallInfo(LONGDATATYPE_CALLNAME, arCallStatusInfo[u8CurrentCallInstance].coarCallerId);
#ifdef VARIANT_S_FTR_ENABLE_PHONECALLMANAGER_SUPPORT
         VehicleSrvClient::poGetInstance()->vUpdateCallInfo(LONGDATATYPE_CALLNUM, arCallStatusInfo[u8CurrentCallInstance].strCallStatusExTelNo);
#endif
         if (BTPHN_EN_DISCONNECTING == arCallStatusInfo[u8CurrentCallInstance].enCallStatus)
         {
            // Set to true as we have an active call
            bIsOngoingCallActive = false;
            bIsIncomingCallPopupActive = false;
            bIsOutgoingCallPopupActive = false;
         }

         if (BTPHN_EN_DISCONNECTING == arCallStatusInfo[u8CurrentCallInstance].enCallStatus)
         {
            // Set to false as we have the call status as disconnecting
            bIsOngoingCallActive = false;
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
            OnUpdMissedIntercomCall();
#endif

            ETG_TRACE_USR4(("vProcessCallStatus : CallStatus = BTPHN_EN_DISCONNECTING"));
            if (TRUE == bIsCallRejected)
            {
               ETG_TRACE_USR4(("vProcessCallStatus: bIsCallRejected = False"));
               /* When an incoming call is rejected, end call timer is not
               started */
               bIsCallRejected = FALSE;
            }
            else
            {
               ETG_TRACE_USR4(("vProcessCallStatus: bIsCallRejected = False"));
               if (FALSE == bIsEndCallTimerRunning)
               {
                  ETG_TRACE_USR4(("vProcessCallStatus: Call disconnecting End call timer is started"));
                  if (OSAL_NULL != hEndCallTimer)
                  {
                     ETG_TRACE_USR4(("vProcessCallStatus: OSAL_NULL != hEndCallTimer"));
                     OSAL_s32TimerSetTime(
                        hEndCallTimer, // Timer handle
                        u32EndCallTimerDuration, // Timer expiry duration
                        BTPHN_C_DEFAULT_VALUE_ZERO // Timer repeat value
                     );
                     bIsEndCallTimerRunning = TRUE;
                     vSendCallStatus(arCallStatusInfo[u8CurrentCallInstance]);
                     //To update the call duration for single call instance to CAN interface
                     if (VehicleSrvClient::poGetInstance() != NULL)
                     {
                        VehicleSrvClient::poGetInstance()->updateCallDuration(arCallStatusInfo[u8CurrentCallInstance]); //CAN Imp
                     }
                  }
               }
            }
         }
         else if (BTPHN_EN_IDLE != arCallStatusInfo[u8CurrentCallInstance].enCallStatus)
         {
            ETG_TRACE_USR4(("vProcessCallStatus: BTPHN_EN_IDLE != enCallStatus"));
            /* The end call timer is reset if it is running */
            if (TRUE == bIsEndCallTimerRunning)
            {
               ETG_TRACE_USR4(("Call is not in IDLE and End call timer is stopped"));
               if (OSAL_NULL != hEndCallTimer)
               {
                  OSAL_s32TimerSetTime(hEndCallTimer,
                                       BTPHN_C_DEFAULT_VALUE_ZERO,
                                       BTPHN_C_DEFAULT_VALUE_ZERO);
                  bIsEndCallTimerRunning = FALSE;
               }

               /* After terminating the end call timer, call status is sent as IDLE
               to HMI so that the end call/disconnecting call view is exited */
               btphn_trCallStatus rIdleStatus;
               //	memset(&rIdleStatus, 0, sizeof(btphn_trCallStatus));
               vSendCallStatus(rIdleStatus);
            }
            vSendCallStatus(arCallStatusInfo[u8CurrentCallInstance]);
            //To update the call duration for single call instance to CAN interface
            if (VehicleSrvClient::poGetInstance() != NULL)
            {
               VehicleSrvClient::poGetInstance()->updateCallDuration(arCallStatusInfo[u8CurrentCallInstance]); //CAN Imp
            }
         }
         else if (BTPHN_EN_IDLE == arCallStatusInfo[u8CurrentCallInstance].enCallStatus)
         {
            ETG_TRACE_USR4(("vProcessCallStatus: BTPHN_EN_IDLE = enCallStatus"));
            //Disconnection State for Call status in a single call instance

            /* If the end call timer is not running, then call status with Idle
            is updated to HMI. If the timer is running, then call status with
            Idle will be updated to HMI when the timer expires */
            if (FALSE == bIsEndCallTimerRunning)
            {
               ETG_TRACE_USR4(("vProcessCallStatus: (BTPHN_EN_IDLE = enCallStatus) and (bIsEndCallTimerRunning = false)"));
               vSendCallStatus(arCallStatusInfo[u8CurrentCallInstance]);
            }
            else
            {
               ETG_TRACE_USR4(("Call is IDLE. End call timer is running"));
            }
         }
      }

      // update the flags for the SWC/MCP Handling
      // Irrespective of the nature of call Single, Multiple and conference call, we update the flags
      // purely based on the status of the call instance

      // flags are reset at start of loop.
      bIsIncomingCallPopupActive = false;
      bIsOutgoingCallPopupActive = false;
      bIsOngoingCallActive = false;
      bIsIdlestatus = false;

      //Bug Fix: Validate call status for the maximum number of supported calls
      for (tU8 u8Index = BTPHN_C_DEFAULT_VALUE_ZERO; u8Index < BTPHN_C_NUM_CALLS_SUPPORTED; u8Index++)
      {
         ETG_TRACE_USR4(("vProcessCallStatus : CallStatus[%d] = %d", u8Index, arCallStatusInfo[u8Index].enCallStatus));
         // As switch case is used only only one case statement will be executed
         // hence, second call instance status, shall not override this value if set by First call instance
         switch (arCallStatusInfo[u8Index].enCallStatus)
         {
            // incoming call scenario
            case BTPHN_EN_RINGTONE :
            {
               bIsIncomingCallPopupActive = true;
               ETG_TRACE_USR4(("vProcessCallStatus : CallStatus[%d] = %d Incoming call flag set ", u8Index, arCallStatusInfo[u8Index].enCallStatus));
            }
            break;

            // active call scenarios
            case BTPHN_EN_ACTIVE :
            case BTPHN_EN_ON_HOLD :
            case BTPHN_EN_BUSY :
            case BTPHN_EN_CONFERENCE :
            case BTPHN_EN_IN_VOICEMAIL :
            case BTPHN_EN_DISCONNECTING :
            {
               bIsOngoingCallActive = true;
               ETG_TRACE_USR4(("vProcessCallStatus : CallStatus[%d] = %d Ongoing call flag set ", u8Index, arCallStatusInfo[u8Index].enCallStatus));
            }
            break;

            // out going call scenarios
            case BTPHN_EN_DIALING :
            case BTPHN_EN_DIAL_START :
            {
               bIsOutgoingCallPopupActive = true;
               ETG_TRACE_USR4(("vProcessCallStatus : CallStatus[%d] = %d outgoing call flag set ", u8Index, arCallStatusInfo[u8Index].enCallStatus));
            }
            break;

            // flag is not updated in the following case, as some disconnection is in progress
            // case BTPHN_EN_DISCONNECTING :
            // break; // Active call shall be disconnected
            case BTPHN_EN_DIAL_ERROR :
               break;   // Outgoing call shall be disconnected
            case BTPHN_EN_IDLE :
            {
               bIsIdlestatus = true;
               //This variable has been provided for GEN CAN purpose alone,
               //since an idle status was also required for phone handling on discussion it can be extended for phone functionality
               //Please do not read this value if its for purposes outside the GEN CAN Scope
            }
            break;
            default:
               break;
         }
      }// end of for loop

      ETG_TRACE_USR4(("ServiceTel::vProcessCallStatus bIsIncomingCallPopupActive = %s", (bIsIncomingCallPopupActive ? "true" : "false")));
      ETG_TRACE_USR4(("ServiceTel::vProcessCallStatus bIsOutgoingCallPopupActive = %s", (bIsOutgoingCallPopupActive ? "true" : "false")));
      ETG_TRACE_USR4(("ServiceTel::vProcessCallStatus bIsOngoingCallActive = %s", (bIsOngoingCallActive ? "true" : "false")));

      //Fix for Bug: 730719, this condition is added for call disconnection
      if (bIsIncomingCallPopupActive == false && bIsOutgoingCallPopupActive == false && bIsOngoingCallActive == false)
      {
         ETG_TRACE_USR4(("vUpdateCallStatus : vDeletecalldetails()"));
         rfoPhoneCallModel.vDeletecalldetails();
         btphn_trCallStatus resetCallDuration;
         //Resetting the call duration value in Generic CAN when the calls have ended
         VehicleSrvClient::poGetInstance()->updateCallDuration(resetCallDuration);
         ETG_TRACE_USR4(("vUpdateCallStatus : updateCallDuration(resetCallDuration)"));
         //LongDataSync
         VehicleSrvClient::poGetInstance()->vUpdateCallInfo(LONGDATATYPE_CALLNUM, BTPHN_C_EMPTY_STRING); //CAN imp
         VehicleSrvClient::poGetInstance()->vUpdateCallInfo(LONGDATATYPE_CALLNAME, BTPHN_C_EMPTY_STRING); //CAN imp
         ETG_TRACE_USR4(("vUpdateCallStatus : vUpdateCallInfo Reset Media Long Data Call name and Number"));
      }
      if (VehicleSrvClient::poGetInstance() != NULL)
      {
         ETG_TRACE_USR4(("vUpdateCallStatus : Sending VehicleSrvClient->updateCallStatus()"));
         VehicleSrvClient::poGetInstance()->updateCallStatus(); //CAN Imp
      }
   }
   //MIC Check for Any Call occuring on Phone
   //To be done only for Coach Variant
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
   vProcessCallMICStatus();
#endif
}


#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
/************************************************************************
*FUNCTION: 		vProcessCallMICStatus
*DESCRIPTION: 	Determines if call status has to be stopped based on the call type and MIC check
*PARAMETER:
*RETURNVALUE:
*AUTHOR: Bruce Netto v1.0 15/10/2020
************************************************************************/
void ServiceTel::vProcessCallMICStatus()
{
   ETG_TRACE_USR4(("ServiceTel::vProcessCallMICStatus"));
   if (MicClientHandler::poGetInstance()->getMicStatus() == true)//MIC Check
   {
      ETG_TRACE_USR4(("ServiceTel::vProcessCallMICStatus : MIC IS ON"));
      if (bIsIncomingCallPopupActive == true)
      {
         ETG_TRACE_USR4(("ServiceTel::vProcessCallMICStatus : bIsIncomingCallPopupActive is true"));
         vIgnoreCallStart(ServiceTel::poGetInstance()->getIncomingCallInstance());
         //POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(MICPhoneCallResp)()));
         if (MicClientHandler::poGetInstance()->getEnMicStatus() == enMIC_IntercomCall)
         {
            ETG_TRACE_USR4(("ServiceTel::vProcessCallMICStatus Intercom call is missed before Bt call"));
            MicClientHandler::poGetInstance()->vSetMicStatus(enMIC_Intercom_BTCall);
         }
         else
         {
            ETG_TRACE_USR4(("ServiceTel::vProcessCallMICStatus BT call is missed"));
            MicClientHandler::poGetInstance()->vSetMicStatus(enMIC_BTCall);
         }
      }
      if (bIsOutgoingCallPopupActive == true)
      {
         ETG_TRACE_USR4(("ServiceTel::vProcessCallMICStatus : bIsOutgoingCallPopupActive is true"));
         POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(CancelDialCallReq)()));
         POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(MICContactsCallHistoryResp)()));
      }
      if (bIsOngoingCallActive == true)
      {
         ETG_TRACE_USR4(("ServiceTel::vProcessCallMICStatus : bIsOngoingCallActive is true"));
         POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(TerminateCallReq)()));
         //POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(MICPhoneCallResp)()));
      }
   }
   //PriorityCall handling
   if ((IsIntercomCallActive == true) || (true == IntercomClienthandler::poGetInstance()->getIntercomIncomingCallStatus()) || (true == IntercomClienthandler::poGetInstance()->getIntercomCallDialingStatus())) //Intercom call active check
   {
      ETG_TRACE_USR4(("ServiceTel::Priority call handling : IsIntercomCallActive == true"));
      if (bIsIncomingCallPopupActive == true)
      {
         ETG_TRACE_USR4(("ServiceTel::Priority call handling : bIsIncomingCallPopupActive is true"));
         vIgnoreCallStart(ServiceTel::poGetInstance()->getIncomingCallInstance());
         // POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ShowPriorityCall_Intercom_Req)()));
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
         m_bMissedBTCall = true;
#endif
      }
      if (bIsOutgoingCallPopupActive == true)
      {
         ETG_TRACE_USR4(("ServiceTel::Priority call handling : bIsOutgoingCallPopupActive is true"));
         POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(CancelDialCallReq)()));
         // POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(MICContactsCallHistoryResp)()));
      }
      if (bIsOngoingCallActive == true)
      {
         ETG_TRACE_USR4(("ServiceTel::Priority call handling : bIsOngoingCallActive is true"));
         POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(TerminateCallReq)()));
         // POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(MICPhoneCallResp)()));
      }
   }
}


void ServiceTel::OnUpdMissedIntercomCall()
{
   ETG_TRACE_USR4(("ServiceTel::OnUpdMissedIntercomCall"));
   if (true  == IntercomClienthandler::poGetInstance()->getMissedIntercomCallStatus())
   {
      IntercomClienthandler::poGetInstance()->setMissedIntercomCallStatus(false);
      IntercomModel::poGetInstance()->OnUpdMissedICallerId(IntercomClienthandler::poGetInstance()->getStrMissedCall());
      POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ShowPriorityCall_BT_Req)()));
   }
}


#endif
/*********************************************************************************/
//void ServiceTel::onCallStatusNoticeExtendedStatus(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >&/*& proxy*/, const ::boost::shared_ptr< ::MOST_Tel_FI:: CallStatusNoticeExtendedStatus >& status)
//{
//	PrintCallStatusNotice(status);
////	HandleCallDuration(status);
//	HandleCallState(status);
//
//}

/************************************************************************
*FUNCTION: 		bIsCallStatusUpdateRequired
*DESCRIPTION: 	Determines if call status has to be updated to HMI. Intermediate
*					call status during multiparty call swap or during change in
*					conference call status (Hold/Active/End) are not updated to HMI.
*PARAMETER:
*					corfenFirstCallStatus:   (I)
*						First call status
*					corfenSecondCallStatus:   (I)
*						Second call status
*
*RETURNVALUE: 	tBool
*					   TRUE, if call status has to be updated to HMI
*						FALSE, otherwise
*
*HISTORY:
*01.11.2018 ess7kor
*Rev 0.1  Initial Revision
03.06.2019 ebg1kor
*Rev 0.2 Initial Revision.
************************************************************************/

tBool ServiceTel::bIsCallStatusUpdateRequired(const btphn_tenPhoneCallStatus& corfenFirstCallStatus, const	btphn_tenPhoneCallStatus& corfenSecondCallStatus)
{
   ETG_TRACE_USR4(("ServiceTel : bIsCallStatusUpdateRequired()"));
   static btphn_tenPhoneCallStatus enPrevCallStatus[BTPHN_C_NUM_CALLS_SUPPORTED] = { BTPHN_EN_IDLE, BTPHN_EN_IDLE };

   static tBool bIgnoreNextStatus = FALSE;
   static tBool bUpdateNextStatus = FALSE;
   tBool bUpdateStatus = TRUE;

   if (TRUE == bIgnoreNextStatus)
   {
      bUpdateStatus = FALSE;
      bIgnoreNextStatus = FALSE;
      bUpdateNextStatus = TRUE;
   }
   else if (TRUE == bUpdateNextStatus)
   {
      bUpdateStatus = TRUE;
      bUpdateNextStatus = FALSE;
   }
   else
   {
      if ((corfenFirstCallStatus == corfenSecondCallStatus) &&
            ((BTPHN_EN_ACTIVE == corfenFirstCallStatus) ||
             (BTPHN_EN_ON_HOLD == corfenFirstCallStatus)))
      {
         if (0 < u8NumSwapRequests) // When swap is initiated from HMI
         {
            ETG_TRACE_USR4(("Ignoring intermediate call "
                            "status update during swap"));
            bUpdateStatus = FALSE;
            u8NumSwapRequests--;
         }
         /* When swap is initiated from phone. Previous state is compared here
         to differentiate scenario where multiparty call is combined to
         active conference call */
         else if ((arCallStatusInfo[BTPHN_C_FIRST_CALL_INDEX].enCallStatus != arCallStatusInfo[BTPHN_C_SECOND_CALL_INDEX].enCallStatus) &&
                  (enPrevCallStatus[BTPHN_C_FIRST_CALL_INDEX] !=	enPrevCallStatus[BTPHN_C_SECOND_CALL_INDEX])
                 )
         {
            bUpdateStatus = FALSE;
         }
      }

      /* The below condition check will be met when conference call is put on
      hold or a conference call is split to multiparty call */
      else if (((BTPHN_EN_CONFERENCE != corfenFirstCallStatus) &&	(BTPHN_EN_CONFERENCE == corfenSecondCallStatus)) ||
               ((BTPHN_EN_CONFERENCE == corfenFirstCallStatus) && (BTPHN_EN_CONFERENCE != corfenSecondCallStatus)))
      {
         ETG_TRACE_USR4(("Ignoring intermediate call status update when "
                         "conference call is being put on hold"));

         /* The below condition check will be met when conference call is
         put on hold */
         if ((BTPHN_EN_ON_HOLD == corfenFirstCallStatus) ||	(BTPHN_EN_ON_HOLD == corfenSecondCallStatus))
         {
            bIgnoreNextStatus = TRUE;
         }
         bUpdateStatus = FALSE;
      }

      /* The below condition check will be met when confernce call status
      is changed from hold to active or when a conference call is
      terminated from HMI */
      else if ((arCallStatusInfo[BTPHN_C_FIRST_CALL_INDEX].enCallStatus == arCallStatusInfo[BTPHN_C_SECOND_CALL_INDEX].enCallStatus) &&
               (corfenFirstCallStatus != corfenSecondCallStatus) &&
               (BTPHN_EN_IDLE != arCallStatusInfo[BTPHN_C_FIRST_CALL_INDEX].enCallStatus))
      {
         bUpdateStatus = FALSE;
         bUpdateNextStatus = TRUE;
      }

      /* Whenever receiving the DIALING status on DIAL_START request, update the flag bUpdateStatus to TRUE as to update the call status */
      if (((BTPHN_EN_DIAL_START == arCallStatusInfo[BTPHN_C_FIRST_CALL_INDEX].enCallStatus) && (BTPHN_EN_DIALING == corfenFirstCallStatus)) ||
            ((BTPHN_EN_DIAL_START == arCallStatusInfo[BTPHN_C_SECOND_CALL_INDEX].enCallStatus) && (BTPHN_EN_DIALING == corfenSecondCallStatus)))
      {
         bUpdateStatus = TRUE;
      }
   }

   /*Whenever the CallStatus goes to IDLE and PrevCallStatus not equal to IDLE,then we will reset the CallFlag to False.
   * It will prevent delays due to service response delays.Because in some scenario, User may trigger new call in the mean time. Such type of use cases will be handled here*/

   if (((BTPHN_EN_IDLE != enPrevCallStatus[BTPHN_C_FIRST_CALL_INDEX]) && (BTPHN_EN_IDLE == corfenFirstCallStatus)) ||
         ((BTPHN_EN_IDLE != enPrevCallStatus[BTPHN_C_SECOND_CALL_INDEX]) && (BTPHN_EN_IDLE == corfenSecondCallStatus)))
   {
      vResetCallFlag();
   }

   if ((BTPHN_EN_IDLE == enPrevCallStatus[BTPHN_C_FIRST_CALL_INDEX]) && (BTPHN_EN_DIALING  ==  corfenFirstCallStatus))

   {
      /* Post the Courier message only when Android auto session and Carplay is not active */
      if ((g_SM_SpiSessionStatus != SPI_SESSION_ANDROID_AUTO_ACTIVE) && (g_SM_SpiSessionStatus != SPI_SESSION_CARPLAY_ACTIVE))
      {
         ETG_TRACE_USR4(("onCallStatusNoticeStatus:Dailing "));
         // Introducing new flow of going through RequestCallConnectingMsg for call connecting, hence avoiding the burden of PhoneGadgetRequestMsg
         //POST_MSG((COURIER_MESSAGE_NEW(PhoneGadgetRequestMsg)(7)));
         POST_MSG((COURIER_MESSAGE_NEW(RequestCallConnectingMsg)(7)));
      }
   }

   if ((BTPHN_EN_RINGTONE == enPrevCallStatus[BTPHN_C_FIRST_CALL_INDEX]) && (BTPHN_EN_ACTIVE == corfenFirstCallStatus))
   {
      /* Post the Courier message only when Android auto session and Carplay is not active */
      if ((g_SM_SpiSessionStatus != SPI_SESSION_ANDROID_AUTO_ACTIVE) && (g_SM_SpiSessionStatus != SPI_SESSION_CARPLAY_ACTIVE))
      {
         ETG_TRACE_USR4(("bIsCallStatusUpdateRequired:Accept"));
         POST_MSG((COURIER_MESSAGE_NEW(OnIncomingCallRespMsg)()));
      }
   }

   /* Add EndCall check criteria for both call instances */
   if ((BTPHN_EN_IDLE == corfenFirstCallStatus) && (BTPHN_EN_IDLE == corfenSecondCallStatus))
   {
      if (((BTPHN_EN_ACTIVE == enPrevCallStatus[BTPHN_C_FIRST_CALL_INDEX]) && (BTPHN_EN_IDLE == corfenFirstCallStatus))
            || ((BTPHN_EN_ACTIVE == enPrevCallStatus[BTPHN_C_SECOND_CALL_INDEX]) && (BTPHN_EN_IDLE == corfenSecondCallStatus))
            || ((BTPHN_EN_ON_HOLD == enPrevCallStatus[BTPHN_C_FIRST_CALL_INDEX]) && (BTPHN_EN_IDLE == corfenFirstCallStatus))
            || ((BTPHN_EN_ON_HOLD == enPrevCallStatus[BTPHN_C_SECOND_CALL_INDEX]) && (BTPHN_EN_IDLE == corfenSecondCallStatus))
            || ((BTPHN_EN_DISCONNECTING == enPrevCallStatus[BTPHN_C_FIRST_CALL_INDEX]) && (BTPHN_EN_IDLE == corfenFirstCallStatus))
            || ((BTPHN_EN_DISCONNECTING == enPrevCallStatus[BTPHN_C_SECOND_CALL_INDEX]) && (BTPHN_EN_IDLE == corfenSecondCallStatus))
            || ((BTPHN_EN_DIALING == enPrevCallStatus[BTPHN_C_FIRST_CALL_INDEX]) && (BTPHN_EN_IDLE  ==  corfenFirstCallStatus)))
      {
         /* Post the Courier message only when Android auto session and Carplay is not active */
         if ((g_SM_SpiSessionStatus != SPI_SESSION_ANDROID_AUTO_ACTIVE) && (g_SM_SpiSessionStatus != SPI_SESSION_CARPLAY_ACTIVE))
         {
            ETG_TRACE_USR4(("bIsCallStatusUpdateRequired:EndCall "));
            POST_MSG((COURIER_MESSAGE_NEW(OnCallEndRespMsg)()));
         }
      }
   }

   enPrevCallStatus[BTPHN_C_FIRST_CALL_INDEX] = corfenFirstCallStatus;
   enPrevCallStatus[BTPHN_C_SECOND_CALL_INDEX] = corfenSecondCallStatus;
   return bUpdateStatus;
}


/************************************************************************
*FUNCTION: 		vSendMultipleCallStatus
*DESCRIPTION: 	Call Status update for Multiple Call Instance.
*PARAMETER:
*					coarCallStatus:   (I)
*						Structure variable containing call status information for each
*					   call
*RETURNVALUE: 	tVoid
*
*HISTORY:
*01.11.2018 ess7kor
*Rev 0.1 Initial Revision.
************************************************************************/
tVoid ServiceTel::vSendMultipleCallStatus(const btphn_trCallStatus coarCallStatus[])
{
   ETG_TRACE_USR4(("ServiceTel : vSendMultipleCallStatus()"));

   /* Creating the call duration string */
   std::string strFirstCallDuration = AppHmiPhone_Helper::ConvertCallDuration(
                                         coarCallStatus[BTPHN_C_FIRST_CALL_INDEX].u8CallDurationHr,
                                         coarCallStatus[BTPHN_C_FIRST_CALL_INDEX].u8CallDurationMin,
                                         coarCallStatus[BTPHN_C_FIRST_CALL_INDEX].u8CallDurationSec);

   std::string strSecondCallDuration = AppHmiPhone_Helper::ConvertCallDuration(
                                          coarCallStatus[BTPHN_C_SECOND_CALL_INDEX].u8CallDurationHr,
                                          coarCallStatus[BTPHN_C_SECOND_CALL_INDEX].u8CallDurationMin,
                                          coarCallStatus[BTPHN_C_SECOND_CALL_INDEX].u8CallDurationSec);

   //To update the call duration based on the active/Conference/hold call for CAN interface

   if ((BTPHN_EN_ACTIVE == coarCallStatus[BTPHN_C_FIRST_CALL_INDEX].enCallStatus) ||
         (BTPHN_EN_CONFERENCE == coarCallStatus[BTPHN_C_FIRST_CALL_INDEX].enCallStatus))
   {
      if (VehicleSrvClient::poGetInstance() != NULL)
      {
         VehicleSrvClient::poGetInstance()->updateCallDuration(coarCallStatus[BTPHN_C_FIRST_CALL_INDEX]); //CAN Imp
      }
   }
   else if ((BTPHN_EN_ACTIVE == coarCallStatus[BTPHN_C_SECOND_CALL_INDEX].enCallStatus) ||
            (BTPHN_EN_CONFERENCE == coarCallStatus[BTPHN_C_SECOND_CALL_INDEX].enCallStatus))
   {
      if (VehicleSrvClient::poGetInstance() != NULL)
      {
         VehicleSrvClient::poGetInstance()->updateCallDuration(coarCallStatus[BTPHN_C_SECOND_CALL_INDEX]); //CAN Imp
      }
   }

   ETG_TRACE_USR3(("First Caller Id = %s",	coarCallStatus[BTPHN_C_FIRST_CALL_INDEX].coarCallerId));
   ETG_TRACE_USR3(("Second Caller Id = %s", coarCallStatus[BTPHN_C_SECOND_CALL_INDEX].coarCallerId));
   ETG_TRACE_USR3(("First Call status = %d", coarCallStatus[BTPHN_C_FIRST_CALL_INDEX].enCallStatus));
   ETG_TRACE_USR3(("Second Call status = %d", coarCallStatus[BTPHN_C_SECOND_CALL_INDEX].enCallStatus));
   ETG_TRACE_USR3(("First Call Duration = %s", strFirstCallDuration.c_str()));
   ETG_TRACE_USR3(("Second Call Duration = %s", strSecondCallDuration.c_str()));

   /* Determining the contact number type */
   tCString acopcContactNumType[] = { BTPHN_C_LIST_ICON_CELL,
                                      BTPHN_C_LIST_ICON_CELL_HIGHLIGHTED,
                                      BTPHN_C_LIST_ICON_HOME,
                                      BTPHN_C_LIST_ICON_HOME_HIGHLIGHTED,
                                      BTPHN_C_LIST_ICON_WORK,
                                      BTPHN_C_LIST_ICON_WORK_HIGHLIGHTED,
                                      BTPHN_C_LIST_ICON_OTHER,
                                      BTPHN_C_LIST_ICON_OTHER_HIGHLIGHTED,
                                      BTPHN_C_LIST_ICON_GENERIC,
                                      BTPHN_C_LIST_ICON_GENERIC_HIGHLIGHTED
                                    };

   tU8 u8FirstType =
      ((BTPHN_EN_ACTIVE == coarCallStatus[BTPHN_C_FIRST_CALL_INDEX].enCallStatus) ||
       (BTPHN_EN_CONFERENCE == coarCallStatus[BTPHN_C_FIRST_CALL_INDEX].enCallStatus))
      ? (((coarCallStatus[BTPHN_C_FIRST_CALL_INDEX].enPhoneType) * 2) + 1)
      : ((coarCallStatus[BTPHN_C_FIRST_CALL_INDEX].enPhoneType) * 2);

   tU8 u8SecondType =
      ((BTPHN_EN_ACTIVE == coarCallStatus[BTPHN_C_SECOND_CALL_INDEX].enCallStatus) ||
       (BTPHN_EN_CONFERENCE == coarCallStatus[BTPHN_C_SECOND_CALL_INDEX].enCallStatus))
      ? (((coarCallStatus[BTPHN_C_SECOND_CALL_INDEX].enPhoneType) * 2) + 1)
      : ((coarCallStatus[BTPHN_C_SECOND_CALL_INDEX].enPhoneType) * 2);

   tCString copcFirstContactType = acopcContactNumType[u8FirstType];
   tCString copcSecondContactType = acopcContactNumType[u8SecondType];
   ETG_TRACE_USR3(("First Contact Type = %s", copcFirstContactType));
   ETG_TRACE_USR3(("Second Contact Type = %s", copcSecondContactType));

   /* Post the update to Model and Controller only when Android auto session and Carplay is not active */
   if ((g_SM_SpiSessionStatus != SPI_SESSION_ANDROID_AUTO_ACTIVE) && (g_SM_SpiSessionStatus != SPI_SESSION_CARPLAY_ACTIVE))
   {
      /* Creating MultipleCallStatusResp courier message */
      MultipleCallStatusResp* poMultipleCallStatusResp =
         COURIER_MESSAGE_NEW(MultipleCallStatusResp)
         ((Candera::String)coarCallStatus[BTPHN_C_FIRST_CALL_INDEX].coarCallerId.c_str(),
          (Candera::String)strFirstCallDuration.c_str(),
          coarCallStatus[BTPHN_C_FIRST_CALL_INDEX].enCallStatus,
          (Candera::String)copcFirstContactType,
          (Candera::String)coarCallStatus[BTPHN_C_SECOND_CALL_INDEX].coarCallerId.c_str(),
          (Candera::String)strSecondCallDuration.c_str(),
          coarCallStatus[BTPHN_C_SECOND_CALL_INDEX].enCallStatus,
          (Candera::String)copcSecondContactType);

      rfoPhoneCallModel.OnUpdMultipleCallStatusResp(coarCallStatus[BTPHN_C_FIRST_CALL_INDEX].coarCallerId.c_str(),
            strFirstCallDuration.c_str(),
            coarCallStatus[BTPHN_C_FIRST_CALL_INDEX].enCallStatus,
            copcFirstContactType,
            coarCallStatus[BTPHN_C_SECOND_CALL_INDEX].coarCallerId.c_str(),
            strSecondCallDuration.c_str(),
            coarCallStatus[BTPHN_C_SECOND_CALL_INDEX].enCallStatus,
            copcSecondContactType);

      if (NULL != poMultipleCallStatusResp)
      {
         ETG_TRACE_USR3(("ServiceTel : Sending MultipleCallStatusResp' message"));
         /* Posting MultipleCallStatusResp courier message to HMI Model */
         tBool bRetVal = poMultipleCallStatusResp->Post();
         if (FALSE == bRetVal)
         {
            ETG_TRACE_ERR(("ServiceTel: vSendMultipleCallStatus()\n Sending MultipleCallStatusResp' message failed"));
         }
      }
   }
}


/************************************************************************
*FUNCTION: 		vSendCallStatus
*DESCRIPTION: 	Call Status update for Single Call Instance
*PARAMETER:
*					corfrCallStatus:   (I)
*						Structure variable containing call status information
*RETURNVALUE: 	tVoid
*
*HISTORY:
*01.11.2018 ess7kor
*Rev 0.1 Initial Revision.
************************************************************************/
tVoid ServiceTel::vSendCallStatus(const btphn_trCallStatus& corfrCallStatus)
{
   ETG_TRACE_USR4(("ServiceTel : vSendCallStatus()"));
   ETG_TRACE_USR3(("Caller Id = %s", corfrCallStatus.coarCallerId.c_str()));
   ETG_TRACE_USR3(("Call status = %d", corfrCallStatus.enCallStatus));

   //static btphn_tenPhoneCallStatus enPrevCallStatus = BTPHN_EN_IDLE;

   /* Creating the call duration string */
   std::string acDuration = AppHmiPhone_Helper::ConvertCallDuration(corfrCallStatus.u8CallDurationHr,
                            corfrCallStatus.u8CallDurationMin,
                            corfrCallStatus.u8CallDurationSec);

   ETG_TRACE_USR3(("Call Duration = %s", acDuration.c_str()));
   ETG_TRACE_USR3(("Contact Type = %d", corfrCallStatus.enPhoneType));

   /* Determining the contact number type */
   std::string acopcContactNumType[] = { BTPHN_C_LIST_ICON_CELL,
                                         BTPHN_C_LIST_ICON_HOME,
                                         BTPHN_C_LIST_ICON_WORK,
                                         BTPHN_C_LIST_ICON_OTHER,
                                         BTPHN_C_LIST_ICON_GENERIC
                                       };

   std::string copcContactType = "";

   tU8 u_ContactNumType = sizeof(acopcContactNumType) / sizeof(acopcContactNumType[0]);

   if ((u_ContactNumType > corfrCallStatus.enPhoneType) && (0 <= corfrCallStatus.enPhoneType))
   {
      copcContactType = acopcContactNumType[corfrCallStatus.enPhoneType];
      ETG_TRACE_USR3(("Contact Type = %s", copcContactType.c_str()));
   }
   else
   {
      copcContactType = acopcContactNumType[BTPHN_EN_OTHER];
      ETG_TRACE_USR3(("Contact Type = %s", copcContactType.c_str()));
   }

   /* Post the update to Model and Controller only when Android auto session and Carplay is not active */
   if ((g_SM_SpiSessionStatus != SPI_SESSION_ANDROID_AUTO_ACTIVE) && (g_SM_SpiSessionStatus != SPI_SESSION_CARPLAY_ACTIVE))
   {
      /* Creating CallStatusResp courier message */
      CallStatusResp* poCallStatusResp = COURIER_MESSAGE_NEW(CallStatusResp)
                                         (corfrCallStatus.u16CallInstance,
                                          (Candera::String)corfrCallStatus.coarCallerId.c_str(),
                                          (Candera::String)acDuration.c_str(),
                                          corfrCallStatus.enCallStatus,
                                          (Candera::String)copcContactType.c_str()
                                         );
      rfoPhoneCallModel.OnUpdCallStatus(corfrCallStatus.u16CallInstance, acDuration.c_str(), corfrCallStatus.coarCallerId.c_str(),
                                        corfrCallStatus.enCallStatus, copcContactType.c_str());

      if (NULL != poCallStatusResp)
      {
         ETG_TRACE_USR3(("ServiceTel : Sending CallStatusResp message"));
         /* Posting CallStatusResp courier message to HMI Model */
         tBool bRetVal = poCallStatusResp->Post();
         if (FALSE == bRetVal)
         {
            ETG_TRACE_ERR(("ServiceTel: vSendCallStatus()\n Sending CallStatusResp message failed"));
         }
      }
      else
      {
         ETG_TRACE_ERR(("ServiceTel: vSendCallStatus()\n Empty CallStatusResp message"));
      }
   }
   //enPrevCallStatus = corfrCallStatus.enCallStatus;

   //supriya enPrevCallStatus is used in suzuki to send a global event to other apps about the call status. this needs
   //to be discussed with HMI Master team and accordingly done
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void ServiceTel::onCellDeviceNameError(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::CellDeviceNameError >& error)
{
   //Add your code here
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void ServiceTel::onCellDeviceNameStatus(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::CellDeviceNameStatus >& status)
{
   //TODO: Varun changes for Device name
   //bruce device name long data check in log if this is the right function
   //bruce OPL what to send if network operator is empty?
   ETG_TRACE_USR2(("onCellDeviceNameStatus: Device Name:%s", status->getSDeviceName().c_str()));
   strDeviceNameCurrent = status->getSDeviceName().c_str();
   //LongDataSync for Device Name Not to be confused with Network provider due to interfaces used
   VehicleSrvClient::poGetInstance()->vUpdatePhoneInfo(LONGDATATYPE_PHONENAME, strDeviceNameCurrent.c_str());
   ETG_TRACE_USR2(("onCellDeviceNameStatus: status->getU8DeviceHandle() :%d", m_u8ActiveDev));
#ifdef VARIANT_S_FTR_ENABLE_BLUETOOTH_ON_OFF
   if ((status->getU8DeviceHandle() == 0) && (Sm_ActiveSource == 0) && (ServiceBT::poGetInstance()->GetServiceAvailableStatus() == true))
   {
      //Fix for Bug: 866525, When No phone is paired/connected
      strDeviceNameCurrent = "";
      ETG_TRACE_USR2(("onCellDeviceNameStatus: getU8DeviceHandle() = 0"));
      rfoPhoneCallModel.OnUpdNetworkOperatorName(strDeviceNameCurrent.c_str());
      return;
   }
#endif
#ifndef VARIANT_S_FTR_ENABLE_BLUETOOTH_ON_OFF
   if ((status->getU8DeviceHandle() == 0) && (Sm_ActiveSource == 0))
   {
      //Fix for Bug: 866525, When No phone is paired/connected
      strDeviceNameCurrent = "";
      ETG_TRACE_USR2(("onCellDeviceNameStatus: getU8DeviceHandle() = 0"));
      rfoPhoneCallModel.OnUpdNetworkOperatorName(strDeviceNameCurrent.c_str());
      return;
   }
#endif
   if ((status->getU8DeviceHandle() == m_u8ActiveDev) && (Sm_ActiveSource == 0))   //check if the status received is of the current active device
   {
#ifdef VARIANT_S_FTR_ENABLE_BLUETOOTH_ON_OFF
      if ((ServiceBT::poGetInstance()->GetBluetoothOnOffStatus() == true) && (ServiceBT::poGetInstance()->GetServiceAvailableStatus() == true))
      {
#endif
         if (!(!strDeviceNameCurrent.empty()))
         {
            ETG_TRACE_USR2(("onCellDeviceNameStatus empty"));
            strDeviceNameCurrent = "";
            rfoPhoneCallModel.OnUpdNetworkOperatorName(strDeviceNameCurrent.c_str());
         }
         else
         {
            ETG_TRACE_USR2(("ServiceTel::onCellDeviceNameStatus :"));
            rfoPhoneCallModel.OnUpdNetworkOperatorName(strDeviceNameCurrent.c_str());
         }
#ifdef VARIANT_S_FTR_ENABLE_BLUETOOTH_ON_OFF
      }
      else if ((ServiceBT::poGetInstance()->GetBluetoothOnOffStatus() == false) && (ServiceBT::poGetInstance()->GetServiceAvailableStatus() == true))
      {
         ETG_TRACE_USR2(("DeviceListCallModel::Bluetooth OFF"));
         rfoPhoneCallModel.OnUpdNetworkOperatorName("Bluetooth Off");
      }
#endif
   }
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void ServiceTel::onCellNetworkStatusError(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::CellNetworkStatusError >& error)
{
   ETG_TRACE_USR2(("ServiceTel::onCellNetworkStatusError"));
   if (error->hasSystemErrorCode() == true)
   {
      ETG_TRACE_USR4(("onCellNetworkStatusError:SYSTEM Error code"));
      VehicleSrvClient::poGetInstance()->updatePhoneRoamingError(PHONEMW_SNA);//Gen Can Imp for roaming error code
   }
   else if (error->hasCcaErrorCode() == true)
   {
      ETG_TRACE_USR4(("onCellNetworkStatusError:CCA Error code"));
      VehicleSrvClient::poGetInstance()->updatePhoneRoamingError(CAN_ERROR);//Gen Can Imp for roaming error code
   }
   else
   {
      ETG_TRACE_USR4(("onCellNetworkStatusError:Unknown Error"));
      VehicleSrvClient::poGetInstance()->updatePhoneRoamingError(PHONEMW_SNA);//Gen Can Imp for roaming error code
   }
}


/*****************************************************************************
* @brief   This function gives the Connectivity status of the cell network. This
is a notification since, network could be lost anytime. Includes roaming true or
false.
*v0.1 Sai Prabha: 360317: Phone Network strenght is observed in the Home screen ,
when the Phone is Paired without SIM Inserted.
*****************************************************************************/
void ServiceTel::onCellNetworkStatusStatus(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::CellNetworkStatusStatus >& status)
{
   //Add your code here
   if (status->getU8DeviceHandle() == 0)
   {
      return;
   }
   if (status->getU8DeviceHandle() == m_u8ActiveDev) //check if the status received is of the current active device
   {
      if (0 == status->getE8CellService())//IDLE Network -0 //Ringing Network -1
      {
         ETG_TRACE_USR2(("Servie Status :%d", status->getE8CellService()));
         _most_tel_fi->sendCellDeviceNameGet(*this);
      }
      else
      {
         ETG_TRACE_USR2(("Roaming Status :%d", status->getBRoamingStatus()));
         rfoCallHistoryModel.OnUpdCellNetworkStatus((btphn_tenNetworkServiceState)status->getE8CellService(), status->getBRoamingStatus());
      }
      if (VehicleSrvClient::poGetInstance() != NULL)
      {
         VehicleSrvClient::poGetInstance()->updatePhoneRoamingStatus(status->getBRoamingStatus());//Gen Can Imp for roaming status
      }
   }
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void ServiceTel::onCellSignalStrengthError(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::CellSignalStrengthError >& error)
{
   ETG_TRACE_USR2(("ServiceTel::onCellSignalStrengthError"));
   //Update the Signal strength error status to the CAN Interface
   if (error->hasSystemErrorCode() == true)
   {
      ETG_TRACE_USR4(("onCellSignalStrengthError:SYSTEM Error code"));
      VehicleSrvClient::poGetInstance()->updateGSMSignalStrengthError(PHONEMW_SNA);//Gen Can Imp for roaming error code
   }
   else if (error->hasCcaErrorCode() == true)
   {
      ETG_TRACE_USR4(("onCellSignalStrengthError:CCA Error code"));
      VehicleSrvClient::poGetInstance()->updateGSMSignalStrengthError(CAN_ERROR);//Gen Can Imp for roaming error code
   }
   else
   {
      ETG_TRACE_USR4(("onCellSignalStrengthError:Unknown Error"));
      VehicleSrvClient::poGetInstance()->updateGSMSignalStrengthError(PHONEMW_SNA);//Gen Can Imp for roaming error code
   }
}


/*****************************************************************************
* @brief   This is a notification
since the signal strength can change.
v0.1 Sai Prabha , BugFix:272500: Signal Strength not correct in Home screen - Phone tile
*****************************************************************************/
void ServiceTel::onCellSignalStrengthStatus(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::CellSignalStrengthStatus >& status)
{
   if (status->getU8DeviceHandle() == 0)
   {
      return;
   }
   if (status->getU8DeviceHandle() == m_u8ActiveDev) //check if the status received is of the current active device
   {
      ETG_TRACE_USR2(("onCellSignalStrengthStatus: signal : %d", ETG_CENUM(tU8, (tU8)status->getU8SignalSrength())));
      u8NetworkStrength = status->getU8SignalSrength();
      rfoCallHistoryModel.OnUpdCellSignalStrength(status->getU8SignalSrength());
      if (VehicleSrvClient::poGetInstance() != NULL)
      {
         VehicleSrvClient::poGetInstance()->updateGSMSignalStrength(status->getU8SignalSrength()); //CAN Imp
      }
   }
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void ServiceTel::onCellStateofChargeError(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::CellStateofChargeError >& error)
{
   //Update the Battery charge error status to the CAN Interface
   ETG_TRACE_USR4(("ServiceTel::onCellStateofChargeError"));
   if (error->hasSystemErrorCode() == true)
   {
      ETG_TRACE_USR4(("onCellStateofChargeError:SYSTEM Error code"));
      VehicleSrvClient::poGetInstance()->updatePhoneBatteryLevelError(PHONEMW_SNA);//Gen Can Imp for roaming error code
   }
   else if (error->hasCcaErrorCode() == true)
   {
      ETG_TRACE_USR4(("onCellStateofChargeError:CCA Error code"));
      VehicleSrvClient::poGetInstance()->updatePhoneBatteryLevelError(CAN_ERROR);//Gen Can Imp for roaming error code
   }
   else
   {
      ETG_TRACE_USR4(("onCellStateofChargeError:Unknown Error"));
      VehicleSrvClient::poGetInstance()->updatePhoneBatteryLevelError(PHONEMW_SNA);
   }
}


/*****************************************************************************
* @brief  Property contains the current cell phone charge status from 1 to 5
v0.1 Sai Prabha 271828: Battery information is not seen when phone is connected and vice versa in Home screen
*****************************************************************************/
void ServiceTel::onCellStateofChargeStatus(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::CellStateofChargeStatus >& status)
{
   ETG_TRACE_USR2(("onCellStateofChargeStatus: u8StateofCharge : %d", ETG_CENUM(tU8, (tU8)status->getU8StateofCharge())));
   if (status->getU8DeviceHandle() == 0)
   {
      return;
   }
   //u8ChargeStatus = status->getU8StateofCharge();
   if (status->getU8DeviceHandle() == m_u8ActiveDev) //check if the status received is of the current active device
   {
      ETG_TRACE_USR2(("onCellStateofChargeStatus: %d", ETG_CENUM(tU8, (tU8)status->getU8StateofCharge())));
      rfoCallHistoryModel.SetBatteryIndicator(); //Bug Fix for 973141
      rfoCallHistoryModel.OnUpdCellStateOfCharge(status->getU8StateofCharge());
      u8ChargeStatus = status->getU8StateofCharge();
      //Update the Battery charge status to the CAN Interface
      if (VehicleSrvClient::poGetInstance() != NULL)
      {
         VehicleSrvClient::poGetInstance()->updatePhoneBatteryLevel(u8ChargeStatus); //CAN Imp
      }
   }
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void ServiceTel::onInComingCallEventError(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::InComingCallEventError >& error)
{
   //Add your code here
}


/************************************************************************
*FUNCTION:     onInComingCallEventStatus
*DESCRIPTION:  Receives the updated value of the property
*              IncomingCallEventStatus.
*PARAMETER:
*              status:   (->I)
*                 Message object
*
*RETURNVALUE:	tVoid
*
*HISTORY:
*01.15.2018 ess7kor
*Rev 0.1 Initial Revision
************************************************************************/
void ServiceTel::onInComingCallEventStatus(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::InComingCallEventStatus >& status)
{
   tU8 u8DeviceType = 0;
   std::string strDeviceName;
   tU8 u8DeviceHandle = 0;

   if (_most_tel_fi)
   {
      printIncomingCallEventStatus(status);

      u8IncomingCallInstance = status->getU16CallInstance();
      btphn_tenPhoneIncomingCallStatus enCallStatus = static_cast<btphn_tenPhoneIncomingCallStatus>(status->getE8IncomingCallEventEnum());
      std::string strTelephoneNumber;
      std::string strContactName;
      tPChar pcContactName = NULL;

      /* Fix for SUZUKI-16076. The dial call timer is reset if it is running as well as either of the call instance is in DIAL_START*/
      //Todo bIsDialCallTimerRunning
      if ((TRUE == bIsDialCallTimerRunning) || (BTPHN_EN_DIAL_START == arCallStatusInfo[BTPHN_C_FIRST_CALL_INDEX].enCallStatus) || (BTPHN_EN_DIAL_START == arCallStatusInfo[BTPHN_C_SECOND_CALL_INDEX].enCallStatus))
      {
         vStopDialCallTimer();

         /* DIAL_START status is updated to RINGING or IDLE depending upon the
         call instance */
         if (BTPHN_EN_DIAL_START == arCallStatusInfo[u8IncomingCallInstance].enCallStatus)
         {
            arCallStatusInfo[u8IncomingCallInstance].enCallStatus = BTPHN_EN_RINGTONE;
            arCallStatusInfo[u8IncomingCallInstance].bIsContactInfoReq = FALSE;
         }
         else
         {
            /* If one call status is DIAL_START and the other call status is
            RINGING, then dialing will not happen. Hence the DIAL_START status
            is updated to IDLE */
            tU16 u16DialCallInstance = (BTPHN_C_FIRST_CALL_INDEX == u8IncomingCallInstance) ? BTPHN_C_SECOND_CALL_INDEX : BTPHN_C_FIRST_CALL_INDEX;
            if (BTPHN_EN_DIAL_START == arCallStatusInfo[u16DialCallInstance].enCallStatus)
            {
               arCallStatusInfo[u16DialCallInstance].coarCallerId = "";
               arCallStatusInfo[u16DialCallInstance].enCallStatus = BTPHN_EN_IDLE;
               arCallStatusInfo[u16DialCallInstance].bIsContactInfoReq = FALSE;
            }
         }
      }

      /* Processing of passive device incoming call as per the feature: SYSFL-5565 */
      /* When incoming call comes for passive device while already call is in progress for active device, show passive incoming call pop-up*/
      if ((m_u8ActiveDev != status->getU8DeviceHandle()) && (BTPHN_EN_INIDLE != enCallStatus) && (bIsOngoingCallActive == true))
      {
         /* Assign device ID of 2 for passive device */
         u8DeviceType = INCOMING_CALL_DEVICETYPE_PASSIVE;
         /*Store active call instance and passive dev handle, to hang-up the ongoing active call on acceptance of passive incoming call */
         u8PreviousCallInstance = arCallStatusInfo[u8CurrentCallInstance].u16CallInstance;
         m_u8PassiveDevHandle = status->getU8DeviceHandle();
         ETG_TRACE_USR3(("Passive Incoming Call :Previous Call Instance = %d", u8PreviousCallInstance));
         ETG_TRACE_USR3(("Passive Incoming Call :Passive dev handle = %d", m_u8PassiveDevHandle));

         /* Maintaining incoming caller information */
         bIsIncomingCallPopupActive = true;

         /* Get the passive device name from ServiceBT */
         if (ServiceBT::poGetInstance() != NULL)
         {
            ServiceBT::poGetInstance()->GetPassivePhoneDetails(strDeviceName, u8DeviceHandle);
            ETG_TRACE_USR3(("Passive Incoming call: Passive DeviceName = %s", strDeviceName.c_str()));
         }
      }
      else
      {
         /* Assign device ID of 1 for active device */
         u8DeviceType = INCOMING_CALL_DEVICETYPE_ACTIVE;
         m_u8PassiveDevHandle = 0;
      }

      if (BTPHN_EN_INIDLE != enCallStatus)
      {
         most_PhonBk_fi_types::T_e8_PhonBkPreferredPhoneBookSortOrder sortorder = T_e8_PhonBkPreferredPhoneBookSortOrder__e8PREFERRED_SORT_ORDER_FIRSTNAME;

         if (ServicePhoneBook::poGetInstance() != NULL)
         {
            sortorder = ServicePhoneBook::poGetInstance()->enGetSortOrder(); //temporary untill P_OPT scene implementation
         }

         std::string strFirstName = status->getSFirstName();
         std::string strLastName = status->getSLastName();

         strContactName = AppHmiPhone_Helper::GetContactname(strFirstName, strLastName, sortorder);
         ETG_TRACE_USR3(("Contact name = %s", strContactName.c_str()));

         /* Maintaining incoming caller information */
         bIsIncomingCallPopupActive = true;
         if (u8NumActiveCalls > 0)
         {
            /* When call is active (active or hold call status), update the index value for the new incoming call based on the current active index, to accommodate the new call in the valid index */
            if ((arCallStatusInfo[BTPHN_C_FIRST_CALL_INDEX].enCallStatus == BTPHN_EN_ACTIVE) ||
                  (arCallStatusInfo[BTPHN_C_FIRST_CALL_INDEX].enCallStatus == BTPHN_EN_ON_HOLD))
            {
               m_index = BTPHN_C_SECOND_CALL_INDEX;
            }
            else
            {
               m_index = BTPHN_C_FIRST_CALL_INDEX;
            }
         }
         arCallStatusInfo[m_index].u16CallInstance = u8IncomingCallInstance;
         ETG_TRACE_USR4(("onInComingCallEventStatus : arCallStatusInfo[%d].u16CallInstance = %d", m_index, arCallStatusInfo[m_index].u16CallInstance));
         /*contact name updated according to logic
         - Display contact name and number if it is in phone book
         - Display number only otherwise
         - Display "unknown" if no number sent by the counterpart*/
         if (strContactName.empty() && status->getSTelephoneNumber().empty())
         {
            arCallStatusInfo[m_index].coarCallerId = copcUnknownText.c_str();
            strTelephoneNumber = copcUnknownText.c_str();
            ETG_TRACE_USR3(("Inside Empty arCallStatusInfo[m_index].coarCallerId = %s",
                            arCallStatusInfo[m_index].coarCallerId.c_str()));
         }

         else if (strContactName.compare(copcUnknownText) == BTPHN_C_EQUAL)
         {
            arCallStatusInfo[m_index].coarCallerId = status->getSTelephoneNumber().c_str();
            ETG_TRACE_USR3(("Inside unknown :arCallStatusInfo[m_index].coarCallerId = %s",
                            arCallStatusInfo[m_index].coarCallerId.c_str()));
            strTelephoneNumber = status->getSTelephoneNumber().c_str();
            arCallStatusInfo[m_index].bIsCallerIdUpdated = TRUE;
         }
         else if (!strContactName.empty())
         {
            arCallStatusInfo[m_index].coarCallerId = strContactName.c_str();
            strTelephoneNumber = status->getSTelephoneNumber().c_str();
            ETG_TRACE_USR3(("inside valid : arCallStatusInfo[m_index].coarCallerId = %s",
                            arCallStatusInfo[m_index].coarCallerId.c_str()));
         }

         ETG_TRACE_USR3(("Caller Id = %s", arCallStatusInfo[m_index].coarCallerId.c_str()));
         ETG_TRACE_USR3(("Telephone number = %s", strTelephoneNumber.c_str()));
         arCallStatusInfo[m_index].enPhoneType = BTPHN_EN_GENERIC;

         strContactName = arCallStatusInfo[m_index].coarCallerId.c_str();
         m_index++;
      }

      /* Post the update to Model and Controller only when Android auto session and Carplay is not active */
      if ((g_SM_SpiSessionStatus != SPI_SESSION_ANDROID_AUTO_ACTIVE) && (g_SM_SpiSessionStatus != SPI_SESSION_CARPLAY_ACTIVE))
      {
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
         //MIC LOGIC
         if (MicClientHandler::poGetInstance()->getMicStatus() == false) //MIC Check for Incoming Call
         {
            ETG_TRACE_USR4(("ServiceTel : onInComingCallEventStatus MIC is not Active"));
#endif
            /* Creating IncomingCallResp courier message */
            IncomingCallResp* poIncomingCallResp =
               COURIER_MESSAGE_NEW(IncomingCallResp)
               ((Candera::String)strContactName.c_str(),
                (Candera::String)strTelephoneNumber.c_str(),
                (btphn_tenPhoneIncomingCallStatus)enCallStatus,
                u8DeviceType);

            rfoPhoneCallModel.OnUpdIncomingCallStatus(strContactName.c_str(), strTelephoneNumber.c_str(), enCallStatus, strDeviceName.c_str());

            if (NULL != poIncomingCallResp)
            {
               ETG_TRACE_USR3(("ServiceTel : Sending poIncomingCallResp' message"));
               /* Posting Courier message with IncomingCallEventStatus to HMI */
               poIncomingCallResp->Post();
            }
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
         }
         else
         {
            ETG_TRACE_USR4(("ServiceTel : onInComingCallEventStatus MIC is Active Rejecting Incoming Call"));
            vIgnoreCallStart(ServiceTel::poGetInstance()->getIncomingCallInstance());
            //POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(MICPhoneCallResp)()));
            if (MicClientHandler::poGetInstance()->getEnMicStatus() == enMIC_IntercomCall)
            {
               ETG_TRACE_USR4(("ServiceTel::onInComingCallEventStatus Intercom call is missed before Bt call"));
               MicClientHandler::poGetInstance()->vSetMicStatus(enMIC_Intercom_BTCall);
            }
            else
            {
               ETG_TRACE_USR4(("ServiceTel::onInComingCallEventStatus BT call is missed"));
               MicClientHandler::poGetInstance()->vSetMicStatus(enMIC_BTCall);
            }
         }
#endif
      }
   }
}


/************************************************************************
*FUNCTION: 		vStopDialCallTimer
*DESCRIPTION: 	Stops DialCall timer execution
*PARAMETER:		tVoid
*RETURNVALUE: 	tVoid
*
*HISTORY:
*01.11.2018 ess7kor
*Rev 0.1 Initial Revision
************************************************************************/

tVoid ServiceTel::vStopDialCallTimer()
{
   ETG_TRACE_USR4(("ServiceTel : vStopDialCallTimer()"));

   /* The dial call timer is reset if it is running */
   if (TRUE == bIsDialCallTimerRunning)
   {
      ETG_TRACE_USR4(("Dial call timer is stopped"));
      bIsDialCallTimerRunning = FALSE;
      if (OSAL_NULL != hDialCallTimer)
      {
         OSAL_s32TimerSetTime(hDialCallTimer,
                              BTPHN_C_DEFAULT_VALUE_ZERO,
                              BTPHN_C_DEFAULT_VALUE_ZERO);
      }
   }
}


/************************************************************************
*FUNCTION: 		vStartDialCallTimer
*DESCRIPTION: 	Stops DialCall timer execution
*PARAMETER:		tVoid
*RETURNVALUE: 	tVoid
*
*HISTORY:
*01.11.2018 ess7kor
*Rev 0.1 Initial Revision
************************************************************************/

tVoid ServiceTel::vStartDialCallTimer()
{
   ETG_TRACE_USR4(("ServiceTel : vStartDialCallTimer()"));

   /* The dial call timer is reset if it is running */
   if (FALSE == bIsDialCallTimerRunning)
   {
      ETG_TRACE_USR4(("Dial call timer is stopped"));
      bIsDialCallTimerRunning = TRUE;
      if (OSAL_NULL != hDialCallTimer)
      {
         OSAL_s32TimerSetTime(
            hDialCallTimer, // Timer handle
            BTPHN_C_DIAL_CALL_TIMER_DURATION, // Timer expiry duration
            BTPHN_C_DEFAULT_VALUE_ZERO // Timer repeat value
         );
      }
   }
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void ServiceTel::onNetworkOperatorError(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::NetworkOperatorError >& error)
{
   //Add your code here
}


/************************************************************************
*FUNCTION:     onNetworkOperatorStatus
*DESCRIPTION:  Receives the updated value of the property
*              Network operator name
*PARAMETER:
*              status:  pointer to MOST_Tel_FI::NetworkOperatorStatus
*
*RETURNVALUE:	tVoid
* Shwetha K : initial revision
v0.1 Sai Prabha 454189: Network operator name is seen in phone tile even if the phone is not connected to Head Unit
************************************************************************/
void ServiceTel::onNetworkOperatorStatus(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::NetworkOperatorStatus >& status)
{
   if (NULL != _most_tel_fi)
   {
      ETG_TRACE_USR2(("ServiceTel::onNetworkOperatorStatus : %s", status->getSNetworkOperator().c_str()));
      //LongDataSync for Network Provider
      VehicleSrvClient::poGetInstance()->vUpdatePhoneInfo(LONGDATATYPE_PHONENETWORK, status->getSNetworkOperator().c_str());
   }
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void ServiceTel::onSLCStatusError(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::SLCStatusError >& error)
{
   //Add your code here
}


/************************************************************************
*FUNCTION:     onSLCStatusStatus
*DESCRIPTION:  Receives the updated value of the property
*              Serial link lost status
*PARAMETER:    status:  pointer to MOST_Tel_FI::NetworkOperatorStatus
*
*RETURNVALUE:	tVoid
* Shwetha K : initial revision
************************************************************************/
void ServiceTel::onSLCStatusStatus(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::SLCStatusStatus >& status)
{
   if (ServiceBT::poGetInstance()->u8GetConnectedDeviceHandle() == 0)
   {
      return;
   }
   tBool bCommandStatus = status->getBCommandStatus();
   ETG_TRACE_USR2(("SLCStatusStatus: bCommandStatus : %d", ETG_CENUM(enBool, (enBool)status->getBCommandStatus())));

   rfoDeviceListModel.OnUpdSLCStatus(bCommandStatus);
   /*set Device type connected to true if connected device is present*/
   if ((BTPHN_C_PAIRED_DEVICE_HANDLE_START_INDEX <= ServiceBT::poGetInstance()->u8GetConnectedDeviceHandle())/* &&(!bAAPConnectedStatus)*/)
   {
      POST_MSG((COURIER_MESSAGE_NEW(SLCStatusResp)(bCommandStatus)));
      DeviceTypeConnected = bCommandStatus;
   }
}


void ServiceTel::onActivePhoneDeviceError(const ::boost::shared_ptr< MOST_Tel_FIProxy >& proxy, const ::boost::shared_ptr< ActivePhoneDeviceError >& error)
{
   ETG_TRACE_USR2(("ServiceTel::onActivePhoneDeviceError "));
}


/**************************************************************************
* This funtion is a Status update on change of an Active Phone
* @param[in] - ActivePhoneDeviceStatus
*
* @param[out]- void
initial rev: Sai Prabha
* Req Ids : SYSFL-475838,475839
**************************************************************************/
void ServiceTel::onActivePhoneDeviceStatus(const ::boost::shared_ptr< MOST_Tel_FIProxy >& proxy, const ::boost::shared_ptr< ActivePhoneDeviceStatus >& status)
{
   ETG_TRACE_USR2(("ServiceTel::onActivePhoneDeviceStatus : active device handle %d", status->getU8DeviceHandle()));
   m_u8ActiveDev = status->getU8DeviceHandle();

   if (TRUE == bBTVRViewShown)
   {
      ETG_TRACE_USR4(("ServiceTel :: onSetActivePhoneResult : Disable the VR"));
      vDisableBTVRView();
   }
   //NetworkOperatorStatus NWOperatorStatus = _most_tel_fi->getNetworkOperator();
   // std::string ProviderInfo;
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void ServiceTel::onVehicleAudioError(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::VehicleAudioError >& error)
{
   //Add your code here
   ETG_TRACE_ERR(("ServiceTel::onVehicleAudioError"));
   if (error->hasSystemErrorCode() == true)
   {
      ETG_TRACE_USR4(("onVehicleAudioError:SYSTEM Error code"));
      VehicleSrvClient::poGetInstance()->updateCallStatusError(PHONEMW_SNA);//Gen Can Imp for roaming error code
   }
   else if (error->hasCcaErrorCode() == true)
   {
      ETG_TRACE_USR4(("onVehicleAudioError:CCA Error code"));
      VehicleSrvClient::poGetInstance()->updateCallStatusError(CAN_ERROR);//Gen Can Imp for roaming error code
   }
   else
   {
      ETG_TRACE_USR4(("onVehicleAudioError:Unknown Error"));
      VehicleSrvClient::poGetInstance()->updateCallStatusError(PHONEMW_SNA);//Gen Can Imp for roaming error code
   }
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void ServiceTel::onVehicleAudioStatus(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::VehicleAudioStatus >& status)
{
   ETG_TRACE_USR4(("ServiceTel :: onVehicleAudioStatus"));
   //   ETG_TRACE_USR2(("VehicleAudioStatus: isUsingVehicleAudio : %d", ETG_CENUM(bool, (bool)status->getBUsingVehicleAudio())));
   rfoPhoneCallModel.OnUpdVehicleAudioSource(status->getBUsingVehicleAudio());
   if (VehicleSrvClient::poGetInstance() != NULL)
   {
      int BTPHN_EN_PRIVATE_MODE = 4;
      ETG_TRACE_USR4(("ServiceTel :: onVehicleAudioStatus Transfer of Audio is: %d", status->getBUsingVehicleAudio()));
      if (status->getBUsingVehicleAudio() == true)
      {
         VehicleSrvClient::poGetInstance()->updatePrivateDialFailMode(BTPHN_EN_PRIVATE_MODE); //CAN Imp
      }
   }
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void ServiceTel::onMicrophoneMuteStateError(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::MicrophoneMuteStateError >& error)
{
   ETG_TRACE_ERR(("ServiceTel::onMicrophoneMuteStateError "));
}


/************************************************************************
*FUNCTION: 		onMicrophoneMuteStateStatus
*DESCRIPTION: 	Read Status update and Sends the microphone mute status to HMI Model
*PARAMETER:
*					status:   (I)
*						1 - Enabled
*						0 - Disabled
*
*RETURNVALUE: 	tVoid
*
*HISTORY:
*01.16.2018 ess7kor
*Rev 0.1 Initial Revision.
************************************************************************/
void ServiceTel::onMicrophoneMuteStateStatus(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::MicrophoneMuteStateStatus >& status)
{
   ETG_TRACE_USR3(("ServiceTel::onMicrophoneMuteStateStatus status is %d", status->getBEnable()));

   // Storing the microphone mute status
   bMicrophoneState = status->getBEnable();

   //send Microphone Status to GUI
   MicMuteStateResp* poMicMuteStateResp = COURIER_MESSAGE_NEW(MicMuteStateResp)(bMicrophoneState);
   if (NULL != poMicMuteStateResp)
   {
      ETG_TRACE_USR3(("ServiceTel : Sending 'MicMuteStateResp' message"));

      /* Posting Courier message with microphone mute status to HMI Model */
      tBool bRetVal = poMicMuteStateResp->Post();
      if (FALSE == bRetVal)
      {
         ETG_TRACE_ERR(("ServiceTel: Sending 'MicMuteStateResp' message failed"));
      }
   }
   rfoPhoneCallModel.OnUpdMicMuteStatus(bMicrophoneState);
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void ServiceTel::onRingtoneMuteStateError(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::RingtoneMuteStateError >& error)
{
   //Add your code here
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void ServiceTel::onRingtoneMuteStateStatus(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::RingtoneMuteStateStatus >& status)
{
   //Add your code here
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void ServiceTel::onHFServiceError(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::HFServiceError >& error)
{
   //Add your code here
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void ServiceTel::onHFServiceStatus(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::HFServiceStatus >& status)
{
   //Add your code here
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void ServiceTel::onBTDeviceVoiceRecognitionExtendedError(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::BTDeviceVoiceRecognitionExtendedError >& error)
{
   ETG_TRACE_USR4(("ServiceTel::onBTDeviceVoiceRecognitionExtendedError called "));
   if (bIsAcceptCallRequestedWhenVREnds == TRUE)
   {
      bIsAcceptCallRequestedWhenVREnds = false;
   }
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void ServiceTel::onBTDeviceVoiceRecognitionExtendedStatus(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::BTDeviceVoiceRecognitionExtendedStatus >& status)
{
   //SIRI Feature availability handling - starts here
   uint8 tempDevicehandle = status->getU8DeviceHandle();
   most_Tel_fi_types_Extended::T_e8_TelSiriAvailabilityState temp_SiriAvailabilityState = status->getE8SiriAvailabilityState();

   ETG_TRACE_USR4((" ServiceTel::onBTDeviceVoiceRecognitionExtendedStatus DeviceHandle (%d), SiriStatus (%d)", tempDevicehandle, temp_SiriAvailabilityState));

   //Take a copy of the current Siri status as to check against new Siri status
   most_Tel_fi_types_Extended::T_e8_TelSiriAvailabilityState previousDeviceOneSiriState = BtvrExtendedStatus[BTPHN_C_HFP_INDEX_ZERO].enSiriAvailabilityState;
   most_Tel_fi_types_Extended::T_e8_TelSiriAvailabilityState previousDeviceTwoSiriState = BtvrExtendedStatus[BTPHN_C_HFP_INDEX_ONE].enSiriAvailabilityState;

   if (tempDevicehandle != 0)
   {
      //If the device handle is already being updated, just update the new Siri status for HFP device 1
      if (BtvrExtendedStatus[BTPHN_C_HFP_INDEX_ZERO].u8DeviceHandle == tempDevicehandle)
      {
         BtvrExtendedStatus[BTPHN_C_HFP_INDEX_ZERO].enSiriAvailabilityState = temp_SiriAvailabilityState;
         ETG_TRACE_USR4((" ServiceTel::onBTDeviceVoiceRecognitionExtendedStatus SiriState update for DeviceOne"));
      }
      //If the device handle is already being updated, just update the new Siri status for HFP device 2
      else if (BtvrExtendedStatus[BTPHN_C_HFP_INDEX_ONE].u8DeviceHandle == tempDevicehandle)
      {
         BtvrExtendedStatus[BTPHN_C_HFP_INDEX_ONE].enSiriAvailabilityState = temp_SiriAvailabilityState;
         ETG_TRACE_USR4((" ServiceTel::onBTDeviceVoiceRecognitionExtendedStatus SiriState update for DeviceTwo"));
      }
      else
      {
         //If the device handle is new, then update both the device handle and Siri status for HFP device 1
         if (BtvrExtendedStatus[BTPHN_C_HFP_INDEX_ZERO].enSiriAvailabilityState != most_Tel_fi_types_Extended::T_e8_TelSiriAvailabilityState__e8AVAILABLE_DISABLED)
         {
            BtvrExtendedStatus[BTPHN_C_HFP_INDEX_ZERO].u8DeviceHandle = tempDevicehandle;
            BtvrExtendedStatus[BTPHN_C_HFP_INDEX_ZERO].enSiriAvailabilityState = temp_SiriAvailabilityState;
            ETG_TRACE_USR4((" ServiceTel::onBTDeviceVoiceRecognitionExtendedStatus SiriState and DeviceHandle update for DeviceOne"));
         }
         //If the device handle is new, then update both the device handle and Siri status for HFP device 2
         else if (BtvrExtendedStatus[BTPHN_C_HFP_INDEX_ONE].enSiriAvailabilityState != most_Tel_fi_types_Extended::T_e8_TelSiriAvailabilityState__e8AVAILABLE_DISABLED)
         {
            BtvrExtendedStatus[BTPHN_C_HFP_INDEX_ONE].u8DeviceHandle = tempDevicehandle;
            BtvrExtendedStatus[BTPHN_C_HFP_INDEX_ONE].enSiriAvailabilityState = temp_SiriAvailabilityState;
            ETG_TRACE_USR4((" ServiceTel::onBTDeviceVoiceRecognitionExtendedStatus SiriState and DeviceHandle update for DeviceTwo"));
         }
         else
         {
         }
      }

      //Show the EyeFree popup only when there is a Siri Status change to DISABLED in any one of the HFP device
      if (((previousDeviceOneSiriState != BtvrExtendedStatus[BTPHN_C_HFP_INDEX_ZERO].enSiriAvailabilityState) && (BtvrExtendedStatus[BTPHN_C_HFP_INDEX_ZERO].enSiriAvailabilityState == most_Tel_fi_types_Extended::T_e8_TelSiriAvailabilityState__e8AVAILABLE_DISABLED)) ||
            ((previousDeviceTwoSiriState != BtvrExtendedStatus[BTPHN_C_HFP_INDEX_ONE].enSiriAvailabilityState) && (BtvrExtendedStatus[BTPHN_C_HFP_INDEX_ONE].enSiriAvailabilityState == most_Tel_fi_types_Extended::T_e8_TelSiriAvailabilityState__e8AVAILABLE_DISABLED)))
      {
         ETG_TRACE_USR4((" ServiceTel::onBTDeviceVoiceRecognitionExtendedStatus: Change in SiriState"));
         if ((BtvrExtendedStatus[BTPHN_C_HFP_INDEX_ZERO].enSiriAvailabilityState == most_Tel_fi_types_Extended::T_e8_TelSiriAvailabilityState__e8AVAILABLE_DISABLED) ||
               (BtvrExtendedStatus[BTPHN_C_HFP_INDEX_ONE].enSiriAvailabilityState == most_Tel_fi_types_Extended::T_e8_TelSiriAvailabilityState__e8AVAILABLE_DISABLED))
         {
            POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_PhoneModule#Sys_Popup#Pfo_SIRI_EyeFree"))));
            ETG_TRACE_USR4((" ServiceTel::onBTDeviceVoiceRecognitionExtendedStatus: Showing of SiriEyeFree Popup"));
         }
      }
   }
   //SIRI Feature availability handling - ends here

   //BTVR Feature handling - starts here
   m_bBTVoiceActive = status->getBBTDeviceVoiceRecActive();
   m_bBTRecSupported = status->getBBTDeviceVoiceRecSupported();
   ETG_TRACE_USR4((" ServiceTel::onBTDeviceVoiceRecognitionExtendedStatus active (%d), supported (%d)", m_bBTVoiceActive, m_bBTRecSupported));
   tBool bVRIconToGUI = FALSE;
   //update the SR icon status
   if (m_bBTVoiceActive == TRUE)
   {
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
      //MIC LOGIC
      if (MicClientHandler::poGetInstance()->getMicStatus() == false) //MIC Check for BT VR
      {
         ETG_TRACE_USR4(("ServiceTel : onBTDeviceVoiceRecognitionStatus MIC is not Active proceed with BTVR Request"));
#endif
         if (FALSE == bBTVRViewShown)
         {
            /*to do :check if we have multipple ways of activating bt siri*/
            //m_u8SIRISessionDeviceState = 1; // Set the variable to 1 which indicates the session & popup was activated via Device
            vEnableBTVRView();
         }
         bVRIconToGUI = TRUE;
         //to do : handle this according to interface from master
         m_u8BTVRSessionActive = 2;// Update SIRI Status to session active
         ETG_TRACE_USR3(("onBTDeviceVoiceRecognitionExtendedStatus - Popup Is VISIBLE, SESSION is ACTIVE "));
         //poSmartPhoneIntegrationApp->covSendGlobalEvt(eAppGlobalEvt_SIRI_STATUS, m_u8BTVRSessionActive);
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
      }
      else
      {
         ETG_TRACE_USR4(("ServiceTel : onBTDeviceVoiceRecognitionStatus MIC is Active Cancelling BTVR Request"));
         vDisableBTVRView();
      }
#endif
   }
   else
   {
      m_u8BTVRSessionActive = 0;// Update SIRI Status to pop up not visible
      if (TRUE == bBTVRViewShown)
      {
         ETG_TRACE_USR3(("onBTDeviceVoiceRecognitionExtendedStatus - Popup Is VISIBLE, SESSION is NOT ACTIVE"));

         //	poSmartPhoneIntegrationApp->covSendGlobalEvt(eAppGlobalEvt_SIRI_STATUS, m_u8BTVRSessionActive);
         //if (m_u8SIRISessionDeviceState == 1)
         //{
         //	vDisableViewCourierMsg();// Exit the SIRI Popup which was displayed via Phone
         //	m_u8BTVRSessionActive = 0;// Update SIRI Status to App manager
         //	ETG_TRACE_USR3(("tclSmartPhoneIntegrationSiriAppLogic - Popup Is NOT VISIBLE, SESSION is NOT ACTIVE and Application Manager has Been Informed"));
         //	poSmartPhoneIntegrationApp->covSendGlobalEvt(eAppGlobalEvt_SIRI_STATUS, m_u8BTVRSessionActive);
         //}
         vDisableBTVRView();
      }
      else  if (bIsAcceptCallRequestedWhenVREnds == TRUE)
      {
         ETG_TRACE_USR3(("onBTDeviceVoiceRecognitionExtendedStatus - bIsAcceptCallRequestedWhenVREnds == true"));
         vAcceptCallStart(u8IncomingCallInstance);
         bIsAcceptCallRequestedWhenVREnds = false;
      }
      else
      {
         ETG_TRACE_USR3(("tclSmartPhoneIntegrationSiriAppLogic - Popup Is NOT VISIBLE, SESSION is NOT ACTIVE"));
         //poSmartPhoneIntegrationApp->covSendGlobalEvt(eAppGlobalEvt_SIRI_STATUS, m_u8BTVRSessionActive);
      }
   }

   rfoPhoneCallModel.OnUpdSRIconStatus(bVRIconToGUI);
   //BTVR Feature handling - ends here
}


tVoid ServiceTel::vSet_BTVoiceRecognition(tBool bBTActiveReq)
{
   ETG_TRACE_USR3(("ServiceTel::vSet_BTVoiceRecognition bBTActiveReq= %d ", bBTActiveReq));
   if ((_most_tel_fi) && (m_u8ActiveDev != 0))
   {
      _most_tel_fi->sendBTDeviceVoiceRecognitionExtendedPureSet(m_u8ActiveDev, bBTActiveReq);
   }
}


tVoid ServiceTel::vEnableBTVRView()
{
   ETG_TRACE_USR3(("ServiceTel::vEnableBTVRView() - show BTVR scene "));
   //POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_PhoneModule#Sys_Popup#SR_SIRI_Main"))));
   bBTVRViewShown = TRUE;

   ETG_TRACE_USR3(("ServiceTel:vEnableBTVRView Enable "));
   //SWITCH_TO_BTVR - Value of 5
   POST_MSG((COURIER_MESSAGE_NEW(TriggerSceneTransitionMsg)(5, true)));
}


tVoid ServiceTel::vDisableBTVRView()
{
   if (!_isCabinOnly)
   {
      ETG_TRACE_USR3(("ServiceTel::vDisableBTVRView() - hide BTVR scene "));
      //POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Hide, Courier::ViewId("AppHmi_PhoneModule#Sys_Popup#SR_SIRI_Main"))));
      if (bBTVRViewShown == TRUE)
      {
         bBTVRViewShown = FALSE;

         ETG_TRACE_USR3(("ServiceTel:vDisableBTVRView Disable "));
         //SWITCH_TO_BTVR - Value of 5
         POST_MSG((COURIER_MESSAGE_NEW(TriggerSceneTransitionMsg)(5, false)));
      }
   }
}


/************************************************************************
*FUNCTION: 		vSwitchToBTVRView
*DESCRIPTION: 	update BT vr status
*PARAMETER:		void
*
*
*RETURNVALUE: 	tVoid
*
*HISTORY:
*14.05.2018 Shwetha K
*Rev 1.0 Initial Revision.

************************************************************************/
tVoid ServiceTel::vSwitchToBTVRView()
{
   ETG_TRACE_USR4(("ServiceTel::vSwitchToBTVRView"));
   if (FALSE == bBTVRViewShown)
   {
      // Check whether the device is supported
      if (m_bBTRecSupported == FALSE)
      {
         // EnablePopup Msg creation failure
         ETG_TRACE_ERR(("ServiceTel::vSwitchToBTVRView() Enable View Failed Since BTVR is not Supported"));
      }
      else
      {
         // On call check whether status is idle or not, if yes, disable the popup
         if (m_nActiveCall != BTPHN_EN_IDLE)
         {
            vDisableBTVRView();
         }
         else
         {
            vEnableBTVRView();
            // Update SIRI Status to App manager
            m_u8BTVRSessionActive = 1;
            ETG_TRACE_USR3(("ServiceTel - POPUP is VISIBLE, SESSION Is Not ACTIVE and Application Manager has Been Informed"));
            ETG_TRACE_USR4(("ServiceTel:: Value Of BT_VR State: (%d)", m_u8BTVRSessionActive));
            //poSmartPhoneIntegrationApp->covSendGlobalEvt(eAppGlobalEvt_SIRI_STATUS, m_u8BTVRSessionActive);
         }
      }
   }
   else
   {
      ETG_TRACE_USR3(("ServiceTel - LONG Press Of PTT hence Quiting bt vr Session + Popup"));
      // Stop Session & Exit btvr on long press of PTT key
      vDisableBTVRPopUpOnHomeButton();
   }
}


/************************************************************************
*FUNCTION:		vStartBTVRSessionViaPTT
*DESCRIPTION:	Start BT VR session via PTT
*
*HISTORY:
*15.05.18 Shwetha K
*Rev 1.0 Initial Revision.
************************************************************************/
tVoid ServiceTel::vStartBTVRSessionViaPTT()
{
   ETG_TRACE_USR3(("ServiceTel::vStartBTVRSessionViaPTT"));

   if (FALSE == bBTVRViewShown)
   {
      ETG_TRACE_USR3(("ServiceTel - BT_VR Popup Is Not Shown Hence No Action Is Taken"));
   }
   else
   {
      ETG_TRACE_USR3(("ServiceTel - BT_VR Popup Is Shown Thus Session Will Start Now"));
      if (PhoneDataPoolConfig::getInstance()->isVRSupported())
      {
         vSet_BTVoiceRecognition(TRUE);
      }
   }
}


/************************************************************************
*FUNCTION:		vDisableSIRIPopUpOnHomeButton
*DESCRIPTION:	Disables SIRI popup
*
*HISTORY:
*15.05.18 Shwetha K
*Rev 1.0 Initial Revision.
************************************************************************/
tVoid ServiceTel::vDisableBTVRPopUpOnHomeButton()
{
   ETG_TRACE_USR3(("ServiceTel - LONG Press Of PTT hence Quiting BT VR Session + Popup"));
   if (m_bBTVoiceActive == TRUE)
   {
      vSet_BTVoiceRecognition(0);
   }
   if (TRUE == bBTVRViewShown)
   {
      // Exit the popup
      vDisableBTVRView();
   }
   // Update SIRI Status to App manager
   m_u8BTVRSessionActive = 0;
   //poSmartPhoneIntegrationApp->covSendGlobalEvt(eAppGlobalEvt_SIRI_STATUS, m_u8BTVRSessionActive);
}


tVoid ServiceTel::vSendCallStatusToBTVR(const std::vector<btphn_trCallStatus>& arCallList)
{
   ETG_TRACE_USR3(("ServiceTel : vSendCallStatusToBTVR"));
   tBool IsAnyOngoingCall = false;
   for (tU8 Index = 0 ; Index < arCallList.size(); Index++)
   {
      ETG_TRACE_USR4(("ServiceTel : vSendCallStatusToBTVR: CAll STATE = (%d)", arCallList[Index].enCallStatus));

      if (arCallList[Index].enCallStatus != BTPHN_EN_IDLE)
      {
         IsAnyOngoingCall = true;
         m_nActiveCall = arCallList[Index].enCallStatus;
      }
   }

   if (!IsAnyOngoingCall == FALSE)
   {
      m_nActiveCall = BTPHN_EN_IDLE;
   }

   ETG_TRACE_USR4(("ServiceTel::vSendCallStatusToBTVR: POPUP IS SHOWN = (%d)", bBTVRViewShown));

   // Check whether call is going on and whether SIRI popup is shown
   if ((IsAnyOngoingCall == true) && (bBTVRViewShown == TRUE))
   {
      vDisableBTVRView();
      // Update SIRI Status to App manager
      //m_u8BTVRSessionActive = 0;

      //if (tclSmartPhoneIntegrationApp::poGetSPIAppInstance() != NULL)
      //{
      //	ETG_TRACE_USR3(("tclSmartPhoneIntegrationSiriAppLogic - Popup Is Not VISIBLE and Application Manager has Been Informed"));
      //	ETG_TRACE_USR4(("tclSmartPhoneIntegrationSiriAppLogic:: Value Of SIRI State: (%d)", m_u8SIRISessionActive));
      //	tclSmartPhoneIntegrationApp::poGetSPIAppInstance()->covSendGlobalEvt(eAppGlobalEvt_SIRI_STATUS, m_u8SIRISessionActive);
      //}
   }
}


/***************************************************************************
* Function name : vSetRingtone
* parma[in]- tU8 u8RingToneID
* param[out]- tVoid
* v 0.1 initial rev :
****************************************************************************/
#ifdef VARIANT_S_FTR_ENABLE_PHONECALLMANAGER_SUPPORT
tVoid ServiceTel::vSetRingtone(tU8 u8RingToneID)
{
   ETG_TRACE_USR4(("ServiceTel::vSetRingtone"));
   /* Set the desired ringtoneID for ALL devices */
   if (_most_tel_fi && _most_tel_fi.get())
   {
      _most_tel_fi->sendSetRingtoneStart(*this, "ALL", u8RingToneID);
      ETG_TRACE_USR4(("ServiceTel::sendSetRingtoneStart"));
   }
   else
   {
      ETG_TRACE_ERR(("_most_tel_fi not available"));
   }
}


#endif

/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void ServiceTel::onHandsFreeEnhancedCallControlError(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::HandsFreeEnhancedCallControlError >& error)
{
   //Add your code here
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void ServiceTel::onHandsFreeEnhancedCallControlStatus(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Tel_FI::HandsFreeEnhancedCallControlStatus >& status)
{
   //Add your code here
}


void ServiceTel::onDialError(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const ::boost::shared_ptr< ::MOST_Tel_FI::DialError >& error)
{
   ETG_TRACE_USR4(("ServiceTel : onDialError()"));
   int BTPHN_EN_DIALING_FAILED = 5;
   if (error->hasSystemErrorCode() == true)
   {
      ETG_TRACE_USR4(("onDialError:SYSTEM Error code"));
      VehicleSrvClient::poGetInstance()->updatePrivateDialFailMode(BTPHN_EN_DIALING_FAILED);//Gen Can Imp for roaming error code
   }
   else if (error->hasCcaErrorCode() == true)
   {
      ETG_TRACE_USR4(("onDialError:CCA Error code"));
      VehicleSrvClient::poGetInstance()->updatePrivateDialFailMode(BTPHN_EN_DIALING_FAILED);//Gen Can Imp for roaming error code
   }
   else
   {
      ETG_TRACE_USR4(("onDialError:Unknown Error"));
      VehicleSrvClient::poGetInstance()->updatePrivateDialFailMode(BTPHN_EN_DIALING_FAILED);//Gen Can Imp for roaming error code
   }
}


void ServiceTel::onDialResult(const ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy >& proxy, const ::boost::shared_ptr< ::MOST_Tel_FI::DialResult >& result)
{
   ETG_TRACE_USR4(("ServiceTel : onDialResult()"));
   vResetCallFlag();
   vStartDialCallTimer();
}


/************************************************************************
*FUNCTION:     vSendDialCallMethodStart
*DESCRIPTION:  Function called to Phone Service to
*              Dial to the requested number
*PARAMETER:
*              strTelephoneNo:   (->I)
*						Telephone number to be dialed
*					enECNRStatus:   (I)
*						ECNR status
*
*RETURNVALUE:	tVoid
*
*HISTORY:
*01.16.2018	ess7kor
*Rev 0.1 Initial Revision
************************************************************************/
tVoid ServiceTel::vSendDialCallMethodStart(std::string strTelephoneNo, btphn_tenECNRSettings enECNRStatus)
{
   ETG_TRACE_ERR(("vSendDialCallMethodStart:: Phone no is %s", strTelephoneNo.c_str()));
   if (_most_tel_fi)
   {
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
      //MIC LOGIC
      if (MicClientHandler::poGetInstance()->getMicStatus() == false) //MIC Check for Outgoing Calls from Contact/Dialpad/Matchlist
      {
         ETG_TRACE_USR4(("ServiceTel : vSendDialCallMethodStart MIC is not Active"));
#endif
         ::fi_basetypes_most::T_String telephoneno = strTelephoneNo.c_str();
         bCallReqFlag = TRUE;
         ETG_TRACE_USR4(("CallReqFlag : %d", bCallReqFlag));
         _most_tel_fi->sendDialStart(*this, telephoneno, static_cast<most_Tel_fi_types::T_e8_TelEchoCancellationNoiseReductionSetting>(enECNRStatus));

         // Set to true as call has been dial from connected phone
         bIsOutgoingCallPopupActive = true;

         for (tU8 u8Index = BTPHN_C_DEFAULT_VALUE_ZERO; u8Index < BTPHN_C_NUM_CALLS_SUPPORTED; u8Index++)
         {
            if (BTPHN_EN_IDLE == arCallStatusInfo[u8Index].enCallStatus)
            {
               arCallStatusInfo[u8Index].coarCallerId = strTelephoneNo.c_str();
               arCallStatusInfo[u8Index].bIsContactInfoReq = TRUE;
               arCallStatusInfo[u8Index].enCallStatus = BTPHN_EN_DIAL_START;
               u8NumActiveCalls++;
               u8CurrentCallInstance = u8Index;
               vProcessCallStatus();
               break;
            }
         }
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
      }
      else
      {
         ETG_TRACE_USR4(("ServiceTel : vSendDialCallMethodStart MIC is Active Cancelling Outgoing Call Request"));
         POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(CancelDialCallReq)()));
         POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(MICContactsCallHistoryResp)()));
      }
#endif
   }
   else
   {
      ETG_TRACE_ERR(("vSendDialCallMethodStart::_most_tel_fi is NULL"));
   }
}


/************************************************************************
*FUNCTION:     vAcceptCallStart
*DESCRIPTION:  Function called from Courier Message to request Phone Service to
*              Accept Call
*
*PARAMETER:
*              u16CallInstance:  (I)
*						Call Instance
*
*RETURNVALUE:	tVoid
*
*HISTORY:
*ess7or: Initial Version
************************************************************************/
tVoid ServiceTel::vAcceptCallStart(tU16 u16CallInstance)
{
   ETG_TRACE_USR4(("ServiceTel : vAcceptCallStart()\t"));
   ETG_TRACE_USR4(("CallReqFlag : %d", bCallReqFlag));
   ETG_TRACE_USR4(("Call instance : %d", u16CallInstance));
   ETG_TRACE_USR3(("m_u8BTVRSessionActive : %d",  m_u8BTVRSessionActive));

   if (m_u8BTVRSessionActive != 0)
   {
      ETG_TRACE_USR4(("ServiceTel : vAcceptCallStart() disable BT VR and then send accept call request"));
      bIsAcceptCallRequestedWhenVREnds = true;
      vSet_BTVoiceRecognition(FALSE);
   }
   else
   {
      ETG_TRACE_USR4(("ServiceTel : vAcceptCallStart() request"));
      if (_most_tel_fi && _most_tel_fi.get())
      {
         _most_tel_fi->sendAcceptCallStart(*this, u16CallInstance);
      }
      /* Sending AcceptCall request if Phone Service is avaialable */
      else
      {
         ETG_TRACE_ERR(("ServiceTel : vAcceptCallStart()\n" "Phone service is unavailable hence request not posted"));
         vResetCallFlag();
         vSendServiceError(BTPHN_EN_PHN_ACCEPTCALL, BTPHN_EN_ERRCODE_NOERROR);
      }
   }
}


/************************************************************************
*FUNCTION:     vIgnoreCallStart
*DESCRIPTION:  Function called from Courier Message to request Phone Service to
*              Ignore Call
*
*PARAMETER:
*              u16CallInstance:  (I)
*						Call Instance
*
*RETURNVALUE:	tVoid
*
*HISTORY:
*ess7or: Initial Version
************************************************************************/
tVoid ServiceTel::vIgnoreCallStart(tU16 u16CallInstance)
{
   ETG_TRACE_USR4(("ServiceTel : vIgnoreCallStart()\t"));
   ETG_TRACE_USR4(("CallReqFlag : %d", bCallReqFlag));
   ETG_TRACE_USR4(("Call instance : %d", u16CallInstance));

   if (_most_tel_fi && _most_tel_fi.get())
   {
      if (bCallReqFlag == FALSE)
      {
         ETG_TRACE_USR4(("ServiceTel : Sending IgnoreCall Method Start"));
         _most_tel_fi->sendIgnoreCallStart(*this, u16CallInstance);
         bCallReqFlag = TRUE;
      }
   }
   /* Sending IgnoreCall request if Phone Service is avaialable */
   else
   {
      ETG_TRACE_ERR(("ServiceTel : IgnoreCall()\n"
                     "Phone service is unavailable hence request not posted"));
      vResetCallFlag();
      vSendServiceError(BTPHN_EN_PHN_REJECTCALL, BTPHN_EN_ERRCODE_NOERROR);
   }
}


/************************************************************************
*FUNCTION: 		 vSendServiceError
*DESCRIPTION: 	 Sends courier message with Phone service error information to HMI
*PARAMETER:
*					corfrServiceType:	(I)
* 					    Phone service type
* 				    corfrErrCode:	(I)
* 					    Service error code
*
*RETURNVALUE: 	tVoid
*
*HISTORY: ess7kor
*Rev 0.1 Initial Revision.
************************************************************************/

tVoid ServiceTel::vSendServiceError(const btphn_tenPhnServiceType& corfrServiceType, const btphn_tenServiceErrCode& corfrErrCode)
{
   ETG_TRACE_USR4(("ServiceTel : vSendServiceError()\t"));
   ETG_TRACE_USR4(("Phone service type = %x\t Service error code : %d",
                   ETG_CENUM(btphn_tenPhnServiceType, (btphn_tenPhnServiceType)corfrServiceType),
                   ETG_CENUM(btphn_tenServiceErrCode, (btphn_tenServiceErrCode)corfrErrCode)));

   POST_MSG((COURIER_MESSAGE_NEW(PhnServiceErrResp)(corfrServiceType, corfrErrCode)));
}


/************************************************************************
*FUNCTION: 		 printIncomingCallEventStatus
*DESCRIPTION: 	 Prints incoming Call Status message for debugging
*PARAMETER:
*					status: InComingCallEventStatus object
*
*RETURNVALUE: 	tVoid
*
*HISTORY: ess7kor
*Rev 0.1 Initial Revision.
************************************************************************/
void ServiceTel::printIncomingCallEventStatus(const boost::shared_ptr< InComingCallEventStatus >& status)
{
   //ETG_TRACE_USR4(("Incoming Call : Status = %d", ETG_CENUM(btphn_tenPhoneIncomingCallStatus, (btphn_tenPhoneIncomingCallStatus)status->getE8IncomingCallEventEnum())));
   ETG_TRACE_USR4(("Incoming Call :Call Instance = %d", status->getU16CallInstance()));
   ETG_TRACE_USR4(("Incoming Call :Telephone Number = %s", status->getSTelephoneNumber().c_str()));
   ETG_TRACE_USR4(("Incoming Call :First name = %s", status->getSFirstName().c_str()));
   ETG_TRACE_USR4(("Incoming Call :Last name = %s", status->getSLastName().c_str()));
   ETG_TRACE_USR4(("Incoming Call :Device Handle = %d", status->getU8DeviceHandle()));
   //ETG_TRACE_USR4(("Incoming Call :isCallWaiting = %d", ETG_CENUM(enBool, (enBool)status->getBCallWait())));
}


/************************************************************************
*FUNCTION: 		vHandleDeviceHandleResp
*DESCRIPTION: 	henever there is a change in the connected device handle or
*					when there is a device disconnected, this message is received.
*PARAMETER:
* 					tU8ConenctedDeviceHanlde
*
*HISTORY:
*01.15.2018 ess7kor
*Rev 0.1 Initial Revision.
************************************************************************/
tVoid ServiceTel::vHandleDeviceHandleResp(tU8 u8ConnectedDeviceHandle)
{
   ETG_TRACE_USR4(("ServiceTel :: vHandleDeviceHandleResp : Device Handle = %d", u8ConnectedDeviceHandle));

   if (u8ConnectedDeviceHandle == 0)
   {
      //Dial Call Flag is reset
      vResetCallFlag();

      for (tU8 u8Index = BTPHN_C_DEFAULT_VALUE_ZERO;
            u8Index < BTPHN_C_NUM_CALLS_SUPPORTED;
            u8Index++)
      {
         /*On device disconnection, If DialCall timer is running, the same is
         terminated and call status corresponding to DIAL_START is reset to IDLE
         and updated to HMI*/
         if (BTPHN_EN_DIAL_START == arCallStatusInfo[u8Index].enCallStatus)
         {
            if (TRUE == bIsDialCallTimerRunning)
            {
               vStopDialCallTimer();
            }

            arCallStatusInfo[u8Index].u16CallInstance = u8Index;
            arCallStatusInfo[u8Index].enPhoneType = BTPHN_EN_GENERIC;
            vSendCallStatus(arCallStatusInfo[u8Index]);
            break;
         }
      }
   }
}


/************************************************************************
*FUNCTION: 		vSendDTMFStart
*DESCRIPTION: 	whenever speller key is pressed, request to service for DTMF start is sent for current call instance
*PARAMETER:		strKeyPressed : pressed key from speller layout

* Return : void
*
*HISTORY:
*14.02.2018 Shwetha K
*Rev 0.1 Initial Revision.
*Rev 0.2 25.07.2019 pka5cob; SYSFL1839
************************************************************************/
tVoid ServiceTel::vSendDTMFStart(std::string strKeyPressed)
{
   ETG_TRACE_USR4(("ServiceTel :: vSendDTMFStart :strKeyPressed = %s  ", strKeyPressed.c_str()));
   ETG_TRACE_USR4(("ServiceTel :: vSendDTMFStart :call instance = %d  ", u8CurrentActiveCallInstance_dtmf));
   _most_tel_fi->sendSendDTMFStart(*this, u8CurrentActiveCallInstance_dtmf, strKeyPressed);
}


/************************************************************************
*FUNCTION: 		vSetActivePhone
*DESCRIPTION: 	Function to Set Active Phone
*PARAMETER:		u8ActiveDev : device handle
strDeviceName : device name

* Return : void
*
*HISTORY:
*04.02.2019 Sai Prabha
*Rev 0.1 Initial Revision.
************************************************************************/
void ServiceTel:: vSetActivePhone(tU8 u8ActiveDev, std::string strDeviceName)
{
   if (!_isCabinOnly)
   {
      ETG_TRACE_USR4(("ServiceTel :: vSetActivePhone = %d  ", u8ActiveDev));
      _most_tel_fi->sendSetActivePhoneStart(*this, u8ActiveDev);

      m_strDeviceName = strDeviceName;
      ETG_TRACE_USR4(("vSetActivePhone::DeviceName :%s", m_strDeviceName));
   }
}


/************************************************************************
*FUNCTION: 		onSetActivePhoneError
*DESCRIPTION: 	Set Active Phone Error
*PARAMETER:		error : Error of set Active phone

* Return : void
*
*HISTORY:
*04.02.2019 Sai Prabha
*Rev 0.1 Initial Revision.
************************************************************************/
void ServiceTel:: onSetActivePhoneError(const ::boost::shared_ptr< MOST_Tel_FIProxy >& proxy, const ::boost::shared_ptr< SetActivePhoneError >& error)
{
   ETG_TRACE_USR4(("ServiceTel :: onSetActivePhoneError"));
}


/************************************************************************
*FUNCTION: 		onSetActivePhoneResult
*DESCRIPTION: 	Set Active Phone Result
*PARAMETER:		result :success of set active phone
* Return : void
*
*HISTORY:
*04.02.2019 Sai Prabha
*Rev 0.1 Initial Revision.
************************************************************************/
void ServiceTel:: onSetActivePhoneResult(const ::boost::shared_ptr< MOST_Tel_FIProxy >& proxy, const ::boost::shared_ptr< SetActivePhoneResult >& result)
{
   if (!_isCabinOnly)
   {
      if (TRUE == bBTVRViewShown)
      {
         ETG_TRACE_USR4(("ServiceTel :: onSetActivePhoneResult : Disable the VR"));
         vDisableBTVRView();
      }
      ETG_TRACE_USR4(("ServiceTel :: onSetActivePhoneResult"));
      POST_MSG((COURIER_MESSAGE_NEW(ResetListDataReq)(BTPHN_C_LIST_ID_CALL_LIST)));
      POST_MSG((COURIER_MESSAGE_NEW(ResetListDataReq)(BTPHN_C_LIST_ID_CONTACT_LIST)));
      ETG_TRACE_USR4(("ServiceTel::poGetInstance()->u8GetActiveDev():%d", ServiceTel::poGetInstance()->u8GetActiveDev()));

      ETG_TRACE_USR4(("onActivePhoneDeviceStatus::OnUpdConnectedDeviceName:%s", m_strDeviceName));
      rfoDeviceListModel.OnUpdConnectedDeviceName(m_strDeviceName.c_str());
      if (ServicePhoneBook::poGetInstance()->bCurrentActiveTab == BTPHN_EN_CALLHISTORY_LIST)
      {
         POST_MSG((COURIER_MESSAGE_NEW(CallListReq)()));
      }
      else
      {
         POST_MSG((COURIER_MESSAGE_NEW(ContactListReq)()));
      }
      rfoCallHistoryModel.SetBatteryIndicator();

      if (proxy == _most_tel_fi && _most_tel_fi.get())
      {
         ETG_TRACE_USR4(("ServiceTel::onSetActivePhoneResult Request for Connected Phone Details"));
         _most_tel_fi->sendCellSignalStrengthGet(*this);
         _most_tel_fi->sendCellDeviceNameGet(*this);
         _most_tel_fi->sendCellStateofChargeGet(*this);
         _most_tel_fi->sendCellNetworkStatusGet(*this);
      }
      else
      {
         ETG_TRACE_USR4(("ServiceTel::onSetActivePhoneResult Most Tel Proxy is not valid"));
      }
   }
}


/************************************************************************
*FUNCTION: 		onSetRingtoneError
*DESCRIPTION: 	Set Ringtone Error
*PARAMETER:		error : Error of set Ringtone

* Return : void
*
*HISTORY:
*
*Rev 0.1 Initial Revision.
************************************************************************/
#ifdef VARIANT_S_FTR_ENABLE_PHONECALLMANAGER_SUPPORT
void ServiceTel:: onSetRingtoneError(const ::boost::shared_ptr< MOST_Tel_FIProxy >& proxy, const ::boost::shared_ptr< SetRingtoneError >& error)
{
   ETG_TRACE_USR4(("ServiceTel :: onSetRingtoneError"));
}


/************************************************************************
*FUNCTION: 		onSetRingtoneResult
*DESCRIPTION: 	Set Ringtone Result
*PARAMETER:		error : Result of set Ringtone

* Return : void
*
*HISTORY:
*
*Rev 0.1 Initial Revision.
************************************************************************/
void ServiceTel:: onSetRingtoneResult(const ::boost::shared_ptr< MOST_Tel_FIProxy >& proxy, const ::boost::shared_ptr< SetRingtoneResult >& result)
{
   ETG_TRACE_USR4(("ServiceTel :: onSetRingtoneResult"));
}


/************************************************************************
*FUNCTION: 		onSystemWideRingtonesListError
*DESCRIPTION: 	SystemWideRingtonesListError
*PARAMETER:		error : SystemWideRingtonesList error

* Return : void
*
*HISTORY:
*
*Rev 0.1 Initial Revision.
************************************************************************/
void ServiceTel:: onSystemWideRingtonesListError(const ::boost::shared_ptr< MOST_Tel_FIProxy >& proxy, const ::boost::shared_ptr< SystemWideRingtonesListError >& error)
{
   ETG_TRACE_USR4(("ServiceTel :: onSystemWideRingtonesListError"));
}


/************************************************************************
*FUNCTION: 		onSystemWideRingtonesListStatus
*DESCRIPTION: 	SystemWideRingtonesListStatus
*PARAMETER:		Status : Result of SystemWideRingtonesList

* Return : void
*
*HISTORY:
*
*Rev 0.1 Initial Revision.
************************************************************************/
void ServiceTel:: onSystemWideRingtonesListStatus(const ::boost::shared_ptr< MOST_Tel_FIProxy >& proxy, const ::boost::shared_ptr< SystemWideRingtonesListStatus >& status)
{
   ETG_TRACE_USR4(("ServiceTel :: onSystemWideRingtonesListStatus"));
   ETG_TRACE_USR4(("ServiceTel::onSystemWideRingtonesListStatus : getU8ActiveSystemRingtoneID : %d", status->getU8ActiveSystemRingtoneID()));

   if ((NULL != PhonebookModelComponent::poGetInstance()))
   {
      PhonebookModelComponent::poGetInstance()->OnUpdChangeRingtoneStatus(status->getU8ActiveSystemRingtoneID());
   }
}


/************************************************************************
*FUNCTION: 		onDeviceRingtoneListError
*DESCRIPTION: 	DeviceRingtoneListError
*PARAMETER:		error : DeviceRingtoneList error

* Return : void
*
*HISTORY:
*
*Rev 0.1 Initial Revision.
************************************************************************/
void ServiceTel:: onDeviceRingtoneListError(const ::boost::shared_ptr< MOST_Tel_FIProxy >& proxy, const ::boost::shared_ptr< DeviceRingtoneListError >& error)
{
   ETG_TRACE_USR4(("ServiceTel :: onDeviceRingtoneListError"));
}


/************************************************************************
*FUNCTION: 		onDeviceRingtoneListStatus
*DESCRIPTION: 	DeviceRingtoneListStatus
*PARAMETER:		Status : Result of DeviceRingtoneList

* Return : void
*
*HISTORY:
*
*Rev 0.1 Initial Revision.
************************************************************************/
void ServiceTel:: onDeviceRingtoneListStatus(const ::boost::shared_ptr< MOST_Tel_FIProxy >& proxy, const ::boost::shared_ptr< DeviceRingtoneListStatus >& status)
{
   ETG_TRACE_USR4(("ServiceTel :: onDeviceRingtoneListStatus"));
}


/************************************************************************
*FUNCTION: 		onCallOperationFailedEventError
*DESCRIPTION: 	CallOperationFailedEventError
*PARAMETER:		error : CallOperationFailedEvent Error

* Return : void
*
*HISTORY:
*
*Rev 0.1 Initial Revision.
************************************************************************/
void ServiceTel:: onCallOperationFailedEventError(const ::boost::shared_ptr< MOST_Tel_FIProxy >& proxy, const ::boost::shared_ptr< CallOperationFailedEventError >& error)
{
   ETG_TRACE_USR4(("ServiceTel :: onCallOperationFailedEventError"));
}


/************************************************************************
*FUNCTION: 		onCallOperationFailedEventStatus
*DESCRIPTION: 	CallOperationFailedEventStatus
*PARAMETER:		Status : Result of CallOperationFailedEventStatus

* Return : void
*
*HISTORY:
*
*Rev 0.1 Initial Revision.
************************************************************************/
void ServiceTel:: onCallOperationFailedEventStatus(const ::boost::shared_ptr< MOST_Tel_FIProxy >& proxy, const ::boost::shared_ptr< CallOperationFailedEventStatus >& status)
{
   ETG_TRACE_USR4(("ServiceTel : onCallOperationFailedEventStatus()"));

   std::string strCallOpFailedEvent = status->getSOperation().c_str();

   if (strCallOpFailedEvent == "AcceptCall Failed")
   {
      ETG_TRACE_USR4(("ServiceTel : onAcceptCallError()"));
      bCallReqFlag = FALSE;
   }

   if (strCallOpFailedEvent == "HangupCall Failed")
   {
      ETG_TRACE_USR4(("ServiceTel : onHangUpCallError()"));

      bCallReqFlag = FALSE;
   }

   if (strCallOpFailedEvent == "SwapCall Failed")
   {
      ETG_TRACE_USR4(("ServiceTel: onSwapCallError"));
   }

   if (strCallOpFailedEvent == "ConferenceCall Failed")
   {
      ETG_TRACE_USR4(("ServiceTel: onConferenceCallError"));
   }

   if (strCallOpFailedEvent == "SplitCall Failed")
   {
      ETG_TRACE_USR4(("ServiceTel: onSplitCallError"));
   }
}


/************************************************************************
*FUNCTION: 		onSendDTMFFailedEventError
*DESCRIPTION: 	SendDTMFFailedEventError
*PARAMETER:		error : onSendDTMFFailedEvent Error

* Return : void
*
*HISTORY:
*
*Rev 0.1 Initial Revision.
************************************************************************/
void ServiceTel:: onSendDTMFFailedEventError(const ::boost::shared_ptr< MOST_Tel_FIProxy >& proxy, const ::boost::shared_ptr< SendDTMFFailedEventError >& error)
{
   ETG_TRACE_USR4(("ServiceTel : onSendDTMFFailedEventError()"));
}


/************************************************************************
*FUNCTION: 		onSendDTMFFailedEventStatus
*DESCRIPTION: 	SendDTMFFailedEventStatus
*PARAMETER:		Status : Result of SendDTMFFailedEventStatus

* Return : void
*
*HISTORY:
*
*Rev 0.1 Initial Revision.
************************************************************************/
void ServiceTel:: onSendDTMFFailedEventStatus(const ::boost::shared_ptr< MOST_Tel_FIProxy >& proxy, const ::boost::shared_ptr< SendDTMFFailedEventStatus >& status)
{
   ETG_TRACE_USR4(("ServiceTel : onSendDTMFFailedEventStatus()"));
}


/************************************************************************
*FUNCTION: 		onAudioTransferFailedEventError
*DESCRIPTION: 	AudioTransferFailedEventError
*PARAMETER:		error : AudioTransferFailedEvent Error

* Return : void
*
*HISTORY:
*
*Rev 0.1 Initial Revision.
************************************************************************/
void ServiceTel:: onAudioTransferFailedEventError(const ::boost::shared_ptr< MOST_Tel_FIProxy >& proxy, const ::boost::shared_ptr< AudioTransferFailedEventError >& error)
{
   ETG_TRACE_USR4(("ServiceTel : onAudioTransferFailedEventError()"));
}


/************************************************************************
*FUNCTION: 		onAudioTransferFailedEventStatus
*DESCRIPTION: 	AudioTransferFailedEventStatus
*PARAMETER:		Status : Result of onAudioTransferFailedEventStatus

* Return : void
*
*HISTORY:
*
*Rev 0.1 Initial Revision.
************************************************************************/
void ServiceTel:: onAudioTransferFailedEventStatus(const ::boost::shared_ptr< MOST_Tel_FIProxy >& proxy, const ::boost::shared_ptr< AudioTransferFailedEventStatus >& status)
{
   ETG_TRACE_USR4(("ServiceTel : onAudioTransferFailedEventStatus()"));

   if (status->getE8AudioDirection() == most_Tel_fi_types_Extended::T_e8_AudioTransferDirection__e8HF_TO_AG)
   {
      ETG_TRACE_FATAL(("ServiceTel : onTransferCallToHandsetError()"));
   }
   else if (status->getE8AudioDirection() == most_Tel_fi_types_Extended::T_e8_AudioTransferDirection__e8AG_TO_HF)
   {
      ETG_TRACE_FATAL(("ServiceTel : onTransferCallToVehicleError()"));
   }
   else
   {
   }
}


#endif

/************************************************************************
*FUNCTION: 		 IncomingCallResponseReq
*DESCRIPTION: 	 Handles GUI Courier Message to Accept of Reject Incoming Call Instance
*
*PARAMETER:
* 					 corfoMsg:	(I)
* 						Courier message received
*
*RETURNVALUE: 	 tBool
*						TRUE, if the message is consumed
*						FALSE, if the message should be forwarded
*
*HISTORY:
*revision 0.1	ess7kor	1/09/2018
************************************************************************/
bool ServiceTel::onCourierMessage(const IncomingCallResponseReq& oMsg)
{
   ETG_TRACE_USR4(("ServiceTel :: onCourierMessage : IncomingCallResponseReq Call status= %d",
                   ETG_CENUM(btphn_tenPhoneIncomingCallResponse, (btphn_tenPhoneIncomingCallResponse)oMsg.GetCallStatus())));

   //Extract the Call Status recieved in the COurier Message
   btphn_tenPhoneIncomingCallResponse enIncomingCallResponse =	oMsg.GetCallStatus();

   if (BTPHN_EN_INCOMINGCALL_RESPONSE_ACCEPT == enIncomingCallResponse)
   {
      /* Requesting Phone Client Handler to Accept Incoming call */
      vAcceptCallStart(u8IncomingCallInstance);
   }
   else if (BTPHN_EN_INCOMINGCALL_RESPONSE_REJECT == enIncomingCallResponse)
   {
      /* Requesting Phone Client Handler to Reject Incoming call */
      m_u8PassiveDevHandle = 0;
      vIgnoreCallStart(u8IncomingCallInstance);
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
      OnUpdMissedIntercomCall();
#endif
      //todo
      //bIsCallRejected = TRUE;
   }

   return TRUE;
}


/************************************************************************
*FUNCTION: 		 TransferVehicleAudioReq
*DESCRIPTION: 	 Handles GUI Courier Message to Transfer Vehicle AUdio Req
*
*PARAMETER:
* 					 corfoMsg:	(I)
* 						Courier message received
*
*RETURNVALUE: 	 tBool
*						TRUE, if the message is consumed
*						FALSE, if the message should be forwarded
*
*HISTORY:
*revision 0.1	ess7kor	1/09/2018
************************************************************************/
bool ServiceTel::onCourierMessage(const TransferVehicleAudioReq& oMsg)
{
   ETG_TRACE_USR4(("ServiceTel :: onCourierMessage : TransferVehicleAudioReq"));

   if (_most_tel_fi && _most_tel_fi.get())
   {
      if ((BTPHN_C_NUM_CALLS_SUPPORTED > u8CurrentCallInstance) &&
            (FALSE == arCallStatusInfo[u8CurrentCallInstance].bUsingVehicleAudio))
      {
         /* Requesting Phone Client Handler to transfer call to vehicle */
         _most_tel_fi->sendTransferCallToVehicleStart(*this, m_u8ActiveDev);
      }
      else
      {
         /* Requesting Phone Client Handler to transfer call to handset */
         _most_tel_fi->sendTransferCallToHandsetStart(*this);
      }
   }

   return TRUE;
}


/************************************************************************
*FUNCTION: 		 MicMuteStateReq
*DESCRIPTION: 	 Handles GUI Courier Message to Transfer Vehicle AUdio Req
*
*PARAMETER:
* 					 corfoMsg:	(I)
* 						Courier message received
*
*RETURNVALUE: 	 tBool
*						TRUE, if the message is consumed
*						FALSE, if the message should be forwarded
*
*HISTORY:
*revision 0.1	ess7kor	1/09/2018
************************************************************************/
bool ServiceTel::onCourierMessage(const MicMuteStateReq& oMsg)
{
   ETG_TRACE_USR4(("ServiceTel :: onCourierMessage : MicMuteStateReq"));
   ETG_TRACE_USR4(("ServiceTel :: bMicrophoneState is %d", bMicrophoneState));
   if (_most_tel_fi && _most_tel_fi.get())

   {
      _most_tel_fi->sendMicrophoneMuteStatePureSet(!bMicrophoneState);
   }

   return TRUE;
}


/************************************************************************
*FUNCTION: 		 TerminateCallReq
*DESCRIPTION: 	 Handles GUI Courier Message to Terminate Active Call
*
*PARAMETER:
* 					 corfoMsg:	(I)
* 						Courier message received
*
*RETURNVALUE: 	 tBool
*						TRUE, if the message is consumed
*						FALSE, if the message should be forwarded
*
*HISTORY:
*revision 0.1	ess7kor	1/09/2018
************************************************************************/
bool ServiceTel::onCourierMessage(const TerminateCallReq& oMsg)
{
   ETG_TRACE_USR4(("ServiceTel :: onCourierMessage : TerminateCallReq"));
   ETG_TRACE_USR4(("ServiceTel :: bMicrophoneState is %d", bMicrophoneState));
   if (_most_tel_fi && _most_tel_fi.get())

   {
      tU8 u8CallInstance = (TRUE == bIsConference) ? BTPHN_C_CALL_INSTANCE_INVALID : arCallStatusInfo[u8CurrentCallInstance].u16CallInstance;
      _most_tel_fi->sendHangUpCallStart(*this, u8CallInstance);
      bCallReqFlag = TRUE;
      ETG_TRACE_USR4(("CallReqFlag : %d", bCallReqFlag));
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
      OnUpdMissedIntercomCall();
#endif
   }
   else
   {
      ETG_TRACE_ERR(("ServiceTel : Phone service is unavailable hence request not posted"));
      /*If BTService is not avilable then error will be returned to HMI*/
      vResetCallFlag();
      vSendServiceError(BTPHN_EN_PHN_HANGUPCALL, BTPHN_EN_ERRCODE_NOERROR);
   }

   return TRUE;
}


/************************************************************************
*FUNCTION: 		 CancelDialCallReq
*DESCRIPTION: 	 Handles GUI Courier Message to Cancel Dialing Call
*
*PARAMETER:
* 					 corfoMsg:	(I)
* 						Courier message received
*
*RETURNVALUE: 	 tBool
*						TRUE, if the message is consumed
*						FALSE, if the message should be forwarded
*
*HISTORY:
*revision 0.1	ess7kor	1/09/2018
************************************************************************/
bool ServiceTel::onCourierMessage(const CancelDialCallReq& oMsg)
{
   ETG_TRACE_USR4(("ServiceTel :: onCourierMessage : CancelDialCallReq"));
   if (_most_tel_fi && _most_tel_fi.get())
   {
      if (FALSE == bCallReqFlag)
      {
         _most_tel_fi->sendCancelOutgoingCallStart(*this, arCallStatusInfo[u8CurrentCallInstance].u16CallInstance);
         bCallReqFlag = TRUE;
         ETG_TRACE_USR4(("CallReqFlag : %d", bCallReqFlag));
      }
      else
      {
         ETG_TRACE_ERR(("ServiceTel : vCancelDialCallStart() Request not sent to Phone Service"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("ServiceTel : Phone service is unavailable hence request not posted"));
      /*If BTService is not avilable then error will be returned to HMI*/
      vResetCallFlag();
      vSendServiceError(BTPHN_EN_PHN_CANCELOUTGOINGCALL, BTPHN_EN_ERRCODE_NOERROR);
   }

   return TRUE;
}


/************************************************************************
*FUNCTION: 		 SwapMultipleCallReq
*DESCRIPTION: 	 Handles GUI Courier Message to Swap 2 calls
*
*PARAMETER:
* 					 oMsg:	(I)
* 						Courier message received
*
*RETURNVALUE: 	 tBool
*						TRUE, if the message is consumed
*						FALSE, if the message should be forwarded
*
*HISTORY:
*revision 0.1	ess7kor	1/09/2018
************************************************************************/
bool ServiceTel::onCourierMessage(const SwapMultipleCallReq& oMsg)
{
   ETG_TRACE_USR4(("ServiceTel :: onCourierMessage : SwapMultipleCallReq"));
   BTPHN_M_UNUSED(oMsg);

   if (_most_tel_fi && _most_tel_fi.get())

   {
      if (FALSE == bIsConference)
      {
         u8NumSwapRequests++;

         /* Requesting Phone Client Handler to send swap multiparty call */
         T_TelStreamOfCallInstances callInstances;
         callInstances.clear();
         callInstances.push_back(arCallStatusInfo[BTPHN_C_FIRST_CALL_INDEX].u16CallInstance);
         callInstances.push_back(arCallStatusInfo[BTPHN_C_SECOND_CALL_INDEX].u16CallInstance);
         if (callInstances.size())
         {
            ETG_TRACE_USR4(("T_TelStreamOfCallInstances :CallInstance1 %d,  CallInstance2 %d", callInstances[0], callInstances[1]));
            _most_tel_fi->sendSwapCallStart(*this, callInstances);
         }
      }
      else
      {
         ETG_TRACE_USR4(("Swap not performed for a conference call"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("ServiceTel : Phone service is unavailable hence request not posted"));
      vSendServiceError(BTPHN_EN_PHN_SWAPCALL, BTPHN_EN_ERRCODE_NOERROR);
   }

   return TRUE;
}


/************************************************************************
*FUNCTION: 		 NetworkOperatorReq
*DESCRIPTION: 	 This is posted when Home screen is on foreground.
Funcxtion is responsible for updating GUI with valid network operator name
*
*PARAMETER: 					 oMsg:	(I)
* 						Courier message received
*
*RETURNVALUE: 	 tBool
*						TRUE, if the message is consumed
*						FALSE, if the message should be forwarded
*
*HISTORY:
*revision 0.1	kis8kor
************************************************************************/
bool ServiceTel::onCourierMessage(const NetworkOperatorReq& oMsg)
{
   if (!_isCabinOnly)
   {
      ETG_TRACE_USR4(("ServiceTel :: onCourierMessage : NetworkOperatorReq"));
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
      if (Sm_ActiveSource == 0)
      {
#endif
         if (ServiceBT::poGetInstance()->GetNoOfHFPDevicesConnected() > 0)
         {
            if (_most_tel_fi && _most_tel_fi.get())
            {
               ETG_TRACE_USR4(("ServiceTel :: onCourierMessage : NetworkOperatorReq - Request for Connected Phone Details"));
               _most_tel_fi->sendCellSignalStrengthGet(*this);
               _most_tel_fi->sendCellDeviceNameGet(*this);
               _most_tel_fi->sendCellStateofChargeGet(*this);
               _most_tel_fi->sendCellNetworkStatusGet(*this);
            }
            else
            {
               ETG_TRACE_USR4(("ServiceTel :: onCourierMessage : NetworkOperatorReq - Proxy is not available"));
            }
         }
         else
         {
            ETG_TRACE_USR4(("ServiceTel :: onCourierMessage : NetworkOperatorReq - No Phone Connected to Head unit"));
            rfoPhoneCallModel.OnUpdDeviceConnectionStatus();
         }

         if (NULL != ServicePhoneBook::poGetInstance())
         {
            rfoCallHistoryModel.OnUpdMissedCallIndicator(ServicePhoneBook::poGetInstance()->u32GetNumOfMissedCall());
         }

#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
      }
      else
      {
         IntercomModel::poGetInstance()->vSetIntercomActive();
      }
#endif
   }
   return TRUE;
}


bool ServiceTel::onCourierMessage(const BTVROnExitMsg& oMsg)
{
   ETG_TRACE_USR4(("ServiceTel :: onCourierMessage : BTVROnExitMsg"));
   /* if (m_bBTVoiceActive == TRUE)
   {
      vSet_BTVoiceRecognition(0);
   }
   m_u8BTVRSessionActive = 0;
   //poSmartPhoneIntegrationApp->covSendGlobalEvt(eAppGlobalEvt_SIRI_STATUS, m_u8BTVRSessionActive);
   bBTVRViewShown = 0; */

   return TRUE;
}


bool ServiceTel::onCourierMessage(const BTVRSessionExitAppReqMsg& oMsg)
{
   ETG_TRACE_USR4(("ServiceTel :: onCourierMessage : BTVRSessionExitAppReqMsg"));
   //SWITCH_TO_BTVR - Value of 5
   vSet_BTVoiceRecognition(FALSE);

   return TRUE;
}


bool ServiceTel::onCourierMessage(const BTVRSessionStartAppReqMsg& oMsg)
{
   ETG_TRACE_USR4(("ServiceTel :: onCourierMessage : BTVRSessionStartAppReqMsg"));

   most_Tel_fi_types_Extended::T_e8_TelSiriAvailabilityState temp_enSiriAvailabilityState = T_e8_TelSiriAvailabilityState__e8NOT_AVAILABLE;

   //Get the Siri availability status for the active device handle
   if (BtvrExtendedStatus[BTPHN_C_HFP_INDEX_ZERO].u8DeviceHandle == m_u8ActiveDev)
   {
      temp_enSiriAvailabilityState = BtvrExtendedStatus[BTPHN_C_HFP_INDEX_ZERO].enSiriAvailabilityState;
   }
   else if (BtvrExtendedStatus[BTPHN_C_HFP_INDEX_ONE].u8DeviceHandle == m_u8ActiveDev)
   {
      temp_enSiriAvailabilityState = BtvrExtendedStatus[BTPHN_C_HFP_INDEX_ONE].enSiriAvailabilityState;
   }
   else
   {
   }

   /* If Siri availability state is disabled, show the Siri EyeFree popup, otherwise activate the BTVR */
   if (temp_enSiriAvailabilityState == most_Tel_fi_types_Extended::T_e8_TelSiriAvailabilityState__e8AVAILABLE_DISABLED)
   {
      POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_PhoneModule#Sys_Popup#Pfo_SIRI_EyeFree"))));
      ETG_TRACE_USR4(("ServiceTel :: onCourierMessage : BTVRSessionStartAppReqMsg : Show SIRI EyeFree Popup"));
   }
   else
   {
      /*AKM2COB: For requirement like Short press SWC -> Continue BTVR in SPI is Active and Inactive both cases*/
      vSet_BTVoiceRecognition(TRUE);
   }

   return TRUE;
}


bool ServiceTel::onCourierMessage(const BTVRSessionStopAppReqMsg& oMsg)
{
   ETG_TRACE_USR4(("ServiceTel :: onCourierMessage : BTVRSessionStopAppReqMsg"));
   if (m_bBTVoiceActive == TRUE)
   {
      vSet_BTVoiceRecognition(FALSE);
   }
   else
   {
      ETG_TRACE_USR4(("ServiceTel:Session Not Active"));
   }
   return TRUE;
}


/************************************************************************
*FUNCTION: 		 TransferVehicleAudioReqFromCAN
*DESCRIPTION: 	 Handles GUI Courier Message to Transfer Vehicle AUdio Req
************************************************************************/
bool ServiceTel::onCourierMessage(const TransferVehicleAudioReqFromCAN& oMsg)
{
   ETG_TRACE_USR4(("ServiceTel :: onCourierMessage : TransferVehicleAudioReqFromCAN"));

   if (_most_tel_fi && _most_tel_fi.get())
   {
      if (BTPHN_C_NUM_CALLS_SUPPORTED > u8CurrentCallInstance)
      {
         if (FALSE == arCallStatusInfo[u8CurrentCallInstance].bUsingVehicleAudio)
         {
            ETG_TRACE_USR4(("TransferVehicleAudioReqFromCAN :Requesting Phone Client Handler to transfer call to vehicle"));
            _most_tel_fi->sendTransferCallToVehicleStart(*this, m_u8ActiveDev);
         }
         else
         {
            ETG_TRACE_USR4(("TransferVehicleAudioReqFromCAN :Requesting Phone Client Handler to transfer call to handset"));
            _most_tel_fi->sendTransferCallToHandsetStart(*this);
         }
      }
      else
      {
         ETG_TRACE_ERR(("unsupported call instance"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("ServiceTel : TransferVehicleAudioReq - Phone service is unavailable hence request not posted"));
   }

   return TRUE;
}


/************************************************************************
*FUNCTION: 		 MicMuteStateReqFromCAN
*DESCRIPTION: 	 Handles GUI Courier Message to mute Mic status Req

************************************************************************/
bool ServiceTel::onCourierMessage(const MicMuteStateReqFromCAN& oMsg)
{
   ETG_TRACE_USR4(("ServiceTel :: onCourierMessage : MicMuteStateReqFromCAN"));
   ETG_TRACE_USR4(("ServiceTel :: bMicrophoneState is %d", bMicrophoneState));
   if (_most_tel_fi && _most_tel_fi.get())

   {
      ETG_TRACE_USR4(("ServiceTel :: bMicrophoneState sent to Middleware %d", !bMicrophoneState));
      _most_tel_fi->sendMicrophoneMuteStatePureSet(!bMicrophoneState);
   }

   return TRUE;
}


/************************************************************************
*FUNCTION: 		bIsMultipleCallActive
*DESCRIPTION: 	Determines if the call number is single or multiple
*              Function is created for handling a specific context switching scenario
*PARAMETER: NA
*RETURNVALUE: bIsMultiCall
*AUTHOR: Bruce Netto v1.0 18/05/2021
************************************************************************/
bool ServiceTel::bIsMultipleCallActive()
{
   ETG_TRACE_USR4(("ServiceTel :: bIsMultipleCallActive"));
   bool bIsMultiCall = false;
   if (BTPHN_VALUE_ONE == u8NumActiveCalls)
   {
      ETG_TRACE_USR4(("ServiceTel :: bIsMultipleCallActive MultiCall is false"));
      bIsMultiCall = false;
   }
   else if ((BTPHN_VALUE_TWO == u8NumActiveCalls) || (bIsConference == true))
   {
      ETG_TRACE_USR4(("ServiceTel :: bIsMultipleCallActive MultiCall is true"));
      bIsMultiCall = true;
   }
   return bIsMultiCall;
}


/************************************************
Function 	: bGetActiveSmartVRDevice
Parameters 	: bIsSmartVRSIRIEnabled
Description : Determines if the current active device is SIRI enabled
SYSFL/SWFL  : CRQ 1053847 Currently Disabled
Author      : Bruce Netto 10/06/2021
************************************************/
bool ServiceTel::bGetActiveSmartVRDevice()
{
   ETG_TRACE_USR4(("ServiceTel::bGetActiveSmartVRDevice"));
   bool bIsSmartVRSIRIEnabled = false;
   if (BTPHN_VALUE_ZERO != m_u8ActiveDev)
   {
      ETG_TRACE_USR4(("ServiceTel::bGetActiveSmartVRDevice Active Device Handle is not 0"));
      if ((m_u8ActiveDev == BtvrExtendedStatus[BTPHN_C_HFP_INDEX_ONE].u8DeviceHandle &&
            most_Tel_fi_types_Extended::T_e8_TelSiriAvailabilityState__e8AVAILABLE_DISABLED !=
            BtvrExtendedStatus[BTPHN_C_HFP_INDEX_ONE].enSiriAvailabilityState) ||
            (m_u8ActiveDev == BtvrExtendedStatus[BTPHN_C_HFP_INDEX_ZERO].u8DeviceHandle &&
             most_Tel_fi_types_Extended::T_e8_TelSiriAvailabilityState__e8AVAILABLE_DISABLED !=
             BtvrExtendedStatus[BTPHN_C_HFP_INDEX_ZERO].enSiriAvailabilityState))
      {
         bIsSmartVRSIRIEnabled = true;
      }
   }
   else
   {
      ETG_TRACE_USR4(("ServiceTel::bGetActiveSmartVRDevice Active Device Handle is 0"));
   }
   ETG_TRACE_USR4(("ServiceTel::bGetActiveSmartVRDevice bIsSmartVRSIRIEnabled is %d", bIsSmartVRSIRIEnabled));
   return bIsSmartVRSIRIEnabled;
}


#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
void ServiceTel::vUpdActivePhone()
{
#ifdef VARIANT_S_FTR_ENABLE_BLUETOOTH_ON_OFF
   if (ServiceBT::poGetInstance()->GetServiceAvailableStatus() == true)
   {
      rfoPhoneCallModel.OnUpdNetworkOperatorName(strDeviceNameCurrent.c_str());
   }
#endif
#ifndef VARIANT_S_FTR_ENABLE_BLUETOOTH_ON_OFF
   rfoPhoneCallModel.OnUpdNetworkOperatorName(strDeviceNameCurrent.c_str());
#endif
   rfoCallHistoryModel.OnUpdCellStateOfCharge(u8ChargeStatus);
   rfoCallHistoryModel.OnUpdCellSignalStrength(u8NetworkStrength);
   rfoCallHistoryModel.OnUpdMissedCallIndicator(ServicePhoneBook::poGetInstance()->u32GetNumOfMissedCall());
}


#endif
} // namespace Core
}// namespace App
