/**************************************************************************************
 * @file         : SpeedLockStatusHandler.cpp
 * @addtogroup   : AppHmi_phone
 * @brief        :
 * @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 *29.07.20 Rev. 0.1 wut4kor(RBEI/EHM)
 * 		  			Initial Revision.
 **************************************************************************************/
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#include "hall_std_if.h"
#include "SpeedLockStatusHandler.h"
#include "AppHmi_PhoneStateMachine.h"
#include "ServiceTel.h"
#include "App/Core/Microphone/MicClientHandler.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_PHONE_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_PHONE
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_Phone_"
#define ETG_I_FILE_PREFIX                 App::Core::SpeedLockStatusHandler::
#include "trcGenProj/Header/SpeedLockStatusHandler.cpp.trc.h"
#endif
#else
#include "SpeedLockStatusHandler.h"

#endif  // End of Unittest Macro

namespace App {
namespace Core {

SpeedLockStatusHandler* SpeedLockStatusHandler::_SpeedLockStatusHandler = NULL;
SpeedLockStatusHandler& SpeedLockStatusHandler::getInstance()
{
   if (_SpeedLockStatusHandler == 0)
   {
      _SpeedLockStatusHandler = new SpeedLockStatusHandler();
   }
   return *_SpeedLockStatusHandler;
}


void SpeedLockStatusHandler::removeInstance()
{
   if (_SpeedLockStatusHandler)
   {
      delete _SpeedLockStatusHandler;
      _SpeedLockStatusHandler = 0;
   }
}


/************************************************************************
*FUNCTION		: SpeedLockStatusHandler
*DESCRIPTION	: Destructor.
*PARAMETER		:
*RETURNVALUE	: NA
************************************************************************/
SpeedLockStatusHandler::~SpeedLockStatusHandler()
{
}


/************************************************************************
*FUNCTION		: SpeedLockStatusHandler
*DESCRIPTION	: Constructor. Initializes the member variables.
*PARAMETER		:
*PARAMETER		:
*RETURNVALUE	: NA
************************************************************************/
SpeedLockStatusHandler::SpeedLockStatusHandler():
   m_SpeedLockActive(false)   //As the speed value will be starting from 0 or < 8kmph by default set false
   , m_SysSpeedlockEnable(true)  //System setting speed lock status will be enabled by default.

{
   ETG_TRACE_USR1(("SpeedLockStatusHandler::SpeedLockStatusHandler Constructor called"));
}


/************************************************************************
*FUNCTION		: updateSpeedValue
*DESCRIPTION	: This function is called whenever there is a change in the vehicle data speed.
	            : [SYSFL-8952,SYSFL-8953,SYSFL-8954,SYSFL-8955]
*RETURNVALUE	: NA
************************************************************************/
void SpeedLockStatusHandler::updateSpeedValue(uint16 speedValue)
{
   ETG_TRACE_USR4(("SpeedLockStatusHandler::updateSpeedValue() speedValue is :%d", (speedValue / 100)));
   ETG_TRACE_USR4(("SpeedLockStatusHandler::updateSpeedValue() m_SpeedLockActive is :%d", m_SpeedLockActive));
   uint16 u16speedValue = speedValue;
   bool isSpeedLockEnabled = false;
   if (SNA_VALUE == u16speedValue)
   {
      u16speedValue = BTPHN_C_DEFAULT_VALUE_ZERO;
      ETG_TRACE_USR4(("SpeedLockStatusHandler::updateSpeedValue() u16speedValue == SNA_VALUE"));
   }
   if (u16speedValue < MAX_SPEED)
   {
      ETG_TRACE_USR4(("SpeedLockStatusHandler::updateSpeedValue() u16speedValue < MAX_SPEED"));
      isSpeedLockEnabled = false;
   }
   else if (u16speedValue >= MAX_SPEED)
   {
      ETG_TRACE_USR4(("SpeedLockStatusHandler::updateSpeedValue() u16speedValue >= MAX_SPEED"));
      isSpeedLockEnabled = true;
   }

   if ((m_SpeedLockActive != isSpeedLockEnabled))
   {
      ETG_TRACE_USR4(("SpeedLockStatusHandler::updateSpeedValue() m_SpeedLockActive != isSpeedLockEnabled"));
      m_SpeedLockActive = isSpeedLockEnabled;
      updatespeedlockenabledStatus();
   }
   ETG_TRACE_USR4(("SpeedLockStatusHandler::updateSpeedValue() m_SpeedLockActive is :%d", m_SpeedLockActive));
   ETG_TRACE_USR4(("SpeedLockStatusHandler::updateSpeedValue() isSpeedLockEnabled is :%d", isSpeedLockEnabled));
}


bool SpeedLockStatusHandler::getSpeedLockStatus()
{
   ETG_TRACE_USR4(("updatespeedlockenabledStatus:getSpeedLockStatus"));

   if ((m_SpeedLockActive == true) && (m_SysSpeedlockEnable == true))
   {
      ETG_TRACE_USR4(("updatespeedlockenabledStatus:getSpeedLockStatus true"));
      return true;
   }
   else if ((m_SpeedLockActive == true) && (m_SysSpeedlockEnable == false))
   {
      ETG_TRACE_USR4(("updatespeedlockenabledStatus:getSpeedLockStatus false"));
      return false;
   }
   else if ((m_SpeedLockActive == false) && (m_SysSpeedlockEnable == true))
   {
      ETG_TRACE_USR4(("updatespeedlockenabledStatus:getSpeedLockStatus false"));
      return false;
   }
   else
   {
      ETG_TRACE_USR4(("updatespeedlockenabledStatus:getSpeedLockStatus false"));
      return false;
   }
}


/************************************************************************
*FUNCTION		: updatespeedlockenabledStatus
*DESCRIPTION	: update the databinding associated with SMS expand button,Dialpad button and Addnew button to enable/Disable the button
            	: SYSFL-8952,SYSFL-8953,SYSFL-8954,SYSFL-8955]
*PARAMETER		:
*RETURNVALUE	: NA
************************************************************************/
void SpeedLockStatusHandler::updatespeedlockenabledStatus()
{
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
   ETG_TRACE_USR4(("updatespeedlockenabledStatus:Syssetting speed lock status  is %d", m_SysSpeedlockEnable));
   ETG_TRACE_USR4(("updatespeedlockenabledStatus:vehicle speed status  is %d", m_SpeedLockActive));

   if (m_SysSpeedlockEnable == true)
   {
      if (m_SpeedLockActive)
      {
         (*speedlock).mSpeedLockStatus = false;
         speedlock.MarkItemModified(ItemKey::SpeedLock::SpeedLockStatusItem);		 // vehicle speed exceeds the threshold value
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
         if (MicClientHandler::poGetInstance()->getMicStatus() == true)
         {
            (*speedlock).mDialPadCallButtonStatus = false;
            speedlock.MarkItemModified(ItemKey::SpeedLock::DialPadCallButtonStatusItem);
            ETG_TRACE_USR4(("SpeedLockStatusHandler:: SpdLck: True MIC: True DialPad Call Button Disabled"));
         }
         if (MicClientHandler::poGetInstance()->getMicStatus() == false)
         {
            (*speedlock).mDialPadCallButtonStatus = false;
            speedlock.MarkItemModified(ItemKey::SpeedLock::DialPadCallButtonStatusItem);
            ETG_TRACE_USR4(("SpeedLockStatusHandler:: SpdLck: True MIC: False DialPad Call Button Disabled"));
         }
#else
         (*speedlock).mDialPadCallButtonStatus = false;
         speedlock.MarkItemModified(ItemKey::SpeedLock::DialPadCallButtonStatusItem);
         ETG_TRACE_USR4(("updatespeedlockenabledStatus: Dont send vabledStatus: Dont send vCheckforEnableDialpadCallButton"));
#endif

         /*send Message to IAR to close the SMS Expand Pop-up*/
         POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(SpeedLockstatusIndMsg)()));

         ETG_TRACE_USR4(("updatespeedlockenabledStatus: Send Cancel pairing to any ongoing pairing scenario"));
         // Posting the below msg, only cancels the pairing in the middleware, doesnot close the popup in the AppHmi_Phone.
         // Separate msg is posted to close the same in the following line.
         POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(SSPResponseReq)(BTPHN_EN_PAIRING_RESPONSE_CANCEL)));
         // Posting courier msg to close the all the pairing popups in the State machine
         POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ClosePairingPopupsOnSpeedLockstatusMsg)()));
      }
      else
      {
         (*speedlock).mSpeedLockStatus = true;
         speedlock.MarkItemModified(ItemKey::SpeedLock::SpeedLockStatusItem);		 // vehicle speed is below the threshold value
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
         if (MicClientHandler::poGetInstance()->getMicStatus() == true)
         {
            (*speedlock).mDialPadCallButtonStatus = false;
            speedlock.MarkItemModified(ItemKey::SpeedLock::DialPadCallButtonStatusItem);
            ETG_TRACE_USR4(("SpeedLockStatusHandler:: SpdLck: False MIC: True DialPad Call Button Disabled"));
         }
         if (MicClientHandler::poGetInstance()->getMicStatus() == false)
         {
            (*speedlock).mDialPadCallButtonStatus = true;
            speedlock.MarkItemModified(ItemKey::SpeedLock::DialPadCallButtonStatusItem);
            ETG_TRACE_USR4(("SpeedLockStatusHandler:: SpdLck: False MIC: False DialPad Call Button Enabled"));
         }
#else
         (*speedlock).mDialPadCallButtonStatus = true;
         speedlock.MarkItemModified(ItemKey::SpeedLock::DialPadCallButtonStatusItem);
         ETG_TRACE_USR4(("updatespeedlockenabledStatus: Dont send vCheckforEnableDialpadCallButton"));
#endif
      }
   }
   else
   {
      (*speedlock).mSpeedLockStatus = true; //system setting speedlock status is disabled so the databinding is set to true
      speedlock.MarkItemModified(ItemKey::SpeedLock::SpeedLockStatusItem);
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
      if (MicClientHandler::poGetInstance()->getMicStatus() == true)
      {
         (*speedlock).mDialPadCallButtonStatus = false;
         speedlock.MarkItemModified(ItemKey::SpeedLock::DialPadCallButtonStatusItem);
         ETG_TRACE_USR4(("SpeedLockStatusHandler:: SystemSettingSpdLck: False MIC: True DialPad Call Button Disabled"));
      }
      if (MicClientHandler::poGetInstance()->getMicStatus() == false)
      {
         (*speedlock).mDialPadCallButtonStatus = true;
         speedlock.MarkItemModified(ItemKey::SpeedLock::DialPadCallButtonStatusItem);
         ETG_TRACE_USR4(("SpeedLockStatusHandler:: SystemSettingSpdLck: False MIC: False DialPad Call Button Enabled"));
      }
#else
      (*speedlock).mDialPadCallButtonStatus = true;
      speedlock.MarkItemModified(ItemKey::SpeedLock::DialPadCallButtonStatusItem);
      ETG_TRACE_USR4(("updatespeedlockenabledStatus: Dont send vCheckforEnableDialpadCallButton"));
#endif
   }
   // speedlock.MarkAllItemsModified();
   speedlock.SendUpdate();
   ETG_TRACE_USR4(("updatespeedlockenabledStatus::speedlock data binding"));
#endif //end of unittest Macro
}


/************************************************************************
*FUNCTION		: updateSysSettingSpeedLockStatus
*DESCRIPTION	: This function is called whenever there is a change in the system setting value.
*DESCRIPTION	  To update the speed lock status value based on the system setting enable/Disable.
            	: [SWFL-3787,SYSFL-8955]
*PARAMETER		:
*RETURNVALUE	: NA
************************************************************************/
void SpeedLockStatusHandler::updateSysSettingSpeedLockStatus(bool speedLockStatus)
{
   ETG_TRACE_USR4(("updateSysSettingSpeedLockStatus:speedLockStatus is %d", speedLockStatus));
   if (m_SysSpeedlockEnable != speedLockStatus)
   {
      m_SysSpeedlockEnable = speedLockStatus;
      updatespeedlockenabledStatus();
   }
}


}//end of namespace Core
}//end of namespace App
