/* ***************************************************************************************
* FILE:          PhoneHall.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  PhoneHall.cpp is part of HMI-Base reference/demo/test applications
*    COPYRIGHT:  (c) 2017-2020 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "hall_std_if.h"
#include "PhoneHall.h"


#include "App/TTFisCmdHandler/Apphmi_PhoneTtfisHmiCmdHandler.h"
#include "App/datapool/PhoneDataPoolConfig.h"
#include "App/DataModel/PhonebookModelComponent.h"
#include "App/DataModel/PhoneGadgetModel.h"
#include "Common/VariantHandling/VariantHandling.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_PHONE_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_PHONE
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_PHONE_"
#define ETG_I_FILE_PREFIX                 App::Core::PhoneHall::
#include "trcGenProj/Header/PhoneHall.cpp.trc.h"
#endif

/*****************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------*/
#define APPLICATION_SWITCH_PORT "applicationSwitchPort"
#define EN_ACTIVITY_IDS bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs

const int TOGGLE_PHONE_HANDLE = 0;
const int SWITCH_TO_ADD_DEVICE = 1;
const int SWITCH_TO_CALL_LIST = 2;
const int SWITCH_TO_ACTIVE_PHONE_CALL_LIST = 3;
const int SWITCH_TO_SPEECH = 4;
const int SWITCH_TO_BTVR = 5;
const int SWITCH_TO_ACTIVE_PHONE = 6;
const int SWITCH_TO_CALL_CONNECTING = 7;
const int SWITCH_TO_CALL_ACCEPT = 8;
const int SWITCH_TO_CALL_BTVR_EXIT = 9;
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
const int SWITCH_TO_INTERCOM_CONTACTS = 10;
const int SWITCH_TO_INTERCOMCALL = 11;
const int EXIT_INTERCOMAPPLICATION = 12;
const int SWITCH_TO_INTERCOMCALLCONNECTING = 14;
//const int PHONEONE = 1;
//const int PHONETWO = 2;
//const int INTERCOM = 3;
#endif
const int SWITCH_TO_MULTICALL_ACCEPT = 13;
using namespace ::App::Core;
using namespace ::CourierTunnelService::CourierMessageReceiver;
//using namespace ::hmibase::appbase;

namespace App {
namespace Core {

DEFINE_CLASS_LOGGER_AND_LEVEL("App/Core", PhoneHall, Info);

PhoneHall::PhoneHall()
   : HallComponentBase("", "App.Core.AppHmi_Phone", "/org/genivi/NodeStateManager/LifeCycleConsumer/AppHmi_Phone"),
     _ServiceBT(NULL),
     _VehicleSrvClient(NULL),
     _ServicePhoneBook(NULL),
     _ServiceTel(NULL),
     _phClock(NULL),
     _SmartPhoneClient(NULL),
     _ServiceMsg(NULL),
     _applicationSwitchClientHandler(NULL),
     _isIncomingCallPopupActive(false),
     _isCallDialingInProgress(false),
     _isThereAnActiveCall(false),
     _isPhonceContextActive(false),
     _MicClientHandler(NULL),
     u16VehicleType(0),
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
     _IntercomClienthandler(NULL),
     u8ActiveSource(PHONEONE),
     _isIntercomIncomingCallPopupActive(false),
     _isIntercomCallDialingInProgress(false),
     _isThereAnActiveIntercomCall(false),
     _isThereConfIntercomCall(false),
#endif
     m_u8PhoneCurrentHandle(0),
     _enPreviousCallStatus(BTPHN_EN_IDLE),
     _enPreviousFirstCallStatus(BTPHN_EN_IDLE),
     _enPreviousSecondCallStatus(BTPHN_EN_IDLE)
{
   ETG_I_REGISTER_FILE();
   ETG_TRACE_FATAL(("PhoneHALL : PhoneHall Constructor \n"));
   ETG_TRACE_FATAL(("PhoneHALL : TR_TTFIS_APPHMI_PHONE is %d", TR_CLASS_APPHMI_PHONE_HALL));
   ETG_TRACE_FATAL(("PhoneHALL : TR_CLASS_APPHMI_PHONE_MAIN is %d", TR_CLASS_APPHMI_PHONE_MAIN));
   //_msgProxy= ::MOST_Msg_FI::MOST_Msg_FIProxy::createProxy("btMsgFiPort", *this);

   _ServiceBT = ServiceBT::poGetInstance();
   _VehicleSrvClient = VehicleSrvClient::poGetInstance();
   _phClock = ClockSrvClient::getInstance();
   _ServicePhoneBook = ServicePhoneBook::poGetInstance();
   _ServiceTel = ServiceTel::poGetInstance();
   _ServiceMsg = ServiceMsg::poGetInstance();
   _SmartPhoneClient = SmartPhoneClient::getInstance();
   _applicationSwitchClientHandler = new ApplicationSwitchClientHandler(APPLICATION_SWITCH_PORT);
   _MicClientHandler = MicClientHandler::poGetInstance();
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
   _IntercomClienthandler = IntercomClienthandler::poGetInstance();
#endif
   // StartupSync::getInstance().registerPropertyRegistrationIF(this, _msgProxy->getPortName());
   _applicationSwitchClientComponent = ApplicationSwitchClientComponent :: poGetInstance();

//Initialise the TraceCommand Handler
   Apphmi_PhoneTtfisHmiCmdHandler::initialize();

   // Initialize _isCabinOnly based on the VariantType
   if (VariantHandling::getInstance() != NULL)
   {
      _isCabinOnly = (VariantHandling::getInstance()->getVariantType() == VARIANTTYPE_CC);
   }
   else
   {
      _isCabinOnly = false; // Default to false if VariantHandling is not available
   }

   //for factory reset
   DefSetServiceBase::s_Intialize("appHmiPhoneServicePort");//port name needs to be same as port name given in cmc file
   _mdefSetServiceBase = DefSetServiceBase::GetInstance();
   if (_mdefSetServiceBase != NULL)
   {
      _mdefSetServiceBase->vRegisterforUpdate(this);
   }

   if (PhoneDataPoolConfig::getInstance())
   {
      ETG_TRACE_USR4(("PhoneHall::PhoneHall() - Retrive vehicleType information from KDS"));
      u16VehicleType = PhoneDataPoolConfig::getInstance()->getKDSVehicleType();
      ETG_TRACE_USR4(("PhoneHall::PhoneHall() - getKDSVehicleType, Vehicle type : %d", u16VehicleType));
   }
   else
   {
      ETG_TRACE_USR4(("PhoneHall::PhoneHall() - Error in getting PhoneDataPoolConfig instance"));
   }

   if ((VEHICLETYPE_COACH == u16VehicleType) || (VEHICLETYPE_TRUCK == u16VehicleType))
   {
      ETG_TRACE_USR4(("PhoneHall::PhoneHall() - Intercom and Kitchen option enabled for Coach"));
      sm_variant_intercom = 1;
      sm_variant_kitchen = 1;
   }
   else if (VEHICLETYPE_SMART == u16VehicleType)
   {
      ETG_TRACE_USR4(("PhoneHall::PhoneHall() - Intercom and Kitchen option diabled for smart"));
      sm_variant_intercom = 0;
      sm_variant_kitchen = 0;
   }
   else
   {
      ETG_TRACE_USR4(("PhoneHall::PhoneHall() - Intercom and Kitchen option disabled - unknown case vehicleType %d", u16VehicleType));
      //default to truck
      sm_variant_intercom = 1;
      sm_variant_kitchen = 1;
   }

   if (_applicationSwitchClientComponent != NULL)
   {
      _applicationSwitchClientComponent->setApplicationID(APPID_APPHMI_PHONE);
   }

   (void)NavigationModeDBHandler::getInstance();
}


PhoneHall::~PhoneHall()
{
   ETG_TRACE_USR4(("PhoneHall : Destructor"));

   delete _ServicePhoneBook;
   _ServicePhoneBook = NULL;

   delete _ServiceBT;
   _ServiceBT = NULL;

   delete _VehicleSrvClient;
   _VehicleSrvClient = NULL;

   delete _ServiceTel;
   _ServiceTel = NULL;

   delete _phClock;
   _phClock = NULL;

   delete _ServiceMsg;
   _ServiceMsg = NULL;

   delete _SmartPhoneClient;
   _SmartPhoneClient = NULL;

   delete _applicationSwitchClientHandler;
   _applicationSwitchClientHandler = NULL;

   delete _MicClientHandler;
   _MicClientHandler = NULL;

#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
   delete _IntercomClienthandler;
   _IntercomClienthandler = NULL;
#endif

   PhoneDataPoolConfig::deleteInstance();

   DefSetServiceBase::s_Destrory(); //to destroy DefSetServiceBase instance
   if (_mdefSetServiceBase != NULL)
   {
      _mdefSetServiceBase->vUnRegisterforUpdate(this);
      //donot delete this instance as its not created by us.
      _mdefSetServiceBase = NULL;
   }
}


void PhoneHall::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_FATAL(("PhoneHall::registerProperties(State:%d)", stateChange.getCurrentState()));

   // if (_ServiceMsg != NULL)
   //{
   //	   _ServiceMsg->registerProperties(proxy, stateChange);
   // }
}


void PhoneHall::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("PhoneHall::deregisterProperties(State:%d)", stateChange.getCurrentState()));

   /*	if (_ServiceMsg != NULL)
   	{
   		 _ServiceMsg->deregisterProperties(proxy, stateChange);
   	}*/
}


/************************************************************************
*Courier message: GuiStartupFinishedUpdMsg
*DESCRIPTION	: Informs once HMI is GUI update is finished
*PARAMETER		: NA
*PARAMETER		: NA
*RETURNVALUE	: TRUE if it consumes the courier message else FALSE
************************************************************************/
bool PhoneHall::onCourierMessage(const GuiStartupFinishedUpdMsg& msg)
{
   ETG_TRACE_USR4(("PhoneHall::onGuiStartupFinishedUpdMsg"));

   return false;
}


bool PhoneHall::onCourierMessage(const MultipleCallStatusResp& msg)
{
   btphn_tenPhoneCallStatus enCurrFirstCallStatus = msg.GetFirstCallStatus();
   btphn_tenPhoneCallStatus enCurrSecondCallStatus = msg.GetSecondCallStatus();
   ETG_TRACE_USR4(("PhoneHall::MultipleCallStatusResp enCurrFirstCallStatus - %d", enCurrFirstCallStatus));
   ETG_TRACE_USR4(("PhoneHall::MultipleCallStatusResp enCurrSecondCallStatus - %d", enCurrSecondCallStatus));

#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
   //If any of the Intercom call related operation is in progress, don't process BT call status
   if ((true == IsIntercomCallActive) || (true == IntercomClienthandler::poGetInstance()->getIntercomIncomingCallStatus()) || (true == IntercomClienthandler::poGetInstance()->getIntercomCallDialingStatus()))
   {
      ETG_TRACE_USR4(("PhoneHall::MultipleCallStatusResp - Intercom call is active, hence BT call status processing is blocked"));
      return false;
   }
#endif
   // use this boolean flag as argument to the restorePhoneApplicationScene function at the end
   bool restorePermanentScene = false;

   if (false == _isPhonceContextActive)
   {
      ETG_TRACE_USR4(("PhoneHall::MultipleCallStatusResp is processed in PhoneHall as phone is in BG"));
      if ((enCurrFirstCallStatus == BTPHN_EN_IDLE) && (enCurrSecondCallStatus == BTPHN_EN_IDLE)) /* && (g_SM_IsRVCViewEnabled == true)*/
      {
         // use case : Call Status is IDLE and RVC is Enaged
         restorePermanentScene = true;
      }
      else
      {
         // use case : Call Status is anything other IDLE and RVC is Enaged (Phone in BG) - update context to Master
         restorePermanentScene = false;
      }
   }
   else
   {
      ETG_TRACE_USR4(("PhoneHall::MultipleCallStatusResp - Not processed in PhoneHall as Phone is in FG. "));
   }

   // we cant update the master based on just CallStatus validate is Intercom call is active or not
   // though if there is no change in callstatus, we won't update. for additional constrainting, check intercom call status as well

   if ((_enPreviousFirstCallStatus != enCurrFirstCallStatus) && (_enPreviousSecondCallStatus != enCurrSecondCallStatus))
   {
      // Special case Patch: Phone stuck in disconnected state - DISCONNECTING to IDLE scene trigger is missed in SM while in Foreground
      if ((enCurrFirstCallStatus == BTPHN_EN_IDLE) && (enCurrSecondCallStatus == BTPHN_EN_IDLE))
      {
         // only reprocess info it is new update
         _applicationSwitchClientHandler->restorePhoneApplicationScene(true);
         // case scenario : previous call status was not IDLE, now only callStatus become idle, so we need to transit to permanent scene
         _applicationSwitchClientHandler->displayPhoneContextScene();
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
         _ServiceTel->OnUpdMissedIntercomCall();
#endif
      }
      else if (((_enPreviousFirstCallStatus == BTPHN_EN_IDLE) && (enCurrFirstCallStatus == BTPHN_EN_ACTIVE)) && ((_enPreviousSecondCallStatus == BTPHN_EN_IDLE) && (enCurrSecondCallStatus == BTPHN_EN_ON_HOLD)) && (false == _isPhonceContextActive))
      {
         ETG_TRACE_USR4(("PhoneHall::MultipleCallStatusResp -Phone is in Background and Voicemail is received "));
         sendRequestForContextSwitching(SWITCH_TO_MULTICALL_ACCEPT);
      }
      else if (((_enPreviousFirstCallStatus == BTPHN_EN_IDLE) && (enCurrFirstCallStatus == BTPHN_EN_ON_HOLD)) && ((_enPreviousSecondCallStatus == BTPHN_EN_IDLE) && (enCurrSecondCallStatus == BTPHN_EN_ACTIVE)) && (false == _isPhonceContextActive))
      {
         ETG_TRACE_USR4(("PhoneHall::MultipleCallStatusResp -Phone is in Background and Voicemail is received "));
         sendRequestForContextSwitching(SWITCH_TO_MULTICALL_ACCEPT);
      }
      else if (((_enPreviousFirstCallStatus == BTPHN_EN_IDLE) && (enCurrFirstCallStatus == BTPHN_EN_ON_HOLD)) && ((_enPreviousSecondCallStatus == BTPHN_EN_IDLE) && (enCurrSecondCallStatus == BTPHN_EN_ON_HOLD)) && (false == _isPhonceContextActive))
      {
         ETG_TRACE_USR4(("PhoneHall::MultipleCallStatusResp -Phone is in Background and Voicemail is received "));
         sendRequestForContextSwitching(SWITCH_TO_MULTICALL_ACCEPT);
      }
      else if (((_enPreviousFirstCallStatus == BTPHN_EN_IDLE) && (enCurrFirstCallStatus == BTPHN_EN_ACTIVE)) && ((_enPreviousSecondCallStatus == BTPHN_EN_IDLE) && (enCurrSecondCallStatus == BTPHN_EN_ACTIVE)) && (false == _isPhonceContextActive))
      {
         ETG_TRACE_USR4(("PhoneHall::MultipleCallStatusResp -Phone is in Background and Voicemail is received "));
         sendRequestForContextSwitching(SWITCH_TO_MULTICALL_ACCEPT);
      }
      else if (((_enPreviousFirstCallStatus == BTPHN_EN_IDLE) && (enCurrFirstCallStatus == BTPHN_EN_CONFERENCE)) && ((_enPreviousSecondCallStatus == BTPHN_EN_IDLE) && (enCurrSecondCallStatus == BTPHN_EN_CONFERENCE)) && (false == _isPhonceContextActive))
      {
         ETG_TRACE_USR4(("PhoneHall::MultipleCallStatusResp -Phone is in Background and Voicemail is received "));
         sendRequestForContextSwitching(SWITCH_TO_MULTICALL_ACCEPT);
      }
      else
      {
         // only reprocess info it is new update
         _applicationSwitchClientHandler->restorePhoneApplicationScene(restorePermanentScene);
      }

      _enPreviousFirstCallStatus = enCurrFirstCallStatus;
      _enPreviousSecondCallStatus = enCurrSecondCallStatus;
      ETG_TRACE_USR4(("PhoneHall::MultipleCallStatusResp -  updated _enPreviousFirstCallStatus to %d.", _enPreviousFirstCallStatus));
      ETG_TRACE_USR4(("PhoneHall::MultipleCallStatusResp -  updated _enPreviousSecondCallStatus to %d.", _enPreviousSecondCallStatus));
   }
   else
   {
      ETG_TRACE_USR4(("PhoneHall::MultipleCallStatusResp -  Signal Already Processed"));
   }

   return false;
}


bool PhoneHall::onCourierMessage(const CallStatusResp& msg)
{
   btphn_tenPhoneCallStatus enCurrCallStatus = msg.GetCallStatus();
   ETG_TRACE_USR4(("PhoneHall::CallStatusResp - %d", enCurrCallStatus));

#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
   //If any of the Intercom call related operation is in progress, don't process BT call status
   if ((true == IsIntercomCallActive) || (true == IntercomClienthandler::poGetInstance()->getIntercomIncomingCallStatus()) || (true == IntercomClienthandler::poGetInstance()->getIntercomCallDialingStatus()))
   {
      ETG_TRACE_USR4(("PhoneHall::CallStatusResp - Intercom call is active, hence BT call status processing is blocked"));
      return false;
   }
#endif
   // use this booleam flag as argument to the restorePhoneApplicationScene function at the end
   bool restorePermanentScene = false;

   if (false == _isPhonceContextActive)
   {
      ETG_TRACE_USR4(("PhoneHall::CallStatusResp is processed in PhoneHall as phone is in BG"));
      if ((enCurrCallStatus == BTPHN_EN_IDLE) /* && (g_SM_IsRVCViewEnabled == true)*/)
      {
         // use case : Call Status is IDLE and RVC is Enaged
         restorePermanentScene = true;
      }
      else
      {
         // use case : Call Status is anything other IDLE and RVC is Enaged (Phone in BG) - update context to Master
         restorePermanentScene = false;
      }
   }
   else
   {
      ETG_TRACE_USR4(("PhoneHall::CallStatusResp - Not processed in PhoneHall as Phone is in FG. "));
   }

   // we cant update the master based on just CallStatus validate is Intercom call is active or not
   // though if there is no change in callstatus, we won't update. for additional constrainting, check intercom call status as well

   if (_enPreviousCallStatus != enCurrCallStatus)
   {
      // Special case Patch: Phone stuck in disconnected state - DISCONNECTING to IDLE scene trigger is missed in SM while in Foreground
      if (enCurrCallStatus == BTPHN_EN_IDLE)
      {
         // only reprocess info it is new update
         _applicationSwitchClientHandler->restorePhoneApplicationScene(true);
         // case scenario : previous call status was not IDLE, now only callStatus become idle, so we need to transit to permanent scene
         _applicationSwitchClientHandler->displayPhoneContextScene();
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
         _ServiceTel->OnUpdMissedIntercomCall();
#endif
      }

      else if ((_enPreviousCallStatus == BTPHN_EN_IDLE) && (enCurrCallStatus == BTPHN_EN_ACTIVE) && (false == _isPhonceContextActive))
      {
         ETG_TRACE_USR4(("PhoneHall::CallStatusResp -Phone is in Background and Voicemail is received "));
         sendRequestForContextSwitching(SWITCH_TO_CALL_ACCEPT);
      }
      else
      {
         // only reprocess info it is new update
         _applicationSwitchClientHandler->restorePhoneApplicationScene(restorePermanentScene);
      }

      _enPreviousCallStatus = enCurrCallStatus;
      ETG_TRACE_USR4(("PhoneHall::CallStatusResp -  updated _enPreviousCallStatus to %d. ", _enPreviousCallStatus));
   }
   else
   {
      ETG_TRACE_USR4(("PhoneHall::CallStatusResp -  Signal Already Processed"));
   }

   return false;
}


bool PhoneHall::onCourierMessage(const ApplicationStateUpdMsg& msg)
{
   ETG_TRACE_USR4(("PhoneHall::onCourierMessage(ApplicationStateUpdMsg) -- %d ", msg.GetState()));

   if (msg.GetState() == hmibase::IN_FOREGROUND)
   {
      ETG_TRACE_USR4(("PhoneHall::Phone App is brought to Foreground"));
      _isPhonceContextActive = true;

      // read setting info from datapool when application is brought to foreground
      if ((NULL != PhonebookModelComponent::poGetInstance()))
      {
         // this is introduced to sync with Factory reset, and delayed Phone datapool update
         ETG_TRACE_USR4(("PhoneHall::Update PhonebookModelComponent with datapool values"));
         PhonebookModelComponent::poGetInstance()->Init();
      }

      _applicationSwitchClientHandler->displayPhoneContextScene();

      // Used keep an updated variable for processing External Scene Transition triggers
      _applicationSwitchClientHandler->SetCurrentPhoneStatus(EN_PHONE_CONTEXT_FG);
   }
   else if (msg.GetState() == hmibase::TO_BACKGROUND)
   {
      ETG_TRACE_USR4(("PhoneHall::Phone App is sent to Background"));
      _isPhonceContextActive = false;
      /* On change of application ID to other than phone, reset the background context flag to false */
      _applicationSwitchClientHandler->SetPhoneContextBgReqFlag(false);

      _applicationSwitchClientHandler->SetCurrentPhoneStatus(EN_PHONE_CONTEXT_BG);
   }
   else
   {
   }
   return false;

   // This function is called whenver change in the state from appmaster like fg,bg...
}


bool PhoneHall::onCourierMessage(const SwitchApplicationReq& oMsg)
{
   ETG_TRACE_USR3(("PhoneHall::onCourierMessage(SwitchApplicationReq)"));
   _applicationSwitchClientComponent -> requestSwitchApplication(APPID_APPHMI_MASTER);
   return true;
}


/**************************************************************************
* This is a courier message received when an Incoming call is accepted
* @param[in] - OnIncomingCallRespMsg
*
* @param[out]- bool
initial rev: Sai Prabha
* Req Ids : Feature 485201
**************************************************************************/


bool PhoneHall::onCourierMessage(const OnIncomingCallRespMsg& oMsg)
{
   ETG_TRACE_USR3(("PhoneHall::onCourierMessage(OnIncomingCallRespMsg)"));
   sendRequestForContextSwitching(SWITCH_TO_CALL_ACCEPT);

   return true;
}


/**************************************************************************
* This is a courier message received when user Hangs up the active call
* @param[in] - OnCallEndRespMsg
*
* @param[out]- void
initial rev: Sai Prabha
* Req Ids : Feature 485201
**************************************************************************/

bool PhoneHall::onCourierMessage(const OnCallEndRespMsg& oMsg)
{
   ETG_TRACE_USR3(("PhoneHall::onCourierMessage(OnCallEndRespMsg)"));
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
   //If any of the Intercom call related operation is in progress, don't process BT call status
   if ((true == IsIntercomCallActive) || (true == IntercomClienthandler::poGetInstance()->getIntercomIncomingCallStatus()) || (true == IntercomClienthandler::poGetInstance()->getIntercomCallDialingStatus()))
   {
      ETG_TRACE_USR4(("PhoneHall::OnCallEndRespMsg - Intercom call is active, hence BT call status processing is blocked"));
      return false;
   }
#endif
   if (_applicationSwitchClientHandler->GetPhoneContextStatus() == EN_PHONE_CONTEXT_BG)
   {
      ETG_TRACE_USR4(("PhoneHall::OnCallEndRespMsg - reset the PhoneContextStatus as we are returning back to last application"));
//changing the source activity id to none due to the changes in master team w.r.t Backward context switching
      _applicationSwitchClientComponent->requestContextSwitchApplication(getActiveregionid(),
            APPID_APPHMI_PHONE,
            (EN_ACTIVITY_IDS)0,//enActivityIDs__eActivityID_PHONE_ON_CALL,
            (EN_ACTIVITY_IDS)0,
            (EN_ACTIVITY_IDS)0);

      // reset the PhoneContextStatus as we are returning back to last application
      _applicationSwitchClientHandler->SetPhoneContextStatus(EN_PHONE_CONTEXT_UNKNOWN);
   }

   return true;
}


/**************************************************************************
* This is a courier message received when user Exits the BTVR using back button
* @param[in] - OnContextSwitchOnExitOfBTVRMsg
*
* @param[out]- void
initial rev: Sai Prabha
* Req Ids : Feature 485201
**************************************************************************/

bool PhoneHall::onCourierMessage(const OnContextSwitchOnExitOfBTVRMsg& oMsg)
{
   ETG_TRACE_USR3(("PhoneHall::onCourierMessage(OnContextSwitchOnExitOfBTVRMsg)"));
   if (_applicationSwitchClientHandler->GetPhoneContextStatus() == EN_PHONE_CONTEXT_BG)
   {
      sendRequestForContextSwitching(SWITCH_TO_CALL_BTVR_EXIT);
   }

   return true;
}


bool PhoneHall::onCourierMessage(const OnBackButtonPressMsg& oMsg)
{
   ETG_TRACE_USR3(("PhoneHall::onCourierMessage(OnBackButtonPressMsg)"));

   _applicationSwitchClientComponent->requestContextSwitchApplication(getActiveregionid(),
         APPID_APPHMI_PHONE,
         (EN_ACTIVITY_IDS)0,
         APPID_APPHMI_MASTER,
         _isCabinOnly ? enActivityIDs__eActivityID_CABINA_HOME : enActivityIDs__eActivityID_HOME);

   // reset the PhoneContextStatus as we are Going to Home Screen
//   _applicationSwitchClientHandler->SetPhoneContextStatus(EN_PHONE_CONTEXT_BG);

   return true;
}


#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
/**************************************************************************
* This is a courier message received when an Intercom Incoming call is accepted
* @param[in] - OnIntercomIncomingCallRespMsg
*
* @param[out]- bool
initial rev: Vidya
**************************************************************************/


bool PhoneHall::onCourierMessage(const OnIntercomIncomingCallRespMsg& oMsg)
{
   ETG_TRACE_USR3(("PhoneHall::onCourierMessage(OnIntercomIncomingCallRespMsg)"));
   if (_isPhonceContextActive == false)
   {
      _applicationSwitchClientComponent->requestContextSwitchApplication(getActiveregionid(),
            (EN_ACTIVITY_IDS)0,
            (EN_ACTIVITY_IDS)0,
            APPID_APPHMI_PHONE,
            enActivityIDs__eActivityID_PHONE_ON_INTERCOMCALL);

      // application is BG so request for context switch
      _applicationSwitchClientHandler->SetPhoneContextStatus(EN_PHONE_CONTEXT_BG);
   }
   else
   {
      _applicationSwitchClientHandler->SetPhoneContextStatus(EN_PHONE_CONTEXT_FG);

      //Application is all ready in foreground just make internal transition
      ETG_TRACE_USR4(("ApplicationSwitchClientHandler::displayPhoneContextScene::Posting ActivateOnIntercomCallSceneMsg"));
      POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateOnIntercomCallSceneMsg)()));
   }
   return true;
}


/**************************************************************************
* This is a courier message received when user Hangs up the active Intercom call
* @param[in] - OnIntercomCallEndRespMsg
*
* @param[out]- void
initial rev: Vidya
**************************************************************************/

bool PhoneHall::onCourierMessage(const OnIntercomCallEndRespMsg& oMsg)
{
   ETG_TRACE_USR3(("PhoneHall::onCourierMessage(OnIntercomCallEndRespMsg)"));

   // when intercom started Phone was in BG. hence req for con switch
   if (_applicationSwitchClientHandler->GetPhoneContextStatus() == EN_PHONE_CONTEXT_BG)
   {
      //changing the source activity id to none due to the changes in master team w.r.t Backward context switching
      _applicationSwitchClientComponent->requestContextSwitchApplication(getActiveregionid(),
            APPID_APPHMI_PHONE,
            (EN_ACTIVITY_IDS)0,//enActivityIDs__eActivityID_PHONE_ON_INTERCOMCALL,
            (EN_ACTIVITY_IDS)0,
            (EN_ACTIVITY_IDS)0);

      _applicationSwitchClientHandler->SetPhoneContextStatus(EN_PHONE_CONTEXT_UNKNOWN);
   }
   else
   {
      // AS Phone was already in foreground, Need not do context switch
      // Special case: On intercom call end status, restore last permanent scene to avoid any SM stuck scenario.
      _applicationSwitchClientHandler->restorePhoneApplicationScene(true);
      _applicationSwitchClientHandler->displayPhoneContextScene();
   }

   return true;
}


/**************************************************************************
* This is a courier message received when an Intercom Incoming call accepted is conference call
* @param[in] - OnIntercomIncomingConfCallRespMsg
*
* @param[out]- bool
initial rev: Vidya
**************************************************************************/


bool PhoneHall::onCourierMessage(const OnIntercomIncomingConfCallRespMsg& oMsg)
{
   ETG_TRACE_USR3(("PhoneHall::onCourierMessage(OnIntercomIncomingConfCallRespMsg)"));
   if (_isPhonceContextActive == false)
   {
      _applicationSwitchClientComponent->requestContextSwitchApplication(getActiveregionid(),
            (EN_ACTIVITY_IDS)0,
            (EN_ACTIVITY_IDS)0,
            APPID_APPHMI_PHONE,
            enActivityIDs__eActivityID_PHONE_ON_I_CONFERENCECALL);

      // application is BG so request for context switch
      _applicationSwitchClientHandler->SetPhoneContextStatus(EN_PHONE_CONTEXT_BG);
   }
   else
   {
      _applicationSwitchClientHandler->SetPhoneContextStatus(EN_PHONE_CONTEXT_FG);

      //Application is all ready in foreground just make internal transition
      ETG_TRACE_USR4(("ApplicationSwitchClientHandler::displayPhoneContextScene::Posting ActivateOnIntercomConfCallSceneMsg"));
      POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateOnIntercomConfCallSceneMsg)()));
   }

   return true;
}


/**************************************************************************
* This is a courier message received when user Hangs up the active Intercom call
* @param[in] - OnIntercomConfCallEndRespMsg
*
* @param[out]- void
initial rev: Vidya
**************************************************************************/

bool PhoneHall::onCourierMessage(const OnIntercomConfCallEndRespMsg& oMsg)
{
   ETG_TRACE_USR3(("PhoneHall::onCourierMessage(OnIntercomConfCallEndRespMsg)"));

   // when intercom started Phone was in BG. hence req for con switch
   if (_applicationSwitchClientHandler->GetPhoneContextStatus() == EN_PHONE_CONTEXT_BG)
   {
      //changing the source activity id to none due to the changes in master team w.r.t Backward context switching
      _applicationSwitchClientComponent->requestContextSwitchApplication(getActiveregionid(),
            APPID_APPHMI_PHONE,
            (EN_ACTIVITY_IDS)0,//enActivityIDs__eActivityID_PHONE_ON_I_CONFERENCECALL,
            (EN_ACTIVITY_IDS)0,
            (EN_ACTIVITY_IDS)0);

      _applicationSwitchClientHandler->SetPhoneContextStatus(EN_PHONE_CONTEXT_UNKNOWN);
   }
   else
   {
      // AS Phone was already in foreground, Need not do context switch
   }
// Special case: On intercom call end status, restore last permanent scene to avoid any SM stuck scenario.
   _applicationSwitchClientHandler->restorePhoneApplicationScene(true);
   _applicationSwitchClientHandler->displayPhoneContextScene();

   return true;
}


#endif

/************************************************
Function 	: HKStatusChangedUpdMsg
Parameters 	: GetHKCode
Description : Implementation for MCP Accept and Reject Btn by consuming hard key press from framework
v 0.1 initial ver :Pooja
SYSFL/SWFL : Implemented SWFL 2645, SWFL 2648 and SYSFL 4709, SYSFL 4712
************************************************/
bool PhoneHall::onCourierMessage(const onHardKeyPress& msg)
{
   ETG_TRACE_USR4(("PhoneHall::onHardKeyPress key = %d", msg.GetHKCode()));
   ETG_TRACE_USR4(("PhoneHall::onHardKeyPress HKCode: %d %d", msg.GetHKCode(), msg.GetHKState()));
   int hardKeyCode = msg.GetHKCode();
   int hardkeyState = msg.GetHKState();
   int retVal = false;

   //Update the flags
   _isIncomingCallPopupActive = _ServiceTel->getIncomingCallStatus();
   _isCallDialingInProgress = _ServiceTel->getCallDialingStatus();
   _isThereAnActiveCall = _ServiceTel->getOngoingCallStatus();

#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
   _isIntercomIncomingCallPopupActive = _IntercomClienthandler->getIntercomIncomingCallStatus();
   _isIntercomCallDialingInProgress = _IntercomClienthandler->getIntercomCallDialingStatus();
   _isThereAnActiveIntercomCall = _IntercomClienthandler->getIntercomOngoingCallStatus();
   _isThereConfIntercomCall = _IntercomClienthandler->getIntercomConfCallStatus();
   ETG_TRACE_USR4(("PhoneHall::onHardKeyPress _isIntercomIncomingCallPopupActive = %s", (_isIntercomIncomingCallPopupActive ? "true" : "false")));
   ETG_TRACE_USR4(("PhoneHall::onHardKeyPress _isIntercomCallDialingInProgress = %s", (_isIntercomCallDialingInProgress ? "true" : "false")));
   ETG_TRACE_USR4(("PhoneHall::onHardKeyPress _isThereAnActiveIntercomCall = %s", (_isThereAnActiveIntercomCall ? "true" : "false")));
   ETG_TRACE_USR4(("PhoneHall::onHardKeyPress _isThereConfIntercomCall = %s", (_isThereConfIntercomCall ? "true" : "false")));
#endif
   ETG_TRACE_USR4(("PhoneHall::onHardKeyPress _isIncomingCallPopupActive = %s", (_isIncomingCallPopupActive ? "true" : "false")));
   ETG_TRACE_USR4(("PhoneHall::onHardKeyPress _isCallDialingInProgress = %s", (_isCallDialingInProgress ? "true" : "false")));
   ETG_TRACE_USR4(("PhoneHall::onHardKeyPress _isThereAnActiveCall = %s", (_isThereAnActiveCall ? "true" : "false")));
   ETG_TRACE_USR4(("PhoneHall::onHardKeyPress _isPhonceContextActive = %s", (_isPhonceContextActive ? "true" : "false")));

   if ((g_SM_SpiSessionStatus == SPI_SESSION_INACTIVE) || (g_SM_SpiSessionStatus == SPI_SESSION_UNKNOWN))
   {
      if (VEHICLETYPE_SMART == u16VehicleType)
      {
         // Hard Key handling
         if (HARDKEYCODE_SWC_PHONE_PICKUP == hardKeyCode)
         {
            if (Enum_hmibase_HARDKEYSTATE_UP == hardkeyState)
            {
               retVal = HandleSWCShortPress(hardkeyState);
            }
            else
            {
               ETG_TRACE_USR4(("PhoneHall::onHardKeyPress invalid key state = %d", hardkeyState));
            }
         }
         else if (HARDKEYCODE_SWC_PHONE_REJECT == hardKeyCode)
         {
            if (Enum_hmibase_HARDKEYSTATE_UP == hardkeyState)
            {
               retVal = HandleSWCLongPress(hardkeyState);
            }
            else
            {
               ETG_TRACE_USR4(("PhoneHall::onHardKeyPress invalid key state = %d", hardkeyState));
            }
         }
         else
         {
            ETG_TRACE_USR4(("PhoneHall::onHardKeyPress invalid key Code = %d", hardKeyCode));
         }
      }
      else
      {
         // Hard Key handling
         if (HARDKEYCODE_SWC_PHONE_PICKUP == hardKeyCode)
         {
            retVal = HandleSWCPickupKeyPress(hardkeyState);
         }
         else if (HARDKEYCODE_SWC_PHONE_REJECT == hardKeyCode)
         {
            retVal = HandleSWCRejectKeyPress(hardkeyState);
         }
         else if (HARDKEYCODE_HK_PHONE_ACCEPT == hardKeyCode)
         {
            retVal = HandleMCPAcceptKeyPress(hardkeyState);
         }
         else if (HARDKEYCODE_HK_PHONE_REJECT == hardKeyCode)
         {
            retVal = HandleMCPRejectKeyPress(hardkeyState);
         }
         else if (HARDKEYCODE_REMOTECONTROL_PHONE == hardKeyCode)
         {
            ETG_TRACE_USR4(("PhoneHall::onHardKeyPress - HARDKEYCODE_REMOTECONTROL_PHONE"));
            retVal = HandleHKRemoteControlPhoneKeyPress(hardkeyState);
         }
         else if (((hardKeyCode >= HARDKEYCODE_REMOTECONTROL_NUM_9) && (hardKeyCode <= HARDKEYCODE_REMOTECONTROL_NUM_0)) || (hardKeyCode == HARDKEYCODE_REMOTECONTROL_HASH) || (hardKeyCode == HARDKEYCODE_REMOTECONTROL_STAR))//multiple conditions checked due to the values not part of the same number series.
         {
            ETG_TRACE_USR4(("PhoneHall::onHardKeyPress - HARDKEYCODE_REMOTECONTROL_DIALPAD_BUTTON: %d", hardKeyCode));
            retVal = HandleHKRemoteControlDialPadKeyPress(hardKeyCode);
         }
         else
         {
         }
      }
   }
   else
   {
      ETG_TRACE_USR4(("PhoneHall::onHardKeyPress Ignoring Hard Key as SPI connection is Active"));
   }

   return retVal;
}


/************************************************************************
*FUNCTION		: HandleSWCPickupKeyPress
*DESCRIPTION	: Member function to handle Steering Wheel Control Pick up button
*PARAMETER		: Hard Key State - Up (release), long (long press)
*RETURNVALUE	: void
*SWFL			: SYSFL-6422, SYSFL-6421, SYSFL-5680,SYSFL-5679, SYSFL-5681
************************************************************************/
bool PhoneHall::HandleSWCPickupKeyPress(const int hardkeyState)
{
   bool retval = false;
   ETG_TRACE_USR4(("PhoneHall::HandleSWCPickupKeyPress "));

   if (_ServiceBT->GetNoOfHFPDevicesConnected() > 0)
   {
      if (Enum_hmibase_HARDKEYSTATE_UP == hardkeyState)
      {
         if (_isIncomingCallPopupActive)
         {
            POST_MSG((COURIER_MESSAGE_NEW(IncomingCallResponseReq)(BTPHN_EN_INCOMINGCALL_RESPONSE_ACCEPT)));
            ETG_TRACE_USR4(("PhoneHall::HandleSWCPickupKeyPress - Accept the incoming Call via SWC pickup "));
            retval = true;
         }
         else if (_isCallDialingInProgress)
         {
            // if it is an outgoing Call - no reaction for pick up key
            ETG_TRACE_USR4(("PhoneHall::HandleSWCPickupKeyPress - SWC short press on outgoing call should not have any reaction."));
            retval = true;
         }
         else if (_isThereAnActiveCall)
         {
            // if in Active Call shall be terminated
            ETG_TRACE_USR4(("PhoneHall::HandleSWCPickupKeyPress - SWC short press on Active call should not have any reaction."));
            retval = true;
         }
         // DONT RE-ORDER THIS IF-ELSE. Only after comfirming that atleast one phone is connected,
         // we are trying to trigger BT-VR.
         // The Case of SPI connected via Android auto is already eliminated before this function call.
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
         else if (_isIntercomIncomingCallPopupActive)
         {
            POST_MSG((COURIER_MESSAGE_NEW(IntercomIncomingCallHardKeyResp)()));
            ETG_TRACE_USR4(("PhoneHall::HandleSWCPickupKeyPress - Accept the Intercom incoming Call via SWC pickup "));
            retval = true;
         }
         else if (_isIntercomCallDialingInProgress)
         {
            // if it is an outgoing Call - no reaction for pick up key
            ETG_TRACE_USR4(("PhoneHall::HandleSWCPickupKeyPress - SWC short press on Intercom outgoing call should not have any reaction."));
            retval = true;
         }
         else if (_isThereAnActiveIntercomCall)
         {
            // if in Active Call shall be terminated
            ETG_TRACE_USR4(("PhoneHall::HandleSWCPickupKeyPress - SWC short press on Intercom Active call should not have any reaction."));
            retval = true;
         }
         else if (_isThereConfIntercomCall)
         {
            // if in Active Call shall be terminated
            ETG_TRACE_USR4(("PhoneHall::HandleSWCPickupKeyPress - SWC short press on Intercom Conference call should not have any reaction."));
            retval = true;
         }
#endif
         else // if (_ServiceBT->GetNoOfHFPDevicesConnected() > 0)
         {
            //context switch to speech code snippet is removed by Vidya as per Ramesh inputs
            // because Speech should be triggered by master and phone component should not handle any key press wrt speech
            // Having a valid HFP connection without any calls (in/out/active) - phone on Foreground or background
            // And also make a BTVR request only when RVC is not active
            if ((bBTVRViewShown == TRUE) && (_applicationSwitchClientHandler->getCurrentActivityID() != enActivityIDs__eActivityID_RVC))
            {
               ETG_TRACE_USR4(("PhoneHall::HandleSWCPickupKeyPress -- Request for Smartphone VR "));
               sendRequestForBTVRFunctionHandling();
            }
         }
      }
      else if (Enum_hmibase_HARDKEYSTATE_LONG1 == hardkeyState)
      {
         if ((!_isIncomingCallPopupActive) && (!_isThereAnActiveCall) && (!_isCallDialingInProgress))//check intercom calls
         {
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
            if ((!_isIntercomIncomingCallPopupActive) && (!_isIntercomCallDialingInProgress) && (!_isThereAnActiveIntercomCall) && (!_isThereConfIntercomCall))
            {
#endif
               // Having a valid HFP connection without any calls (in/out/active)
               // And also make a BTVR request only when RVC is not active
               ETG_TRACE_USR4(("PhoneHall::HandleSWCPickupKeyPress:bBTVRViewShown =%d ", bBTVRViewShown));
               if (_applicationSwitchClientHandler->getCurrentActivityID() != enActivityIDs__eActivityID_RVC)
               {
                  /*AKM2cob: For requirement like Long press SWC -> Terminate BTVR in SPI is Active */
                  if (bBTVRViewShown == TRUE)
                  {
                     POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(BTVRSessionExitAppReqMsg)()));
                     ETG_TRACE_USR4(("PhoneHall::HandleSWCPickupKeyPress -- ExitBTVR "));
                  }
                  else
                  {
                     ETG_TRACE_USR4(("PhoneHall::HandleSWCPickupKeyPress -- Request for external VR - Phone is in Foreground or background "));
                     sendRequestForBTVRFunctionHandling();
                  }
               }
               retval = true;
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
            }
            else
            {
               // Intercom is not Idle, no Reaction on Long press
               ETG_TRACE_USR4(("PhoneHall::HandleSWCPickupKeyPress -- No reaction for long press when there is Intercom incoming/ outgoing/ Active call. "));
               retval = true;
            }
#endif
         }
         else
         {
            // Phone is not Idle, no Reaction on Long press
            ETG_TRACE_USR4(("PhoneHall::HandleSWCPickupKeyPress -- No reaction for long press when there is incoming/ outgoing/ Active call. "));
            retval = true;
         }
      }
      else
      {
         // Invalid Key state
         ETG_TRACE_USR4(("PhoneHall::HandleSWCPickupKeyPress -- Invalid Key State. "));
      }
   }
   else if (_ServiceBT->GetNoOfHFPDevicesConnected() == 0)
   {
      if (Enum_hmibase_HARDKEYSTATE_UP == hardkeyState)
      {
         //intercom to be handled
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
         if (_isIntercomIncomingCallPopupActive)
         {
            POST_MSG((COURIER_MESSAGE_NEW(IntercomIncomingCallHardKeyResp)()));
            ETG_TRACE_USR4(("PhoneHall::HandleSWCPickupKeyPress - Accept the Intercom incoming Call via SWC pickup "));
            retval = true;
         }
         else if (_isIntercomCallDialingInProgress)
         {
            // if it is an outgoing Call - no reaction for pick up key
            ETG_TRACE_USR4(("PhoneHall::HandleSWCPickupKeyPress - SWC short press on Intercom outgoing call should not have any reaction."));
            retval = true;
         }
         else if (_isThereAnActiveIntercomCall)
         {
            // if in Active Call shall be terminated
            ETG_TRACE_USR4(("PhoneHall::HandleSWCPickupKeyPress - SWC short press on Intercom Active call should not have any reaction."));
            retval = true;
         }
         else if (_isThereConfIntercomCall)
         {
            // if in Active Call shall be terminated
            ETG_TRACE_USR4(("PhoneHall::HandleSWCPickupKeyPress - SWC short press on Intercom Conference call should not have any reaction."));
            retval = true;
         }
#endif
         //context switch to speech code snippet is removed by Vidya as per Ramesh inputs
         // because Speech should be triggered by master and phone component should not handle any key press wrt speech
      }
      else if (Enum_hmibase_HARDKEYSTATE_LONG1 == hardkeyState)
      {
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
         if ((!_isIntercomIncomingCallPopupActive) && (!_isIntercomCallDialingInProgress) && (!_isThereAnActiveIntercomCall) && (!_isThereConfIntercomCall))
         {
#endif
            // No phone connected trigger BT pairing scene
            ETG_TRACE_USR4(("PhoneHall::HandleSWCPickupKeyPress - No Phone is connected to Head unit, so switching to Add Device as BT VR is not fessible"));
            sendRequestForContextSwitching(SWITCH_TO_ADD_DEVICE);
            retval = true;
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
         }
         else
         {
            // Invalid Key state
            ETG_TRACE_USR4(("PhoneHall::HandleSWCPickupKeyPress -- Invalid Key State. "));
         }
#endif
      }
      else
      {
         // Invalid Key state
         ETG_TRACE_USR4(("PhoneHall::HandleSWCPickupKeyPress -- Invalid Key State. "));
      }
   }
   else
   {
      //default case
      ETG_TRACE_USR4(("PhoneHall::HandleSWCPickupKeyPress -- No Requirement to Handle. Hence return false "));
      retval = false;
   }

   return retval;
}


/************************************************************************
*FUNCTION		: HandleSWCRejectKeyPress
*DESCRIPTION	: Member function to handle Steering Wheel Control Reject button
*PARAMETER		: Hard Key State - Up (release), long (long press)
*RETURNVALUE	: void
*SWFL			: SYSFL-6424, SYSFL-6423, SYSFL-5683, SYSFL-5685, SYSFL-5684
************************************************************************/
bool PhoneHall::HandleSWCRejectKeyPress(const int hardkeyState)
{
   bool retval = false;
   ETG_TRACE_USR4(("PhoneHall::HandleSWCRejectKeyPress "));

   if (_ServiceBT->GetNoOfHFPDevicesConnected() > 0)
   {
      if (Enum_hmibase_HARDKEYSTATE_UP == hardkeyState)
      {
         if (_isIncomingCallPopupActive)
         {
            POST_MSG((COURIER_MESSAGE_NEW(IncomingCallResponseReq)(BTPHN_EN_INCOMINGCALL_RESPONSE_REJECT)));
            ETG_TRACE_USR4(("PhoneHall::HandleSWCRejectKeyPress - Reject the incoming Call via SWC Reject "));
            retval = true;
         }
         else if (_isCallDialingInProgress)
         {
            POST_MSG((COURIER_MESSAGE_NEW(CancelDialCallReq)()))
            ETG_TRACE_USR4(("PhoneHall::HandleSWCRejectKeyPress - Cancel the outgoing Call via SWC Reject "));
            retval = true;
         }
         else if (_isThereAnActiveCall)
         {
            // if in Active Call shall be terminated
            POST_MSG((COURIER_MESSAGE_NEW(TerminateCallReq)()))
            ETG_TRACE_USR4(("PhoneHall::HandleSWCRejectKeyPress - Terminate the active Call via SWC Reject "));
            retval = true;
         }
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
         else if (_isIntercomIncomingCallPopupActive)
         {
            POST_MSG((COURIER_MESSAGE_NEW(IntercomCallResponseReq)(INTERCOMCALL_RESPONSE_REJECT)));
            ETG_TRACE_USR4(("PhoneHall::HandleSWCRejectKeyPress - Reject the Intercom incoming Call via SWC Reject "));
            retval = true;
         }
         else if (_isIntercomCallDialingInProgress)
         {
            POST_MSG((COURIER_MESSAGE_NEW(IntercomCallResponseReq)(INTERCOMCALL_RESPONSE_REJECT)));
            ETG_TRACE_USR4(("PhoneHall::HandleSWCRejectKeyPress - Cancel the Intercom outgoing Call via SWC Reject "));
            retval = true;
         }
         else if (_isThereAnActiveIntercomCall)
         {
            POST_MSG((COURIER_MESSAGE_NEW(IntercomCallResponseReq)(INTERCOMCALL_RESPONSE_REJECT)));
            ETG_TRACE_USR4(("PhoneHall::HandleSWCRejectKeyPress - Terminate the active Call via SWC Reject "));
            retval = true;
         }
         else if (_isThereConfIntercomCall)
         {
            POST_MSG((COURIER_MESSAGE_NEW(IntercomCallResponseReq)(INTERCOMCALL_RESPONSE_CONF_REJECT)));
            ETG_TRACE_USR4(("PhoneHall::HandleSWCRejectKeyPress - Terminate the conf Call via SWC Reject "));
            retval = true;
         }
#endif
         else
         {
            // Connected device is IDLE and SWC reject short press
            ETG_TRACE_USR4(("PhoneHall::HandleSWCRejectKeyPress - No requirement defined for Short press of SWC Reject when connected phone is IDLE."));
         }
      }
      else if (Enum_hmibase_HARDKEYSTATE_LONG1 == hardkeyState)
      {
         // No Long press scenario
         ETG_TRACE_USR4(("PhoneHall::HandleSWCRejectKeyPress - No requirement defined for Long press of SWC Reject when there is connected phone is Idle."));
      }
      else
      {
         // Invalid Key state
         ETG_TRACE_USR4(("PhoneHall::HandleSWCRejectKeyPress -- Invalid Key State. "));
      }
   }
   else if (_ServiceBT->GetNoOfHFPDevicesConnected() == 0)
   {
      //intercom to be handled
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT

      if (_isIntercomIncomingCallPopupActive)
      {
         POST_MSG((COURIER_MESSAGE_NEW(IntercomCallResponseReq)(INTERCOMCALL_RESPONSE_REJECT)));
         ETG_TRACE_USR4(("PhoneHall::HandleSWCRejectKeyPress - Reject the Intercom incoming Call via SWC Reject "));
         retval = true;
      }
      else if (_isIntercomCallDialingInProgress)
      {
         POST_MSG((COURIER_MESSAGE_NEW(IntercomCallResponseReq)(INTERCOMCALL_RESPONSE_REJECT)));
         ETG_TRACE_USR4(("PhoneHall::HandleSWCRejectKeyPress - Cancel the Intercom outgoing Call via SWC Reject "));
         retval = true;
      }
      else if (_isThereAnActiveIntercomCall)
      {
         POST_MSG((COURIER_MESSAGE_NEW(IntercomCallResponseReq)(INTERCOMCALL_RESPONSE_REJECT)));
         ETG_TRACE_USR4(("PhoneHall::HandleSWCRejectKeyPress - Terminate the active Call via SWC Reject "));
         retval = true;
      }
      else if (_isThereConfIntercomCall)
      {
         POST_MSG((COURIER_MESSAGE_NEW(IntercomCallResponseReq)(INTERCOMCALL_RESPONSE_CONF_REJECT)));
         ETG_TRACE_USR4(("PhoneHall::HandleSWCRejectKeyPress - Terminate the conf Call via SWC Reject "));
         retval = true;
      }
#endif

      // No Device is connected - MCP long press reject
      ETG_TRACE_USR4(("PhoneHall::HandleSWCRejectKeyPress - No requirement defined for Long press of SWC Reject when no Device is connected."));
   }
   else
   {
      // other scenario
      ETG_TRACE_USR4(("PhoneHall::HandleSWCRejectKeyPress - Invalid scenario."));
   }
   return retval;
}


/************************************************************************
*FUNCTION		: HandleMCPAcceptKeyPress
*DESCRIPTION	: Member function to handle Media control Panel Accept button
*PARAMETER		: Hard Key State - Up (release), long (long press)
*RETURNVALUE	: void
*SWFL			:
************************************************************************/
bool PhoneHall::HandleMCPAcceptKeyPress(const int hardkeyState)
{
   bool retval = false;
   ETG_TRACE_USR4(("PhoneHall::HandleMCPAcceptKeyPress "));
   if (_ServiceBT->GetNoOfHFPDevicesConnected() > 0)
   {
      if (Enum_hmibase_HARDKEYSTATE_UP == hardkeyState)
      {
         if (_isIncomingCallPopupActive)
         {
            POST_MSG((COURIER_MESSAGE_NEW(IncomingCallResponseReq)(BTPHN_EN_INCOMINGCALL_RESPONSE_ACCEPT)));
            ETG_TRACE_USR4(("PhoneHall::HandleMCPAcceptKeyPress -- Accept the incoming Call via MCP pickup "));
            retval = true;
         }
         else if (_isCallDialingInProgress)
         {
            // if it is an outgoing Call - no reaction for pick up key
            ETG_TRACE_USR4(("PhoneHall::HandleMCPAcceptKeyPress - MCP short press on outgoing call should not have any reaction."));
            retval = true;
         }
         else if (_isThereAnActiveCall)
         {
            // if in Active Call shall be terminated
            ETG_TRACE_USR4(("PhoneHall::HandleMCPAcceptKeyPress - MCP press on Active call should not have any reaction."));
            retval = true;
         }
         // DONT RE-ORDER THIS IF-ELSE. Only after comfirming that atleast one phone is connected,
         // we are trying to trigger BT-VR.
         // The Case of SPI connected via Android auto is already eliminated before this function call.
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
         else if (_isIntercomIncomingCallPopupActive)
         {
            POST_MSG((COURIER_MESSAGE_NEW(IntercomIncomingCallHardKeyResp)()));
            ETG_TRACE_USR4(("PhoneHall::HandleMCPAcceptKeyPress - Accept the Intercom incoming Call via MCP pickup "));
            retval = true;
         }
         else if (_isIntercomCallDialingInProgress)
         {
            // if it is an outgoing Call - no reaction for pick up key
            ETG_TRACE_USR4(("PhoneHall::HandleMCPAcceptKeyPress - MCP short press on Intercom outgoing call should not have any reaction."));
            retval = true;
         }
         else if (_isThereAnActiveIntercomCall)
         {
            // if in Active Call shall be terminated
            ETG_TRACE_USR4(("PhoneHall::HandleMCPAcceptKeyPress - MCP short press on Intercom Active call should not have any reaction."));
            retval = true;
         }
         else if (_isThereConfIntercomCall)
         {
            // if in Active Call shall be terminated
            ETG_TRACE_USR4(("PhoneHall::HandleMCPAcceptKeyPress - MCP short press on Intercom Conference call should not have any reaction."));
            retval = true;
         }
#endif
         else // if (_ServiceBT->GetNoOfHFPDevicesConnected() > 0)
         {
            //context switch to speech code snippet is removed by Vidya as per Ramesh inputs
            // because Speech should be triggered by master and phone component should not handle any key press wrt speech

            // Having a valid HFP connection without any calls (in/out/active) - phone on Foreground or background
            // And also make a BTVR request only when RVC is not active
            if ((bBTVRViewShown == TRUE) && (_applicationSwitchClientHandler->getCurrentActivityID() != enActivityIDs__eActivityID_RVC))
            {
               ETG_TRACE_USR4(("PhoneHall::HandleSWCShortPress -- Request for Smartphone VR "));
               sendRequestForBTVRFunctionHandling();
            }
         }
      }
      else if (Enum_hmibase_HARDKEYSTATE_LONG1 == hardkeyState)
      {
         if ((!_isIncomingCallPopupActive) && (!_isThereAnActiveCall) && (!_isCallDialingInProgress))
         {
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
            if ((!_isIntercomIncomingCallPopupActive) && (!_isIntercomCallDialingInProgress) && (!_isThereAnActiveIntercomCall) && (!_isThereConfIntercomCall))
            {
#endif
               // Having a valid HFP connection without any calls (in/out/active)
               // And also make a BTVR request only when RVC is not active
               ETG_TRACE_USR4(("PhoneHall::HandleSWCLongPress:bBTVRViewShown =%d ", bBTVRViewShown));
               if (_applicationSwitchClientHandler->getCurrentActivityID() != enActivityIDs__eActivityID_RVC)
               {
                  /*AKM2cob: For requirement like Long press SWC -> Terminate BTVR in SPI is Active */
                  if (bBTVRViewShown == TRUE)
                  {
                     POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(BTVRSessionExitAppReqMsg)()));
                  }
                  else
                  {
                     ETG_TRACE_USR4(("PhoneHall::HandleSWCLongPress -- Request for external VR - Phone is in Foreground or background "));
                     sendRequestForBTVRFunctionHandling();
                  }
               }
               retval = true;
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
            }
            else
            {
               ETG_TRACE_USR4(("PhoneHall::HandleMCPAcceptKeyPress -- No reaction for long press when there is incoming/ outgoing/ Active call. "));
               retval = true;
            }
#endif
         }
         else
         {
            ETG_TRACE_USR4(("PhoneHall::HandleMCPAcceptKeyPress -- No reaction for long press when there is incoming/ outgoing/ Active call. "));
            retval = true;
         }
      }
      else
      {
         // Invalid Key state
         ETG_TRACE_USR4(("PhoneHall::HandleMCPAcceptKeyPress -- Invalid Key State. "));
      }
   }
   else if (_ServiceBT->GetNoOfHFPDevicesConnected() == 0)
   {
      ETG_TRACE_USR4(("PhoneHall::HandleMCPAcceptKeyPress no device connected "));
      if (Enum_hmibase_HARDKEYSTATE_UP == hardkeyState)
      {
         ETG_TRACE_USR4(("PhoneHall::HandleMCPAcceptKeyPress Enum_hmibase_HARDKEYSTATE_UP "));
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
         if (_isIntercomIncomingCallPopupActive)
         {
            POST_MSG((COURIER_MESSAGE_NEW(IntercomIncomingCallHardKeyResp)()));
            ETG_TRACE_USR4(("PhoneHall::HandleMCPAcceptKeyPress - Accept the Intercom incoming Call via MCP pickup "));
            retval = true;
         }
         else if (_isIntercomCallDialingInProgress)
         {
            // if it is an outgoing Call - no reaction for pick up key
            ETG_TRACE_USR4(("PhoneHall::HandleMCPAcceptKeyPress - MCP short press on Intercom outgoing call should not have any reaction."));
            retval = true;
         }
         else if (_isThereAnActiveIntercomCall)
         {
            // if in Active Call shall be terminated
            ETG_TRACE_USR4(("PhoneHall::HandleMCPAcceptKeyPress - MCP short press on Intercom Active call should not have any reaction."));
            retval = true;
         }
         else if (_isThereConfIntercomCall)
         {
            // if in Active Call shall be terminated
            ETG_TRACE_USR4(("PhoneHall::HandleMCPAcceptKeyPress - MCP short press on Intercom Conference call should not have any reaction."));
            retval = true;
         }
#endif


         //context switch to speech code snippet is removed by Vidya as per Ramesh inputs
         // because Speech should be triggered by master and phone component should not handle any key press wrt speech
      }
      else if (Enum_hmibase_HARDKEYSTATE_LONG1 == hardkeyState)
      {
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
         if ((!_isIntercomIncomingCallPopupActive) && (!_isIntercomCallDialingInProgress) && (!_isThereAnActiveIntercomCall) && (!_isThereConfIntercomCall))
         {
#endif
            // No phone connected trigger BT pairing scene
            ETG_TRACE_USR4(("PhoneHall::HandleMCPAcceptKeyPress - No Phone is connected to Head unit, so switching to Add Device as BT VR is not fessible"));
            sendRequestForContextSwitching(SWITCH_TO_ADD_DEVICE);
            retval = true;
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
         }
         else
         {
            // Invalid Key state
            ETG_TRACE_USR4(("PhoneHall::HandleMCPAcceptKeyPress -- Invalid Key State. "));
         }
#endif
      }
      else
      {
         // Invalid Key state
         ETG_TRACE_USR4(("PhoneHall::HandleMCPAcceptKeyPress -- Invalid Key State. "));
      }
   }
   else
   {
      //default case
      ETG_TRACE_USR4(("PhoneHall::HandleMCPAcceptKeyPress -- No Requirement to Handle. Hence return false "));
   }

   return retval;
}


/************************************************************************
*FUNCTION		: HandleMCPRejectKeyPress
*DESCRIPTION	: Member function to handle Media control Panel Reject button
*PARAMETER		: Hard Key State - Up (release), long (long press)
*RETURNVALUE	: void
*SWFL			:
************************************************************************/
bool PhoneHall::HandleMCPRejectKeyPress(const int hardkeyState)
{
   bool retval = false;
   ETG_TRACE_USR4(("PhoneHall::HandleMCPRejectKeyPress "));

   if (_ServiceBT->GetNoOfHFPDevicesConnected() > 0)
   {
      if (Enum_hmibase_HARDKEYSTATE_UP == hardkeyState)
      {
         if (_isIncomingCallPopupActive)
         {
            POST_MSG((COURIER_MESSAGE_NEW(IncomingCallResponseReq)(BTPHN_EN_INCOMINGCALL_RESPONSE_REJECT)));
            ETG_TRACE_USR4(("PhoneHall::HandleMCPRejectKeyPress - Reject the incoming Call via MCP Reject "));
            retval = true;
         }
         else if (_isCallDialingInProgress)
         {
            POST_MSG((COURIER_MESSAGE_NEW(CancelDialCallReq)()));
            ETG_TRACE_USR4(("PhoneHall::HandleMCPRejectKeyPress - CancelDialCallReq the outgoing Call via MCP Reject "));
            retval = true;
         }
         else if (_isThereAnActiveCall)
         {
            // if in Active Call shall be terminated
            POST_MSG((COURIER_MESSAGE_NEW(TerminateCallReq)()));
            ETG_TRACE_USR4(("PhoneHall::HandleMCPRejectKeyPress - TerminateCallReq the active Call via MCP Reject "));
            retval = true;
         }

#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
         else if (_isIntercomIncomingCallPopupActive)
         {
            POST_MSG((COURIER_MESSAGE_NEW(IntercomCallResponseReq)(INTERCOMCALL_RESPONSE_REJECT)));
            ETG_TRACE_USR4(("PhoneHall::HandleMCPRejectKeyPress - Accept the Intercom incoming Call via MCP Reject "));
            retval = true;
         }
         else if (_isIntercomCallDialingInProgress)
         {
            POST_MSG((COURIER_MESSAGE_NEW(IntercomCallResponseReq)(INTERCOMCALL_RESPONSE_REJECT)));
            ETG_TRACE_USR4(("PhoneHall::HandleMCPRejectKeyPress - CancelDialCallReq the Intercom outgoing Call via MCP Reject"));
            retval = true;
         }
         else if (_isThereAnActiveIntercomCall)
         {
            // if in Active Call shall be terminated
            POST_MSG((COURIER_MESSAGE_NEW(IntercomCallResponseReq)(INTERCOMCALL_RESPONSE_REJECT)));
            ETG_TRACE_USR4(("PhoneHall::HandleMCPRejectKeyPress - TerminateCallReq the Intercom active Call via MCP Reject"));
            retval = true;
         }
         else if (_isThereConfIntercomCall)
         {
            POST_MSG((COURIER_MESSAGE_NEW(IntercomCallResponseReq)(INTERCOMCALL_RESPONSE_CONF_REJECT)));
            ETG_TRACE_USR4(("PhoneHall::HandleMCPRejectKeyPress - Terminate the conf Call via MCP Reject "));
            retval = true;
         }
#endif


         else
         {
            //intercom to be handled
            //context switch to speech code snippet is removed by Vidya as per Ramesh inputs
            // because Speech should be triggered by master and phone component should not handle any key press wrt speech
            // Connected device is IDLE and MCP reject short press
            ETG_TRACE_USR4(("PhoneHall::HandleMCPRejectKeyPress - No requirement defined for Short press of MCP Reject when connected phone is IDLE."));
         }
      }
      else if (Enum_hmibase_HARDKEYSTATE_LONG1 == hardkeyState)
      {
         // No Long press scenario
         ETG_TRACE_USR4(("PhoneHall::HandleMCPRejectKeyPress - No requirement defined for Long press of MCP Reject when there is connected phone is Idle."));
      }
      else
      {
         // Invalid Key state
         ETG_TRACE_USR4(("PhoneHall::HandleMCPRejectKeyPress -- Invalid Key State. "));
      }
   }
   else if (_ServiceBT->GetNoOfHFPDevicesConnected() == 0)
   {
      if (Enum_hmibase_HARDKEYSTATE_UP == hardkeyState)
      {
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
         if (_isIntercomIncomingCallPopupActive)
         {
            POST_MSG((COURIER_MESSAGE_NEW(IntercomCallResponseReq)(INTERCOMCALL_RESPONSE_REJECT)));
            ETG_TRACE_USR4(("PhoneHall::HandleMCPRejectKeyPress - Accept the Intercom incoming Call via MCP Reject "));
            retval = true;
         }
         else if (_isIntercomCallDialingInProgress)
         {
            POST_MSG((COURIER_MESSAGE_NEW(IntercomCallResponseReq)(INTERCOMCALL_RESPONSE_REJECT)));
            ETG_TRACE_USR4(("PhoneHall::HandleMCPRejectKeyPress - CancelDialCallReq the Intercom outgoing Call via MCP Reject"));
            retval = true;
         }
         else if (_isThereAnActiveIntercomCall)
         {
            // if in Active Call shall be terminated
            POST_MSG((COURIER_MESSAGE_NEW(IntercomCallResponseReq)(INTERCOMCALL_RESPONSE_REJECT)));
            ETG_TRACE_USR4(("PhoneHall::HandleMCPRejectKeyPress - TerminateCallReq the Intercom active Call via MCP Reject"));
            retval = true;
         }
         else if (_isThereConfIntercomCall)
         {
            POST_MSG((COURIER_MESSAGE_NEW(IntercomCallResponseReq)(INTERCOMCALL_RESPONSE_CONF_REJECT)));
            ETG_TRACE_USR4(("PhoneHall::HandleMCPRejectKeyPress - Terminate the conf Call via MCP Reject "));
            retval = true;
         }
#endif
      }
      else if (Enum_hmibase_HARDKEYSTATE_LONG1 == hardkeyState)
      {
         // No Device is connected - MCP long press reject
         ETG_TRACE_USR4(("PhoneHall::HandleMCPRejectKeyPress - No requirement defined for Long press of MCP Reject when no Device is connected."));
      }
      else
      {
         // Invalid Key state
         ETG_TRACE_USR4(("PhoneHall::HandleMCPRejectKeyPress - No Device Connected -- Invalid Key State. "));
      }
   }
   else
   {
      // other scenario
      ETG_TRACE_USR4(("PhoneHall::HandleMCPRejectKeyPress - Invalid Key State."));
   }
   return retval;
}


/************************************************************************
*FUNCTION		: HandleSWCShortPress
*DESCRIPTION	: Member function to handle Steering Wheel Control Pick up button
*PARAMETER		: Hard Key State - Up (short release),
*RETURNVALUE	: void
*SWFL			:
************************************************************************/
bool PhoneHall::HandleSWCShortPress(const int hardkeyState)
{
   bool retval = false;
   ETG_TRACE_USR4(("PhoneHall::HandleSWCShortPress (PTT)"));

   if (_ServiceBT->GetNoOfHFPDevicesConnected() > 0)
   {
      if (_isIncomingCallPopupActive)
      {
         // Accept incoming call on short press
         POST_MSG((COURIER_MESSAGE_NEW(IncomingCallResponseReq)(BTPHN_EN_INCOMINGCALL_RESPONSE_ACCEPT)));
         ETG_TRACE_USR4(("Accept the incoming Call via SWC pickup "));
         retval = true;
      }
      else if (_isCallDialingInProgress)
      {
         // Cancel outgoing call on short press
         POST_MSG((COURIER_MESSAGE_NEW(CancelDialCallReq)()))
         ETG_TRACE_USR4(("PhoneHall::HandleSWCShortPress - Reject the outgoing Call via SWC Pickup "));
         retval = true;
      }
      else if (_isThereAnActiveCall)
      {
         // if in Active Call shall be terminated
         POST_MSG((COURIER_MESSAGE_NEW(TerminateCallReq)()))
         ETG_TRACE_USR4(("PhoneHall::HandleSWCShortPress - Reject the active Call via SWC Pickup "));
         retval = true;
      }
      // DONT RE-ORDER THIS IF ELSE, only after comfirmating that atleast one phone is connected,
      // we are trying to trigger BT-VR.
      // The Case of SPI connected via Android auto is already eliminated before this function call.
      else // if (_ServiceBT->GetNoOfHFPDevicesConnected() > 0)
      {
         // Having a valid HFP connection without any calls (in/out/active) - phone on Foreground or background
         // And also make a BTVR request only when RVC is not active
         if (_applicationSwitchClientHandler->getCurrentActivityID() != enActivityIDs__eActivityID_RVC)
         {
            ETG_TRACE_USR4(("PhoneHall::HandleSWCShortPress -- Request for Smartphone VR "));
            sendRequestForBTVRFunctionHandling();
         }
         retval = true;
      }
   }
   else if (_ServiceBT->GetNoOfHFPDevicesConnected() == 0)
   {
      // No phone connected trigger BT pairing scene
      ETG_TRACE_USR4(("PhoneHall::HandleSWCShortPress - No Phone is connected to Head unit - Add Device "));
      sendRequestForContextSwitching(SWITCH_TO_ADD_DEVICE);
      retval = true;
   }
   else
   {
      //default case
      ETG_TRACE_USR4(("PhoneHall::HandleSWCShortPress -- No Requirement to Handle. Hence return false "));
      retval = false;
   }

   return retval;
}


/************************************************************************
*FUNCTION		: HandleSWCLongPress
*DESCRIPTION	: Member function to handle Steering Wheel Control Reject button
*PARAMETER		: Hard Key State -long (long press)
*RETURNVALUE	: void
*SWFL			:
************************************************************************/
bool PhoneHall::HandleSWCLongPress(const int hardkeyState)
{
   bool retval = false;
   ETG_TRACE_USR4(("PhoneHall::HandleSWCLongPress "));

   if (_ServiceBT->GetNoOfHFPDevicesConnected() > 0)
   {
      if (_isIncomingCallPopupActive)
      {
         // Reject incoming call on long press
         POST_MSG((COURIER_MESSAGE_NEW(IncomingCallResponseReq)(BTPHN_EN_INCOMINGCALL_RESPONSE_REJECT)));
         ETG_TRACE_USR4(("PhoneHall::HandleSWCLongPress - Reject the incoming Call via SWC Reject "));
         retval = true;
      }
      else if (_isCallDialingInProgress)
      {
         // Cancel outgoing call on long press
         POST_MSG((COURIER_MESSAGE_NEW(CancelDialCallReq)()))
         ETG_TRACE_USR4(("PhoneHall::HandleSWCLongPress - Reject the outgoing Call via SWC Reject "));
         retval = true;
      }
      else if (_isThereAnActiveCall)
      {
         // if in Active Call shall be terminated
         POST_MSG((COURIER_MESSAGE_NEW(TerminateCallReq)()))
         ETG_TRACE_USR4(("PhoneHall::HandleSWCLongPress - Reject the active Call via SWC Reject "));
         retval = true;
      }
      // DONT RE-ORDER THIS IF ELSE, only after comfirmating that atleast one phone is connected,
      // we are trying to trigger BT-VR.
      // The Case of SPI connected via Android auto is already eliminated before this function call.
      else // (_ServiceBT->GetNoOfHFPDevicesConnected() > 0)
      {
         // And also make a BTVR request only when RVC is not active
         ETG_TRACE_USR4(("PhoneHall::HandleSWCLongPress:bBTVRViewShown =%d ", bBTVRViewShown));
         if (_applicationSwitchClientHandler->getCurrentActivityID() != enActivityIDs__eActivityID_RVC)
         {
            /*AKM2cob: For requirement like Long press SWC -> Terminate BTVR in SPI is Active */
            if (bBTVRViewShown == TRUE)
            {
               POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(BTVRSessionExitAppReqMsg)()));
            }
            else
            {
               ETG_TRACE_USR4(("PhoneHall::HandleSWCLongPress -- Request for external VR - Phone is in Foreground or background "));
               sendRequestForBTVRFunctionHandling();
            }
         }
         retval = true;
      }
   }
   else if (_ServiceBT->GetNoOfHFPDevicesConnected() == 0)
   {
      // No Phone is connected Trigger BT Pairing scene
      ETG_TRACE_USR4(("PhoneHall::HandleSWCLongPress - No Phone is connected to Head unit - Add Device "));
      sendRequestForContextSwitching(SWITCH_TO_ADD_DEVICE);
      retval = true;
   }
   else
   {
      //default case
      ETG_TRACE_USR4(("PhoneHall::HandleSWCLongPress -- No Requirement to Handle. Hence return false "));
      retval = false;
   }

   return retval;
}


/************************************************************************
*FUNCTION		: on android auto/carplay session msg from SmartPhoneClient
*DESCRIPTION	: Handling of phone handle on android auto session change
*PARAMETER		: NA
*RETURNVALUE	: bool
*SWFL			:
************************************************************************/
bool PhoneHall::onCourierMessage(const SpiSessionMsg& msg)
{
   ETG_TRACE_USR4(("PhoneHall::onCourierMessage(SpiSessionMsg)"));
   /* If android auto session or carplay is active, update the value of 3 as to show phone not connected icon else show respective device handle icon */
   if ((g_SM_SpiSessionStatus == SPI_SESSION_ANDROID_AUTO_ACTIVE) || (g_SM_SpiSessionStatus == SPI_SESSION_CARPLAY_ACTIVE))
   {
      (*_phoneGadgetInfoData).mPhoneHandle = 3;
   }
   else
   {
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
      (*_phoneGadgetInfoData).mPhoneHandle = 5; //m_u8PhoneCurrentHandle;

#else
      (*_phoneGadgetInfoData).mPhoneHandle = m_u8PhoneCurrentHandle;
#endif
   }

   _phoneGadgetInfoData.MarkAllItemsModified();
   _phoneGadgetInfoData.SendUpdate(true);
   (*_PhoneButtonText).mSource = Candera::String(TextId_0x13A5);
   _PhoneButtonText.MarkAllItemsModified();
   _PhoneButtonText.SendUpdate(true);

   return false;
}


/************************************************************************
*FUNCTION		: SendContextSwitchRequestForVR
*DESCRIPTION	: Member function to to trigger COntext Switch to VR (both internal and SPI)
*PARAMETER		: bool - is the key press is long press or not
*RETURNVALUE	: void
*SWFL			:
************************************************************************/
void PhoneHall::SendContextSwitchRequestForVR(bool bIsLongKeyPress)
{
   if (false == bIsLongKeyPress)
   {
      ETG_TRACE_USR4(("Context Switch request to Internal VR "));
      // context switch to internal VR
      sendRequestForContextSwitching(SWITCH_TO_SPEECH);
   }
   else
   {
      ETG_TRACE_USR4(("Context Switch request to SPI VR "));
      // context switch to External VR
      sendRequestForBTVRFunctionHandling();
   }
}


/************************************************************************
*FUNCTION		: on TriggerSceneTransitionMsg
*DESCRIPTION	: to handle the context switching for BTVR
*PARAMETER		: SceneIndex, SceneEnableValue
*RETURNVALUE	: bool
*SWFL			: NA
************************************************************************/
bool PhoneHall::onCourierMessage(const TriggerSceneTransitionMsg& msg)
{
   ETG_TRACE_USR4(("PhoneHall::onCourierMessage(TriggerSceneTransitionMsg)"));

   bool retVal = false;
   int requestSceneIndex = msg.GetRequestSceneIndex();
   bool SceneEnableStatus = msg.GetSceneEnableStatus();

   //Enable BTVR scene on enable request
   if ((requestSceneIndex == SWITCH_TO_BTVR) && (SceneEnableStatus == true))
   {
      if (_isPhonceContextActive == true)
      {
         /* Creating and posting BTVRStatusResp courier message */
         ETG_TRACE_USR4(("PhoneHall TriggerSceneTransitionMsg: Sending BTVRStatusResp message TRUE"));
         POST_MSG((COURIER_MESSAGE_NEW(BTVRStatusResp)(true)));
      }
      else
      {
         sendRequestForContextSwitching(SWITCH_TO_BTVR);
      }
   }
   //Disable BTVR scene on disable request
   else if ((requestSceneIndex == SWITCH_TO_BTVR) && (SceneEnableStatus == false))
   {
      if (_applicationSwitchClientHandler->GetPhoneContextStatus() == EN_PHONE_CONTEXT_BG)
      {
         sendRequestForContextSwitching(SWITCH_TO_CALL_BTVR_EXIT);
      }
      else
      {
         /* Creating and posting BTVRStatusResp courier message */
         ETG_TRACE_USR4(("PhoneHall TriggerSceneTransitionMsg: Sending BTVRStatusResp message FALSE"));
         POST_MSG((COURIER_MESSAGE_NEW(BTVRStatusResp)(false)));
      }
   }
   else
   {
   }

   return true;
}


/************************************************************************
*FUNCTION		: on ButtonReactionMsg From Master Cobsumer Gadget
*DESCRIPTION	: to handle the context switching in Phone Gadget
*PARAMETER		: NA
*RETURNVALUE	: bool
*SWFL			: [SWFL-4347]
************************************************************************/
bool PhoneHall::onCourierMessage(const PhoneGadgetRequestMsg& msg)
{
   ETG_TRACE_USR4(("PhoneHall::onCourierMessage(PhoneGadgetRequestMsg)"));

   bool retVal = false;
   int requestType = msg.GetRequestType();
   bool IsBTCallActive = false;
   /*   bool IsPhoneIntercomCallActive = false;
   #ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
     if ((true == IsIntercomCallActive) || (true == IntercomClienthandler::poGetInstance()->getIntercomIncomingCallStatus()) || (true == IntercomClienthandler::poGetInstance()->getIntercomCallDialingStatus()))
     {
        ETG_TRACE_USR4(("PhoneHall::Intercom Call is active hence toggle shouldnt be processed"));
        IsPhoneIntercomCallActive = true;
     }
   #endif */
   if ((true == ServiceTel::poGetInstance()->getIncomingCallStatus()) || (true == ServiceTel::poGetInstance()->getCallDialingStatus()) || (true == ServiceTel::poGetInstance()->getOngoingCallStatus()))
   {
      ETG_TRACE_USR4(("PhoneHall::BT Call is active hence toggle shouldnt be processed"));
      IsBTCallActive = true;
   }

   if ((TOGGLE_PHONE_HANDLE == requestType)) /* && (false == IsPhoneIntercomCallActive)) */
   {
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
      Sm_ActiveSource = 0;

      (*_Intercomenable).mIntercomenable = true;
      _Intercomenable.MarkAllItemsModified();
      _Intercomenable.SendUpdate(true);

      if (_ServiceBT->GetBluetoothOnOffStatus() == false)
      {
         if (0 == Sm_ActiveSource)
         {
            ETG_TRACE_USR4(("PhoneHall::onCourierMessage(PhoneGadgetRequestMsg) :: Active Source is BT now so BT Tile Disable"));
            DeviceListModel::poGetInstance()->OnUpdBTONOFFStatus(false);
         }
      }
      else
      {
         if (0 == Sm_ActiveSource)
         {
            ETG_TRACE_USR4(("PhoneHall::onCourierMessage(PhoneGadgetRequestMsg) :: Active Source is BT now so BT Tile Enable"));
            DeviceListModel::poGetInstance()->OnUpdBTONOFFStatus(true);
         }
      }

      ETG_TRACE_USR4(("PhoneHall::onCourierMessage(PhoneGadgetRequestMsg) IntercomAvailable"));
      ETG_TRACE_USR4(("PhoneHall::onCourierMessage(PhoneGadgetRequestMsg) u8ActiveSource %d)", u8ActiveSource));
      if ((((_ServiceBT->GetNoOfHFPDevicesConnected() > 1) && (u8ActiveSource != PHONETWO) && (false == IsBTCallActive)) || (u8ActiveSource == INTERCOM)))
      {
         Sm_ActiveSource = 0;
         (*_Intercomenable).mIntercomenable = true;
         _Intercomenable.MarkAllItemsModified();
         _Intercomenable.SendUpdate(true);

         if (_ServiceBT->GetBluetoothOnOffStatus() == true)
         {
            DeviceListModel::poGetInstance()->OnUpdBTONOFFStatus(true);
         }
         else
         {
            DeviceListModel::poGetInstance()->OnUpdBTONOFFStatus(false);
         }

         ETG_TRACE_USR4(("PhoneHall::onCourierMessage(PhoneGadgetRequestMsg toogle between phone)"));
         /* If android auto session or carplay is active, update the value of 3 as to show phone not connected icon else show respective device handle icon */
         if ((g_SM_SpiSessionStatus == SPI_SESSION_ANDROID_AUTO_ACTIVE) || (g_SM_SpiSessionStatus == SPI_SESSION_CARPLAY_ACTIVE))
         {
            /* Update with value of 3 as to show phone not connected icon*/
            (*_phoneGadgetInfoData).mPhoneHandle = 3;
         }
         else
         {
            /* Update the respective device handle icon index*/
            // m_u8PhoneCurrentHandle = (0 == m_u8PhoneCurrentHandle) ? 1 : 0;
            (*_phoneGadgetInfoData).mPhoneHandle = 5;/* m_u8PhoneCurrentHandle; */
         }
         _phoneGadgetInfoData.MarkAllItemsModified();
         _phoneGadgetInfoData.SendUpdate(true);

         (*_PhoneButtonText).mSource = Candera::String(TextId_0x13A5);
         _PhoneButtonText.MarkAllItemsModified();
         _PhoneButtonText.SendUpdate(true);
         /*If android auto session or carplay is active, do not allow Phone Toggle operation*/
         if ((g_SM_SpiSessionStatus != SPI_SESSION_ANDROID_AUTO_ACTIVE) && (g_SM_SpiSessionStatus != SPI_SESSION_CARPLAY_ACTIVE))
         {
            if (false == IsBTCallActive)
            {
               _ServiceBT->vToggleActivePhone();
            }
            _ServiceTel->vUpdActivePhone();
         }
         if ((u8ActiveSource == PHONEONE) && (_ServiceBT->GetNoOfHFPDevicesConnected() > 1))
         {
            u8ActiveSource = PHONETWO;
            ETG_TRACE_USR4(("PhoneHall::onCourierMessage(PhoneGadgetRequestMsg) in if change to u8ActiveSource %d)", u8ActiveSource));
         }
         else if (u8ActiveSource == INTERCOM)
         {
            u8ActiveSource = PHONEONE;
            ETG_TRACE_USR4(("PhoneHall::onCourierMessage(PhoneGadgetRequestMsg) in else if change to u8ActiveSource %d)", u8ActiveSource));
         }
         else if (u8ActiveSource == PHONETWO)
         {
            u8ActiveSource = INTERCOM;
            ETG_TRACE_USR4(("PhoneHall::onCourierMessage(PhoneGadgetRequestMsg) in else if change to u8ActiveSource %d)", u8ActiveSource));
         }
         ETG_TRACE_USR4(("PhoneHall::onCourierMessage(PhoneGadgetRequestMsg) u8ActiveSource %d)", u8ActiveSource));
      }
      else
      {
         Sm_ActiveSource = 1;
         (*_Intercomenable).mIntercomenable = false;
         _Intercomenable.MarkAllItemsModified();
         _Intercomenable.SendUpdate(true);
         if (_ServiceBT->GetBluetoothOnOffStatus() == false)
         {
            DeviceListModel::poGetInstance()->OnUpdBTONOFFStatus(true);
         }
         ETG_TRACE_USR4(("PhoneHall::onCourierMessage(PhoneGadgetRequestMsg) inside intercom u8ActiveSource %d)", u8ActiveSource));
         u8ActiveSource = INTERCOM;
         IntercomModel::poGetInstance()->vSetIntercomActive();
         (*_phoneGadgetInfoData).mPhoneHandle = 4;
         _phoneGadgetInfoData.MarkAllItemsModified();
         _phoneGadgetInfoData.SendUpdate(true);

         (*_PhoneButtonText).mSource = Candera::String(TextId_0x14B0); //"INTERCOM";
         _PhoneButtonText.MarkAllItemsModified();
         _PhoneButtonText.SendUpdate(true);
      }

      // Ensure Phone Gadget state is correctly updated.
      PhoneGadgetModel::poGetInstance()->UpdatePhoneGadgetInfo();
#endif

#ifndef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
      ETG_TRACE_USR4(("PhoneGadgetRequestMsg - Toggle Phone Handle"));
      if ((_ServiceBT->GetNoOfHFPDevicesConnected() > 1))
      {
         /* If android auto session or carplay is active, update the value of 3 as to show phone not connected icon else show respective device handle icon */
         if ((g_SM_SpiSessionStatus == SPI_SESSION_ANDROID_AUTO_ACTIVE) || (g_SM_SpiSessionStatus == SPI_SESSION_CARPLAY_ACTIVE))
         {
            /* Update with value of 3 as to show phone not connected icon*/
            (*_phoneGadgetInfoData).mPhoneHandle = 3;
         }
         else
         {
            /* Update the respective device handle icon index*/
            m_u8PhoneCurrentHandle = (0 == m_u8PhoneCurrentHandle) ? 1 : 0;
            (*_phoneGadgetInfoData).mPhoneHandle = m_u8PhoneCurrentHandle;
         }
         _phoneGadgetInfoData.MarkAllItemsModified();
         _phoneGadgetInfoData.SendUpdate(true);

         (*_PhoneButtonText).mSource = Candera::String(TextId_0x13A5);
         _PhoneButtonText.MarkAllItemsModified();
         _PhoneButtonText.SendUpdate(true);
         /*If android auto session or carplay is active, do not allow Phone Toggle operation*/
         if ((g_SM_SpiSessionStatus != SPI_SESSION_ANDROID_AUTO_ACTIVE) && (g_SM_SpiSessionStatus != SPI_SESSION_CARPLAY_ACTIVE))
         {
            _ServiceBT->vToggleActivePhone();
         }
      }
#endif
   }
   /* When outgoing call status is active, enter into call connecting scene on activation of phone application */
   /* Call connecting scenario is handled with new courier message - RequestCallConnectingMsg*/
   /*else if ((_ServiceTel->getCallDialingStatus() == true))
    {
      retVal = sendRequestForContextSwitching(SWITCH_TO_CALL_CONNECTING);
   } */
   else if (_ServiceTel->getCallDialingStatus() == true && requestType != SWITCH_TO_ADD_DEVICE)
   {
      ETG_TRACE_USR4(("PhoneHall::PhoneGadgetRequestMsg, SWITCH_TO_CALL_CONNECTING"));
      POST_MSG((COURIER_MESSAGE_NEW(RequestCallConnectingMsg)(SWITCH_TO_CALL_CONNECTING)));
   }
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
   else if ((true == _isPhonceContextActive) && (EXIT_INTERCOMAPPLICATION == requestType))
   {
      if (Sm_ActiveSource == 1)
      {
         _applicationSwitchClientComponent->requestContextSwitchApplication(getActiveregionid(),
               (EN_ACTIVITY_IDS)0,
               (EN_ACTIVITY_IDS)0,
               APPID_APPHMI_MASTER,
               enActivityIDs__eActivityID_HOME);
      }
   }
#endif
   else
   {
      retVal = sendRequestForContextSwitching(requestType);
      ETG_TRACE_USR4(("PhoneHall : PhoneGadgetRequestMsg else"));
   }

   return retVal;
}


/************************************************************************
*FUNCTION		: sendRequestForBTVRFunctionHandling
*DESCRIPTION	: BTVR request to activate/deactivate the function
*PARAMETER		: NA
*RETURNVALUE	: void
*SWFL			:
*Rev 0.1: 15.10.2020 Palani K , BTVR changes
************************************************************************/
void PhoneHall::sendRequestForBTVRFunctionHandling()
{
   ETG_TRACE_USR4(("PhoneHall :: sendRequestForBTVRFunctionHandling - Processing sendRequestForBTVRFunctionHandling "));
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
   if (MicClientHandler::poGetInstance()->getMicStatus() == false)//check mic status
   {
      ETG_TRACE_USR4(("PhoneHall::sendRequestForBTVRFunctionHandling::Mic announcement is false"));
#endif
      /* Handle BTVR request only when Android auto session and Carplay is not active */
      if ((g_SM_SpiSessionStatus != SPI_SESSION_ANDROID_AUTO_ACTIVE) && (g_SM_SpiSessionStatus != SPI_SESSION_CARPLAY_ACTIVE))
      {
         ETG_TRACE_USR4(("PhoneHall::sendRequestForBTVRFunctionHandling::BT_VR request handling"));
         /* Check for BTVR feature coding and HFP active status */
         if ((PhoneDataPoolConfig::getInstance()->isVRSupported()) && (_ServiceBT->GetNoOfHFPDevicesConnected() > 0))
         {
            ETG_TRACE_USR4(("PhoneHall::sendRequestForBTVRFunctionHandling:isVRSupported = true and HFP device is active "));

            POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(BTVRSessionStartAppReqMsg)()));
         }
      }
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
   }
   else
   {
      ETG_TRACE_USR4(("PhoneHall::sendRequestForBTVRFunctionHandling::Mic announcement is true"));
   }
#endif
}


uint8 PhoneHall::getActiveregionid()
{
   uint8 u8Activeregionid = REGION_INVALID;
   if (_applicationSwitchClientHandler != NULL)
   {
      u8Activeregionid = _applicationSwitchClientHandler->getActiveregionid();
      ETG_TRACE_USR4(("PhoneHall::GetActiveregionid = %d ", u8Activeregionid));
   }

   return u8Activeregionid ;
}


/************************************************************************
*FUNCTION		: sendRequestForContextSwitching
*DESCRIPTION	: Actual member function to send the request to context switch
*PARAMETER		: NA
*RETURNVALUE	: bool
*SWFL			: [SWFL-4347]
*31.05.2019: ebg1kor
*Rev 0.1: Initial Version , SYSFL-5561,SYSFL-5562
*Rev 0.2: 11.09.2019 Palani K , Android Auto changes
************************************************************************/
bool PhoneHall::sendRequestForContextSwitching(const int requestType)
{
   ETG_TRACE_USR4(("sendRequestForContextSwitching - Processing sendRequestForContextSwitching.. "));
   bool isPhoneReqMade = false;
//Check both active phone context status and the RVC Status flag which updates whether phone context was in foreground if RVC is currently active.
   if ((_isPhonceContextActive == false) && (_applicationSwitchClientHandler->GetPhoneContextRVCStatus() == false))
   {
#ifdef VARIANT_S_FTR_ENABLE_BLUETOOTH_ON_OFF
      if (_ServiceBT->GetBluetoothOnOffStatus())
      {
#endif
         if (!_isCabinOnly)
         {
            switch (requestType)
            {
               case SWITCH_TO_ADD_DEVICE:
               {
                  ETG_TRACE_USR4(("sendRequestForContextSwitching - Request Application Switch to Add Device "));
                  // Context switching from Master to Add Device
                  _applicationSwitchClientComponent->requestContextSwitchApplication(getActiveregionid(),
                        APPID_APPHMI_MASTER,
                        enActivityIDs__eActivityID_HOME,
                        APPID_APPHMI_PHONE,
                        enActivityIDs__eActivityID_PHONE_DEVICE_SELECTION);
                  // isPhoneReqMade = true;
                  break;
               }
               case SWITCH_TO_CALL_LIST:
               {
                  /* When Android Auto session or CarPlay is active, switch to add device list page on receiving call list request as call list page is not applicable in this scenario */
                  if ((g_SM_SpiSessionStatus == SPI_SESSION_ANDROID_AUTO_ACTIVE) ||
                        (g_SM_SpiSessionStatus == SPI_SESSION_CARPLAY_ACTIVE)      ||
                        (_ServiceBT->GetNoOfHFPDevicesConnected() < 1))   // Fix for 707123 - AppFly in Flicker. As phone not connected, Switch directly to Phone Selection
                  {
                     ETG_TRACE_USR4(("sendRequestForContextSwitching - Request Application Switch to Add Device "));
                     // Context switching from Master to Add Device
                     _applicationSwitchClientComponent->requestContextSwitchApplication(getActiveregionid(),
                           APPID_APPHMI_MASTER,
                           enActivityIDs__eActivityID_HOME,
                           APPID_APPHMI_PHONE,
                           enActivityIDs__eActivityID_PHONE_DEVICE_SELECTION);
                  }
                  else
                  {
                     ETG_TRACE_USR4(("sendRequestForContextSwitching - Request Application Switch to Call List "));
                     // Context switching from Master to Call list
                     _applicationSwitchClientComponent->requestContextSwitchApplication(getActiveregionid(),
                           APPID_APPHMI_MASTER,
                           enActivityIDs__eActivityID_HOME,
                           APPID_APPHMI_PHONE,
                           enActivityIDs__eActivityID_PHONE_CALL_LIST);
                  }
                  // isPhoneReqMade = true;
                  break;
               }
               case SWITCH_TO_SPEECH:
               {
                  ETG_TRACE_USR4(("sendRequestForContextSwitching - Request Application Switch to Internal Voice Recognition"));
                  _applicationSwitchClientComponent->requestContextSwitchApplication(getActiveregionid(),
                        (EN_ACTIVITY_IDS)0,
                        (EN_ACTIVITY_IDS)0,
                        APPID_APPHMI_SPEECH,
                        enActivityIDs__eActivityID_SPEECH);
                  break;
               }
               case SWITCH_TO_BTVR:
               {
                  if ((g_SM_SpiSessionStatus != SPI_SESSION_ANDROID_AUTO_ACTIVE) && (g_SM_SpiSessionStatus != SPI_SESSION_CARPLAY_ACTIVE))
                  {
                     ETG_TRACE_USR4(("sendRequestForContextSwitching - Request Application Switch to Smart Phone VR"));
                     _applicationSwitchClientComponent->requestContextSwitchApplication(getActiveregionid(),
                           (EN_ACTIVITY_IDS)0,
                           (EN_ACTIVITY_IDS)0,
                           APPID_APPHMI_PHONE,
                           enActivityIDs__eActivityID_PHONE_BT_VR);
                     isPhoneReqMade = true;
                  }
                  else
                  {
                     ETG_TRACE_USR4(("sendRequestForContextSwitching - Android active and Carplay session active"));
                  }

                  break;
               }
               case SWITCH_TO_ACTIVE_PHONE:
                  if (_ServiceBT->GetNoOfHFPDevicesConnected() == 1)
                  {
                     ETG_TRACE_USR4(("sendRequestForContextSwitching - Only one phone is connected"));
                     /* When only one device is active, set the device handle index to zero */
                     m_u8PhoneCurrentHandle = 0;
                     /* If android auto session or carplay is active, update the value of 3 as to show phone not connected icon else show first device handle icon */
                     if ((g_SM_SpiSessionStatus == SPI_SESSION_ANDROID_AUTO_ACTIVE) || (g_SM_SpiSessionStatus == SPI_SESSION_CARPLAY_ACTIVE))
                     {
                        /* Update with value of 3 as to show phone not connected icon*/
                        (*_phoneGadgetInfoData).mPhoneHandle =  3;
                     }
                     else
                     {
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
                        (*_phoneGadgetInfoData).mPhoneHandle = 5; //m_u8PhoneCurrentHandle;

#else
                        (*_phoneGadgetInfoData).mPhoneHandle = m_u8PhoneCurrentHandle;
#endif
                     }
                     _phoneGadgetInfoData.MarkAllItemsModified();
                     _phoneGadgetInfoData.SendUpdate(true);

                     (*_PhoneButtonText).mSource = Candera::String(TextId_0x13A5);
                     _PhoneButtonText.MarkAllItemsModified();
                     _PhoneButtonText.SendUpdate(true);
                  }
                  break;

               case SWITCH_TO_CALL_CONNECTING:
               {
                  ETG_TRACE_USR4(("sendRequestForContextSwitching - Request Application Switch to Call Connecting"));
                  /// Context switching from Master to call connecting
                  _applicationSwitchClientComponent->requestContextSwitchApplication(getActiveregionid(),
                        (EN_ACTIVITY_IDS)0,
                        (EN_ACTIVITY_IDS)0,
                        APPID_APPHMI_PHONE,
                        enActivityIDs__eActivityID_PHONE_CALL_CONNECTING);
               }
               isPhoneReqMade = true;
               break;

               case SWITCH_TO_CALL_ACCEPT:
               {
                  ETG_TRACE_USR4(("sendRequestForContextSwitching - Request Application Switch to Call Accept"));
                  _applicationSwitchClientComponent->requestContextSwitchApplication(getActiveregionid(),
                        (EN_ACTIVITY_IDS)0,
                        (EN_ACTIVITY_IDS)0,
                        APPID_APPHMI_PHONE,
                        enActivityIDs__eActivityID_PHONE_ON_CALL);
               }
               isPhoneReqMade = true;
               break;

               case SWITCH_TO_MULTICALL_ACCEPT:
               {
                  ETG_TRACE_USR4(("sendRequestForContextSwitching - Request Application Switch to Call Accept"));
                  _applicationSwitchClientComponent->requestContextSwitchApplication(getActiveregionid(),
                        (EN_ACTIVITY_IDS)0,
                        (EN_ACTIVITY_IDS)0,
                        APPID_APPHMI_PHONE,
                        enActivityIDs__eActivityID_PHONE_ON_MULTIPARTY_CALL);
               }
               isPhoneReqMade = true;
               break;

               case SWITCH_TO_CALL_BTVR_EXIT:
               {
                  //changing the source activity id to none due to the changes in master team w.r.t Backward context switching Bug 1361301
                  ETG_TRACE_USR4(("sendRequestForContextSwitching - Request Application Switch to BTVR Exit"));
                  _applicationSwitchClientComponent->requestContextSwitchApplication(getActiveregionid(),
                        APPID_APPHMI_PHONE,
                        (EN_ACTIVITY_IDS)0,//enActivityIDs__eActivityID_PHONE_BT_VR,
                        (EN_ACTIVITY_IDS)0,
                        (EN_ACTIVITY_IDS)0);

                  // reset the PhoneContextStatus as we are returning back to last application
                  _applicationSwitchClientHandler->SetPhoneContextStatus(EN_PHONE_CONTEXT_UNKNOWN);
               }
               break;
               default:
                  ETG_TRACE_USR4(("PhoneGadgetRequestMsg - Unknown Value"));
                  break;
            }
         }
         else if ((_isCabinOnly) && (requestType == SWITCH_TO_ADD_DEVICE))
         {
            ETG_TRACE_USR4(("sendRequestForContextSwitching - Request Application Switch to Add Device "));
            // Context switching from Master to Add Device
            _applicationSwitchClientComponent->requestContextSwitchApplication(getActiveregionid(),
                  APPID_APPHMI_MASTER,
                  enActivityIDs__eActivityID_HOME,
                  APPID_APPHMI_PHONE,
                  enActivityIDs__eActivityID_PHONE_DEVICE_SELECTION);
            // isPhoneReqMade = true;
         }

#ifdef VARIANT_S_FTR_ENABLE_BLUETOOTH_ON_OFF
      }
#endif
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
      switch (requestType)
      {
         case SWITCH_TO_INTERCOM_CONTACTS:
         {
            ETG_TRACE_USR4(("sendRequestForContextSwitching - Request Application SWITCH_TO_INTERCOM_CONTACTS"));

            _applicationSwitchClientComponent->requestContextSwitchApplication(getActiveregionid(),
                  APPID_APPHMI_MASTER,
                  enActivityIDs__eActivityID_HOME,
                  APPID_APPHMI_PHONE,
                  enActivityIDs__eActivityID_PHONE_ON_I_CONTACTS);
         }
            //isPhoneReqMade = true;
         break;
         case SWITCH_TO_INTERCOMCALL:
         {
            ETG_TRACE_USR4(("sendRequestForContextSwitching - Request Application SWITCH_TO_INTERCOMCALL"));

            _applicationSwitchClientComponent->requestContextSwitchApplication(getActiveregionid(),
                  APPID_APPHMI_MASTER,
                  enActivityIDs__eActivityID_HOME,
                  APPID_APPHMI_PHONE,
                  enActivityIDs__eActivityID_PHONE_ON_INTERCOMCALL);
         }
         isPhoneReqMade = true;
         break;
         case SWITCH_TO_INTERCOMCALLCONNECTING:
         {
            ETG_TRACE_USR4(("sendRequestForContextSwitching - Request Application SWITCH_TO_INTERCOMCALL"));

            _applicationSwitchClientComponent->requestContextSwitchApplication(getActiveregionid(),
                  APPID_APPHMI_MASTER,
                  enActivityIDs__eActivityID_HOME,
                  APPID_APPHMI_PHONE,
                  enActivityIDs__eActivityID_PHONE_ON_I_CALL_CONNECTING);
         }
         isPhoneReqMade = true;
         break;
         default:
            ETG_TRACE_USR4(("PhoneGadgetRequestMsg - Unknown Value"));
            break;
      }
#endif

      if (isPhoneReqMade == true)
      {
         _applicationSwitchClientHandler->SetPhoneContextStatus(EN_PHONE_CONTEXT_BG);
      }
   }
   else
   {
      ETG_TRACE_USR4(("sendRequestForContextSwitching - Forwarding to internal ContextSwitching.. "));
      sendRequestForContextSwitching_internal(requestType);
   }
   return true;
}


/************************************************************************
*FUNCTION		: sendRequestForContextSwitching_internal
*DESCRIPTION	: Actual member function to send the request to context switch
*PARAMETER		: NA
*RETURNVALUE	: bool
*SWFL			: [SWFL-4347]
*31.05.2019: ebg1kor
*Rev 0.1: Initial Version , SYSFL-5561,SYSFL-5562
*Rev 0.2: 11.09.2019 Palani K , Android Auto changes
************************************************************************/
bool PhoneHall::sendRequestForContextSwitching_internal(const int requestType)
{
   ETG_TRACE_USR4(("sendRequestForContextSwitching_internal - Processing sendRequestForContextSwitching_internal.. "))
   bool isPhoneReqMade = false;
#ifdef VARIANT_S_FTR_ENABLE_BLUETOOTH_ON_OFF
   if (_ServiceBT->GetBluetoothOnOffStatus())
   {
#endif
      if (!_isCabinOnly)
      {
         switch (requestType)
         {
            case SWITCH_TO_ADD_DEVICE:
            {
               ETG_TRACE_USR4(("PhoneHall::sendRequestForContextSwitching_internal::Posting ActivateAddDeviceMsg"));
               POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateAddDeviceMsg)()));
               isPhoneReqMade = true;
               break;
            }
            case SWITCH_TO_CALL_LIST:
            {
               /* When Android Auto session or CarPlay is active, switch to add device list page on receiving call list request as call list page is not applicable in this scenario */
               if ((g_SM_SpiSessionStatus == SPI_SESSION_ANDROID_AUTO_ACTIVE) || (g_SM_SpiSessionStatus == SPI_SESSION_CARPLAY_ACTIVE))
               {
                  ETG_TRACE_USR4(("PhoneHall::sendRequestForContextSwitching_internal::Posting ActivateAddDeviceMsg"));
                  POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateAddDeviceMsg)()));
               }
               else
               {
                  ETG_TRACE_USR4(("PhoneHall::sendRequestForContextSwitching_internal::Posting ActivateCallListMsg"));
                  POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateCallListMsg)()));
               }
               isPhoneReqMade = true;
               break;
            }
            case SWITCH_TO_SPEECH:
            {
               ETG_TRACE_USR4(("PhoneHall::sendRequestForContextSwitching_internal - Request Application Switch to Internal Voice Recognition"));
               _applicationSwitchClientComponent->requestContextSwitchApplication(getActiveregionid(),
                     APPID_APPHMI_PHONE,
                     enActivityIDs__eActivityID_NONE,
                     APPID_APPHMI_SPEECH,
                     enActivityIDs__eActivityID_SPEECH);
               break;
            }
            // move this to PhoneGadget function
            case SWITCH_TO_ACTIVE_PHONE:
               if (_ServiceBT->GetNoOfHFPDevicesConnected() == 1)
               {
                  ETG_TRACE_USR4(("PhoneHall::sendRequestForContextSwitching_internal - Only one phone is connected"));
                  /* When only one device is active, set the device handle index to zero */
                  m_u8PhoneCurrentHandle = 0;
                  /* If android auto session or carplay is active, update the value of 3 as to show phone not connected icon else show first device handle icon */
                  if ((g_SM_SpiSessionStatus == SPI_SESSION_ANDROID_AUTO_ACTIVE) || (g_SM_SpiSessionStatus == SPI_SESSION_CARPLAY_ACTIVE))
                  {
                     /* Update with value of 3 as to show phone not connected icon*/
                     (*_phoneGadgetInfoData).mPhoneHandle =  3;
                  }
                  else
                  {
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
                     (*_phoneGadgetInfoData).mPhoneHandle = 5; //m_u8PhoneCurrentHandle;

#else
                     (*_phoneGadgetInfoData).mPhoneHandle = m_u8PhoneCurrentHandle;
#endif
                  }
                  _phoneGadgetInfoData.MarkAllItemsModified();
                  _phoneGadgetInfoData.SendUpdate(true);

                  (*_PhoneButtonText).mSource = Candera::String(TextId_0x13A5);
                  _PhoneButtonText.MarkAllItemsModified();
                  _PhoneButtonText.SendUpdate(true);
               }
               break;

            case SWITCH_TO_CALL_CONNECTING:
            {
               /* When Call Connecting request comes when phone is transitioning to background, then send the foreground request to Master since the application is transitioning to background */
               /* This usecase shall become valid whenever initiating a call through BTVR session when system was in other than phone application */
               if (_applicationSwitchClientHandler->GetPhoneContextBgReqFlag() == true)
               {
                  ETG_TRACE_USR4(("PhoneHall::sendRequestForContextSwitching_internal - Request Application Switch to Call Connecting"));
                  _applicationSwitchClientComponent->requestContextSwitchApplication(getActiveregionid(),
                        (EN_ACTIVITY_IDS)0,
                        (EN_ACTIVITY_IDS)0,
                        APPID_APPHMI_PHONE,
                        enActivityIDs__eActivityID_PHONE_CALL_CONNECTING);

                  // Set the PhoneContextStatus as BG, as we need to return back to last application on phone context exit
                  _applicationSwitchClientHandler->SetPhoneContextStatus(EN_PHONE_CONTEXT_BG);
                  //Set the background context request flag to false
                  _applicationSwitchClientHandler->SetPhoneContextBgReqFlag(false);
               }
               else
               {
                  ETG_TRACE_USR4(("PhoneHall::sendRequestForContextSwitching_internal::Posting ActivateCallConnectingSceneMsg"));
                  POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateCallConnectingSceneMsg)()));
                  isPhoneReqMade = true;
               }
            }

            break;

            case SWITCH_TO_CALL_ACCEPT:
            {
               ETG_TRACE_USR4(("PhoneHall::sendRequestForContextSwitching_internal::Posting ActivateOnCallSceneMsg"));
               POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateOnCallSceneMsg)()));
            }
            isPhoneReqMade = true;
            break;

            case SWITCH_TO_MULTICALL_ACCEPT:
            {
               ETG_TRACE_USR4(("PhoneHall::sendRequestForContextSwitching_internal::Posting ActivateOnMultipleCallSceneMsg"));
               POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateOnMultipleCallSceneMsg)()));
            }
            isPhoneReqMade = true;
            break;

            case SWITCH_TO_CALL_BTVR_EXIT:
            {
               ETG_TRACE_USR4(("sendRequestForContextSwitching_internal - Request Application Switch to BTVR Exit"));
               //changing the source activity id to none due to the changes in master team w.r.t Backward context switching Bug 1361301
               _applicationSwitchClientComponent->requestContextSwitchApplication(getActiveregionid(),
                     APPID_APPHMI_PHONE,
                     (EN_ACTIVITY_IDS)0,//enActivityIDs__eActivityID_PHONE_BT_VR,
                     (EN_ACTIVITY_IDS)0,
                     (EN_ACTIVITY_IDS)0);

               // reset the PhoneContextStatus as we are returning back to last application
               _applicationSwitchClientHandler->SetPhoneContextStatus(EN_PHONE_CONTEXT_UNKNOWN);
               //Set the background context request flag to true
               _applicationSwitchClientHandler->SetPhoneContextBgReqFlag(true);
            }
            break;
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
            case SWITCH_TO_INTERCOM_CONTACTS:
            {
               ETG_TRACE_USR4(("sendRequestForContextSwitching_internal::Request Application Switch to SWITCH_TO_INTERCOM_CONTACTS"));
               POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateOnIContactsSceneMsg)()));
               isPhoneReqMade = true;
               break;
            }
            case SWITCH_TO_INTERCOMCALL:
            {
               ETG_TRACE_USR4(("sendRequestForContextSwitching_internal::Request Application Switch to SWITCH_TO_INTERCOMCALL"));
               POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateOnIntercomCallSceneMsg)()));
               isPhoneReqMade = true;
               break;
            }
            case SWITCH_TO_INTERCOMCALLCONNECTING:
            {
               ETG_TRACE_USR4(("sendRequestForContextSwitching_internal::Request Application Switch to SWITCH_TO_INTERCOMCALL"));
               POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(SwitchToICallConnecting)()));
               isPhoneReqMade = true;
               break;
            }
#endif
            default:
               ETG_TRACE_USR3(("sendRequestForContextSwitching_internal - Unknown Value"));
               break;
         }
      }
      else if ((_isCabinOnly) && (requestType == SWITCH_TO_ADD_DEVICE))
      {
         ETG_TRACE_USR4(("sendRequestForContextSwitching - Request Application Switch to Add Device "));
         // Context switching from Master to Add Device
         _applicationSwitchClientComponent->requestContextSwitchApplication(getActiveregionid(),
               APPID_APPHMI_MASTER,
               enActivityIDs__eActivityID_HOME,
               APPID_APPHMI_PHONE,
               enActivityIDs__eActivityID_PHONE_DEVICE_SELECTION);
         // isPhoneReqMade = true;
      }

#ifdef VARIANT_S_FTR_ENABLE_BLUETOOTH_ON_OFF
   }
#endif
   if (isPhoneReqMade == true)
   {
      _applicationSwitchClientHandler->SetPhoneContextStatus(EN_PHONE_CONTEXT_FG);
   }
   return true;
}


#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
//Removed Pluginendstaion update as it is depreciated from avdec plugin

bool PhoneHall::onCourierMessage(const PhoneAppflyinReq& msg)
{
   ETG_TRACE_USR4(("PhoneAppflyinReq "));
   OnUpdAppFlyinContextSwitch();
   return true;
}


void PhoneHall::OnUpdAppFlyinContextSwitch()
{
   ETG_TRACE_USR4(("OnUpdAppFlyinContextSwitch "));
   if ((_isCabinOnly) && (PhoneGadgetModel::poGetInstance()->GetIntercomStatus() == true))
   {
      if (true == IsIntercomCallActive)
      {
         ETG_TRACE_USR4((" SWITCH_TO_INTERCOMCALL"));
         POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateOnIntercomCallSceneMsg)()));
      }
      else
      {
         ETG_TRACE_USR4((" SWITCH_TO_INTERCOM_CONTACTS"));
         POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateOnIContactsSceneMsg)()));
      }
   }
   else if (!_isCabinOnly)
   {
      if ((_ServiceBT->GetBluetoothOnOffStatus() == false) && ((NULL != PhoneGadgetModel::poGetInstance()) && (PhoneGadgetModel::poGetInstance()->GetIntercomStatus() == true)))
      {
         ETG_TRACE_USR4(("OnUpdAppFlyinContextSwitch Bluetooth = Off, Intercom = ON "));
         Sm_ActiveSource = 1;
         DeviceListModel::poGetInstance()->OnUpdGadgetStatus(INTERCOM);
         if (true == IsIntercomCallActive)
         {
            ETG_TRACE_USR4((" SWITCH_TO_INTERCOMCALL"));
            POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateOnIntercomCallSceneMsg)()));
         }
         else
         {
            ETG_TRACE_USR4((" SWITCH_TO_INTERCOM_CONTACTS"));
            POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateOnIContactsSceneMsg)()));
         }
      }
      else if ((_ServiceBT->GetBluetoothOnOffStatus() == true) && (PhoneGadgetModel::poGetInstance()->GetIntercomStatus() == false))
      {
         ETG_TRACE_USR4(("OnUpdAppFlyinContextSwitch Bluetooth = ON, Intercom = Off "));
         Sm_ActiveSource = 0;
         DeviceListModel::poGetInstance()->OnUpdGadgetStatus(PHONEONE);
         POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateCallListMsg)()));
      }
      else if ((_ServiceBT->GetBluetoothOnOffStatus() == true) && (PhoneGadgetModel::poGetInstance()->GetIntercomStatus() == true))
      {
         if (true == IsIntercomCallActive)
         {
            DeviceListModel::poGetInstance()->OnUpdGadgetStatus(INTERCOM);
            ETG_TRACE_USR4((" SWITCH_TO_INTERCOM_CALL"));
            POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateOnIntercomCallSceneMsg)()));
         }
         else if ((0 == Sm_ActiveSource) || ((true ==  _ServiceTel->getCallDialingStatus()) || (true == _ServiceTel->getOngoingCallStatus())))
         {
            Sm_ActiveSource = 0;
            DeviceListModel::poGetInstance()->OnUpdGadgetStatus(PHONEONE);
            ETG_TRACE_USR4((" SWITCH_TO_Calllist"));
            POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateCallListMsg)()));
         }
         else if (1 == Sm_ActiveSource)
         {
            DeviceListModel::poGetInstance()->OnUpdGadgetStatus(INTERCOM);
            ETG_TRACE_USR4((" SWITCH_TO_INTERCOM_CONTACTS"));
            POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateOnIContactsSceneMsg)()));
         }
         else
         {
            ETG_TRACE_USR4(("OnUpdAppFlyinContextSwitch other "));
         }
      }
      else
      {
         ETG_TRACE_USR4(("OnUpdAppFlyinContextSwitch Bluetooth = Off, Intercom = Off "));
      }
   }
   else
   {
      ETG_TRACE_USR4(("OnUpdAppFlyinContextSwitch error scenario "));
   }
}


#endif
//Reset To Factory//////////
/* reqPrepareResponse: To know the response of the respective APP when Restore factory setting is pressed.
* @param[in]: None
* @param[in] :None
*/
void PhoneHall::reqPrepareResponse(const PrepareResDiagMsgData& /*oPrepareResDiagMsgData*/)
{
   ETG_TRACE_USR4(("PhoneHall:: reqPrepareResponse"));
   if (_mdefSetServiceBase)
   {
      _mdefSetServiceBase->sendPrepareResponse(0, this);
      ETG_TRACE_USR4(("PhoneHall:: Prepared"));
   }
}


/************************************************************************
*FUNCTION		: reqExecuteResponse
*DESCRIPTION	: Reset To Factory Settings, to Execute the response when
				  restore factory setting has been executed
*PARAMETER		: NA
*RETURNVALUE	: void
*02.07.2019: ebg1kor
**Rev 0.1: Initial Version , SYSFL- 529,SWFL-2684: Factory Reset_Phone
************************************************************************/
void PhoneHall::reqExecuteResponse(const ExecuteResDiagMsgData& /*oExecuteResDiagMsgData*/)
{
   ETG_TRACE_USR4(("PhoneHall:: reqExecuteResponse"));
   if (_mdefSetServiceBase)
   {
      _mdefSetServiceBase->sendExecuteResponse(static_cast<int>(bPhoneRestoreFactorySetting()), this);

      ETG_TRACE_USR4(("PhoneHall:: Executed"));
   }
}


/**
 * bPhoneRestoreFactorySetting: To Execute the reset of Phone Data base & clear all the settings to default settings
 * @param[in]: None
 * @param[in] :None
 *
 */
/************************************************************************
*FUNCTION		: PhoneRestoreFactorySetting
*DESCRIPTION	: Reset To Factory Settings, To Execute the reset of Phone
                 Data base & clear all the settings to default settings
*PARAMETER		: NA
*RETURNVALUE	: bool
*02.07.2019: ebg1kor
*Rev 0.1: Initial Version , SYSFL- 529,SWFL-2684: Factory Reset_Phone
************************************************************************/
tBool PhoneHall::bPhoneRestoreFactorySetting()
{
   ETG_TRACE_USR4(("PhoneHall bPhoneRestoreFactorySetting"));
   tBool bresult = false; //0 - Restore success (for Diagnosis)

   //PhoneDataPoolConfig::getInstance()->resetDPCallListoryTabSel();
   //PhoneDataPoolConfig::getInstance()->resetDpSWCSettingForSiriOrVA();
   //PhoneDataPoolConfig::getInstance()->resetDpLanguage();
   //PhoneDataPoolConfig::getInstance()->resetDpAutoReplyStatus();
   //PhoneDataPoolConfig::getInstance()->resetDpFixedTextMsgIndex();
   //PhoneDataPoolConfig::getInstance()->resetDpTextMessageStatus();
   //PhoneDataPoolConfig::getInstance()->resetDpTextMessageSignatureStatus();
   //PhoneDataPoolConfig::getInstance()->resetDpTextMessageRingToneStatus();
   //PhoneDataPoolConfig::getInstance()->resetDpPhoneRingToneStatus();
   //PhoneDataPoolConfig::getInstance()->resetDpSelectedAutoReplyTab();
   //PhoneDataPoolConfig::getInstance()->resetDpCustomMsgHandle();
   //PhoneDataPoolConfig::getInstance()->resetDpHotSpotPassphrase();
   //PhoneDataPoolConfig::getInstance()->resetDpHotSpotSSID();
   _ServiceBT->onClearDeviceInfo();

   if (PhoneDataPoolConfig::getInstance())
   {
      tU8 SortOrderDefaultvalue = 0; // Default value for sort order on factory reset is assigned as first name
      tU8 ChangeRingtone = 1; //Default value = ON for changeringtone on factory reset.
      bool SMSSettings = true; //Default Value = ON for SMS settings on factory reset.
      ETG_TRACE_USR4(("PhoneHall Making call to Reset Message settings"));
      PhoneDataPoolConfig::getInstance()->resetMessageSettings();
      _ServiceMsg->vSetShowMessageStatus(SMSSettings);
      // _ServiceMsg->vOnChangeSMSSettings();

      ETG_TRACE_USR4(("PhoneHall Making call to Reset Sort order settings"));
      PhoneDataPoolConfig::getInstance()->resetDpSortOrderValue();
      POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ResetSortOrderReq)(SortOrderDefaultvalue)));

      ETG_TRACE_USR4(("PhoneHall Making call to Reset Phone Ringtone settings"));
      PhoneDataPoolConfig::getInstance()->resetDpPhoneRingToneStatus();

#ifndef VARIANT_S_FTR_ENABLE_PHONECALLMANAGER_SUPPORT
      _ServiceBT->vSetChangeRingtone(ChangeRingtone);
#else
      _ServiceTel->vSetRingtone(ChangeRingtone);
#endif
   }
   //MessageListHandler::getInstance()->onClearUserMessagingInfo();
   //PhoneDataPoolConfig::getInstance()->resetDpOutgoingCallVolume();
   //PhoneUserProfile::getInstance()->performUserProfileBTToggling();

   return bresult;
}


/**
 * reqExecuteResponse: To Finalize the response when restore factory setting has been executed.
 * @param[in]: None
 * @param[in] :None
 */
void PhoneHall:: reqFinalizeResponse(const FinalizeResDiagMsgData& /*oFinalizeResDiagMsgData*/)
{
   ETG_TRACE_USR4(("PhoneHall:: reqFinalizeResponse"));
   if (_mdefSetServiceBase)
   {
      _mdefSetServiceBase->sendFinalizeResponse(0, this);
      ETG_TRACE_USR4(("PhoneHall:: Finalized"));
   }
}


bool PhoneHall::onCourierMessage(const RequestCallConnectingMsg& msg)
{
   bool retVal = false;
   int requestType = msg.GetRequestType();
   ETG_TRACE_USR4(("PhoneHall::onCourierMessage RequestCallConnectingMsg: requestType = %d", requestType));
   if ((_ServiceTel->getCallDialingStatus() == true))
   {
      if (requestType != SWITCH_TO_ADD_DEVICE) //&& CurrentView != AppHmi_PhoneModule_PhoneScenes_P_PhoneSelection)
      {
         ETG_TRACE_USR4(("PhoneHall::RequestCallConnectingMsg, SWITCH_TO_CALL_CONNECTING"));
         retVal = sendRequestForContextSwitching(SWITCH_TO_CALL_CONNECTING);
      }
   }
   return retVal;
}


/************************************************************************
*FUNCTION:onCourierMessage(const ControlPanelButtonPressInfo& oMsg)
*DESCRIPTION:courier message is posted from master when there is a long press on the speech button in the control panel notch.The courier message is captured in phone.
************************************************************************/
bool PhoneHall::onCourierMessage(const ControlPanelButtonPressInfo& oMsg)
{
   /* When Long press on speech button is made */
   ETG_TRACE_USR4(("PhoneHall::onCourierMessage ControlPanelButtonPressInfo Entered"));
   ETG_TRACE_USR4(("PhoneHall::onCourierMessage ControlPanelButtonPressInfo: Buttonstate = %d", oMsg.GetButtonstate()));
   ETG_TRACE_USR4(("PhoneHall::onCourierMessage ControlPanelButtonPressInfo: ControlPanelButton = %d", oMsg.GetControlPanelButton()));
   if (oMsg.GetControlPanelButton() == CONTROLPANEL_BUTTON_SPEECH) //(oMsg.GetButtonstate() == CONTROLPANEL_BUTTON_SPEECH)
   {
      ETG_TRACE_USR4(("ApplicationSwitchClientHandler::Launch BTVR"));
      handleBTVRViaControlPanel();
   }
   else
   {
      ETG_TRACE_USR4(("ApplicationSwitchClientHandler::Not Launch BTVR"));
   }
   return true;
}


/************************************************
Function 	: onCourierMessage(GenericCANBTVRReq&)
Parameters 	:
Description : This courier launches the Voice Recognition
*             Feature when trigerred from Generic CAN command
SYSFL/SWFL :
Author     : Bruce Netto 31/05/2021
************************************************/
bool PhoneHall::onCourierMessage(const GenericCANBTVRReq& oMsg)
{
   /* When VR command is trigerred From Generic CAN*/
   ETG_TRACE_USR4(("PhoneHall::onCourierMessage GenericCANBTVRReq Smart VR Generic CAN Trigger"));
   handleBTVRViaControlPanel();
   return true;
}


bool PhoneHall::onCourierMessage(const NavigationCarModeInfo& oMsg)
{
   ETG_TRACE_USR4(("PhoneHall::onCourierMessage carMode :%d coachMode :%d", oMsg.GetCarMode(),  oMsg.GetCoachMode()));

   (*_carModeVisibilityStatus).mVisibleStatus = oMsg.GetCarMode();
   _carModeVisibilityStatus.MarkItemModified(ItemKey::CarModeVisibility::VisibleStatusItem);
   _carModeVisibilityStatus.SendUpdate(true);
   return true;
}


/************************************************************************
*FUNCTION:handleBTVRViaControlPanel
*DESCRIPTION:This function handles launching the BTVR.
************************************************************************/
bool PhoneHall::handleBTVRViaControlPanel()
{
   ETG_TRACE_USR4(("PhoneHall::handleBTVRViaControlPanel Entered"));
   bool retval = false;
   if (_ServiceBT->GetNoOfHFPDevicesConnected() > 0)
   {
      // Having a valid HFP connection without any calls (in/out/active) - phone on Foreground or background
      ETG_TRACE_USR4(("PhoneHall::handleBTVRViaControlPanel -- Request for Smartphone VR "));
      sendRequestForBTVRFunctionHandling();
      retval = true;
   }
   else if (_ServiceBT->GetNoOfHFPDevicesConnected() == 0)
   {
      // No phone connected trigger BT pairing scene
      ETG_TRACE_USR4(("PhoneHall::handleBTVRViaControlPanel - No Phone is connected to Head unit - Add Device "));
      sendRequestForContextSwitching(SWITCH_TO_ADD_DEVICE);
      retval = true;
   }
   return retval;
}


/************************************************************************
*FUNCTION		: HandleHKRemoteControlPhoneKeyPress
*DESCRIPTION	: Member function to handle RemoteC ontrol Phone Key Press button
*PARAMETER		: Hard Key State - Up (release), long (long press)
*RETURNVALUE	: bool
*SWFL			: SYSFL-
************************************************************************/
bool PhoneHall::HandleHKRemoteControlPhoneKeyPress(const int hardkeyState)
{
   bool retval = false;
   if (_isPhonceContextActive == false)
   {
      ETG_TRACE_USR4(("PhoneHall::HandleHKRemoteControlPhoneKeyPress - OnUpdAppFlyinContextSwitch"));
#ifndef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
      //Takes care of moving to the correct Phone scene
      //since in this case Phone is in BG but this is a non intercom variant
      POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateCallListMsg)()));
#else
      if ((0 == Sm_ActiveSource) && (_ServiceBT->GetNoOfHFPDevicesConnected() == 0))
      {
         ETG_TRACE_USR4(("PhoneHall::HandleHKRemoteControlPhoneKeyPress Phone in BG - SWITCH_TO_ADD_DEVICE_LIST"));
         sendRequestForContextSwitching(SWITCH_TO_ADD_DEVICE);
      }
      else if ((0 == Sm_ActiveSource) && (_ServiceBT->GetNoOfHFPDevicesConnected() > 0))
      {
         ETG_TRACE_USR4(("PhoneHall::HandleHKRemoteControlPhoneKeyPress -Phone in BG  Phone is connected to Head unit -call list "));
         sendRequestForContextSwitching(SWITCH_TO_CALL_LIST);
      }
      else
      {
         OnUpdAppFlyinContextSwitch();
      }
#endif
      retval = true;
   }
   else
   {
      ETG_TRACE_USR4(("PhoneHall::HandleHKRemoteControlPhoneKeyPress - Phone application - Current Source (Sm_ActiveSource)is %d", Sm_ActiveSource));
      //Source Checks Done to check if the active one is Phone or Intercom
      //and react accoprdingly using the remote control phone Button
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
      if ((_ServiceBT->GetBluetoothOnOffStatus() == false) &&
            (PhoneGadgetModel::poGetInstance()->GetIntercomStatus() == true))
      {
         ETG_TRACE_USR4(("PhoneHall::HandleHKRemoteControlPhoneKeyPress Bluetooth = Off, Intercom = ON "));
         if (true == IsIntercomCallActive)
         {
            ETG_TRACE_USR4(("PhoneHall::HandleHKRemoteControlPhoneKeyPress::SWITCH_TO_INTERCOMCALL"));
            //POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateOnIntercomCallSceneMsg)()));
         }
         else
         {
            ETG_TRACE_USR4(("PhoneHall::HandleHKRemoteControlPhoneKeyPress::SWITCH_TO_INTERCOM_CONTACTS"));
            POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateOnIContactsSceneMsg)()));
         }
      }
      else if (_ServiceBT->GetBluetoothOnOffStatus() == true)
      {
         ETG_TRACE_USR4(("PhoneHall::HandleHKRemoteControlPhoneKeyPress Bluetooth = ON"));
#endif
         if ((ACTIVE_SOURCE_PHONE == Sm_ActiveSource) &&
               ((false == ServiceTel::poGetInstance()->getCallDialingStatus()) &&
                (false == ServiceTel::poGetInstance()->getIncomingCallStatus()) &&
                (false == ServiceTel::poGetInstance()->getOngoingCallStatus())))
         {
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
            ETG_TRACE_USR4(("PhoneHall::HandleHKRemoteControlPhoneKeyPress - Phone in FG No Phone Calls, Active Source is Phone"));
            if ((false == IsIntercomCallActive) && (false == IntercomClienthandler::poGetInstance()->getIntercomIncomingCallStatus()) && (false == IntercomClienthandler::poGetInstance()->getIntercomCallDialingStatus()))
            {
#endif
               ETG_TRACE_USR4(("PhoneHall::HandleHKRemoteControlPhoneKeyPress - No Intercom Calls"));
               if (_ServiceBT->GetNoOfHFPDevicesConnected() == 0)
               {
                  // No phone connected trigger BT pairing scene
                  ETG_TRACE_USR4(("PhoneHall::HandleHKRemoteControlPhoneKeyPress - Phone in FG No - Phone is connected to Head unit - Add Device "));
                  sendRequestForContextSwitching(SWITCH_TO_ADD_DEVICE);
                  retval = true;
               }
               else
               {
                  ETG_TRACE_USR4(("PhoneHall::HandleHKRemoteControlPhoneKeyPress -Phone in FG - Phone is connected to Head unit -call list "));
                  sendRequestForContextSwitching(SWITCH_TO_CALL_LIST);
                  retval = true;
               }
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
            }
            else
            {
               ETG_TRACE_USR4(("PhoneHall::HandleHKRemoteControlPhoneKeyPress - Intercom Calls are Active"));
            }
#endif
         }
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
         else if ((ACTIVE_SOURCE_INTERCOM == Sm_ActiveSource) &&
                  ((false == ServiceTel::poGetInstance()->getCallDialingStatus()) &&
                   (false == ServiceTel::poGetInstance()->getIncomingCallStatus()) &&
                   (false == ServiceTel::poGetInstance()->getOngoingCallStatus())))
         {
            ETG_TRACE_USR4(("PhoneHall::HandleHKRemoteControlPhoneKeyPress - No Phone Calls, Active Source is Intercom"));
            if ((false == IsIntercomCallActive) && (false == IntercomClienthandler::poGetInstance()->getIntercomIncomingCallStatus()) && (false == IntercomClienthandler::poGetInstance()->getIntercomCallDialingStatus()))
            {
               ETG_TRACE_USR4(("PhoneHall::HandleHKRemoteControlPhoneKeyPress::SWITCH_TO_INTERCOM_CONTACTS"));
               POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateOnIContactsSceneMsg)()));
            }
         }
#endif
         else
         {
            ETG_TRACE_USR4(("PhoneHall::HandleHKRemoteControlPhoneKeyPress - Phone application - Current Source (Sm_ActiveSource) is Intercom"));
         }
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
      }
      else
      {
         ETG_TRACE_USR4(("PhoneHall::HandleHKRemoteControlPhoneKeyPress Bluetooth = Off, Intercom = Off "));
      }
#endif
      retval = true;
   }
   return retval;
}


/************************************************************************
*FUNCTION		: HandleHKRemoteControlDialPadKeyPress
*DESCRIPTION	: Member function to handle Remote Control Dial Pad Key Press button
*PARAMETER		: Hard Key Code - Dial Pad Number
*RETURNVALUE	: bool
*SWFL			: SYSFL-
*AUTHOR     : Bruce Netto     Date:19/04/2021
************************************************************************/
bool PhoneHall::HandleHKRemoteControlDialPadKeyPress(const int hardkeyCode)
{
   bool retval = false;
   ETG_TRACE_USR4(("PhoneHall::HandleHKRemoteControlDialPadKeyPress DialPad: %d", hardkeyCode));

   if (_applicationSwitchClientHandler->GetPhoneContextStatus() != EN_PHONE_CONTEXT_BG)
   {
      ETG_TRACE_USR4(("PhoneHall::HandleHKRemoteControlDialPadKeyPress Phone is in Foreground"));
      switch (hardkeyCode)
      {
         case HARDKEYCODE_REMOTECONTROL_NUM_9:
            ETG_TRACE_USR4(("PhoneHall::HandleHKRemoteControlDialPadKeyPress - DialPad Number 9 is Pressed "));
            ServicePhoneBook::poGetInstance()->onRemoteControlKeyPress("9");
            retval = true;
            break;

         case HARDKEYCODE_REMOTECONTROL_NUM_8:
            ETG_TRACE_USR4(("PhoneHall::HandleHKRemoteControlDialPadKeyPress - DialPad Number 8 is Pressed "));
            ServicePhoneBook::poGetInstance()->onRemoteControlKeyPress("8");
            retval = true;
            break;

         case HARDKEYCODE_REMOTECONTROL_NUM_7:
            ETG_TRACE_USR4(("PhoneHall::HandleHKRemoteControlDialPadKeyPress - DialPad Number 7 is Pressed "));
            ServicePhoneBook::poGetInstance()->onRemoteControlKeyPress("7");
            retval = true;
            break;

         case HARDKEYCODE_REMOTECONTROL_NUM_6:
            ETG_TRACE_USR4(("PhoneHall::HandleHKRemoteControlDialPadKeyPress - DialPad Number 6 is Pressed "));
            ServicePhoneBook::poGetInstance()->onRemoteControlKeyPress("6");
            retval = true;
            break;

         case HARDKEYCODE_REMOTECONTROL_NUM_5:
            ETG_TRACE_USR4(("PhoneHall::HandleHKRemoteControlDialPadKeyPress - DialPad Number 5 is Pressed "));
            ServicePhoneBook::poGetInstance()->onRemoteControlKeyPress("5");
            retval = true;
            break;

         case HARDKEYCODE_REMOTECONTROL_NUM_4:
            ETG_TRACE_USR4(("PhoneHall::HandleHKRemoteControlDialPadKeyPress - DialPad Number 4 is Pressed "));
            ServicePhoneBook::poGetInstance()->onRemoteControlKeyPress("4");
            retval = true;
            break;

         case HARDKEYCODE_REMOTECONTROL_NUM_3:
            ETG_TRACE_USR4(("PhoneHall::HandleHKRemoteControlDialPadKeyPress - DialPad Number 3 is Pressed "));
            ServicePhoneBook::poGetInstance()->onRemoteControlKeyPress("3");
            retval = true;
            break;

         case HARDKEYCODE_REMOTECONTROL_NUM_2:
            ETG_TRACE_USR4(("PhoneHall::HandleHKRemoteControlDialPadKeyPress - DialPad Number 2 is Pressed "));
            ServicePhoneBook::poGetInstance()->onRemoteControlKeyPress("2");
            retval = true;
            break;

         case HARDKEYCODE_REMOTECONTROL_NUM_1:
            ETG_TRACE_USR4(("PhoneHall::HandleHKRemoteControlDialPadKeyPress - DialPad Number 1 is Pressed "));
            ServicePhoneBook::poGetInstance()->onRemoteControlKeyPress("1");
            retval = true;
            break;

         case HARDKEYCODE_REMOTECONTROL_NUM_0:
            ETG_TRACE_USR4(("PhoneHall::HandleHKRemoteControlDialPadKeyPress - DialPad Number 0 is Pressed "));
            ServicePhoneBook::poGetInstance()->onRemoteControlKeyPress("0");
            retval = true;
            break;

         case HARDKEYCODE_REMOTECONTROL_HASH:
            ETG_TRACE_USR4(("PhoneHall::HandleHKRemoteControlDialPadKeyPress - DialPad Number # is Pressed "));
            ServicePhoneBook::poGetInstance()->onRemoteControlKeyPress("#");
            retval = true;
            break;

         case HARDKEYCODE_REMOTECONTROL_STAR:
            ETG_TRACE_USR4(("PhoneHall::HandleHKRemoteControlDialPadKeyPress - DialPad Number * is Pressed "));
            ServicePhoneBook::poGetInstance()->onRemoteControlKeyPress("*");
            retval = true;
            break;

         default:
            ETG_TRACE_USR4(("PhoneHall::HandleHKRemoteControlDialPadKeyPress - Not a valid Key Code %d", hardkeyCode));
            //Default Case shall return false as no Key code has matched our Requirement hence we do not consume it.
            break;
      }
   }
   else
   {
      ETG_TRACE_USR4(("PhoneHall::HandleHKRemoteControlDialPadKeyPress Phone is in BackGround"));
   }

   return retval;
}


} // namespace Core
} // namespace App
