/****************************************************************************
* Copyright (c) 2019-2020 Robert Bosch Car Multimedia GmbH
* duplication and disclosure to third parties is prohibited.
*
* FILE                : MicClientHandler.cpp
* COMPONENT Name      : Apphmi_phone
* DESCRIPTION         : The file consists of handling of Microphone announcement
*                       in the phone component
* AUTHOR              : Bruce Netto
* Revision History    : 0.1
* Date 04.10.2020     : Initial version
****************************************************************************/
#include "MicClientHandler.h"
#include "App/Core/SpeedLockStatusHandler/SpeedLockStatusHandler.h"
#include "PluginConstants.h"
#include "hmi_trace_if.h"
#include <string>
#include "hall_std_if.h"
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
#include "IntercomClienthandler.h"
#endif
#include "ServiceTel.h"
#include "../VehicleSrvClient/VehicleSrvClient.h"
#include "AppHmi_MasterBase/AudioInterface/AudioDefines.h"


#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_PHONE_HALL
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_I_TRACE_CHANNEL TR_TTFIS_APPHMI_PHONE
#define ETG_I_TTFIS_CMD_PREFIX "APPHMI_PHONE_"
#define ETG_I_FILE_PREFIX App::Core::MicClientHandler::
#include "trcGenProj/Header/MicClientHandler.cpp.trc.h"
#endif

using namespace MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange;
using namespace ::bosch::cm::ai::hmi::masteraudioservice::SoundProperties;
using namespace asf::core;
using namespace ::MOST_Tel_FI;
using namespace ::most_Tel_fi_types;

namespace App {
namespace Core {

MicClientHandler* MicClientHandler::_MicInstance = NULL;

/************************************************
Function 	: MicClientHandler Constructor
Parameters 	:
Description :
SYSFL/SWFL  :
Author      : Bruce Netto 5/10/2020
************************************************/
MicClientHandler::MicClientHandler():
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
   rfoIntercomModel(*IntercomModel::poGetInstance()),
#endif
   _audioSourceChangeProxy(AudioSourceChangeProxy::createProxy("audioSourceChangePort", *this)),
   _audioSoundPropertiesProxy(SoundPropertiesProxy::createProxy("soundPropertiesPort", *this)),
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
   m_bPrivateMode(false),
#endif
   m_bMIC1Status(false),
   _enMicStatus(enMIC_Off)

{
   ETG_TRACE_USR4(("MicClientHandler Constructor is called"));
   StartupSync::getInstance().registerPropertyRegistrationIF(this, "audioSourceChangePort");
   StartupSync::getInstance().registerPropertyRegistrationIF(this, "soundPropertiesPort");
}


/************************************************
Function 	: ~MicClientHandler Destructor
Parameters 	:
Description :
SYSFL/SWFL  :
Author      : Bruce Netto 5/10/2020
************************************************/
MicClientHandler::~MicClientHandler()
{
   _audioSourceChangeProxy.reset();
}


/************************************************
Function 	: poGetInstance
Parameters 	:
Description :
SYSFL/SWFL  :
Author      : Bruce Netto 5/10/2020
************************************************/
MicClientHandler* MicClientHandler::poGetInstance()
{
   ETG_TRACE_USR4(("MicClientHandler poGetInstance"));
   if (_MicInstance == NULL)
   {
      _MicInstance = new MicClientHandler();
   }
   return _MicInstance;
}


/************************************************
Function 	: registerProperties
Parameters 	:
Description :
SYSFL/SWFL  :
Author      : Bruce Netto 5/10/2020
************************************************/
void MicClientHandler::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("MicClientHandler register property is called"));
}


/************************************************
Function 	: deregisterProperties
Parameters 	:
Description :
SYSFL/SWFL  :
Author      : Bruce Netto 5/10/2020
************************************************/
void MicClientHandler::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("MicClientHandler deregister property is called"));
}


/************************************************
Function 	: onUnavailable
Parameters 	:
Description :
SYSFL/SWFL  :
Author      : Bruce Netto 5/10/2020
************************************************/
void MicClientHandler::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("MicClientHandler::onUnavailable()"));
   if (_audioSourceChangeProxy && _audioSourceChangeProxy == proxy)
   {
      _audioSourceChangeProxy->sendDeregisterAll();
   }

   if (_audioSoundPropertiesProxy != NULL)
   {
      _audioSoundPropertiesProxy->sendSystemPropertyListDeregisterAll();
      _audioSoundPropertiesProxy->sendSystemPropertyChangedDeregisterAll();
   }
}


/************************************************
Function 	: onAvailable
Parameters 	:
Description :
SYSFL/SWFL  :
Author      : Bruce Netto 5/10/2020
************************************************/
void MicClientHandler::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("MicClientHandler::onAvailable()"));
   if (_audioSourceChangeProxy && _audioSourceChangeProxy == proxy)
   {
      _audioSourceChangeProxy->sendActiveSourceListRegister(*this);
   }

   if (_audioSoundPropertiesProxy != NULL)
   {
      _audioSoundPropertiesProxy->sendSystemPropertyChangedRegister(*this);
      _audioSoundPropertiesProxy->sendSystemPropertyListRegister(*this);
      _audioSoundPropertiesProxy->sendSystemPropertyListGet(*this);
   }
}


/************************************************
Function 	: onActiveSourceListError
Parameters 	:
Description : This status is received from HMI_Master
*             if there are any errors in
*             ActiveSourceList request.
SYSFL/SWFL  :
Author      : Bruce Netto 9/10/2020
************************************************/
void MicClientHandler::onActiveSourceListError(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::AudioSourceChangeProxy >& proxy,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::ActiveSourceListError >& error)
{
   ETG_TRACE_USR4(("MicClientHandler::onActiveSourceListError\n"));
}


/************************************************
Function 	: onActiveSourceListUpdate
Parameters 	:
Description : This status is received from HMI_Master
*             on ActiveSourceList request.
SYSFL/SWFL  :
Author      : Bruce Netto 9/10/2020
************************************************/
void MicClientHandler::onActiveSourceListUpdate(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::AudioSourceChangeProxy >& proxy,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::ActiveSourceListUpdate >& signal)
{
   ETG_TRACE_USR4(("MicClientHandler::onActiveSourceListUpdate \n"));
   std::vector<sourceData>AudioSourceList = signal->getActiveSourceList();
   std::vector<sourceData>::const_iterator iter = AudioSourceList.begin(); // The list's bottom is the top of the stack
   m_bMIC1Status = false;

   while (iter != AudioSourceList.end())
   {
      ETG_TRACE_USR4(("MicClientHandler::onActiveSourceListUpdate() iAudioSourceID  %d", iter->getSrcId()));
      if ((iter->getSrcId() == SRC_MIC1_ANNOUNCEMENT) && (iter->getConnectionState() == STATE_CONNECTED))
      {
         ETG_TRACE_USR4(("MicClientHandler::onActiveSourceListUpdate() AudioSourceID  is MIC 1 %d", iter->getSrcId()));//SRC_MIC1_ANNOUNCEMENT 445
         ETG_TRACE_USR4(("MicClientHandler::onActiveSourceListUpdate() AudioSinkId  %d", iter->getSinkId()));//COCKPIT_SINK_ID 1
         ETG_TRACE_USR4(("MicClientHandler::onActiveSourceListUpdate() audioSourceConnStatus  %d", iter->getConnectionState()));//STATE_CONNECTED 2

         m_bMIC1Status = true;
         break;
      }
      iter++;
   }
   //Call MIC Check everytime an update happens to Source Activation
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
   vHandleMIC(m_bMIC1Status);
#endif
}


/************************************************
Function 	: onSystemPropertyChangedError
Parameters 	:
Description : Property updating the HMI the error
			  in systempropertychanged
SYSFL/SWFL  :
*Author      : Vidya K S
************************************************/

void MicClientHandler::onSystemPropertyChangedError(const ::boost::shared_ptr<AUDIO_SOUND_PROPERTIES::SoundPropertiesProxy >& proxy, const ::boost::shared_ptr<AUDIO_SOUND_PROPERTIES::SystemPropertyChangedError >& error)
{
   ETG_TRACE_ERR(("MicClientHandler::onSystemPropertyChangedError "));
}


/************************************************
Function 	: onSystemPropertyChangedSignal
Parameters 	:
Description : Property updating the HMI the change
			  in property type and value of
			  systempropertychangedSignal
SYSFL/SWFL  :
*Author      : Vidya K S
************************************************/
void MicClientHandler::onSystemPropertyChangedSignal(const ::boost::shared_ptr<AUDIO_SOUND_PROPERTIES::SoundPropertiesProxy >& proxy, const ::boost::shared_ptr< AUDIO_SOUND_PROPERTIES::SystemPropertyChangedSignal >& signal)
{
   ETG_TRACE_USR4(("MicClientHandler::onSystemPropertyChangedSignal "));
   ETG_TRACE_USR4(("MicClientHandler::onSystemPropertyChangedSignal getpropertyType = %d", signal->getPropertyType()));
   ETG_TRACE_USR4(("MicClientHandler::onSystemPropertyChangedSignal getvalue =%d", signal->getValue()));
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
   if (_audioSoundPropertiesProxy != NULL)
   {
      int propertyType = signal->getPropertyType();
      if (SYP_PRIVATE_MODE == propertyType)
      {
         ETG_TRACE_USR4(("AudioClientHandler::onSystemPropertyChangedSignal(), propertyType : %d", propertyType));
         onPrivateModeUpdate(signal->getValue());
      }
   }
#endif
}


/************************************************
Function 	: onSetSystemPropertyError
Parameters 	:
Description : method updating the HMI the error
			  in setting the systemproperty
SYSFL/SWFL  :
*Author      : Vidya K S
************************************************/
void MicClientHandler::onSetSystemPropertyError(const ::boost::shared_ptr< AUDIO_SOUND_PROPERTIES::SoundPropertiesProxy >& proxy, const ::boost::shared_ptr< AUDIO_SOUND_PROPERTIES::SetSystemPropertyError >& error)
{
   ETG_TRACE_ERR(("MicClientHandler::onSetSystemPropertyError "));
}


/************************************************
Function 	: onSetSystemPropertyResponse
Parameters 	:
Description : method updating the HMI the response
			  received in setting the systemproperty
SYSFL/SWFL  :
*Author      : Vidya K S
************************************************/

void MicClientHandler::onSetSystemPropertyResponse(const ::boost::shared_ptr< AUDIO_SOUND_PROPERTIES::SoundPropertiesProxy >& proxy, const ::boost::shared_ptr< AUDIO_SOUND_PROPERTIES::SetSystemPropertyResponse >& response)
{
   ETG_TRACE_USR4(("MicClientHandler::onSetSystemPropertyResponse "));
}


/************************************************
Function 	: onSystemPropertyListError
Parameters 	:
Description : property updating the HMI the error
			  received while getting the systempropertyList
SYSFL/SWFL  :
*Author      : Vidya K S
************************************************/
void MicClientHandler::onSystemPropertyListError(const ::boost::shared_ptr<AUDIO_SOUND_PROPERTIES::SoundPropertiesProxy >& proxy, const ::boost::shared_ptr< AUDIO_SOUND_PROPERTIES::SystemPropertyListError >& error)
{
   ETG_TRACE_ERR(("MicClientHandler::onSystemPropertyListError "));
}


/************************************************
Function 	: onSystemPropertyListError
Parameters 	:
Description : property updating the HMI property type
			  and value received while getting the systempropertyList
SYSFL/SWFL  :
*Author      : Vidya K S
************************************************/
void MicClientHandler::onSystemPropertyListUpdate(const ::boost::shared_ptr<AUDIO_SOUND_PROPERTIES::SoundPropertiesProxy >& proxy, const ::boost::shared_ptr<AUDIO_SOUND_PROPERTIES::SystemPropertyListUpdate >& update)
{
   ETG_TRACE_USR4(("MicClientHandler::onSystemPropertyListUpdate "));
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT

   if (_audioSoundPropertiesProxy != NULL)
   {
      std::vector< SystemProperty > audioPropertyVector = update->getSystemPropertyList();
      std::vector< SystemProperty >::iterator itr = audioPropertyVector.begin();
      for (; itr < audioPropertyVector.end(); ++itr)
      {
         int propertyType = itr->getPropertyType();
         if (SYP_PRIVATE_MODE == propertyType)
         {
            ETG_TRACE_USR4(("MicClientHandler::onSystemPropertyListUpdate(), propertyType : %d", propertyType));
            ETG_TRACE_USR4(("MicClientHandler::onSystemPropertyListUpdate getvalue =%d", itr->getValue()));
            onPrivateModeUpdate(itr->getValue());
         }
      }
   }
#endif
}


/************************************************
Function 	: vHandleMIC
Parameters 	:
Description : This function is called when the MICrophone
*             gets enabled to check and disable phone and intercom
*             functionalities
SYSFL/SWFL :
Author     : Bruce Netto 5/10/2020
************************************************/
void MicClientHandler::vHandleMIC(bool l_bmicStatus)
{
   ETG_TRACE_USR4(("MicClientHandler::vHandleMIC MIC Value  = %d", l_bmicStatus));
   //Dial Pad button should be disabled
   //vCheckforEnableDialpadCallButton();
   SpeedLockStatusHandler::getInstance().updatespeedlockenabledStatus();
   //Check if the MIC Announcement is happening based on value
   if (l_bmicStatus == true)
   {
      ETG_TRACE_USR4(("MicClientHandler::vHandleMIC Entered IF Condition"));
      //PHONE
      //Check if Incoming Call Pop up is present on screen when MIC is activating
      if (ServiceTel::poGetInstance()->getIncomingCallStatus() == true)
      {
         ETG_TRACE_USR4(("MicClientHandler::vHandleMIC getIncomingCallStatus = %d", ServiceTel::poGetInstance()->getIncomingCallStatus()));
         //Incoming call should be rejected
         //ServiceTel::poGetInstance()->vIgnoreCallStart(ServiceTel::poGetInstance()->getIncomingCallInstance());
         POST_MSG((COURIER_MESSAGE_NEW(IncomingCallResponseReq)(BTPHN_EN_INCOMINGCALL_RESPONSE_REJECT)));
         //Pop up to be shown
         POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(MICPhoneCallResp)()));
         /*  if (getEnMicStatus() == enMIC_IntercomCall)
          {
             ETG_TRACE_USR4(("MicClientHandler::vHandleMIC Intercom call is missed before Bt call"));
             vSetMicStatus(enMIC_Intercom_BTCall);
          }
          else
          {
             ETG_TRACE_USR4(("MicClientHandler::vHandleMIC BT call is missed"));
             vSetMicStatus(enMIC_BTCall);
          } */
         ETG_TRACE_USR4(("MicClientHandler::vHandleMIC Incoming Call MICPhoneCallResp"));
      }
      else
      {
         //There is no Incoming Pop up on the screen when MIC became Active
         ETG_TRACE_USR4(("MicClientHandler::vHandleMIC No Incoming call pop up on screen during MIC Activation"));
      }
      //Check if a call is in active state when the MIC is becoming active
      if (ServiceTel::poGetInstance()->getOngoingCallStatus() == true)
      {
         ETG_TRACE_USR4(("MicClientHandler::vHandleMIC getActiveCallStatus = %d", ServiceTel::poGetInstance()->getOngoingCallStatus()));
         //Active call should be rejected via courier message TerminateCallReq
         POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(TerminateCallReq)()));
         //Pop up to be shown
         //POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(MICPhoneCallResp)()));
         ETG_TRACE_USR4(("MicClientHandler::vHandleMIC Active Call MICPhoneCallResp"));
      }
      else
      {
         //There is no Active Call Pop up on the screen when MIC became Active
         ETG_TRACE_USR4(("MicClientHandler::vHandleMIC No Active call pop up on screen during MIC Activation"));
      }
      //Check if a call is in Outgoing state when the MIC is becoming active
      if (ServiceTel::poGetInstance()->getCallDialingStatus() == true)
      {
         ETG_TRACE_USR4(("MicClientHandler::vHandleMIC getCallDialingStatus = %d", ServiceTel::poGetInstance()->getCallDialingStatus()));
         //Outgoing call should be rejected via courier message CancelDialCallReq
         POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(CancelDialCallReq)()));
         //Pop up to be shown
         POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(MICContactsCallHistoryResp)()));
         ETG_TRACE_USR4(("MicClientHandler::vHandleMIC OutGoing Call MICContactsCallHistoryResp"));
      }
      else
      {
         //There is no Active Call Pop up on the screen when MIC became Active
         ETG_TRACE_USR4(("MicClientHandler::vHandleMIC No Outgoing call pop up on screen during MIC Activation"));
      }
      //Check if a BTVR is ON screen when the MIC is becoming active
      if (ServiceTel::poGetInstance()->getBTVRStatus() == true)
      {
         ETG_TRACE_USR4(("MicClientHandler::vHandleMIC getBTVRStatus = %d", ServiceTel::poGetInstance()->getBTVRStatus()));
         //BTVR should be Closed via vDisableBTVRView()
         ServiceTel::poGetInstance()->vDisableBTVRView();
      }
      else
      {
         //There is no Active Call Pop up on the screen when MIC became Active
         ETG_TRACE_USR4(("MicClientHandler::vHandleMIC No BTVR on screen during MIC Activation"));
      }
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
      //INTERCOM
      //Check if Intercom Incoming Call Pop up is present on screen when MIC is activating
      if (IntercomClienthandler::poGetInstance()->getIntercomIncomingCallStatus() == true)
      {
         ETG_TRACE_USR4(("MicClientHandler::vHandleMIC getIntercomIncomingCallStatus = %d", IntercomClienthandler::poGetInstance()->getIntercomIncomingCallStatus()));
         //Intercom Incoming call should be rejected
         POST_MSG((COURIER_MESSAGE_NEW(IntercomCallResponseReq)(INTERCOMCALL_RESPONSE_REJECT)));
         //Pop up to be shown
         POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(MICIntercomCallResp)()));//send courier message to show mic anounment pop up
         /*  if (getEnMicStatus() == enMIC_BTCall)
          {
             ETG_TRACE_USR3(("MicClientHandler::vHandleMIC Bt call is missed before intercom missed call"));
             vSetMicStatus(enMIC_BTCall_Intercom);
          }
          else
          {
             ETG_TRACE_USR3(("MicClientHandler::vHandleMIC intercom call is missed "));
             vSetMicStatus(enMIC_IntercomCall);
          } */
         ETG_TRACE_USR4(("MicClientHandler::vHandleMIC Intercom Incoming Call MICIntercomCallResp"));
      }
      else
      {
         //There is no Intercom Incoming Pop up on the screen when MIC became Active
         ETG_TRACE_USR4(("MicClientHandler::vHandleMIC No Intercom Incoming call pop up on screen during MIC Activation"));
      }

      //Check if Intercom Active Call Pop up is present on screen when MIC is activating
      if (IntercomClienthandler::poGetInstance()->getIntercomOngoingCallStatus() == true)
      {
         ETG_TRACE_USR4(("MicClientHandler::vHandleMIC getIntercomOngoingCallStatus = %d", IntercomClienthandler::poGetInstance()->getIntercomOngoingCallStatus()));
         //Intercom Active call should be rejected
         POST_MSG((COURIER_MESSAGE_NEW(IntercomCallResponseReq)(INTERCOMCALL_RESPONSE_REJECT)));
         //Pop up to be shown
         //POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(MICIntercomCallResp)()));
         ETG_TRACE_USR4(("MicClientHandler::vHandleMIC Intercom Active Call MICIntercomCallResp"));
      }
      else
      {
         //There is no Intercom Active Pop up on the screen when MIC became Active
         ETG_TRACE_USR4(("MicClientHandler::vHandleMIC No Intercom Active call pop up on screen during MIC Activation"));
      }
      //Check if Intercom Outgoing Call Pop up is present on screen when MIC is activating
      if (IntercomClienthandler::poGetInstance()->getIntercomCallDialingStatus() == true)
      {
         ETG_TRACE_USR4(("MicClientHandler::vHandleMIC getIntercomCallDialingStatus = %d", IntercomClienthandler::poGetInstance()->getIntercomCallDialingStatus()));
         //Intercom Outgoing call should be rejected
         POST_MSG((COURIER_MESSAGE_NEW(IntercomCallResponseReq)(INTERCOMCALL_RESPONSE_REJECT)));
         //Pop up to be shown
         //POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(MICIntercomCallResp)()));
         ETG_TRACE_USR4(("MicClientHandler::vHandleMIC Intercom Outgoing Call MICIntercomCallResp"));
      }
      else
      {
         //There is no Intercom Active Pop up on the screen when MIC became Active
         ETG_TRACE_USR4(("MicClientHandler::vHandleMIC No Intercom Outgoing call pop up on screen during MIC Activation"));
      }
      if (IntercomClienthandler::poGetInstance()->getIntercomConfCallStatus() == true)
      {
         ETG_TRACE_USR4(("MicClientHandler::vHandleMIC getIntercomConfCallStatus = %d", IntercomClienthandler::poGetInstance()->getIntercomConfCallStatus()));
         //Intercom Outgoing call should be rejected
         POST_MSG((COURIER_MESSAGE_NEW(IntercomCallResponseReq)(INTERCOMCALL_RESPONSE_CONF_REJECT)));
         //Pop up to be shown
         // POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(MICIntercomCallResp)()));
         ETG_TRACE_USR4(("MicClientHandler::vHandleMIC Intercom Conference Call MICIntercomCallResp"));
      }
      else
      {
         //There is no Intercom Active Pop up on the screen when MIC became Active
         ETG_TRACE_USR4(("MicClientHandler::vHandleMIC No Intercom Conference call pop up on screen during MIC Activation"));
      }
#endif
   }
   else if (l_bmicStatus == false)
   {
      ETG_TRACE_USR4(("MicClientHandler::vHandleMIC() _enMicStatus: %d", _enMicStatus));
      if (getEnMicStatus() == enMIC_BTCall)
      {
         ETG_TRACE_USR4(("MicClientHandler::vHandleMIC() BT call missed"));
         POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(MICPhoneCallResp)()));
      }
      else if (getEnMicStatus() == enMIC_IntercomCall)
      {
         ETG_TRACE_USR4(("MicClientHandler::vHandleMIC() Intecom call missed"));
         POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(MICIntercomCallResp)()));
      }
      else if (getEnMicStatus() == enMIC_BTCall_Intercom)
      {
         ETG_TRACE_USR4(("MicClientHandler::vHandleMIC() BT and Intecom missed call"));
         POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(MICPhoneCallResp)()));
         POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(MICIntercomCallResp)()));
      }
      else if (getEnMicStatus() == enMIC_Intercom_BTCall)
      {
         ETG_TRACE_USR4(("MicClientHandler::vHandleMIC() Intecom and BT missed call"));
         POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(MICIntercomCallResp)()));
         POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(MICPhoneCallResp)()));
      }
      vSetMicStatus(enMIC_Off);
   }
}


/************************************************
Function 	: vSetMicStatus
Parameters 	:
Description : This function is called when the
*             MIC status is to be set to a respective state
SYSFL/SWFL :
Author     : Bruce Netto 11/05/2021
************************************************/
void MicClientHandler::vSetMicStatus(enMicStatus micStatus)
{
   _enMicStatus = micStatus;
   ETG_TRACE_USR4(("MicClientHandler::vSetMicStatus _enMicStatus = %d", _enMicStatus));
}


#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
/************************************************
Function 	: vTogglePrivateMode
Parameters 	:
Description : Function call for updating the
			  private mode ON/OFF on toggle in
			  any kind of Phone/Intercom calls
SYSFL/SWFL  :
*Author      : Vidya K S
************************************************/
void MicClientHandler::vTogglePrivateMode()
{
   ETG_TRACE_USR4(("MicClientHandler : vTogglePrivateMode()"));
   ETG_TRACE_USR4(("MicClientHandler : vTogglePrivateMode()  m_bPrivateMode = %d", m_bPrivateMode));
   m_bPrivateMode = !m_bPrivateMode;
   ETG_TRACE_USR4(("MicClientHandler : vTogglePrivateMode() changed  m_bPrivateMode = %d", m_bPrivateMode));
   int16_t privatemodevalue = (true == m_bPrivateMode) ? SYP_PRIVATE_MODE_2 : SYP_PRIVATE_MODE_1;
   ETG_TRACE_USR4(("MicClientHandler : vTogglePrivateMode() privatemodevalue = %d", privatemodevalue));
   _audioSoundPropertiesProxy->sendSetSystemPropertyRequest(*this, SYP_PRIVATE_MODE, privatemodevalue) ;
   rfoIntercomModel.OnUpdPrivatemode(m_bPrivateMode);
}


/************************************************
Function 	: onPrivateModeUpdate
Parameters 	:
Description : Function call for updating the
			  private mode ON/OFF received on property
			  update in any kind of Phone/Intercom calls
SYSFL/SWFL  :
*Author      : Vidya K S
************************************************/
void MicClientHandler::onPrivateModeUpdate(int privateMode)
{
   ETG_TRACE_USR4(("MicClientHandler::onPrivateModeUpdate(), privateMode : %d", privateMode));

   if ((privateMode == SYP_PRIVATE_MODE_2) || (privateMode == SYP_PRIVATE_MODE_3))
   {
      m_bPrivateMode = true;
      ETG_TRACE_USR4(("MicClientHandler : onPrivateModeUpdate() - Private mode ON"));
   }
   else if (privateMode == SYP_PRIVATE_MODE_1)
   {
      m_bPrivateMode = false;
      ETG_TRACE_USR4(("MicClientHandler : onPrivateModeUpdate() - Private mode OFF"));
   }
   else
   {
      ETG_TRACE_USR4(("MicClientHandler : onPrivateModeUpdate() - Invalid"));
   }
   rfoIntercomModel.OnUpdPrivatemode(m_bPrivateMode);
}


/************************************************
Function 	: PrivateModeUpdateMsg
Parameters 	:
Description : courier message to update the
			  private mode value on startup
SYSFL/SWFL  :
*Author      : Vidya K S
************************************************/
bool MicClientHandler::onCourierMessage(const PrivateModeUpdateMsg& oMsg)
{
   ETG_TRACE_USR4(("MicClientHandler : PrivateModeUpdateMsg - courier message"));
   ETG_TRACE_USR4(("MicClientHandler::PrivateModeUpdateMsg m_bPrivateMode : %d", m_bPrivateMode));
   rfoIntercomModel.OnUpdPrivatemode(m_bPrivateMode);
   return true;
}


#endif
/************************************************
Function 	: ButtonReactionMsg
Parameters 	:
Description : courier message to update the
			  buttonreactions on the press of buttons
SYSFL/SWFL  :
*Author      : Vidya K S
************************************************/
bool MicClientHandler::onCourierMessage(const ButtonReactionMsg& oMsg)
{
   ETG_TRACE_USR4(("MicClientHandler : ButtonReactionMsg()"));
   tBool bRetVal = false;
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
   const Courier::Identifier  IdBtn2 = Courier::Identifier("Layer_PrivatePublicSpeaker/ButtonWidget2D");

   const Courier::Identifier senderInfo = oMsg.GetSender();
   if (oMsg.GetEnReaction() == enRelease)
   {
      ETG_TRACE_USR4(("MicClientHandler : ButtonReactionMsg - enRelease"));
      if (senderInfo == IdBtn2)
      {
         ETG_TRACE_USR4(("MicClientHandler : ButtonReactionMsg - enRelease -senderinfo"));
         vTogglePrivateMode();
      }
   }
#endif
   return bRetVal;
}


/************************************************
Function 	: vCheckforEnableDialpadCallButton
Parameters 	:
Description : This function is called when the
*             call button needs to be disabled or enabled
*             based on speed treshold or Mic status
SYSFL/SWFL :
Author     : Bruce Netto 5/10/2020
************************************************/
/* void MicClientHandler::vCheckforEnableDialpadCallButton()
{
   //If Speed is greater that treshold or MIC status true disable the dial call button
   if (m_bMIC1Status == true) //|| (SpeedLockStatusHandler::getInstance().getSpeedLockActive() == true && SpeedLockStatusHandler::getInstance().getSysSpeedLockEnable() == true))
   {
      ETG_TRACE_USR4(("MicClientHandler::vCheckforEnableDialpadCallButton DialPad Call Button False"));
      (*speedlock).mDialPadCallButtonStatus = false;
   }
   else
   {
      ETG_TRACE_USR4(("MicClientHandler::vCheckforEnableDialpadCallButton DialPad Call Button True"));
      (*speedlock).mDialPadCallButtonStatus = true;
   }
   speedlock.MarkItemModified(ItemKey::SpeedLock::DialPadCallButtonStatusItem);
   speedlock.SendUpdate(true);
}


*/
}//namespace Core
}//namespace App
