/***************************************************************************
* Copyright(c) 2017-2020 Robert Bosch Car Multimedia GmbH
* This software is property of Robert Bosch GmbH.
***************************************************************************/
/*****************************************************************************
 * (c) Robert Bosch Car Multimedia GmbH
 *
 * Add details here
****************************************************************************/
#include "hall_std_if.h"
#include "ServiceMsg.h"
#include "ServiceTel.h"
#include "AppHmi_PhoneTypes.h"
#include "ServiceBT.h"
#include "App/datapool/PhoneDataPoolConfig.h"
#include "Common/VariantHandling/VariantHandling.h"


#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_PHONE_HALL
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ServiceMsg.cpp.trc.h"
#endif

using namespace ::MOST_Msg_FI;
using namespace ::most_Msg_fi_types;

//Add include files here

namespace App {
namespace Core {

ServiceMsg* ServiceMsg::_poServiceMsg = NULL;

/************************************************************************
*FUNCTION: 		 ServiceMsg()
*DESCRIPTION: 	 Constructor of ServiceMsg Class, here _MSGProxy is initialised
*PARAMETER:
*
*RETURNVALUE: 	 tVoid
*
*HISTORY:
*revision 0.1	ess7kor	11/08/2017
*************************************************************************/
ServiceMsg::ServiceMsg():
   _MsgProxy(MOST_Msg_FIProxy::createProxy("btMsgFiPort", *this)),
   rfoMSGModelComponent(*MSGModelComponent::poGetInstance()),
   u8ConnectedDeviceHandle(0),
   m_bShowMessageStatus(true),
   m_u16MessageListHandle(BTPHN_C_LIST_HANDLE_INVALID)

{
   // Initialize _isCabinOnly based on the VariantType
   if (VariantHandling::getInstance() != NULL)
   {
      _isCabinOnly = (VariantHandling::getInstance()->getVariantType() == VARIANTTYPE_CC);
   }
   else
   {
      _isCabinOnly = false; // Default to false if VariantHandling is not available
   }

   memset(&rMsgMessageHandle, 0, sizeof(btphn_trMsgHandle));
   if (_MsgProxy.get())
   {
      ETG_TRACE_USR2(("ServiceMsg _MsgProxy.get is TRUE"));
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _MsgProxy->getPortName());
   }
   vInit();
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
ServiceMsg::~ServiceMsg()
{
}


void ServiceMsg::vInit()
{
   ETG_TRACE_USR2(("ServiceMsg vInit"));

   uint16 u16VehicleType = VEHICLETYPE_TRUCK;
   //Read VehicleType from KDS and register for the MSG services based on the Vehicletype value
   if (PhoneDataPoolConfig::getInstance())
   {
      ETG_TRACE_FATAL(("ServiceMsg:: Retrive vehicleType information from KDS"));
      u16VehicleType = PhoneDataPoolConfig::getInstance()->getKDSVehicleType();
      ETG_TRACE_FATAL(("ServiceMsg:: getKDSVehicleType, Vehicle type : %d", u16VehicleType));
   }
   else
   {
      ETG_TRACE_FATAL(("ServiceMsg:: Error in getting PhoneDataPoolConfig instance"));
   }
   // MSG services are available only when vehicle variant is other than Smart
   if (VEHICLETYPE_SMART != u16VehicleType)
   {
      if (PhoneDataPoolConfig::getInstance())
      {
         m_bShowMessageStatus = PhoneDataPoolConfig::getInstance()->getMessageSettings();
         //Also set the settings checkbox accordingly TRUE = checked, FALSE = unchecked
         if (PhonebookModelComponent::poGetInstance())
         {
            if (m_bShowMessageStatus == TRUE)
            {
               ETG_TRACE_USR2(("ServiceMsg BTPHN_C_CHECKED"));
               PhonebookModelComponent::poGetInstance()->vSetShowIncomingMessage(BTPHN_C_CHECKED);
            }
            else
            {
               ETG_TRACE_USR2(("ServiceMsg BTPHN_C_CHECKED"));
               PhonebookModelComponent::poGetInstance()->vSetShowIncomingMessage(BTPHN_C_UNCHECKED);
            }
         }
         //ETG_TRACE_USR2(("m_bShowMessageStatus is %d", ETG_CENUM(tBool, m_bShowMessageStatus)));
      }
   }
}


void ServiceMsg::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _MsgProxy)
   {
      ETG_TRACE_USR2(("ServiceMsg registerProperties"));
   }
}


void ServiceMsg::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _MsgProxy)
   {
      ETG_TRACE_USR2(("ServiceMsg deregisterProperties"));
   }
}


/************************************************************************
*FUNCTION: 		 onAvailable
*DESCRIPTION: 	 ASF Framework callback of Messaging service Available(), on Service available, do a upreg of all properities used here.
*PARAMETER:
* 					 proxy :	(I)
* 						asf::core::Proxy Shared pointer,
*					stateChange :
*						ServiceStateChange Object
*
*RETURNVALUE: 	 tVoid
*
*HISTORY:
*revision 0.1	ess7kor	11/08/2017
*************************************************************************/
void ServiceMsg::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_FATAL(("ServiceMsg onAvailable(_MsgProxy)"));

   uint16 u16VehicleType = VEHICLETYPE_TRUCK;
   //Read VehicleType from KDS and register for the MSG services based on the Vehicletype value
   if (PhoneDataPoolConfig::getInstance())
   {
      ETG_TRACE_FATAL(("ServiceMsg:: Retrive vehicleType information from KDS"));
      u16VehicleType = PhoneDataPoolConfig::getInstance()->getKDSVehicleType();
      ETG_TRACE_FATAL(("ServiceMsg:: getKDSVehicleType, Vehicle type : %d", u16VehicleType));
   }
   else
   {
      ETG_TRACE_FATAL(("ServiceMsg:: Error in getting PhoneDataPoolConfig instance"));
   }

   //Register for MSG services only when vehicle variant is other than Smart
   if ((VEHICLETYPE_SMART != u16VehicleType) && (!_isCabinOnly))
   {
      ETG_TRACE_FATAL(("ServiceMsg:: Register for the MSG services"));
      if ((proxy == _MsgProxy) && (_MsgProxy.get()))
      {
         _MsgProxy->sendMessageListChangeUpReg(*this);
         _MsgProxy->sendMessagingDeviceConnectionUpReg(*this);
         _MsgProxy->sendMessagingAudioPlaybackStateUpReg(*this);
         _MsgProxy->sendMapDeviceCapabilitiesExtendedUpReg(*this);
      }
      else
      {
         if (proxy != _MsgProxy)
         {
            ETG_TRACE_FATAL(("ServiceMsg registerProperties(_MsgProxy != proxy)"));
         }
      }
   }
}


/************************************************************************
*FUNCTION: 		 onUnavailable
*DESCRIPTION: 	 ASF Framework callback of Messaging service unavailable(), on Service unavailable, do a rel-upreg of all properities used here.
*PARAMETER:
* 					 proxy :	(I)
* 						asf::core::Proxy Shared pointer,
*					stateChange :
*						ServiceStateChange Object
*
*RETURNVALUE: 	 tVoid
*
*HISTORY:
*revision 0.1	ess7kor	11/08/2017
*************************************************************************/
void ServiceMsg::onUnavailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const ServiceStateChange& stateChange)
{
   uint16 u16VehicleType = VEHICLETYPE_TRUCK;
   //Read VehicleType from KDS and deregister for the MSG services based on the Vehicletype value
   if (PhoneDataPoolConfig::getInstance())
   {
      ETG_TRACE_FATAL(("ServiceMsg:: onUnavailable: Retrive vehicleType information from KDS"));
      u16VehicleType = PhoneDataPoolConfig::getInstance()->getKDSVehicleType();
      ETG_TRACE_FATAL(("ServiceMsg:: onUnavailable: getKDSVehicleType, Vehicle type : %d", u16VehicleType));
   }
   else
   {
      ETG_TRACE_FATAL(("ServiceMsg:: onUnavailable:Error in getting PhoneDataPoolConfig instance"));
   }

   //Deregister for MSG services only when vehicle variant is other than Smart
   if ((VEHICLETYPE_SMART != u16VehicleType)  && (!_isCabinOnly))
   {
      if (proxy == _MsgProxy && (_MsgProxy.get()))
      {
         ETG_TRACE_FATAL(("ServiceMsg deregisterProperties(_MsgProxy)"));
         _MsgProxy->sendMessageListChangeRelUpRegAll();
         _MsgProxy->sendMessagingDeviceConnectionRelUpRegAll();
         _MsgProxy->sendMessagingAudioPlaybackStateRelUpRegAll();
         _MsgProxy->sendMapDeviceCapabilitiesExtendedRelUpRegAll();
      }
   }
}


/************************************************************************
*FUNCTION: 		 onCreateMessageListError
*DESCRIPTION: 	 Method result error callback of onCreateMessageList Method start
*PARAMETER:
* 					 proxy :	(I)
* 						MOST_Msg_FI::MOST_Msg_FIProxy Shared pointer,
*					error :
*						CreateMessageListError Object
*
*RETURNVALUE: 	 tVoid
*
*HISTORY:
*revision 0.1	ess7kor	11/08/2017
*************************************************************************/
void ServiceMsg::onCreateMessageListError(const :: boost::shared_ptr< ::MOST_Msg_FI::MOST_Msg_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Msg_FI::CreateMessageListError >& error)
{
   ETG_TRACE_FATAL(("ServiceMsg::onCreateMessageListError"));
}


/************************************************************************
*FUNCTION: 		 onCreateMessageListResult
*DESCRIPTION: 	 Method result  callback of onCreateMessageList Method start, extract only the list handle
*PARAMETER:
* 					 proxy :	(I)
* 						MOST_Msg_FI::MOST_Msg_FIProxy Shared pointer,
*					result :
*						CreateMessageListResult Object
*
*RETURNVALUE: 	 tVoid
*
*HISTORY:
*revision 0.1	ess7kor	11/08/2017
*************************************************************************/
void ServiceMsg::onCreateMessageListResult(const :: boost::shared_ptr< ::MOST_Msg_FI::MOST_Msg_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Msg_FI::CreateMessageListResult >& result)
{
   ETG_TRACE_USR4(("ServiceMsg::onCreateMessageListResult"));
   if (_MsgProxy)
   {
      ETG_TRACE_USR4(("ServiceMsg::onCreateMessage ListHandle:%d, ListLength:%d", result->getU16ListHandle(), result->getU16ListLength()));
      m_u16MessageListHandle = result->getU16ListHandle();
      messageHandleList.push_back(m_u16MessageListHandle);
   }
}


/************************************************************************
*FUNCTION: 		 onGetMessageError
*DESCRIPTION: 	 Method result Error callback of GetMessage Method start
*PARAMETER:
* 					 proxy :	(I)
* 						MOST_Msg_FI::MOST_Msg_FIProxy Shared pointer,
*					error :
*						GetMessageError Object
*
*RETURNVALUE: 	 tVoid
*
*HISTORY:
*revision 0.1	ess7kor	11/08/2017
*************************************************************************/
void ServiceMsg::onGetMessageError(const ::boost::shared_ptr< MOST_Msg_FIProxy >& /*proxy*/, const boost::shared_ptr< GetMessageError >& /*error*/)
{
   ETG_TRACE_FATAL(("ServiceMsg::onGetMessageError"));
}


/************************************************************************
*FUNCTION: 		 onGetMessageResult
*DESCRIPTION: 	 Method result callback of GetMessage Method start
*PARAMETER:
* 					 proxy :	(I)
* 						MOST_Msg_FI::MOST_Msg_FIProxy Shared pointer,
*					result :
*						GetMessageResult Object
*
*RETURNVALUE: 	 tVoid
*
*HISTORY:
*revision 0.1	ess7kor	11/08/2017
*revision 0.2	pka5cob	11/09/2019; android auto changes
*************************************************************************/
void ServiceMsg::onGetMessageResult(const ::boost::shared_ptr< ::MOST_Msg_FI::MOST_Msg_FIProxy >& /*proxy*/, const boost::shared_ptr< ::MOST_Msg_FI::GetMessageResult >& result)
{
   ETG_TRACE_USR4(("ServiceMsg::onGetMessageResult"));
   if (_MsgProxy)
   {
      /*extract The Short Message and Subject in the rMessageDetails Structure */
      rMessageDetails.copcSubject = result->getOMessageDetailsResult().getSSubject();
      rMessageDetails.copcShortMessage = result->getOMessageDetailsResult().getSShortMessage();

      ETG_TRACE_USR3(("----Subject = %s", rMessageDetails.copcSubject.c_str()));
      ETG_TRACE_USR3(("----ShortMessage = %s", rMessageDetails.copcShortMessage.c_str()));
      if (m_bShowMessageStatus)
      {
         /* Send the Msg update to Model component only when Android auto session and Carplay is not active */
         if ((g_SM_SpiSessionStatus != SPI_SESSION_ANDROID_AUTO_ACTIVE) && (g_SM_SpiSessionStatus != SPI_SESSION_CARPLAY_ACTIVE))
         {
            /*Send update to MsgDataModel to update the details in HMI*/
            rfoMSGModelComponent.OnUpdSendIncomingMsg(rMessageDetails);
            ETG_TRACE_USR3(("update to model"));
         }
      }
   }
}


/************************************************************************
*FUNCTION: 		 onRequestSliceMessageListError
*DESCRIPTION: 	 Error status of RequestSliceMessageListError
*
*PARAMETER:
* 					 proxy :	(I)
* 						MOST_Msg_FI::MOST_Msg_FIProxy Shared pointer,
*					error :
*						RequestSliceMessageListError Object
*
*RETURNVALUE: 	 tVoid
*
*HISTORY:
*revision 0.1	ess7kor	11/08/2017
*************************************************************************/

void ServiceMsg::onRequestSliceMessageListError(const :: boost::shared_ptr< ::MOST_Msg_FI::MOST_Msg_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Msg_FI::RequestSliceMessageListError >& error)
{
   ETG_TRACE_FATAL(("ServiceMsg::onRequestSliceMessageListError"));
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/

void ServiceMsg::onRequestSliceMessageListResult(const ::boost::shared_ptr< ::MOST_Msg_FI::MOST_Msg_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Msg_FI::RequestSliceMessageListResult >& result)
{
   ETG_TRACE_USR4(("ServiceMsg::onRequestSliceMessageListResult()"));

   if (_MsgProxy)
   {
      btphn_trMsgListSupport	rMsgListEntry;

      /* Extract Message Header*/
      vExtractMessageHeader(rMsgListEntry, result);

      /*Print detials in Console*/
      vLogMessageHeader(rMsgListEntry);

      /*If Msg type is SMS or MMS Get the Message from Service*/
      if ((BTPHN_EN_E8MSG_TYPE_SMS == rMsgListEntry.enMsgType) || (BTPHN_EN_E8MSG_TYPE_MMS == rMsgListEntry.enMsgType))
      {
         static tU32 u32UpperMsgHandle = BTPHN_C_DEFAULT_VALUE_ZERO;

         ETG_TRACE_USR3(("MsgHandleUpper = %d", u32UpperMsgHandle));

         if (u32UpperMsgHandle != rMsgListEntry.rMsgHandle.u32MsgHandleUpper)
         {
            ETG_TRACE_USR3(("sendGetMessageStart"));
            u32UpperMsgHandle = rMsgListEntry.rMsgHandle.u32MsgHandleUpper;
            T_MsgMessageHandle msgHandle(rMsgListEntry.rMsgHandle.u8DeviceHandle, rMsgListEntry.rMsgHandle.u32MsgHandleUpper, rMsgListEntry.rMsgHandle.u32MsgHandleLower);
            _MsgProxy->sendGetMessageStart(*this, msgHandle);
         }
      }
   }
}


/************************************************************************
*FUNCTION: 		 vExtractMessageHeader
*DESCRIPTION: 	 Message header extraction and storing in local data
*
*PARAMETER:
*				rMsgListEntry
*					btphn_trMsgListSupport Obj reference
*				result:
*					RequestSliceMessageListResult obj reference
*RETURNVALUE: 	 tVoid
*
*HISTORY:
*revision 0.1	ess7kor	11/08/2017
*************************************************************************/

void ServiceMsg::vExtractMessageHeader(btphn_trMsgListSupport& rMsgListEntry, const boost::shared_ptr< ::MOST_Msg_FI::RequestSliceMessageListResult >& result)
{
   ETG_TRACE_USR4(("ServiceMsg::vExtractMessageHeader()"));

   int u32SizeOfList = result->getOMessageListSliceResult().size();
   if (u32SizeOfList <= 0 || result->getOMessageListSliceResult().empty())
   {
      return;
   }
   u32SizeOfList = 0;

   rMsgListEntry.rMsgHandle.u8DeviceHandle = result->getOMessageListSliceResult().at(u32SizeOfList).getOMessageHandle().getU8DeviceHandle();
   rMsgListEntry.rMsgHandle.u32MsgHandleUpper = result->getOMessageListSliceResult().at(u32SizeOfList).getOMessageHandle().getU32MsgHandleUpper();
   rMsgListEntry.rMsgHandle.u32MsgHandleLower = result->getOMessageListSliceResult().at(u32SizeOfList).getOMessageHandle().getU32MsgHandleLower();

   rMsgMessageHandle.u32MsgHandleUpper = rMsgListEntry.rMsgHandle.u32MsgHandleUpper;
   rMsgMessageHandle.u32MsgHandleLower = rMsgListEntry.rMsgHandle.u32MsgHandleLower;

   //rMsgListEntry.rMsgDateTime.rMsgdate.copcCldrYear = result->getOMessageListSliceResult().at(u32SizeOfList).getOMessageDateTime().getOMessageDate().getSCldrYear();
   //rMsgListEntry.rMsgDateTime.rMsgdate.copcCldrMonth = result->getOMessageListSliceResult().at(u32SizeOfList).getOMessageDateTime().getOMessageDate().getSCldrMonth();
   //rMsgListEntry.rMsgDateTime.rMsgdate.copcCldrDay = result->getOMessageListSliceResult().at(u32SizeOfList).getOMessageDateTime().getOMessageDate().getSCldrDay();

   //rMsgListEntry.rMsgDateTime.rMsgTime.copcHours = result->getOMessageListSliceResult().at(u32SizeOfList).getOMessageDateTime().getOMessageTime().getSHours();
   //rMsgListEntry.rMsgDateTime.rMsgTime.copcMinutes = result->getOMessageListSliceResult().at(u32SizeOfList).getOMessageDateTime().getOMessageTime().getSMinutes();
   //rMsgListEntry.rMsgDateTime.rMsgTime.copcSecounds = result->getOMessageListSliceResult().at(u32SizeOfList).getOMessageDateTime().getOMessageTime().getSSeconds();

//	rMsgListEntry.copcSubject = result->getOMessageListSliceResult().at(u32SizeOfList).getSSubject();
//	rMsgListEntry.copcShortMessage = result->getOMessageListSliceResult().at(u32SizeOfList).getSShortMessage().c_str();
   rMsgListEntry.bAttachmentFlag = result->getOMessageListSliceResult().at(u32SizeOfList).getBAttachmentFlag();

   rMsgListEntry.enMsgType = static_cast <btphn_MsgType>(result->getOMessageListSliceResult().at(u32SizeOfList).getE8MessageType());

   rMsgListEntry.enMsgSendMsgResultType = static_cast <btphn_MsgSendMsgResultType>(result->getOMessageListSliceResult().at(u32SizeOfList).getE8SendMessageResultType());

   rMsgListEntry.bReadStatusFlag = result->getOMessageListSliceResult().at(u32SizeOfList).getBReadStatusFlag();

   rMsgListEntry.copcFirstName = result->getOMessageListSliceResult().at(u32SizeOfList).getSFirstName().c_str();
   rMsgListEntry.copcLastName = result->getOMessageListSliceResult().at(u32SizeOfList).getSLastName().c_str();
   rMsgListEntry.copcEmailAddress = result->getOMessageListSliceResult().at(u32SizeOfList).getSEmailAddress().c_str();
   rMsgListEntry.copcPhoneNumber = result->getOMessageListSliceResult().at(u32SizeOfList).getSPhoneNumber().c_str();

   rMessageDetails.copcPhoneNumber = rMsgListEntry.copcPhoneNumber.c_str();
   rMessageDetails.copcFirstName = rMsgListEntry.copcFirstName.c_str();
   rMessageDetails.copcLastName = rMsgListEntry.copcLastName.c_str();
   rMessageDetails.rMsgDateTime.rMsgdate.copcCldrYear = result->getOMessageListSliceResult().at(u32SizeOfList).getOMessageDateTime().getOMessageDate().getSCldrYear().c_str();
   rMessageDetails.rMsgDateTime.rMsgdate.copcCldrMonth = result->getOMessageListSliceResult().at(u32SizeOfList).getOMessageDateTime().getOMessageDate().getSCldrMonth().c_str();
   rMessageDetails.rMsgDateTime.rMsgdate.copcCldrDay = result->getOMessageListSliceResult().at(u32SizeOfList).getOMessageDateTime().getOMessageDate().getSCldrDay().c_str();

   rMessageDetails.rMsgDateTime.rMsgTime.copcHours = result->getOMessageListSliceResult().at(u32SizeOfList).getOMessageDateTime().getOMessageTime().getSHours().c_str();
   rMessageDetails.rMsgDateTime.rMsgTime.copcMinutes = result->getOMessageListSliceResult().at(u32SizeOfList).getOMessageDateTime().getOMessageTime().getSMinutes().c_str();
   rMessageDetails.rMsgDateTime.rMsgTime.copcSecounds = result->getOMessageListSliceResult().at(u32SizeOfList).getOMessageDateTime().getOMessageTime().getSSeconds().c_str();
}


/************************************************************************
*FUNCTION: 		 vLogMessageHeader
*DESCRIPTION: 	 Message header logging for debugging purpose.
*
*PARAMETER:
*				rMsgListEntry
*					btphn_trMsgListSupport Obj reference
*
*RETURNVALUE: 	 tVoid
*
*HISTORY:
*revision 0.1	ess7kor	11/08/2017
*************************************************************************/
void ServiceMsg::vLogMessageHeader(btphn_trMsgListSupport& rMsgListEntry)
{
   u8ConnectedDeviceHandle = rMsgListEntry.rMsgHandle.u8DeviceHandle;
   ETG_TRACE_USR3(("----DeviceHandle = %d", rMsgListEntry.rMsgHandle.u8DeviceHandle));
   ETG_TRACE_USR3(("----MsgHandleUpper = %d", rMsgListEntry.rMsgHandle.u32MsgHandleUpper));
   ETG_TRACE_USR3(("----MsgHandleLower = %d", rMsgListEntry.rMsgHandle.u32MsgHandleLower));

   ETG_TRACE_USR3(("----DateTimeHours = %s", rMsgListEntry.rMsgDateTime.rMsgTime.copcHours.c_str()));
   ETG_TRACE_USR3(("----DateTimeMinutes = %s", rMsgListEntry.rMsgDateTime.rMsgTime.copcMinutes.c_str()));

   ETG_TRACE_USR3(("----FirstName = %s", rMsgListEntry.copcFirstName.c_str()));
   ETG_TRACE_USR3(("----LastName = %s", rMsgListEntry.copcLastName.c_str()));
   ETG_TRACE_USR3(("----PhoneNumber = %s", rMsgListEntry.copcPhoneNumber.c_str()));
}


/************************************************************************
*FUNCTION: 		 onReleaseMessageListError
*DESCRIPTION: 	 Method Result Error callback of onReleaseMessageListEror
*
*PARAMETER:
* 					 proxy :	(I)
* 						MOST_Msg_FI::MOST_Msg_FIProxy Shared pointer,
*					result :
*						onReleaseMessageListError Object
*
*RETURNVALUE: 	 tVoid
*
*HISTORY:
*revision 0.1	ess7kor	11/08/2017
*************************************************************************/
void ServiceMsg::onReleaseMessageListError(const :: boost::shared_ptr< ::MOST_Msg_FI::MOST_Msg_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Msg_FI::ReleaseMessageListError >& error)
{
   ETG_TRACE_USR3(("ServiceMsg:: onReleaseMessageListError "));
}


/************************************************************************
*FUNCTION: 		 onReleaseMessageListResult
*DESCRIPTION: 	 Method Result callback of onReleaseMessageListResult
*
*PARAMETER:
* 					 proxy :	(I)
* 						MOST_Msg_FI::MOST_Msg_FIProxy Shared pointer,
*					result :
*						ReleaseMessageListResult Object
*
*RETURNVALUE: 	 tVoid
*
*HISTORY:
*revision 0.1	ess7kor	11/08/2017
*************************************************************************/
void ServiceMsg::onReleaseMessageListResult(const :: boost::shared_ptr< ::MOST_Msg_FI::MOST_Msg_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Msg_FI::ReleaseMessageListResult >& result)
{
   ETG_TRACE_USR3(("ServiceMsg:: onReleaseMessageListResult\n"));
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/


/************************************************************************
*FUNCTION: 		 onReadMessageAloudExtendedError
*DESCRIPTION: 	 Error status of onReadMessageAloudExtendedError
*
*PARAMETER:
* 					 proxy :	(I)
* 						MOST_Msg_FI::MOST_Msg_FIProxy Shared pointer,
*					error :
*						ReadMessageAloudError Object
*
*RETURNVALUE: 	 tVoid
*
*HISTORY:
*revision 0.1	ess7kor	11/08/2017
*************************************************************************/
void ServiceMsg::onReadMessageAloudExtendedError(const ::boost::shared_ptr< ::MOST_Msg_FI::MOST_Msg_FIProxy >& /*proxy*/, const boost::shared_ptr< ::MOST_Msg_FI::ReadMessageAloudExtendedError >& error)
{
   ETG_TRACE_FATAL(("ServiceMsg::onReadMessageAloudExtendedError:"));
}


/************************************************************************
*FUNCTION: 		 onReadMessageAlouExtended Result
*DESCRIPTION: 	 Method Result callback of onReadMessageAloudExtendedResult
*
*PARAMETER:
* 					 proxy :	(I)
* 						MOST_Msg_FI::MOST_Msg_FIProxy Shared pointer,
*					error :
*						ReadMessageAloudResult Object
*
*RETURNVALUE: 	 tVoid
*
*HISTORY:
*revision 0.1	ess7kor	11/08/2017
*************************************************************************/
void ServiceMsg::onReadMessageAloudExtendedResult(const ::boost::shared_ptr< ::MOST_Msg_FI::MOST_Msg_FIProxy >& /*proxy*/, const boost::shared_ptr< ::MOST_Msg_FI::ReadMessageAloudExtendedResult >& result)
{
   ETG_TRACE_FATAL(("ServiceMsg::onReadMessageAloudExtendedResult"));
}


/************************************************************************
*FUNCTION: 		 onMessageListChangeError
*DESCRIPTION: 	 Error status of onMessageListChangeError
*
*PARAMETER:
* 					 proxy :	(I)
* 						MOST_Msg_FI::MOST_Msg_FIProxy Shared pointer,
*					error :
*						MessageListChangeError Object
*
*RETURNVALUE: 	 tVoid
*
*HISTORY:
*revision 0.1	ess7kor	11/08/2017
*************************************************************************/
void ServiceMsg::onMessageListChangeError(const :: boost::shared_ptr< ::MOST_Msg_FI::MOST_Msg_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Msg_FI::MessageListChangeError >& error)
{
   ETG_TRACE_FATAL(("ServiceMsg::onMessageListChangeError"));
}


/************************************************************************
*FUNCTION: 		 onMessageListChangeStatus
*DESCRIPTION: 	 Change in the MessageList status notification from Service. This means there is a new message, hence request the Message Slice
*
*PARAMETER:
* 					 proxy :	(I)
* 						MOST_Msg_FI::MOST_Msg_FIProxy Shared pointer,
*					status :
*						MessageListChangeStatus Object
*
*RETURNVALUE: 	 tVoid
*
*HISTORY:
*revision 0.1	ess7kor	11/08/2017
*************************************************************************/
void ServiceMsg::onMessageListChangeStatus(const ::boost::shared_ptr< ::MOST_Msg_FI::MOST_Msg_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Msg_FI::MessageListChangeStatus >& status)
{
   if (_MsgProxy)
   {
      tU16 U16ListHandle = status->getU16ListHandle();
      ETG_TRACE_USR3(("onMessageListChangeStatus ListHandle = %d, ListChangeType = %d, ListLength = %d, NumItems = %d",
                      U16ListHandle,
                      ETG_CENUM(btphn_MsgListChangeType, status->getE8ListChangeType()),
                      status->getU16ListLength(),
                      status->getU32NumItems()));

      /*If the recieved list has changes it means that there is a new sms , hence request send sendRequestSliceMessageListStart*/
      ETG_TRACE_USR3(("T_e8_MsgListChangeType__e8LCH_ITEMS_ADDED:%d", T_e8_MsgListChangeType__e8LCH_ITEMS_ADDED));
      ETG_TRACE_USR3(("T_e8_MsgListChangeType__e8LCH_ITEMS_CHANGED:%d", T_e8_MsgListChangeType__e8LCH_ITEMS_CHANGED));
      ETG_TRACE_USR3(("U16ListHandle:%d", U16ListHandle));
      ETG_TRACE_USR3(("m_u16MessageListHandle:%d", m_u16MessageListHandle));
      if (T_e8_MsgListChangeType__e8LCH_ITEMS_ADDED == status->getE8ListChangeType() || T_e8_MsgListChangeType__e8LCH_ITEMS_CHANGED == status->getE8ListChangeType())
      {
         for (std::vector<int>::iterator it = messageHandleList.begin() ; it != messageHandleList.end(); ++it)
         {
            if (std::find(messageHandleList.begin(), messageHandleList.end(), U16ListHandle) != messageHandleList.end())
            {
               ETG_TRACE_USR3(("onMessageListChangeStatus: sendRequestSliceMessageListStart"));
               _MsgProxy->sendRequestSliceMessageListStart(*this, U16ListHandle, BTPHNMSG_C_WINDOW_START, BTPHNMSG_C_WINDOW_SIZE);
            }
         }
      }
      else
      {
         ETG_TRACE_USR3(("onMessageListChangeStatus , No addition or Change in List"));
      }
   }
}


/************************************************************************
*FUNCTION: 		 onMessagingDeviceConnectionError
*DESCRIPTION: 	 Error status of MessageDeviceConenction
*
*PARAMETER:
* 					 proxy :	(I)
* 						MOST_Msg_FI::MOST_Msg_FIProxy Shared pointer,
*					error :
*						MessagingDeviceConnectionError Object
*
*RETURNVALUE: 	 tVoid
*
*HISTORY:
*revision 0.1	ess7kor	11/08/2017
*************************************************************************/
void ServiceMsg::onMessagingDeviceConnectionError(const :: boost::shared_ptr< ::MOST_Msg_FI::MOST_Msg_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Msg_FI::MessagingDeviceConnectionError >& error)
{
   ETG_TRACE_FATAL(("ServiceMsg::onMessagingDeviceConnectionError"));
}


/************************************************************************
*FUNCTION: 		 onMessagingDeviceConnectionStatus
*DESCRIPTION: 	 On recieveing the Device Cnenction Status upDATE from service, trigger CreateMessageList Method Start
*
*PARAMETER:
* 					 proxy :	(I)
* 						MOST_Msg_FI::MOST_Msg_FIProxy Shared pointer,
*					status :
*						MessagingDeviceConnectionStatus Object
*
*RETURNVALUE: 	 tVoid
*HISTORY:
*revision 0.1	ess7kor	11/08/2017
*************************************************************************/
void ServiceMsg::onMessagingDeviceConnectionStatus(const ::boost::shared_ptr< ::MOST_Msg_FI::MOST_Msg_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Msg_FI::MessagingDeviceConnectionStatus >& status)
{
   ETG_TRACE_USR3(("ServiceMsg::onMessagingDeviceConnectionStatus"));
   ETG_TRACE_USR3(("ServiceMsg::onMessagingDeviceConnectionStatus:%d", status->getBSMSSupport()));
   if (_MsgProxy && (TRUE == status->getBSMSSupport()))
   {
      tU8 u8DeviceHandle = status->getU8DeviceHandle();
      ETG_TRACE_USR3(("ServiceMsg::u8DeviceHandle:%d", u8DeviceHandle));
      //_MsgProxy->sendCreateMessageListStart(*this, u8DeviceHandle, T_e8_MsgFolderType__e8MSG_FOLDER_INBOX, 0, T_e8_MsgMessageListType__e8MSG_LIST_SMS_MMS, T_e8_MsgMessageListSortType__e8MSG_LIST_SORT_TIMESTAMP, T_e8_MsgMessageListFilterType__e8MSG_LIST_FILTER_ALL);
   }
}


void ServiceMsg::onMapDeviceCapabilitiesExtendedStatus(const ::boost::shared_ptr< MOST_Msg_FIProxy >& proxy, const ::boost::shared_ptr< MapDeviceCapabilitiesExtendedStatus >& status)
{
   if (status->getOMapDeviceCapabilitiesList().size() != 0)
   {
      for (int nIndex = 0 ; nIndex <= status->getOMapDeviceCapabilitiesList().size() ; ++nIndex)
      {
         ETG_TRACE_USR3(("ServiceMsg::onMapDeviceCapabilitiesExtendedStatus Handle:%d", status->getOMapDeviceCapabilitiesList()[nIndex].getU8DeviceHandle()));
         ETG_TRACE_USR3(("ServiceMsg::onMapDeviceCapabilitiesExtendedStatus:%d", status->getOMapDeviceCapabilitiesList()[nIndex].getU16SupportedMessageTypes()));

         if (_MsgProxy && (BT_SUPPORTED_MESSAGE_TYPE == status->getOMapDeviceCapabilitiesList()[nIndex].getU16SupportedMessageTypes()))
         {
            ETG_TRACE_USR3(("ServiceMsg::onMapDeviceCapabilitiesExtendedStatus 0"));
            tU8 u8DeviceHandle1 = status->getOMapDeviceCapabilitiesList()[nIndex].getU8DeviceHandle();
            u8ConnectedDeviceHandle = u8DeviceHandle1;
            _MsgProxy->sendCreateMessageListStart(*this, u8DeviceHandle1, T_e8_MsgFolderType__e8MSG_FOLDER_INBOX, 0, T_e8_MsgMessageListType__e8MSG_LIST_SMS_MMS, T_e8_MsgMessageListSortType__e8MSG_LIST_SORT_TIMESTAMP, T_e8_MsgMessageListFilterType__e8MSG_LIST_FILTER_ALL);
         }
      }
   }
}


void ServiceMsg::onMapDeviceCapabilitiesExtendedError(const ::boost::shared_ptr< MOST_Msg_FIProxy >& proxy, const ::boost::shared_ptr< MapDeviceCapabilitiesExtendedError >& error)
{
   ETG_TRACE_FATAL(("ServiceMsg::onMapDeviceCapabilitiesExtendedError"));
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void ServiceMsg::onMessagingAudioPlaybackStateError(const :: boost::shared_ptr< ::MOST_Msg_FI::MOST_Msg_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Msg_FI::MessagingAudioPlaybackStateError >& error)
{
   ETG_TRACE_USR3(("ServiceMsg::onMessagingAudioPlaybackStateError"));
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void ServiceMsg::onMessagingAudioPlaybackStateStatus(const :: boost::shared_ptr< ::MOST_Msg_FI::MOST_Msg_FIProxy >& proxy, const boost::shared_ptr< ::MOST_Msg_FI::MessagingAudioPlaybackStateStatus >& status)
{
   /*ETG_TRACE_USR3(("ServiceMsg::onMessagingAudioPlaybackStateStatus, Play Back Status is %d", ETG_CENUM(most_Msg_fi_types::T_e8_MsgPlaybackState,
   	status->getE8PlaybackState())));*/

   most_Msg_fi_types::T_e8_MsgPlaybackState EnumPlayBackState;
   //EnumPlayBackState = static_cast <btphn_trMsgPlayBackState>(status->getE8PlaybackState());
   EnumPlayBackState = status->getE8PlaybackState();

   //Update GUI about the Message Play Status
   rfoMSGModelComponent.OnUpdMsgAudioPlayStatus(EnumPlayBackState);
}


/************************************************************************
*FUNCTION: 		 vDeleteMsgList
*DESCRIPTION: 	 Deletes all the lists maintained by tclMessagingLogic namely
*
*PARAMETER:     None
*RETURNVALUE: 	 tVoid
*
*HISTORY:
*06.11.13 Anarasi Bikash Patro
*Rev 0.1 Initial Revision.
************************************************************************/

tVoid ServiceMsg::vDeleteMsgList()
{
   ETG_TRACE_USR4(("ServiceMsg : vDeleteMsgList()"));

   if (BTPHN_C_LIST_HANDLE_INVALID != m_u16MessageListHandle)
   {
      if (NULL != _MsgProxy)
      {
         _MsgProxy->sendReleaseMessageListStart(*this, m_u16MessageListHandle);
      }
   }
   m_u16MessageListHandle = BTPHN_C_LIST_HANDLE_INVALID;
}


/************************************************************************
*FUNCTION: 		 ReadMessageAloudReq
*DESCRIPTION: 	 Handles request from HMI to call ReadMessageAloud
*					 with PlayBack action as PBA_PLAY
*
*PARAMETER:
* 					 ReadMessageAloudReq :	(I)
* 						Courier message received
*
*RETURNVALUE: 	 tBool
*						TRUE, if the message is consumed
*						FALSE, if the message should be forwarded
*
*HISTORY:
*revision 0.1	ebg1kor	11/08/2019
*************************************************************************/
bool ServiceMsg::onCourierMessage(const ReadMessageAloudReq& oMsg)
{
   ETG_TRACE_USR3(("ServiceMsg::cobOnReadMessageAloudReq(): Received ReadMessageAloudReq"));
   if (_MsgProxy)
   {
      btphn_trMsgPlayBackAction enPlayBackAction = oMsg.GetPlayBackAction();

      ETG_TRACE_USR4(("PlayBack Action = %d", enPlayBackAction));
      tU8 u8DeviceHandle;
      /*Forward the request to Service by Sending method start ReadMessageAloud*/
      /*todo LogicalAV channel input from HMI is unknow, and in Suzuki it is unspecified, hecne using default value of 0 in the paramter as midwfi sets default value 0 in suzuki,
      This needs to be checked from Service */
      ::most_Msg_fi_types::T_e8_MsgPlaybackAction e8PlaybackAction = static_cast<T_e8_MsgPlaybackAction>(enPlayBackAction);
      //_MsgProxy->sendReadMessageAloudStart(*this, msgHandle, e8PlaybackAction, 0, T_e8_MsgTTSLanguage__e8USE_SYSTEM_LANGUAGE_SELECTION);
      std::string strMsgAloud = ("\x1B\\tn=sms\\ ") + rMessageDetails.copcShortMessage;
      _MsgProxy->sendReadMessageAloudExtendedStart(*this, e8PlaybackAction, strMsgAloud);
   }
   else
   {
      ETG_TRACE_ERR(("ServiceMsg::_MsgProxy is NULL"));
   }

   return TRUE;
}


/************************************************************************
*FUNCTION: 		 MessageDialReq
*DESCRIPTION: 	 Handles request from HMI to call the sender of the message
*
*PARAMETER:
* 					 corfoMsg:	(I)
* 						Courier message received
*
*RETURNVALUE: 	 tBool
*						TRUE, if the message is consumed
*						FALSE, if the message should be forwarded
*
*HISTORY:
*revision 0.1	ess7kor	11/08/2017
*************************************************************************/
bool ServiceMsg::onCourierMessage(const MessageDialReq& oMsg)
{
   ETG_TRACE_USR3(("ServiceMsg::MessageDialReq(): Received MessageDialReq"));

   btphn_tenECNRSettings corfenECNRStatus = BTPHN_EN_ECNR_NOCHANGE;
   ServiceTel* poServiceTel = ServiceTel::poGetInstance();
   if (poServiceTel)
   {
      std::string strTelephoneNo = rMessageDetails.copcPhoneNumber.c_str();
      //tVoid vSendDialCallMethodStart(std::string strTelephoneNo, btphn_tenECNRSettings enECNRStatus);
      poServiceTel->vSendDialCallMethodStart(strTelephoneNo, corfenECNRStatus);
   }
   else
   {
      ETG_TRACE_ERR(("ServiceMsg::poServiceBT is NULL"));
   }

   return TRUE;
}


/************************************************************************
*FUNCTION: 		 DeviceHandleResp
*DESCRIPTION: 	 Handles request from BTService to update the Connected DeviceHandle update
*
*PARAMETER:
* 					 corfoMsg:	(I)
* 						Courier message received
*
*RETURNVALUE: 	 tBool
*						TRUE, if the message is consumed
*						FALSE, if the message should be forwarded
*
*HISTORY:
*revision 0.1	ess7kor	11/08/2017
*************************************************************************/
bool ServiceMsg::onCourierMessage(const DeviceHandleResp& oMsg)
{
   ETG_TRACE_USR3(("ServiceMsg::DeviceHandleResp(): Received DeviceHandleResp"));

   tU8 u8DeviceHandle = oMsg.GetDeviceHandle();
   ETG_TRACE_USR4(("Connected device handle = %d", u8DeviceHandle));

   if (u8ConnectedDeviceHandle != u8DeviceHandle)
   {
      u8ConnectedDeviceHandle = u8DeviceHandle;
      vDeleteMsgList();
   }

   return FALSE;
}


void ServiceMsg::vOnChangeSMSSettings()
{
   m_bShowMessageStatus = !m_bShowMessageStatus;
   ETG_TRACE_USR4(("ServiceMsg::ChangeSMSSettingsReq : m_bShowMessageStatus = %d", m_bShowMessageStatus));

   if (PhoneDataPoolConfig::getInstance())
   {
      PhoneDataPoolConfig::getInstance()->setMessageSettings(m_bShowMessageStatus);
      //ETG_TRACE_USR3(("PhoneDataPoolConfig Message Settngs is set to %d", ETG_CENUM(tBool, PhoneDataPoolConfig::getInstance()->getMessageSettings())));
   }
   rfoMSGModelComponent.OnUpdChangeSMSSettingsReq(m_bShowMessageStatus);
}


}
}
