/**************************************************************************************
* @file         : NaviDataPoolConfig.h
* @author       : ECG5-Atchuta Sashank Kappagantu
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) -2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#ifndef NAVIDATAPOOLCONFIG_H_
#define NAVIDATAPOOLCONFIG_H_

#define ECO_DRIVE_REPORT_AVAILABLE 1

#include "AppHmi_NavigationTypes.h"

#ifdef DP_DATAPOOL_ID

#include "hall_std_if.h"
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_generic_if.h"
#include "dp_hmi_03_if.h"
#include "App/DataModel/Common/Util/HMIModelIncludes.h"
#include "NaviDataPoolConsts.h"

class dp_tclAppHmi_NavigationLanguageIndex;


/**
 * Hold implementation related to Datapool elements. It provides interfaces to read/write DP elements
 */
class NaviDataPoolConfig
{
   public:
      /**
       * Singleton Class. Method to retrieve the instance of the class
       * @return Returns instance of the class
       */
      static NaviDataPoolConfig* getInstance();

      /** Method to delete instance of the class */
      static void deleteInstance();

      void setDpVDLanguageIndexSystem(uint8 languageIndexSystem);
      void setDpVDLanguageIndexSds(uint8 languageIndexSds);
      void setDpVDLanguageIndexCluster(uint8 languageIndexCluster);
      void setDpVDLanguageIndexTrafficText(uint8 languageIndexTrafficText);

      void setDpLanguageIndexSystem(uint8 languageIndexSystem);
      void setDpLanguageISOCodeSystem(std::string languageISOCodeSystem);
      void setDpLanguageIndexSds(uint8 languageIndexSds);
      void setDpLanguageISOCodeSds(std::string languageISOCodeSds);
      void setDpLanguageIndexCluster(uint8 languageIndexCluster);
      void setDpLanguageISOCodeCluster(std::string languageISOCodeCluster);
      void setDpLanguageIndexTrafficText(uint8 languageIndexTrafficText);
      void setDpLanguageISOCodeTrafficText(std::string languageISOCodeTrafficText);

      void setDpCoachWeight(float coachWeight);
      void setDpCoachLoadPerAxle(float coachLoadPerAxle);

      void setDpLightMode(enLightMode lightMode);
      void setDpDateFormat(uint8 dateFormat);
      void setDpTimeFormat(enTimeFormat timeFormat);
      void setDpLocalTimeOffset(int localTimeOffset);
      void setDpClockAutoModeActivated(bool clockAutoModeActivated);
      void setDpEstimatedTimeMode(unsigned char estimatedTimeMode);
      void setDpEstimatedTimeType(unsigned char estimatedTimeType);

      void setDpDistanceUnitSystem(unsigned char distanceUnitSystem);

      /**
      * Setter function to write the status of guidance
      * @param[in] bool is guidance active
      */
      void setDpIsGuidanceActive(bool isGuidanceActive);

      /**
      * Setter function to write the setting of Lat/long input
      */
      void setDpLatitudeDirection(enLatitudeCoordinatesDirection latitudeDirection);
      void setDpLongitudeDirection(enLongitudeCoordinatesDirection longitudeDirection);
      void setDpCoordinatiateDisplayFormat(enCoordinatesDisplayType coordinateDisplayFormat);

      void setMyPOIsAvailability(bool isAvailable);
      void setSatelliteSource(uint8_t satelliteSource);

      void setDpVehicleProfile(uint8 vehicleProfile);
      void setDpProfileAndGuidanceStateLastWrittenLineNo(int profileAndGuidanceStateLastWrittenLineNo);
      void setDpShowETAOnSecMap(bool isETASettingsEnabled);

      /**
      * Getter function to read lat,long and coordinatiate display format
      */
      enLatitudeCoordinatesDirection getDpLatitudeDirection();
      enLongitudeCoordinatesDirection getDpLongitudeDirection();
      enCoordinatesDisplayType getDpCoordinatiateDisplayFormat();

      uint8 getDpLanguageIndexSystem();
      uint8 getDpVDLanguageIndexSystem();
      std::string getDpLanguageISOCodeSystem();
      uint8 getDpLanguageIndexSds();
      uint8 getDpVDLanguageIndexSds();
      std::string getDpLanguageISOCodeSds();
      uint8 getDpLanguageIndexCluster();
      uint8 getDpVDLanguageIndexCluster();
      std::string getDpLanguageISOCodeCluster();
      uint8 getDpLanguageIndexTrafficText();
      uint8 getDpVDLanguageIndexTrafficText();
      std::string getDpLanguageISOCodeTrafficText();
      float getDpCoachWeight();
      float getDpCoachLoadPerAxle();

      enRegionType getDpRegionType();

      enLightMode getDpLightMode();
      uint8 getDpDateFormat();
      enTimeFormat getDpTimeFormat();
      int getDpLocalTimeOffset();
      bool getDpClockAutoModeActivated();
      unsigned char getDpEstimatedTimeMode();
      unsigned char getDpEstimatedTimeType();

      unsigned char getDpDistanceUnitSystem();

      /* default factory configurable position */
      float getDpDefaultPositionLongitude() const;
      float getDpDefaultPositionLatitude() const;
      float getDpDefaultPositionHeading() const;

      /* unit identifier, unique for every device*/
      //std::string getDpNavigationUnitId() const;

      /**
       * Getter function to read the status of guidance
       * @return bool Status of guidance
       */
      bool getDpIsGuidanceActive();

      // Vehicle related information
      enVehicleType getKDSVehicleType();
      uint16 getKDSHeightOfVehicle();
      uint16 getKDSLengthOfVehicle();
      uint16 getKDSWidthOfVehicle();
      uint16 getKDSWeightOfVehicle();
      uint8 getKDSTrailerType();
      uint8 getKDSNumberOfAxes();
      uint16 getKDSMaximumLoadPerAxle();

      uint8 getKDSSatelliteSource();

      enVariantSelection getVariantSelectionType();
      bool getMyPOIsAvailability();

      uint8 getDpVehicleProfile();
      int getDpProfileAndGuidanceStateLastWrittenLineNo();
      bool getDpShowETAOnSecMap();

      /*
      unsigned int getTcuConfig();

      unsigned int getDisplayAspectRatio(); */

      /**
      * Gets gyro mounting angle configuration
      * @param angleRX on succesfull return will contain configuration value for angle RX
      * @param angleRY on succesfull return will contain configuration value for angle RY
      * @param angleRZ on succesfull return will contain configuration value for angle RZ
      * @param angleSX on succesfull return will contain configuration value for angle SX
      * @param angleSY on succesfull return will contain configuration value for angle SY
      * @param angleSZ on succesfull return will contain configuration value for angle SZ
      * @param angleTX on succesfull return will contain configuration value for angle TX
      * @param angleTY on succesfull return will contain configuration value for angle TY
      * @param angleTZ on succesfull return will contain configuration value for angle TZ
      * @return Returns true if all mounting angle parameters have been succesfully fetched
      */
      /* bool getGyroMountingAngleConfiguration(
         std::string& angleRX,
         std::string& angleRY,
         std::string& angleRZ,
         std::string& angleSX,
         std::string& angleSY,
         std::string& angleSZ,
         std::string& angleTX,
         std::string& angleTY,
         std::string& angleTZ); */

   private:
      /** Method to only create an instance */
      static void createInstance();

      /** Destructor of class DataPoolConfig */
      ~NaviDataPoolConfig();

      /** Constructor of class DataPoolConfig */
      NaviDataPoolConfig();

      /**
       * Generalized utility function for getting configuration item values of type uint8 as string from
       * specific configuration category.
       *
       * @todo: sav1szh: Depending on future needs, consider exposing this as public function so that extensive
       *                 amount of configuration items can be fetched through single function (using shared keys).
       *
       * @param category configuration category
       * @param configurationItem configuration item key in category
      */
      /* bool getUInt8ConfigurationValueAsString(
         const std::string& category,
         const std::string& configurationItem,
         std::string& value); */

      /** Instance of the class */
      static NaviDataPoolConfig* _dpNavi;

      /** Datapool elements */

      /** Language Indexes from Vehicle */
      dp_tclAppHmi_NavigationVDLanguageIndexSystem _dpVDLanguageIndexSystem;
      dp_tclAppHmi_NavigationVDLanguageIndexSds _dpVDLanguageIndexSds;
      dp_tclAppHmi_NavigationVDLanguageIndexCluster _dpVDLanguageIndexCluster;
      dp_tclAppHmi_NavigationVDLanguageIndexTrafficText _dpVDLanguageIndexTrafficText;

      /** languages relevant to navigation */
      dp_tclAppHmi_NavigationLanguageIndexSystem _dpLanguageIndexSystem;
      dp_tclAppHmi_NavigationLanguageISOCodeSystem _dpLanguageISOCodeSystem;
      dp_tclAppHmi_NavigationLanguageNaviRelevantFromSds _dpLanguageIndexSds;
      dp_tclAppHmi_NavigationLanguageIsoCodeNaviRelevantFromSds _dpLanguageISOCodeSds;
      dp_tclAppHmi_NavigationLanguageNaviRelevantFromCluster _dpLanguageIndexCluster;
      dp_tclAppHmi_NavigationLanguageIsoCodeNaviRelevantFromCluster _dpLanguageISOCodeCluster;
      dp_tclAppHmi_NavigationLanguageNaviRelevantFromTrafficText _dpLanguageIndexTrafficText;
      dp_tclAppHmi_NavigationLanguageIsoCodeNaviRelevantFromTrafficText _dpLanguageISOCodeTrafficText;

      /** Coach Weight and Load per axle */
      dp_tclAppHmi_NavigationCoachWeight _dpCoachWeight;
      dp_tclAppHmi_NavigationCoachLoadPerAxle _dpCoachLoadPerAxle;

      /** Guidance State*/
      dp_tclAppHmi_NavigationIsGuidanceActive _dpIsGuidanceActive;

      /** Light Mode */
      dp_tclAppHmi_NavigationLightMode _dpLightMode;

      /** Estimated time mode setting in route preference */
      dp_tclAppHmi_NavigationEstimatedTimeMode _dpEstimatedTimeMode;
      dp_tclAppHmi_NavigationEstimatedTimeType _dpEstimatedTimeType;

      /** Date/Time formats */
      dp_tclAppHmi_NavigationDateFormat _dpDateFormat;
      dp_tclAppHmi_NavigationTimeFormat _dpTimeFormat;

      /** Local time offset */
      dp_tclAppHmi_NavigationLocalTimeOffset _dpLocalTimeOffset;

      /** Clock auto mode state */
      dp_tclAppHmi_NavigationClockAutoModeActivated _dpClockAutoModeActivated;

      /** Distance Unit System configuration */
      dp_tclAppHmi_NavigationDistanceUnitSystem _dpDistanceUnitSystem;

      dp_tclAppHmi_NavigationLatitudeCoordinatesDirection _dpLatitudeCoordinatesDirection;
      dp_tclAppHmi_NavigationLongitudeCoordinatesDirection _dpLongitudeCoordinatesDirection;
      dp_tclAppHmi_NavigationCoordinateDisplayFormat _dpCoordinateDisplayFormat;
      dp_tclAppHmi_NavigationMyPOIsAvailability _dpMyPOIsAvailability;

      /** Current vehicle profile */
      dp_tclAppHmi_NavigationVehicleProfile _dpVehicleProfile;
      dp_tclAppHmi_NavigationProfileAndGuidanceStateLastWrittenLineNo _dpProfileAndGuidanceStateLastWrittenLineNo;

      /** ETA settings to show/hide ETA and Duration on Mapout */
      dp_tclAppHmi_NavigationShowETAOnSecMap _dpShowETAOnSecMap;
      void readKDSEntry(tU16 u16KdsKey, tU16 u16Len, tU8* u8Buf) const;
};


#else

#include "sys_std_if.h"
#include "App/DataModel/Common/Util/HMIModelIncludes.h"
#include "App/DataModel/Common/Util/EnvironmentUtils.h"
#include "NaviDataPoolConsts.h"
typedef unsigned char uint8;

// SystemLanguage is defined in FI-catalogue "\nincg3_GEN\ai_projects\generated\components\vehicle_fi\types\vehicle_main_fi_types.xml"
const int SYSTEM_DEFAULT_LANGUAGE = 13; /*T_e8_Language_Code__English_US*/

// MiddlewareLanguages are defined in "\navi_development\nav_business_ctrl\pres_ctrl\aivi_pres_ctrl\src\navmiddleware\info\InfoTypes.h"
const int SYSTEM_DEFAULT_MW_LANGUAGE_CODE = 15; /*LANGUAGE_CODE__ENGLISH_US*/

// SystemLanguage ISoCode is defined "\nincg3\ai_nissan_hmi\products\NINCG3\Apps\Common\VehicleDataHandler\VehicleDataHandler.cpp"
const char  SYSTEM_DEFAULT_LANGUAGE_ISOCODE[] = "en_US"; /*en_US*/

/* KDS values for truck and coach */
const enVehicleType KDS_VEHICLE_TYPE = TRUCK;
const unsigned int KDS_TRUCK_HEIGTH_OF_VEHICLE        = 4000;    // vehicle dimensions in mm
const unsigned int KDS_TRUCK_LENGTH_OF_VEHICLE        = 12000;
const unsigned int KDS_TRUCK_WIDTH_OF_VEHICLE         = 2550;
const unsigned int KDS_TRUCK_WEIGHT_OF_VEHICLE        = 36287;   // 40 ton. Converted to kg

const unsigned int KDS_COACH_HEIGTH_OF_VEHICLE        = 3700;    // vehicle dimensions in mm
const unsigned int KDS_COACH_LENGTH_OF_VEHICLE        = 13000;
const unsigned int KDS_COACH_WIDTH_OF_VEHICLE         = 2550;
const unsigned int KDS_COACH_WEIGHT_OF_VEHICLE        = 36287;   // 40 ton. Converted to kg

class NaviDataPoolConfig
{
   public:
      static NaviDataPoolConfig* getInstance()
      {
         if (_dpNavi == NULL)
         {
            _dpNavi = new NaviDataPoolConfig();
         }
         return _dpNavi;
      }

      static void deleteInstance()
      {
         if (_dpNavi != NULL)
         {
            delete _dpNavi;
            _dpNavi = NULL;
         }
      }

      void setDpVDLanguageIndexSystem(uint8) {}
      void setDpVDLanguageIndexSds(uint8) {}
      void setDpVDLanguageIndexCluster(uint8) {}
      void setDpVDLanguageIndexTrafficText(uint8) {}

      void setDpLanguageIndexSystem(uint8 languageReceived)
      {
         _languageIndexSystem = languageReceived;
      }

      void setDpLanguageISOCodeSystem(std::string languageISOCodeSystem)
      {
         _languageISOCodeSystem = languageISOCodeSystem;
      }

      void setDpLanguageIndexHmi(uint8 languageNaviRelevantFromHmi)
      {
         _languageIndexHmi = languageNaviRelevantFromHmi;
      }

      void setDpLanguageISOCodeHmi(std::string languageISOCodeHmi)
      {
         _languageISOCodeHmi = languageISOCodeHmi;
      }

      void setDpLanguageIndexSds(uint8 languageNaviRelevantFromSds)
      {
         _languageIndexSds = languageNaviRelevantFromSds;
      }

      void setDpLanguageISOCodeSds(std::string languageISOCodeSds)
      {
         _languageISOCodeSds = languageISOCodeSds;
      }

      void setDpLanguageIndexCluster(uint8 languageNaviRelevantFromCluster)
      {
         _languageIndexCluster = languageNaviRelevantFromCluster;
      }

      void setDpLanguageISOCodeCluster(std::string languageISOCodeCluster)
      {
         _languageISOCodeCluster = languageISOCodeCluster;
      }

      void setDpLanguageIndexTrafficText(uint8 languageNaviRelevantFromTrafficText)
      {
         _languageIndexTrafficText = languageNaviRelevantFromTrafficText;
      }

      void setDpLanguageISOCodeTrafficText(std::string languageISOCodeTrafficText)
      {
         _languageISOCodeTrafficText = languageISOCodeTrafficText;
      }

      void setDpIsGuidanceActive(bool isGuidanceActive)
      {
         _isGuidanceActive = isGuidanceActive;
      }

      void setDpLightMode(enLightMode lightMode)
      {
         _lightMode = (uint8)lightMode;
      }
      void setDpDateFormat(uint8 dateFormat)
      {
         _dateFormat = dateFormat;
      }

      void setDpTimeFormat(enTimeFormat timeFormat)
      {
         _timeFormat = (uint8)timeFormat;
      }

      void setDpLocalTimeOffset(int localTimeOffset)
      {
         _localTimeOffset = localTimeOffset;
      }

      void setDpEstimatedTimeMode(unsigned char estimatedTimeMode) // Estimated Time mode setting
      {
         _estimatedTimeMode = estimatedTimeMode;
      }

      void setDpEstimatedTimeType(unsigned char estimatedTimeType) // Estimated Time mode setting
      {
         _estimatedTimeType = estimatedTimeType;
      }

      void setDpClockAutoModeActivated(bool clockAutoModeActivated)
      {
         _clockAutoModeActivated = clockAutoModeActivated;
      }

      void setDpDistanceUnitSystem(unsigned char distanceUnitSystem)
      {
         _distanceUnitSystem = distanceUnitSystem;
      }

      void setDpLatitudeDirection(enLatitudeCoordinatesDirection latitudeDirection)
      {
         _latitudeDirection = (tU8)latitudeDirection;
      }

      void setDpLongitudeDirection(enLongitudeCoordinatesDirection longitudeDirection)
      {
         _longitudeDirection = (tU8)longitudeDirection;
      }

      void setDpCoordinatiateDisplayFormat(enCoordinatesDisplayType coordinateDisplayFormat)
      {
         _coordinateDisplayFormat = (tU8)coordinateDisplayFormat;
      }

      void setDpVehicleProfile(uint8 vehicleProfile)
      {
         _vehicleProfile = vehicleProfile;
      }

      void setDpShowETAOnSecMap(bool isETASettingsEnabled)
      {
         _isETASettingsEnabled = isETASettingsEnabled;
      }

      void setDpProfileAndGuidanceStateLastWrittenLineNo(int profileAndGuidanceStateLastWrittenLineNo)
      {
         _profileAndGuidanceStateLastWrittenLineNo = profileAndGuidanceStateLastWrittenLineNo;
      }

      enLatitudeCoordinatesDirection getDpLatitudeDirection()
      {
         return (enLatitudeCoordinatesDirection)_latitudeDirection;
      }

      enLongitudeCoordinatesDirection getDpLongitudeDirection()
      {
         return (enLongitudeCoordinatesDirection)_longitudeDirection;
      }

      enCoordinatesDisplayType getDpCoordinatiateDisplayFormat()
      {
         return (enCoordinatesDisplayType)_coordinateDisplayFormat;
      }

      uint8 getDpVDLanguageIndexSystem()
      {
         return 0;
      }
      uint8 getDpVDLanguageIndexSds()
      {
         return 0;
      }
      uint8 getDpVDLanguageIndexCluster()
      {
         return 0;
      }
      uint8 getDpVDLanguageIndexTrafficText()
      {
         return 0;
      }

      uint8 getDpLanguageIndexSystem()
      {
         _languageIndexSystem = (uint8)getEnvVarAsInt(ENV_HMI_DP_LANGUAGE_INDEX_SYSTEM, SYSTEM_DEFAULT_LANGUAGE);
         return _languageIndexSystem;
      }

      std::string getDpLanguageISOCodeSystem()
      {
         return getEnvVarAsString(ENV_HMI_DP_LANGUAGE_ISOCODE_SYSTEM, SYSTEM_DEFAULT_LANGUAGE_ISOCODE);
      }

      float NaviDataPoolConfig::getDpCoachWeight()
      {
         return _coachWeight;
      }

      float NaviDataPoolConfig::getDpCoachLoadPerAxle()
      {
         return _coachLoadPerAxle;
      }

      uint8 NaviDataPoolConfig::getDpLanguageIndexSds()
      {
         _languageIndexSds = (uint8)getEnvVarAsInt(ENV_HMI_DP_LANGUAGE_INDEX_SDS, SYSTEM_DEFAULT_MW_LANGUAGE_CODE);
         return _languageIndexSds;
      }

      std::string getDpLanguageISOCodeSds()
      {
         return getEnvVarAsString(ENV_HMI_DP_LANGUAGE_ISOCODE_SDS, SYSTEM_DEFAULT_LANGUAGE_ISOCODE);
      }

      uint8 NaviDataPoolConfig::getDpLanguageIndexCluster()
      {
         _languageIndexCluster = (uint8)getEnvVarAsInt(ENV_HMI_DP_LANGUAGE_INDEX_CLUSTER, SYSTEM_DEFAULT_MW_LANGUAGE_CODE);
         return _languageIndexCluster;
      }

      std::string getDpLanguageISOCodeCluster()
      {
         return getEnvVarAsString(ENV_HMI_DP_LANGUAGE_ISOCODE_CLUSTER, SYSTEM_DEFAULT_LANGUAGE_ISOCODE);
      }

      uint8 NaviDataPoolConfig::getDpLanguageIndexTrafficText()
      {
         _languageIndexTrafficText = (uint8)getEnvVarAsInt(ENV_HMI_DP_LANGUAGE_INDEX_TRAFFICTEXT, SYSTEM_DEFAULT_MW_LANGUAGE_CODE);
         return _languageIndexTrafficText;
      }

      std::string getDpLanguageISOCodeTrafficText()
      {
         return getEnvVarAsString(ENV_HMI_DP_LANGUAGE_ISOCODE_TRAFFICTEXT, SYSTEM_DEFAULT_LANGUAGE_ISOCODE);
      }

      bool NaviDataPoolConfig::getDpIsGuidanceActive()
      {
         _isGuidanceActive = getEnvVarAsBool(ENV_HMI_DP_GUIDANCE_ACTIVE, _isGuidanceActive);
         if (_isGuidanceActive)
         {
            EXT_bIsGuidanceActive = TRUE;
         }
         else
         {
            EXT_bIsGuidanceActive = FALSE;
         }
         return _isGuidanceActive;
      }

      enRegionType getDpRegionType()
      {
         return (enRegionType)getEnvVarAsInt(ENV_HMI_DP_REGION_TYPE, OTH_EUR);
      }

      unsigned int getTcuConfig()
      {
         return _tcuConfig;
      }

      float getDpDefaultPositionLongitude()
      {
         return getEnvVarAsFloat(ENV_HMI_DP_DEFAULT_POSITION_LON, 9.891959f); //Building 500 HI, Germany
      }

      float getDpDefaultPositionLatitude()
      {
         return getEnvVarAsFloat(ENV_HMI_DP_DEFAULT_POSITION_LAT, 52.117013f);
      }

      float getDpDefaultPositionHeading()
      {
         return getEnvVarAsFloat(ENV_HMI_DP_DEFAULT_POSITION_HEADING, 0.f);
      }

      std::string getDpNavigationUnitId()
      {
         return getEnvVarAsString(ENV_HMI_DP_NAVIGATION_UNIT_ID, "");
      }

      enLightMode getDpLightMode()
      {
         _lightMode = (uint8)getEnvVarAsInt(ENV_HMI_DP_LIGHT_MODE, 0);
         return (enLightMode)_lightMode;
      }

      uint8 getDpDateFormat()
      {
         _dateFormat = (uint8)getEnvVarAsInt(ENV_HMI_DP_DATE_FORMAT, 0);
         return _dateFormat;
      }

      enTimeFormat getDpTimeFormat()
      {
         _timeFormat = (uint8)getEnvVarAsInt(ENV_HMI_DP_TIME_FORMAT, 0);
         return (enTimeFormat)_timeFormat;
      }

      int getDpLocalTimeOffset()
      {
         _localTimeOffset = getEnvVarAsInt(ENV_HMI_DP_LOCALTIME_OFFSET, 0);
         return _localTimeOffset;
      }

      unsigned char getDpEstimatedTimeMode()
      {
         return  _estimatedTimeMode;
      }

      unsigned char getDpEstimatedTimeType()
      {
         return  _estimatedTimeType;
      }
      bool getDpClockAutoModeActivated()
      {
         _clockAutoModeActivated = getEnvVarAsBool(ENV_HMI_DP_CLOCK_AUTOMODE_ON, true);
         return _clockAutoModeActivated;
      }

      // Vehicle related information
      enVehicleType getKDSVehicleType()
      {
         return TRUCK;   // Vehicle type corresponds to truck
      }
      unsigned int getKDSHeightOfVehicle()
      {
         unsigned int height = (TRUCK == getKDSVehicleType()) ? KDS_TRUCK_HEIGTH_OF_VEHICLE : KDS_COACH_HEIGTH_OF_VEHICLE;
         return height;
      }
      unsigned int getKDSLengthOfVehicle()
      {
         unsigned int length = (TRUCK == getKDSVehicleType()) ? KDS_TRUCK_LENGTH_OF_VEHICLE : KDS_COACH_LENGTH_OF_VEHICLE;
         return length;
      }
      unsigned int getKDSWidthOfVehicle()
      {
         unsigned int width = (TRUCK == getKDSVehicleType()) ? KDS_TRUCK_WIDTH_OF_VEHICLE : KDS_COACH_WIDTH_OF_VEHICLE;
         return width;
      }
      unsigned int getKDSWeightOfVehicle()
      {
         unsigned int weight = (TRUCK == getKDSVehicleType()) ? KDS_TRUCK_WEIGHT_OF_VEHICLE : KDS_COACH_WEIGHT_OF_VEHICLE;
         return weight;
      }

      enVariantSelection getVariantSelectionType()
      {
         return (enVariantSelection)getEnvVarAsInt(ENV_HMI_DP_VEHICLE_TYPE, NAVI);
      }

      unsigned char getDpDistanceUnitSystem()
      {
         return (unsigned char)getEnvVarAsInt(ENV_HMI_DP_DISTANCEUNITSYSTEM, _distanceUnitSystem);
      }

      /** Gyro mounting angle configuration */
      bool getGyroMountingAngleConfiguration(
         std::string& /*angleRX*/,
         std::string& /*angleRY*/,
         std::string& /*angleRZ*/,
         std::string& /*angleSX*/,
         std::string& /*angleSY*/,
         std::string& /*angleSZ*/,
         std::string& /*angleTX*/,
         std::string& /*angleTY*/,
         std::string& /*angleTZ*/)
      {
         return true;
      }

      uint8 getDpVehicleProfile()
      {
         return _vehicleProfile;
      }

      int getDpProfileAndGuidanceStateLastWrittenLineNo()
      {
         return _profileAndGuidanceStateLastWrittenLineNo;
      }

      bool getDpShowETAOnSecMap()
      {
         return _isETASettingsEnabled;
      }

   private:
      ~NaviDataPoolConfig() {}
      NaviDataPoolConfig()
      {
         _languageIndexSystem = 0;
         _languageIndexHmi = 0;
         _languageIndexSds = 0;
         _languageIndexCluster = 0;
         _lightMode = 0;
         _dateFormat = 0;
         _timeFormat = 0;
         _vehicleProfile = 0;
         _profileAndGuidanceStateLastWrittenLineNo = 0;
         _localTimeOffset = 0;
         _clockAutoModeActivated = false;
         _isGuidanceActive = false;
         _isETASettingsEnabled = true;
         _tcuConfig = 0;
         _estimatedTimeMode = 0;
         _estimatedTimeType = 0;
         _distanceUnitSystem = 0;
         _latitudeDirection = 0;
         _longitudeDirection = 0;
         _coordinateDisplayFormat = 0;
         _coachWeight = 0.0;
         _coachLoadPerAxle = 0.0;
      }

      static NaviDataPoolConfig* _dpNavi;

      uint8 _languageIndexSystem;
      std::string _languageISOCodeSystem;
      uint8 _languageIndexHmi;
      std::string _languageISOCodeHmi;
      uint8 _languageIndexSds;
      std::string _languageISOCodeSds;
      uint8 _languageIndexCluster;
      std::string _languageISOCodeCluster;
      uint8 _languageIndexTrafficText;
      std::string _languageISOCodeTrafficText;
      float _coachWeight;
      float _coachLoadPerAxle;

      uint8 _lightMode;
      uint8 _dateFormat;
      uint8 _timeFormat;
      uint8 _vehicleProfile;
      int _carModeCountHistory;
      int _localTimeOffset;
      bool _clockAutoModeActivated;
      bool _isGuidanceActive;
      bool _isETASettingsEnabled;
      bool _tcuConfig;
      unsigned char _estimatedTimeMode;
      unsigned char _estimatedTimeType;
      unsigned char _distanceUnitSystem;
      tU8 _latitudeDirection;
      tU8 _longitudeDirection;
      tU8 _coordinateDisplayFormat;
};


#endif /* DP_DATAPOOL_ID */
#endif /* NAVIDATAPOOLCONFIG_H_ */
