/**************************************************************************************
* @file         : TrafficListHandler.h
* @author       : KanagaDurga Balakrishnan
* @addtogroup   :
* @brief        :
* @copyright    : (c) 2018-2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include "util/StrUtf8.h"
#include "AppHmi_NavigationStateMachine.h"
#include "TrafficListHandler.h"
#include "hmi_trace_if.h"
#include "CgiExtensions/ImageLoader.h"
#include "Common/Util/StringUtils.h"
#include "POI/DestinationDataUtils.h"   // For direction icons

//List Identifier
static const char* const DATA_CONTEXT_LIST_TMC_ROUTE = "TMC_Route_ListItem";
static const Candera::String NO_TRAFFIC_MESSAGES_TXT = LANGUAGE_STRING(TextId_0x1A5A, "No traffic messages are available");
static const Candera::String DISTANCE_STRING = LANGUAGE_STRING(TextId_0x1974, "Distance:");
static const Candera::String TIME_STRING = LANGUAGE_STRING(TextId_0x1973, "Time:");
static const Candera::String USE_DETOUR_STRING = LANGUAGE_STRING(TextId_0x1975, "Use detour in:");
static const std::string SPACE_STRING = " ";
static const unsigned int DELAY_TIME_START_TRAFFIC_DETOUR = 60000;

static const Courier::Identifier  IdClickRefreshOnRoute = Courier::Identifier("Btn_Refresh_TMC_OnRoute/ButtonWidget");
static const Courier::Identifier  IdClickRefreshActiveRG = Courier::Identifier("Btn_Refresh_TMC_ActiveRG/ButtonWidget");
static const Courier::Identifier  IdClickRefreshNoRG = Courier::Identifier("Btn_Refresh_TMC_No_RG/ButtonWidget");

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/TrafficListHandler.cpp.trc.h"
#endif

using namespace navmiddleware;

TrafficListHandler::TrafficListHandler(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore)
   : HMIModelBase(navMiddleware, infoStore)
   , _startIndex(0)
   , _windowElementSize(0)
   , _listID(0)
   , _xUrgentTrafficMsgId(0)
   , _isTrafficDetourDisplayed(false)
{
}


TrafficListHandler::~TrafficListHandler()
{
}


/********************************************
/* NAME    : TrafficListHandler
/* SYSFL    : null,
****************************************************/
void TrafficListHandler::initialize()
{
   _infoStore.registerDataPropertyUpdateCallback(*this);
   _navMiddleware.registerTrafficMessagePropertyUpdateCallback(*this);
   _navMiddleware.registerRoutePropertyUpdateCallback(*this);
   _navMiddleware.registerGuidancePropertyUpdateCallback(*this);
}


/********************************************
/* NAME    : TrafficListHandler
/* SYSFL    : null,
****************************************************/
void TrafficListHandler::deinitialize()
{
   _infoStore.unregisterDataPropertyUpdateCallback(*this);
   _navMiddleware.unregisterTrafficMessagePropertyUpdateCallback(*this);
   _navMiddleware.unregisterRoutePropertyUpdateCallback(*this);
   _navMiddleware.unregisterGuidancePropertyUpdateCallback(*this);
}


/********************************************
/* NAME    : TrafficListHandler
/* SYSFL    : null,
****************************************************/
bool TrafficListHandler::onCourierMessage(const ListDateProviderReqMsg& oMsg)
{
   int listId = oMsg.GetListId();
   if ((listId == LIST_ID_TMC_ONROUTE) || (listId == LIST_ID_TMC_AllACTIVE_RG) || (listId == LIST_ID_TMC_AllNO_RG))
   {
      _listID = listId;
      _startIndex = (unsigned int)oMsg.GetStartIndex();
      _windowElementSize = oMsg.GetWindowElementSize();
      ETG_TRACE_USR4(("TMCListHandler::onCourierMessage(ListDateProviderReqMsg), listID : %d, start index : %d, window size : %d", _listID, _startIndex, _windowElementSize));

      navmiddleware::TrafficMessageListType trafficMessageListType = ::navmiddleware::TRAFFICMESSAGELIST_TYPE__ALONG_CURRENT_ROUTE;
      if ((listId == LIST_ID_TMC_AllACTIVE_RG) || (listId == LIST_ID_TMC_AllNO_RG))
      {
         trafficMessageListType = ::navmiddleware::TRAFFICMESSAGELIST_TYPE__ALL_TM;
      }

      _navMiddleware.requestTrafficMessageList(_startIndex, _windowElementSize, trafficMessageListType, TRAFFICMESSAGE_TEXT_TYPE__LIST);
      return true;
   }

   return false;
}


/********************************************
/* NAME    : TrafficListHandler
/* SYSFL    : null,
****************************************************/
bool TrafficListHandler::onCourierMessage(const ButtonReactionMsg& oMsg)
{
   bool isMessageProcessed = false;
   if (oMsg.GetEnReaction() == enRelease)
   {
      ListProviderEventInfo info;
      const Courier::Identifier senderInfo = oMsg.GetSender();
      if ((ListProviderEventInfo::GetItemIdentifierInfo(senderInfo, info)))
      {
         int listId = info.getListId();
         if ((listId == LIST_ID_TMC_ONROUTE) || (listId == LIST_ID_TMC_AllACTIVE_RG) || (listId == LIST_ID_TMC_AllNO_RG))
         {
            ETG_TRACE_USR4(("TMCListHandler::onCourierMessage(ButtonReactionMsg(listID %d, row %d, column %d))", info.getListId(), info.getHdlRow(), info.getHdlCol()));
            requestTrafficMessageDetail((unsigned int)info.getHdlRow());
            isMessageProcessed = true;
         }
      }
      if (senderInfo == IdClickRefreshOnRoute)
      {
         _navMiddleware.requestTrafficMessageList(_startIndex, _windowElementSize, TRAFFICMESSAGELIST_TYPE__ALONG_CURRENT_ROUTE, TRAFFICMESSAGE_TEXT_TYPE__LIST);
         isMessageProcessed = true;
      }
      else if ((senderInfo == IdClickRefreshActiveRG) || (senderInfo == IdClickRefreshNoRG))
      {
         _navMiddleware.requestTrafficMessageList(_startIndex, _windowElementSize, TRAFFICMESSAGELIST_TYPE__ALL_TM, TRAFFICMESSAGE_TEXT_TYPE__LIST);
         isMessageProcessed = true;
      }
      else
      {
         //do nothing
      }
   }
   return isMessageProcessed;
}


/********************************************
/* NAME    : TrafficListHandler
/* SYSFL    : SYSFL-3506, SYSFL-3507,
****************************************************/
bool TrafficListHandler::onPropertyUpdateTrafficImpactAlternativeChanged()
{
   ETG_TRACE_USR4(("TrafficListHandler::onPropertyUpdateTrafficImpactAlternativeChanged()"));
   navmiddleware::TrafficImpactRouteAlternativeInfos trafficImpactRouteAlternativeInfos = _navMiddleware.getTrafficImpactRouteAlternativeInfos();
   navmiddleware::TrafficImpactRouteAlternativeInfos::TrafficImpactCause trafficImpactCause = trafficImpactRouteAlternativeInfos.getTrafficImpactCause();

   switch (trafficImpactCause)
   {
      case navmiddleware::TrafficImpactRouteAlternativeInfos::TRAFFIC_IMPACT_CAUSE_BLOCKING_ON_ROUTE:
      case navmiddleware::TrafficImpactRouteAlternativeInfos::TRAFFIC_IMPACT_CAUSE_IMPROVEMENT:
      case navmiddleware::TrafficImpactRouteAlternativeInfos::TRAFFIC_IMPACT_CAUSE_WORSENING:
      {
         ETG_TRACE_USR4(("TrafficListHandler::onPropertyUpdateTrafficImpactAlternativeChanged(NOTCLOSURE)"));
         updateTrafficDetourPopupDetails(trafficImpactRouteAlternativeInfos);
         POST_MSG((COURIER_MESSAGE_NEW(ContextSwitchReqMsg)(CONTEXT_SWITCH_FROM_NONE, CONTEXT_SWITCH_TO_TRAFFIC_DETOUR)));
         break;
      }
      case navmiddleware::TrafficImpactRouteAlternativeInfos::TRAFFIC_IMPACT_CAUSE_INVALID:
         ETG_TRACE_USR4(("TrafficListHandler::onPropertyUpdateTrafficImpactAlternativeChanged(TRAFFIC_IMPACT_CAUSE_INVALID)"));
         if (true == _isTrafficDetourDisplayed)
         {
            POST_MSG((COURIER_MESSAGE_NEW(SwitchContextFromTrafficDetourReqMsg)()));
         }
         break;
      default:
         break;
   }
   return true;
}


void TrafficListHandler::updateTrafficDetourPopupDetails(const navmiddleware::TrafficImpactRouteAlternativeInfos& trafficImpactRouteAlternativeInfos)
{
   ETG_TRACE_USR4(("TrafficListHandler::updateTrafficDetourPopupDetails()"));

   std::string currentString = "";
   (*_trafficDetourDetails).mTimeDifference = TIME_STRING;
   (*_trafficDetourDetails).mDistanceDifference = DISTANCE_STRING;
   (*_trafficDetourDetails).mDistanceToTraffic = USE_DETOUR_STRING;
   (*_trafficDetourDetails).mTrafficIncidentDetails = "";
   (*_trafficDetourDetails).mTrafficIncidentName = "";

   if (!(trafficImpactRouteAlternativeInfos.getTimeDifferenceAsString().empty()))
   {
      std::string timeStr = TIME_STRING.GetCString();
      std::string timeDifferenceSignStr = (trafficImpactRouteAlternativeInfos.getTimeDifference().getValue() > 0) ? "+" : "-";
      currentString = timeStr + SPACE_STRING + timeDifferenceSignStr + trafficImpactRouteAlternativeInfos.getTimeDifferenceAsString();
      (*_trafficDetourDetails).mTimeDifference = currentString.c_str();
      ETG_TRACE_USR4(("TrafficListHandler::updateTrafficDetourPopupDetails() Time Difference (%s)", trafficImpactRouteAlternativeInfos.getTimeDifferenceAsString().c_str()));
   }
   if (!(trafficImpactRouteAlternativeInfos.getDistanceDifferenceAsString().empty()))
   {
      std::string distanceStr = DISTANCE_STRING.GetCString();
      std::string distanceDifferenceSignStr = (trafficImpactRouteAlternativeInfos.getDistanceDifference().getValue() > 0) ? "+" : "-";
      currentString = distanceStr + SPACE_STRING + distanceDifferenceSignStr + trafficImpactRouteAlternativeInfos.getDistanceDifferenceAsString();
      (*_trafficDetourDetails).mDistanceDifference = currentString.c_str();
      ETG_TRACE_USR4(("TrafficListHandler::updateTrafficDetourPopupDetails() Distance Difference (%s)", trafficImpactRouteAlternativeInfos.getDistanceDifferenceAsString().c_str()));
   }
   if (!(trafficImpactRouteAlternativeInfos.getDistanceToTrafficMessageWithUnit().empty()))
   {
      std::string useDetourStr = USE_DETOUR_STRING.GetCString();
      currentString = useDetourStr + SPACE_STRING + trafficImpactRouteAlternativeInfos.getDistanceToTrafficMessageWithUnit();
      (*_trafficDetourDetails).mDistanceToTraffic = currentString.c_str();
      ETG_TRACE_USR4(("TrafficListHandler::updateTrafficDetourPopupDetails() Distance to Traffic (%s)", trafficImpactRouteAlternativeInfos.getDistanceToTrafficMessageWithUnit().c_str()));
   }
   if (!(trafficImpactRouteAlternativeInfos.getTrafficMessageDescription().empty()))
   {
      (*_trafficDetourDetails).mTrafficIncidentDetails = trafficImpactRouteAlternativeInfos.getTrafficMessageDescription().c_str();
      ETG_TRACE_USR4(("TrafficListHandler::updateTrafficDetourPopupDetails() Traffic Message (%s)", trafficImpactRouteAlternativeInfos.getTrafficMessageDescription().c_str()));
   }
   if (!(trafficImpactRouteAlternativeInfos.getTrafficMessageHeadline().empty()))
   {
      (*_trafficDetourDetails).mTrafficIncidentName = trafficImpactRouteAlternativeInfos.getTrafficMessageHeadline().c_str();
      ETG_TRACE_USR4(("TrafficListHandler::updateTrafficDetourPopupDetails() Traffic Incident name (%s)", trafficImpactRouteAlternativeInfos.getTrafficMessageHeadline().c_str()));
   }
   if (trafficImpactRouteAlternativeInfos.getTrafficImage().isValid())
   {
      navmiddleware::Image icon = trafficImpactRouteAlternativeInfos.getTrafficImage().getValue();
      Candera::Bitmap::SharedPointer bitmap = ImageLoader::loadBitmapData(icon.getBlob(), icon.getBlobSize());
      (*_trafficDetourDetails).mTrafficIncidentIcon = ImageLoader::createImage(bitmap);
   }
   _trafficDetourDetails.MarkAllItemsModified();
   _trafficDetourDetails.SendUpdate(true);
}


bool TrafficListHandler::onCourierMessage(const StartTrafficDetourTimerReqMsg& oMsg)
{
   ETG_TRACE_USR4(("TrafficListHandler::onCourierMessage(StartTrafficDetourTimerReqMsg)"));
   COURIER_UNUSED(oMsg);
   _isTrafficDetourDisplayed = true;
   _trafficDetourTimer.setTimeout(0, DELAY_TIME_START_TRAFFIC_DETOUR, COURIER_MESSAGE_NEW(TrafficDetourTimerExpiredReqMsg)());
   _trafficDetourTimer.start();
   return true;
}


bool TrafficListHandler::onCourierMessage(const TrafficDetourTimerExpiredReqMsg& oMsg)
{
   ETG_TRACE_USR4(("TrafficListHandler::onCourierMessage(TrafficDetourTimerExpiredReqMsg)"));
   COURIER_UNUSED(oMsg);
   _isTrafficDetourDisplayed = false;
   _trafficDetourTimer.stop();
   _navMiddleware.rejectTrafficImpactRouteAlternative();
   POST_MSG((COURIER_MESSAGE_NEW(SwitchContextFromTrafficDetourReqMsg)()));
   return true;
}


/********************************************
/* NAME    : TrafficListHandler
/* SYSFL    : SYSFL-3506, SYSFL-3507,
****************************************************/
bool TrafficListHandler::onCourierMessage(const AcceptTrafficAlternativeReqMsg& oMsg)
{
   //selecting OK button on block traffic pop-up
   ETG_TRACE_USR4(("TrafficListHandler::onCourierMessage(AcceptTrafficAlternativeReqMsg)"));
   COURIER_UNUSED(oMsg);
   _isTrafficDetourDisplayed = false;
   _trafficDetourTimer.stop();
   _navMiddleware.acceptTrafficImpactRouteAlternative();
   return true;
}


/********************************************
/* NAME    : TrafficListHandler
/* SYSFL    : SYSFL-3506, SYSFL-3507, SYSFL-3364
****************************************************/
bool TrafficListHandler::onCourierMessage(const RejectTrafficAlternativeReqMsg& oMsg)
{
   //selecting cancel button on block traffic pop-up
   ETG_TRACE_USR4(("TrafficListHandler::onCourierMessage(RejectTrafficAlternativeReqMsg)"));
   COURIER_UNUSED(oMsg);
   _isTrafficDetourDisplayed = false;
   _trafficDetourTimer.stop();
   _navMiddleware.rejectTrafficImpactRouteAlternative();
   return true;
}


/********************************************
/* NAME    : TrafficListHandler
/* SYSFL    : SYSFL-3506, SYSFL-3507, SYSFL-3364
****************************************************/
void TrafficListHandler::onPropertyUpdateGuidanceStateChanged()
{
   bool isGuidanceActive = _navMiddleware.isGuidanceActive();
   ETG_TRACE_USR4(("TrafficListHandler::onPropertyUpdateGuidanceStateChanged(), Is guidance active : %d", isGuidanceActive));

   if ((false == isGuidanceActive) && (true == _isTrafficDetourDisplayed))
   {
      _isTrafficDetourDisplayed = false;
      _trafficDetourTimer.stop();
      POST_MSG((COURIER_MESSAGE_NEW(SwitchContextFromTrafficDetourReqMsg)()));
   }
}


/********************************************
/* NAME    : TrafficListHandler
/* SYSFL    : SYSFL-3503,
****************************************************/
bool TrafficListHandler::onPropertyUpdateTrafficmessageInfoChanged()
{
   ETG_TRACE_USR4(("TrafficListHandler::onPropertyUpdateTrafficmessageInfoChanged()"));

   // just post a ListDateProviderResMsg with our TrafficInfo list data provider
   tSharedPtrDataProvider dataProvider = getTMCListDataProvider(_startIndex, _windowElementSize);
   POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));

   return true;
}


/********************************************
/* NAME    : TrafficListHandler
/* SYSFL    : SYSFL-3504,
****************************************************/
void TrafficListHandler::requestTrafficMessageDetail(unsigned int index)
{
   ETG_TRACE_USR4(("TrafficListHandler::requestTrafficMessageDetail(), Index = %d", index));
   _listID = 0;

   const navmiddleware::TrafficMessageListInfos& tmInfos = _navMiddleware.getTrafficMessageListInfos();
   const std::vector< navmiddleware::TrafficMessageListInfo>& tmInfoVector = tmInfos.getTrafficMessageListInfos();
   if (index < tmInfoVector.size())
   {
      unsigned int tmId = tmInfoVector[index].m_id;
      navmiddleware::ValidValue<int> selectedTrafficMsg;
      selectedTrafficMsg.setValue((int)tmId);
      _infoStore.setTmId(selectedTrafficMsg);
      _navMiddleware.requestTrafficMessageDetails(tmId);
   }
}


/********************************************
/* NAME    : TrafficListHandler
/* SYSFL    : null,
****************************************************/
tSharedPtrDataProvider TrafficListHandler::getTMCListDataProvider(unsigned int listStartIndex, unsigned int windowElementSize)
{
   ListDataProviderBuilder listBuilder(_listID, DATA_CONTEXT_LIST_TMC_ROUTE);
   const navmiddleware::TrafficMessageListInfos& tmInfos = _navMiddleware.getTrafficMessageListInfos();
   unsigned int actTMCListSize = tmInfos.getCountOfAllTrafficMessages();
   ETG_TRACE_USR4(("TrafficListHandler::getTMCListDataProvider(), listID : %d, start index : %d, list size : %d", _listID, listStartIndex, actTMCListSize));

   TrafficMessagesData item;
   unsigned int idx = 0;
   const ::std::vector<navmiddleware::TrafficMessageListInfo>& messages = tmInfos.getTrafficMessageListInfos();

   if ((_listID == LIST_ID_TMC_ONROUTE) || (_listID == LIST_ID_TMC_AllACTIVE_RG) || (_listID == LIST_ID_TMC_AllNO_RG))
   {
      if (0 < actTMCListSize)
      {
         for (::std::vector<navmiddleware::TrafficMessageListInfo>::const_iterator message = messages.begin();
               message != messages.end(); message++)
         {
            item.mTrafficMessageText = message->m_text.c_str();
            item.mDistanceInfo = message->m_distance.c_str();
            if (message->m_trafficIconData.getBlobSize() != 0)
            {
               item.mTrafficIconActiveBitmap = loadImage((const char*)message->m_trafficIconData.getBlob(), message->m_trafficIconData.getBlobSize(), true);
            }
            //To Do:item.mDirectionIconActiveBitmap = message-> (Show traffic icons)
            listBuilder.AddItem(idx,
                                0UL,
                                DATA_CONTEXT_LIST_TMC_ROUTE).AddDataBindingUpdater<TrafficMessagesDataBindingSource>(item);

            idx++;
         }
      }
      else
      {
         item.mTrafficMessageText = NO_TRAFFIC_MESSAGES_TXT;
         listBuilder.AddItem(idx,
                             0UL,
                             DATA_CONTEXT_LIST_TMC_ROUTE).AddDataBindingUpdater<TrafficMessagesDataBindingSource>(item);
      }
   }

   return listBuilder.CreateDataProvider(listStartIndex, actTMCListSize);
}


/********************************************
/* NAME    : TrafficListHandler
/* SYSFL    : ,
****************************************************/
bool TrafficListHandler::onPropertyUpdateTrafficmessageListInvalidate()
{
   ETG_TRACE_USR4(("TrafficListHandler::onPropertyUpdateTrafficmessageListInvalidate()"));

   unsigned int actTMCListSize = 0;
   _startIndex = 0;

   // hand over last stored list type to the list provider
   ListDataProviderBuilder listBuilder(_listID, DATA_CONTEXT_LIST_TMC_ROUTE);
   tSharedPtrDataProvider dataProvider = listBuilder.CreateDataProvider(_startIndex, actTMCListSize);
   POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));

   return true;
}


/********************************************
/* NAME    : TrafficListHandler
/* SYSFL    : SYSFL-3504,
****************************************************/
bool TrafficListHandler::onPropertyUpdateTrafficmessageDetailsChanged()
{
   const ValidValue<TrafficMessageDetailsInfo>& tmDetailsInfo = _navMiddleware.getTrafficMessageDetailsInfo();
   const navmiddleware::TrafficMessageDetailsInfo& tmDetailsInfoValue = tmDetailsInfo.getValue();

   GeoCoordinateDegree tmcCoordinate = tmDetailsInfoValue.getCoordinate();
   ETG_TRACE_USR4(("TrafficListHandler::onPropertyUpdateTrafficmessageDetailsChanged(): GeoCoordinateDegree= (%f, %f)\n",
                   tmcCoordinate.getLongitude(), tmcCoordinate.getLatitude()));

   // Set Co-Ordinate Position of Selected traffic message on DetailedInfo Scene
   std::vector<PosWGS84<double> > positionVector;
   positionVector.push_back(PosWGS84<double>(tmcCoordinate.getLongitude(), tmcCoordinate.getLatitude()));
   _infoStore.setLatitude(tmcCoordinate.getLatitude());
   _infoStore.setLongitude(tmcCoordinate.getLongitude());
   InfoStoreBase::CoordinatesToBeShownInMap& coordinatesToBeShownInMap = _infoStore.getCoordinatesToBeShownInMap();
   coordinatesToBeShownInMap._coordinates = positionVector;

   std::string outputText;
   for (int textIndex = 1; textIndex < tmDetailsInfoValue.m_textSections.size(); textIndex++)
   {
      outputText += tmDetailsInfoValue.m_textSections[textIndex] + " ";
   }
   _infoStore.setAddressInfoDetailsAddress(outputText.c_str());
   _infoStore.setAddressInfoDetailsName("");
   _infoStore.setAddressInfoDetailsPhoneNumber("");
   _infoStore.setIsDetailInfoRequested(false, InfoStoreBase::DETAIL_INFO_TMC);   // Set for disabling favorites button (in AddressDetailedInfo) when TMC detailed info is shown
   POST_MSG((COURIER_MESSAGE_NEW(Show_TMC_DetailedInfoReqMsg)()));
   POST_MSG((COURIER_MESSAGE_NEW(ShowDetailedAddressMsg)()));
   POST_MSG((COURIER_MESSAGE_NEW(ShowFavEditIcon_Enable_Delete_Guidance_Buttons)(0, 0, 0, 0, 0)));

   return true;
}


void TrafficListHandler::onPropertyUpdateTrafficmessageXUrgentChanged()
{
   ETG_TRACE_USR4(("TrafficListHandler::onPropertyUpdateTrafficmessageXUrgentChanged()"));

   const navmiddleware::TrafficMessageListInfos& tmInfos = _navMiddleware.getXUrgentTrafficMessageListInfos();
   if (tmInfos.getTrafficMessageListInfos().size())
   {
      navmiddleware::TrafficMessageListInfo xUrgentMessage = tmInfos.getTrafficMessageListInfos().front();
      _xUrgentTrafficMsgId = xUrgentMessage.m_id;

      // Distance
      if ((*_xUrgentTrafficMessage).mDistance != xUrgentMessage.m_distance.c_str())
      {
         (*_xUrgentTrafficMessage).mDistance = xUrgentMessage.m_distance.c_str();
         _xUrgentTrafficMessage.MarkItemModified(ItemKey::XUrgentTrafficMessage::DistanceItem);
      }

      // X-urgent traffic incident details
      std::string trafficDetails;
      std::vector< std::string>::const_iterator text = xUrgentMessage.m_textSections.begin();
      std::vector< std::string>::const_iterator textEnd = xUrgentMessage.m_textSections.end();
      for (; text != textEnd; ++text)
      {
         trafficDetails += *text + "\n";
      }
      if ((*_xUrgentTrafficMessage).mDetails != trafficDetails.c_str())
      {
         (*_xUrgentTrafficMessage).mDetails = trafficDetails.c_str();
         _xUrgentTrafficMessage.MarkItemModified(ItemKey::XUrgentTrafficMessage::DetailsItem);
      }

      // Direction
      unsigned int directionIdx = (unsigned int)(xUrgentMessage.m_direction.isValid() ? xUrgentMessage.m_direction.getValue() : DIRECTIONDESCRIPTION_SIZE);
      (*_xUrgentTrafficMessage).mDirection = ImageLoader::getAssetBitmapImage(getDirectionIcon(directionIdx)._normalBitmap);
      _xUrgentTrafficMessage.MarkItemModified(ItemKey::XUrgentTrafficMessage::DirectionItem);

      // Traffic incident icon
      Candera::Bitmap::SharedPointer bitmap = ImageLoader::loadBitmapData(xUrgentMessage.m_trafficIconData.getBlob(), xUrgentMessage.m_trafficIconData.getBlobSize());
      (*_xUrgentTrafficMessage).mTrafficIncidentIcon = ImageLoader::createImage(bitmap);
      _xUrgentTrafficMessage.MarkItemModified(ItemKey::XUrgentTrafficMessage::TrafficIncidentIconItem);

      _xUrgentTrafficMessage.SendUpdate(true);
      if (!_infoStore.isNonNaviVariant())
      {
         POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_XUrgent_Message"))));
      }
   }
}


bool TrafficListHandler::onCourierMessage(const ReleaseXUrgentTrafficMessageReqMsg& oMsg)
{
   ETG_TRACE_USR4(("TrafficListHandler::onCourierMessage(ReleaseXUrgentTrafficMessageReqMsg)"));
   COURIER_UNUSED(oMsg);

   _navMiddleware.releaseXUrgentTrafficMessage(_xUrgentTrafficMsgId);
   return true;
}
