/**************************************************************************************
* @file         : VehicleSettingsHelper.cpp
* @author       : Preethi Alagappan
* @addtogroup   : AppHmi_Navigation
* @brief        : Helper class for vehicle parameter list handler
* @copyright    : (c) 2019-2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "VehicleSettingsHelper.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#include "trcGenProj/Header/VehicleSettingsHelper.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

#define MAX_LENGTH_DIMENSIONS_STRING 8 // Length of floating point value of height/length/width/speed when converted to string

// Returns inputValue if it falls within min-max range
// Else, value is set to min/max and returned
unsigned int VehicleSettingsHelper::resetDimensionsToBound(VehicleSettingsListItemValue settingsItemIndex, unsigned int vehicleDimension, navmiddleware::DistanceUnitSystem distanceUnitSystem)
{
   ETG_TRACE_USR4(("VehicleSettingsHelper::resetDimensionsToBound(), List index : %d, Vehicle Dimension : %d, Units : %d", settingsItemIndex, vehicleDimension, distanceUnitSystem));

   bool bIsMetric = false;
   unsigned int inputValue = vehicleDimension;
   const VehicleSettingsMetaData* vehicleSettingsMetaData = NULL;
   if (navmiddleware::DISTANCEUNITSYSTEM_METRIC == distanceUnitSystem)
   {
      vehicleSettingsMetaData = &vehicleSettingsMetricMetaData[0];
      bIsMetric = true;
   }
   else
   {
      vehicleSettingsMetaData = &vehicleSettingsImperialMetaData[0];

      // KDS values for dimensions are in millimeters. KDS value is converted to inches for imperial units
      switch (settingsItemIndex)
      {
         case VEHICLE_SETTINGS_HEIGHT:
         case VEHICLE_SETTINGS_LENGTH:
         case VEHICLE_SETTINGS_WIDTH:
         {
            inputValue = VehicleDimensionsConverter::millimetersToInches(vehicleDimension);
            break;
         }

         case VEHICLE_SETTINGS_TOTAL_WEIGHT:
         case VEHICLE_SETTINGS_LOAD_PER_AXLE:
         {
            // For BRITISH_IMPERIAL, weight is in kilogram
            inputValue = (navmiddleware::DISTANCEUNITSYSTEM_IMPERIAL == distanceUnitSystem)
                         ? VehicleDimensionsConverter::kilogramsToPounds(vehicleDimension)
                         : vehicleDimension;
            break;
         }

         default:
         {
            inputValue = vehicleDimension;
         }
      }
   }

   float hmiUnitValue = VehicleSettingsHelper::convertDimensionToHmiUnit(settingsItemIndex, inputValue, distanceUnitSystem);
   float hmiResetValue = ROUNDF_2DECIMAL(hmiUnitValue);

   if (hmiUnitValue < vehicleSettingsMetaData[settingsItemIndex].minValue)
   {
      hmiResetValue = vehicleSettingsMetaData[settingsItemIndex].minValue;
   }
   else if (hmiUnitValue > vehicleSettingsMetaData[settingsItemIndex].maxValue)
   {
      hmiResetValue = vehicleSettingsMetaData[settingsItemIndex].maxValue;
   }

   unsigned int presCtrlResetValue = VehicleSettingsHelper::convertDimensionToPresCtrlUnit(settingsItemIndex, hmiResetValue, distanceUnitSystem);

   ETG_TRACE_USR4(("VehicleSettingsHelper::resetDimensionsToBound(), Vehicle Dimension reset to : %d", presCtrlResetValue));
   return presCtrlResetValue;
}


void VehicleSettingsHelper::convertVehicleProfileUnits(VehicleProfile& vehicleProfile, navmiddleware::DistanceUnitSystem distanceUnitSystem)
{
   ETG_TRACE_USR4(("VehicleSettingsHelper::convertVehicleProfileUnits(), Units : %d", distanceUnitSystem));
   if (navmiddleware::DISTANCEUNITSYSTEM_METRIC == distanceUnitSystem)
   {
      convertVehicleProfileUnitsToMetric(vehicleProfile, distanceUnitSystem);
   }
   else
   {
      convertVehicleProfileUnitsToImperial(vehicleProfile, distanceUnitSystem);
   }
}


Candera::String VehicleSettingsHelper::convertHmiDimensionToString(VehicleSettingsListItemValue settingsItemIndex, float dimensionValue, navmiddleware::DistanceUnitSystem distanceUnitSystem)
{
   ETG_TRACE_USR4(("VehicleSettingsHelper::convertHmiDimensionToString(), List index : %d, Vehicle Dimension : %f, Unit system : %d", settingsItemIndex, dimensionValue, distanceUnitSystem));
   Candera::String hmiUnitsValueStr;
   if (navmiddleware::DISTANCEUNITSYSTEM_METRIC == distanceUnitSystem)
   {
      hmiUnitsValueStr = convertHmiMetricDimensionToString(settingsItemIndex, dimensionValue);
   }
   else
   {
      hmiUnitsValueStr = convertHmiImperialDimensionToString(settingsItemIndex, dimensionValue);
   }
   return hmiUnitsValueStr;
}


unsigned int VehicleSettingsHelper::convertDimensionToPresCtrlUnit(VehicleSettingsListItemValue settingsItemIndex, float dimensionValue, navmiddleware::DistanceUnitSystem distanceUnitSystem)
{
   ETG_TRACE_USR4(("VehicleSettingsHelper::convertDimensionToPresCtrlUnit(), List index : %d, Vehicle Dimension : %f, Unit system : %d", settingsItemIndex, dimensionValue, distanceUnitSystem));

   unsigned int presCtrlUnitValue = 0;
   if (navmiddleware::DISTANCEUNITSYSTEM_METRIC == distanceUnitSystem)
   {
      presCtrlUnitValue = convertDimensionToPresCtrlMetricUnit(settingsItemIndex, dimensionValue);
   }
   else
   {
      bool isBritishImperial = (navmiddleware::DISTANCEUNITSYSTEM_BRITISH_IMPERIAL == distanceUnitSystem) ? true : false;
      presCtrlUnitValue = convertDimensionToPresCtrlImperialUnit(settingsItemIndex, dimensionValue, isBritishImperial);
   }
   return presCtrlUnitValue;
}


float VehicleSettingsHelper::convertDimensionToHmiUnit(VehicleSettingsListItemValue settingsItemIndex, unsigned int dimensionValue, navmiddleware::DistanceUnitSystem distanceUnitSystem)
{
   ETG_TRACE_USR4(("VehicleSettingsHelper::convertDimensionToHmiUnit(), List index : %d, Vehicle Dimension : %d, Unit system : %d", settingsItemIndex, dimensionValue, distanceUnitSystem));

   float hmiUnitValue = 0.0F;
   if (navmiddleware::DISTANCEUNITSYSTEM_METRIC == distanceUnitSystem)
   {
      hmiUnitValue = convertDimensionToHmiMetricUnit(settingsItemIndex, dimensionValue);
   }
   else
   {
      bool isBritishImperial = (navmiddleware::DISTANCEUNITSYSTEM_BRITISH_IMPERIAL == distanceUnitSystem) ? true : false;
      hmiUnitValue = convertDimensionToHmiImperialUnit(settingsItemIndex, dimensionValue, isBritishImperial);
   }
   return hmiUnitValue;
}


void VehicleSettingsHelper::convertVehicleProfileUnitsToMetric(VehicleProfile& vehicleProfile, navmiddleware::DistanceUnitSystem distanceUnitSystem)
{
   ETG_TRACE_USR4(("VehicleSettingsHelper::convertVehicleProfileUnitsToMetric()"));

   navmiddleware::DistanceUnitSystem previousDistanceUnitSystem = vehicleProfile.getUnitSystem();
   vehicleProfile.setUnitSystem(distanceUnitSystem);

   unsigned int presCtrlImperialValue = 0;
   unsigned int presCtrlMetricValue = 0;
   float hmiMetricValue = 0.0;

   presCtrlImperialValue = vehicleProfile.getHeightOfVehicle();
   hmiMetricValue = ROUNDF_1DECIMAL(VehicleDimensionsConverter::inchesToMeters(presCtrlImperialValue));
   presCtrlMetricValue = convertDimensionToPresCtrlMetricUnit(VEHICLE_SETTINGS_HEIGHT, hmiMetricValue);
   vehicleProfile.setHeightOfVehicle(presCtrlMetricValue);
   ETG_TRACE_USR4(("VehicleSettingsHelper::convertVehicleProfileUnitsToMetric(), Height (%d inch, %d mm)", presCtrlImperialValue, presCtrlMetricValue));

   presCtrlImperialValue = vehicleProfile.getLengthOfVehicle();
   hmiMetricValue = ROUNDF_1DECIMAL(VehicleDimensionsConverter::inchesToMeters(presCtrlImperialValue));
   hmiMetricValue = roundToNearestStepValue(hmiMetricValue, vehicleSettingsMetricMetaData[VEHICLE_SETTINGS_LENGTH].step);
   presCtrlMetricValue = convertDimensionToPresCtrlMetricUnit(VEHICLE_SETTINGS_LENGTH, hmiMetricValue);
   vehicleProfile.setLengthOfVehicle(presCtrlMetricValue);
   ETG_TRACE_USR4(("VehicleSettingsHelper::convertVehicleProfileUnitsToMetric(), Length (%d inch, %d mm)", presCtrlImperialValue, presCtrlMetricValue));

   presCtrlImperialValue = vehicleProfile.getWidthOfVehicle();
   hmiMetricValue = ROUNDF_2DECIMAL(VehicleDimensionsConverter::inchesToMeters(presCtrlImperialValue));
   hmiMetricValue = roundToNearestStepValue(hmiMetricValue, vehicleSettingsMetricMetaData[VEHICLE_SETTINGS_WIDTH].step);
   presCtrlMetricValue = convertDimensionToPresCtrlMetricUnit(VEHICLE_SETTINGS_WIDTH, hmiMetricValue);
   vehicleProfile.setWidthOfVehicle(presCtrlMetricValue);
   ETG_TRACE_USR4(("VehicleSettingsHelper::convertVehicleProfileUnitsToMetric(), Width (%d inch, %d mm)", presCtrlImperialValue, presCtrlMetricValue));

   // For BRITISH_IMPERIAL, weight units will be set in kilogram in VehicleProfile. So no unit conversion is done.
   if (navmiddleware::DISTANCEUNITSYSTEM_IMPERIAL == previousDistanceUnitSystem)
   {
      presCtrlImperialValue = vehicleProfile.getMaxWeightOfVehicle();
      hmiMetricValue = ROUNDF_1DECIMAL(VehicleDimensionsConverter::poundsToTonnes(presCtrlImperialValue));
      presCtrlMetricValue = convertDimensionToPresCtrlMetricUnit(VEHICLE_SETTINGS_TOTAL_WEIGHT, hmiMetricValue);
      vehicleProfile.setMaxWeightOfVehicle(presCtrlMetricValue);
      ETG_TRACE_USR4(("VehicleSettingsHelper::convertVehicleProfileUnitsToMetric(), Weight (%d lb, %d kg)", presCtrlImperialValue, presCtrlMetricValue));

      presCtrlImperialValue = vehicleProfile.getMaxLoadPerAxleOfVehicle();
      hmiMetricValue = ROUNDF_1DECIMAL(VehicleDimensionsConverter::poundsToTonnes(presCtrlImperialValue));
      presCtrlMetricValue = convertDimensionToPresCtrlMetricUnit(VEHICLE_SETTINGS_LOAD_PER_AXLE, hmiMetricValue);
      vehicleProfile.setMaxLoadPerAxleOfVehicle(presCtrlMetricValue);
      ETG_TRACE_USR4(("VehicleSettingsHelper::convertVehicleProfileUnitsToMetric(), Load per axle (%d lb, %d kg)", presCtrlImperialValue, presCtrlMetricValue));
   }

   presCtrlImperialValue = vehicleProfile.getMaxSpeed();
   presCtrlMetricValue = UINT(ROUNDF(VehicleDimensionsConverter::mphToKmph(presCtrlImperialValue)));
   unsigned int stepValue = UINT(vehicleSettingsMetricMetaData[VEHICLE_SETTINGS_MAX_SPEED].step);
   presCtrlMetricValue = roundToNearestStepValue(presCtrlMetricValue, stepValue);
   vehicleProfile.setMaxSpeed(presCtrlMetricValue);
   ETG_TRACE_USR4(("VehicleSettingsHelper::convertVehicleProfileUnitsToMetric(), Max speed (%d mph, %d kmph)", presCtrlImperialValue, presCtrlMetricValue));

   return;
}


void VehicleSettingsHelper::convertVehicleProfileUnitsToImperial(VehicleProfile& vehicleProfile, navmiddleware::DistanceUnitSystem distanceUnitSystem)
{
   ETG_TRACE_USR4(("VehicleSettingsHelper::convertVehicleProfileUnitsToImperial()"));

   vehicleProfile.setUnitSystem(distanceUnitSystem);

   unsigned int presCtrlMetricValue = 0;
   unsigned int presCtrlImperialValue = 0;
   float hmiImperialValue = 0.0;

   presCtrlMetricValue = vehicleProfile.getHeightOfVehicle();
   hmiImperialValue = ROUNDF_1DECIMAL(VehicleDimensionsConverter::millimetersToYards(presCtrlMetricValue));
   presCtrlImperialValue = convertDimensionToPresCtrlImperialUnit(VEHICLE_SETTINGS_HEIGHT, hmiImperialValue);
   vehicleProfile.setHeightOfVehicle(presCtrlImperialValue);
   ETG_TRACE_USR4(("VehicleSettingsHelper::convertVehicleProfileUnitsToImperial(), Height (%d mm, %d inch)", presCtrlMetricValue, presCtrlImperialValue));

   presCtrlMetricValue = vehicleProfile.getLengthOfVehicle();
   hmiImperialValue = ROUNDF_1DECIMAL(VehicleDimensionsConverter::millimetersToYards(presCtrlMetricValue));
   presCtrlImperialValue = convertDimensionToPresCtrlImperialUnit(VEHICLE_SETTINGS_LENGTH, hmiImperialValue);
   vehicleProfile.setLengthOfVehicle(presCtrlImperialValue);
   ETG_TRACE_USR4(("VehicleSettingsHelper::convertVehicleProfileUnitsToImperial(), Length (%d mm, %d inch)", presCtrlMetricValue, presCtrlImperialValue));

   presCtrlMetricValue = vehicleProfile.getWidthOfVehicle();
   hmiImperialValue = ROUNDF_2DECIMAL(VehicleDimensionsConverter::millimetersToYards(presCtrlMetricValue));
   presCtrlImperialValue = convertDimensionToPresCtrlImperialUnit(VEHICLE_SETTINGS_WIDTH, hmiImperialValue);
   vehicleProfile.setWidthOfVehicle(presCtrlImperialValue);
   ETG_TRACE_USR4(("VehicleSettingsHelper::convertVehicleProfileUnitsToImperial(), Width (%d mm, %d inch)", presCtrlMetricValue, presCtrlImperialValue));

   // For BRITISH_IMPERIAL, weight units should be set in kilogram in VehicleProfile. So no unit conversion is done
   if (navmiddleware::DISTANCEUNITSYSTEM_IMPERIAL == distanceUnitSystem)
   {
      presCtrlMetricValue = vehicleProfile.getMaxWeightOfVehicle();
      hmiImperialValue = ROUNDF_1DECIMAL(VehicleDimensionsConverter::kilogramsToTonnes(presCtrlMetricValue));
      presCtrlImperialValue = convertDimensionToPresCtrlImperialUnit(VEHICLE_SETTINGS_TOTAL_WEIGHT, hmiImperialValue);
      vehicleProfile.setMaxWeightOfVehicle(presCtrlImperialValue);
      ETG_TRACE_USR4(("VehicleSettingsHelper::convertVehicleProfileUnitsToImperial(), Weight (%d kg, %d lb)", presCtrlMetricValue, presCtrlImperialValue));

      presCtrlMetricValue = vehicleProfile.getMaxLoadPerAxleOfVehicle();
      hmiImperialValue = ROUNDF_1DECIMAL(VehicleDimensionsConverter::kilogramsToTonnes(presCtrlMetricValue));
      presCtrlImperialValue = convertDimensionToPresCtrlImperialUnit(VEHICLE_SETTINGS_LOAD_PER_AXLE, hmiImperialValue);
      vehicleProfile.setMaxLoadPerAxleOfVehicle(presCtrlImperialValue);
      ETG_TRACE_USR4(("VehicleSettingsHelper::convertVehicleProfileUnitsToImperial(), Load per axle (%d kg, %d lb)", presCtrlMetricValue, presCtrlImperialValue));
   }

   presCtrlMetricValue = vehicleProfile.getMaxSpeed();
   presCtrlImperialValue = UINT(ROUNDF(VehicleDimensionsConverter::kmphToMph(presCtrlMetricValue)));
   unsigned int stepValue = UINT(vehicleSettingsImperialMetaData[VEHICLE_SETTINGS_MAX_SPEED].step);
   presCtrlImperialValue = roundToNearestStepValue(presCtrlImperialValue, stepValue);
   vehicleProfile.setMaxSpeed(presCtrlImperialValue);
   ETG_TRACE_USR4(("VehicleSettingsHelper::convertVehicleProfileUnitsToImperial(), Max speed (%d kmph, %d mph)", presCtrlMetricValue, presCtrlImperialValue));

   return;
}


Candera::String VehicleSettingsHelper::convertHmiMetricDimensionToString(VehicleSettingsListItemValue settingsItemIndex, float dimensionValue)
{
   ETG_TRACE_USR4(("VehicleSettingsHelper::convertHmiMetricDimensionToString(), List index : %d, Vehicle Dimension : %f", settingsItemIndex, dimensionValue));

   char hmiUnitsValueStr[MAX_LENGTH_DIMENSIONS_STRING];
   memset(hmiUnitsValueStr, '\0', MAX_LENGTH_DIMENSIONS_STRING * sizeof(hmiUnitsValueStr[0]));

   switch (settingsItemIndex)
   {
      case VEHICLE_SETTINGS_LENGTH:
      case VEHICLE_SETTINGS_HEIGHT:
      {
         snprintf(hmiUnitsValueStr, MAX_LENGTH_DIMENSIONS_STRING, "%.1f%1s", dimensionValue, "m");
         break;
      }

      case VEHICLE_SETTINGS_WIDTH:
      {
         snprintf(hmiUnitsValueStr, MAX_LENGTH_DIMENSIONS_STRING, "%.2f%1s", dimensionValue, "m");
         break;
      }

      case VEHICLE_SETTINGS_TOTAL_WEIGHT:
      case VEHICLE_SETTINGS_LOAD_PER_AXLE:
      {
         snprintf(hmiUnitsValueStr, MAX_LENGTH_DIMENSIONS_STRING, "%.1f%s", dimensionValue, "t");
         break;
      }

      case VEHICLE_SETTINGS_MAX_SPEED:
      {
         snprintf(hmiUnitsValueStr, MAX_LENGTH_DIMENSIONS_STRING, "%.0f%s", dimensionValue, "kmph");
         break;
      }

      default:
      {
         snprintf(hmiUnitsValueStr, MAX_LENGTH_DIMENSIONS_STRING, "%.0f", dimensionValue);
      }
   }
   ETG_TRACE_USR4(("VehicleSettingsHelper::convertHmiMetricDimensionToString(), HMI Vehicle Dimension : %s", hmiUnitsValueStr));
   return hmiUnitsValueStr;
}


Candera::String VehicleSettingsHelper::convertHmiImperialDimensionToString(VehicleSettingsListItemValue settingsItemIndex, float dimensionValue)
{
   ETG_TRACE_USR4(("VehicleSettingsHelper::convertHmiImperialDimensionToString(), List index : %d, Vehicle Dimension : %f", settingsItemIndex, dimensionValue));

   char hmiUnitsValueStr[MAX_LENGTH_DIMENSIONS_STRING];
   memset(hmiUnitsValueStr, '\0', MAX_LENGTH_DIMENSIONS_STRING * sizeof(hmiUnitsValueStr[0]));

   switch (settingsItemIndex)
   {
      case VEHICLE_SETTINGS_LENGTH:
      case VEHICLE_SETTINGS_HEIGHT:
      {
         snprintf(hmiUnitsValueStr, MAX_LENGTH_DIMENSIONS_STRING, "%.1f%1s", dimensionValue, "yd");
         break;
      }

      case VEHICLE_SETTINGS_WIDTH:
      {
         snprintf(hmiUnitsValueStr, MAX_LENGTH_DIMENSIONS_STRING, "%.2f%1s", dimensionValue, "yd");
         break;
      }

      case VEHICLE_SETTINGS_TOTAL_WEIGHT:
      case VEHICLE_SETTINGS_LOAD_PER_AXLE:
      {
         snprintf(hmiUnitsValueStr, MAX_LENGTH_DIMENSIONS_STRING, "%.1f%s", dimensionValue, "t");
         break;
      }

      case VEHICLE_SETTINGS_MAX_SPEED:
      {
         snprintf(hmiUnitsValueStr, MAX_LENGTH_DIMENSIONS_STRING, "%.0f%s", dimensionValue, "mph");
         break;
      }

      default:
      {
         snprintf(hmiUnitsValueStr, MAX_LENGTH_DIMENSIONS_STRING, "%.0f", dimensionValue);
      }
   }
   ETG_TRACE_USR4(("VehicleSettingsHelper::convertHmiImperialDimensionToString(), HMI Vehicle Dimension : %s", hmiUnitsValueStr));
   return hmiUnitsValueStr;
}


unsigned int VehicleSettingsHelper::convertDimensionToPresCtrlMetricUnit(VehicleSettingsListItemValue settingsItemIndex, float dimensionValue)
{
   ETG_TRACE_USR4(("VehicleSettingsHelper::convertDimensionToPresCtrlMetricUnit(), List index : %d, Vehicle Dimension : %f", settingsItemIndex, dimensionValue));

   unsigned int presCtrlUnitValue = 0;

   switch (settingsItemIndex)
   {
      case VEHICLE_SETTINGS_LENGTH:
      case VEHICLE_SETTINGS_HEIGHT:
      case VEHICLE_SETTINGS_WIDTH:
      {
         presCtrlUnitValue = UINT(ROUNDF(VehicleDimensionsConverter::metersToMillimeters(dimensionValue)));
         ETG_TRACE_USR4(("VehicleSettingsHelper::convertDimensionToPresCtrlMetricUnit(), PresCtrl Vehicle Dimension (mm) : %d", presCtrlUnitValue));
         break;
      }

      case VEHICLE_SETTINGS_TOTAL_WEIGHT:
      case VEHICLE_SETTINGS_LOAD_PER_AXLE:
      {
         presCtrlUnitValue = UINT(ROUNDF(VehicleDimensionsConverter::tonnesToKilograms(dimensionValue)));
         ETG_TRACE_USR4(("VehicleSettingsHelper::convertDimensionToPresCtrlMetricUnit(), PresCtrl Vehicle Dimension (kg) : %d", presCtrlUnitValue));
         break;
      }

      default:
      {
         presCtrlUnitValue = UINT(ROUNDF(dimensionValue));
         ETG_TRACE_USR4(("VehicleSettingsHelper::convertDimensionToPresCtrlMetricUnit(), PresCtrl Vehicle Dimension : %d", presCtrlUnitValue));
      }
   }
   return presCtrlUnitValue;
}


unsigned int VehicleSettingsHelper::convertDimensionToPresCtrlImperialUnit(VehicleSettingsListItemValue settingsItemIndex, float dimensionValue, bool isBritishImperial)
{
   ETG_TRACE_USR4(("VehicleSettingsHelper::convertDimensionToPresCtrlImperialUnit(), List index : %d, Vehicle Dimension : %f, isBritishImperial : %d", settingsItemIndex, dimensionValue, isBritishImperial));

   unsigned int presCtrlUnitValue = 0;

   switch (settingsItemIndex)
   {
      case VEHICLE_SETTINGS_LENGTH:
      case VEHICLE_SETTINGS_HEIGHT:
      case VEHICLE_SETTINGS_WIDTH:
      {
         presCtrlUnitValue = UINT(ROUNDF(VehicleDimensionsConverter::yardsToInches(dimensionValue)));
         ETG_TRACE_USR4(("VehicleSettingsHelper::convertDimensionToPresCtrlImperialUnit(), PresCtrl Vehicle Dimension (inch) : %d", presCtrlUnitValue));
         break;
      }

      case VEHICLE_SETTINGS_TOTAL_WEIGHT:
      case VEHICLE_SETTINGS_LOAD_PER_AXLE:
      {
         if (false == isBritishImperial)
         {
            presCtrlUnitValue = UINT(ROUNDF(VehicleDimensionsConverter::tonnesToPounds(dimensionValue)));
            ETG_TRACE_USR4(("VehicleSettingsHelper::convertDimensionToPresCtrlImperialUnit(), PresCtrl Vehicle Dimension (lb) : %d", presCtrlUnitValue));
         }
         else
         {
            presCtrlUnitValue = UINT(ROUNDF(VehicleDimensionsConverter::tonnesToKilograms(dimensionValue)));
            ETG_TRACE_USR4(("VehicleSettingsHelper::convertDimensionToPresCtrlImperialUnit(), PresCtrl Vehicle Dimension (kg) : %d", presCtrlUnitValue));
         }
         break;
      }

      default:
      {
         presCtrlUnitValue = UINT(ROUNDF(dimensionValue));
         ETG_TRACE_USR4(("VehicleSettingsHelper::convertDimensionToPresCtrlImperialUnit(), PresCtrl Vehicle Dimension : %d", presCtrlUnitValue));
      }
   }
   return presCtrlUnitValue;
}


float VehicleSettingsHelper::convertDimensionToHmiMetricUnit(VehicleSettingsListItemValue settingsItemIndex, unsigned int dimensionValue)
{
   ETG_TRACE_USR4(("VehicleSettingsHelper::convertDimensionToHmiMetricUnit(), List index : %d, Vehicle Dimension : %d", settingsItemIndex, dimensionValue));

   float hmiUnitValue = 0.0F;

   switch (settingsItemIndex)
   {
      case VEHICLE_SETTINGS_HEIGHT:
      {
         hmiUnitValue = VehicleDimensionsConverter::millimetersToMeters(dimensionValue);
         hmiUnitValue = ROUNDF_1DECIMAL(hmiUnitValue);
         ETG_TRACE_USR4(("VehicleSettingsHelper::convertDimensionToHmiMetricUnit(), Hmi Vehicle Dimension (m) : %f", hmiUnitValue));
         break;
      }

      case VEHICLE_SETTINGS_LENGTH:
      {
         hmiUnitValue = VehicleDimensionsConverter::millimetersToMeters(dimensionValue);
         hmiUnitValue = ROUNDF_1DECIMAL(hmiUnitValue);
         hmiUnitValue = roundToNearestStepValue(hmiUnitValue, vehicleSettingsMetricMetaData[settingsItemIndex].step);
         ETG_TRACE_USR4(("VehicleSettingsHelper::convertDimensionToHmiMetricUnit(), Hmi Vehicle Dimension (m) : %f", hmiUnitValue));
         break;
      }

      case VEHICLE_SETTINGS_WIDTH:
      {
         hmiUnitValue = VehicleDimensionsConverter::millimetersToMeters(dimensionValue);
         hmiUnitValue = ROUNDF_2DECIMAL(hmiUnitValue);
         hmiUnitValue = roundToNearestStepValue(hmiUnitValue, vehicleSettingsMetricMetaData[settingsItemIndex].step);
         ETG_TRACE_USR4(("VehicleSettingsHelper::convertDimensionToHmiMetricUnit(), Hmi Vehicle Dimension (m) : %f", hmiUnitValue));
         break;
      }

      case VEHICLE_SETTINGS_TOTAL_WEIGHT:
      case VEHICLE_SETTINGS_LOAD_PER_AXLE:
      {
         hmiUnitValue = VehicleDimensionsConverter::kilogramsToTonnes(dimensionValue);
         hmiUnitValue = ROUNDF_1DECIMAL(hmiUnitValue);
         hmiUnitValue = roundToNearestStepValue(hmiUnitValue, vehicleSettingsMetricMetaData[settingsItemIndex].step);

         ETG_TRACE_USR4(("VehicleSettingsHelper::convertDimensionToHmiMetricUnit(), Hmi Vehicle Dimension (tonne) : %f", hmiUnitValue));
         break;
      }

      default:
      {
         hmiUnitValue = static_cast<float>(dimensionValue);
         ETG_TRACE_USR4(("VehicleSettingsHelper::convertDimensionToHmiMetricUnit(), Hmi Vehicle Dimension : %f", hmiUnitValue));
      }
   }
   if (hmiUnitValue < vehicleSettingsMetricMetaData[settingsItemIndex].minValue)
   {
      hmiUnitValue = vehicleSettingsMetricMetaData[settingsItemIndex].minValue;
   }
   else if (hmiUnitValue > vehicleSettingsMetricMetaData[settingsItemIndex].maxValue)
   {
      hmiUnitValue = vehicleSettingsMetricMetaData[settingsItemIndex].maxValue;
   }
   return hmiUnitValue;
}


float VehicleSettingsHelper::convertDimensionToHmiImperialUnit(VehicleSettingsListItemValue settingsItemIndex, unsigned int dimensionValue, bool isBritishImperial)
{
   ETG_TRACE_USR4(("VehicleSettingsHelper::convertDimensionToHmiImperialUnit(), List index : %d, Vehicle Dimension : %d, isBritishImperial : %d", settingsItemIndex, dimensionValue, isBritishImperial));

   float hmiUnitValue = 0.0F;

   switch (settingsItemIndex)
   {
      case VEHICLE_SETTINGS_LENGTH:
      case VEHICLE_SETTINGS_HEIGHT:
      {
         hmiUnitValue = VehicleDimensionsConverter::inchesToYards(dimensionValue);
         hmiUnitValue = ROUNDF_1DECIMAL(hmiUnitValue);
         ETG_TRACE_USR4(("VehicleSettingsHelper::convertDimensionToHmiImperialUnit(), Hmi Vehicle Dimension (yard) : %f", hmiUnitValue));
         break;
      }

      case VEHICLE_SETTINGS_WIDTH:
      {
         hmiUnitValue = VehicleDimensionsConverter::inchesToYards(dimensionValue);
         hmiUnitValue = ROUNDF_2DECIMAL(hmiUnitValue);
         ETG_TRACE_USR4(("VehicleSettingsHelper::convertDimensionToHmiImperialUnit(), Hmi Vehicle Dimension (yard) : %f", hmiUnitValue));
         break;
      }

      case VEHICLE_SETTINGS_TOTAL_WEIGHT:
      case VEHICLE_SETTINGS_LOAD_PER_AXLE:
      {
         hmiUnitValue = (false == isBritishImperial)
                        ? VehicleDimensionsConverter::poundsToTonnes(dimensionValue)
                        : VehicleDimensionsConverter::kilogramsToTonnes(dimensionValue);
         hmiUnitValue = ROUNDF_1DECIMAL(hmiUnitValue);
         hmiUnitValue = roundToNearestStepValue(hmiUnitValue, vehicleSettingsImperialMetaData[settingsItemIndex].step);
         ETG_TRACE_USR4(("VehicleSettingsHelper::convertDimensionToHmiImperialUnit(), Hmi Vehicle Dimension (tonne) : %f", hmiUnitValue));
         break;
      }

      default:
      {
         hmiUnitValue = static_cast<float>(dimensionValue);
         ETG_TRACE_USR4(("VehicleSettingsHelper::convertDimensionToHmiImperialUnit(), Hmi Vehicle Dimension : %f", hmiUnitValue));
      }
   }
   if (hmiUnitValue < vehicleSettingsImperialMetaData[settingsItemIndex].minValue)
   {
      hmiUnitValue = vehicleSettingsImperialMetaData[settingsItemIndex].minValue;
   }
   else if (hmiUnitValue > vehicleSettingsImperialMetaData[settingsItemIndex].maxValue)
   {
      hmiUnitValue = vehicleSettingsImperialMetaData[settingsItemIndex].maxValue;
   }
   return hmiUnitValue;
}


std::string VehicleSettingsHelper::convertVehicleRestrictionsDimensionToString(VehicleRestrictionsPriorityType vehicleRestrictionsType, unsigned int vehicleRestrictionValue, navmiddleware::DistanceUnitSystem distanceUnitSystem, bool isRestrictionValueInMetric)
{
   ETG_TRACE_USR4(("VehicleSettingsHelper::convertVehicleRestrictionsDimensionToString(), VehicleRestrictionsPriorityType : %d, vehicleRestrictionValue :%d,", vehicleRestrictionsType, vehicleRestrictionValue));

   float hmiUnitValue = 0.0F;
   std::string currrentUnitString = "";
   std::stringstream hmiUnitsValueStr;

   switch (vehicleRestrictionsType)
   {
      case VEHICLE_RESTRICTIONS_HEIGHT:
      case VEHICLE_RESTRICTIONS_LENGTH:
      {
         if (distanceUnitSystem == navmiddleware::DISTANCEUNITSYSTEM_METRIC)
         {
            hmiUnitValue = isRestrictionValueInMetric ? VehicleDimensionsConverter::millimetersToMeters(vehicleRestrictionValue) : VehicleDimensionsConverter::inchesToMeters(vehicleRestrictionValue);
            currrentUnitString = "m";
            ETG_TRACE_USR4(("VehicleSettingsHelper::convertVehicleRestrictionsDimensionToString(), Hmi Vehicle Restrictions Dimension (Meters): %f", hmiUnitValue));
         }
         else
         {
            hmiUnitValue = isRestrictionValueInMetric ? VehicleDimensionsConverter::millimetersToYards(vehicleRestrictionValue) : VehicleDimensionsConverter::inchesToYards(vehicleRestrictionValue);
            currrentUnitString = "yd";
            ETG_TRACE_USR4(("VehicleSettingsHelper::convertVehicleRestrictionsDimensionToString(), Hmi Vehicle Restrictions Dimension (Yards): %f", hmiUnitValue));
         }
         hmiUnitValue = ROUNDF_1DECIMAL(hmiUnitValue);
         break;
      }
      case VEHICLE_RESTRICTIONS_WIDTH:
      {
         if (distanceUnitSystem == navmiddleware::DISTANCEUNITSYSTEM_METRIC)
         {
            hmiUnitValue = isRestrictionValueInMetric ? VehicleDimensionsConverter::millimetersToMeters(vehicleRestrictionValue) : VehicleDimensionsConverter::inchesToMeters(vehicleRestrictionValue);
            currrentUnitString = "m";
            ETG_TRACE_USR4(("VehicleSettingsHelper::convertVehicleRestrictionsDimensionToString(), Hmi Vehicle Restrictions Dimension (Meters): %f", hmiUnitValue));
         }
         else
         {
            hmiUnitValue = isRestrictionValueInMetric ? VehicleDimensionsConverter::millimetersToYards(vehicleRestrictionValue) : VehicleDimensionsConverter::inchesToYards(vehicleRestrictionValue);
            currrentUnitString = "yd";
            ETG_TRACE_USR4(("VehicleSettingsHelper::convertVehicleRestrictionsDimensionToString(), Hmi Vehicle Restrictions Dimension (Yards): %f", hmiUnitValue));
         }
         hmiUnitValue = ROUNDF_2DECIMAL(hmiUnitValue);
         break;
      }
      case VEHICLE_RESTRICTIONS_WEIGHT:
      case VEHICLE_RESTRICTIONS_WEIGHT_PER_AXLE:
      {
         hmiUnitValue = isRestrictionValueInMetric ? VehicleDimensionsConverter::kilogramsToTonnes(vehicleRestrictionValue) : VehicleDimensionsConverter::poundsToTonnes(vehicleRestrictionValue);
         ETG_TRACE_USR4(("VehicleSettingsHelper::convertVehicleRestrictionsDimensionToString(), Hmi Vehicle Restrictions Dimension (tonne): %f", hmiUnitValue));
         hmiUnitValue = ROUNDF_1DECIMAL(hmiUnitValue);
         currrentUnitString = "t";
         break;
      }
      default:
      {
         break;
      }
   }
   hmiUnitsValueStr << hmiUnitValue << currrentUnitString;
   return hmiUnitsValueStr.str();
}


unsigned int VehicleSettingsHelper::roundToNearestStepValue(unsigned int value, unsigned int step)
{
   unsigned int roundedValue = value;
   unsigned int remainder = value % step;
   if (0 != remainder)
   {
      roundedValue = (remainder <= (step / 2)) ? (value - remainder) : (value + (step - remainder));
   }
   ETG_TRACE_USR4(("VehicleSettingsHelper::roundToNearestStepValue(), Input value: %d, Rounded value: %d, Step : %d", value, roundedValue, step));
   return roundedValue;
}


float VehicleSettingsHelper::roundToNearestStepValue(float value, float step)
{
   float roundedValue = value;
   if (!isMultipleOfStepValue(value, step))
   {
      float remainder = fabs(remainderf(value, step));
      roundedValue = value + remainder;
      if (!isMultipleOfStepValue(roundedValue, step))
      {
         roundedValue = value - remainder;
      }
   }
   ETG_TRACE_USR4(("VehicleSettingsHelper::roundToNearestStepValue(), Input value: %f, Rounded value: %f, Step: %f", value, roundedValue, step));
   return roundedValue;
}


bool VehicleSettingsHelper::isMultipleOfStepValue(float value, float step)
{
   float remainder = fabs(remainderf(value, step));
   float conversionFactor = pow(10, (MAX_HMI_DIMENSION_VALUE_PRECISION + 1));
   int convertedRemainder = remainder * conversionFactor;
   return ((0 == convertedRemainder) ? true : false);
}
