/**************************************************************************************
* @file         : VehicleParameterListHandler.cpp
* @author       : ECG-Prieethi Narayanaswamy
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) -2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#include "VehicleParameterListHandler.h"
#include "Route/RouteDataUtils.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/VehicleParameterListHandler.cpp.trc.h"
#endif
#else
#include  <vector>
#include "VehicleParameterListHandler.h"
#define LIST_ID_NAV_SETTINGS_VEHICLE_PARAMETER 1043
#define LIST_ID_NAV_SETTINGS_VEHICLE_PROFILE 1028
#define LIST_ID_NAV_SETTINGS_VEHICLE_HAZARDOUS 1029
#define LIST_ID_NAV_SETTINGS_VEHICLE_TRAILER_TYPE 1046
#endif   // end of VARIANT_S_FTR_ENABLE_UNITTEST

#ifdef  HALL_TO_MDW_COM

// Scene composer related values for vehicle settings list
static const int DATA_ITEM_DECREMENT = 1;
static const int DATA_ITEM_INCREMENT = 2;
static const char* DATA_ITEM_CONTEXT_SPINNER      = "Layer_Spinner";
static const char* const DATA_CONTEXT_COMBOBOX_LIST_ITEMS_4  = "Combobox_ListItem4";
static const char* const DATA_CONTEXT_COMBOBOX_LIST_ITEMS_2 = "Combobox_ListItem2";
static const char* DATA_ITEM_DROP_DOWN            = "DropdownListButton";
static const char* const DATA_CONTEXT_COMBOBOX_LIST_ITEMS_5 = "Combobox_ListItem5";
static const float CARMODE_TOTAL_WEIGHT_IN_TONNE = 2.0f;
static const float CARMODE_LOAD_PER_AXLE_TONNE = 1.0f;
static const float COACH_MIN_WEIGHT_IN_TONNE = 10.0f;
static const float COACH_MIN_LOAD_PER_AXLE_TONNE = 7.0f;
static const int CAR_VEHICLE_PARAMETER_X_POSITION = 540;
static const int CAR_COACH_VEHICLE_PARAMETER_Y_POSITION = 160;
static const int COACH_VEHICLE_PARAMETER_X_POSITION = 424;
static const char* CMN_IMG_GUIDANCE_CARMODE_ICON_NORMAL = "AppHmi_NavigationModule#Images#Common#Icon_Guidance_carmode_normalBmp";
static const char* CMN_IMG_GUIDANCE_CARMODE_ICON_PRESSED = "AppHmi_NavigationModule#Images#Common#Icon_Guidance_carmode_touchedBmp";
static const char* CMN_IMG_GUIDANCE_COACH_ICON_NORMAL = "AppHmi_NavigationModule#Images#Common#Icon_Guidance_normalBmp";
static const char* CMN_IMG_GUIDANCE_COACH_ICON_PRESSED = "AppHmi_NavigationModule#Images#Common#Icon_Guidance_touchedBmp";

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST

VehicleParameterListHandler::VehicleParameterListHandler(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore)
   : HMIModelBase(navMiddleware, infoStore)
   , _listId(LIST_ID_NAV_SETTINGS_VEHICLE_PARAMETER)
   , _isCoach(false)
   , _isStaticList(false)
   , _isVehicleProfileChangedDuringActiveRG(false)
   , _profileId(USE_VEHICLETYPE_USER_1)
   , _currentVehicleProfileId(COACH_PROFILE_1)
   , _previousActiveProfileId(USE_VEHICLETYPE_USER_1)
   , vehicleSettingsMetaData(NULL)
   , _distanceUnitSystem(navmiddleware::DISTANCEUNITSYSTEM_METRIC)
#else
VehicleParameterListHandler::VehicleParameterListHandler()
   : _navMiddleware()
   , _infoStore()
   , _mocklistItemObj()
   , listBuilder(_mocklistItemObj)
#endif //VARIANT_S_FTR_ENABLE_UNITTEST
{
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_NAV_SETTINGS_VEHICLE_PARAMETER, this);
#endif

   memset(_previousVehicleProfile, 0, MAX_SIZE_VEHICLE_SETTINGS * sizeof(_previousVehicleProfile[0]));
   memset(_currentVehicleProfile,  0, MAX_SIZE_VEHICLE_SETTINGS * sizeof(_currentVehicleProfile[0]));
   memset(_cargoType,              0, MAX_SIZE_HAZARDOUS_DROP_DOWN_LIST * sizeof(_cargoType[0]));
}


VehicleParameterListHandler::~VehicleParameterListHandler()
{
   ETG_TRACE_USR4(("VehicleParameterListHandler::VehicleParameterListHandler Destructor"));
}


/********************************************
/* NAME    : VehicleParameterListHandler
/* SYSFL    : null,
****************************************************/
void VehicleParameterListHandler::initialize()
{
   ETG_TRACE_USR4(("VehicleParameterListHandler::initialize()"));
   _navMiddleware.registerGuidancePropertyUpdateCallback(*this);
   _navMiddleware.registerRoutePropertyUpdateCallback(*this);
   VehicleTypeValue vehicleType = USE_VEHICLETYPE_TRUCK;
   if (false == _infoStore.isTruckType())
   {
      _isCoach = true;
      vehicleType = USE_VEHICLETYPE_LIGHT_TRUCK;
   }

   _VehicleSettingsList.reserve(MAX_SIZE_VEHICLE_SETTINGS);
   _VehicleSettingsList.push_back(VehicleSettingsListItem(VehicleSettingsLabel[0], IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_DROPDOWN, VEHICLE_SETTINGS_PROFILE, ProfileDropDownListLabel[0], LIST_ID_NAV_SETTINGS_VEHICLE_PROFILE));
   _VehicleSettingsList.push_back(VehicleSettingsListItem(VehicleSettingsLabel[1], IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_INC_DEC_VALUE, VEHICLE_SETTINGS_HEIGHT));
   _VehicleSettingsList.push_back(VehicleSettingsListItem(VehicleSettingsLabel[2], IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_INC_DEC_VALUE, VEHICLE_SETTINGS_LENGTH));
   _VehicleSettingsList.push_back(VehicleSettingsListItem(VehicleSettingsLabel[3], IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_INC_DEC_VALUE, VEHICLE_SETTINGS_WIDTH));
   _VehicleSettingsList.push_back(VehicleSettingsListItem(VehicleSettingsLabel[4], IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_INC_DEC_VALUE, VEHICLE_SETTINGS_TOTAL_WEIGHT));
   _VehicleSettingsList.push_back(VehicleSettingsListItem(VehicleSettingsLabel[5], IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_INC_DEC_VALUE, VEHICLE_SETTINGS_LOAD_PER_AXLE));
   _VehicleSettingsList.push_back(VehicleSettingsListItem(VehicleSettingsLabel[6], IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_INC_DEC_VALUE, VEHICLE_SETTINGS_MAX_SPEED));
   if (!_isCoach)
   {
      _VehicleSettingsList.push_back(VehicleSettingsListItem(VehicleSettingsLabel[7], IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_DROPDOWN,        VEHICLE_SETTINGS_TRAILER_TYPE, TrailerTypeDropDownListLabel[0], LIST_ID_NAV_SETTINGS_VEHICLE_TRAILER_TYPE));
      _VehicleSettingsList.push_back(VehicleSettingsListItem(VehicleSettingsLabel[8], IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_DROPDOWN,        VEHICLE_SETTINGS_GOODS_TYPE, HazardousDropDownListLabel[0], LIST_ID_NAV_SETTINGS_VEHICLE_HAZARDOUS));
   }

   (*_vehicleProfileInfo).mIsTruckType = _infoStore.isTruckType();
   _vehicleProfileInfo.MarkItemModified(ItemKey::VehicleProfileInfo::IsTruckTypeItem);
   _vehicleProfileInfo.SendUpdate(true);

   VehicleProfileSettings vehicleProfileSettings = _navMiddleware.getVehicleProfileSettings();
   if (_isCoach)
   {
      ActiveVehicleProfileType enNewProfileSelected = navmiddleware::settings::USE_VEHICLETYPE_DEFAULT;    // Initialized to Coach
      _currentVehicleProfile[VEHICLE_SETTINGS_PROFILE] = float(navmiddleware::settings::USE_VEHICLETYPE_USER_1);
      enNewProfileSelected = USE_VEHICLETYPE_USER_1;
      vehicleProfileSettings.setCurrentVehicleProfileType(enNewProfileSelected);
      VehicleProfile vehicleProfile = vehicleProfileSettings.getUserVehicleProfile(USE_VEHICLETYPE_USER_1);
      vehicleProfileSettings.setUserVehicleProfile(USE_VEHICLETYPE_USER_1, vehicleProfile);
      if (vehicleType != vehicleProfile.getVehicleType())
      {
         vehicleProfile = _infoStore.getDefaultVehicleProfile();
         vehicleProfileSettings.setUserVehicleProfile(USE_VEHICLETYPE_USER_1, vehicleProfile);
      }
   }
   else
   {
      vehicleProfileSettings.setDefaultVehicleProfile(_infoStore.getDefaultVehicleProfile());
      // Maps the enum value in HMI with the enum value in navmiddleware
      _cargoType[VEHICLE_SETTINGS_NO_LOAD]                         = USE_CARGOTYPE_HAS_NO_LOAD;
      _cargoType[VEHICLE_SETTINGS_NO_HAZARDOUS]                    = USE_CARGOTYPE_HAS_SPECIAL_LOAD;
      _cargoType[VEHICLE_SETTINGS_WATER_POLLUTING]                 = USE_CARGOTYPE_HAS_WATER_POLLUTING;
      _cargoType[VEHICLE_SETTINGS_EXPLOSIVES]                      = USE_CARGOTYPE_HAS_EXPLOSIVE;
      _cargoType[VEHICLE_SETTINGS_GAS]                             = USE_CARGOTYPE_HAS_GASSES;

      // If the vehicle type for user profile from navmiddleware is set to CAR, then default KDS values are stored for the user profile
      for (int index = USE_VEHICLETYPE_USER_1; index < USE_VEHICLETYPE_DEFAULT; index++)
      {
         VehicleProfile vehicleProfile = vehicleProfileSettings.getUserVehicleProfile(index);

         if (vehicleType != vehicleProfile.getVehicleType())
         {
            vehicleProfile = _infoStore.getDefaultVehicleProfile();
            vehicleProfileSettings.setUserVehicleProfile(index, vehicleProfile);
         }
      }
   }
   _previousActiveProfileId = _navMiddleware.getVehicleProfileSettings().getCurrentVehicleProfileType();
   updateUnitSystem();
}


/********************************************
/* NAME    : VehicleParameterListHandler
/* SYSFL    : null,
****************************************************/
void VehicleParameterListHandler::deinitialize()
{
   ETG_TRACE_USR4(("VehicleParameterListHandler::deinitialize()"));
   _navMiddleware.unregisterGuidancePropertyUpdateCallback(*this);
   _navMiddleware.unregisterRoutePropertyUpdateCallback(*this);
   //do nothing
}


/********************************************
/* NAME    : VehicleParameterListHandler
/* SYSFL    : null,
****************************************************/
void VehicleParameterListHandler::updateUnitSystem()
{
   ETG_TRACE_USR4(("VehicleParameterListHandler::updateUnitSystem(), Unit is %d", _infoStore.getDistanceUnitSystem()));
   VEHICLE_PROFILE_CHANGED = true;

   _distanceUnitSystem = static_cast<navmiddleware::DistanceUnitSystem>(_infoStore.getDistanceUnitSystem());
   if (navmiddleware::DISTANCEUNITSYSTEM_METRIC == _distanceUnitSystem)
   {
      vehicleSettingsMetaData = &vehicleSettingsMetricMetaData[0];
      ETG_TRACE_USR4(("VehicleParameterListHandler::vehicleSettingsMetricMetaData is set"));
   }
   else
   {
      vehicleSettingsMetaData = &vehicleSettingsImperialMetaData[0];
      ETG_TRACE_USR4(("VehicleParameterListHandler::vehicleSettingsImperialMetaData is set"));
   }

   VehicleProfileSettings vehicleProfileSettings = _navMiddleware.getVehicleProfileSettings();
   if (_isCoach)
   {
      VehicleProfile vehicleProfile = vehicleProfileSettings.getUserVehicleProfile(USE_VEHICLETYPE_USER_1);
      navmiddleware::DistanceUnitSystem vehicleProfileUnit = vehicleProfile.getUnitSystem();
      ETG_TRACE_USR4(("VehicleParameterListHandler::updateUnitSystem(), Unit for User 1 vehicle profile : %d", vehicleProfileUnit));

      if (_distanceUnitSystem != vehicleProfileUnit)
      {
         VehicleSettingsHelper::convertVehicleProfileUnits(vehicleProfile, _distanceUnitSystem);
         vehicleProfileSettings.setUserVehicleProfile(USE_VEHICLETYPE_USER_1, vehicleProfile);
      }
   }
   else
   {
      for (int index = USE_VEHICLETYPE_USER_1; index < USE_VEHICLETYPE_DEFAULT; index++)
      {
         VehicleProfile vehicleProfile = vehicleProfileSettings.getUserVehicleProfile(index);
         navmiddleware::DistanceUnitSystem vehicleProfileUnit = vehicleProfile.getUnitSystem();
         ETG_TRACE_USR4(("VehicleParameterListHandler::updateUnitSystem(), Unit for %d vehicle profile : %d", index, vehicleProfileUnit));

         if (_distanceUnitSystem != vehicleProfileUnit)
         {
            VehicleSettingsHelper::convertVehicleProfileUnits(vehicleProfile, _distanceUnitSystem);
            vehicleProfileSettings.setUserVehicleProfile(index, vehicleProfile);
         }
      }
   }

   VehicleProfile defaultProfile = vehicleProfileSettings.getDefaultVehicleProfile();
   navmiddleware::DistanceUnitSystem vehicleProfileUnit = defaultProfile.getUnitSystem();
   ETG_TRACE_USR4(("VehicleParameterListHandler::updateUnitSystem(), Unit for default vehicle profile : %d", vehicleProfileUnit));

   if (_distanceUnitSystem != vehicleProfileUnit)
   {
      VehicleSettingsHelper::convertVehicleProfileUnits(defaultProfile, _distanceUnitSystem);
      vehicleProfileSettings.setDefaultVehicleProfile(defaultProfile);
   }
   _navMiddleware.updateVehicleProfileSettings(vehicleProfileSettings);
   _navMiddleware.applySettings();

   getCurrentVehicleProfile();
}


/********************************************
/* NAME    : VehicleParameterListHandler
/* SYSFL    : null,
****************************************************/
bool VehicleParameterListHandler::onCourierMessage(const ButtonListItemUpdMsg& oMsg)
{
   bool isListChanged         = false;
   bool isMsgConsumed         = false;
   Candera::UInt32 listId     = oMsg.GetListId();
   unsigned int rowIdx        = (unsigned int)oMsg.GetHdl();
   unsigned int colIdx        = (unsigned int)oMsg.GetSubHdl();
   const enReaction reaction  = oMsg.GetReaction();

   if (enRelease == reaction)
   {
      if ((LIST_ID_NAV_SETTINGS_VEHICLE_PARAMETER == listId) && (MAX_SIZE_VEHICLE_SETTINGS > rowIdx))
      {
         ETG_TRACE_USR4(("VehicleParameterListHandler::onCourierMessage(ButtonListItemUpdMsg), List Id : %d, Row Id : %d", listId, rowIdx));
         switch (rowIdx)
         {
            case VEHICLE_SETTINGS_PROFILE:
            case VEHICLE_SETTINGS_TRAILER_TYPE:
            case VEHICLE_SETTINGS_GOODS_TYPE:
            {
               processComboBox(rowIdx, colIdx);
               isMsgConsumed = true;
            }
            break;
            default:
            {
               float stepValue = vehicleSettingsMetaData[rowIdx].step;
               // Increment or decrement settings value
               if ((DATA_ITEM_DECREMENT == colIdx))
               {
                  _currentVehicleProfile[rowIdx] = _currentVehicleProfile[rowIdx] - stepValue;
                  if (VEHICLE_SETTINGS_WIDTH != rowIdx)
                  {
                     _currentVehicleProfile[rowIdx] = ROUNDF_1DECIMAL(_currentVehicleProfile[rowIdx]);
                  }
                  else
                  {
                     _currentVehicleProfile[rowIdx] = ROUNDF_2DECIMAL(_currentVehicleProfile[rowIdx]);
                  }
                  if (_currentVehicleProfile[rowIdx] < vehicleSettingsMetaData[rowIdx].minValue)
                  {
                     _currentVehicleProfile[rowIdx] = vehicleSettingsMetaData[rowIdx].minValue;
                  }
               }

               else if ((DATA_ITEM_INCREMENT == colIdx))
               {
                  _currentVehicleProfile[rowIdx] = _currentVehicleProfile[rowIdx] + stepValue;
                  if (VEHICLE_SETTINGS_WIDTH != rowIdx)
                  {
                     _currentVehicleProfile[rowIdx] = ROUNDF_1DECIMAL(_currentVehicleProfile[rowIdx]);
                  }
                  else
                  {
                     _currentVehicleProfile[rowIdx] = ROUNDF_2DECIMAL(_currentVehicleProfile[rowIdx]);
                  }
                  if (_currentVehicleProfile[rowIdx] > vehicleSettingsMetaData[rowIdx].maxValue)
                  {
                     _currentVehicleProfile[rowIdx] = vehicleSettingsMetaData[rowIdx].maxValue;
                  }
                  ETG_TRACE_USR4(("_currentVehicleProfile[rowIdx]: %d", _currentVehicleProfile[rowIdx]));
               }

               isListChanged = true;
               isMsgConsumed = true;
            }
            break;
         }
      }
      if (true == isListChanged)
      {
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
         ListRegistry::s_getInstance().updateList(LIST_ID_NAV_SETTINGS_VEHICLE_PARAMETER);
#endif
      }
   }
   return isMsgConsumed;
}


bool VehicleParameterListHandler::onCourierMessage(const ButtonReactionMsg& oMsg)
{
   bool bIsMsgConsumed = false;

   const Courier::ViewId VEHICLE_PROFILEINFO_SCENENAME = Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_Dest_VehicleInfo");
   const Courier::Identifier IdCoachBtn = Courier::Identifier("Button_Coach/ButtonWidget");
   const Courier::Identifier IdCarBtn = Courier::Identifier("Button_Car/ButtonWidget");

   const Courier::Identifier senderInfo = oMsg.GetSender();
   const Courier::ViewId sceneName = oMsg.GetView();
   const enReaction reaction = oMsg.GetEnReaction();

   if ((VEHICLE_PROFILEINFO_SCENENAME == sceneName) && (enRelease == reaction))
   {
      ETG_TRACE_USR4(("VehicleParameterListHandler::onCourierMessage(ButtonReactionMsg), Scene name : %s", sceneName.CStr()));
      // Vehicle profile
      Candera::String currentProfileLength;
      Candera::String currentProfileHeight;
      Candera::String currentProfileWidth;
      Candera::String currentProfileTotalWeight;
      Candera::String currentProfileMaxSpeed;
      Candera::String currentProfileLoadPerAxle;

      currentProfileLength = VehicleSettingsHelper::convertHmiDimensionToString(VEHICLE_SETTINGS_LENGTH, _currentVehicleProfile[VEHICLE_SETTINGS_LENGTH], _distanceUnitSystem);
      currentProfileHeight = VehicleSettingsHelper::convertHmiDimensionToString(VEHICLE_SETTINGS_HEIGHT, _currentVehicleProfile[VEHICLE_SETTINGS_HEIGHT], _distanceUnitSystem);
      currentProfileWidth = VehicleSettingsHelper::convertHmiDimensionToString(VEHICLE_SETTINGS_WIDTH, _currentVehicleProfile[VEHICLE_SETTINGS_WIDTH], _distanceUnitSystem);
      currentProfileMaxSpeed = VehicleSettingsHelper::convertHmiDimensionToString(VEHICLE_SETTINGS_MAX_SPEED, _currentVehicleProfile[VEHICLE_SETTINGS_MAX_SPEED], _distanceUnitSystem);

      if ((IdCarBtn == senderInfo))
      {
         currentProfileTotalWeight = VehicleSettingsHelper::convertHmiDimensionToString(VEHICLE_SETTINGS_TOTAL_WEIGHT, CARMODE_TOTAL_WEIGHT_IN_TONNE, _distanceUnitSystem);
         currentProfileLoadPerAxle = VehicleSettingsHelper::convertHmiDimensionToString(VEHICLE_SETTINGS_LOAD_PER_AXLE, CARMODE_LOAD_PER_AXLE_TONNE, _distanceUnitSystem);

         (*_vehicleProfileInfo).mPosition = Candera::Vector2(CAR_VEHICLE_PARAMETER_X_POSITION, CAR_COACH_VEHICLE_PARAMETER_Y_POSITION);
         (*_vehicleProfileInfo).mCoachActive = false;
         (*_vehicleProfileInfo).mCarActive = true;
         setStaticVehicleParameter(false);
         (*_vehicleProfileInfo).mLength = currentProfileLength;
         (*_vehicleProfileInfo).mHeight = currentProfileHeight;
         (*_vehicleProfileInfo).mWidth = currentProfileWidth;
         (*_vehicleProfileInfo).mTotalWeight = currentProfileTotalWeight;
         (*_vehicleProfileInfo).mMaxSpeed = currentProfileMaxSpeed;
         (*_vehicleProfileInfo).mLoadPerAxle = currentProfileLoadPerAxle;
         (*_vehicleProfileInfo).mIconNormalBitmap = ImageLoader::getAssetBitmapImage(CMN_IMG_GUIDANCE_CARMODE_ICON_NORMAL);
         (*_vehicleProfileInfo).mIconPressedBitmap = ImageLoader::getAssetBitmapImage(CMN_IMG_GUIDANCE_CARMODE_ICON_PRESSED);
         _vehicleProfileInfo.MarkAllItemsModified();
         _vehicleProfileInfo.SendUpdate(true);
         EXT_bIsCarModeWarningApplicable = true;
      }
      else if ((IdCoachBtn == senderInfo))
      {
         currentProfileTotalWeight = VehicleSettingsHelper::convertHmiDimensionToString(VEHICLE_SETTINGS_TOTAL_WEIGHT, _currentVehicleProfile[VEHICLE_SETTINGS_TOTAL_WEIGHT], _distanceUnitSystem);
         currentProfileLoadPerAxle = VehicleSettingsHelper::convertHmiDimensionToString(VEHICLE_SETTINGS_LOAD_PER_AXLE, _currentVehicleProfile[VEHICLE_SETTINGS_LOAD_PER_AXLE], _distanceUnitSystem);
         (*_vehicleProfileInfo).mPosition = Candera::Vector2(COACH_VEHICLE_PARAMETER_X_POSITION, CAR_COACH_VEHICLE_PARAMETER_Y_POSITION);
         (*_vehicleProfileInfo).mCoachActive = true;
         (*_vehicleProfileInfo).mCarActive = false;
         setStaticVehicleParameter(true);
         (*_vehicleProfileInfo).mLength = currentProfileLength;
         (*_vehicleProfileInfo).mHeight = currentProfileHeight;
         (*_vehicleProfileInfo).mWidth = currentProfileWidth;
         (*_vehicleProfileInfo).mTotalWeight = currentProfileTotalWeight;
         (*_vehicleProfileInfo).mMaxSpeed = currentProfileMaxSpeed;
         (*_vehicleProfileInfo).mLoadPerAxle = currentProfileLoadPerAxle;
         (*_vehicleProfileInfo).mIconNormalBitmap = ImageLoader::getAssetBitmapImage(CMN_IMG_GUIDANCE_COACH_ICON_NORMAL);
         (*_vehicleProfileInfo).mIconPressedBitmap = ImageLoader::getAssetBitmapImage(CMN_IMG_GUIDANCE_COACH_ICON_PRESSED);
         _vehicleProfileInfo.MarkAllItemsModified();
         _vehicleProfileInfo.SendUpdate(true);
         EXT_bIsCarModeWarningApplicable = false;
      }
      else
      {
         // do nothing
      }
   }
}


void VehicleParameterListHandler::processComboBox(Courier::UInt32 hdl, Courier::UInt32 subHdl)
{
   ETG_TRACE_USR4(("VehicleParameterListHandler::processComboBox: Hdl: %d, SubHdl: %d", hdl, subHdl));
   subHdl = subHdl - 1;  //subtracting 1 as the dropdown item values start with the index of 0..
   switch (hdl)
   {
      case VEHICLE_SETTINGS_PROFILE:
      {
         if (!_isCoach)
         {
            if ((MAX_SIZE_PROFILE_DROP_DOWN_LIST > subHdl) && (UINT(_currentVehicleProfile[VEHICLE_SETTINGS_PROFILE]) != subHdl))
            {
               _profileId = subHdl;
               bool hasSettingsChanged = hasVehicleSettingsChanged();
               if (true == hasSettingsChanged)
               {
                  updateCurrentVehicleProfile();
               }
               changeActiveProfile();
            }
         }
      }
      break;
      case VEHICLE_SETTINGS_TRAILER_TYPE:
      {
         if ((MAX_SIZE_TRAILER_DROP_DOWN_LIST > subHdl) && (_currentVehicleProfile[VEHICLE_SETTINGS_TRAILER_TYPE] != float(subHdl)))
         {
            _currentVehicleProfile[VEHICLE_SETTINGS_TRAILER_TYPE] = float(subHdl);
         }
      }
      break;
      case VEHICLE_SETTINGS_GOODS_TYPE:
      {
         if ((MAX_SIZE_HAZARDOUS_DROP_DOWN_LIST > subHdl) && (_currentVehicleProfile[VEHICLE_SETTINGS_GOODS_TYPE] != float(_cargoType[subHdl])))
         {
            _currentVehicleProfile[VEHICLE_SETTINGS_GOODS_TYPE] = float(_cargoType[subHdl]);
         }
      }
      break;
      default:
         break;
   }

   ::hmibase::widget::combobox::ComboBoxReqMsg* cbToggleMsg = COURIER_MESSAGE_NEW(::hmibase::widget::combobox::ComboBoxReqMsg)(0, ::hmibase::widget::combobox::enComboBoxAction::Close); //ComboBoxAction enComboBoxAction::Enum::ComboBoxClose:
   if (cbToggleMsg != NULL)
   {
      cbToggleMsg->Post();
   }
   ListRegistry::s_getInstance().updateList(LIST_ID_NAV_SETTINGS_VEHICLE_PARAMETER);
}


/********************************************
/* NAME    : VehicleParameterListHandler
/* SYSFL    : null,
****************************************************/
bool VehicleParameterListHandler::onCourierMessage(const CheckIsVehicleSettingsChangedReqMsg& oMsg)
{
   ETG_TRACE_USR4(("VehicleParameterListHandler::onCourierMessage(CheckIsVehicleSettingsChangedReqMsg()"));

   bool hasSettingsChanged = hasVehicleSettingsChanged();
   bool isActiveProfileChanged = false;
   if (_previousActiveProfileId != _navMiddleware.getVehicleProfileSettings().getCurrentVehicleProfileType())
   {
      _previousActiveProfileId = _navMiddleware.getVehicleProfileSettings().getCurrentVehicleProfileType();// Update the previous active vehicle profile on exit from vehicle settings.
      isActiveProfileChanged = true;
   }
   // Message is posted to Controller if settings has changed or not
   if ((true == hasSettingsChanged) || (true == isActiveProfileChanged))
   {
      updateCurrentVehicleProfile();
      if (_navMiddleware.isGuidanceActive() == true)
      {
         _isVehicleProfileChangedDuringActiveRG = true;
         POST_MSG((COURIER_MESSAGE_NEW(PopupTypeUpdateMsg)(POPUP_VEHICLE_SETTINGS_SAVE_DURING_RG))); // Update the popuptype to controller
         POST_MSG((COURIER_MESSAGE_NEW(UpdateDeletePopupTextReqMsg)(POPUP_VEHICLE_SETTINGS_SAVE_DURING_RG))); // updates the text and displays the popup
      }
      else
      {
         _isVehicleProfileChangedDuringActiveRG = false;
      }
   }
   return true;
}


void VehicleParameterListHandler::onPropertyUpdateRouteCalculationStopped()
{
   ETG_TRACE_USR4(("VehicleParameterListHandler::onPropertyUpdateRouteCalculationStopped()"));
   restoreCoachProfileValue();
}


void VehicleParameterListHandler::onPropertyUpdateGuidanceStateChanged()
{
   ETG_TRACE_USR4(("VehicleParameterListHandler::onPropertyUpdateGuidanceStateChanged"));
   if (!EXT_bIsCarModeWarningApplicable)
   {
      setStaticVehicleParameter(true);
   }
   if (false == EXT_bIsGuidanceActive)
   {
      restoreCoachProfileValue();
   }
   _currentVehicleProfileId = _infoStore.getVehicleProfile();

   if (_isVehicleProfileChangedDuringActiveRG)
   {
      ETG_TRACE_USR4(("VehicleParameterListHandler::onPropertyUpdateGuidanceStateChanged(inside condition)"));
      _isVehicleProfileChangedDuringActiveRG = false;
      POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Hide, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_MultiLine_TwoButton"))));
      POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Destroy, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_MultiLine_TwoButton"))));
   }
}


/********************************************
/* NAME    : VehicleParameterListHandler
/* SYSFL    : null,
****************************************************/
void VehicleParameterListHandler::restoreCoachProfileValue()
{
   ETG_TRACE_USR4(("VehicleParameterListHandler::restoreCoachProfileValue()"));
   VehicleProfileSettings vehicleProfileSettings = _navMiddleware.getVehicleProfileSettings();
   VehicleProfile vehicleProfile = vehicleProfileSettings.getUserVehicleProfile(USE_VEHICLETYPE_USER_1);
   unsigned int weightPressCtrl = VehicleSettingsHelper::convertDimensionToPresCtrlUnit(VEHICLE_SETTINGS_TOTAL_WEIGHT, _currentVehicleProfile[VEHICLE_SETTINGS_TOTAL_WEIGHT], _distanceUnitSystem);
   vehicleProfile.setMaxWeightOfVehicle(weightPressCtrl);
   unsigned int loadPerAxlePressCtrl = VehicleSettingsHelper::convertDimensionToPresCtrlUnit(VEHICLE_SETTINGS_LOAD_PER_AXLE, _currentVehicleProfile[VEHICLE_SETTINGS_LOAD_PER_AXLE], _distanceUnitSystem);
   vehicleProfile.setMaxLoadPerAxleOfVehicle(loadPerAxlePressCtrl);
   vehicleProfileSettings.setUserVehicleProfile(USE_VEHICLETYPE_USER_1, vehicleProfile);
   _navMiddleware.updateVehicleProfileSettings(vehicleProfileSettings);
   _navMiddleware.applySettings();
}


bool VehicleParameterListHandler::onCourierMessage(const UpdateVehicleProfilePopupStatusReqMsg& oMsg)
{
   _isVehicleProfileChangedDuringActiveRG = oMsg.GetIsGuidanceRetriggered();
   ETG_TRACE_USR4(("VehicleParameterListHandler::onCourierMessage(UpdateVehicleProfilePopupStatusReqMsg(%d))", _isVehicleProfileChangedDuringActiveRG));
   return true;
}


/********************************************
/* NAME    : VehicleParameterListHandler
/* SYSFL    : null,
****************************************************/
bool VehicleParameterListHandler::onCourierMessage(const ChangeActiveProfileReqMsg& oMsg)
{
   ETG_TRACE_USR4(("VehicleParameterListHandler::onCourierMessage(ChangeActiveProfileReqMsg()"));

   changeActiveProfile();

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
   ListRegistry::s_getInstance().updateList(LIST_ID_NAV_SETTINGS_VEHICLE_PARAMETER);
#endif

   return true;
}


/********************************************
/* NAME    : VehicleParameterListHandler
/* SYSFL    : null,
****************************************************/
bool VehicleParameterListHandler::onCourierMessage(const SaveVehicleProfileSettingsReqMsg& oMsg)
{
   ETG_TRACE_USR4(("VehicleParameterListHandler::onCourierMessage(SaveVehicleProfileSettingsReqMsg()"));
   VEHICLE_PROFILE_CHANGED = true;
   updateCurrentVehicleProfile();
   if (_navMiddleware.isGuidanceActive() == true)
   {
      _isVehicleProfileChangedDuringActiveRG = true;
      POST_MSG((COURIER_MESSAGE_NEW(VehicleParameterStatusUpdMsg)()));
   }
   else
   {
      _isVehicleProfileChangedDuringActiveRG = false;
   }

   return true;
}


bool VehicleParameterListHandler::onCourierMessage(const CheckVehicleParameterStatusMsg& oMsg)
{
   ETG_TRACE_USR4(("VehicleParameterListHandler::onCourierMessage(CheckVehicleParameterStatusMsg()"));

   if ((true == _isVehicleProfileChangedDuringActiveRG) && (true == _navMiddleware.isGuidanceActive()))
   {
      _isVehicleProfileChangedDuringActiveRG = false;
      POST_MSG((COURIER_MESSAGE_NEW(RetriggerLastGuidedRouteReqMsg)()));
   }

   return true;
}


/********************************************
/* NAME    : VehicleParameterListHandler
/* SYSFL    : null,
****************************************************/
bool VehicleParameterListHandler::onCourierMessage(const ResetVehicleProfileSettingsReqMsg& oMsg)
{
   ETG_TRACE_USR4(("VehicleParameterListHandler::onCourierMessage(ResetVehicleProfileSettingsReqMsg()"));
   VEHICLE_PROFILE_CHANGED = false;
   if (false == _isStaticList)
   {
      // Reset the current value with the previous value if the settings is not changed
      for (int index = VEHICLE_SETTINGS_HEIGHT; index < MAX_SIZE_VEHICLE_SETTINGS; index++)
      {
         _currentVehicleProfile[index] = _previousVehicleProfile[index];
      }
   }
   return true;
}


/********************************************
/* NAME    : VehicleParameterListHandler
/* SYSFL    : null,
****************************************************/
bool VehicleParameterListHandler::onCourierMessage(const ResetVehicleProfileInfoReqMsg& oMsg)
{
   ETG_TRACE_USR4(("VehicleParameterListHandler::onCourierMessage ResetVehicleProfileInfoReqMsg"));
   EXT_bIsCarModeWarningApplicable = false;
   VEHICLE_PROFILE_CHANGED = false;
   return true;
}


/********************************************
/* NAME    : VehicleParameterListHandler
/* SYSFL    : null,
****************************************************/
bool VehicleParameterListHandler::onCourierMessage(const UpdateStartupVehicleProfileReqMsg& oMsg)
{
   ETG_TRACE_USR4(("VehicleParameterListHandler::onCourierMessage(UpdateStartupVehicleProfileReqMsg()"));
   // Vehicle profile
   Candera::String currentProfileInfoText;
   Candera::String currentProfileLength;
   Candera::String currentProfileHeight;
   Candera::String currentProfileWidth;
   Candera::String currentProfileTotalWeight;
   Candera::String currentProfileMaxSpeed;
   Candera::String currentProfileLoadPerAxle;

   currentProfileLength = VehicleSettingsHelper::convertHmiDimensionToString(VEHICLE_SETTINGS_LENGTH, _currentVehicleProfile[VEHICLE_SETTINGS_LENGTH], _distanceUnitSystem);
   currentProfileHeight = VehicleSettingsHelper::convertHmiDimensionToString(VEHICLE_SETTINGS_HEIGHT, _currentVehicleProfile[VEHICLE_SETTINGS_HEIGHT], _distanceUnitSystem);
   currentProfileWidth = VehicleSettingsHelper::convertHmiDimensionToString(VEHICLE_SETTINGS_WIDTH, _currentVehicleProfile[VEHICLE_SETTINGS_WIDTH], _distanceUnitSystem);
   currentProfileTotalWeight = VehicleSettingsHelper::convertHmiDimensionToString(VEHICLE_SETTINGS_TOTAL_WEIGHT, _currentVehicleProfile[VEHICLE_SETTINGS_TOTAL_WEIGHT], _distanceUnitSystem);
   currentProfileMaxSpeed = VehicleSettingsHelper::convertHmiDimensionToString(VEHICLE_SETTINGS_MAX_SPEED, _currentVehicleProfile[VEHICLE_SETTINGS_MAX_SPEED], _distanceUnitSystem);
   currentProfileLoadPerAxle = VehicleSettingsHelper::convertHmiDimensionToString(VEHICLE_SETTINGS_LOAD_PER_AXLE, _currentVehicleProfile[VEHICLE_SETTINGS_LOAD_PER_AXLE], _distanceUnitSystem);
   EXT_bIsCarModeWarningApplicable = false;

   // Trailer type
   currentProfileInfoText = vehicleProfileTextUpdations(VEHICLE_SETTINGS_TRAILER_TYPE, UINT(_currentVehicleProfile[VEHICLE_SETTINGS_TRAILER_TYPE]), _isCoach);
   (*_vehicleProfileInfo).mTrailerType = currentProfileInfoText;

   // Cargo type
   int index = 0;
   for (; index < MAX_SIZE_HAZARDOUS_DROP_DOWN_LIST; index++)
   {
      if ((unsigned int)_cargoType[index] == UINT(_currentVehicleProfile[VEHICLE_SETTINGS_GOODS_TYPE]))
      {
         break;
      }
   }
   if (index < MAX_SIZE_HAZARDOUS_DROP_DOWN_LIST)
   {
      currentProfileInfoText = HazardousDropDownListLabel[index];
   }
   (*_vehicleProfileInfo).mGoodsType = currentProfileInfoText;
   (*_vehicleProfileInfo).mPosition = Candera::Vector2(COACH_VEHICLE_PARAMETER_X_POSITION, CAR_COACH_VEHICLE_PARAMETER_Y_POSITION);
   (*_vehicleProfileInfo).mCoachActive = true;
   (*_vehicleProfileInfo).mCarActive = false;
   (_infoStore.getVehicleProfile()) ? (setStaticVehicleParameter(false)) : (setStaticVehicleParameter(true));

   (*_vehicleProfileInfo).mLength = currentProfileLength;
   (*_vehicleProfileInfo).mHeight = currentProfileHeight;
   (*_vehicleProfileInfo).mWidth = currentProfileWidth;
   (*_vehicleProfileInfo).mTotalWeight = currentProfileTotalWeight;
   (*_vehicleProfileInfo).mMaxSpeed = currentProfileMaxSpeed;
   (*_vehicleProfileInfo).mLoadPerAxle = currentProfileLoadPerAxle;
   (*_vehicleProfileInfo).mIconNormalBitmap = ImageLoader::getAssetBitmapImage(CMN_IMG_GUIDANCE_COACH_ICON_NORMAL);
   (*_vehicleProfileInfo).mIconPressedBitmap = ImageLoader::getAssetBitmapImage(CMN_IMG_GUIDANCE_COACH_ICON_PRESSED);
   _vehicleProfileInfo.MarkAllItemsModified();
   _vehicleProfileInfo.SendUpdate(true);
   return true;
}


/********************************************
/* NAME    : VehicleParameterListHandler
/* SYSFL    : null,
****************************************************/
bool VehicleParameterListHandler::onCourierMessage(const UpdateDeletePopupTextReqMsg& oMsg)
{
   bool bIsMsgConsumed = false;
   unsigned int popupType = oMsg.GetPopupTextID();
   ETG_TRACE_USR4(("VehicleParameterListHandler::UpdateDeletePopupTextReqMsg: popupType= %d", popupType));

   const Candera::String DATA_CONTEXT__TEXT_VEHICLESETTINGS_SAVE_DURING_RG_FIRSTLINE = "Do you want to restart the guidance";
   const Candera::String DATA_CONTEXT__TEXT_VEHICLESETTINGS_SAVE_DURING_RG_SECONDLINE = "with the new vehicle settings?";

   if (POPUP_VEHICLE_SETTINGS_SAVE_DURING_RG == popupType)
   {
      (*_PopupText).mPopuptextLine1 = DATA_CONTEXT__TEXT_VEHICLESETTINGS_SAVE_DURING_RG_FIRSTLINE;
      (*_PopupText).mPopuptextLine2 = DATA_CONTEXT__TEXT_VEHICLESETTINGS_SAVE_DURING_RG_SECONDLINE;
      _PopupText.MarkAllItemsModified();
      _PopupText.SendUpdate(true);
      POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_MultiLine_TwoButton"))));
      bIsMsgConsumed = true;
   }
   return bIsMsgConsumed;
}


/********************************************
/* NAME    : VehicleParameterListHandler
/* SYSFL    : null,
****************************************************/
bool VehicleParameterListHandler::onCourierMessage(const ResetAllNaviSettingsReqMsg& oMsg)
{
   ETG_TRACE_USR4(("VehicleParameterListHandler::onCourierMessage(ResetAllNaviSettingsReqMsg)"));

   bool bIsMsgConsumed = false;   // Set to false so that other navigation settings handler can process this message

   VehicleProfileSettings vehicleProfileSettings = _navMiddleware.getVehicleProfileSettings();
   VehicleProfile defaultProfile = vehicleProfileSettings.getDefaultVehicleProfile();
   navmiddleware::DistanceUnitSystem vehicleProfileUnit = defaultProfile.getUnitSystem();
   navmiddleware::DistanceUnitSystem distanceUnit = static_cast<navmiddleware::DistanceUnitSystem>(_infoStore.getDistanceUnitSystem());
   ETG_TRACE_USR4(("VehicleParameterListHandler::onCourierMessage(ResetAllNaviSettingsReqMsg), Unit for default vehicle profile : %d, Distance unit : %d", vehicleProfileUnit, distanceUnit));

   if (distanceUnit != vehicleProfileUnit)
   {
      VehicleSettingsHelper::convertVehicleProfileUnits(defaultProfile, distanceUnit);
      vehicleProfileSettings.setUserVehicleProfile(USE_VEHICLETYPE_USER_1, defaultProfile);
      _navMiddleware.updateVehicleProfileSettings(vehicleProfileSettings);
      _navMiddleware.applySettings();
   }
   getCurrentVehicleProfile();   // After factory settings reset, current vehicle profile values are fetched again
   float coachWeight = _currentVehicleProfile[VEHICLE_SETTINGS_TOTAL_WEIGHT];
   float coachLoadPerAxle = _currentVehicleProfile[VEHICLE_SETTINGS_LOAD_PER_AXLE];
   _infoStore.setCoachWeight(coachWeight);
   _infoStore.setCoachLoadPerAxle(coachLoadPerAxle);
   return bIsMsgConsumed;
}


/********************************************
/* NAME    : VehicleParameterListHandler
/* SYSFL    : null,
****************************************************/
void VehicleParameterListHandler::setStaticVehicleParameter(bool isEnabled)
{
   ETG_TRACE_USR4(("VehicleParameterListHandler::setStaticVehicleParameter(%d)", isEnabled));

   if ((*_switchIndexData).mIndexValue != isEnabled)
   {
      (*_switchIndexData).mIndexValue = isEnabled;
      _switchIndexData.MarkAllItemsModified();
      _switchIndexData.SendUpdate();
   }
}


/********************************************
/* NAME    : VehicleParameterListHandler
/* SYSFL    : null,
****************************************************/
bool VehicleParameterListHandler::onCourierMessage(const AcceptCarModeWarningReqMsg& oMsg)
{
   ETG_TRACE_USR4(("VehicleParameterListHandler::onCourierMessage AcceptCarModeWarningReqMsg, Current vehicle profile = %d", UINT(_currentVehicleProfile[VEHICLE_SETTINGS_PROFILE])));

   VehicleProfileSettings vehicleProfileSettings = _navMiddleware.getVehicleProfileSettings();
   VehicleProfile vehicleProfile = vehicleProfileSettings.getUserVehicleProfile(USE_VEHICLETYPE_USER_1);
   float coachWeight = _previousVehicleProfile[VEHICLE_SETTINGS_TOTAL_WEIGHT];
   float coachLoadPerAxle = _previousVehicleProfile[VEHICLE_SETTINGS_LOAD_PER_AXLE];
   _infoStore.setCoachWeight(coachWeight);
   _infoStore.setCoachLoadPerAxle(coachLoadPerAxle);
   unsigned int weightPressCtrl = VehicleSettingsHelper::convertDimensionToPresCtrlUnit(VEHICLE_SETTINGS_TOTAL_WEIGHT, CARMODE_TOTAL_WEIGHT_IN_TONNE, _distanceUnitSystem);
   vehicleProfile.setMaxWeightOfVehicle(weightPressCtrl);
   unsigned int loadPerAxlePressCtrl = VehicleSettingsHelper::convertDimensionToPresCtrlUnit(VEHICLE_SETTINGS_LOAD_PER_AXLE, CARMODE_LOAD_PER_AXLE_TONNE, _distanceUnitSystem);
   vehicleProfile.setMaxLoadPerAxleOfVehicle(loadPerAxlePressCtrl);
   vehicleProfileSettings.setUserVehicleProfile(USE_VEHICLETYPE_USER_1, vehicleProfile);
   _navMiddleware.updateVehicleProfileSettings(vehicleProfileSettings);
   _navMiddleware.applySettings();

   if (EXT_bIsGuidanceReqFromSds)
   {
      POST_MSG((COURIER_MESSAGE_NEW(StartRouteGuidanceFromSpeechReqMsg)()));
   }
   else
   {
      POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(StartRouteGuidanceReqMsg)()));
   }
   return true;
}


/********************************************
/* NAME    : VehicleParameterListHandler
/* SYSFL    : null,
****************************************************/
bool VehicleParameterListHandler::onCourierMessage(const DeclineCarModeWarningReqMsg& oMsg)
{
   ETG_TRACE_USR4(("VehicleParameterListHandler::onCourierMessage DeclineCarModeWarningReqMsg"));
   return true;
}


/********************************************
/* NAME    : VehicleParameterListHandler
/* SYSFL    : null,
****************************************************/
bool VehicleParameterListHandler::onCourierMessage(const UpdateContextonBackgroundReqMsg& oMsg)
{
   ETG_TRACE_USR4(("VehicleParameterListHandler::onCourierMessage UpdateContextonBackgroundReqMsg"));
   EXT_bIsGuidanceReqFromSds = false;
   EXT_bDisplayCalcRoutePopup = true;
   return true;
}


/********************************************
/* NAME    : VehicleParameterListHandler
/* SYSFL    : null,
****************************************************/
bool VehicleParameterListHandler::hasVehicleSettingsChanged()
{
   ETG_TRACE_USR4(("VehicleParameterListHandler::hasVehicleSettingsChanged()"));

   bool hasSettingsChanged = false;
   if (false == _isStaticList)
   {
      // Compares current vehicle settings with previous vehicle settings
      for (int index = VEHICLE_SETTINGS_HEIGHT; index < MAX_SIZE_VEHICLE_SETTINGS; index++)
      {
         if (_previousVehicleProfile[index] != _currentVehicleProfile[index])
         {
            hasSettingsChanged = true;
            break;
         }
      }
   }
   return hasSettingsChanged;
}


/********************************************
/* NAME    : VehicleParameterListHandler
/* SYSFL    : null,
****************************************************/
void VehicleParameterListHandler::changeActiveProfile()
{
   ETG_TRACE_USR4(("VehicleParameterListHandler::changeActiveProfile(), Profile selected = %d", _profileId));
   VEHICLE_PROFILE_CHANGED = true;
   // Reset the existing settings value
   memset(_currentVehicleProfile, 0, MAX_SIZE_VEHICLE_SETTINGS * sizeof(_currentVehicleProfile[0]));
   ActiveVehicleProfileType enNewProfileSelected = navmiddleware::settings::USE_VEHICLETYPE_DEFAULT;    // Initialized to Default
   if (_isCoach)
   {
      if (COACH_PROFILE_1 == static_cast<CoachProfileType>(_profileId))
      {
         _currentVehicleProfile[VEHICLE_SETTINGS_PROFILE] = float(navmiddleware::settings::USE_VEHICLETYPE_USER_1);
         enNewProfileSelected = USE_VEHICLETYPE_USER_1;
      }
   }
   else
   {
      if (navmiddleware::settings::USE_VEHICLETYPE_DEFAULT >= static_cast<ActiveVehicleProfileType>(_profileId))
      {
         enNewProfileSelected = static_cast<ActiveVehicleProfileType>(_profileId);
         _currentVehicleProfile[VEHICLE_SETTINGS_PROFILE] = float(_profileId);
      }
   }

   VehicleProfileSettings vehicleProfileSettings = _navMiddleware.getVehicleProfileSettings();
   vehicleProfileSettings.setCurrentVehicleProfileType(enNewProfileSelected);
   ETG_TRACE_USR4(("VehicleParameterListHandler::changeActiveProfile(), Current vehicle profile = %d", enNewProfileSelected));

   _navMiddleware.updateVehicleProfileSettings(vehicleProfileSettings);
   _navMiddleware.applySettings();

   // Get the data from navmiddleware corresponding to the new profile selected
   getCurrentVehicleProfile();
}


/********************************************
/* NAME    : VehicleParameterListHandler
/* SYSFL    : null,
****************************************************/
tSharedPtrDataProvider VehicleParameterListHandler::getListDataProvider(const ListDateProviderReqMsg& oMsg)
{
   _listId = oMsg.GetListId();
   tSharedPtrDataProvider dataProvider;

   if (_listId == LIST_ID_NAV_SETTINGS_VEHICLE_PARAMETER)
   {
      unsigned int vehicleProfileTypeCoach = 0;
      unsigned int vehicleProfileTypeTruck = 0;
      unsigned int vehicleTrailerType = 0;
      unsigned int vehicleGoodsType = 0;
      Candera::String currentTrailerText = "";
      Candera::String currentGoodsText = "";
      bool profileTypeCoach[MAX_SIZE_PROFILE_DROP_DOWN_LIST_COACH] = {false};
      bool profileTypeTruck[MAX_SIZE_PROFILE_DROP_DOWN_LIST] = {false};
      bool trailerType[MAX_SIZE_TRAILER_DROP_DOWN_LIST] = {false};
      bool goodsType[MAX_SIZE_HAZARDOUS_DROP_DOWN_LIST] = {false};
      int listIndex = 0;
      ListDataProviderBuilder listBuilder(LIST_ID_NAV_SETTINGS_VEHICLE_PARAMETER);
      vehicleProfileTypeTruck = static_cast<unsigned int>(_currentVehicleProfile[VEHICLE_SETTINGS_PROFILE]);
      vehicleTrailerType = static_cast<unsigned int>(_currentVehicleProfile[VEHICLE_SETTINGS_TRAILER_TYPE]);
      vehicleGoodsType = getGoodsTypeHmiIndex();

      for (std::vector<VehicleSettingsListItem>::iterator listItem = _VehicleSettingsList.begin();
            (listItem != _VehicleSettingsList.end());
            ++listItem, ++listIndex)
      {
         if (listItem->_listButtonType == LIST_BUTTON_TYPE_INC_DEC_VALUE)
         {
            buildSliderListItem(listBuilder, static_cast<const VehicleSettingsListItem>(*listItem), listIndex);
         }
         else if (listItem->_listButtonType == LIST_BUTTON_TYPE_DROPDOWN)
         {
            switch (listItem->_order)
            {
               case VEHICLE_SETTINGS_PROFILE:
                  currentProfileText = vehicleProfileTextUpdations(listItem->_order, _currentVehicleProfile[VEHICLE_SETTINGS_PROFILE], _isCoach);
                  if (!_isCoach)
                  {
                     profileTypeTruck[vehicleProfileTypeTruck] = true;
                     listBuilder.AddItem(listIndex, 0, DATA_CONTEXT_COMBOBOX_LIST_ITEMS_4)
                     .AddData(VehicleSettingsLabel[listIndex])
                     .AddId(listIndex, 1).AddData(ProfileDropDownListLabel[0]).AddData(profileTypeTruck[0])
                     .AddId(listIndex, 2).AddData(ProfileDropDownListLabel[1]).AddData(profileTypeTruck[1])
                     .AddId(listIndex, 3).AddData(ProfileDropDownListLabel[2]).AddData(profileTypeTruck[2])
                     .AddId(listIndex, 4).AddData(ProfileDropDownListLabel[3]).AddData(profileTypeTruck[3])
                     .AddData(currentProfileText);
                  }
                  else
                  {
                     profileTypeCoach[vehicleProfileTypeCoach] = true;
                     listBuilder.AddItem(listIndex, 0, DATA_CONTEXT_COMBOBOX_LIST_ITEMS_2)
                     .AddData(VehicleSettingsLabel[listIndex]);
                  }
                  break;
               case VEHICLE_SETTINGS_TRAILER_TYPE:
               {
                  currentTrailerText = vehicleProfileTextUpdations(listItem->_order, _currentVehicleProfile[VEHICLE_SETTINGS_TRAILER_TYPE], _isCoach);
                  trailerType[vehicleTrailerType] = true;
                  listBuilder.AddItem(listIndex, 0, DATA_CONTEXT_COMBOBOX_LIST_ITEMS_4)
                  .AddData(VehicleSettingsLabel[listIndex])
                  .AddId(listIndex, 1).AddData(TrailerTypeDropDownListLabel[0]).AddData(trailerType[0])
                  .AddId(listIndex, 2).AddData(TrailerTypeDropDownListLabel[1]).AddData(trailerType[1])
                  .AddId(listIndex, 3).AddData(TrailerTypeDropDownListLabel[2]).AddData(trailerType[2])
                  .AddId(listIndex, 4).AddData(TrailerTypeDropDownListLabel[3]).AddData(trailerType[3])
                  .AddData(currentTrailerText);
               }
               break;
               case VEHICLE_SETTINGS_GOODS_TYPE:
               {
                  currentGoodsText = HazardousDropDownListLabel[vehicleGoodsType];
                  goodsType[vehicleGoodsType] = true;

                  listBuilder.AddItem(listIndex, 0, DATA_CONTEXT_COMBOBOX_LIST_ITEMS_5)
                  .AddData(VehicleSettingsLabel[listIndex])
                  .AddId(listIndex, 1).AddData(HazardousDropDownListLabel[0]).AddData(goodsType[0])
                  .AddId(listIndex, 2).AddData(HazardousDropDownListLabel[1]).AddData(goodsType[1])
                  .AddId(listIndex, 3).AddData(HazardousDropDownListLabel[2]).AddData(goodsType[2])
                  .AddId(listIndex, 4).AddData(HazardousDropDownListLabel[3]).AddData(goodsType[3])
                  .AddId(listIndex, 5).AddData(HazardousDropDownListLabel[4]).AddData(goodsType[4])
                  .AddData(currentGoodsText);
               }
               break;
            }
         }
      }
      dataProvider = listBuilder.CreateDataProvider();
      if ((*_vehicleProfileInfo).mGoodsType != currentGoodsText)
      {
         (*_vehicleProfileInfo).mGoodsType = currentGoodsText;
         _vehicleProfileInfo.MarkItemModified(ItemKey::VehicleProfileInfo::GoodsTypeItem);
      }
      if ((*_vehicleProfileInfo).mTrailerType != currentTrailerText)
      {
         (*_vehicleProfileInfo).mTrailerType = currentTrailerText;
         _vehicleProfileInfo.MarkItemModified(ItemKey::VehicleProfileInfo::TrailerTypeItem);
      }
      _vehicleProfileInfo.SendUpdate(true);
   }

   return dataProvider;
}


unsigned int VehicleParameterListHandler::getGoodsTypeHmiIndex()
{
   unsigned int index = 0;
   unsigned int goodsType = 0;
   for (; index < MAX_SIZE_HAZARDOUS_DROP_DOWN_LIST; index++)
   {
      if ((unsigned int)_cargoType[index] == UINT(_currentVehicleProfile[VEHICLE_SETTINGS_GOODS_TYPE]))
      {
         break;
      }
   }
   if (index < MAX_SIZE_HAZARDOUS_DROP_DOWN_LIST)
   {
      goodsType = index;
   }

   return goodsType;
}


/********************************************
/* NAME    : VehicleParameterListHandler
/* SYSFL    : null,
****************************************************/
void VehicleParameterListHandler::buildSliderListItem(ListDataProviderBuilder& listBuilder, const VehicleSettingsListItem& listItem, const int& listItemPosition)
{
   ETG_TRACE_USR4(("VehicleParameterListHandler::buildSliderListItem()"));
   float dimensionValue = _currentVehicleProfile[listItem._order];
   ETG_TRACE_USR4(("VehicleParameterListHandler::buildSliderListItem(), List index : %d, Vehicle dimension : %f", listItem._order, dimensionValue));

   Candera::String displayValueStr = VehicleSettingsHelper::convertHmiDimensionToString(listItem._order, dimensionValue, _distanceUnitSystem);

   if ((VEHICLE_SETTINGS_LENGTH == listItem._order) &&
         ((*_vehicleProfileInfo).mLength != displayValueStr))
   {
      (*_vehicleProfileInfo).mLength = displayValueStr;
      _vehicleProfileInfo.MarkItemModified(ItemKey::VehicleProfileInfo::LengthItem);
   }
   if ((VEHICLE_SETTINGS_HEIGHT == listItem._order) &&
         ((*_vehicleProfileInfo).mHeight != displayValueStr))
   {
      (*_vehicleProfileInfo).mHeight = displayValueStr;
      _vehicleProfileInfo.MarkItemModified(ItemKey::VehicleProfileInfo::HeightItem);
   }
   if ((VEHICLE_SETTINGS_WIDTH == listItem._order) &&
         ((*_vehicleProfileInfo).mWidth != displayValueStr))
   {
      (*_vehicleProfileInfo).mWidth = displayValueStr;
      _vehicleProfileInfo.MarkItemModified(ItemKey::VehicleProfileInfo::WidthItem);
   }
   if ((VEHICLE_SETTINGS_TOTAL_WEIGHT == listItem._order) &&
         ((*_vehicleProfileInfo).mTotalWeight != displayValueStr))
   {
      (*_vehicleProfileInfo).mTotalWeight = displayValueStr;
      _vehicleProfileInfo.MarkItemModified(ItemKey::VehicleProfileInfo::TotalWeightItem);
   }
   if ((VEHICLE_SETTINGS_MAX_SPEED == listItem._order) &&
         ((*_vehicleProfileInfo).mMaxSpeed != displayValueStr))
   {
      (*_vehicleProfileInfo).mMaxSpeed = displayValueStr;
      _vehicleProfileInfo.MarkItemModified(ItemKey::VehicleProfileInfo::MaxSpeedItem);
   }
   if ((VEHICLE_SETTINGS_LOAD_PER_AXLE == listItem._order) &&
         ((*_vehicleProfileInfo).mLoadPerAxle != displayValueStr))
   {
      (*_vehicleProfileInfo).mLoadPerAxle = displayValueStr;
      _vehicleProfileInfo.MarkItemModified(ItemKey::VehicleProfileInfo::LoadPerAxleItem);
   }
   _vehicleProfileInfo.SendUpdate(true);

   SettingSliderData settingSliderData;
   if (false == _isStaticList)
   {
      // Check if currentValue lies within boundary range and enable/disable increment/decrement button accordingly
      settingSliderData.mButtonIsActive = true;
      settingSliderData.mDecreasable = (dimensionValue <= vehicleSettingsMetaData[listItem._order].minValue) ? false : true;
      settingSliderData.mIncreasable = (dimensionValue >= vehicleSettingsMetaData[listItem._order].maxValue) ? false : true;
   }
   else
   {
      settingSliderData.mButtonIsActive = false;
   }

   // Settings value displayed here is a string value
   listBuilder.AddItem(listItemPosition, 0UL, DATA_ITEM_CONTEXT_SPINNER)
   .AddData(listItem._text)        // Name of the vehicle setting
   .AddData(listItem._isVisible)   // Button visibility
   .AddId(listItemPosition, DATA_ITEM_DECREMENT)   // Name(Id) of the Decerement button
   .AddId(listItemPosition, DATA_ITEM_INCREMENT)   // Name(Id) of the Increment button
   .AddData(displayValueStr)       // Value to be displayed
   .AddDataBindingUpdater<SettingSliderDataBindingSource>(settingSliderData);
}


Candera::String VehicleParameterListHandler::vehicleProfileTextUpdations(VehicleSettingsListItemValue listValue, unsigned int currentVehicleProfileValue, bool isCoach)
{
   Candera::String currentProfileText;
   if (listValue == VEHICLE_SETTINGS_PROFILE)
   {
      currentProfileText = ProfileDropDownListLabel[currentVehicleProfileValue];
   }
   else if (listValue == VEHICLE_SETTINGS_TRAILER_TYPE)
   {
      currentProfileText = TrailerTypeDropDownListLabel[currentVehicleProfileValue];
   }
   else
   {
      // do nothing
   }
   return currentProfileText;
}


/********************************************
/* NAME    : VehicleParameterListHandler
/* SYSFL    : null,
****************************************************/
void VehicleParameterListHandler::getCurrentVehicleProfile()
{
   ETG_TRACE_USR4(("VehicleParameterListHandler::getCurrentVehicleProfile()"));

   VehicleProfileSettings vehicleProfileSettings = _navMiddleware.getVehicleProfileSettings();
   ActiveVehicleProfileType activeProfileType = vehicleProfileSettings.getCurrentVehicleProfileType();
   ETG_TRACE_USR4(("VehicleParameterListHandler::getCurrentVehicleProfile(), Active profile type : %d", activeProfileType));

   VehicleProfile vehicleProfile;
   if (_isCoach)
   {
      vehicleProfile = vehicleProfileSettings.getUserVehicleProfile(USE_VEHICLETYPE_USER_1);
      _isStaticList = false;
   }
   else
   {
      if (USE_VEHICLETYPE_DEFAULT > activeProfileType)
      {
         vehicleProfile = vehicleProfileSettings.getUserVehicleProfile(activeProfileType);
         _isStaticList = false;
      }
      else
      {
         vehicleProfile = vehicleProfileSettings.getDefaultVehicleProfile();
         _isStaticList = true;
      }
   }

   // On resetting to factory settings, all profiles will have their vehicle type's reset to CAR.
   // In this case, the KDS value maintained by infostore should be fetched and shown
   if (USE_VEHICLETYPE_PASSENGER_CAR == vehicleProfile.getVehicleType())
   {
      ETG_TRACE_USR4(("VehicleParameterListHandler::getCurrentVehicleProfile(), Resetting with default profile value"));
      vehicleProfile = vehicleProfileSettings.getDefaultVehicleProfile();
      vehicleProfileSettings.setCurrentVehicleProfileType(USE_VEHICLETYPE_USER_1);
      vehicleProfileSettings.setUserVehicleProfile(USE_VEHICLETYPE_USER_1, vehicleProfile);
      _navMiddleware.updateVehicleProfileSettings(vehicleProfileSettings);
      _navMiddleware.applySettings();
   }
   _currentVehicleProfile[VEHICLE_SETTINGS_PROFILE] = float(navmiddleware::settings::USE_VEHICLETYPE_USER_1);
   float height = VehicleSettingsHelper::convertDimensionToHmiUnit(VEHICLE_SETTINGS_HEIGHT, vehicleProfile.getHeightOfVehicle(), _distanceUnitSystem);
   _currentVehicleProfile[VEHICLE_SETTINGS_HEIGHT]        = height;

   float length = VehicleSettingsHelper::convertDimensionToHmiUnit(VEHICLE_SETTINGS_LENGTH, vehicleProfile.getLengthOfVehicle(), _distanceUnitSystem);
   _currentVehicleProfile[VEHICLE_SETTINGS_LENGTH]        = length;

   float width = VehicleSettingsHelper::convertDimensionToHmiUnit(VEHICLE_SETTINGS_WIDTH, vehicleProfile.getWidthOfVehicle(), _distanceUnitSystem);
   _currentVehicleProfile[VEHICLE_SETTINGS_WIDTH]         = width;
   float weight = VehicleSettingsHelper::convertDimensionToHmiUnit(VEHICLE_SETTINGS_TOTAL_WEIGHT, vehicleProfile.getMaxWeightOfVehicle(), _distanceUnitSystem);
   if (weight <= COACH_MIN_WEIGHT_IN_TONNE)
   {
      _currentVehicleProfile[VEHICLE_SETTINGS_TOTAL_WEIGHT] = _infoStore.getCoachWeight();
   }
   else
   {
      _currentVehicleProfile[VEHICLE_SETTINGS_TOTAL_WEIGHT] = weight;
   }

   _currentVehicleProfile[VEHICLE_SETTINGS_TRAILER_TYPE]  = (float)vehicleProfile.getTrailer();
   _currentVehicleProfile[VEHICLE_SETTINGS_MAX_SPEED]   = (float)vehicleProfile.getMaxSpeed();
   float loadPerAxle = VehicleSettingsHelper::convertDimensionToHmiUnit(VEHICLE_SETTINGS_LOAD_PER_AXLE, vehicleProfile.getMaxLoadPerAxleOfVehicle(), _distanceUnitSystem);
   if (loadPerAxle <= COACH_MIN_LOAD_PER_AXLE_TONNE)
   {
      _currentVehicleProfile[VEHICLE_SETTINGS_LOAD_PER_AXLE] = _infoStore.getCoachLoadPerAxle();
   }
   else
   {
      _currentVehicleProfile[VEHICLE_SETTINGS_LOAD_PER_AXLE] = loadPerAxle;
   }

   ::std::vector<navmiddleware::settings::CargoType> cargoTypes = vehicleProfile.getCargoTypes();
   _currentVehicleProfile[VEHICLE_SETTINGS_GOODS_TYPE]    = (float)((0 < cargoTypes.size()) ? cargoTypes[0] : navmiddleware::settings::USE_CARGOTYPE_HAS_NO_LOAD);

   // Store vehicle settings from navmiddleware in _previousVehicleProfile. Any change made henceforth from HMI will be stored in _currentVehicleProfile
   if (false == _isStaticList)   // Not required for default profile because their values cannot be changed
   {
      for (int index = 0; index < MAX_SIZE_VEHICLE_SETTINGS; index++)
      {
         _previousVehicleProfile[index] = _currentVehicleProfile[index];
      }
   }
}


/********************************************
/* NAME    : VehicleParameterListHandler
/* SYSFL    : null,
****************************************************/
void VehicleParameterListHandler::updateCurrentVehicleProfile()
{
   ETG_TRACE_USR4(("VehicleParameterListHandler::updateCurrentVehicleProfile(), Active profile type : %d", _currentVehicleProfile[VEHICLE_SETTINGS_PROFILE]));

   VehicleProfileSettings vehicleProfileSettings = _navMiddleware.getVehicleProfileSettings();
   VehicleProfile vehicleProfile;
   vehicleProfile = vehicleProfileSettings.getUserVehicleProfile(USE_VEHICLETYPE_USER_1);

   navmiddleware::settings::VehicleTypeValue enVehicleTypeValue = (false == _isCoach) ? navmiddleware::settings::USE_VEHICLETYPE_TRUCK : navmiddleware::settings::USE_VEHICLETYPE_LIGHT_TRUCK;
   vehicleProfile.setVehicleType(enVehicleTypeValue);

   unsigned int height = VehicleSettingsHelper::convertDimensionToPresCtrlUnit(VEHICLE_SETTINGS_HEIGHT, _currentVehicleProfile[VEHICLE_SETTINGS_HEIGHT], _distanceUnitSystem);
   vehicleProfile.setHeightOfVehicle(height);

   unsigned int length = VehicleSettingsHelper::convertDimensionToPresCtrlUnit(VEHICLE_SETTINGS_LENGTH, _currentVehicleProfile[VEHICLE_SETTINGS_LENGTH], _distanceUnitSystem);
   vehicleProfile.setLengthOfVehicle(length);

   unsigned int width = VehicleSettingsHelper::convertDimensionToPresCtrlUnit(VEHICLE_SETTINGS_WIDTH, _currentVehicleProfile[VEHICLE_SETTINGS_WIDTH], _distanceUnitSystem);
   vehicleProfile.setWidthOfVehicle(width);

   unsigned int weight = VehicleSettingsHelper::convertDimensionToPresCtrlUnit(VEHICLE_SETTINGS_TOTAL_WEIGHT, _currentVehicleProfile[VEHICLE_SETTINGS_TOTAL_WEIGHT], _distanceUnitSystem);
   vehicleProfile.setMaxWeightOfVehicle(weight);

   navmiddleware::settings::TrailerValue trailerValue = static_cast<navmiddleware::settings::TrailerValue>(_currentVehicleProfile[VEHICLE_SETTINGS_TRAILER_TYPE]);
   vehicleProfile.setTrailer(trailerValue);

   unsigned int loadPerAxle = VehicleSettingsHelper::convertDimensionToPresCtrlUnit(VEHICLE_SETTINGS_LOAD_PER_AXLE, _currentVehicleProfile[VEHICLE_SETTINGS_LOAD_PER_AXLE], _distanceUnitSystem);
   vehicleProfile.setMaxLoadPerAxleOfVehicle(loadPerAxle);

   navmiddleware::settings::CargoType cargoType = static_cast<navmiddleware::settings::CargoType>(_currentVehicleProfile[VEHICLE_SETTINGS_GOODS_TYPE]);
   ::std::vector<navmiddleware::settings::CargoType> cargoTypes;
   cargoTypes.push_back(cargoType);
   vehicleProfile.setCargoTypes(cargoTypes);

   vehicleProfile.setMaxSpeed(UINT(_currentVehicleProfile[VEHICLE_SETTINGS_MAX_SPEED]));
   vehicleProfileSettings.setUserVehicleProfile(USE_VEHICLETYPE_USER_1, vehicleProfile);
   _navMiddleware.updateVehicleProfileSettings(vehicleProfileSettings);
   _navMiddleware.applySettings();

   // After updating the new settings to navmiddleware, the settings value is updated in _previousVehicleProfile
   if (false == _isStaticList)   // Not required for default/car profile because their values cannot be changed
   {
      for (int index = 0; index < MAX_SIZE_VEHICLE_SETTINGS; index++)
      {
         _previousVehicleProfile[index] = _currentVehicleProfile[index];
      }
   }
   float coachWeight = _previousVehicleProfile[VEHICLE_SETTINGS_TOTAL_WEIGHT];
   float coachLoadPerAxle = _previousVehicleProfile[VEHICLE_SETTINGS_LOAD_PER_AXLE];
   _infoStore.setCoachWeight(coachWeight);
   _infoStore.setCoachLoadPerAxle(coachLoadPerAxle);
}


#endif // HALL_TO_MDW_COM
