/**************************************************************************************
* @file         : UserPOISettingsListHandler.cpp
* @author       : ECG-Atchuta Sashank Kappagantu
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (C) 2020 Robert Bosch GmbH
*                 (C) 2020 Robert Bosch Engineering and Business Solutions Limited
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include "UserPOISettingsListHandler.h"
#include "NaviSettingsUtil.h"
#include "NaviDataPoolConfig.h"
#include <string>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/UserPOISettingsListHandler.cpp.trc.h"
#endif

#ifdef  HALL_TO_MDW_COM

const char* EMPTY_POI_STR = "";

IsButtonToggleEnabled getPOIWarningStatus(navmiddleware::NavMiddleware& navMiddleware)
{
   navmiddleware::settings::GuidanceSettings guidanceSettings = navMiddleware.getGuidanceSettings();
   IsButtonToggleEnabled retValue = guidanceSettings.isUsbPoiWarningsEnabled() ? BUTTON_TOGGLE_ENABLED_TRUE : BUTTON_TOGGLE_ENABLED_FALSE;
   ETG_TRACE_USR4(("getPOIWarningStatus(%d)", retValue));

   return retValue;
}


UserPOISettingsListHandler::UserPOISettingsListHandler(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore)
   : HMIModelBase(navMiddleware, infoStore)
   , _myPOIsPath(EMPTY_POI_STR)
   , _hmiImportStatus(IMPORT_STATUS__INVALID)
   , _deviceStatus(false)
{
   //Register the ListID
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_MY_POI, this);
}


UserPOISettingsListHandler::~UserPOISettingsListHandler()
{
}


void UserPOISettingsListHandler::initialize()
{
   _navMiddleware.registerLocationPropertyUpdateCallback(*this);

   NaviDataPoolConfig* dpConfig = NaviDataPoolConfig::getInstance();
   if (NULL != dpConfig)
   {
      bool isAvailable = dpConfig->getMyPOIsAvailability();
      changeMyPOIsButtonState(isAvailable);
   }

   _myPOIsList.push_back(MyPOIsListItem(MyPOIListItemText[0], IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_TOGGLE, USER_POI_ALERT, getPOIWarningStatus));
   _myPOIsList.push_back(MyPOIsListItem(MyPOIListItemText[1], IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_BUTTON, USER_POI_IMPORT_FROM_USB));
   _myPOIsList.push_back(MyPOIsListItem(MyPOIListItemText[2], IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_BUTTON, USER_POI_DELETE));
}


void UserPOISettingsListHandler::deinitialize()
{
   _navMiddleware.unregisterLocationPropertyUpdateCallback(*this);
}


/********************************************
/* NAME    : getListDataProvider(const ListDateProviderReqMsg& oMsg)
/* SYSFL    : null,
****************************************************/
tSharedPtrDataProvider UserPOISettingsListHandler::getListDataProvider(const ListDateProviderReqMsg& oMsg)
{
   unsigned int _listId = oMsg.GetListId();
   tSharedPtrDataProvider dataProvider;
   if (LIST_ID_MY_POI == _listId)
   {
      ETG_TRACE_USR4(("UserPOISettingsListHandler::getListDataProvider(), List Id : (%d)", _listId));
      dataProvider = getListDataProviderMainList();
   }

   return dataProvider;
}


/********************************************
/* NAME    : getListDataProviderMainList
/* SYSFL   : null,
****************************************************/
tSharedPtrDataProvider UserPOISettingsListHandler::getListDataProviderMainList()
{
   ETG_TRACE_USR4(("UserPOISettingsListHandler::getListDataProviderMainList()"));

   const char* const DATA_CONTEXT_TEXT_LIST_ITEM = "Layer_TextListItem";
   const char* const DATA_CONTEXT_TEXT_CHECKBOX_ITEM = "Layer_CheckboxOptions";

   ListDataProviderBuilder listBuilder(LIST_ID_MY_POI);
   DataItemContext listItemName;

   for (std::vector<MyPOIsListItem>::iterator itemNumber = _myPOIsList.begin(); itemNumber != _myPOIsList.end(); ++itemNumber)
   {
      if (itemNumber->_listButtonType == LIST_BUTTON_TYPE_TOGGLE)
      {
         listItemName = DATA_CONTEXT_TEXT_CHECKBOX_ITEM;
         listBuilder.AddItem(
            itemNumber->_order,
            0UL,  //    - unused
            listItemName)  //0 - Name(Id) of the button
         .AddData(itemNumber->_text)
         .AddData((itemNumber->_isVisible == IS_VISIBLE_FALSE) ? true : itemNumber->getButtonToggleState(_navMiddleware));
      }
      else if (itemNumber->_listButtonType == LIST_BUTTON_TYPE_BUTTON)
      {
         listItemName = DATA_CONTEXT_TEXT_LIST_ITEM;
         listBuilder.AddItem(
            itemNumber->_order,
            0UL,  //    - unused
            listItemName)  //0 - Name(Id) of the button
         .AddData(itemNumber->_text)
         .AddData(itemNumber->_isVisible);
      }
      else
      {
      }
   }
   return listBuilder.CreateDataProvider();
}


/********************************************
/* NAME    : onPropertyUpdateImportStatusChanged
/* SYSFL   : null,
****************************************************/
void UserPOISettingsListHandler::onPropertyUpdateImportStatusChanged()
{
   _hmiImportStatus = IMPORT_STATUS__INVALID;
   POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Hide, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_ThreeLine_Text"))));

   const navmiddleware::ImportUserPoisStatus& importUserPoisStatus = _navMiddleware.getImportUserPoisStatus();
   const navmiddleware::ImportUserPoisStatus::ImportStatus& importStatus = importUserPoisStatus.getStatus();
   uint32_t numOfFilesImported = importUserPoisStatus.getNumOfFilesImported();
   uint32_t numOfTotalFiles = importUserPoisStatus.getNumOfTotalFiles();
   ETG_TRACE_USR4(("UserPOISettingsListHandler::onPropertyUpdateImportStatusChanged(), ImportStatus :%d, numOfFilesImported : %d, numOfTotalFiles : %d",
                   importStatus, numOfFilesImported, numOfTotalFiles));

   if (navmiddleware::ImportUserPoisStatus::IMPORT_STATUS__SUCCESS == importStatus)
   {
      if ((0 != numOfFilesImported) && (0 != numOfTotalFiles))
      {
         setMyPOIsAvailability(true);

         std::string importSucessLine1 = LANGUAGE_STRING(TextId_0x197E, "%1 out of %2 file(s) downloaded successfully.").GetCString();
         std::string importSucessLine2 = LANGUAGE_STRING(TextId_0x197F, "").GetCString();

         // Replace %1 and %2 in dynamic pop-up text with number of files imported and the total number of files
         std::stringstream strNumOfFilesImported, strNumOfTotalFiles;
         strNumOfFilesImported << numOfFilesImported;
         strNumOfTotalFiles << numOfTotalFiles;
         importSucessLine1 = replaceInTranslationTextID(importSucessLine1, DynamicPopupText[0], strNumOfFilesImported.str());
         importSucessLine2 = replaceInTranslationTextID(importSucessLine2, DynamicPopupText[0], strNumOfFilesImported.str());
         importSucessLine1 = replaceInTranslationTextID(importSucessLine1, DynamicPopupText[1], strNumOfTotalFiles.str());
         importSucessLine2 = replaceInTranslationTextID(importSucessLine2, DynamicPopupText[1], strNumOfTotalFiles.str());

         (*_popupText).mTextLine1 = importSucessLine1.c_str();
         (*_popupText).mTextLine2 = importSucessLine2.c_str();
         _popupText.MarkAllItemsModified();
         _popupText.SendUpdate(true);
         POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_LatLong_Invalid"))));
      }
      else
      {
         vUPOIPopUpChangeRequest(POPUP_USER_POI_IMPORT_ERROR);
      }
   }
   else if (navmiddleware::ImportUserPoisStatus::IMPORT_STATUS__ERROR == importStatus)
   {
      vUPOIPopUpChangeRequest(POPUP_USER_POI_IMPORT_ERROR);
   }
   else
   {
   }
}


/********************************************
/* NAME    : onPropertyUpdateUPoisDeleted
/* SYSFL   : null,
****************************************************/
void UserPOISettingsListHandler::onPropertyUpdateUPoisDeleted()
{
   ETG_TRACE_USR4(("UserPOISettingsListHandler::onPropertyUpdateUPoisDeleted(), Hmi import status : %d", _hmiImportStatus));

   setMyPOIsAvailability(false);
   if (true == EXT_bIsUPOIContext)
   {
      POST_MSG((COURIER_MESSAGE_NEW(UPOIInvalidateUpdateMsg)()));
   }
   if (IMPORT_STATUS__INITIATED == _hmiImportStatus)
   {
      _navMiddleware.importUserPois(_myPOIsPath);
      _hmiImportStatus = IMPORT_STATUS__IN_PROGRESS;
   }
}


/********************************************
/* NAME    : onCourierMessage(const ButtonListItemUpdMsg& oMsg)
/* SYSFL   : null,
****************************************************/
bool UserPOISettingsListHandler::onCourierMessage(const ButtonListItemUpdMsg& oMsg)
{
   bool msgProcessed = false;
   unsigned int listID = oMsg.GetListId();
   unsigned int rowIdx = (unsigned int)oMsg.GetHdl();

   if ((oMsg.GetReaction() == enRelease) && (LIST_ID_MY_POI == listID))
   {
      ETG_TRACE_USR4(("UserPOISettingsListHandler::onCourierMessage(ButtonListItemUpdMsg),  List Id: %d,  Row Id: %d", listID , rowIdx));
      switch (rowIdx)
      {
         case USER_POI_ALERT:
         {
            navmiddleware::settings::GuidanceSettings guidanceSettings = _navMiddleware.getGuidanceSettings();
            guidanceSettings.isUsbPoiWarningsEnabled() ? guidanceSettings.setUsbPoiWarningsEnabled(false) : guidanceSettings.setUsbPoiWarningsEnabled(true);
            _navMiddleware.updateGuidanceSettings(guidanceSettings);
            ListRegistry::s_getInstance().updateList(LIST_ID_MY_POI);
            break;
         }
         case USER_POI_IMPORT_FROM_USB:
         {
            if (!(_myPOIsPath.empty()) && (true == _deviceStatus))
            {
               _hmiImportStatus = IMPORT_STATUS__AWAITING_USER_RESPONSE;
               vUPOIPopUpChangeRequest(POPUP_USER_POI_IMPORT);
            }
            else
            {
               vUPOIPopUpChangeRequest(POPUP_USER_POI_DEVICE_NOT_FOUND);
            }
            break;
         }
         case USER_POI_DELETE:
         {
            vUPOIPopUpChangeRequest(POPUP_USER_POI_DELETE);
            break;
         }
         default:
         {
            ETG_TRACE_ERR(("UserPOISettingsListHandler::onCourierMessage(ButtonListItemUpdMsg), Incorrect rowId: %d ", rowIdx));
            break;
         }
      }
      msgProcessed = true;
   }

   return msgProcessed;
}


/********************************************
/* NAME    : onCourierMessage(const DeleteAllUPOIsReqMsg& oMsg)
/* SYSFL   : null,
****************************************************/
bool UserPOISettingsListHandler::onCourierMessage(const DeleteAllUPOIsReqMsg& oMsg)
{
   ETG_TRACE_USR4(("UserPOISettingsListHandler::onCourierMessage(DeleteAllUPOIsReqMsg)"));
   _navMiddleware.deleteUserPois();
   return true;
}


/********************************************
/* NAME    : onCourierMessage(const ImportUPOIsReqMsg& oMsg)
/* SYSFL   : null,
****************************************************/
bool UserPOISettingsListHandler::onCourierMessage(const ImportUPOIsReqMsg& oMsg)
{
   bool importUPOIsRequest = oMsg.GetImportUPOIsRequest();
   ETG_TRACE_USR4(("UserPOISettingsListHandler::onCourierMessage(ImportUPOIsReqMsg), ImportUPOIsRequest : %d", importUPOIsRequest));
   if (IMPORT_STATUS__AWAITING_USER_RESPONSE == _hmiImportStatus)
   {
      if (true == importUPOIsRequest)
      {
         _hmiImportStatus = IMPORT_STATUS__INITIATED;
         _navMiddleware.deleteUserPois();
         vUPOIPopUpChangeRequest(POPUP_USER_POI_IMPORTING);   //to display the importing status.
      }
      else
      {
         _hmiImportStatus = IMPORT_STATUS__INVALID;
      }
   }
   return true;
}


/********************************************
/* NAME    : onCourierMessage(const UPOIDevicePathReqMsg& oMsg)
/* SYSFL   : null,
****************************************************/
bool UserPOISettingsListHandler::onCourierMessage(const UPOIDeviceStatusUpdMsg& oMsg)
{
   _deviceStatus = oMsg.GetDeviceStatus();
   _myPOIsPath = oMsg.GetMyPOIsPath().GetCString();
   ETG_TRACE_USR4(("UserPOISettingsListHandler::onCourierMessage(UPOIDeviceStatusUpdMsg), USB status : %d, MyPOIs Path : %s",
                   _deviceStatus, _myPOIsPath.c_str()));

   if ((IMPORT_STATUS__AWAITING_USER_RESPONSE == _hmiImportStatus) && (false == _deviceStatus))
   {
      _hmiImportStatus = IMPORT_STATUS__INVALID;
      POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Hide, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_FiveLine_TwoButton"))));
      vUPOIPopUpChangeRequest(POPUP_USER_POI_DEVICE_REMOVED);
   }
   return true;
}


/********************************************
/* NAME    : vUPOIPopUpChangeRequest
/* SYSFL   : null,
****************************************************/
void UserPOISettingsListHandler::vUPOIPopUpChangeRequest(const Courier::UInt8& popupType)
{
   ETG_TRACE_USR4(("UserPOISettingsListHandler::vUPOIPopUpChangeRequest(), popupType= %d", popupType));

   const Candera::String DATA_CONTEXT_USER_POI_EMPTY_LINE = LANGUAGE_STRING(TEXTID_UNAVAILABLE, EMPTY_POI_STR);
   const Candera::String DATA_CONTEXT_USER_POI_IMPORT_ERROR = LANGUAGE_STRING(TextId_0x0EAB, "Download failed.");
   const Candera::String DATA_CONTEXT_USER_POI_DEVICE_REMOVED = LANGUAGE_STRING(TextId_0x1374, "The USB flash drive has been removed.");

   const Candera::String DATA_CONTEXT_USER_POI_IMPORT_LINE1 = LANGUAGE_STRING(TextId_0x0A0B, "");
   const Candera::String DATA_CONTEXT_USER_POI_IMPORT_LINE2 = LANGUAGE_STRING(TextId_0x0A0C, "MyPOIs are already present. Download");
   const Candera::String DATA_CONTEXT_USER_POI_IMPORT_LINE3 = LANGUAGE_STRING(TextId_0x0A0D, "will overwrite the current MyPOI entries.");
   const Candera::String DATA_CONTEXT_USER_POI_IMPORT_LINE4 = LANGUAGE_STRING(TextId_0x0E77, "Do you want to start the download?");
   const Candera::String DATA_CONTEXT_USER_POI_IMPORT_LINE5 = LANGUAGE_STRING(TextId_0x0EB0, "");

   const Candera::String DATA_CONTEXT_USER_POI_IMPORTING_LINE1 = LANGUAGE_STRING(TextId_0x0A10, "MyPOIs will be copied into the system,");
   const Candera::String DATA_CONTEXT_USER_POI_IMPORTING_LINE2 = LANGUAGE_STRING(TextId_0x0E79, "please wait...");
   const Candera::String DATA_CONTEXT_USER_POI_IMPORTING_LINE3 = LANGUAGE_STRING(TextId_0x0E7A, "");

   const Candera::String DATA_CONTEXT_USER_POI_DELETE_LINE1 = LANGUAGE_STRING(TextId_0x0A0E, "Are you sure you want to delete all MyPOIs?");
   const Candera::String DATA_CONTEXT_USER_POI_DELETE_LINE2 = LANGUAGE_STRING(TextId_0x0E78, "");

   const Candera::String DATA_CONTEXT_USER_POI_DEVICE_NOT_FOUND_LINE1 = LANGUAGE_STRING(TextId_0x1980, "Please insert USB flash drive containing valid");
   const Candera::String DATA_CONTEXT_USER_POI_DEVICE_NOT_FOUND_LINE2 = LANGUAGE_STRING(TextId_0x1981, "MyPOIs data.");

   POST_MSG((COURIER_MESSAGE_NEW(PopupTypeUpdateMsg)(popupType)));   //Send popup type information to controller.

   switch (popupType)
   {
      case POPUP_USER_POI_IMPORT:
      {
         (*_fiveLinePopupText).mPopupTextLine1 = DATA_CONTEXT_USER_POI_IMPORT_LINE1;
         (*_fiveLinePopupText).mPopupTextLine2 = DATA_CONTEXT_USER_POI_IMPORT_LINE2;
         (*_fiveLinePopupText).mPopupTextLine3 = DATA_CONTEXT_USER_POI_IMPORT_LINE3;
         (*_fiveLinePopupText).mPopupTextLine4 = DATA_CONTEXT_USER_POI_IMPORT_LINE4;
         (*_fiveLinePopupText).mPopupTextLine5 = DATA_CONTEXT_USER_POI_IMPORT_LINE5;
         _fiveLinePopupText.MarkAllItemsModified();
         _fiveLinePopupText.SendUpdate(true);
         POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_FiveLine_TwoButton"))));
         break;
      }
      case POPUP_USER_POI_IMPORTING:
      {
         (*_multiLinePopupText).mThreeLinePopupText1 = DATA_CONTEXT_USER_POI_IMPORTING_LINE1;
         (*_multiLinePopupText).mThreeLinePopupText2 = DATA_CONTEXT_USER_POI_IMPORTING_LINE2;
         (*_multiLinePopupText).mThreeLinePopupText3 = DATA_CONTEXT_USER_POI_IMPORTING_LINE3;
         _multiLinePopupText.MarkAllItemsModified();
         _multiLinePopupText.SendUpdate(true);
         POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_ThreeLine_Text"))));
         break;
      }
      case POPUP_USER_POI_DEVICE_NOT_FOUND:
      {
         (*_popupText).mTextLine1 = DATA_CONTEXT_USER_POI_DEVICE_NOT_FOUND_LINE1;
         (*_popupText).mTextLine2 = DATA_CONTEXT_USER_POI_DEVICE_NOT_FOUND_LINE2;
         _popupText.MarkAllItemsModified();
         _popupText.SendUpdate(true);
         POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_LatLong_Invalid"))));
         break;
      }
      case POPUP_USER_POI_IMPORT_ERROR:
      {
         (*_twoLinePopupText).mPopuptextLine1 = DATA_CONTEXT_USER_POI_IMPORT_ERROR;
         (*_twoLinePopupText).mPopuptextLine2 = DATA_CONTEXT_USER_POI_EMPTY_LINE;
         _twoLinePopupText.MarkAllItemsModified();
         _twoLinePopupText.SendUpdate(true);
         POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_SingleLine_SingleButton"))));
         break;
      }
      case POPUP_USER_POI_DEVICE_REMOVED:
      {
         (*_twoLinePopupText).mPopuptextLine1 = DATA_CONTEXT_USER_POI_DEVICE_REMOVED;
         (*_twoLinePopupText).mPopuptextLine2 = DATA_CONTEXT_USER_POI_EMPTY_LINE;
         _twoLinePopupText.MarkAllItemsModified();
         _twoLinePopupText.SendUpdate(true);
         POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_SingleLine_SingleButton"))));
         break;
      }
      case POPUP_USER_POI_DELETE:
      {
         (*_twoLinePopupText).mPopuptextLine1 = DATA_CONTEXT_USER_POI_DELETE_LINE1;
         (*_twoLinePopupText).mPopuptextLine2 = DATA_CONTEXT_USER_POI_DELETE_LINE2;
         _twoLinePopupText.MarkAllItemsModified();
         _twoLinePopupText.SendUpdate(true);
         POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_MultiLine_TwoButton"))));
         break;
      }
      default:
      {
         ETG_TRACE_USR4(("UserPOISettingsListHandler::vUPOIPopUpChangeRequest(), Invalid popupType"));
         break;
      }
   }
}


/********************************************
/* NAME    : setMyPOIsAvailability
/* SYSFL   : null,
****************************************************/
void UserPOISettingsListHandler::setMyPOIsAvailability(bool isAvailable)
{
   ETG_TRACE_USR4(("UserPOISettingsListHandler::setMyPOIsAvailability(), isAvailable : %d", isAvailable));

   NaviDataPoolConfig* dpConfig = NaviDataPoolConfig::getInstance();
   if (NULL != dpConfig)
   {
      dpConfig->setMyPOIsAvailability(isAvailable);
   }
   changeMyPOIsButtonState(isAvailable);

   return;
}


/********************************************
/* NAME    : changeMyPOIsButtonState
/* SYSFL   : null,
****************************************************/
void UserPOISettingsListHandler::changeMyPOIsButtonState(bool isEnabled)
{
   ETG_TRACE_USR4(("UserPOISettingsListHandler::changeMyPOIsButtonState(), isEnabled : %d", isEnabled));

   (*_myPOIsAvailability).mIsMyPOIsAvailable = isEnabled;
   _myPOIsAvailability.MarkAllItemsModified();
   _myPOIsAvailability.SendUpdate(true);
}


#endif // HALL_TO_MDW_COM
