/**************************************************************************************
* @file         : TrafficIconsSettingsListHandler.cpp
* @author       : ECG-Ramesh Kesavan
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) -2018 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include "TrafficIconsSettingsListHandler.h"
#include <iostream>
#include <vector>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/TrafficIconsSettingsListHandler.cpp.trc.h"
#endif

static const char* const DATA_CONTEXT_TRAFFIC_ICONS_LIST_ITEM = "Layer_AllTrafficInformation";

#ifdef  HALL_TO_MDW_COM

static const int MAX_LISTSIZE = 9;
const unsigned int ENABLED = 0;
const unsigned int DISABLED = 1;

const Candera::String ALL_TRAFFIC_INFO_TEXT = LANGUAGE_STRING(TextId_0x141D, "All traffic information");
const Candera::String ALL_OTHER_EVENT_TEXT = LANGUAGE_STRING(TextId_0x177D, "All other events");

TrafficIconsSettingsListHandler::TrafficIconsSettingsListHandler(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore)
   : NavMiddlewarePropertyUpdateConsumer(navMiddleware)
   , _startIndex(0)
   , _windowElementSize(MAX_LISTSIZE)
{
}


TrafficIconsSettingsListHandler::~TrafficIconsSettingsListHandler()
{
}


void TrafficIconsSettingsListHandler::initialize()
{
   _navMiddleware.registerTrafficMessagePropertyUpdateCallback(*this);
}


void TrafficIconsSettingsListHandler::deinitialize()
{
   _navMiddleware.unregisterTrafficMessagePropertyUpdateCallback(*this);
}


bool TrafficIconsSettingsListHandler::onCourierMessage(const ListDateProviderReqMsg& oMsg)
{
   bool retValue = false;
   if (oMsg.GetListId() == LIST_ID_NAV_TRAFFIC_ICONS)
   {
      unsigned int listId = oMsg.GetListId();
      unsigned int _startIndex = oMsg.GetStartIndex();
      unsigned int _windowElementSize = oMsg.GetWindowElementSize();

      ETG_TRACE_USR4(("TrafficIconsSettingsListHandler::onCourierMessage(ListDateProviderReqMsg(listID %d, startIndex %d, elementSize %d)))", listId, _startIndex, _windowElementSize));
      _navMiddleware.requestTrafficIconSelectionList();

      retValue = true;
   }

   return retValue;
}


bool TrafficIconsSettingsListHandler::onPropertyTrafficIconInfoAvailable()
{
   ETG_TRACE_USR4(("TrafficIconsSettingsListHandler::onPropertyTrafficIconInfoAvailable"));
   updateTrafficIconSettingsList();

   return true;
}


tSharedPtrDataProvider TrafficIconsSettingsListHandler::getTrafficIconSelectionList()
{
   ListDataProviderBuilder listBuilder(LIST_ID_NAV_TRAFFIC_ICONS);
   const navmiddleware::TrafficIconSelectionListInfo& trafficSelectionListInfo = _navMiddleware.getTrafficIconSelectionListInfo();
   ::navmiddleware::TrafficIconSelectionListInfo::T_TrafficIconSelectionCriteriaDetailsMap _trafficIconInfo = trafficSelectionListInfo.getTrafficIconSelectionItemList();

   unsigned int index = _startIndex;
   IsButtonToggleEnabled buttonStatus = BUTTON_TOGGLE_ENABLED_TRUE;

   TrafficIconsData item;
   item.mTrafficIconName = ALL_TRAFFIC_INFO_TEXT;
   buttonStatus = getTrafficIconListElementState(navmiddleware::settings::TRAFFIC_ICON_SELECTION_CRITERIA__ALL_TRAFFIC_INFORMATION);

   if (BUTTON_TOGGLE_ENABLED_TRUE == buttonStatus)
   {
      item.mToggleIconSwitchIndex = 0;
   }
   else
   {
      item.mToggleIconSwitchIndex = 1;
   }

   listBuilder.AddItem(index, 0UL, DATA_CONTEXT_TRAFFIC_ICONS_LIST_ITEM)
   .AddDataBindingUpdater<TrafficIconsDataBindingSource>(item)
   .AddData(item.mToggleIconSwitchIndex);
   ++index;

   ::navmiddleware::TrafficIconSelectionListInfo::T_TrafficIconSelectionCriteriaDetailsMap::iterator iterator = _trafficIconInfo.begin();
   for (; (index < _windowElementSize) && (iterator != _trafficIconInfo.end()); ++iterator, ++index)
   {
      if (iterator->first == navmiddleware::settings::TRAFFIC_ICON_SELECTION_CRITERIA__ALL_OTHER_EVENTS)
      {
         item.mTrafficIconName = ALL_OTHER_EVENT_TEXT;
      }
      else
      {
         // the Icon Name string obtained from Middleware are all lower case.
         // Hence converting the First letter to Upper case To match with PSD requirement
         iterator->second.first[0] = toupper(iterator->second.first[0]);
         item.mTrafficIconName = iterator->second.first.c_str();
      }

      item.mTrafficIconImage = loadImage((const char*)iterator->second.second.getBlob(), iterator->second.second.getBlobSize(), true);

      buttonStatus = getTrafficIconListElementState(iterator->first);
      if (BUTTON_TOGGLE_ENABLED_TRUE == buttonStatus)
      {
         item.mToggleIconSwitchIndex = 0;
      }
      else
      {
         item.mToggleIconSwitchIndex = 1;
      }

      listBuilder.AddItem(
         index,
         0UL,
         DATA_CONTEXT_TRAFFIC_ICONS_LIST_ITEM)
      .AddDataBindingUpdater<TrafficIconsDataBindingSource>(item).AddData(item.mToggleIconSwitchIndex);
   }

   return listBuilder.CreateDataProvider();
}


IsButtonToggleEnabled TrafficIconsSettingsListHandler::getTrafficIconListElementState(navmiddleware::settings::TrafficIconSelectionCriteria selectedTrafficIcon) const
{
   IsButtonToggleEnabled retValue = BUTTON_TOGGLE_ENABLED_TRUE;
   const navmiddleware::settings::TrafficSettings& trafficSettings = _navMiddleware.getTrafficSettings();
   const ::std::vector<navmiddleware::settings::TrafficIconSelectionCriteria>& trafficSelectionCriteria = trafficSettings.getTrafficIconFilterCriteriaList();

   if (selectedTrafficIcon == navmiddleware::settings::TRAFFIC_ICON_SELECTION_CRITERIA__ALL_TRAFFIC_INFORMATION)
   {
      retValue = trafficSelectionCriteria.empty() ? BUTTON_TOGGLE_ENABLED_TRUE : BUTTON_TOGGLE_ENABLED_FALSE;
   }
   else
   {
      for (::std::vector<navmiddleware::settings::TrafficIconSelectionCriteria>::const_iterator iterator = trafficSelectionCriteria.begin(); iterator != trafficSelectionCriteria.end(); ++iterator)
      {
         if (selectedTrafficIcon == *iterator)
         {
            retValue = BUTTON_TOGGLE_ENABLED_FALSE;
            break;
         }
      }
   }

   return retValue;
}


bool TrafficIconsSettingsListHandler::onCourierMessage(const ButtonListItemUpdMsg& oMsg)
{
   bool msgProcessed = false;

   ListProviderEventInfo info;

   unsigned int listID = oMsg.GetListId();

   if (LIST_ID_NAV_TRAFFIC_ICONS == listID)
   {
      if (oMsg.GetReaction() == enRelease)
      {
         unsigned int index = oMsg.GetHdl();
         toggleTafficIconSelection(static_cast<navmiddleware::settings::TrafficIconSelectionCriteria>(index));
         updateTrafficIconSettingsList();
         msgProcessed = true;
      }
   }

   return msgProcessed;
}


void TrafficIconsSettingsListHandler::updateTrafficIconSettingsList()
{
   tSharedPtrDataProvider dataProvider = getTrafficIconSelectionList();
   if (dataProvider != NULL)
   {
      ETG_TRACE_USR4(("TrafficIconsSettingsListHandler::updateTrafficIconSettingsList() dataProvider->listSize = %d)", dataProvider->listSize()));
      POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
   }
}


void TrafficIconsSettingsListHandler::toggleTafficIconSelection(navmiddleware::settings::TrafficIconSelectionCriteria selectedcriteria)
{
   navmiddleware::settings::TrafficSettings trafficSettings = _navMiddleware.getTrafficSettings();
   ::std::vector<navmiddleware::settings::TrafficIconSelectionCriteria> trafficSelectionCriteria = trafficSettings.getTrafficIconFilterCriteriaList();
   ::std::vector<navmiddleware::settings::TrafficIconSelectionCriteria >::iterator iterator = find(trafficSelectionCriteria.begin(), trafficSelectionCriteria.end(), selectedcriteria);

   if (iterator != trafficSelectionCriteria.end())
   {
      trafficSelectionCriteria.erase(std::remove(trafficSelectionCriteria.begin(), trafficSelectionCriteria.end(), selectedcriteria));
   }
   else if (selectedcriteria == navmiddleware::settings::TRAFFIC_ICON_SELECTION_CRITERIA__ALL_TRAFFIC_INFORMATION)
   {
      // check the size of the list to determine ON or OFF
      if (!trafficSelectionCriteria.empty())
      {
         //clearing the vector so that we need everything to ON
         trafficSelectionCriteria.clear();
      }
      else //send everything as OFF
      {
         //Push all the items to vector except TRAFFIC_ICON_SELECTION_CRITERIA__ALL_TRAFFIC_INFORMATION
         //We are now making all the items to OFF from ON
         trafficSelectionCriteria.push_back(navmiddleware::settings::TRAFFIC_ICON_SELECTION_CRITERIA__TRAFFIC_JAM);
         trafficSelectionCriteria.push_back(navmiddleware::settings::TRAFFIC_ICON_SELECTION_CRITERIA__SLOW_TRAFFIC);
         trafficSelectionCriteria.push_back(navmiddleware::settings::TRAFFIC_ICON_SELECTION_CRITERIA__TRAFFIC_FLOWING_FREELY);
         trafficSelectionCriteria.push_back(navmiddleware::settings::TRAFFIC_ICON_SELECTION_CRITERIA__ROAD_CLOSED);
         trafficSelectionCriteria.push_back(navmiddleware::settings::TRAFFIC_ICON_SELECTION_CRITERIA__ACCIDENT);
         trafficSelectionCriteria.push_back(navmiddleware::settings::TRAFFIC_ICON_SELECTION_CRITERIA__ROADWORKS);
         trafficSelectionCriteria.push_back(navmiddleware::settings::TRAFFIC_ICON_SELECTION_CRITERIA__NARROW_LANES);
         trafficSelectionCriteria.push_back(navmiddleware::settings::TRAFFIC_ICON_SELECTION_CRITERIA__ALL_OTHER_EVENTS);
      }
   }
   else
   {
      trafficSelectionCriteria.push_back(selectedcriteria);
   }

   trafficSettings.setTrafficIconFilterCriteriaList(trafficSelectionCriteria);
   _navMiddleware.updateTrafficSettings(trafficSettings);
}


bool TrafficIconsSettingsListHandler::onCourierMessage(const ApplyTrafficIconsSettingsReqMsg& oMsg)
{
   COURIER_UNUSED(oMsg);
   ETG_TRACE_USR4(("TrafficIconsSettingsListHandler::onCourierMessage(ApplyTrafficIconsSettingsReqMsg)"));
   _navMiddleware.applySettings();

   return true;
}


#endif // HALL_TO_MDW_COM
