/**************************************************************************************
* @file         : RoutePreferenceListHandler.cpp
* @author       : ECG-Ramesh Kesavan
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) -2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include "RoutePreferenceListHandler.h"
#include  <vector>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/RoutePreferenceListHandler.cpp.trc.h"
#endif

static const char* const DATA_CONTEXT_TEXT_LIST_ITEM = "Layer_TextListItem";
static const char* const DATA_CONTEXT_COMBOBOX_LIST_ITEMS_3 = "Combobox_ListItem3";
static const char* const DATA_CONTEXT_COMBOBOX_LIST_ITEMS_2 = "Combobox_ListItem2";
static const char* const DATA_CONTEXT_ROUTE_SETTINGS_OTHER_OPTIONS = "Layer_OtherOptions";
static const char* const DATA_CONTEXT_ROUTE_SETTINGS_TOGGLE_OPTION = "Layer_CheckboxOptions";

#ifdef  HALL_TO_MDW_COM

static const int MAX_LISTSIZE = 5;
static const int NUM_ROUTE_CRITERION = 3;
static const int MAX_ITEMS_2 = 2;


Candera::String RoutePreferenceListHandler::getRouteCondition(navmiddleware::NavMiddleware& navMiddleware)
{
   Candera::String retValue = "";
   navmiddleware::settings::RouteCriterion routeCriterion = navMiddleware.getRouteCriteriaSettings().getRouteCriterion();
   ETG_TRACE_USR4(("RoutePreferenceListHandler::getGetRouteCondition(), Navmid route criterion : %d", routeCriterion));

   switch (routeCriterion)
   {
      case navmiddleware::settings::ROUTE_CRITERION_FAST:
         retValue = RouteTypeStrings[1];
         _currentRouteOption = static_cast<enRouteTypeOptions>(1);
         break;
      case navmiddleware::settings::ROUTE_CRITERION_ECONOMIC:
         retValue = RouteTypeStrings[2];
         _currentRouteOption = static_cast<enRouteTypeOptions>(2);
         break;
      case navmiddleware::settings::ROUTE_CRITERION_SHORT:
         retValue = RouteTypeStrings[0];
         _currentRouteOption = static_cast<enRouteTypeOptions>(0);
         break;
      default:
         retValue = RouteTypeStrings[0];
         _currentRouteOption = static_cast<enRouteTypeOptions>(0);
         break;
   }
   ETG_TRACE_USR4(("RoutePreferenceListHandler::getGetRouteCondition(), Hmi route criterion : %s", retValue.GetCString()));
   return retValue;
}


Candera::String RoutePreferenceListHandler::getRouteCalculationType(navmiddleware::NavMiddleware& navMiddleware)
{
   Candera::String retValue = "";
   switch (navMiddleware.getRouteOptionSettings().getMoreRoutesType())
   {
      case navmiddleware::settings::MORE_ROUTES_TYPE_ALTERNATIVE:
         retValue = CustomRouteOptionStrings[0];
         _currentCustomRouteOption = static_cast<enCustomizeRouteOptions>(0);
         break;
      case navmiddleware::settings::MORE_ROUTES_TYPE_MULTIPLE:
         retValue = CustomRouteOptionStrings[1];
         _currentCustomRouteOption = static_cast<enCustomizeRouteOptions>(1);
         break;
      default:
         retValue = CustomRouteOptionStrings[1];
         _currentCustomRouteOption = static_cast<enCustomizeRouteOptions>(1);
         break;
   }
   ETG_TRACE_USR4(("RoutePreferenceListHandler::getRouteCalculationType(%s)", retValue.GetCString()));
   return retValue;
}


Candera::String RoutePreferenceListHandler::getEstimatedTimeOption()
{
   Candera::String retValue = "";
   switch (_infoStore.getEstimatedTimeType())
   {
      case InfoStore::DESTINATION:
         retValue = EstimatedTimeStrings[0];
         break;
      case InfoStore::WAYPOINT:
         retValue = EstimatedTimeStrings[1];
         break;
      default:
         retValue = EstimatedTimeStrings[0];
         break;
   }
   _currentEstimatedTimeOption = static_cast<enEstimatedTimeOptions>(_infoStore.getEstimatedTimeType());
   ETG_TRACE_USR4(("RoutePreferenceListHandler::getEstimatedTimeOption(%s)", retValue.GetCString()));
   return retValue;
}


Candera::String RoutePreferenceListHandler::getTimeOnMapOption()
{
   Candera::String retValue = "";
   switch (_infoStore.getEstimatedTimeMode())
   {
      case InfoStore::ARRIVAL_TIME:
         retValue = TimeOnMapStrings[0];
         _currentTimeOnMapOption = static_cast<enTimeOnMapOptions>(0);
         break;
      case InfoStore::TRAVEL_TIME:
         retValue = TimeOnMapStrings[1];
         _currentTimeOnMapOption = static_cast<enTimeOnMapOptions>(1);
         break;
      default:
         retValue = TimeOnMapStrings[0];
         _currentTimeOnMapOption = static_cast<enTimeOnMapOptions>(0);
         break;
   }
   ETG_TRACE_USR4(("RoutePreferenceListHandler::getEstimatedTimeOption(%s)", retValue.GetCString()));
   return retValue;
}


RoutePreferenceListHandler::RoutePreferenceListHandler(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore)
   : InfoStorePropertyUpdateConsumer(infoStore)
   , NavMiddlewarePropertyUpdateConsumer(navMiddleware)
   , _isDropdownEnabled(false)
   , _listId(LIST_ID_NAV_ROUTE_PREFERENCE)
   , _currentRouteOption(ROUTE_TYPE_SHORTEST)
   , _currentCustomRouteOption(SINGLE_ROUTE)
   , _currentEstimatedTimeOption(ETA_DESTINATION)
   , _currentTimeOnMapOption(ONMAP_ETA)
   , _currentRouteOptionStr(RouteTypeStrings[0])
   , _currentCustomRouteOptionStr(CustomRouteOptionStrings[0])
   , _currentEstimatedTimeStr(EstimatedTimeStrings[0])
   , _currentTimeOnMapStr(TimeOnMapStrings[0])
{
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_NAV_ROUTE_PREFERENCE, this);
   resetSelectedOption();
}


RoutePreferenceListHandler::~RoutePreferenceListHandler()
{
}


void RoutePreferenceListHandler::initialize()
{
}


void RoutePreferenceListHandler::deinitialize()
{
}


void RoutePreferenceListHandler::resetSelectedOption()
{
   memset(_routeType, 0, MAX_SIZE_ROUTE_TYPE * sizeof(_routeType[0]));
   memset(_customisedRoute, 0, MAX_SIZE_CUSTOM_ROUTE_OPTION * sizeof(_customisedRoute[0]));
   memset(_estimatedTime, 0, MAX_SIZE_ESTIMATED_TIME * sizeof(_estimatedTime[0]));
   memset(_timeOnMap, 0, MAX_SIZE_TIME_ON_MAP * sizeof(_timeOnMap[0]));
}


tSharedPtrDataProvider RoutePreferenceListHandler::getListDataProvider(const ListDateProviderReqMsg& oMsg)
{
   tSharedPtrDataProvider dataProvider;

   if (oMsg.GetListId() == LIST_ID_NAV_ROUTE_PREFERENCE)
   {
      _currentRouteOptionStr = getRouteCondition(_navMiddleware);
      _currentCustomRouteOptionStr = getRouteCalculationType(_navMiddleware);
      _currentEstimatedTimeStr     = getEstimatedTimeOption();
      _currentTimeOnMapStr         = getTimeOnMapOption();

      resetSelectedOption();

      _routeType[_currentRouteOption] = true;
      _customisedRoute[_currentCustomRouteOption] = true;
      _estimatedTime[_currentEstimatedTimeOption] = true;
      _timeOnMap[_currentTimeOnMapOption] = true;

      ETG_TRACE_USR4(("RoutePreferenceListHandler::getListDataProvider(%d)", oMsg.GetListId()));
      ListDataProviderBuilder listBuilder(LIST_ID_NAV_ROUTE_PREFERENCE);
      listBuilder.AddItem(0, 0, DATA_CONTEXT_COMBOBOX_LIST_ITEMS_3)
      .AddData(RoutePreference_ListItemText[0])
      .AddId(0, 1).AddData(RouteTypeStrings[0]).AddData(_routeType[0])
      .AddId(0, 2).AddData(RouteTypeStrings[1]).AddData(_routeType[1])
      .AddId(0, 3).AddData(RouteTypeStrings[2]).AddData(_routeType[2])
      .AddData(_currentRouteOptionStr);
      listBuilder.AddItem(1, 0, DATA_CONTEXT_ROUTE_SETTINGS_OTHER_OPTIONS)
      .AddData(RoutePreference_ListItemText[1]);
      listBuilder.AddItem(2, 0, DATA_CONTEXT_COMBOBOX_LIST_ITEMS_2)
      .AddData(RoutePreference_ListItemText[2])
      .AddId(2, 1).AddData(CustomRouteOptionStrings[0]).AddData(_customisedRoute[0])
      .AddId(2, 2).AddData(CustomRouteOptionStrings[1]).AddData(_customisedRoute[1])
      .AddData(_currentCustomRouteOptionStr);
      listBuilder.AddItem(3, 0, DATA_CONTEXT_COMBOBOX_LIST_ITEMS_2)
      .AddData(RoutePreference_ListItemText[3])
      .AddId(3, 1).AddData(EstimatedTimeStrings[0]).AddData(_estimatedTime[0])
      .AddId(3, 2).AddData(EstimatedTimeStrings[1]).AddData(_estimatedTime[1])
      .AddData(_currentEstimatedTimeStr);
      listBuilder.AddItem(4, 0, DATA_CONTEXT_COMBOBOX_LIST_ITEMS_2)
      .AddData(RoutePreference_ListItemText[4])
      .AddId(4, 1).AddData(TimeOnMapStrings[0]).AddData(_timeOnMap[0])
      .AddId(4, 2).AddData(TimeOnMapStrings[1]).AddData(_timeOnMap[1])
      .AddData(_currentTimeOnMapStr);
      dataProvider = listBuilder.CreateDataProvider();
   }
   return dataProvider;
}


bool RoutePreferenceListHandler::onCourierMessage(const ButtonListItemUpdMsg& oMsg)
{
   bool isMsgConsumed = false;
   if (oMsg.GetListId() == LIST_ID_NAV_ROUTE_PREFERENCE && oMsg.GetReaction() == enRelease)
   {
      unsigned int _hdl = oMsg.GetHdl();
      ETG_TRACE_USR4(("RoutePreferenceListHandler::ButtonListItemUpdMsg: ListId: %d, Hdl: %d", oMsg.GetListId(), _hdl));
      if ((_hdl == ROUTE_TYPE) || (_hdl == CUSTOMIZE_MORE_ROUTES) || (_hdl == ESTIMATED_TIME_OPTIONS) || (_hdl == ETA_SHOWN_ON_MAP))
      {
         ::hmibase::widget::combobox::ComboBoxReqMsg* cbToggleMsg = COURIER_MESSAGE_NEW(::hmibase::widget::combobox::ComboBoxReqMsg)(0, ::hmibase::widget::combobox::enComboBoxAction::Close); //ComboBoxAction enComboBoxAction::Enum::ComboBoxClose:
         if (cbToggleMsg != NULL)
         {
            cbToggleMsg->Post();
         }
      }
      else
      {
      }

      isMsgConsumed = true;
   }
   return isMsgConsumed;
}


bool RoutePreferenceListHandler::onCourierMessage(const ChangeRouteTypeItemReqMsg& oMsg)
{
   unsigned int selectedOption = oMsg.GetRouteOptions();
   if (0 < selectedOption)
   {
      selectedOption--;   // Decremented by 1 since Hmi index starts from 1 and route criterion index starts from 0
   }

   ETG_TRACE_USR4(("RoutePreferenceListHandler::onCourierMessage(ChangeRouteTypeItemReqMsg), Hmi route criterion : %d", selectedOption));

   if (selectedOption < NUM_ROUTE_CRITERION)
   {
      _currentRouteOption = static_cast<enRouteTypeOptions>(selectedOption);
      _currentRouteOptionStr = RouteTypeStrings[selectedOption];
      setRouteType();
   }

   ListRegistry::s_getInstance().updateList(LIST_ID_NAV_ROUTE_PREFERENCE);

   return true;
}


bool RoutePreferenceListHandler::onCourierMessage(const ChangeCustomRouteItemReqMsg& oMsg)
{
   ETG_TRACE_USR4(("RoutePreferenceListHandler::onCourierMessage(ChangeCustomRouteItemReqMsg)"));
   unsigned int selectedOption = oMsg.GetCustomRouteTypeOption();

   if (selectedOption <= MAX_ITEMS_2)
   {
      // Subtract 1 to match the selection and enumeration index,
      // handle for first option in dropdown is 1. However, the enumeration value for same is 0
      _currentCustomRouteOptionStr = CustomRouteOptionStrings[selectedOption - 1U];
      _currentCustomRouteOption = static_cast<enCustomizeRouteOptions>(selectedOption);
      setRouteCalculationType();
   }

   ListRegistry::s_getInstance().updateList(LIST_ID_NAV_ROUTE_PREFERENCE);

   return true;
}


bool RoutePreferenceListHandler::onCourierMessage(const ChangeEstimatedTimeItemReqMsg& oMsg)
{
   ETG_TRACE_USR4(("RoutePreferenceListHandler::onCourierMessage(ChangeEstimatedTimeItemReqMsg)"));
   unsigned int selectedOption = oMsg.GetEstimatedTimeOption();

   if (selectedOption <= MAX_ITEMS_2)
   {
      // Subtract 1 to match the selection and enumeration index,
      // handle for first option in dropdown is 1. However, the enumeration value for same is 0
      _currentEstimatedTimeStr = EstimatedTimeStrings[selectedOption - 1U];
      _currentEstimatedTimeOption = static_cast<enEstimatedTimeOptions>(selectedOption);
      if (selectedOption == 1)
      {
         _infoStore.setEstimatedTimeType(InfoStore::DESTINATION);
      }
      else
      {
         _infoStore.setEstimatedTimeType(InfoStore::WAYPOINT);
      }

      //TODO: Sent as Navi Settings changes as notification to navimiddleware via NaviDataPoolConfig
   }

   ListRegistry::s_getInstance().updateList(LIST_ID_NAV_ROUTE_PREFERENCE);

   return true;
}


bool RoutePreferenceListHandler::onCourierMessage(const ChangeTimeOnMapItemReqMsg& oMsg)
{
   ETG_TRACE_USR4(("RoutePreferenceListHandler::onCourierMessage(ChangeTimeOnMapItemReqMsg)"));
   unsigned int selectedOption = oMsg.GetTimeOnMapOption();

   if (selectedOption <= MAX_ITEMS_2)
   {
      // Subtract 1 to match the selection and enumeration index,
      // handle for first option in dropdown is 1. However, the enumeration value for same is 0
      _currentTimeOnMapStr = TimeOnMapStrings[selectedOption - 1U];
      _currentTimeOnMapOption = static_cast<enTimeOnMapOptions>(selectedOption);
      if (selectedOption == 1)
      {
         _infoStore.setEstimatedTimeMode(InfoStore::ARRIVAL_TIME);
      }
      else
      {
         _infoStore.setEstimatedTimeMode(InfoStore::TRAVEL_TIME);
      }
      //TODO: Sent as Navi Settings changes as notification to navimiddleware via NaviDataPoolConfig
   }

   ListRegistry::s_getInstance().updateList(LIST_ID_NAV_ROUTE_PREFERENCE);

   return true;
}


bool RoutePreferenceListHandler::onCourierMessage(const ApplyRoutePreferenceOptionsReqMsg& oMsg)
{
   COURIER_UNUSED(oMsg);
   ETG_TRACE_USR4(("RoutePreferenceListHandler::onCourierMessage(ApplyRoutePreferenceOptionsReqMsg)"));
   _navMiddleware.applySettings();

   return true;
}


void RoutePreferenceListHandler::setRouteCalculationType() const
{
   navmiddleware::settings::RouteOptionsSettings routeOptionsSettings = _navMiddleware.getRouteOptionSettings();
   ETG_TRACE_USR4(("RoutePreferenceListHandler::toggleRouteCalculationType(%d)", routeOptionsSettings.getMoreRoutesType()));

   switch (_currentCustomRouteOption)
   {
      case SINGLE_ROUTE:
         routeOptionsSettings.setMoreRoutesType(navmiddleware::settings::MORE_ROUTES_TYPE_ALTERNATIVE);
         break;
      case MULTIPLE_ROUTE:
         routeOptionsSettings.setMoreRoutesType(navmiddleware::settings::MORE_ROUTES_TYPE_MULTIPLE);
         break;
      default:
         ETG_TRACE_ERR(("RoutePreferenceListHandler::toggleRouteCalculationType()  wrong index (%d)", routeOptionsSettings.getMoreRoutesType()));
         break;
   }

   _navMiddleware.updateRouteOptionSettings(routeOptionsSettings);
}


void RoutePreferenceListHandler::setRouteType()
{
   ETG_TRACE_USR4(("RoutePreferenceListHandler::setRouteType(), Hmi route criterion : %d", _currentRouteOption));
   navmiddleware::settings::RouteCriteriaSettings routeCriteriaSettings = _navMiddleware.getRouteCriteriaSettings();

   switch (_currentRouteOption)
   {
      case ROUTE_TYPE_FASTEST:
      {
         routeCriteriaSettings.setRouteCriterion(navmiddleware::settings::ROUTE_CRITERION_FAST);
         break;
      }
      case ROUTE_TYPE_ECONOMIC:
      {
         routeCriteriaSettings.setRouteCriterion(navmiddleware::settings::ROUTE_CRITERION_ECONOMIC);
         break;
      }
      case ROUTE_TYPE_SHORTEST:
      {
         routeCriteriaSettings.setRouteCriterion(navmiddleware::settings::ROUTE_CRITERION_SHORT);
         break;
      }
      default:
      {
         ETG_TRACE_ERR(("RoutePreferenceListHandler::SetRouteType()  wrong index"));
         break;
      }
   }

   _navMiddleware.updateRouteCriteriaSettings(routeCriteriaSettings);
   ETG_TRACE_USR4(("RoutePreferenceListHandler::setRouteType(), Navmid route criterion : %d", _navMiddleware.getRouteCriteriaSettings().getRouteCriterion()));
}


#endif // HALL_TO_MDW_COM
