/**************************************************************************************
* @file         : HMIModelSettings.cpp
* @author       :
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) -2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include "HMIModelSettings.h"
#include "NaviSettingsUtil.h"
#include "PositionInfoDataUtils.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/HMIModelSettings.cpp.trc.h"
#endif

#ifdef HALL_TO_MDW_COM

HMIModelSettings::HMIModelSettings(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore)
   : HMIModelBase(navMiddleware, infoStore)
   , _navigationSettingsListHandler(_navMiddleware, infoStore)
   , _routePreferenceListHandler(_navMiddleware, infoStore)
   , _detailedRoutePreferenceListHandler(_navMiddleware, infoStore)
   , _mapSettingsListHandler(_navMiddleware, infoStore)
   , _mapSettingsShowPOIListHandler(_navMiddleware, infoStore)
   , _mapSettingsOtherListHandler(_navMiddleware, infoStore)
   , _mapColorStyelListHandler(_navMiddleware, infoStore)
   , _vehicleParameterListHandler(_navMiddleware, infoStore)
   , _guidanceSettingsHandler(_navMiddleware, infoStore)
   , _trafficIconsSettingsListHandler(_navMiddleware, infoStore)
   , _streamedMapSettingListHandler(_navMiddleware, infoStore)
   , _streamedMapSubOptionListHandler(_navMiddleware, infoStore)
   , _premiumTrafficListHandler(_navMiddleware, infoStore)
   , _speedLimitSettingListHandler(_navMiddleware, infoStore)
   , _userPOISettingsListHandler(_navMiddleware, infoStore)
{
}


HMIModelSettings::~HMIModelSettings()
{
}


void HMIModelSettings::initialize()
{
   ETG_TRACE_USR1(("HMIModelSettings::initialize()"));

   _navigationSettingsListHandler.initialize();
   _routePreferenceListHandler.initialize();
   _detailedRoutePreferenceListHandler.initialize();
   _mapSettingsListHandler.initialize();
   _mapSettingsShowPOIListHandler.initialize();
   _mapSettingsOtherListHandler.initialize();
   _mapColorStyelListHandler.initialize();
   _vehicleParameterListHandler.initialize();
   _guidanceSettingsHandler.initialize();
   _trafficIconsSettingsListHandler.initialize();
   _streamedMapSettingListHandler.initialize();
   _streamedMapSubOptionListHandler.initialize();
   _premiumTrafficListHandler.initialize();
   _speedLimitSettingListHandler.initialize();
   _userPOISettingsListHandler.initialize();
   _infoStore.registerDataPropertyUpdateCallback(*this);
}


void HMIModelSettings::deinitialize()
{
   ETG_TRACE_USR1(("HMIModelSettings::deinitialize()"));

   _navigationSettingsListHandler.deinitialize();
   _routePreferenceListHandler.deinitialize();
   _detailedRoutePreferenceListHandler.deinitialize();
   _mapSettingsListHandler.deinitialize();
   _mapSettingsShowPOIListHandler.deinitialize();
   _mapSettingsOtherListHandler.deinitialize();
   _mapColorStyelListHandler.deinitialize();
   _vehicleParameterListHandler.deinitialize();
   _guidanceSettingsHandler.deinitialize();
   _trafficIconsSettingsListHandler.deinitialize();
   _streamedMapSettingListHandler.deinitialize();
   _streamedMapSubOptionListHandler.deinitialize();
   _premiumTrafficListHandler.deinitialize();
   _speedLimitSettingListHandler.deinitialize();
   _userPOISettingsListHandler.deinitialize();
   _infoStore.unregisterDataPropertyUpdateCallback(*this);
}


bool HMIModelSettings::onPropertyUpdateDistanceUnitSystemChanged()
{
   ETG_TRACE_USR1(("HMIModelSettings::onPropertyUpdateDistanceUnitSystemChanged(%d)", _infoStore.getDistanceUnitSystem()));
   _navMiddleware.setDistanceUnitSystem((navmiddleware::DistanceUnitSystem)_infoStore.getDistanceUnitSystem());
   _vehicleParameterListHandler.updateUnitSystem();
   POST_MSG((COURIER_MESSAGE_NEW(UpdateStartupVehicleProfileReqMsg)()));
   return true;
}


bool HMIModelSettings::onPropertyUpdateLanguageChanged()
{
   std::stringstream traceOut;
   traceOut << "System: " << (int)_infoStore.getSystemLanguageIndex() << "/" << _infoStore.getSystemLanguageISOCode().c_str() << ", ";
   traceOut << "SDS: " << (int)_infoStore.getSDSLanguageIndex() << "/" << _infoStore.getSDSLanguageISOCode().c_str() << ", ";
   traceOut << "Traffic: " << (int)_infoStore.getTrafficTextLanguageIndex() << "/" << _infoStore.getTrafficTextISOCode().c_str() << ", ";
   traceOut << "Cluster: " << (int)_infoStore.getClusterLanguageIndex() << "/" << _infoStore.getClusterLanguageISOCode().c_str();

   ETG_TRACE_COMP(("HMIModelSettings::onPropertyUpdateLanguageChanged(%s)", traceOut.str().c_str()));

   _navMiddleware.setLanguage(
      _infoStore.getSystemLanguageISOCode(),
      (navmiddleware::LanguageCode)_infoStore.getSystemLanguageIndex(),
      _infoStore.getSDSLanguageISOCode(),
      (navmiddleware::LanguageCode)_infoStore.getSDSLanguageIndex(),
      _infoStore.getTrafficTextISOCode(),
      (navmiddleware::LanguageCode)_infoStore.getTrafficTextLanguageIndex(),
      _infoStore.getClusterLanguageISOCode(),
      (navmiddleware::LanguageCode)_infoStore.getClusterLanguageIndex());

   return true;
}


bool HMIModelSettings::onPropertyUpdateCountryCodeChanged()
{
   Candera::UInt32 countryCode = _infoStore.getCountryCode();
   ETG_TRACE_USR4(("HMIModelSettings::onPropertyUpdateCountryCodeChanged(%d)", countryCode));

   (*_naviSpellerInfo).mCountryCode = countryCode;
   _naviSpellerInfo.MarkItemModified(ItemKey::NaviSpellerInfo::CountryCodeItem);
   _naviSpellerInfo.SendUpdate(true);
   (*_spellerPageInfoData).mSpellerPageNumber = "ÆØÜ";
   _spellerPageInfoData.SendUpdate(true);

   return true;
}


bool HMIModelSettings::onPropertyUpdateLightModeChanged()
{
   ETG_TRACE_USR1(("HMIModelSettings::onPropertyUpdateLightModeChanged(%d)", _infoStore.getLightMode()));

   bool isDayModeActive = false;

   switch (_infoStore.getLightMode())
   {
      case LIGHT_MODE_DAY_PERMANENT:
         _navMiddleware.configureLightMode(navmiddleware::ConfigurationInfos::LIGHT_MODE_DAY_PERMANENT);
         isDayModeActive = true;
         break;
      case LIGHT_MODE_NIGHT_PERMANENT:
         _navMiddleware.configureLightMode(navmiddleware::ConfigurationInfos::LIGHT_MODE_NIGHT_PERMANENT);
         break;
      case LIGHT_MODE_DAY_AUTO:
         _navMiddleware.configureLightMode(navmiddleware::ConfigurationInfos::LIGHT_MODE_DAY_AUTOMATIC);
         isDayModeActive = true;
         break;
      case LIGHT_MODE_NIGHT_AUTO:
         _navMiddleware.configureLightMode(navmiddleware::ConfigurationInfos::LIGHT_MODE_NIGHT_AUTOMATIC);
         break;
      default:
         ETG_TRACE_ERR(("Invalid lightMode received!"));
         break;
   }
   if ((*_mapModeData).mIsDayModeActive != isDayModeActive)
   {
      (*_mapModeData).mIsDayModeActive = isDayModeActive;
      _mapModeData.MarkAllItemsModified();
      _mapModeData.SendUpdate(true);
   }
   setSatelliteIcon(isDayModeActive);
   _navigationSettingsListHandler.configureRouteOptionsBorder(isDayModeActive);
   return true;
}


bool HMIModelSettings::onPropertySpeedLimitStatusChanged() const
{
   ETG_TRACE_USR1(("HMIModelSettings::onPropertySpeedLimitStatusChanged()"));

   bool display = (_infoStore.getSpeedLimitDisplay() != 0);

   switch (_infoStore.getSpeedLimitUnit())
   {
      case KM_PER_HOUR:
      {
         _navMiddleware.setSpeedLimitData(::navmiddleware::SpeedLimitData(display ? navmiddleware::SPEEDLIMITUSAGE_SPEEDLIMIT_FROM_EXTERNAL_SOURCE : navmiddleware::SPEEDLIMITUSAGE_SPEEDLIMIT_FROM_EXTERNAL_SOURCE_UNAVAILABLE,
                                          _infoStore.getSpeedLimit(),
                                          ::navmiddleware::KM_PER_HOUR,
                                          false,
                                          false));
         break;
      }
      case MI_PER_HOUR:
      {
         _navMiddleware.setSpeedLimitData(::navmiddleware::SpeedLimitData(display ? navmiddleware::SPEEDLIMITUSAGE_SPEEDLIMIT_FROM_EXTERNAL_SOURCE : navmiddleware::SPEEDLIMITUSAGE_SPEEDLIMIT_FROM_EXTERNAL_SOURCE_UNAVAILABLE,
                                          _infoStore.getSpeedLimit(),
                                          ::navmiddleware::MI_PER_HOUR,
                                          false,
                                          false));
         break;
      }
      default:
         ETG_TRACE_USR1(("Invalid SpeedUnit received."));
         break;
   }

   return true;
}


bool HMIModelSettings::onPropertyDateFormatChanged()const
{
   ETG_TRACE_USR1(("HMIModelSettings::onPropertyDateFormatChanged(%d)", _infoStore.getDateFormat()));
   _navMiddleware.setDateFormat(_infoStore.getDateFormat());
   return true;
}


bool HMIModelSettings::onPropertyTimeFormatChanged()const
{
   ETG_TRACE_USR1(("HMIModelSettings::onPropertyTimeFormatChanged(%d)", _infoStore.getTimeFormat()));
   _navMiddleware.setTimeFormat(static_cast<navmiddleware::TimeFormat>(_infoStore.getTimeFormat()));
   return true;
}


bool HMIModelSettings::onPropertyUpdateLocalTimeOffsetInfoChanged() const
{
   ETG_TRACE_USR1(("HMIModelSettings::onPropertyUpdateLocalTimeOffsetInfoChanged(%d %d)", _infoStore.getLocalTimeOffset(), _infoStore.isClockAutoMode()));

   if (_infoStore.isClockAutoMode())
   {
      _navMiddleware.setDisplayTimeOffset(_infoStore.getLocalTimeOffset(), navmiddleware::ConfigurationInfos::TIMEZONE_AND_DST_OFFSET_CONSIDERATION__REQUIRED);
   }
   else
   {
      _navMiddleware.setDisplayTimeOffset(_infoStore.getLocalTimeOffset(), navmiddleware::ConfigurationInfos::TIMEZONE_AND_DST_OFFSET_CONSIDERATION__NOT_REQUIRED);
   }

   return true;
}


bool HMIModelSettings::onPropertyUpdateResetFactoryChanged()
{
   ETG_TRACE_USR1(("HMIModelSettings::onPropertyUpdateResetFactoryChanged()"));
   uint8_t previousSatelliteSource = _infoStore.getSatelliteSource();
   _navMiddleware.restoreFactorySettings(navmiddleware::settings::RESTORE_TYPE__USER_DATA, navmiddleware::settings::USER_PROFILE_GROUP__ALL_USERS);
   // Reset HMI settings which are handled from Infostore
   _navigationSettingsListHandler.resetNaviHmiSettings();
   _navMiddleware.applySettings();

   _navigationSettingsListHandler.configureCarsor();
   _vehicleParameterListHandler.getCurrentVehicleProfile();   // Fetch vehicle data values after factory reset

   uint8_t currentSatelliteSource = _infoStore.getSatelliteSource();
   if (previousSatelliteSource != currentSatelliteSource)
   {
      POST_MSG((COURIER_MESSAGE_NEW(UpdateDeletePopupTextReqMsg)(POPUP_SATELLITE_SOURCE_CHANGE)));
   }

   return true;
}


bool HMIModelSettings::onPropertyUpdateResetFactoryProductionChanged()
{
   ETG_TRACE_USR1(("HMIModelSettings::onPropertyUpdateResetFactoryProductionChanged()"));
   uint8_t previousSatelliteSource = _infoStore.getSatelliteSource();

   //reload KDS values
   _infoStore.loadPersistencyFromDataPool();

   //update default position in middleware
   const PosHeadingWGS84<float>& posHeadingWGS84 = _infoStore.getDefaultPosition();
   _navMiddleware.setDefaultPosition(
      ::navmiddleware::GeoCoordinateDegree(
         static_cast<double>(posHeadingWGS84._pos._latitude),
         static_cast<double>(posHeadingWGS84._pos._longitude)), posHeadingWGS84._heading);

   _navMiddleware.restoreFactorySettings(navmiddleware::settings::RESTORE_TYPE__ALL, navmiddleware::settings::USER_PROFILE_GROUP__ALL_USERS);
   // Reset HMI settings which are handled from Infostore
   _navigationSettingsListHandler.resetNaviHmiSettings();
   _navMiddleware.applySettings();

   _navigationSettingsListHandler.configureCarsor();
   _vehicleParameterListHandler.getCurrentVehicleProfile();   // Fetch vehicle data values after factory reset

   uint8_t currentSatelliteSource = _infoStore.getSatelliteSource();
   if (previousSatelliteSource != currentSatelliteSource)
   {
      POST_MSG((COURIER_MESSAGE_NEW(UpdateDeletePopupTextReqMsg)(POPUP_SATELLITE_SOURCE_CHANGE)));
   }
   return true;
}


#endif // HALL_TO_MDW_COM
