/**************************************************************************************
* @file         : RouteListHandler.h
* @author       : KanagaDurga Balakrishnan
* @addtogroup   :
* @brief        :
* @copyright    : (c) 2018-2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include "RouteListHandler.h"
#include "RouteDataUtils.h"


#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/RouteListHandler.cpp.trc.h"
#endif

static const Courier::Identifier  IdClickRecalculateButton             = Courier::Identifier("Recalculate_Route/ButtonWidget");
static const Courier::Identifier  IdClick_BlockUnBlock_Button          = Courier::Identifier("Layer_BlockRouteList/ButtonWidget");
static const Courier::ViewId BLOCK_ROUTE_LIST_SCENENAME                = Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#N_Guidance_BlockRouteList");

static const char* RECALCULATE_ROUTE_DISABLED_ICON                     = "AppHmi_NavigationModule#Images#N_BlockRouteList#Icon_RecalcRoute_disabled";
static const char* RECALCULATE_ROUTE_TOUCHED_ICON                      = "AppHmi_NavigationModule#Images#N_BlockRouteList#Icon_RecalcRoute_touched";
static const char* RECALCULATE_ROUTE_NORMAL_ICON                       = "AppHmi_NavigationModule#Images#N_BlockRouteList#Icon_RecalcRoute_normal";
static const char* ROUTELIST_SELECTION_ACTIVEBITMAP                    = "AppHmi_NavigationModule#Images#N_Speller_MatchList#MatchListEntry_active_1";
static const char* ROUTELIST_SELECTION_NORMALBITMAP                    = "AppHmi_NavigationModule#Images#N_BlockRouteList#Toucharea_RouteSegmentListEntry_1";
static const char* ROUTELIST_BLOCKED_ICON_NORMAL                       = "AppHmi_NavigationModule#Images#N_BlockRouteList#Icon_Blocked_1_normal";
static const char* ROUTELIST_BLOCKED_ICON_TOUCHED                      = "AppHmi_NavigationModule#Images#N_BlockRouteList#Icon_Blocked_2_touched";
static const char* ROUTELIST_UNBLOCKED_ICON_NORMAL                     = "AppHmi_NavigationModule#Images#N_BlockRouteList#Icon_Unblocked_1_normal";
static const char* ROUTELIST_UNBLOCKED_ICON_TOUCHED                    = "AppHmi_NavigationModule#Images#N_BlockRouteList#Icon_Unblocked_2_touched";

static const char* const TURNLIST_CURRENT_LOCATION_ITEM = "GuidanceRouteListItem";
static const Candera::String LOADING = LANGUAGE_STRING(TextId_0x1458, "Loading...");
static const unsigned int INVALID_INDEX = -1;

using namespace navmiddleware;

RouteListHandler::RouteListHandler(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore)
   : HMIModelBase(navMiddleware, infoStore)
   , _listStartIndex(0)
   , _windowElementSize(0)
   , _listId(0)
   , _firstSelectionIndex(INVALID_INDEX)
   , _lastSelectionIndex(INVALID_INDEX)
   , _currentWindowStartIndex(0)
   , _routeListSize(0)
   , _isblockrouteListSelectionActive(false)
   , _listSelectionIndex(false)
   , _resetSelectionValue(false)
   , _recalculateRouteSelected(false)
   , _isStartRouteActive(false)
   , _isListInvalidated(false)
   , _isListRequested(false)
   , _hasRoadNumberImages(false)
   , _avoidElementStatus(RouteListInfos::AVOID_ELEMENTS_STATUS_INACTIVE)
   , _isRouteListShown(false)
{
}


RouteListHandler::~RouteListHandler()
{
}


void RouteListHandler::initialize()
{
   ETG_TRACE_USR4(("RouteListHandler::initialize()"));

   _navMiddleware.registerRoutePropertyUpdateCallback(*this);
   _navMiddleware.registerGuidancePropertyUpdateCallback(*this);
   _infoStore.registerDataPropertyUpdateCallback(*this);

   ShowBlockUnblockTextReqMsg(true);
}


void RouteListHandler::deinitialize()
{
   ETG_TRACE_USR4(("RouteListHandler::deinitialize()"));

   _navMiddleware.unregisterRoutePropertyUpdateCallback(*this);
   _navMiddleware.unregisterGuidancePropertyUpdateCallback(*this);
   _infoStore.unregisterDataPropertyUpdateCallback(*this);
}


bool RouteListHandler::onCourierMessage(const ListDateProviderReqMsg& oMsg)
{
   bool bIsMsgConsumed = false;

   unsigned int listId = oMsg.GetListId();
   if ((listId == LIST_ID_GUIDANCE_SHOWROUTELIST) || (listId == LIST_ID_GUIDANCE_BLOCKROUTELIST))
   {
      _isListRequested = true;
      _listStartIndex = oMsg.GetStartIndex();
      _windowElementSize = oMsg.GetWindowElementSize();
      _listId = listId;
      ETG_TRACE_USR4(("RouteListHandler::onCourierMessage(ListDateProviderReqMsg(listID %d, startIndex %d, windowSize %d))", _listId, _listStartIndex, _windowElementSize));

      if (0 == _routeListSize)
      {
         sendListLoadingStatus();   // Sends list loading data to list widget
      }

      _navMiddleware.requestRouteList(_infoStore.getUserSelectedRouteAlternative(), _listStartIndex, _windowElementSize);
      bIsMsgConsumed = true;
   }

   return bIsMsgConsumed;
}


bool RouteListHandler::onCourierMessage(const ShowRouteListViewReqMsg& oMsg)
{
   _isRouteListShown = oMsg.GetIsRouteListShown();
   unsigned int indexAlternative = _infoStore.getUserSelectedRouteAlternative();

   ETG_TRACE_USR4(("RouteListHandler::onCourierMessage(ShowRouteListViewReqMsg), Route index : %d", indexAlternative));

   if (true == _isRouteListShown)   // On entry into Show/Block route list scene
   {
      _listId = _routeListSize = _listStartIndex = 0;
      _isListRequested = _isListInvalidated = _hasRoadNumberImages = false;
      _navMiddleware.startRouteListInfo(indexAlternative, navmiddleware::ROUTE_LIST_CONFIGURATION__WITH_FREEWAY_EXITS);
   }
   else   // On exit from Show/Block route list scene
   {
      if ((false == _recalculateRouteSelected) && (RouteListInfos::AVOID_ELEMENTS_STATUS_ACTIVE != _avoidElementStatus))
      {
         _navMiddleware.stopRouteListInfo(indexAlternative);
      }

      if (LIST_ID_GUIDANCE_BLOCKROUTELIST == _listId)
      {
         _listSelectionIndex = _resetSelectionValue = _isblockrouteListSelectionActive = false;
         _firstSelectionIndex = _lastSelectionIndex = INVALID_INDEX;
         enableRouteCalcButton(false);
      }
   }
   return true;
}


void RouteListHandler::onPropertyUpdateRouteListInvalidate()
{
   ETG_TRACE_USR4(("RouteListHandler::onPropertyUpdateRouteListInvalidate, List start index : %d", _listStartIndex));

   _isListInvalidated = true;
   _navMiddleware.requestRouteList(_infoStore.getUserSelectedRouteAlternative(), _listStartIndex, _windowElementSize);
}


void RouteListHandler::onPropertyUpdateRouteListInfoChanged()
{
   ETG_TRACE_USR4(("RouteListHandler::onPropertyUpdateRouteListInfoChanged()"));

   const navmiddleware::RouteListInfos& routeListInfos = _navMiddleware.getRouteListInfos();
   navmiddleware::RouteListInfos::AvoidElementsStatus avoidElementStatus = routeListInfos.getAvoidElementsStatus();
   unsigned int actDestListSize = routeListInfos.getTotalNumberOfElements();

   ETG_TRACE_USR4(("RouteListHandler::onPropertyUpdateRouteListInfoChanged(), Route list size(Previous, New) : (%d, %d)", _routeListSize, actDestListSize));
   ETG_TRACE_USR4(("RouteListHandler::onPropertyUpdateRouteListInfoChanged(), Are road number images present : %d", routeListInfos.hasRoadNumberImages()));
   ETG_TRACE_USR4(("RouteListHandler::onPropertyUpdateRouteListInfoChanged(), Avoid route element status : %d", avoidElementStatus));

   bool prevHasRoadNumberImages = _hasRoadNumberImages;
   _hasRoadNumberImages = routeListInfos.hasRoadNumberImages();

   // Below are the condition checks for updating the list in HMI
   // 1 - List is requested by widget on scene entry or on scroll
   // 2 - Updated list is available from Navmiddleware
   // 3 - When there is a change in route list size
   if ((true == _isListRequested) || (true == _isListInvalidated) || (_routeListSize != actDestListSize))
   {
      // The below if block is executed when there is a change in route list size as RG progresses
      if ((_routeListSize != 0) && ((_routeListSize - 1) == actDestListSize))
      {
         _routeListSize = actDestListSize;
         _isListRequested = true;

         // When there is one less entry in route list from Navmiddleware, the list start index is decremented by 1 and the list is requested again
         // Without this, the last entry in the route list is not visible when list size is decremented
         if (0 < _listStartIndex)
         {
            _listStartIndex--;
         }
         _navMiddleware.requestRouteList(_infoStore.getUserSelectedRouteAlternative(), _listStartIndex, _windowElementSize);
      }
      else
      {
         _routeListSize = actDestListSize;
         _isListInvalidated = false;

         if (0 < _routeListSize)
         {
            _isListRequested = false;
            tSharedPtrDataProvider dataProvider = (LIST_ID_GUIDANCE_BLOCKROUTELIST == _listId) ? getBlockRouteListDataProvider() : getShowRouteListDataProvider();
            sendRouteListData(dataProvider);
         }
      }
   }
   // When road number images are available from navmiddleware, list is updated with the images and sent to list widget
   else if ((LIST_ID_GUIDANCE_SHOWROUTELIST == _listId) && (false == prevHasRoadNumberImages) && (true == _hasRoadNumberImages))
   {
      updateRouteListData();
   }

   if (avoidElementStatus != _avoidElementStatus)
   {
      // When blocking is initiated from HMI, status is changed from INACTIVE to ACTIVE or NO_DETOUR
      if (RouteListInfos::AVOID_ELEMENTS_STATUS_INACTIVE != avoidElementStatus)
      {
         onAvoidElementStatusChange(avoidElementStatus);
      }
      // When blocking is successful, but removed later because of adding waypoint
      else if ((_avoidElementStatus == RouteListInfos::AVOID_ELEMENTS_STATUS_ACTIVE)   // Previous state - ACTIVE
               && (avoidElementStatus == RouteListInfos::AVOID_ELEMENTS_STATUS_INACTIVE))    // Current state - INACTIVE
      {
         unsigned int indexAlternative = _infoStore.getUserSelectedRouteAlternative();
         _navMiddleware.stopRouteListInfo(indexAlternative);
         ShowBlockUnblockTextReqMsg(true);   // Text changed from Unblock to Block
      }
      else
      {
         // do nothing
      }
      _avoidElementStatus = avoidElementStatus;
   }
}


void RouteListHandler::onAvoidElementStatusChange(const navmiddleware::RouteListInfos::AvoidElementsStatus& avoidElementStatus)
{
   ETG_TRACE_USR4(("RouteListHandler::onAvoidElementStatusChange(), Avoid element status : %d", avoidElementStatus));

   if (true == _recalculateRouteSelected)
   {
      _recalculateRouteSelected = false;
      // If "Block route list" button is disabled, then it is enabled once avoid element status is changed
      if ((false == (*_guidanceBlockUnblockButton).mBlockUnblockEnable) && (false == _infoStore.getIsHardRestrictionPresent()))
      {
         (*_guidanceBlockUnblockButton).mBlockUnblockEnable = true;
         _guidanceBlockUnblockButton.MarkItemModified(ItemKey::GuidanceBlockUnblockButton::BlockUnblockEnableItem);
         _guidanceBlockUnblockButton.SendUpdate();
      }

      if (RouteListInfos::AVOID_ELEMENTS_STATUS_NO_DETOUR_POSSIBLE == avoidElementStatus)
      {
         ETG_TRACE_USR4(("RouteListHandler::onAvoidElementStatusChange(), No detour possible"));

         const Candera::String DATA_CONTEXT__TEXT_LINE1 = LANGUAGE_STRING(TextId_0x1723, "There is no alternative route ");
         const Candera::String DATA_CONTEXT__TEXT_LINE2 = LANGUAGE_STRING(TextId_0x1724, "from the current location");

         (*_PopupText).mTextLine1 = DATA_CONTEXT__TEXT_LINE1;
         (*_PopupText).mTextLine2 = DATA_CONTEXT__TEXT_LINE2;
         _PopupText.MarkAllItemsModified();
         _PopupText.SendUpdate(true);
         POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_MultiLine_Text"))));

         unsigned int indexAlternative = _infoStore.getUserSelectedRouteAlternative();
         _navMiddleware.stopRouteListInfo(indexAlternative);
         _avoidElementStatus = RouteListInfos::AVOID_ELEMENTS_STATUS_INACTIVE;
      }
      else // Avoid element status is set to ACTIVE
      {
         ShowBlockUnblockTextReqMsg(false);
      }
   }
}


void RouteListHandler::onPropertyUpdateGuidanceStateChanged()
{
   bool guidanceState = _navMiddleware.isGuidanceActive();
   ETG_TRACE_USR4(("RouteListHandler::onPropertyUpdateGuidanceStateChanged(), Guidance Active: %d", guidanceState));

   if (false == guidanceState)
   {
      _recalculateRouteSelected = false;
      ShowBlockUnblockTextReqMsg(true);

      // When guidance is stopped, if "Block route list" button is disabled, then it is enabled so that button is active for next guidance
      if (false == (*_guidanceBlockUnblockButton).mBlockUnblockEnable)
      {
         (*_guidanceBlockUnblockButton).mBlockUnblockEnable = true;
         _guidanceBlockUnblockButton.MarkItemModified(ItemKey::GuidanceBlockUnblockButton::BlockUnblockEnableItem);
         _guidanceBlockUnblockButton.SendUpdate();
      }
   }
}


tSharedPtrDataProvider RouteListHandler::getShowRouteListDataProvider()
{
   ETG_TRACE_USR4(("RouteListHandler::getShowRouteListDataProvider()"));

   ListDataProviderBuilder listBuilder(_listId);
   ShowRouteListItemData item;

   // get route list elements
   const navmiddleware::RouteListInfos& routeListInfos = _navMiddleware.getRouteListInfos();
   const std::vector< ::navmiddleware::RouteListInfos::RouteListElementInfo >& routeListElementInfos = routeListInfos.getRouteListElementInfos();
   ::std::vector<navmiddleware::RouteListInfos::RouteListElementInfo>::const_iterator info = routeListElementInfos.begin();

   for (unsigned int idx = _listStartIndex; (idx < _listStartIndex + _windowElementSize) && (info != routeListElementInfos.end()); idx++, info++)
   {
      ETG_TRACE_USR4(("RouteListHandler::getShowRouteListDataProvider, Index in HMI list : %d, Index in Navmiddleware list : %d", idx, info->m_indexInRouteList));
      ETG_TRACE_USR4(("RouteListHandler::getShowRouteListDataProvider, Route name : %s", info->m_roadInfo.c_str()));

      getFullTurnlistTurnDataItem(item, info);

      item.mDistanceInfo = info->m_lengthOfElement.c_str();
      item.mRouteList_BlockUnblockNormalIcon = ImageLoader::getAssetBitmapImage(ROUTELIST_UNBLOCKED_ICON_NORMAL);
      item.mRouteList_BlockUnblockPressedIcon = ImageLoader::getAssetBitmapImage(ROUTELIST_UNBLOCKED_ICON_TOUCHED);
      item.mRouteSelectionActiveBitmap = ImageLoader::getAssetBitmapImage(ROUTELIST_SELECTION_NORMALBITMAP);

      listBuilder.AddItem(
         idx,
         0UL,
         TURNLIST_CURRENT_LOCATION_ITEM)
      .AddDataBindingUpdater<ShowRouteListItemDataBindingSource>(item);
   }

   return listBuilder.CreateDataProvider(_listStartIndex, _routeListSize);
}


tSharedPtrDataProvider RouteListHandler::getBlockRouteListDataProvider()
{
   ETG_TRACE_USR4(("RouteListHandler::getBlockRouteListDataProvider()"));

   ListDataProviderBuilder listBuilder(_listId);
   ShowRouteListItemData item;

   // get route list elements
   const navmiddleware::RouteListInfos& routeListInfos = _navMiddleware.getRouteListInfos();
   const std::vector< ::navmiddleware::RouteListInfos::RouteListElementInfo >& routeListElementInfos = routeListInfos.getRouteListElementInfos();
   ::std::vector<navmiddleware::RouteListInfos::RouteListElementInfo>::const_iterator info = routeListElementInfos.begin();

   for (unsigned int idx = _listStartIndex; (idx < _listStartIndex + _windowElementSize) && (info != routeListElementInfos.end()); idx++, info++)
   {
      ETG_TRACE_USR4(("RouteListHandler::getBlockRouteListDataProvider, Index in HMI list : %d, Index in Navmiddleware list : %d", idx, info->m_indexInRouteList));
      ETG_TRACE_USR4(("RouteListHandler::getBlockRouteListDataProvider, Route name : %s", info->m_roadInfo.c_str()));

      getFullTurnlistTurnDataItem(item, info);
      item.mDistanceInfo = info->m_lengthOfElement.c_str();

      if ((INVALID_INDEX != _firstSelectionIndex) && (info->m_indexInRouteList >= _firstSelectionIndex) && (info->m_indexInRouteList <= _lastSelectionIndex))
      {
         ETG_TRACE_USR4(("RouteListHandler::getBlockRouteListDataProvider(), Active image"));
         item.mRouteList_BlockUnblockNormalIcon = ImageLoader::getAssetBitmapImage(ROUTELIST_BLOCKED_ICON_NORMAL);
         item.mRouteList_BlockUnblockPressedIcon = ImageLoader::getAssetBitmapImage(ROUTELIST_BLOCKED_ICON_TOUCHED);
         item.mRouteSelectionActiveBitmap = ImageLoader::getAssetBitmapImage(ROUTELIST_SELECTION_ACTIVEBITMAP);
      }
      else
      {
         ETG_TRACE_USR4(("RouteListHandler::getBlockRouteListDataProvider(), Normal image"));
         item.mRouteList_BlockUnblockNormalIcon = ImageLoader::getAssetBitmapImage(ROUTELIST_UNBLOCKED_ICON_NORMAL);
         item.mRouteList_BlockUnblockPressedIcon = ImageLoader::getAssetBitmapImage(ROUTELIST_UNBLOCKED_ICON_TOUCHED);
         item.mRouteSelectionActiveBitmap = ImageLoader::getAssetBitmapImage(ROUTELIST_SELECTION_NORMALBITMAP);
      }

      listBuilder.AddItem(
         idx,
         0UL,
         TURNLIST_CURRENT_LOCATION_ITEM)
      .AddDataBindingUpdater<ShowRouteListItemDataBindingSource>(item);
   }

   return listBuilder.CreateDataProvider(_listStartIndex, _routeListSize);
}


void RouteListHandler::sendListLoadingStatus()
{
   ETG_TRACE_USR4(("RouteListHandler::sendListLoadingStatus()"));

   ListDataProviderBuilder listBuilder(_listId);
   ShowRouteListItemData item;

   item.mRouteListItem = LOADING;
   item.mRoadIconActiveBitmap = loadImage(NULL, 0, false);
   item.mDistanceInfo = "";
   listBuilder.AddItem(
      0UL,
      0UL,
      TURNLIST_CURRENT_LOCATION_ITEM)
   .AddDataBindingUpdater<ShowRouteListItemDataBindingSource>(item);

   tSharedPtrDataProvider dataProvider = listBuilder.CreateDataProvider(_listStartIndex, 1);
   sendRouteListData(dataProvider);
}


void RouteListHandler::sendRouteListData(tSharedPtrDataProvider& listDataProvider)
{
   ETG_TRACE_USR4(("RouteListHandler::sendRouteListData()"));

   if (!listDataProvider.PointsToNull())
   {
      if (listDataProvider->listSize() > 0)
      {
         listDataProvider->setCacheOnOff(false);
         POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(listDataProvider)));
      }
   }
}


void RouteListHandler::updateRouteListData()
{
   ETG_TRACE_USR4(("RouteListHandler::updateRouteListData()"));

   // get route list elements
   const navmiddleware::RouteListInfos& routeListInfos = _navMiddleware.getRouteListInfos();
   const std::vector< ::navmiddleware::RouteListInfos::RouteListElementInfo >& routeListElementInfos = routeListInfos.getRouteListElementInfos();
   ::std::vector<navmiddleware::RouteListInfos::RouteListElementInfo>::const_iterator info = routeListElementInfos.begin();

   for (unsigned int idx = _listStartIndex; (idx < _listStartIndex + _windowElementSize) && (info != routeListElementInfos.end()); idx++, info++)
   {
      bool isRoadIconAvail = false;
      if (0 < info->m_roadNumberImages.size())
      {
         navmiddleware::Image roadIcon = info->m_roadNumberImages[0];
         if (Image::TYPE__UNDEFINED != roadIcon.getType())
         {
            isRoadIconAvail = true;
         }
      }

      if (true == isRoadIconAvail)
      {
         ListDataProviderBuilder listBuilder(_listId);
         ShowRouteListItemData item;

         ETG_TRACE_USR4(("RouteListHandler::getShowRouteListDataProviderUpdater, Index in HMI list : %d, Index in Navmiddleware list : %d", idx, info->m_indexInRouteList));
         ETG_TRACE_USR4(("RouteListHandler::getShowRouteListDataProviderUpdater, Route name : %s", info->m_roadInfo.c_str()));

         getFullTurnlistTurnDataItem(item, info);

         item.mDistanceInfo = info->m_lengthOfElement.c_str();
         item.mRouteList_BlockUnblockNormalIcon = ImageLoader::getAssetBitmapImage(ROUTELIST_UNBLOCKED_ICON_NORMAL);
         item.mRouteList_BlockUnblockPressedIcon = ImageLoader::getAssetBitmapImage(ROUTELIST_UNBLOCKED_ICON_TOUCHED);
         item.mRouteSelectionActiveBitmap = ImageLoader::getAssetBitmapImage(ROUTELIST_SELECTION_NORMALBITMAP);

         listBuilder.AddItem(
            idx,
            0UL,
            TURNLIST_CURRENT_LOCATION_ITEM,
            0UL,
            idx)
         .AddDataBindingUpdater<ShowRouteListItemDataBindingSource>(item);

         // DataProviderUpdater updates only 1 item at a time
         tSharedPtrDataProviderUpdater dataProvider = listBuilder.CreateDataProviderUpdater();
         if (!dataProvider.PointsToNull())
         {
            POST_MSG((COURIER_MESSAGE_NEW(ListDataProviderUpdMsg)(dataProvider)));
         }
      }
   }
}


void RouteListHandler::getFullTurnlistTurnDataItem(ShowRouteListItemData& item, ::std::vector<navmiddleware::RouteListInfos::RouteListElementInfo>::const_iterator info)
{
   ETG_TRACE_USR4(("RouteListHandler::getFullTurnlistTurnDataItem()"));

   getTurnlistIcon(*info, item, false);

   if (false == getRouteSpecificIcon(item, info))
   {
      bool isRoadIconAvail = false;
      if (0 < info->m_roadNumberImages.size())
      {
         navmiddleware::Image roadIcon = info->m_roadNumberImages[0];
         if (Image::TYPE__UNDEFINED != roadIcon.getType())
         {
            item.mRoadIconActiveBitmap = loadImage((const char*)roadIcon.getBlob(), roadIcon.getBlobSize(), true);
            isRoadIconAvail = true;
         }
      }
      if (false == isRoadIconAvail) //avoid showing icons from previous elements, if no current icon is available
      {
         ETG_TRACE_USR4(("RouteListHandler::getFullTurnlistTurnDataItem(), Icon not present"));
         item.mRoadIconActiveBitmap = loadImage(NULL, 0, false);
      }
   }
   //Update turn message and distance from car position to start of element
   getTurnlistMessageDistance(item, info);
}


bool RouteListHandler::getRouteSpecificIcon(ShowRouteListItemData& item, ::std::vector<navmiddleware::RouteListInfos::RouteListElementInfo>::const_iterator info)
{
   bool isSpecificIconAvail = false;
   if (info->m_routeInfoProperties.m_ferry != navmiddleware::RouteInfoProperties::STATUS__NOT_PRESENT)
   {
      getSpecificTurnListIconUtil(item, TURNLIST__ICON_FERRY);
      isSpecificIconAvail = true;
   }
   else if (info->m_routeInfoProperties.m_autoTrain != navmiddleware::RouteInfoProperties::STATUS__NOT_PRESENT)
   {
      getSpecificTurnListIconUtil(item, TURNLIST__ICON_AUTOTRAIN);
      isSpecificIconAvail = true;
   }
   else if (info->m_routeInfoProperties.m_seasonalClosed != navmiddleware::RouteInfoProperties::STATUS__NOT_PRESENT)
   {
      getSpecificTurnListIconUtil(item, TURNLIST__ICON_SEASON_RESTRICTED);
      isSpecificIconAvail = true;
   }
   else if (info->m_routeInfoProperties.m_timeRestrictedRoad != navmiddleware::RouteInfoProperties::STATUS__NOT_PRESENT)
   {
      getSpecificTurnListIconUtil(item, TURNLIST__ICON_TIME_RESTRICTED);
      isSpecificIconAvail = true;
   }
   else if (info->m_routeInfoProperties.m_tollRoad != navmiddleware::RouteInfoProperties::STATUS__NOT_PRESENT)
   {
      getSpecificTurnListIconUtil(item, TURNLIST__ICON_TOLL_ROAD);
      isSpecificIconAvail = true;
   }
   else
   {
      // do nothing
   }

   ETG_TRACE_USR4(("RouteListHandler::getRouteSpecificIcon m_ferry %d, m_autoTrain %d, m_freeway %d, m_tollroad %d, m_seasonalClosed %d, m_timeRestrictedRoad %d, Road Info %s"
                   , info->m_routeInfoProperties.m_ferry
                   , info->m_routeInfoProperties.m_autoTrain
                   , info->m_routeInfoProperties.m_freeway
                   , info->m_routeInfoProperties.m_tollRoad
                   , info->m_routeInfoProperties.m_seasonalClosed
                   , info->m_routeInfoProperties.m_timeRestrictedRoad
                   , info->m_roadInfo.c_str()));
   return isSpecificIconAvail;
}


void RouteListHandler::getTurnlistMessageDistance(ShowRouteListItemData& item, ::std::vector<navmiddleware::RouteListInfos::RouteListElementInfo>::const_iterator info)
{
   ETG_TRACE_USR4(("RouteListHandler::getTurnlistMessageDistance()"));

   const Candera::String UNKNOWNSTREET = LANGUAGE_STRING(TextId_0x1418, "Unknown street name");
   const Candera::String EXITNUMBER = LANGUAGE_STRING(TEXTID_UNAVAILABLE, "Exit");
   std::string turnMessage;

   //Update turn message with road name
   //When road name is empty, show UNKNOWN STREET NAME
   turnMessage = (info->m_roadInfo.empty()) ? UNKNOWNSTREET.GetCString() : info->m_roadInfo;
   ETG_TRACE_USR4(("RouteListHandler::getTurnlistMessageDistance(), Road name : %s", turnMessage.c_str()));

   const ::std::vector<RouteListInfos::RouteListElementInfo::SignPostInfo> signPostInfos = info->getSignPostInfoList();
   for (int index = 0; index < signPostInfos.size(); index++)
   {
      std::string exitNumber = signPostInfos[index].m_exitNumber;
      if (0 < exitNumber.size())
      {
         ETG_TRACE_USR4(("RouteListHandler::getTurnlistMessageDistance(), Exit number : %s", exitNumber.c_str()));
         turnMessage = turnMessage + ", " + EXITNUMBER.GetCString() + " " + exitNumber;
         break;
      }
   }
   item.mRouteListItem = turnMessage.c_str();
}


void RouteListHandler::recalculateWithBlockedRoute()
{
   ETG_TRACE_USR4(("RouteListHandler::recalculateWithBlockedRoute(), First selection index : %d, Last selection index : %d", _firstSelectionIndex, _lastSelectionIndex));

   _listSelectionIndex = false;
   _resetSelectionValue = true;

   const navmiddleware::RouteListInfos& routeListInfos = _navMiddleware.getRouteListInfos();
   if (routeListInfos.getAvoidElementsStatus() == ::navmiddleware::RouteListInfos::AVOID_ELEMENTS_STATUS_INACTIVE)
   {
      _navMiddleware.avoidRouteListElements(routeListInfos.getIndexAlternative(), _firstSelectionIndex, _lastSelectionIndex);
      _isblockrouteListSelectionActive = false;
      _recalculateRouteSelected = true;

      // "Block route list" button is disabled until there is update for "Avoid element status"
      if (true == (*_guidanceBlockUnblockButton).mBlockUnblockEnable)
      {
         (*_guidanceBlockUnblockButton).mBlockUnblockEnable = false;
         _guidanceBlockUnblockButton.MarkItemModified(ItemKey::GuidanceBlockUnblockButton::BlockUnblockEnableItem);
         _guidanceBlockUnblockButton.SendUpdate();
      }
   }

   POST_MSG((COURIER_MESSAGE_NEW(ShowMapActiveRGReqMsg)()));
   return;
}


void RouteListHandler::onBlockRouteListSelection(unsigned int listIndex)
{
   ETG_TRACE_USR4(("RouteListHandler::onBlockRouteListSelection(), Selected list index : %d", listIndex));

   _isblockrouteListSelectionActive = true;

   if (true == _resetSelectionValue)
   {
      _resetSelectionValue = false;
      enableRouteCalcButton(false);
   }

   if ((false == _listSelectionIndex) && (0 < _routeListSize))
   {
      _firstSelectionIndex = getNavmidIndexInRouteList(listIndex);
      _lastSelectionIndex = _firstSelectionIndex;
      _listSelectionIndex = true;

      ETG_TRACE_USR4(("RouteListHandler::onBlockRouteListSelection(), Blocked index (%d to %d)", _firstSelectionIndex, _lastSelectionIndex));
      tSharedPtrDataProvider dataProvider = getBlockRouteListDataProvider();
      sendRouteListData(dataProvider);
   }

   else
   {
      _lastSelectionIndex = getNavmidIndexInRouteList(listIndex);
      _listSelectionIndex = false;
      _resetSelectionValue = true;

      if (_firstSelectionIndex > _lastSelectionIndex)
      {
         unsigned int temp = _firstSelectionIndex;
         _firstSelectionIndex = _lastSelectionIndex;
         _lastSelectionIndex = temp;
      }
      ETG_TRACE_USR4(("RouteListHandler::onBlockRouteListSelection(), Blocked index (%d to %d)", _firstSelectionIndex, _lastSelectionIndex));
      tSharedPtrDataProvider dataProvider = getBlockRouteListDataProvider();
      sendRouteListData(dataProvider);
      enableRouteCalcButton(true);
   }
}


int RouteListHandler::getNavmidIndexInRouteList(int hmiIndex)
{
   ETG_TRACE_USR4(("RouteListHandler::getNavmidIndexInRouteList(), Hmi Index : %d", hmiIndex));

   const navmiddleware::RouteListInfos& routeListInfos = _navMiddleware.getRouteListInfos();
   const std::vector< ::navmiddleware::RouteListInfos::RouteListElementInfo >& routeListElementInfos = routeListInfos.getRouteListElementInfos();

   int relativeIndex = hmiIndex - _listStartIndex;
   int navmidIndex = routeListElementInfos[relativeIndex].m_indexInRouteList;

   ETG_TRACE_USR4(("RouteListHandler::getNavmidIndexInRouteList(), Navmid Index : %d", navmidIndex));
   return navmidIndex;
}


bool RouteListHandler::onCourierMessage(const ButtonReactionMsg& oMsg)
{
   bool isMessageConsumed = false;
   if (oMsg.GetEnReaction() == enRelease)
   {
      const Courier::Identifier senderInfo = oMsg.GetSender();
      const Courier::ViewId sceneName      = oMsg.GetView();

      if (BLOCK_ROUTE_LIST_SCENENAME == sceneName)
      {
         ETG_TRACE_USR4(("RouteListHandler::onCourierMessage(ButtonReactionMsg), Scene name : %s", sceneName.CStr()));
         if (IdClickRecalculateButton == senderInfo)
         {
            recalculateWithBlockedRoute();
            isMessageConsumed = true;
         }
         else
         {
            ListProviderEventInfo info;
            ListProviderEventInfo::GetItemIdentifierInfo(oMsg.GetSender(), info);
            unsigned int index = (unsigned int)info.getHdlRow();
            onBlockRouteListSelection(index);
            isMessageConsumed = true;
         }
      }   // End of selection in Block route list

      else if (senderInfo == IdClick_BlockUnBlock_Button)   // Click on Block route list button
      {
         ETG_TRACE_USR4(("RouteListHandler::onCourierMessage(ButtonReactionMsg), Block/Unblock Route List button press"));

         // If route recalculation due to blocking is in progress, then button press on "Block route list" is not handled
         if (false == _recalculateRouteSelected)
         {
            // If route entries were previously blocked, then unblock it
            if (RouteListInfos::AVOID_ELEMENTS_STATUS_ACTIVE == _avoidElementStatus)
            {
               ETG_TRACE_USR4(("RouteListHandler::onCourierMessage(ButtonReactionMsg), Cancel avoid route list elements"));
               _navMiddleware.cancelAvoidRouteListElements();
               _avoidElementStatus = RouteListInfos::AVOID_ELEMENTS_STATUS_INACTIVE;
               unsigned int indexAlternative = _infoStore.getUserSelectedRouteAlternative();
               _navMiddleware.stopRouteListInfo(indexAlternative);

               POST_MSG((COURIER_MESSAGE_NEW(ShowMapActiveRGReqMsg)()));
               ShowBlockUnblockTextReqMsg(true);
            }
            else
            {
               POST_MSG((COURIER_MESSAGE_NEW(ShowRouteBlockListReqMsg)()));
            }
         }
         isMessageConsumed = true;
      }
   }

   return isMessageConsumed;
}


bool RouteListHandler::ShowBlockUnblockTextReqMsg(bool isShowBlockText)
{
   ETG_TRACE_USR4(("RouteListHandler::ShowBlockUnblockTextReqMsg(), isShowBlockText : %d", isShowBlockText));

   const Candera::String BLOCK_BUTTON_TEXT = LANGUAGE_STRING(TextId_0x063C, "Block route list");
   const Candera::String UNBLOCK_BUTTON_TEXT = LANGUAGE_STRING(TextId_0x0AE0, "Unblock route list");

   if (true == isShowBlockText)
   {
      (*_guidanceBlockUnblockButton).mBlockUnblockNormalIcon = ImageLoader::getAssetBitmapImage(ROUTELIST_BLOCKED_ICON_NORMAL);
      (*_guidanceBlockUnblockButton).mBlockUnblockPressedIcon = ImageLoader::getAssetBitmapImage(ROUTELIST_UNBLOCKED_ICON_TOUCHED);
      (*_guidanceBlockUnblockButton).mBlockUnblockText = BLOCK_BUTTON_TEXT;
   }
   else
   {
      (*_guidanceBlockUnblockButton).mBlockUnblockNormalIcon = ImageLoader::getAssetBitmapImage(ROUTELIST_UNBLOCKED_ICON_NORMAL);
      (*_guidanceBlockUnblockButton).mBlockUnblockPressedIcon = ImageLoader::getAssetBitmapImage(ROUTELIST_UNBLOCKED_ICON_TOUCHED);
      (*_guidanceBlockUnblockButton).mBlockUnblockText = UNBLOCK_BUTTON_TEXT;
   }
   _guidanceBlockUnblockButton.MarkAllItemsModified();
   _guidanceBlockUnblockButton.SendUpdate(true);

   return true;
}


void RouteListHandler::enableRouteCalcButton(bool isEnabled)
{
   ETG_TRACE_USR4(("RouteListHandler::enableRouteCalcButton(), isRouteCalcButtonEnabled : %d", isEnabled));

   if (true == isEnabled)
   {
      (*_blockRouteListItem).mRecalculateIcon = ImageLoader::getAssetBitmapImage(RECALCULATE_ROUTE_NORMAL_ICON);
      (*_blockRouteListItem).mRecalcEnableDisable = true;
   }
   else
   {
      (*_blockRouteListItem).mRecalculateIcon = ImageLoader::getAssetBitmapImage(RECALCULATE_ROUTE_DISABLED_ICON);
      (*_blockRouteListItem).mRecalcEnableDisable = false;
   }
   _blockRouteListItem.MarkAllItemsModified();
   _blockRouteListItem.SendUpdate(true);
   return;
}


bool RouteListHandler::onPropertyUpdateHardRestrictionStatusChanged()
{
   bool isHardRestrictionPresent = _infoStore.getIsHardRestrictionPresent();
   ETG_TRACE_USR4(("RouteListHandler::onPropertyUpdateHardRestrictionStatusChanged(), isHardRestrictionPresent : %d", isHardRestrictionPresent));

   (*_guidanceBlockUnblockButton).mBlockUnblockEnable = !isHardRestrictionPresent;
   _guidanceBlockUnblockButton.MarkItemModified(ItemKey::GuidanceBlockUnblockButton::BlockUnblockEnableItem);
   _guidanceBlockUnblockButton.SendUpdate();
   return true;
}
