/**************************************************************************************
* @file         : DestinationAddressInputHandler.cpp
* @author       : ECG5-Prieethi Narayanaswamy
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) 2018-2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include "string"

#include "util/StrUtf8.h"
#include "AppHmi_NavigationStateMachine.h"
#include "MapPickingOptionHandler.h"
#include "hmi_trace_if.h"
#include "RouteDataUtils.h"
#include "CgiExtensions/ImageLoader.h"


#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/MapPickingOptionHandler.cpp.trc.h"
#endif


#ifdef HALL_TO_MDW_COM

static const Courier::Identifier  IdClickReplaceDestination = Courier::Identifier("Layer_ReplaceDestination/ButtonWidget");
static const Courier::Identifier  IdClickAddDestination = Courier::Identifier("Layer_AddDestination/ButtonWidget");
static const Courier::Identifier  IdClickAddAsWaypoint = Courier::Identifier("Layer_AddAsWaypoint/ButtonWidget");
static const int ADD_AS_WAYPOINT_CLICK = 1;
static const int TRIP_COUNT_ZERO = 0;
static const int TRIP_COUNT_ONE = 1;
static const int TRIP_COUNT_TWO = 2;
static const Candera::UInt8 MAX_LIST_SIZE_WITH_GUIDANCE = 5;
static const Candera::UInt8 MAX_LIST_SIZE_WITHOUT_GUIDANCE = 4;
static const int MAX_WAYPOINT = 9;
static const Candera::String GUIDANCE_COORDINATES_ICONS[MAX_LIST_SIZE_WITH_GUIDANCE] = { Candera::String("AppHmi_NavigationModule#Images#N_Map_ActiveGuidance#Icon_ReplaceDestination_normal"),
                                                                                         Candera::String("AppHmi_NavigationModule#Images#N_Map_ActiveGuidance#Icon_Waypoints_normal"),
                                                                                         Candera::String("AppHmi_NavigationModule#Images#N_Map_ActiveGuidance#Icon_DestinationInfo_normal"),
                                                                                         Candera::String("AppHmi_NavigationModule#Images#N_Map_ActiveGuidance#Icon_POIsAround_normal"),
                                                                                         Candera::String("AppHmi_NavigationModule#Images#N_Map_ActiveGuidance#Icon_ReplaceDestination_normal")
                                                                                       };

static const Candera::String GUIDANCE_COORDINATES_ICONS_TOUCHED[MAX_LIST_SIZE_WITH_GUIDANCE] = { Candera::String("AppHmi_NavigationModule#Images#N_Map_ActiveGuidance#Icon_ReplaceDestination_touched"),
                                                                                                 Candera::String("AppHmi_NavigationModule#Images#N_Map_ActiveGuidance#Icon_Waypoints_touched"),
                                                                                                 Candera::String("AppHmi_NavigationModule#Images#N_Map_ActiveGuidance#Icon_DestinationInfo_touched"),
                                                                                                 Candera::String("AppHmi_NavigationModule#Images#N_Map_ActiveGuidance#Icon_POIsAround_touched"),
                                                                                                 Candera::String("AppHmi_NavigationModule#Images#N_Map_ActiveGuidance#Icon_ReplaceDestination_touched")
                                                                                               };


MapPickingOptionHandler::MapPickingOptionHandler(navmiddleware::NavMiddleware& _navMiddleware, InfoStore& _infoStore)
   : HMIModelBase(_navMiddleware, _infoStore)
   , _isAddAsWaypointOptionEnabled(false)
   , _isReplaceDestination(false)
   , _mapPickListSize(MAX_LIST_SIZE_WITHOUT_GUIDANCE)
{
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_NAV_COORDINATES, this);
}


MapPickingOptionHandler::~MapPickingOptionHandler()
{
}


void MapPickingOptionHandler::initialize()
{
   _RouteGuidanceCoordinates.reserve(MAX_LIST_SIZE_WITH_GUIDANCE);
   _RouteGuidanceCoordinates.push_back(RouteGuidanceCoordinates(LANGUAGE_STRING(TextId_0x0222, "Replace destination"), IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_CHOICE_ONE_IMAGE, GUIDANCE_COORDINATES_REPLACEDEST));
   _RouteGuidanceCoordinates.push_back(RouteGuidanceCoordinates(LANGUAGE_STRING(TextId_0x0F31, "Add as waypoint"), IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_CHOICE_ONE_IMAGE, GUIDANCE_COORDINATES_WAYPOINT));
   _RouteGuidanceCoordinates.push_back(RouteGuidanceCoordinates(LANGUAGE_STRING(TextId_0x0220, "Details"), IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_CHOICE_ONE_IMAGE, GUIDANCE_COORDINATES_DETAILS));
   _RouteGuidanceCoordinates.push_back(RouteGuidanceCoordinates(LANGUAGE_STRING(TextId_0x0221, "POIs around"), IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_CHOICE_ONE_IMAGE, GUIDANCE_COORDINATES_POISAROUND));
   _RouteGuidanceCoordinates.push_back(RouteGuidanceCoordinates(LANGUAGE_STRING(TextId_0x141F, "Add as destination"), IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_CHOICE_ONE_IMAGE, GUIDANCE_COORDINATES_ADDDEST));
   _navMiddleware.registerLocationPropertyUpdateCallback(*this);
   _navMiddleware.registerGuidancePropertyUpdateCallback(*this);
   _navMiddleware.registerRoutePropertyUpdateCallback(*this);
}


void MapPickingOptionHandler::deinitialize()
{
   _navMiddleware.unregisterLocationPropertyUpdateCallback(*this);
}


void MapPickingOptionHandler::onPropertyUpdateGuidanceStateChanged()
{
   ETG_TRACE_USR4(("MapPickingOptionHandler::onPropertyUpdateGuidanceStateChanged()"));

   // After guidance is stopped, if "Replace destination" is selected by the user, then guidance is started to new destination
   bool guidanceActive = _navMiddleware.isGuidanceActive();
   if ((false == guidanceActive) && (true == _isReplaceDestination))
   {
      _isReplaceDestination = false;
      EXT_bIsCarModeWarningApplicable = false;
   }
}


tSharedPtrDataProvider MapPickingOptionHandler::getListDataProvider(const ListDateProviderReqMsg& oMsg)
{
   ETG_TRACE_USR4(("MapPickingOptionHandler::getListDataProvider(List ID %d)", oMsg.GetListId()));
   DataItemContext  listItemName = "MapPicking_ListOptions";
   Candera::UInt32 listID = oMsg.GetListId();
   ListDataProviderBuilder listBuilder(listID);

   RouteGuidancetextImageUpdationData item;
   Candera::UInt8 idx = 0;
   for (std::vector<RouteGuidanceCoordinates>::iterator itemNumber = _RouteGuidanceCoordinates.begin(); itemNumber < _RouteGuidanceCoordinates.end() && idx < _mapPickListSize ; ++itemNumber, idx++)
   {
      if ((itemNumber->_order == 1) && (_isAddAsWaypointOptionEnabled == false))
      {
         continue;
      }
      if (itemNumber->_listButtonType == LIST_BUTTON_TYPE_CHOICE_ONE_IMAGE)
      {
         item.mText = itemNumber->_text;
         item.mIcon_Normal = ImageLoader::getAssetBitmapImage(GUIDANCE_COORDINATES_ICONS[idx].GetCString());
         item.mIcon_Pressed = ImageLoader::getAssetBitmapImage(GUIDANCE_COORDINATES_ICONS_TOUCHED[idx].GetCString());
         listBuilder.AddItem(
            itemNumber->_order,
            0UL,  //    - unused
            listItemName)  //0 - Name(Id) of the button
         .AddDataBindingUpdater<RouteGuidancetextImageUpdationDataBindingSource>(item);
      }
   }

   return listBuilder.CreateDataProvider();
}


bool MapPickingOptionHandler::onCourierMessage(const SetWayPointEnableReqMsg& oMsg)
{
   ETG_TRACE_USR4(("MapPickingOptionHandler::onCourierMessage(SetWayPointEnableReqMsg())"));
   _isAddAsWaypointOptionEnabled = oMsg.GetWaypointoptionactivated();
   _RouteGuidanceCoordinates.at(GUIDANCE_COORDINATES_REPLACEDEST)._text = (_isAddAsWaypointOptionEnabled == true) ? LANGUAGE_STRING(TextId_0x0222, "Replace destination") :
         LANGUAGE_STRING(TextId_0x141F, "Add as destination");
   //Map pick List size decided based on the Add as Waypoint button appearing or not
   _mapPickListSize = MAX_LIST_SIZE_WITHOUT_GUIDANCE;
   return true;
}


bool MapPickingOptionHandler::onCourierMessage(const ButtonListItemUpdMsg& oMsg)
{
   bool isMsgConsumed = false;
   Candera::UInt32 listId = oMsg.GetListId();
   unsigned int rowIdx = (unsigned int)oMsg.GetHdl();
   const enReaction  reaction = oMsg.GetReaction();
   if (enRelease == reaction)
   {
      if (LIST_ID_NAV_COORDINATES == listId)
      {
         ETG_TRACE_USR4(("MapPickingOptionHandler::onCourierMessage(ButtonListItemUpdMsg), List Id : %d, Row Id : %d", listId, rowIdx));
         if (ADD_AS_WAYPOINT_CLICK == rowIdx)
         {
            unsigned int waypointlistSize = _infoStore.getWaypointListSize();
            if (waypointlistSize < MAX_WAYPOINT)
            {
               POST_MSG((COURIER_MESSAGE_NEW(WaypointInsertReqMsg)()));
               POST_MSG((COURIER_MESSAGE_NEW(ShowMapActiveRGReqMsg)()));
            }
            else
            {
               POST_MSG((COURIER_MESSAGE_NEW(ShowPopupTextMsg)(POPUP_MAX_WAYPOINTS_REACHED)));
            }
         }
         else if (0 == rowIdx) // Corresponds to Replace destination
         {
            if (_navMiddleware.isGuidanceActive() == true)   // Replace destination
            {
               // Request is first sent to stop active RG.
               // After getting property update GUIDANCE_STATE_CHANGED for guidance stopped, request is sent to start route calculation for new destination
               _isReplaceDestination = true;
               POST_MSG((COURIER_MESSAGE_NEW(CancelRouteGuidanceReqMsg)()));
               EXT_bIsGuidanceActive = false;
               POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(VehicleProfileChangedResMsg)()));
            }
            else   // Add as destination
            {
               VEHICLE_PROFILE_CHANGED = true;
               recalculateRouteGuidance(_navMiddleware, _infoStore, false);
            }
         }
         else if (4 == rowIdx) //Corresponds to Add Destination once an initial destination is set.
         {
            ETG_TRACE_USR4(("MapPickingOptionHandler::AddDestination from Coordinates List is clicked"));
            static const int ADD_DESTINATION_POPUP = 11;
            const Candera::String DATA_CONTEXT__TEXT_DESTPOPUP_LINE1 = "Maximum number of destination";
            const Candera::String DATA_CONTEXT__TEXT_DESTPOPUP_LINE2 = "entries has been reached";
            //Maximum three destinations are supported
            if ((_infoStore.tripCount < TRIP_COUNT_TWO))
            {
               ETG_TRACE_USR4(("MapPickingOptionHandler::AddDestination from Coordinates entered If"));
               POST_MSG((COURIER_MESSAGE_NEW(UpdateDeletePopupTextReqMsg)(ADD_DESTINATION_POPUP)));
               _infoStore.setIsTourEnabled(true);
               _infoStore.setIsAddDestinationSelected(true);
               _infoStore.tripCount++;
            }
            else
            {
               ETG_TRACE_USR4(("MapPickingOptionHandler::AddDestination from Coordinates entered Else"));
               _infoStore.setIsTourEnabled(false);
               //Show informative popup saying already maximum number of destination has been added
               (*_PopupText).mTextLine1 = DATA_CONTEXT__TEXT_DESTPOPUP_LINE1;
               (*_PopupText).mTextLine2 = DATA_CONTEXT__TEXT_DESTPOPUP_LINE2;
               _PopupText.MarkAllItemsModified();
               _PopupText.SendUpdate(true);
               POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_MultiLine_Text"))));
            }
         }
         else
         {
            POST_MSG((COURIER_MESSAGE_NEW(PickInMapOptionsUpdateMsg)(rowIdx)));
            isMsgConsumed = true;
         }
      }
   }

   return isMsgConsumed;
}


bool MapPickingOptionHandler::onCourierMessage(const ButtonReactionMsg& oMsg)
{
   static const int ADD_DESTINATION_POPUP = 11;
   bool isMsgConsumed = false;
   const Courier::Identifier senderInfo = oMsg.GetSender();
   const Candera::String DATA_CONTEXT__TEXT_DESTPOPUP_LINE1 = "Maximum number of destination";
   const Candera::String DATA_CONTEXT__TEXT_DESTPOPUP_LINE2 = "entries has been reached";

   if (oMsg.GetEnReaction() == enRelease)
   {
      if (senderInfo == IdClickReplaceDestination)
      {
         ETG_TRACE_USR4(("MapPickingOptionHandler::onCourierMessage(ButtonReactionMsg), Replace destination"));
         if (_navMiddleware.isGuidanceActive() == true)
         {
            // Request is first sent to stop active RG.
            // After getting property update GUIDANCE_STATE_CHANGED for guidance stopped, request is sent to start route calculation for new destination
            _isReplaceDestination = true;
            POST_MSG((COURIER_MESSAGE_NEW(CancelRouteGuidanceReqMsg)()));
            EXT_bIsGuidanceActive = false;
            POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(VehicleProfileChangedResMsg)()));
         }
      }
      if (senderInfo == IdClickAddDestination)
      {
         ETG_TRACE_USR4(("MapPickingOptionHandler::onCourierMessage(ButtonReactionMsg), Add destination"));
         //Maximum three destinations are supported
         if ((_infoStore.tripCount < TRIP_COUNT_TWO))
         {
            POST_MSG((COURIER_MESSAGE_NEW(UpdateDeletePopupTextReqMsg)(ADD_DESTINATION_POPUP)));
            _infoStore.setIsTourEnabled(true);
            _infoStore.setIsAddDestinationSelected(true);
            _infoStore.tripCount++;
         }
         else
         {
            _infoStore.setIsTourEnabled(false);
            //Show informative popup saying already maximum number of destination has been added
            (*_PopupText).mTextLine1 = DATA_CONTEXT__TEXT_DESTPOPUP_LINE1;
            (*_PopupText).mTextLine2 = DATA_CONTEXT__TEXT_DESTPOPUP_LINE2;
            _PopupText.MarkAllItemsModified();
            _PopupText.SendUpdate(true);
            POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_MultiLine_Text"))));
         }
      }

      if (senderInfo == IdClickAddAsWaypoint)
      {
         ETG_TRACE_USR4(("MapPickingOptionHandler::onCourierMessage(ButtonReactionMsg), Add waypoint"));

         unsigned int waypointlistSize = _infoStore.getWaypointListSize();
         if (waypointlistSize < MAX_WAYPOINT)
         {
            POST_MSG((COURIER_MESSAGE_NEW(WaypointInsertReqMsg)()));
            POST_MSG((COURIER_MESSAGE_NEW(ShowMapActiveRGReqMsg)()));
         }
         else
         {
            POST_MSG((COURIER_MESSAGE_NEW(ShowPopupTextMsg)(POPUP_MAX_WAYPOINTS_REACHED)));
         }
      }
   }
   return isMsgConsumed;
}


bool MapPickingOptionHandler::onCourierMessage(const ResetTripCountReqMsg& oMsg)
{
   ETG_TRACE_USR4(("MapPickingOptionHandler::onCourierMessage(ResetTripCountReqMsg())"));

   //Maintaining trip count value on selection of cancel button on Add destination popup
   _infoStore.tripCount--;

   return true;
}


#endif // HALL_TO_MDW_COM
