/**************************************************************************************
* @file         : CalculateRouteHandler.h
* @author       : ECG5-Naveen Thangamalayan
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) 2018-2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#ifndef CALCULATE_ROUTE_HANDLER_H
#define CALCULATE_ROUTE_HANDLER_H

#include "Common/Util/HMIModelIncludes.h"
#include "Route/RouteDataUtils.h"
#include "Candera/System/MemoryManagement/SharedPointer.h"


#ifdef HALL_TO_MDW_COM
/**
* Main hmi model component
*/
class CalculateRouteHandler : public HMIModelBase
{
   public:
      CalculateRouteHandler(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore);
      virtual ~CalculateRouteHandler();

      void initialize();
      void deinitialize();

      // Incoming property updates from middleware
      PROPERTY_UPDATE_MIDDLEWARE_BEGIN()
      ON_PROPERTY_MIDDLEWARE_UPDATE(navmiddleware::ROUTE_CALCULATION_STARTED, onPropertyUpdateRouteCalculationStarted)
      ON_PROPERTY_MIDDLEWARE_UPDATE(navmiddleware::ROUTE_CALCULATION_STOPPED, onPropertyUpdateRouteCalculationStopped)
      ON_PROPERTY_MIDDLEWARE_UPDATE(navmiddleware::GUIDANCE_STATE_CHANGED, onPropertyUpdateGuidanceStateChanged)
      ON_PROPERTY_MIDDLEWARE_UPDATE(navmiddleware::GUIDANCE_LAST_GUIDED_ITEM_DETAILS_AVAILABLE, onPropertyUpdateGuidanceLastGuidedItemDetailsAvailable)
      ON_PROPERTY_MIDDLEWARE_UPDATE(navmiddleware::ROUTE_ARRIVALINFOS_ALTERNATIVES_CHANGED, onPropertyUpdateArrivalInfosAlternativesChanged)
      ON_PROPERTY_MIDDLEWARE_UPDATE(navmiddleware::ROUTE_DESTINATIONS_CHANGED, onPropertyUpdateRouteDestinationsChanged)
      ON_PROPERTY_MIDDLEWARE_UPDATE(navmiddleware::GUIDANCE_HORIZONEVENT_CHANGED, onPropertyUpdateGuidanceHorizonEventChanged)
      ON_PROPERTY_MIDDLEWARE_UPDATE(navmiddleware::ROUTE_CALCULATION_INFO_CHANGED, onPropertyUpdateRouteCalcInfoChanged)
      ON_PROPERTY_MIDDLEWARE_UPDATE(navmiddleware::ROUTE_SOFTPHYSICAL_RESTRICTION_VIOLATION_STATUS_CHANGED, onPropertyUpdateSoftPhysicalRestrictionChanged)
      ON_PROPERTY_MIDDLEWARE_UPDATE(navmiddleware::ROUTE_HARDPHYSICAL_RESTRICTION_VIOLATION_STATUS_CHANGED, onPropertyUpdateHardRestrictionChanged)
      ON_PROPERTY_MIDDLEWARE_UPDATE(navmiddleware::ROUTE_HARDGENERAL_RESTRICTION_VIOLATION_STATUS_CHANGED, onPropertyUpdateHardRestrictionChanged)
      PROPERTY_UPDATE_MIDDLEWARE_END()

      void onPropertyUpdateRouteCalculationStarted();
      void onPropertyUpdateRouteCalculationStopped();
      void onPropertyUpdateGuidanceStateChanged();
      void onPropertyUpdateGuidanceLastGuidedItemDetailsAvailable();
      void onPropertyUpdateArrivalInfosAlternativesChanged();
      void onPropertyUpdateRouteDestinationsChanged();
      void onPropertyUpdateGuidanceHorizonEventChanged();
      void onPropertyUpdateRouteCalcInfoChanged();
      void onPropertyUpdateSoftPhysicalRestrictionChanged();
      void onPropertyUpdateHardRestrictionChanged();
      bool onPropertyUpdateResetFactoryChanged();
      bool onPropertyUpdateResetFactoryProductionChanged();

      // Incoming property updates from info store
      PROPERTY_UPDATE_INFOSTORE_BEGIN()
      ON_PROPERTY_INFOSTORE_UPDATE(NAVI_ESTIMATED_TIME_MODE_CHANGED, onPropertyUpdateEstimatedTimeModeChanged)
      ON_PROPERTY_INFOSTORE_UPDATE(NAVI_DISCLAIMER_STATUS_CHANGED, onPropertyUpdateDisclaimerStatusChanged)
      ON_PROPERTY_INFOSTORE_UPDATE(NAVI_HMI_APP_STATE_CHANGED, onPropertyUpdateHmiAppStateChanged)
      ON_PROPERTY_INFOSTORE_UPDATE(NAVI_RESET_FACTORY_CHANGED, onPropertyUpdateResetFactoryChanged);
      ON_PROPERTY_INFOSTORE_UPDATE(NAVI_RESET_FACTORY_PRODUCTION_CHANGED, onPropertyUpdateResetFactoryProductionChanged);
      PROPERTY_UPDATE_INFOSTORE_END()

      bool onPropertyUpdateEstimatedTimeModeChanged();
      bool onPropertyUpdateDisclaimerStatusChanged();
      bool onPropertyUpdateHmiAppStateChanged();

      // Incoming courier messages from HMI
      COURIER_MSG_MAP_BEGIN(0)
      ON_COURIER_MESSAGE(ButtonReactionMsg)
      ON_COURIER_MESSAGE(StartRouteGuidanceFromSpeechReqMsg)
      ON_COURIER_MESSAGE(StartRouteGuidanceReqMsg)
      ON_COURIER_MESSAGE(CancelRouteGuidanceReqMsg)
      ON_COURIER_MESSAGE(ToggleRouteOverViewBtnMsg)
      ON_COURIER_MESSAGE(HighlightActiveRouteReqMsg)
      ON_COURIER_MESSAGE(RouteStartGuidanceReqMsg)
      ON_COURIER_MESSAGE(AvoidTollRoadReqMsg)
      ON_COURIER_MESSAGE(CancelMultipleRouteReqMsg)
      ON_COURIER_MESSAGE(RetriggerLastGuidedRouteReqMsg)
      ON_COURIER_MESSAGE(ShowRouteCalcPopupReqMsg)
      ON_COURIER_MESSAGE(SoftRestrictionMsg)
      ON_COURIER_MESSAGE(HardRestrictionConfirmMultipleReqMsg)
      COURIER_MSG_MAP_END()

      bool onCourierMessage(const StartRouteGuidanceFromSpeechReqMsg& oMsg);
      bool onCourierMessage(const StartRouteGuidanceReqMsg& oMsg);
      bool onCourierMessage(const CancelRouteGuidanceReqMsg& oMsg);
      bool onCourierMessage(const ToggleRouteOverViewBtnMsg& oMsg);
      bool onCourierMessage(const HighlightActiveRouteReqMsg& oMsg);
      bool onCourierMessage(const RouteStartGuidanceReqMsg& oMsg);
      bool onCourierMessage(const AvoidTollRoadReqMsg& oMsg);
      bool onCourierMessage(const CancelMultipleRouteReqMsg& oMsg);
      bool onCourierMessage(const ButtonReactionMsg& oMsg);
      bool onCourierMessage(const RetriggerLastGuidedRouteReqMsg& oMsg);
      bool onCourierMessage(const ShowRouteCalcPopupReqMsg& oMsg);
      bool onCourierMessage(const SoftRestrictionMsg& oMsg);
      bool onCourierMessage(const HardRestrictionConfirmMultipleReqMsg& oMsg);
      void loadLastDestinationOnStartup();

   private:
      void initializeMoreRoute();
      void activeMoreRouteOption(enRouteTypeOptions activeRouteCriterion);
      void handleTollRoadOnRoute();
      void showTollRoadPopup();
      void startRouteGuidance(navmiddleware::settings::MoreRoutesTypeValue routeType);
      void startGuidanceWithRouteCriterion();
      void bindToMultipleRouteCriteria();
      void writeVehicleProfileGuidanceState(bool isGuidanceActive);
      int getNoOfLines(const std::string& filePath);
      std::string getFormattedCurrentDateTime();
      bool replaceLineAtOffset(const std::string& filePath, std::streampos offset, const std::string& newContent);
      void performResetFactoryOperations();

      bool getMultipleRouteGuidanceBtnState() const
      {
         return (*_multipleRouteGuidanceBtn).mButtonIsActive;
      }
      void setMultipleRouteGuidanceBtnState(bool isEnabled);
      void setMultipleRouteOptionsState(bool isEnabled);
      void setRouteTabState(bool isEnabled);
      void resetMultipleRouteButtonConfiguration();
      void resetGuidanceData();
      void sendAlternativeRouteItemsUpdate();

      void stopPlannedOrGuidedRoute();
      void cancelRouteCalculationAndGuidance(); // To cancel the existing route calculation & guidance. It is called on CancelRouteGuidanceReqMsg.
      void cancelMultipleRouteCalculation();

      void displayCalculateRoutePopup(bool displayPopup); // Set to true to display popup and false to hide it
      void displayHardRestrictionsMultipleInfoPopup(bool isMultipleRoutesRestricted = false);
      void displaySoftRestrictionsPopup();
      void setSoftRestrictionStatus(bool isPresent);
      void setHardRestrictionStatus(bool isPresent);
      std::string getVehicleRestrictionsInfoText(navmiddleware::VehicleRestrictionsInfo& vehicleRestrictionsInfo, int vehicleRestrictionsType);
      enRouteTypeOptions convertRouteCriterionFromNavmidToHmi(::navmiddleware::settings::RouteCriterion navmidRouteCriterion) const;
      ::navmiddleware::settings::RouteCriterion convertRouteCriterionFromHmiToNavmid(enRouteTypeOptions hmiRouteCriterion) const;

      DataBindingItem<MultiLinePopuptextDataBindingSource>         _PopupText;
      DataBindingItem<SwitchindexDataBindingSource>                _switchIndexData;                    // DataBinding used to show Destinations scene with 3 or 4 tabs
      DataBindingItem<RouteSwitchbtnDataBindingSource>             _switchRouteIndex;                   // DataBinding used to show Compass/MapScale/StreetLabel or Fastest/Shortest/Economy in N_Map scene
      DataBindingItem<RouteOverviewBtnDataBindingSource>           _routeOverViewBtnState;
      DataBindingItem<AlternativeRouteItemsDataBindingSource>      _alternativeRouteItems;
      DataBindingItem<GuidanceBlockUnblockButtonDataBindingSource> _guidanceBlockUnblockButton;
      DataBindingItem<RouteInfoPopupTextDataBindingSource>         _routeInfoPopupText;                 // DataBinding used to update Toll Road/Border/CalcError text in Pfo_RouteInfo
      DataBindingItem<RouteRestrictionsPopupTextDataBindingSource> _routeRestrictionsPopupText;         // DataBinding used to update restrictions text in Pfo_LCVRestrictionInfo
      DataBindingItem<FourLinePopupTextDataBindingSource>          _vehicleRestrictionsInfoPopupText;
      DataBindingItem<SoftRestrictionDataBindingSource>            _softRestrictionIcon;
      DataBindingItem<GuidanceRouteSwitchbtnDataBindingSource>     _multipleRouteGuidanceBtn;
      DataBindingItem<HardRestrictionDataBindingSource>            _hardRestrictionData;

      bool _isCalculatingRoutePopUpVisible;
      bool _isGuidanceActive;
      bool _isGuidanceRetriggered;
      bool _isGuidanceTriggerFromGadget;
      bool _checkForTollRoadOnRoute;
      bool _isSoftRestrictionPopupTriggered;
      int _activeAlternativeIndex;   // Index of active multiple route
      ::navmiddleware::settings::RouteCriterion _navmidActiveRouteCriterion;
      std::vector< ButtonMoreRouteType > _buttonConfiguration;
};


#endif // HALL_TO_MDW_COM
#endif // CALCULATE_ROUTE_HANDLER_H
