/**************************************************************************************
* @file         : UserPOIListHandler.cpp
* @author       : ECH-Venkata Sairam Dubba
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (C) 2020 Robert Bosch GmbH
*                 (C) 2020 Robert Bosch Engineering and Business Solutions Limited
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include "UserPOIListHandler.h"
#include "DestinationDataUtils.h"	// For invoking getDirectionIcon()
#include "Common/Util/ImageUtils.h" // For invoking loadImage()
#include "string.h"
#include <vector>


#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/UserPOIListHandler.cpp.trc.h"
#endif

#if defined (HALL_TO_MDW_COM)

static const char* const NEARBY_LIST_ITEM = "BtnPoiMultiLineText";
static const char* const DATA_CONTEXT_EMPTY_LIST_ITEM = "NoDataListItem";
static const char* const DATA_CONTEXT_POI_LOADING_LIST_ITEM = "ListLoadingItem";
static const Candera::String POI_NO_LIST_ITEM = LANGUAGE_STRING(TextId_0x1459, "No data item");
static const Candera::String POI_LIST_LOADING = LANGUAGE_STRING(TextId_0x1458, "Loading...");
static const Candera::UInt8 POI_IN_MAP_COUNT = 4;  // Number of UPOI icons to be shown in map

UserPOIListHandler::UserPOIListHandler(NavMiddleware& navMiddleware, InfoStore& infoStore)
   : HMIModelBase(navMiddleware, infoStore)
   , _isListRequested(false)
   , _isListAvailable(false)
   , _isFreeTextSearch(false)
   , _listId(0)
   , _startIndexListRequest(0)
   , _startIndexVisibleArea(0)
   , _windowElementSize(0)
{
   //Register the ListID
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_NAV_DEST_NEARBY_LIST, this);
}


UserPOIListHandler::~UserPOIListHandler()
{
}


void UserPOIListHandler::initialize()
{
   _navMiddleware.registerLocationPropertyUpdateCallback(*this);
}


void UserPOIListHandler::deinitialize()
{
   _navMiddleware.unregisterLocationPropertyUpdateCallback(*this);
}


bool UserPOIListHandler::onCourierMessage(const ButtonReactionMsg& oMsg)
{
   bool bIsMsgConsumed = false;
   if (oMsg.GetEnReaction() == enRelease)
   {
      ListProviderEventInfo info;
      ListProviderEventInfo::GetItemIdentifierInfo(oMsg.GetSender(), info);

      unsigned int listId = info.getListId();
      if ((true == EXT_bIsUPOIContext) && (listId == LIST_ID_NAV_DEST_NEARBY_LIST))
      {
         ETG_TRACE_USR4(("UserPOIListHandler::onCourierMessage(ButtonReactionMsg(listID %d, row %d, column %d))", listId, info.getHdlRow(), info.getHdlCol()));
         _listId = 0;

         unsigned int listIndex = (unsigned int)info.getHdlRow();
         LocationInfos locationInfos;
         fetchLocationInfo(locationInfos);
         const ::std::vector<navmiddleware::LocationInfos::LocationInfo>& locationInfoVector = locationInfos.getLocationInfos();

         unsigned int relativeIndex = listIndex - _startIndexListRequest;
         if (relativeIndex < locationInfoVector.size())
         {
            navmiddleware::LocationInfos::LocationInfo locationInfo = locationInfoVector[relativeIndex];

            if (false == _isFreeTextSearch)
            {
               _navMiddleware.selectPoiEntry(listIndex);
            }
            else
            {
               _navMiddleware.setLocationWithFreeTextSearchResult(listIndex);
               _navMiddleware.leaveFreeTextSearchInput();
            }

            // Set Co-Ordinate Position of Selected POI on DetailedInfo Scene
            std::vector<PosWGS84<double> > positionVector;
            positionVector.push_back(PosWGS84<double>(locationInfo.m_coordinate.getValue().getLongitude(), locationInfo.m_coordinate.getValue().getLatitude()));
            _infoStore.setLatitude(locationInfo.m_coordinate.getValue().getLatitude());
            _infoStore.setLongitude(locationInfo.m_coordinate.getValue().getLongitude());
            InfoStoreBase::CoordinatesToBeShownInMap& coordinatesToBeShownInMap = _infoStore.getCoordinatesToBeShownInMap();
            coordinatesToBeShownInMap._coordinates = positionVector;

            // Set POI location name and address
            _infoStore.setAddressInfoDetailsName(locationInfo.m_elementName.c_str());

            LocationAttributeInfos attributeInfos = locationInfo.m_attributes;
            _infoStore.setAddressInfoDetailsAddress(getUPOIDescription(attributeInfos));

            if (locationInfo.m_attributes.getPhoneNumberInfo() != NULL)
            {
               _infoStore.setAddressInfoDetailsPhoneNumber(locationInfo.m_attributes.getPhoneNumberInfo()->m_content.c_str());
            }
            else
            {
               _infoStore.setAddressInfoDetailsPhoneNumber("");
            }

            // Courier msg has been posted to show POI detailed info with the detailed address details
            POST_MSG((COURIER_MESSAGE_NEW(ShowDetailedAddressMsg)()));
            POST_MSG((COURIER_MESSAGE_NEW(ShowFavEditIcon_Enable_Delete_Guidance_Buttons)(0, false, true, 2, false))); // To show fav icon and to disable delete buttons
            POST_MSG((COURIER_MESSAGE_NEW(UPOIListBtnSelectionMsg)()));
         }

         bIsMsgConsumed = true;
      } // End of check for POI list ID
   }   // End of check for release button reaction

   return bIsMsgConsumed;
}


bool UserPOIListHandler::onCourierMessage(const ListDateProviderReqMsg& oMsg)
{
   bool isMsgConsumed = false;
   unsigned int listId = oMsg.GetListId();
   if ((true == EXT_bIsUPOIContext) && (listId == LIST_ID_NAV_DEST_NEARBY_LIST))
   {
      isMsgConsumed = true;
      _isListRequested = true;
      _startIndexListRequest = oMsg.GetStartIndex();
      _windowElementSize = oMsg.GetWindowElementSize();
      _listId = listId;
      ETG_TRACE_USR4(("UserPOIListHandler::onCourierMessage(ListDateProviderReqMsg(_listId %d, _startIndexListRequest %d, _windowElementSize %d))",
                      _listId, _startIndexListRequest, _windowElementSize));
      updateList();
   }
   return isMsgConsumed;
}


void UserPOIListHandler::updateList()
{
   ETG_TRACE_USR4(("UserPOIListHandler::updateList(), Start index : %d, Window size : %d", _startIndexListRequest, _windowElementSize));

   if ((true == _isListRequested) && (true == _isListAvailable))
   {
      _isListRequested = false;
      if (false == _isFreeTextSearch)      // UPOI nearby list
      {
         _navMiddleware.requestPoiEntries(_startIndexListRequest, _windowElementSize);
      }
      else                                 // UPOI Free text search list
      {
         _navMiddleware.requestFreeTextSearchResults(_startIndexListRequest, _windowElementSize, FREETEXTSEARCHTYPE__POI);
      }
   }
}


void UserPOIListHandler::onPropertyUpdateUPOIInputInfoChanged()
{
   if (true == EXT_bIsUPOIContext)
   {
      ETG_TRACE_USR4(("UserPOIListHandler::onPropertyUpdateUPOIInputInfoChanged()"));
      _isListAvailable = true;

      const navmiddleware::PoiInputInfo& poiInputInfos = _navMiddleware.getPoiInputInfo();
      if (poiInputInfos.isValidDestination())
      {
         ETG_TRACE_USR4(("UserPOIListHandler::onPropertyUpdateUPOIInputInfoChanged(), Valid destination"));
         _navMiddleware.setLocationWithCurrentPoiInput();
         _navMiddleware.leaveHierarchicalPoiInput();
      }
      else
      {
         updateList();
      }
   }
}


void UserPOIListHandler::onPropertyUpdateUPOIEntriesChanged()
{
   if ((true == EXT_bIsUPOIContext) && (_listId == LIST_ID_NAV_DEST_NEARBY_LIST))
   {
      ETG_TRACE_USR4(("UserPOIListHandler::onPropertyUpdateUPOIEntriesChanged()"));
      tSharedPtrDataProvider dataProvider = getListDataProvider();
      if ((!dataProvider.PointsToNull()) && (dataProvider->listSize() > 0))
      {
         dataProvider->setCacheOnOff(false);
         POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
      }
   }
}


tSharedPtrDataProvider UserPOIListHandler::getListDataProvider()
{
   ETG_TRACE_USR4(("UserPOIListHandler::getListDataProvider(), List ID : %d, Start index : %d, Window Size : %d",
                   _listId, _startIndexListRequest, _windowElementSize));

   ListDataProviderBuilder listBuilder(_listId);
   LocationInfos locationInfos;
   fetchLocationInfo(locationInfos);

   const ::std::vector<LocationInfos::LocationInfo>& infos = locationInfos.getLocationInfos();
   ::std::vector<LocationInfos::LocationInfo>::const_iterator info = infos.begin();
   unsigned int idx = _startIndexListRequest;
   unsigned int actDestListSize = locationInfos.getCountOfAllLocations();
   bool isResultOutstanding = locationInfos.areResultsOutstanding();
   ETG_TRACE_USR4(("UserPOIListHandler::getListDataProvider(), Number of list entries : %d, Are results outstanding : %d",
                   actDestListSize, isResultOutstanding));

   if ((actDestListSize == 0) && (false == isResultOutstanding))
   {
      ETG_TRACE_USR4(("UserPOIListHandler::getListDataProvider(), No list items"));
      listBuilder.AddItem(idx, // - identifies the row
                          0UL, // - unused
                          DATA_CONTEXT_EMPTY_LIST_ITEM).AddData(POI_NO_LIST_ITEM); // 0 - Name(Id) of the button
      actDestListSize = 1;
      showDefaultUPOIMapView();	// For showing CCP in map
   }
   else
   {
      if (((0 == actDestListSize) || (0 == infos.size())) && (true == isResultOutstanding))
      {
         ETG_TRACE_USR4(("UserPOIListHandler::getListDataProvider(), List loading"));
         listBuilder.AddItem(idx, // - identifies the row
                             0UL, // - unused
                             DATA_CONTEXT_POI_LOADING_LIST_ITEM).AddData(POI_LIST_LOADING);
         actDestListSize = 1;
      }
      else
      {
         POICategoryListItemData item;
         for (; (idx < _startIndexListRequest + _windowElementSize) && (info != infos.end()); ++info, ++idx)
         {
            unsigned int turnIdx = (unsigned int)(info->m_direction.isValid() ? info->m_direction.getValue() : DIRECTIONDESCRIPTION_NORTH);
            std::stringstream indexedAddress;
            indexedAddress << idx + 1 << ". " << info->m_elementName;
            item.mName = indexedAddress.str().c_str();
            item.mDistance = info->m_distance.c_str();

            LocationAttributeInfos attributeInfos = info->m_attributes;
            std::string description = getUPOIDescription(attributeInfos);
            item.mAddress = description.c_str();

            ETG_TRACE_USR4(("UserPOIListHandler::getListDataProvider(), List Details populated at index : %d", idx));
            ETG_TRACE_USR4(("UserPOIListHandler::getListDataProvider(), UPOI name : %s", item.mName.GetCString()));
            ETG_TRACE_USR4(("UserPOIListHandler::getListDataProvider(), UPOI distance : %s", item.mDistance.GetCString()));
            ETG_TRACE_USR4(("UserPOIListHandler::getListDataProvider(), UPOI address : %s", item.mAddress.GetCString()));

            //This is for setting the direction icons ,will be used once bitmaps available.
            item.mDirectionIconActiveBitmap = ImageLoader::getAssetBitmapImage(getDirectionIcon(turnIdx)._activeBitmap);
            item.mDirectionIconActiveFocusesBitmap = ImageLoader::getAssetBitmapImage(getDirectionIcon(turnIdx)._activeFocussedBitmap);
            item.mDirectionIconFocusedBitmap = ImageLoader::getAssetBitmapImage(getDirectionIcon(turnIdx)._focusedBitmap);
            item.mDirectionIconNormalBitmap = ImageLoader::getAssetBitmapImage(getDirectionIcon(turnIdx)._normalBitmap);
            item.mDirectionIconPressedBitmap = ImageLoader::getAssetBitmapImage(getDirectionIcon(turnIdx)._pressedBitmap);
            listBuilder.AddItem(idx, 0UL, NEARBY_LIST_ITEM).AddDataBindingUpdater<POICategoryListItemDataBindingSource>(item);
         }
         updateUPOIsInMap();
      }
   }
   return listBuilder.CreateDataProvider(_startIndexListRequest, actDestListSize);
}


bool UserPOIListHandler::onCourierMessage(const UPOIStartInputReqMsg& oMsg)
{
   ETG_TRACE_USR4(("UserPOIListHandler::onCourierMessage(UPOIStartInputReqMsg())"));
   _isListAvailable = _isListRequested = false;
   _listId = _startIndexListRequest = _windowElementSize = _startIndexVisibleArea = 0;
   _isFreeTextSearch = oMsg.GetIsFreeTextSearch();
   if (false == _isFreeTextSearch)
   {
      (*_poiTitleInfoData).mPOITitleValue = LANGUAGE_STRING(TextId_0x0934, "MyPOIs nearby");
      _poiTitleInfoData.MarkAllItemsModified();
      _poiTitleInfoData.SendUpdate();

      _navMiddleware.startPoiInputWithPredefinedCategory(
         PREDEFINED_POI_CATEGORY__UNDEFINED,
         LocationSearchScope(LocationSearchScope::TYPE__AROUND_CURRENT_VEHICLE_POSITION),
         LOCATIONSEARCHSOURCE__USB_POI);
   }
   else
   {
      _navMiddleware.startFreeTextSearchInput(
         FREETEXTSEARCHTYPE__POI,
         LocationSearchScope(LocationSearchScope::TYPE__AROUND_CURRENT_VEHICLE_POSITION),
         LOCATIONSEARCHSOURCE__USB_POI);
      _isListAvailable = true;
   }
   return true;
}


bool UserPOIListHandler::onCourierMessage(const UPOIBrowseBackReqMsg& oMsg)
{
   ETG_TRACE_USR4(("UserPOIListHandler::onCourierMessage(UPOIBrowseBackReqMsg())"));

   COURIER_UNUSED(oMsg);

   _isListRequested = false;
   if (false == _isFreeTextSearch)
   {
      _navMiddleware.undoLastPoiEntrySelection();
   }
   else
   {
      _navMiddleware.undoFreeTextSearchElementSelection();
   }
   return true;
}


bool UserPOIListHandler::onCourierMessage(const ListChangedUpdMsg& oMsg)
{
   ETG_TRACE_USR4(("UserPOIListHandler::onCourierMessage(ListChangedUpdMsg()"));
   bool msgProcessed = false;
   unsigned int listID = oMsg.GetListId();

   if ((true == EXT_bIsUPOIContext) && (LIST_ID_NAV_DEST_NEARBY_LIST == listID))
   {
      if (ListMovementFinished == oMsg.GetMovementStatus())
      {
         _startIndexVisibleArea = oMsg.GetFirstVisibleIndex();
         updateUPOIsInMap();
      }
      msgProcessed = true;
   }
   return msgProcessed;
}


bool UserPOIListHandler::onCourierMessage(const SpellerOKBtnPressedMsg& oMsg)
{
   ETG_TRACE_USR4(("UserPOIListHandler::onCourierMessage(SpellerOKBtnPressedMsg())"));

   bool isMessageProcessed = false;
   const Courier::UInt32 sceneName = oMsg.GetSceneType();
   if ((true == EXT_bIsUPOIContext) && (AppHmi_NavigationModule_NavigationScenes_N_SpellerEdit == sceneName))
   {
      // Sets the POI search string entered through FTS and requests corresponding POI list
      SpellerHandler* spellerHandler = SpellerHandler::getInstance();
      const char* poiFtsText = spellerHandler->GetEditFieldText().GetCString();
      _navMiddleware.setFreeTextSearchString(poiFtsText);
      _navMiddleware.setFreeTextSearchSortOrder(LOCATIONSORTORDER__BY_AIR_DISTANCE_NEAREST_FIRST);

      // Updating POI title
      (*_poiTitleInfoData).mPOITitleValue = poiFtsText;
      _poiTitleInfoData.MarkAllItemsModified();
      _poiTitleInfoData.SendUpdate();

      isMessageProcessed = true;
   }
   return isMessageProcessed;
}


void UserPOIListHandler::updateUPOIsInMap()
{
   ETG_TRACE_USR4(("UserPOIListHandler::updateUPOIsInMap()"));

   LocationInfos locationInfos;
   fetchLocationInfo(locationInfos);
   const ::std::vector<navmiddleware::LocationInfos::LocationInfo>& locationInfoVector = locationInfos.getLocationInfos();

   // set icon infos for location overview
   MapIconInfo mapIconInfo;
   mapIconInfo.m_startOffset = _startIndexVisibleArea + 1;
   mapIconInfo.m_showNumberedIcons = true;

   // set coordinates of UPOI locations to be shown in map
   unsigned int startIndexInfosArray = _startIndexVisibleArea - _startIndexListRequest;
   std::vector<PosWGS84<double> > coordinateVector;

   for (unsigned int i = startIndexInfosArray; (i < (startIndexInfosArray + POI_IN_MAP_COUNT)) && (i < locationInfoVector.size()); ++i)
   {
      if (true == locationInfoVector[i].m_coordinate.isValid())
      {
         PosWGS84<double> coordinate(locationInfoVector[i].m_coordinate.getValue().getLongitude(), locationInfoVector[i].m_coordinate.getValue().getLatitude());
         coordinateVector.push_back(coordinate);
         ETG_TRACE_USR4(("UserPOIListHandler::updateUPOIsInMap(), UPOI Index (%d), UPOI Longitude (%f), UPOI Latitude (%f)", i, coordinate._longitude, coordinate._latitude));
      }
   }
   if (!coordinateVector.empty())
   {
      InfoStoreBase::CoordinatesToBeShownInMap& coordinatesToBeShownInMap = _infoStore.getCoordinatesToBeShownInMap();
      coordinatesToBeShownInMap._mapIconInfo = mapIconInfo;
      coordinatesToBeShownInMap._coordinates = coordinateVector;
      POST_MSG((COURIER_MESSAGE_NEW(SetMapCameraModeReqMsg)(MAP_CAMERA_AND_VIEW_MODE_PICK_POI_50PERCENT)));
   }
}


void UserPOIListHandler::showDefaultUPOIMapView()
{
   ETG_TRACE_USR4(("UserPOIListHandler::showDefaultUPOIMapView()"));

   MapIconInfo mapIconInfo;
   std::vector<PosWGS84<double> > coordinateVector;
   PosWGS84<double> coordinate(_navMiddleware.getPositionStatusInfo().getLongitude(), _navMiddleware.getPositionStatusInfo().getLatitude());
   coordinateVector.push_back(coordinate);

   if (!coordinateVector.empty())
   {
      InfoStoreBase::CoordinatesToBeShownInMap& coordinatesToBeShownInMap = _infoStore.getCoordinatesToBeShownInMap();
      coordinatesToBeShownInMap._mapIconInfo = mapIconInfo;
      coordinatesToBeShownInMap._coordinates = coordinateVector;
      POST_MSG((COURIER_MESSAGE_NEW(SetMapCameraModeReqMsg)(MAP_CAMERA_AND_VIEW_MODE_PICK_POI_50PERCENT)));
   }
}


bool UserPOIListHandler::onCourierMessage(const UPOIResetCoordinatesReqMsg& oMsg)
{
   COURIER_UNUSED(oMsg);
   ETG_TRACE_USR4(("UserPOIListHandler::onCourierMessage(UPOIResetCoordinatesReqMsg())"));
   _startIndexVisibleArea = 0; // Reset visible index to zero when list is shown from the beginning
   // Clear UPOI positions and icons shown in map
   _infoStore.resetCoordinatesToBeShownInMap();
   InfoStoreBase::CoordinatesToBeShownInMap coordinates = _infoStore.getCoordinatesToBeShownInMap();
   ValidValue<int> scale(coordinates._scale);
   const ::std::vector< GeoCoordinateDegree > optionalcoordinate;
   _navMiddleware.showLocationsInMap(MAP_VIEW_ID__PRIMARY, convert(coordinates._coordinates), optionalcoordinate, coordinates._mapIconInfo, scale, CameraAnimation__ADAPTIVE);
   return true;
}


bool UserPOIListHandler::onCourierMessage(const UPOILeaveInputReqMsg& oMsg)
{
   ETG_TRACE_USR4(("UserPOIListHandler::onCourierMessage(UPOILeaveInputReqMsg())"));
   COURIER_UNUSED(oMsg);
   if (false == _isFreeTextSearch)
   {
      _navMiddleware.leaveHierarchicalPoiInput();
   }
   else
   {
      _navMiddleware.leaveFreeTextSearchInput();
      _navMiddleware.clearFreeTextSearchHistory();
   }
   _listId = _startIndexListRequest = _windowElementSize = _startIndexVisibleArea = 0;
   return true;
}


void UserPOIListHandler::fetchLocationInfo(LocationInfos& locationInfos)
{
   if (false == _isFreeTextSearch)
   {
      locationInfos = _navMiddleware.getPoiEntries();
   }
   else
   {
      locationInfos = _navMiddleware.getFreeTextSearchResultInfos();
   }
}


std::string UserPOIListHandler::getUPOIDescription(const navmiddleware::LocationAttributeInfos& infos)
{
   std::string description;
   const ::std::vector<const LocationAttributeInfos::LocationAttributeInfo*>& locationAttributeInfos = infos.getAttributeInfos();
   ::std::vector<const LocationAttributeInfos::LocationAttributeInfo*>::const_iterator infosIterator = locationAttributeInfos.begin();
   for (; infosIterator != locationAttributeInfos.end(); ++infosIterator)
   {
      if ((NULL != *infosIterator) &&
            (LocationAttributeInfos::LocationAttributeInfo::ATTRIBUTE_TYPE__DESCRIPTION == (*infosIterator)->m_type))
      {
         const LocationAttributeInfos::StringAttributeInfo* descriptionInfo = infos.getDescriptionInfo();
         if (NULL != descriptionInfo)
         {
            description = descriptionInfo->m_content;
         }
      }
   }
   return description;
}


#endif // HALL_TO_MDW_COM
