/* ***************************************************************************************
* @file         : StackedPOIOnMapListHandler.cpp
* @author       : ECG5-Naveen Thangamalayan
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) 2018-2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"
#include "util/StrUtf8.h"
#include "AppHmi_NavigationStateMachine.h"
#include "StackedPOIOnMapListHandler.h"
#include "hmi_trace_if.h"
#include "CgiExtensions/ImageLoader.h"
#include "Common/Util/StringUtils.h"
#include "POI/DestinationDataUtils.h"


#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/StackedPOIOnMapListHandler.cpp.trc.h"
#endif
#ifdef HALL_TO_MDW_COM
using namespace navmiddleware;

StackedPOIOnMapListHandler::StackedPOIOnMapListHandler(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore)
   : HMIModelBase(navMiddleware, infoStore)
   , _listId(0)
   , _startIndex(0)
   , _windowSize(0)
{
}


StackedPOIOnMapListHandler::~StackedPOIOnMapListHandler()
{
}


void StackedPOIOnMapListHandler::initialize()
{
   _navMiddleware.registerMapPropertyUpdateCallback(*this);
}


void StackedPOIOnMapListHandler::deinitialize()
{
   _navMiddleware.unregisterMapPropertyUpdateCallback(*this);
}


bool StackedPOIOnMapListHandler::onCourierMessage(const ListDateProviderReqMsg& oMsg)
{
   bool isMsgConsumed = false;
   _listId = oMsg.GetListId();
   if (_listId == LIST_ID_MAP_STACKED_POI)
   {
      _startIndex = oMsg.GetStartIndex();
      _windowSize = oMsg.GetWindowElementSize();
      ETG_TRACE_USR4(("StackedPOIOnMapListHandler::onCourierMessage(ListDateProviderReqMsg(listID %d, startIndex %d, elementSize %d))", _listId, _startIndex, _windowSize));
      tSharedPtrDataProvider dataProvider = getListDataProvider();
      if (!dataProvider.PointsToNull())
      {
         if (dataProvider->listSize() > 0)
         {
            POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
         }
      }
      isMsgConsumed = true;
   }

   return isMsgConsumed;
}


tSharedPtrDataProvider StackedPOIOnMapListHandler::getListDataProvider()
{
   ListDataProviderBuilder listBuilder(_listId);
   const std::vector <navmiddleware::PickingResults::PickingResult>& pickingResults = _navMiddleware.getPickingResults().getPickingResultList();
   ::std::vector<navmiddleware::PickingResults::PickingResult>::const_iterator pickingResultInfo = pickingResults.begin();
   const Candera::UInt32 totalListSize = pickingResults.size();
   unsigned int idx = _startIndex;
   const char* itemID = "DynamicPOIItem";
   Candera::UInt32 totalPOI = 0;
   for (; (idx < _startIndex + _windowSize) && (pickingResultInfo != pickingResults.end()); pickingResultInfo++, ++idx)
   {
      if (pickingResultInfo->m_elementType == PICKING_RESULT_TYPE__POI)
      {
         POICategoryRootItemData item;
         Candera::String mapPickingTitle;
         if (pickingResultInfo->m_title.isValid() && pickingResultInfo->m_title.getValue().size())
         {
            mapPickingTitle = pickingResultInfo->m_title.getValue().c_str();
         }
         else if (pickingResultInfo->getPickingResultDetailPoi() &&
                  (pickingResultInfo->getPickingResultDetailPoi()->m_locationAttributeInfos.getLocationNameInfo() != NULL) &&
                  (!pickingResultInfo->getPickingResultDetailPoi()->m_locationAttributeInfos.getLocationNameInfo()->toString().empty()))
         {
            mapPickingTitle = pickingResultInfo->getPickingResultDetailPoi()->m_locationAttributeInfos.getLocationNameInfo()->toString().c_str();
         }
         if (false == mapPickingTitle.IsEmpty())
         {
            item.mName = mapPickingTitle.GetCString();
            navmiddleware::PickingResults::PickingResultDetailPoi* pickingResultDetailPoi = pickingResultInfo->getPickingResultDetailPoi();
            item.mIconNormalBitmap = (NULL != pickingResultDetailPoi) ?
                                     loadImage((const char*)pickingResultDetailPoi->m_iconData.getBlob(), pickingResultDetailPoi->m_iconData.getBlobSize(), true) :
                                     ImageLoader::getAssetBitmapImage("AppHmi_NavigationModule#Images#N_CategoryRoot#Icon_Category_3_normal");
            listBuilder.AddItem(idx, 0UL, itemID).AddDataBindingUpdater<POICategoryRootItemDataBindingSource>(item);
            totalPOI++;
         }
      }
   }

   return listBuilder.CreateDataProvider(_startIndex, totalPOI);
}


bool StackedPOIOnMapListHandler::onCourierMessage(const ButtonReactionMsg& oMsg)
{
   bool isMsgConsumed = false;
   if (oMsg.GetEnReaction() == enRelease)
   {
      ListProviderEventInfo info;
      ListProviderEventInfo::GetItemIdentifierInfo(oMsg.GetSender(), info);

      unsigned int index = (unsigned int)info.getHdlRow();

      const Courier::Identifier senderInfo = oMsg.GetSender();
      if (info.getListId() == LIST_ID_MAP_STACKED_POI)
      {
         const std::vector <navmiddleware::PickingResults::PickingResult>& pickingResults = _navMiddleware.getPickingResults().getPickingResultList();
         Candera::UInt32 totalSize = pickingResults.size();
         unsigned int index = (unsigned int)info.getHdlRow();
         index = index - _startIndex;

         if (index < totalSize)
         {
            double latitude = pickingResults.at(index).m_geoCoordinate.getLatitude();
            double longitude = pickingResults.at(index).m_geoCoordinate.getLongitude();
            _navMiddleware.setLocationWithPickingResult(pickingResults.at(index));
            setPickedPOIDetailInfo(pickingResults.at(index), _infoStore);

            // Set Co-Ordinate Position of Selected POI on DetailedInfo Scene
            std::vector<PosWGS84<double> > positionVector;
            positionVector.push_back(PosWGS84<double>(longitude, latitude));
            _infoStore.setLatitude(latitude);
            _infoStore.setLongitude(longitude);
            InfoStoreBase::CoordinatesToBeShownInMap& coordinatesToBeShownInMap = _infoStore.getCoordinatesToBeShownInMap();
            coordinatesToBeShownInMap._coordinates = positionVector;

            POST_MSG((COURIER_MESSAGE_NEW(ShowDetailInfoOfPickedPOIReqMsg)()));
            //Courire msg has been posted to show POI detailed info with the detailed address details
            POST_MSG((COURIER_MESSAGE_NEW(ShowDetailedAddressMsg)()));
            POST_MSG((COURIER_MESSAGE_NEW(ShowFavEditIcon_Enable_Delete_Guidance_Buttons)(0, false, true, 2, false))); // To show fav icon and to disable delete buttons
         }
         isMsgConsumed = true;
      }
   }

   return isMsgConsumed;
}


bool StackedPOIOnMapListHandler::onPropertyUpdatePickingDetailResultsAvailable()
{
   if (_listId == LIST_ID_MAP_STACKED_POI)
   {
      tSharedPtrDataProvider dataProvider = getListDataProvider();
      if (!dataProvider.PointsToNull())
      {
         if (dataProvider->listSize() > 0)
         {
            dataProvider->setCacheOnOff(false);
            POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
         }
      }
   }
   return true;
}


#endif // HALL_TO_MDW_COM
