/* ***************************************************************************************
* FILE:          POIListHandler.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  POIListHandler.cpp is part of HMI-Base reference/demo/test applications
*    COPYRIGHT:  (c) -2020 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"
#include "POIListHandler.h"
#include "DestinationDataUtils.h"	// For invoking getDirectionIcon()
#include "Common/Util/ImageUtils.h" // For invoking loadImage()
#include "string.h"

static const char* const DATA_CONTEXT_POI_LOADING_LIST_ITEM = "ListLoadingItem";
static const char* const DATA_CONTEXT_EMPTY_LIST_ITEM = "NoDataListItem";
static const char* const CATEGORY_BROWSE_LIST_ITEM = "DynamicPOIItem";
static const char* const NEARBY_LIST_ITEM = "BtnPoiMultiLineText";
static const Candera::String POI_LIST_ALL_CATEGORIES = LANGUAGE_STRING(TextId_0x0985, "All Categories");
static const Candera::String POI_FTS_TITLE = LANGUAGE_STRING(TextId_0x04B3, "Enter name");
static const Candera::UInt8 POI_IN_MAP_COUNT = 4;  // Number of POI icons to be shown in map

static const Candera::String POI_NO_LIST_ITEM = LANGUAGE_STRING(TextId_0x0BC5, "No list entries");
static const Candera::String POI_LIST_LOADING = LANGUAGE_STRING(TextId_0x1458, "Loading...");

static const Candera::String poiTopCategoryTitle[] =    // Text to be displayed at the top level of POI category list
{
   LANGUAGE_STRING(TextId_0x0937, "POIs nearby"),
   LANGUAGE_STRING(TextId_0x0B7A, "POIs around a point on the map"),
   LANGUAGE_STRING(TextId_0x0B78, "POIs at destination"),
   LANGUAGE_STRING(TextId_0x0219, "POIs on route"),
   LANGUAGE_STRING(TextId_0x0B79, "POIs at address")
};


#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/POIListHandler.cpp.trc.h"
#endif

#if defined (HALL_TO_MDW_COM)

POIListHandler::POIListHandler(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore)
   : HMIModelBase(navMiddleware, infoStore)
   , _listId(0)
   , _poiLevel(0)
   , _startIndexListRequest(0)
   , _startIndexVisibleArea(0)
   , _windowElementSize(0)
   , _numCategoryEntriesFTS(0)
   , _isCategoryEntryUpdatedFTS(false)
   , _isPOICategoryBrowseBack(false)
   , _poiListReady(false)
   , _isListRequested(false)
   , _enSearchScope(LocationSearchScope::TYPE__AROUND_CURRENT_VEHICLE_POSITION)
{
}


POIListHandler::~POIListHandler()
{
}


void POIListHandler::initialize()
{
   _navMiddleware.registerLocationPropertyUpdateCallback(*this);
   _infoStore.registerDataPropertyUpdateCallback(*this);
}


void POIListHandler::deinitialize()
{
   _navMiddleware.unregisterLocationPropertyUpdateCallback(*this);
   _infoStore.unregisterDataPropertyUpdateCallback(*this);
}


void POIListHandler::onPropertyUpdatePOIInputInfoChanged()
{
   if (false == EXT_bIsUPOIContext)
   {
      ETG_TRACE_USR4(("POIListHandler::onPropertyUpdatePOIInputInfoChanged"));
      _poiListReady = true;

      const navmiddleware::PoiInputInfo& poiInputInfos = _navMiddleware.getPoiInputInfo();
      if ((*_isPOICategoryBrowse).mPOITopCategoryStatus != poiInputInfos.isTopLevelEntryList())
      {
         ETG_TRACE_USR4(("POIListHandler::onPropertyUpdatePOIInputInfoChanged, Top level category : %d", poiInputInfos.isTopLevelEntryList()));
         (*_isPOICategoryBrowse).mPOITopCategoryStatus = poiInputInfos.isTopLevelEntryList();
         _isPOICategoryBrowse.MarkItemModified(ItemKey::POICategoryBrowseCheck::POITopCategoryStatusItem);
      }

      if ((*_isPOICategoryBrowse).mPOIResultListStatus != poiInputInfos.isPoiElementList())
      {
         ETG_TRACE_USR4(("POIListHandler::onPropertyUpdatePOIInputInfoChanged, List element : %d", poiInputInfos.isPoiElementList()));
         (*_isPOICategoryBrowse).mPOIResultListStatus = poiInputInfos.isPoiElementList();
         _isPOICategoryBrowse.MarkItemModified(ItemKey::POICategoryBrowseCheck::POIResultListStatusItem);

         if (true == poiInputInfos.isPoiElementList())
         {
            ETG_TRACE_USR4(("POIListHandler::onPropertyUpdatePOIInputInfoChanged, Sort order set to AIR_DISTANCE"));
            _navMiddleware.setPoiSortOrder(LOCATIONSORTORDER__BY_AIR_DISTANCE_NEAREST_FIRST);
         }
      }

      if (_isPOICategoryBrowse.HasModifiedItems())
      {
         if (!(_isPOICategoryBrowse.SendUpdate()))
         {
            ETG_TRACE_ERR(("_isPOICategoryBrowse update failed!"));
         }
      }

      // If the list data request from HMI is received before this property update from navmiddleware,
      // then updateList() is called so that requestPoiEntries() is invoked before getPoiEntries()
      if (true == _isListRequested)
      {
         updateList();
      }

      if (((true == (*_isPOICategoryBrowse).mPOITopCategoryStatus) && (true == _isPOICategoryBrowseBack)) ||						// top level category
            ((false == (*_isPOICategoryBrowse).mPOITopCategoryStatus) && (false == (*_isPOICategoryBrowse).mPOIResultListStatus)) ||	// sub category
            (true == (*_isPOICategoryBrowse).mPOIResultListStatus))																	// poi list
      {
         POST_MSG((COURIER_MESSAGE_NEW(POICategoryBrowseResponseMsg)()));
         _isPOICategoryBrowseBack = false;
      }

      if (poiInputInfos.isValidDestination())
      {
         _navMiddleware.setLocationWithCurrentPoiInput();
         _navMiddleware.leaveHierarchicalPoiInput();
      }
   }
}


void POIListHandler::onPropertyUpdatePOIEntriesChanged()
{
   // just post a ListDateProviderResMsg with our list data provider
   if ((false == EXT_bIsUPOIContext) && _isListRequested && _poiListReady)
   {
      ETG_TRACE_USR4(("POIListHandler::onPropertyUpdatePOIEntriesChanged"));
      tSharedPtrDataProvider dataProvider;
      //category list
      if (_listId == LIST_ID_NAV_DEST_POI_CAT || _listId == LIST_ID_NAV_DEST_POI_SUB_MENU)
      {
         dataProvider = getListDataProvider(_listId, CATEGORY_BROWSE_LIST_ITEM, _startIndexListRequest, _windowElementSize);
      }
      else if (_listId == LIST_ID_NAV_DEST_NEARBY_LIST)
      {
         //result list
         dataProvider = getListDataProvider(LIST_ID_NAV_DEST_NEARBY_LIST, NEARBY_LIST_ITEM, _startIndexListRequest, _windowElementSize);
      }
      else
      {
         //do nothing
      }
      if (!dataProvider.PointsToNull())
      {
         if (dataProvider->listSize() > 0)
         {
            dataProvider->setCacheOnOff(false);
            POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
         }
      }
   }
}


void POIListHandler::onPropertyUpdateFtsResultsChanged()
{
   ETG_TRACE_USR4(("POIListHandler::onPropertyUpdateFtsResultsChanged"));

   // If HMI request for creation of FTS list is received before this property update, then list is created
   if ((false == EXT_bIsUPOIContext) && (true == _isListRequested))
   {
      tSharedPtrDataProvider dataProvider = getListDataProvider(LIST_ID_NAV_DEST_NEARBY_LIST, NEARBY_LIST_ITEM, _startIndexListRequest, _windowElementSize);
      if (!dataProvider.PointsToNull() && (dataProvider->listSize() > 0))
      {
         dataProvider->setCacheOnOff(false);
         POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
      }
   }
}


bool POIListHandler::onCourierMessage(const POICategoryBrowseBackReqMsg& oMsg)
{
   ETG_TRACE_USR4(("POIListHandler::onCourierMessage(POICategoryBrowseBackReqMsg())"));

   COURIER_UNUSED(oMsg);

   _isListRequested = false;

   if ((false == EXT_bIsPOIFts) && (_poiLevel > 0))
   {
      _poiListReady = false;
      _isPOICategoryBrowseBack = true;
      _navMiddleware.undoLastPoiEntrySelection();
      _poiLevel--;
      updatePOITitle("", true);
   }
   else
   {
      _navMiddleware.undoFreeTextSearchElementSelection();
   }
   return true;
}


bool POIListHandler::onCourierMessage(const ListDateProviderReqMsg& oMsg)
{
   bool isMsgConsumed = false;
   if ((oMsg.GetListId() == LIST_ID_NAV_DEST_POI_CAT) ||
         (oMsg.GetListId() == LIST_ID_NAV_DEST_NEARBY_LIST) ||
         (oMsg.GetListId() == LIST_ID_NAV_DEST_POI_SUB_MENU))
   {
      isMsgConsumed = true;
      _isListRequested = true;
      _listId = oMsg.GetListId();
      _startIndexListRequest = oMsg.GetStartIndex();
      _windowElementSize = oMsg.GetWindowElementSize();
      ETG_TRACE_USR4(("POIListHandler::onCourierMessage(ListDateProviderReqMsg(listID %d, startIndex %d, elementSize %d))",
                      _listId, _startIndexListRequest, _windowElementSize));
      updateList();
   }

   return isMsgConsumed;
}


void POIListHandler::updateList()
{
   ETG_TRACE_USR4(("POIListHandler::updateList(), Start index : %d, Window size : %d", _startIndexListRequest, _windowElementSize));

   if (true == _poiListReady)
   {
      if (false == EXT_bIsPOIFts)		// POI category list
      {
         _navMiddleware.requestPoiEntries(_startIndexListRequest, _windowElementSize);
      }
      else								// POI Free text search list
      {
         _navMiddleware.requestFreeTextSearchResults(_startIndexListRequest, _windowElementSize, FREETEXTSEARCHTYPE__POI);
      }
   }
}


//
tSharedPtrDataProvider POIListHandler::getListDataProvider(unsigned int listID, const char* itemID, unsigned int startIndex, unsigned int windowElementSize)
{
   ETG_TRACE_USR4(("POIListHandler::getListDataProvider(), List ID : %d, Start index : %d, Window Size : %d", listID, startIndex, windowElementSize));
   ListDataProviderBuilder listBuilder(listID);
   LocationInfos locationInfos;
   fetchLocationInfo(locationInfos);

   const ::std::vector<navmiddleware::LocationInfos::LocationInfo>& infos = locationInfos.getLocationInfos();
   ::std::vector<navmiddleware::LocationInfos::LocationInfo>::const_iterator info = infos.begin();
   unsigned int idx = startIndex;
   unsigned int actDestListSize = locationInfos.getCountOfAllLocations();
   bool isResultOutstanding = locationInfos.areResultsOutstanding();
   ETG_TRACE_USR4(("POIListHandler::getListDataProvider(), Number of list entries : %d, Are results outstanding : %d", actDestListSize, isResultOutstanding));

   if ((actDestListSize == 0) && (false == isResultOutstanding))
   {
      if (listID == LIST_ID_NAV_DEST_NEARBY_LIST)
      {
         ETG_TRACE_USR4(("POIListHandler::getListDataProvider(), No list items"));
         listBuilder.AddItem(idx, // - identifies the row
                             0UL, // - unused
                             DATA_CONTEXT_EMPTY_LIST_ITEM).AddData(POI_NO_LIST_ITEM); // 0 - Name(Id) of the button
         actDestListSize = 1;
         showDefaultPOIMapView();	// For showing CCP in map
      }
   }
   else
   {
      if ((listID == LIST_ID_NAV_DEST_POI_CAT) || (listID == LIST_ID_NAV_DEST_POI_SUB_MENU))
      {
         POICategoryRootItemData item;
         for (; (idx < startIndex + windowElementSize) && (info != infos.end()); ++info, ++idx)
         {
            item.mName = info->m_elementName.c_str();
            ETG_TRACE_USR4(("POIListHandler::getListDataProvider(), Category name : %s", info->m_elementName.c_str()));
            ETG_TRACE_USR4(("POIListHandler::getListDataProvider(), Image length : %d", info->m_imageBlobLength));
            if (0 != info->m_imageBlobLength)
            {
               item.mIconNormalBitmap = loadImage((const char*)info->m_imageBlob, info->m_imageBlobLength, true);
            }
            else
            {
               item.mIconNormalBitmap = ImageLoader::getAssetBitmapImage("AppHmi_NavigationModule#Images#N_CategoryRoot#Icon_Category_3_normal");
            }
            listBuilder.AddItem(idx, 0UL, itemID).AddDataBindingUpdater<POICategoryRootItemDataBindingSource>(item);
         }
      }
      else if ((listID == LIST_ID_NAV_DEST_NEARBY_LIST))
      {
         if (((0 == actDestListSize) || (0 == infos.size())) && (true == isResultOutstanding))
         {
            ETG_TRACE_USR4(("POIListHandler::getListDataProvider(), List loading"));
            listBuilder.AddItem(idx, // - identifies the row
                                0UL, // - unused
                                DATA_CONTEXT_POI_LOADING_LIST_ITEM).AddData(POI_LIST_LOADING);
            actDestListSize = 1;
         }
         else if ((true == isResultOutstanding) && (true == EXT_bIsPOIFts) && (false == _isCategoryEntryUpdatedFTS))
         {
            // When FTS is initiated, the poi category entries are loaded first
            // and after the list is fully loaded, the poi category entries are moved to the end of the list
            // Hence number of category entries in POI FTS is determined up-front before list is fully loaded
            _isCategoryEntryUpdatedFTS = true;
            for (; info != infos.end(); ++info)
            {
               if (LocationInfos::ENTRY_TYPE__POI_CATEGORY == info->m_entryType)
               {
                  ETG_TRACE_USR4(("POIListHandler::getListDataProvider(), Category name : %s", info->m_elementName.c_str()));
                  _numCategoryEntriesFTS++;
               }
            }
            ETG_TRACE_USR4(("POIListHandler::getListDataProvider(), Number of POI category entries in FTS : %d", _numCategoryEntriesFTS));
         }
         else
         {
            ETG_TRACE_USR4(("POIListHandler::getListDataProvider(), Current sort order : %d", locationInfos.getCurrentSortOrder()));

            POICategoryListItemData item;
            for (; (idx < startIndex + windowElementSize) && (info != infos.end()); ++info, ++idx)
            {
               if (LocationInfos::ENTRY_TYPE__POI_LOCATION == info->m_entryType) // If FTS, entries with type set to POI CATEGORY is not added to the list
               {
                  unsigned int turnIdx = (unsigned int)(info->m_direction.isValid() ? info->m_direction.getValue() : DIRECTIONDESCRIPTION_NORTH);
                  std::stringstream indexedAddress;
                  indexedAddress << idx + 1 << ". " << info->m_elementName;
                  item.mName = indexedAddress.str().c_str();
                  item.mDistance = info->m_distance.c_str();
                  item.mAddress = info->m_description.c_str();

                  ETG_TRACE_USR4(("POIListHandler::getListDataProvider(), List Details populated at index : %d", idx));
                  ETG_TRACE_USR4(("POIListHandler::getListDataProvider(), POI name : %s", item.mName.GetCString()));
                  ETG_TRACE_USR4(("POIListHandler::getListDataProvider(), POI distance : %s", item.mDistance.GetCString()));
                  ETG_TRACE_USR4(("POIListHandler::getListDataProvider(), POI address : %s", item.mAddress.GetCString()));

                  //This is for setting the direction icons ,will be used once bitmaps available.
                  item.mDirectionIconActiveBitmap = ImageLoader::getAssetBitmapImage(getDirectionIcon(turnIdx)._activeBitmap);
                  item.mDirectionIconActiveFocusesBitmap = ImageLoader::getAssetBitmapImage(getDirectionIcon(turnIdx)._activeFocussedBitmap);
                  item.mDirectionIconFocusedBitmap = ImageLoader::getAssetBitmapImage(getDirectionIcon(turnIdx)._focusedBitmap);
                  item.mDirectionIconNormalBitmap = ImageLoader::getAssetBitmapImage(getDirectionIcon(turnIdx)._normalBitmap);
                  item.mDirectionIconPressedBitmap = ImageLoader::getAssetBitmapImage(getDirectionIcon(turnIdx)._pressedBitmap);
                  listBuilder.AddItem(idx, 0UL, itemID).AddDataBindingUpdater<POICategoryListItemDataBindingSource>(item);
               }
            }
            updatePOIsInMap();	// For showing POI icons in map
         }
         if (true == EXT_bIsPOIFts)
         {
            // Category entries are not displayed in POI FTS result. Hence the list size is calculated accordingly
            actDestListSize = actDestListSize - _numCategoryEntriesFTS;
            ETG_TRACE_USR4(("POIListHandler::getListDataProvider(), Number of list entries (FTS) : %d", actDestListSize));
         }
      }
   }
   return listBuilder.CreateDataProvider(startIndex, actDestListSize);
}


bool POIListHandler::onCourierMessage(const InitPOICategoryBrowseReqMsg& oMsg)
{
   ETG_TRACE_USR4(("POIListHandler::onCourierMessage(InitPOICategoryBrowseReqMsg())"));

   _poiLevel = 0;
   _isListRequested = false;
   _poiListReady = false;

   enLocationSearchScope searchScope = oMsg.GetSearchScope();	// Search scope from state machine

   // Updating title of POI top level category list (depending upon the state from which POI browse was initiated)
   (*_poiTopLevelTitleInfoData).mPOITopCategoryTitleValue = poiTopCategoryTitle[static_cast<unsigned int>(searchScope)];

   _poiTopLevelTitleInfoData.MarkAllItemsModified();
   _poiTopLevelTitleInfoData.SendUpdate();

   // Both SEARCHSCOPE_AROUND_CURRENT_LOCATION_POSITION (from By Map) and SEARCHSCOPE_AT_ADDRESS (from New destination --> POI search) correspond to TYPE__AROUND_CURRENT_LOCATION_POSITION
   // Only POI titles differ in these 2 cases
   _enSearchScope = (searchScope == SEARCHSCOPE_AT_ADDRESS) ? LocationSearchScope::TYPE__AROUND_CURRENT_LOCATION_POSITION : static_cast<LocationSearchScope::Type>(searchScope);
   ETG_TRACE_USR4(("POI search scope : %d", _enSearchScope));

   // Reset the list status before starting the POI hierarchical search
   if ((*_isPOICategoryBrowse).mPOIResultListStatus)
   {
      (*_isPOICategoryBrowse).mPOIResultListStatus = false;
      _isPOICategoryBrowse.MarkItemModified(ItemKey::POICategoryBrowseCheck::POIResultListStatusItem);
   }

   if ((*_isPOICategoryBrowse).mPOITopCategoryStatus)
   {
      (*_isPOICategoryBrowse).mPOITopCategoryStatus = false;
      _isPOICategoryBrowse.MarkItemModified(ItemKey::POICategoryBrowseCheck::POITopCategoryStatusItem);
   }

   if (_isPOICategoryBrowse.HasModifiedItems())
   {
      if (!(_isPOICategoryBrowse.SendUpdate()))
      {
         ETG_TRACE_ERR(("_isPOICategoryBrowse update failed!"));
      }
   }

   navmiddleware::LocationSearchScope locationSearchScope(_enSearchScope);
   if (LocationSearchScope::TYPE__AROUND_WAYPOINT_POSITION == _enSearchScope)
   {
      // For listing the POIs around destination, waypointIndex is set with destination index
      locationSearchScope.m_waypointIndex = _navMiddleware.getWaypointInfos().getWaypointInfos().size() - 1;
   }
   _navMiddleware.startHierarchicalPoiInput(
      locationSearchScope,
      navmiddleware::LOCATIONSEARCHSOURCE__OFFLINE,
      navmiddleware::LocationContext(navmiddleware::LocationContext::HMICONTEXT__GUI, navmiddleware::APPLICATIONID__NAVIGATION));

   return true;
}


bool POIListHandler::onCourierMessage(const ButtonReactionMsg& oMsg)
{
   bool bIsMsgConsumed = false;
   if (oMsg.GetEnReaction() == enRelease)
   {
      ListProviderEventInfo info;
      ListProviderEventInfo::GetItemIdentifierInfo(oMsg.GetSender(), info);

      unsigned int listId = info.getListId();
      if ((listId == LIST_ID_NAV_DEST_POI_CAT) ||
            (listId == LIST_ID_NAV_DEST_POI_SUB_MENU) ||
            (listId == LIST_ID_NAV_DEST_NEARBY_LIST))
      {
         ETG_TRACE_USR4(("POIListHandler::onCourierMessage(ButtonReactionMsg(listID %d, row %d, column %d))", listId, info.getHdlRow(), info.getHdlCol()));

         unsigned int listIndex = (unsigned int)info.getHdlRow();
         LocationInfos locationInfos;
         fetchLocationInfo(locationInfos);
         const ::std::vector<navmiddleware::LocationInfos::LocationInfo>& locationInfoVector = locationInfos.getLocationInfos();

         unsigned int relativeIndex = listIndex - _startIndexListRequest;
         if (relativeIndex < locationInfoVector.size())
         {
            _isListRequested = false;
            navmiddleware::LocationInfos::LocationInfo locationInfo = locationInfoVector[relativeIndex];

            if (false == EXT_bIsPOIFts)
            {
               _poiListReady = false;
               _navMiddleware.selectPoiEntry(listIndex);
               _poiLevel++;
               if (listId != LIST_ID_NAV_DEST_NEARBY_LIST)  // For selection of POI list entry to see detailed info, POI title need not be updated
               {
                  updatePOITitle(locationInfo.m_elementName.c_str(), false);
               }
            }
            else
            {
               _navMiddleware.setLocationWithFreeTextSearchResult(listIndex);
               _navMiddleware.leaveFreeTextSearchInput();
            }

            if (listId == LIST_ID_NAV_DEST_NEARBY_LIST)
            {
               // Set Co-Ordinate Position of Selected POI on DetailedInfo Scene
               std::vector<PosWGS84<double> > positionVector;
               positionVector.push_back(PosWGS84<double>(locationInfo.m_coordinate.getValue().getLongitude(), locationInfo.m_coordinate.getValue().getLatitude()));
               _infoStore.setLatitude(locationInfo.m_coordinate.getValue().getLatitude());
               _infoStore.setLongitude(locationInfo.m_coordinate.getValue().getLongitude());
               InfoStoreBase::CoordinatesToBeShownInMap& coordinatesToBeShownInMap = _infoStore.getCoordinatesToBeShownInMap();
               coordinatesToBeShownInMap._coordinates = positionVector;

               // Set POI location name and address
               _infoStore.setAddressInfoDetailsName(locationInfo.m_elementName.c_str());
               _infoStore.setAddressInfoDetailsAddress(locationInfo.m_description.c_str());

               if (locationInfo.m_attributes.getPhoneNumberInfo() != NULL)
               {
                  _infoStore.setAddressInfoDetailsPhoneNumber(locationInfo.m_attributes.getPhoneNumberInfo()->m_content.c_str());
               }
               else
               {
                  _infoStore.setAddressInfoDetailsPhoneNumber("");
               }

               // Courier msg has been posted to show POI detailed info with the detailed address details
               POST_MSG((COURIER_MESSAGE_NEW(ShowDetailedAddressMsg)()));
               POST_MSG((COURIER_MESSAGE_NEW(ShowFavEditIcon_Enable_Delete_Guidance_Buttons)(0, false, true, 2, false))); // To show fav icon and to disable delete buttons
               POST_MSG((COURIER_MESSAGE_NEW(POIListBtnSelectionMsg)()));
            }
         }
         bIsMsgConsumed = true;
      } // End of check for POI list IDs
   }   // End of check for release button reaction

   return bIsMsgConsumed;
}


bool POIListHandler::onCourierMessage(const ShowAllPOICategoryReqMsg& oMsg)
{
   COURIER_UNUSED(oMsg);

   ETG_TRACE_USR4(("POIListHandler::onCourierMessage(ShowAllPoiCategoryReqMsg())"));

   // Resetting list related flags
   _isListRequested = false;
   _poiListReady = false;

   _navMiddleware.selectAllPoiCategory();
   _poiLevel++;
   updatePOITitle(POI_LIST_ALL_CATEGORIES.GetCString(), false);

   return true;
}


bool POIListHandler::onCourierMessage(const SpellerOKBtnPressedMsg& oMsg)
{
   ETG_TRACE_USR4(("POIListHandler::onCourierMessage(SpellerOKBtnPressedMsg())"));

   static const char* previousPoiFtsText = "";

   bool isMessageProcessed = false;
   const Courier::UInt32 sceneName = oMsg.GetSceneType();
   if (AppHmi_NavigationModule_NavigationScenes_N_POI_List == sceneName)
   {
      // Sets the POI search string entered through FTS and requests corresponding POI list
      SpellerHandler* spellerHandler = SpellerHandler::getInstance();
      const char* poiFtsText = spellerHandler->GetEditFieldText().GetCString();
      if (0 != strcmp(previousPoiFtsText, poiFtsText))
      {
         _numCategoryEntriesFTS = 0;
         _isCategoryEntryUpdatedFTS = false;
      }
      previousPoiFtsText = poiFtsText;
      _navMiddleware.setFreeTextSearchString(poiFtsText);
      _navMiddleware.setFreeTextSearchSortOrder(LOCATIONSORTORDER__BY_AIR_DISTANCE_NEAREST_FIRST);

      // Updating POI title
      (*_poiTitleInfoData).mPOITitleValue = poiFtsText;
      _poiTitleInfoData.MarkAllItemsModified();
      _poiTitleInfoData.SendUpdate();

      isMessageProcessed = true;
   }
   return isMessageProcessed;
}


bool POIListHandler::onCourierMessage(const POIStartFtsReqMsg& oMsg)
{
   COURIER_UNUSED(oMsg);
   ETG_TRACE_USR4(("POIListHandler::onCourierMessage(POIStartFtsReqMsg())"));

   // Resetting list related flags
   _isListRequested = false;

   // Update title for FTS speller scene
   (*_FtsTitleData).mTitleValue = POI_FTS_TITLE.GetCString();
   _FtsTitleData.MarkAllItemsModified();
   _FtsTitleData.SendUpdate();

   _navMiddleware.startFreeTextSearchInput(
      FREETEXTSEARCHTYPE__POI,
      LocationSearchScope(_enSearchScope),
      LOCATIONSEARCHSOURCE__OFFLINE,
      LocationContext(LocationContext::HMICONTEXT__GUI, APPLICATIONID__NAVIGATION),
      ValidValue<LanguageEntryInfo>());
   return true;
}


bool POIListHandler::onCourierMessage(const POILeaveFtsReqMsg& oMsg)
{
   COURIER_UNUSED(oMsg);
   ETG_TRACE_USR4(("POIListHandler::onCourierMessage(POILeaveFtsReqMsg())"));

   _navMiddleware.leaveFreeTextSearchInput();
   _navMiddleware.clearFreeTextSearchHistory();

   return true;
}


bool POIListHandler::onCourierMessage(const POILeaveInputReqMsg& oMsg)
{
   COURIER_UNUSED(oMsg);
   ETG_TRACE_USR4(("POIListHandler::onCourierMessage(POILeaveInputReqMsg())"));

   _navMiddleware.leaveHierarchicalPoiInput();

   return true;
}


bool POIListHandler::onCourierMessage(const ResetCoordinatesReqMsg& oMsg)
{
   COURIER_UNUSED(oMsg);
   ETG_TRACE_USR4(("POIListHandler::onCourierMessage(ResetCoordinatesReqMsg())"));

   _startIndexVisibleArea = 0; // Reset visible index to zero when list is shown from the beginning

   // Clear POI positions and icons shown in map
   _infoStore.resetCoordinatesToBeShownInMap();
   InfoStoreBase::CoordinatesToBeShownInMap coordinates = _infoStore.getCoordinatesToBeShownInMap();
   navmiddleware::ValidValue<int> scale(coordinates._scale);
   const ::std::vector< GeoCoordinateDegree > optionalcoordinate;
   _navMiddleware.showLocationsInMap(MAP_VIEW_ID__PRIMARY, convert(coordinates._coordinates), optionalcoordinate, coordinates._mapIconInfo, scale, navmiddleware::CameraAnimation__ADAPTIVE);

   return true;
}


bool POIListHandler::onCourierMessage(const ListChangedUpdMsg& oMsg)
{
   bool msgProcessed = false;
   unsigned int listID = oMsg.GetListId();

   if (_listId == listID)
   {
      ETG_TRACE_USR4(("POIListHandler::onCourierMessage(ListChangedUpdMsg()"));
      if (oMsg.GetMovementStatus() == ListMovementFinished)
      {
         _startIndexVisibleArea = oMsg.GetFirstVisibleIndex();
         updatePOIsInMap();
      }
      msgProcessed = true;
   }
   return msgProcessed;
}


void POIListHandler::updatePOIsInMap()
{
   ETG_TRACE_USR4(("POIListHandler::updatePOIsInMap()"));

   LocationInfos locationInfos;
   fetchLocationInfo(locationInfos);
   const ::std::vector<navmiddleware::LocationInfos::LocationInfo>&  locationInfoVector = locationInfos.getLocationInfos();

   // set icon infos for location overview
   MapIconInfo mapIconInfo;
   mapIconInfo.m_startOffset = _startIndexVisibleArea + 1;
   mapIconInfo.m_showNumberedIcons = true;

   // set coordinates of POI locations to be shown in map
   unsigned int startIndexInfosArray = _startIndexVisibleArea - _startIndexListRequest;
   std::vector<PosWGS84<double> > coordinateVector;

   for (unsigned int i = startIndexInfosArray; (i < (startIndexInfosArray + POI_IN_MAP_COUNT)) && (i < locationInfoVector.size()); ++i)
   {
      if (true == locationInfoVector[i].m_coordinate.isValid())
      {
         PosWGS84<double> coordinate(locationInfoVector[i].m_coordinate.getValue().getLongitude(), locationInfoVector[i].m_coordinate.getValue().getLatitude());
         coordinateVector.push_back(coordinate);
         ETG_TRACE_USR4(("POI Index (%d)", i));
         ETG_TRACE_USR4(("POI Longitude (%f)", coordinate._longitude));
         ETG_TRACE_USR4(("POI Latitude (%f)", coordinate._latitude));
      }
   }
   if (!coordinateVector.empty())
   {
      InfoStoreBase::CoordinatesToBeShownInMap& coordinatesToBeShownInMap = _infoStore.getCoordinatesToBeShownInMap();
      coordinatesToBeShownInMap._mapIconInfo = mapIconInfo;
      coordinatesToBeShownInMap._coordinates = coordinateVector;
      POST_MSG((COURIER_MESSAGE_NEW(SetMapCameraModeReqMsg)(MAP_CAMERA_AND_VIEW_MODE_PICK_POI_50PERCENT)));
   }
}


void POIListHandler::showDefaultPOIMapView()
{
   ETG_TRACE_USR4(("POIListHandler::showDefaultPOIMapView()"));

   MapIconInfo mapIconInfo;
   std::vector<PosWGS84<double> > coordinateVector;
   PosWGS84<double> coordinate(_navMiddleware.getPositionStatusInfo().getLongitude(), _navMiddleware.getPositionStatusInfo().getLatitude());
   coordinateVector.push_back(coordinate);

   if (!coordinateVector.empty())
   {
      InfoStoreBase::CoordinatesToBeShownInMap& coordinatesToBeShownInMap = _infoStore.getCoordinatesToBeShownInMap();
      coordinatesToBeShownInMap._mapIconInfo = mapIconInfo;
      coordinatesToBeShownInMap._coordinates = coordinateVector;
      POST_MSG((COURIER_MESSAGE_NEW(SetMapCameraModeReqMsg)(MAP_CAMERA_AND_VIEW_MODE_PICK_POI_50PERCENT)));
   }
}


void POIListHandler::updatePOITitle(Candera::String poiTitleValue, bool isPOICategoryBrowseBack)
{
   ETG_TRACE_USR4(("POIListHandler::updatePOITitle(), POI title : %s", poiTitleValue.GetCString()));
   const std::string SEPERATOR(">");

   if (_poiLevel > 0)
   {
      if (_poiLevel > MAX_POI_LEVEL)
      {
         _poiLevel = MAX_POI_LEVEL;
      }
      if (false == poiTitleValue.IsEmpty())
      {
         _poiTitleValue[_poiLevel - 1] = poiTitleValue;
      }
      std::string POITitle("");
      for (unsigned int idx = 0; idx < _poiLevel; idx++)
      {
         POITitle = POITitle + _poiTitleValue[idx].GetCString();
         if (idx != (_poiLevel - 1))
         {
            POITitle = POITitle + SEPERATOR;
         }
      }
      (*_poiTitleInfoData).mPOITitleValue = POITitle.c_str();
      _poiTitleInfoData.MarkAllItemsModified();
      _poiTitleInfoData.SendUpdate();
   }
}


void POIListHandler::fetchLocationInfo(LocationInfos& locationInfos)
{
   if (false == EXT_bIsPOIFts)
   {
      locationInfos = _navMiddleware.getPoiEntries();
   }
   else
   {
      locationInfos = _navMiddleware.getFreeTextSearchResultInfos();
   }
}


#endif
