/**************************************************************************************
* @file         : HMIModelNavDataUpdate.cpp
* @author       : ECG5-Naveen Thangamalayan
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include <hmibase/gui_std_if.h>
#include <iomanip>
#include "HMIModelNavDataUpdate.h"
#include "Common/Util/HMIModelIncludes.h"
#include "Common/Util/EnvironmentUtils.h"
#include "Common/Util/SurfaceSynchronizationHandler.h"
#include "NavDataUpdateUtils.h"
#include "Map/MapScreenDataUtils.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/HMIModelNavDataUpdate.cpp.trc.h"
#endif

#if defined (HALL_TO_MDW_COM)

static const char* const ENV_HMI_MAPUPDATE_ENABLE_PARTITION_LOCK = "HMI_MAPUPDATE_ENABLE_PARTITION_LOCK";

#ifdef MAPUPDATE_PARTITION_LOCK_FEATURE
static const bool FORCE_ENABLE_PARTITION_LOCK = true;
#else
static const bool FORCE_ENABLE_PARTITION_LOCK = false;
#endif

using namespace navmiddleware;

static const bool DIAGNOSTIC_ON  = true;
static const bool DIAGNOSTIC_OFF = false;


bool HMIModelNavDataUpdate::NeedTOEnteredInMapUpdate = false;
bool HMIModelNavDataUpdate::isMapUpdateInProgress()
{
   return NeedTOEnteredInMapUpdate;
}


HMIModelNavDataUpdate::HMIModelNavDataUpdate(NavMiddleware& navMiddleware, InfoStore& infoStore)
   : HMIModelBase(navMiddleware, infoStore)
   , _currentUpdateStatus(NAVDATAUPDATE__UPDATESTATUS__NOT_HANDLED)
     //, _exportStatus(NAVDATAUPDATE__EXPORTSTATUS__NOT_HANDLED)
   , _deviceType(NAVDATAUPDATE__DEVICETYPE__NONE)
   , _diagnosticModeON(DIAGNOSTIC_OFF)
   , _PARTITION_LOCK_ENABLED(getEnvVarAsBool(ENV_HMI_MAPUPDATE_ENABLE_PARTITION_LOCK, FORCE_ENABLE_PARTITION_LOCK))
   , _partitionLockRequested(false)
   , _partitionLockError(PARTITION_LOCK_ERR___UNKNOWN)
   , mapUpadateLaunchFromNaviHMI(false)
   , _activePopup(MAP_UPDATE_POPUP_NONE)
{
}


HMIModelNavDataUpdate::~HMIModelNavDataUpdate()
{
}


void HMIModelNavDataUpdate::initialize()
{
   ETG_TRACE_USR1(("HMIModelNavDataUpdate::initialize()"));

   static bool initialized = false;

   if (!initialized)
   {
      _navMiddleware.registerNavDataUpdatePropertyUpdateCallback(*this);
      _infoStore.registerDataPropertyUpdateCallback(*this);
      initialized = true;

      (*_navDataUpdatePopupText).mCurrentNavDataVersion = LANGUAGE_STRING(TextId_0x0AC0, "Unknown").GetCString();
      (*_navDataUpdatePopupText).mNewNavDataVersion = LANGUAGE_STRING(TextId_0x0AC0, "Unknown").GetCString();
      _navDataUpdatePopupText.MarkItemModified(ItemKey::NavDataPopupText::CurrentNavDataVersionItem);
      _navDataUpdatePopupText.MarkItemModified(ItemKey::NavDataPopupText::NewNavDataVersionItem);
      sendPopupDynamicTextNavDataUpdate();
   }
   else
   {
      ETG_TRACE_USR1(("HMIModelNavDataUpdate was initialized before!"));
   }
}


void HMIModelNavDataUpdate::deinitialize()
{
   ETG_TRACE_USR1(("HMIModelNavDataUpdate::deinitialize()"));

   _navMiddleware.unregisterNavDataUpdatePropertyUpdateCallback(*this);
   _infoStore.unregisterDataPropertyUpdateCallback(*this);
}


void HMIModelNavDataUpdate::recoverNavData()
{
   ETG_TRACE_USR1(("HMIModelNavDataUpdate::recoverNavData()"));

   initialize();

   _currentUpdateStatus = NAVDATAUPDATE__UPDATESTATUS__FULL_UPDATE_REQUIRED;
}


void HMIModelNavDataUpdate::onPropertyUpdateDeviceStatusChanged()
{
   const NavDataDeviceInfos& deviceInfo = _navMiddleware.getNavDataDeviceInfos();
   ETG_TRACE_USR4(("HMIModelNavDataUpdate::onPropertyUpdateDeviceStatusChanged()\n%s", deviceInfo.toString().c_str()));

   enNavDataUpdate__DeviceStatus usbDeviceStatus = NAVDATAUPDATE__DEVICESTATUS__NOT_AVAILABLE;
   ::std::vector< NavDataDeviceInfo>::const_iterator itemNumber = deviceInfo.getNavDataDeviceInfos().begin();
   for (; ((itemNumber != deviceInfo.getNavDataDeviceInfos().end()) && (NAVDATAUPDATE__DEVICESTATUS__NOT_AVAILABLE == usbDeviceStatus)); ++itemNumber)
   {
      if (DEVICETYPE__USB == itemNumber->getDeviceType())
      {
         usbDeviceStatus = getMappedDeviceStatus(itemNumber->getDeviceStatus());
      }
   }

   ETG_TRACE_USR4(("HMIModelNavDataUpdate::onPropertyUpdateDeviceStatusChanged(), USB device status: %d, Is diagnostic mode : %d", usbDeviceStatus, _diagnosticModeON));
   if (_diagnosticModeON)
   {
      switch (usbDeviceStatus)
      {
         case NAVDATAUPDATE__DEVICESTATUS__NOT_AVAILABLE:
         case NAVDATAUPDATE__DEVICESTATUS__AVAILABLE:
         {
            // FIXME: MDW send usb device status TWICE (AVAILABLE then INC_UPDATE/FULL_UPDATE)
            //       - Causes: Navigation will be in Diag Mode till valid USB stick inserted (maybe forever)
            // TODO: 2 possibilities:
            //       - MDW should send status only ONCE?
            //       - HMI needs timer
            // sendUpdateResultToService(NAVDATAUPDATE__DIAG_RESULT__FAILED__INVALID_USB_DATA);
            break;
         }
         // FIXME: update will happen in Diag Mode even with INC_UPDATE
         case NAVDATAUPDATE__DEVICESTATUS__INC_UPDATE:
         case NAVDATAUPDATE__DEVICESTATUS__FULL_UPDATE:
         {
            ETG_TRACE_USR4(("[Diag Mode ON] USB is valid, requesting AvailableUpdates."));
            requestAvailableUpdates(NAVDATAUPDATE__DEVICETYPE__USB);
            break;
         }
         default:
         {
            sendUpdateResultToService(NAVDATAUPDATE__DIAG_RESULT__FAILED__OTHERS);
            break;
         }
      }
   }
   else
   {
      //Currently only Full USB update is implemented
      //if (usbDeviceStatus == NAVDATAUPDATE__DEVICESTATUS__INC_UPDATE || usbDeviceStatus == NAVDATAUPDATE__DEVICESTATUS__FULL_UPDATE)
      if (usbDeviceStatus == NAVDATAUPDATE__DEVICESTATUS__FULL_UPDATE)
      {
         ETG_TRACE_USR4(("[Diag Mode OFF] USB is valid, requesting AvailableUpdates."));
         requestAvailableUpdates(NAVDATAUPDATE__DEVICETYPE__USB);
      }
      else if (usbDeviceStatus == NAVDATAUPDATE__DEVICESTATUS__NOT_AVAILABLE)
      {
         //No Popup should be shown when USB is unavailable
      }
   }
}


bool HMIModelNavDataUpdate::requestAvailableUpdates(enNavDataUpdate__DeviceType deviceType)
{
   ETG_TRACE_USR4(("HMIModelNavDataUpdate::requestAvailableUpdates(deviceType %d, isSWUpdating %d)", deviceType, _infoStore.isSWUpdating()));

   bool retVal = false;

   if (_infoStore.isSWUpdating())
   {
      mapUpadateLaunchFromNaviHMI = false;
      _partitionLockError = PARTITION_LOCK_ERR___SW_UPDATING;
      updatePopupText(MAP_UPDATE_ONGOING);
      setPopupVisibility(MAP_UPDATE_ERROR_POPUP, true);
   }
   else
   {
      // TODO: Partition lock is mandatory
      // The implementaion will be permanent when functionalities become stable
      ///
      if (_PARTITION_LOCK_ENABLED)
      {
         // FIXME: cannot return value with async progress, always pass true
         performRequestAvailableUpdatesAfterLock(deviceType);
         retVal = true;
      }
      else
      {
         retVal = performRequestAvailableUpdates(deviceType, std::string(""));
      }
   }

   return retVal;
}


/**
  * Request MapDownloadLock to SPM via SpmClientHandler
  * If MapDownloadLock is successful, HMIModelNavDataUpdate perform requestAvailableUpdates to MDW
  */
void HMIModelNavDataUpdate::performRequestAvailableUpdatesAfterLock(enNavDataUpdate__DeviceType deviceType)
{
   ETG_TRACE_USR4(("HMIModelNavDataUpdate::performRequestAvailableUpdatesAfterLock(%d)", deviceType));

   POST_MSG((COURIER_MESSAGE_NEW(NAVDATAUPDATE_SpmStartMapUpdateLockReqMsg)()));

   _deviceType = deviceType;
   _partitionLockRequested = true;
   _partitionLockError = PARTITION_LOCK_ERR___UNKNOWN;
}


bool HMIModelNavDataUpdate::performRequestAvailableUpdates(enNavDataUpdate__DeviceType deviceType, const std::string& path)
{
   ETG_TRACE_USR4(("HMIModelNavDataUpdate::performRequestAvailableUpdates(device %d, path: %s)", deviceType, path.c_str()));

   bool retVal = false;

   NAVDATA_INIT_DEVICEID();

   switch (deviceType)
   {
      case NAVDATAUPDATE__DEVICETYPE__USB:
      {
         //if (IS_DEVICE_CONTAINS_INC_UPDATE(DEVICETYPE__USB) || IS_DEVICE_CONTAINS_FULL_UPDATE(DEVICETYPE__USB))
         if (IS_DEVICE_CONTAINS_FULL_UPDATE(DEVICETYPE__USB))
         {
            AvailableUpdatesRequestType requestType = (_diagnosticModeON ? AVAILABLE_UPDATES_REQUEST_TYPE__INITIAL_REQUEST_BY_DIAGNOSTIC : AVAILABLE_UPDATES_REQUEST_TYPE__INITIAL_REQUEST);
            _navMiddleware.requestAvailableUpdates(deviceId, path, requestType);
            retVal = true;
            mapUpadateLaunchFromNaviHMI = true;
         }
         else
         {
            ETG_TRACE_ERR(("USB device does not contain any updates!"));
            //To check
            //POST_MSG((COURIER_MESSAGE_NEW(NAVDATAUPDATE_InternalUpdateStatusUpdMsg)(NAVDATAUPDATE__INTERNALUPDATESTATUS__USB_FAST_REMOVED)));
            releaseSpmMapUpdateLock();
         }
         break;
      }
      default:
      {
         ETG_TRACE_ERR(("Unhandled deviceType!"));
      }
   }

   return retVal;
}


bool HMIModelNavDataUpdate::onCourierMessage(const NAVDATAUPDATE_DiagPerformFullUpdateViaUSBReqMsg& oMsg)
{
   COURIER_UNUSED(oMsg);
   ETG_TRACE_USR4(("HMIModelNavDataUpdate::onCourierMessage(NAVDATAUPDATE_DiagPerformFullUpdateViaUSBReqMsg)"));
   if (_diagnosticModeON)
   {
      POST_MSG((COURIER_MESSAGE_NEW(NAVDATAUPDATE_DiagFullUpdateViaUSBAckUpdMsg)(NAVDATAUPDATE__DIAG_RESULT__DECLINED__UPDATE_IN_PROGRESS)));
   }
   else
   {
      setDiagnosticMode(DIAGNOSTIC_ON);
      // send response accepting request
      POST_MSG((COURIER_MESSAGE_NEW(NAVDATAUPDATE_DiagFullUpdateViaUSBAckUpdMsg)(NAVDATAUPDATE__DIAG_RESULT__ACCEPTED)));

      if ((NAVDATAUPDATE__UPDATESTATUS__AVAILABLE_UPDATES_STARTED != _currentUpdateStatus)
            && (NAVDATAUPDATE__UPDATESTATUS__AVAILABLE_UPDATES_IN_PROGRESS != _currentUpdateStatus)
            && (NAVDATAUPDATE__UPDATESTATUS__DOWNLOAD_STARTED != _currentUpdateStatus)
            && (NAVDATAUPDATE__UPDATESTATUS__DOWNLOAD_FINISHED != _currentUpdateStatus)
            && (NAVDATAUPDATE__UPDATESTATUS__INSTALLATION_STARTED != _currentUpdateStatus)
            && (NAVDATAUPDATE__UPDATESTATUS__INSTALLATION_FINISHED != _currentUpdateStatus))
      {
         ETG_TRACE_USR4(("[Diag Mode] check USB availability and request AvailableUpdates."));
         // FIXME: update will happen in Diag Mode even with INC_UPDATE
         if (!requestAvailableUpdates(NAVDATAUPDATE__DEVICETYPE__USB))
         {
            ETG_TRACE_USR4(("[Diag Mode] USB is not yet plugged or invalid data."));
         }
      }
   }
   return true;
}


bool HMIModelNavDataUpdate::onCourierMessage(const NAVDATAUPDATE_SpmStartMapUpdateLockResMsg& oMsg)
{
   ETG_TRACE_USR4(("HMIModelNavDataUpdate::onCourierMessage(NAVDATAUPDATE_SpmStartMapUpdateLockResMsg(%d))", oMsg.GetIsSuccessfull()));

   if (_partitionLockRequested)
   {
      if (oMsg.GetIsSuccessfull())
      {
         if (_infoStore.isSWUpdating())
         {
            ETG_TRACE_ERR(("This should not happen!"));
            _partitionLockError = PARTITION_LOCK_ERR___SW_UPDATING;
            updatePopupText(MAP_UPDATE_ONGOING);
            setPopupVisibility(MAP_UPDATE_ERROR_POPUP, true);
         }
         else
         {
            // Successfully locked, perform requestAvailableUpdates to MDW.
            performRequestAvailableUpdates(_deviceType, std::string(""));

            // clear _deviceType
            _deviceType = NAVDATAUPDATE__DEVICETYPE__NONE;
         }
      }
      else
      {
         _partitionLockError = PARTITION_LOCK_ERR___LOCK_FAILED;
         if (true == NeedTOEnteredInMapUpdate)
         {
            NeedTOEnteredInMapUpdate = false;
            setPopupVisibility(MAP_UPDATE_PROGRESS_POPUP, false);
         }
         POST_MSG((COURIER_MESSAGE_NEW(NAVDATAUPDATE_UpdateBlockedBySWUpdateUpdMsd)()));
      }
   }

   return true;
}


void HMIModelNavDataUpdate::releaseSpmMapUpdateLock()
{
   mapUpadateLaunchFromNaviHMI = false;
   if (_PARTITION_LOCK_ENABLED)
   {
      if (_partitionLockRequested)
      {
         ETG_TRACE_USR4(("HMIModelNavDataUpdate::releaseSpmMapUpdateLock()"));
         POST_MSG((COURIER_MESSAGE_NEW(NAVDATAUPDATE_SpmMapUpdateUnLockReqMsg)()));
         _partitionLockRequested = false;
      }
      else
      {
         ETG_TRACE_USR4(("MapUpdateLock was released or did not request the lock!"));
      }
   }
}


void HMIModelNavDataUpdate::setDiagnosticMode(bool isDiagON)
{
   ETG_TRACE_USR4(("HMIModelNavDataUpdate::setDiagnosticMode(%d)", isDiagON));
   _diagnosticModeON = true;
   if (isDiagON != _diagnosticModeON)
   {
      _diagnosticModeON = isDiagON;
   }
   else
   {
      ETG_TRACE_USR4(("Same DiagnosticMode is already set!"));
   }
}


/** Only for failed cases */
void HMIModelNavDataUpdate::sendUpdateResultToService(enNavDataUpdate_DiagFullUpdateResult result)
{
   ETG_TRACE_USR4(("HMIModelNavDataUpdate::sendUpdateResultToService(diagON %d, result %d)", _diagnosticModeON, result));
   if (_diagnosticModeON)
   {
      POST_MSG((COURIER_MESSAGE_NEW(NAVDATAUPDATE_DiagFullUpdateViaUSBResultUpdMsg)(result)));
      setDiagnosticMode(DIAGNOSTIC_OFF);
   }
}


// Property update for USB prepration result
void HMIModelNavDataUpdate::onPropertyUpdatePreparationStatusChanged()
{
   const NavDataUpdateUSBDeviceStatus deviceStatus = _navMiddleware.getPreparedUSBUpdateDeviceResult();
   ETG_TRACE_USR4(("HMIModelNavDataUpdate::onPropertyUpdatePreparationStatusChanged(%d)", deviceStatus));
   switch (deviceStatus)
   {
      case NAVDATAUPDATE_USB_DEVICE_STATUS__SUCCESS:
      case NAVDATAUPDATE_USB_DEVICE_STATUS__FAILURE:
      case NAVDATAUPDATE_USB_DEVICE_STATUS__OUT_OF_SPACE_ERROR:
      case NAVDATAUPDATE_USB_DEVICE_STATUS__REMOVED_ERROR:
      {
         //_exportStatus = getMappedExportStatus(deviceStatus);
         //POST_MSG((COURIER_MESSAGE_NEW(NAVDATAUPDATE_ExportToUsbResultUpdMsg)(_exportStatus)));
         break;
      }
      default:
         break;
   }
}


void HMIModelNavDataUpdate::onPropertyNavDataUpdateInfoChanged()
{
   if (!mapUpadateLaunchFromNaviHMI)
   {
      ETG_TRACE_USR4(("HMIModelNavDataUpdate::onPropertyNavDataUpdateInfoChanged:Map Update Triggerd By Some Other External client "));
      return;
   }
   const navmiddleware::NavDataUpdateOverviewInfo& navDataUpdateOverviewInfo = _navMiddleware.getAvailableUpdateInfo();
   ETG_TRACE_USR4(("HMIModelNavDataUpdate::onPropertyNavDataUpdateInfoChanged()\n%s", navDataUpdateOverviewInfo.toString().c_str()));
   const bool updatesFound = (navDataUpdateOverviewInfo.getTotalNumberOfUpdates() > 0);

   ETG_TRACE_USR4(("_diagnosticModeON %d, updatesFound %d", _diagnosticModeON, updatesFound));

   if (!updatesFound)
   {
      releaseSpmMapUpdateLock();
      updatePopupText(MAP_UPDATE_FILE_OLDER);
      _navMiddleware.requestNavigationDataVersionInfo();
      setPopupVisibility(MAP_UPDATE_ERROR_POPUP, true);
   }
   else
   {
      if (!_diagnosticModeON)
      {
         ETG_TRACE_USR4(("[InfoChanged:Diagnostic Mode OFF ] requestUpdateBySelection()"));
         _navMiddleware.requestNavigationDataVersionInfo();
         updatePopupText(MAP_UPDATE_INTERACTIVE);
         setPopupVisibility(MAP_UPDATE_AVAILABLE_POPUP, true);
      }
   }

   if (_diagnosticModeON)
   {
      if (updatesFound)
      {
         ETG_TRACE_USR4(("[InfoChanged:Diagnostic Mode ON ] requestUpdateBySelection()"));
         // confirm the update automatically
         triggerFullMapUpdate();
      }
      else
      {
         ETG_TRACE_USR4(("[Diagnostic Mode] no updates found!"));
         sendUpdateResultToService(NAVDATAUPDATE__DIAG_RESULT__FAILED__NO_UPDATES_FOUND);
      }
   }
}


void HMIModelNavDataUpdate::triggerFullMapUpdate()
{
   ETG_TRACE_USR4(("HMIModelNavDataUpdate::triggerFullMapUpdate()"));

   NeedTOEnteredInMapUpdate = true;

   setPopupVisibility(MAP_UPDATE_PROGRESS_POPUP, true);
   _navMiddleware.requestUpdateBySelection("");
   if (false == EXT_bIsNaviAppRunning)
   {
      POST_MSG((COURIER_MESSAGE_NEW(ContextSwitchReqMsg)(CONTEXT_SWITCH_FROM_NONE, CONTEXT_SWITCH_TO_MAP)));
   }
}


bool HMIModelNavDataUpdate::onCourierMessage(const NAVDATAUPDATE_Nav_FullMapUpdateYesReqMsg& oMsg)
{
   ETG_TRACE_USR4(("HMIModelNavDataUpdate::onCourierMessage(NAVDATAUPDATE_Nav_FullMapUpdateYesReqMsg), Yes button press"));
   _activePopup = MAP_UPDATE_POPUP_NONE;

   // This courier message is fired on the exit of Pfo_MapUpdate_Interactive pop-up and not immediately on press of yes button in the pop-up.
   // This is because invocation of _navMiddleware.requestUpdateBySelection() on press of yes button lead to a delay of ~3 seconds for pop-up closure.
   bool isMapUpdateTriggered = oMsg.GetIsMapUpdateTriggered();
   if (true == isMapUpdateTriggered)
   {
      triggerFullMapUpdate();
   }
   return true;
}


bool HMIModelNavDataUpdate::onCourierMessage(const NAVDATAUPDATE_Nav_FullMapUpdateNoReqMsg& oMsg)
{
   //Download canceled by user Before Started
   ETG_TRACE_USR4(("NavigationPopups#Popups#Nav_Upgr_MapUpdateLaunched_ToastPopup press No"));
   _activePopup = MAP_UPDATE_POPUP_NONE;

   releaseSpmMapUpdateLock();
   NeedTOEnteredInMapUpdate = false;
   return true;
}


bool HMIModelNavDataUpdate::onCourierMessage(const NAVDATAUPDATE_Nav_ErrorMsgDuringUpdatePressOkReqMsg& oMsg)
{
   ETG_TRACE_USR4(("HMIModelNavDataUpdate::onCourierMessage(NAVDATAUPDATE_Nav_ErrorMsgDuringUpdatePressOkReqMsg)"));
   _activePopup = MAP_UPDATE_POPUP_NONE;
   return true;
}


void HMIModelNavDataUpdate::onPropertyNavDataUpdateStatusChanged()
{
   const NavDataUpdateStatus& navDataUpdateStatus = _navMiddleware.getUpdateStatus();
   ETG_TRACE_USR4(("onPropertyNavDataUpdateStatusChanged:navDataUpdateStatus: %d", ETG_CENUM(NavDataUpdateStatus::DataUpdateStatus, navDataUpdateStatus.getDataUpdateStatus())));

   if (NavDataUpdateStatus::DATAUPDATESTATUS__FULL_UPDATE_REQUIRED == navDataUpdateStatus.getDataUpdateStatus())
   {
      ETG_TRACE_USR4(("HMIModelNavDataUpdate::onPropertyNavDataUpdateStatusChanged:DATAUPDATESTATUS__FULL_UPDATE_REQUIRED"));
      if (_activePopup != MAP_UPDATE_ERROR_POPUP)
      {
         updatePopupText(MAP_UPDATE_INCONSISTENT);
         setPopupVisibility(MAP_UPDATE_ERROR_POPUP, true);
      }
      _currentUpdateStatus = getMappedUpdateStatus(navDataUpdateStatus.getDataUpdateStatus());

      setNavigationGadgetStatus(NAVI_GADGET_STATUS__NO_DATA, false, _infoStore.getVehicleProfile());
      if (true == EXT_bIsNaviAppRunning)
      {
         // Move navi app to BG and request master to bring last active app to FG.
         POST_MSG((COURIER_MESSAGE_NEW(SwitchContextFromNavDataUpdateReqMsg)()));
      }
   }
   else if (!mapUpadateLaunchFromNaviHMI)
   {
      ETG_TRACE_USR4(("HMIModelNavDataUpdate::onPropertyNavDataUpdateStatusChanged:Map Update Triggerd By Some Other External client "));
      return;
   }
   else
   {
      ETG_TRACE_USR4(("HMIModelNavDataUpdate::onPropertyNavDataUpdateStatusChanged:Map Update Triggerd By Navi HMI "));
      const unsigned int remainingTimeInSec = navDataUpdateStatus.getRemainingTimeInSeconds().getValue();
      const Courier::Float dataUpdateProgress = static_cast<Courier::Float>(navDataUpdateStatus.getProgress());

      bool updateStatusToSM = false;

      switch (navDataUpdateStatus.getDataUpdateStatus())
      {
         case NavDataUpdateStatus::DATAUPDATESTATUS__AVAILABLE_UPDATES_STARTED:
         case NavDataUpdateStatus::DATAUPDATESTATUS__AVAILABLE_UPDATES_IN_PROGRESS:
         case NavDataUpdateStatus::DATAUPDATESTATUS__AVAILABLE_UPDATES_FINISHED:
         case NavDataUpdateStatus::DATAUPDATESTATUS__DOWNLOAD_STARTED:
         case NavDataUpdateStatus::DATAUPDATESTATUS__DOWNLOAD_FINISHED:
         case NavDataUpdateStatus::DATAUPDATESTATUS__INSTALLATION_STARTED:
         case NavDataUpdateStatus::DATAUPDATESTATUS__INSTALLATION_FINISHED:
         {
            updateStatusToSM = true;
            ETG_TRACE_USR4(("HMIModelNavDataUpdate::onPropertyNavDataUpdateStatusChanged:Nothing to be done..No popups currently for these status"));
            break;
         }
         case NavDataUpdateStatus::DATAUPDATESTATUS__AVAILABLE_UPDATES_ABORTED_BY_ERROR:
         case NavDataUpdateStatus::DATAUPDATESTATUS__AVAILABLE_UPDATES_ABORTED_BY_USER:
         case NavDataUpdateStatus::DATAUPDATESTATUS__DOWNLOAD_ABORTED_BY_ERROR:
         case NavDataUpdateStatus::DATAUPDATESTATUS__UPDATE_ABORTED_BY_USER:
         case NavDataUpdateStatus::DATAUPDATESTATUS__INSTALLATION_STOPPED:
         case NavDataUpdateStatus::DATAUPDATESTATUS__UPDATE_SOURCE_ERROR:
         case NavDataUpdateStatus::DATAUPDATESTATUS__USB_REMOVED:
         case NavDataUpdateStatus::DATAUPDATESTATUS__UNKNOWN:
         case NavDataUpdateStatus::DATAUPDATESTATUS__INCOMPATIBLE_UPDATE_PACKAGE:
         case NavDataUpdateStatus::DATAUPDATESTATUS__SUBSCRIPTION_EXPIRED:
         case NavDataUpdateStatus::DATAUPDATESTATUS__SUBSCRIPTION_VALIDATION_IMPOSSIBLE:
         case NavDataUpdateStatus::DATAUPDATESTATUS__OUT_OF_SPACE:
         case NavDataUpdateStatus::DATAUPDATESTATUS__INVALID_UPDATE_PACKAGE:
         case NavDataUpdateStatus::DATAUPDATESTATUS__COMPATIBILITY_MATRIX_NOT_FOUND:
         case NavDataUpdateStatus::DATAUPDATESTATUS__COMPATIBILITY_MATRIX_OUTDATED:
         {
            updateStatusToSM = true;
            _navMiddleware.requestNavigationDataVersionInfo();
            setDiagnosticMode(DIAGNOSTIC_OFF);
            ETG_TRACE_USR4(("HMIModelNavDataUpdate::onPropertyNavDataUpdateStatusChanged:nav data update(%d)", _infoStore.isNavDataUpdateRunning()));
            if (_infoStore.isNavDataUpdateRunning() && NavDataUpdateStatus::DATAUPDATESTATUS__USB_REMOVED == navDataUpdateStatus.getDataUpdateStatus())
            {
               ETG_TRACE_USR4(("HMIModelNavDataUpdate::onPropertyNavDataUpdateStatusChanged:DATAUPDATESTATUS__USB_REMOVED"));
               updatePopupText(MAP_UPDATE_USB_UNPLUGGED);
               setPopupVisibility(MAP_UPDATE_ERROR_POPUP, true);
            }
            else if ((NavDataUpdateStatus::DATAUPDATESTATUS__AVAILABLE_UPDATES_ABORTED_BY_ERROR == navDataUpdateStatus.getDataUpdateStatus() || NavDataUpdateStatus::DATAUPDATESTATUS__AVAILABLE_UPDATES_ABORTED_BY_USER == navDataUpdateStatus.getDataUpdateStatus()))
            {
               ETG_TRACE_USR4(("HMIModelNavDataUpdate::onPropertyNavDataUpdateStatusChanged:Same Version Or Older version"));
               updatePopupText(MAP_UPDATE_FILE_OLDER);
               setPopupVisibility(MAP_UPDATE_ERROR_POPUP, true);
            }
            else if (NavDataUpdateStatus::DATAUPDATESTATUS__INCOMPATIBLE_UPDATE_PACKAGE == navDataUpdateStatus.getDataUpdateStatus())
            {
               ETG_TRACE_USR4(("HMIModelNavDataUpdate::onPropertyNavDataUpdateStatusChanged:DATAUPDATESTATUS__INCOMPATIBLE_UPDATE_PACKAGE"));
               updatePopupText(MAP_UPDATE_NOT_COMPATIBLE);
               setPopupVisibility(MAP_UPDATE_ERROR_POPUP, true);
            }
            else if (NavDataUpdateStatus::DATAUPDATESTATUS__INVALID_UPDATE_PACKAGE == navDataUpdateStatus.getDataUpdateStatus())
            {
               ETG_TRACE_USR4(("HMIModelNavDataUpdate::onPropertyNavDataUpdateStatusChanged:DATAUPDATESTATUS__INVALID_UPDATE_PACKAGE"));
               updatePopupText(MAP_UPDATE_FILE_NOT_VALID);
               setPopupVisibility(MAP_UPDATE_ERROR_POPUP, true);
            }
            else
            {
               ETG_TRACE_USR4(("HMIModelNavDataUpdate::onPropertyNavDataUpdateStatusChanged:Nav_InstallationFailure_InteractiveCenterPopup"));
               updatePopupText(MAP_UPDATE_FILE_INSTALLATION_FAILURE);
               setPopupVisibility(MAP_UPDATE_ERROR_POPUP, true);
            }
            NeedTOEnteredInMapUpdate = false;
            releaseSpmMapUpdateLock();
            break;
         }
         case NavDataUpdateStatus::DATAUPDATESTATUS__DOWNLOAD_IN_PROGRESS:
         case NavDataUpdateStatus::DATAUPDATESTATUS__INSTALLATION_IN_PROGRESS:
         {
            setRemainingTimeValue(remainingTimeInSec);
            sendPopupDynamicTextNavDataUpdate();

            setPopupDynamicProgressValue(dataUpdateProgress);
            break;
         }
         case NavDataUpdateStatus::DATAUPDATESTATUS__DOWNLOAD_UPDATE_SUCCESSFULLY_FINISHED:
         {
            updatePopupText(MAP_UPDATE_SUCCESSFULL);
            setPopupVisibility(MAP_UPDATE_ERROR_POPUP, true);
            updateStatusToSM = true;
            NeedTOEnteredInMapUpdate = false;
            _navMiddleware.requestNavigationDataVersionInfo();
            releaseSpmMapUpdateLock();
            setDiagnosticMode(DIAGNOSTIC_OFF);
            break;
         }
         default:
         {
            ETG_TRACE_USR4(("Unhandled NavDataUpdateStatus %d", navDataUpdateStatus.getDataUpdateStatus()));
            break;
         }
      }

      if (updateStatusToSM)
      {
         _currentUpdateStatus = getMappedUpdateStatus(navDataUpdateStatus.getDataUpdateStatus());
      }
   }
}


void HMIModelNavDataUpdate::onPropertyNavDataVersionInfoChanged()
{
   const std::string UNKNOWN_VERSION = "0_0_0";
   std::string currentVersion = _navMiddleware.getNavigationDataVersionInfo().getCurrentVersion();
   ETG_TRACE_USR4(("HMIModelNavDataUpdate::onPropertyNavDataVersionInfoChanged(), Current version : %s", currentVersion.c_str()));

   (*_navDataUpdatePopupText).mCurrentNavDataVersion = (!currentVersion.empty() && currentVersion.compare(UNKNOWN_VERSION))
         ? currentVersion.c_str() : LANGUAGE_STRING(TextId_0x0AC0, "Unknown").GetCString();
   (*_navDataUpdatePopupText).mNewNavDataVersion = _navMiddleware.getAvailableUpdateInfo().getNewVersion().c_str();
   _navDataUpdatePopupText.MarkItemModified(ItemKey::NavDataPopupText::CurrentNavDataVersionItem);
   _navDataUpdatePopupText.MarkItemModified(ItemKey::NavDataPopupText::NewNavDataVersionItem);
   sendPopupDynamicTextNavDataUpdate();
}


void HMIModelNavDataUpdate::setRemainingTimeValue(unsigned int remainingTimeInSec)
{
   ETG_TRACE_USR4(("HMIModelNavDataUpdate::setRemainingTimeValue(%d)", remainingTimeInSec));
   static unsigned int preRemainingTimeInSec = remainingTimeInSec;
   static bool first = true;
   if ((remainingTimeInSec != preRemainingTimeInSec) || (first))
   {
      unsigned int remainingTimeInMin = remainingTimeInSec / 60;
      if (remainingTimeInSec == 0)
      {
         // TODO: handle invalid remainingTime??
         remainingTimeInMin = 0;
      }
      else if (remainingTimeInMin == 0)
      {
         remainingTimeInMin = 1;
      }

      App::DataModel::util::MultiLangTextBuilder remainingTimeInMinStr;
      remainingTimeInMinStr.AddData(remainingTimeInMin);

      (*_navDataUpdatePopupText).mRemainingTime = remainingTimeInMinStr.GetData();
      _navDataUpdatePopupText.MarkItemModified(ItemKey::NavDataPopupText::RemainingTimeItem);

      preRemainingTimeInSec = remainingTimeInSec;
      if (first)
      {
         first = false;
      }
   }
}


void HMIModelNavDataUpdate::sendPopupDynamicTextNavDataUpdate()
{
   if (_navDataUpdatePopupText.HasModifiedItems())
   {
      if (!(_navDataUpdatePopupText.SendUpdate()))
      {
         ETG_TRACE_ERR(("_navDataUpdatePopupText update failed"));
      }
   }
}


bool HMIModelNavDataUpdate::onCourierMessage(const NAVDATAUPDATE_PerformFullUpdateViaUSBReqMsg& oMsg)
{
   COURIER_UNUSED(oMsg);
   ETG_TRACE_USR4(("HMIModelNavDataUpdate::onCourierMessage(NAVDATAUPDATE_PerformFullUpdateViaUSBReqMsg)"));

   if ((NAVDATAUPDATE__UPDATESTATUS__AVAILABLE_UPDATES_STARTED != _currentUpdateStatus)
         && (NAVDATAUPDATE__UPDATESTATUS__AVAILABLE_UPDATES_IN_PROGRESS != _currentUpdateStatus)
         && (NAVDATAUPDATE__UPDATESTATUS__DOWNLOAD_STARTED != _currentUpdateStatus)
         && (NAVDATAUPDATE__UPDATESTATUS__DOWNLOAD_FINISHED != _currentUpdateStatus)
         && (NAVDATAUPDATE__UPDATESTATUS__INSTALLATION_STARTED != _currentUpdateStatus)
         && (NAVDATAUPDATE__UPDATESTATUS__INSTALLATION_FINISHED != _currentUpdateStatus))
   {
      ETG_TRACE_USR4(("check USB availability and request AvailableUpdates."));
      if (!requestAvailableUpdates(NAVDATAUPDATE__DEVICETYPE__USB))
      {
         ETG_TRACE_USR4((" USB is not yet plugged or invalid data."));
         return false;
      }
   }
   return true;
}


bool HMIModelNavDataUpdate::onPropertyUpdateSpmSystemStateChanged()
{
   ETG_TRACE_USR1(("HMIModelNavDataUpdate::onPropertyUpdateSpmSystemStateChanged()(%d)", _infoStore.getSpmSystemState()));

   switch (_infoStore.getSpmSystemState())
   {
      case SPM_SYSTEM_STATE__BACKGROUND:
         _navMiddleware.setSystemState(navmiddleware::ConfigurationInfos::SYSTEM_STATE__BACKGROUND);
         break;
      case SPM_SYSTEM_STATE__PREPARE_SHUTDOWN:
         _navMiddleware.setSystemState(navmiddleware::ConfigurationInfos::SYSTEM_STATE__PREPARE_SHUTDOWN);
         break;
      case SPM_SYSTEM_STATE__SHUTDOWN:
         _navMiddleware.setSystemState(navmiddleware::ConfigurationInfos::SYSTEM_STATE__SHUTDOWN);
         break;
      case SPM_SYSTEM_STATE__SUSPEND:
         _navMiddleware.setSystemState(navmiddleware::ConfigurationInfos::SYSTEM_STATE__SUSPEND);
         break;
      case SPM_SYSTEM_STATE__STANDBY:
         _navMiddleware.setSystemState(navmiddleware::ConfigurationInfos::SYSTEM_STATE__STANDBY);
         break;
      case SPM_SYSTEM_STATE__OFF:
         _navMiddleware.setSystemState(navmiddleware::ConfigurationInfos::SYSTEM_STATE__OFF);
         break;
      case SPM_SYSTEM_STATE__DOWNLOAD:
         _navMiddleware.setSystemState(navmiddleware::ConfigurationInfos::SYSTEM_STATE__DOWNLOAD);
         break;
      case SPM_SYSTEM_STATE__ON:
         _navMiddleware.setSystemState(navmiddleware::ConfigurationInfos::SYSTEM_STATE__ON);
         break;
      case SPM_SYSTEM_STATE__DOOR_OPEN:
         _navMiddleware.setSystemState(navmiddleware::ConfigurationInfos::SYSTEM_STATE__DOOR_OPEN);
         break;
      case SPM_SYSTEM_STATE__IGNITION:
         _navMiddleware.setSystemState(navmiddleware::ConfigurationInfos::SYSTEM_STATE__IGNITION);
         break;
      case SPM_SYSTEM_STATE__DIAGNOSIS:
         _navMiddleware.setSystemState(navmiddleware::ConfigurationInfos::SYSTEM_STATE__DIAGNOSIS);
         break;
      case SPM_SYSTEM_STATE__PROFILE:
         _navMiddleware.setSystemState(navmiddleware::ConfigurationInfos::SYSTEM_STATE__PROFILE);
         break;
      case SPM_SYSTEM_STATE__MMI_STANDBY_PWR_SAVE_1:
         _navMiddleware.setSystemState(navmiddleware::ConfigurationInfos::SYSTEM_STATE__MMI_STANDBY_PWR_SAVE_1);
         break;
      case SPM_SYSTEM_STATE__MMI_STANDBY_PWR_SAVE_2:
         _navMiddleware.setSystemState(navmiddleware::ConfigurationInfos::SYSTEM_STATE__MMI_STANDBY_PWR_SAVE_2);
         break;
      case SPM_SYSTEM_STATE__MMI_STANDBY:
         _navMiddleware.setSystemState(navmiddleware::ConfigurationInfos::SYSTEM_STATE__MMI_STANDBY);
         break;
      case SPM_SYSTEM_STATE__MMI_ON:
         _navMiddleware.setSystemState(navmiddleware::ConfigurationInfos::SYSTEM_STATE__MMI_ON);
         break;
      case SPM_SYSTEM_STATE__MMI_STANDBY_RESCTRICTED:
         _navMiddleware.setSystemState(navmiddleware::ConfigurationInfos::SYSTEM_STATE__MMI_STANDBY_RESCTRICTED);
         break;
      case SPM_SYSTEM_STATE__MMI_ON_DIAG:
         _navMiddleware.setSystemState(navmiddleware::ConfigurationInfos::SYSTEM_STATE__MMI_ON_DIAG);
         break;
      case SPM_SYSTEM_STATE__MMI_ON_TEL:
         _navMiddleware.setSystemState(navmiddleware::ConfigurationInfos::SYSTEM_STATE__MMI_ON_TEL);
         break;
      case SPM_SYSTEM_STATE__MMI_ON_SWDL:
         _navMiddleware.setSystemState(navmiddleware::ConfigurationInfos::SYSTEM_STATE__MMI_ON_SWDL);
         break;
      case SPM_SYSTEM_STATE__MMI_STANDBY_CUSTOMER_SWDL:
         _navMiddleware.setSystemState(navmiddleware::ConfigurationInfos::SYSTEM_STATE__MMI_STANDBY_CUSTOMER_SWDL);
         break;
      case SPM_SYSTEM_STATE__MMI_STANDBY_PWR_SAVE:
         _navMiddleware.setSystemState(navmiddleware::ConfigurationInfos::SYSTEM_STATE__MMI_STANDBY_PWR_SAVE);
         break;
      case SPM_SYSTEM_STATE__OVER_TMP:
         _navMiddleware.setSystemState(navmiddleware::ConfigurationInfos::SYSTEM_STATE__OVER_TMP);
         break;
      case SPM_SYSTEM_STATE__SAFE:
         _navMiddleware.setSystemState(navmiddleware::ConfigurationInfos::SYSTEM_STATE__SAFE);
         break;
      case SPM_SYSTEM_STATE__STATE_MAX:
         _navMiddleware.setSystemState(navmiddleware::ConfigurationInfos::SYSTEM_STATE__STATE_MAX);
         break;
      default:
         break;
   }
   return true;
}


void HMIModelNavDataUpdate::setPopupVisibility(MapUpdatePopupType popupType, bool isVisible)
{
   ETG_TRACE_USR4(("HMIModelNavDataUpdate::setPopupVisibility(), activePopup : %d, requestPopup : %d, isVisible : %d", _activePopup, popupType, isVisible));

   static const char* mapUpdateSceneNames[] =
   {
      "",
      "AppHmi_NavigationModule#NavigationScenes#Pfo_MapUpdate_Interactive",
      "AppHmi_NavigationModule#NavigationScenes#Pfo_MapUpdate_Progress",
      "AppHmi_NavigationModule#NavigationScenes#Pfo_MapUpdate_Error"
   };

   if (MAP_UPDATE_POPUP_NONE != popupType)
   {
      if (true == isVisible)
      {
         if (_activePopup != popupType)
         {
            if (MAP_UPDATE_POPUP_NONE != _activePopup)
            {
               POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Hide, Courier::ViewId(mapUpdateSceneNames[_activePopup]))));
            }
            _activePopup = popupType;
            if (!_infoStore.isNonNaviVariant())
            {
               POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId(mapUpdateSceneNames[_activePopup]))));
            }
            if ((MAP_UPDATE_ERROR_POPUP == _activePopup) && (true == NeedTOEnteredInMapUpdate))
            {
               NeedTOEnteredInMapUpdate = false;
            }
         }
      }
      else
      {
         if (_activePopup == popupType)
         {
            POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Hide, Courier::ViewId(mapUpdateSceneNames[_activePopup]))));
            _activePopup = MAP_UPDATE_POPUP_NONE;
         }
      }
   }
}


void HMIModelNavDataUpdate::updatePopupText(Candera::UInt8 MsgID)
{
   ETG_TRACE_USR4(("HMIModelNavDataUpdate::updatePopupText (%d)", MsgID));

   const Candera::String ERRORMSGS[10] =
   {
      LANGUAGE_STRING(TextId_0x146E, "Map update was successful."),
      LANGUAGE_STRING(TextId_0x1470, "Map update is in progress. Please wait, do not remove the USB flash drive."),
      LANGUAGE_STRING(TextId_0x1919, "Map database version is up to date. No update is required."),
      LANGUAGE_STRING(TextId_0x1474, "Do you want to update the map data?"),
      LANGUAGE_STRING(TextId_0x1475, "Map update is starting."),
      LANGUAGE_STRING(TextId_0x1476, "Map data is inconsistent. Please insert a USB flash drive containing valid map data."),
      LANGUAGE_STRING(TextId_0x1477, "Map update failed because the USB flash drive was unplugged. Please plug in the USB flash drive and start the update again."),
      LANGUAGE_STRING(TextId_0x1479, "Map data is incompatible"),
      LANGUAGE_STRING(TextId_0x147A, "Map data is invalid. Please insert a USB flash drive containing valid map data."),
      LANGUAGE_STRING(TextId_0x147B, "Map update failed. Please try again."),
   };

   (*_navDataUpdatePopupText).mErrorMsgLine1 = ERRORMSGS[MsgID];
   _navDataUpdatePopupText.MarkItemModified(ItemKey::NavDataPopupText::ErrorMsgLine1Item);
   _navDataUpdatePopupText.SendUpdate();
}


void HMIModelNavDataUpdate::setPopupDynamicProgressValue(float dataUpdateProgress)
{
   ETG_TRACE_USR4(("HMIModelNavDataUpdate::Map Update Progress value (%d)", dataUpdateProgress));
   if (dataUpdateProgress != (*_navDataUpdatePopupText).mProgressValue)
   {
      (*_navDataUpdatePopupText).mProgressValue = dataUpdateProgress;
      _navDataUpdatePopupText.MarkItemModified(ItemKey::NavDataPopupText::ProgressValueItem);
      _navDataUpdatePopupText.SendUpdate();
   }
}


///Check and delete
bool HMIModelNavDataUpdate::onCourierMessage(const UpdateNAVDATAUPDATE_PopupTextMsg& oMsg)
{
   updatePopupText(oMsg.GetPopupID());
   return true;
}


#endif // HALL_TO_MDW_COM
