/**************************************************************************************
* @file         : HMIModelMap.cpp
* @author       :
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) -2018 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include "MapScreenDataUtils.h"
//#include "../Common/Util/DatabindingUtils.h"
#include "../Common/Util/NavMiddlewareUtils.h"
#include "../Settings/NaviSettingsUtil.h"
#include <iomanip>


#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/MapScreenDataUtils.cpp.trc.h"
#endif

#if defined (HALL_TO_MDW_COM)

static const unsigned int NIGHTMODE_OFFSET_COMPASS_DIRECTION_ICON = 16;
static const unsigned int NIGHTMODE_OFFSET_TRAFFIC_SERVICE_ICON = 3;
static const int SHOW_DESTINATION_DAY_MODE_FLAG_ICON = 0;
static const int SHOW_WAYPOINT_FLAG_ICON = 1;
static const int SHOW_DESTINATION_NIGHT_MODE_FLAG_ICON = 2;
static const int SHOW_NONE_FLAG = 3;
static const float TBT_WITH_AUDIOBAR_POSITION_Y = 56.0f;
static const float EXIT_TEXT_OFFSET_POSITION_X = 9.0f;
static const float EXIT_TEXT_OFFSET_POSITION_Y = 1.0f;
static const int NO_MANEUVER_SYMBOL_INDEX = 55;
static const int FOLLOW_SYMBOL_INDEX = 2;
static const int OFF_ROAD_SYMBOL_INDEX = 39; //Starting Index for OffRoad
static const int TIME_STR_MAX_LENGTH = 8;   // For example, 12:00pm for ETA calculation cabin only

const float RedColor[4] = { 1.0F, 0.0F, 0.0F, 1.0F };
const float BlackColor[4] = { 0.0F, 0.0F, 0.0F, 1.0F };
const float GreenColor[4] = { 0.0F, 1.0F, 0.0F, 1.0F };
const float GreyColor[4] = { 0.6667F, 0.6667F, 0.7333F, 1.0F };

const float compassStepSize = 22.5f;
const int maxNumCompassSteps = 16;

using namespace navmiddleware;


DataBindingItem<CompassButtonDataBindingSource>& getCompassButtonInfoData()
{
   static DataBindingItem<CompassButtonDataBindingSource> sCompassButtonData;
   return sCompassButtonData;
}


DataBindingItem<EnableOkButtonDataBindingSource>& getOkButtonInfoData()
{
   static DataBindingItem<EnableOkButtonDataBindingSource> sOkButtonData;
   return sOkButtonData;
}


DataBindingItem<MapScreenDataDataBindingSource>& getMapScreenData()
{
   static DataBindingItem<MapScreenDataDataBindingSource> sMapScreenData;
   return sMapScreenData;
}


DataBindingItem<MapoutScreenDataDataBindingSource>& getMapoutScreenData()
{
   static DataBindingItem<MapoutScreenDataDataBindingSource> sMapoutScreenData;
   return sMapoutScreenData;
}


DataBindingItem<ODRDataDataBindingSource>& getODRData()
{
   static DataBindingItem<ODRDataDataBindingSource> sODRData;
   return sODRData;
}


DataBindingItem<NavigationGadgetInfoDataBindingSource>& getNavigationGadgetInfoData()
{
   static DataBindingItem<NavigationGadgetInfoDataBindingSource> sNavigationGadgetInfoData;
   return sNavigationGadgetInfoData;
}


DataBindingItem<NaviMiddlewareStatusDataBindingSource>& getNaviMiddlewareStatusData()
{
   static DataBindingItem<NaviMiddlewareStatusDataBindingSource> sNaviMiddlewareStatusData;
   return sNaviMiddlewareStatusData;
}


bool sendMapScreenData()
{
   bool retVal = true;
   if (getMapScreenData().HasModifiedItems() &&
         (getMapScreenData().SendUpdate() == false))
   {
      ETG_TRACE_ERR(("getMapScreenData() update failed!"));
      retVal = false;
   }
   else
   {
      // to nothing
   }

   return retVal;
}


bool sendMapoutScreenData()
{
   bool retVal = true;
   if (getMapoutScreenData().HasModifiedItems() &&
         (getMapoutScreenData().SendUpdate() == false))
   {
      ETG_TRACE_ERR(("getMapoutScreenData() update failed!"));
      retVal = false;
   }
   else
   {
      // to nothing
   }

   return retVal;
}


void sendODRData()
{
   if (getODRData().HasModifiedItems() &&
         (getODRData().SendUpdate() == false))
   {
      ETG_TRACE_ERR(("getODRData() update failed!"));
   }
   else
   {
      // to nothing
   }
}


void setMapScreenDataMapCameraInfos(const navmiddleware::MapCameraInfos& mapCameraInfos, const navmiddleware::NominalScaleInfos& nominalScaleInfos)
{
   DataBindingItem<MapScreenDataDataBindingSource>& mapScreenData = getMapScreenData();
   // update zoom level
   if ((*mapScreenData).mZoomLevel != mapCameraInfos.getFormattedMapScale().c_str())
   {
      (*mapScreenData).mZoomLevel = mapCameraInfos.getFormattedMapScale().c_str();
      mapScreenData.MarkItemModified(ItemKey::MapScreenData::ZoomLevelItem);
   }

   // update zoomSliderIndex
   unsigned int zoomSliderIndex = 0;
   for (zoomSliderIndex = 0; zoomSliderIndex < nominalScaleInfos.getNominalScaleInfos().size(); zoomSliderIndex++)
   {
      if (mapCameraInfos.getMapScale() <= nominalScaleInfos.getNominalScaleInfos()[zoomSliderIndex].getMapScale())
      {
         break;
      }
   }
   zoomSliderIndex = nominalScaleInfos.getNominalScaleInfos().size() - zoomSliderIndex;
   if ((*mapScreenData).mZoomSliderCurrentValue != (float)zoomSliderIndex)
   {
      (*mapScreenData).mZoomSliderCurrentValue = (float)zoomSliderIndex;
      mapScreenData.MarkItemModified(ItemKey::MapScreenData::ZoomSliderCurrentValueItem);
   }

   // update compass
   const float cameraHeadingDegree = mapCameraInfos.getCameraHeading();
   unsigned int compassHeadingStep = static_cast<unsigned int>((cameraHeadingDegree + COMPASS_STEP_SIZE / 2.) / COMPASS_STEP_SIZE);

   if (compassHeadingStep >= MAX_NUM_COMPASS_STEPS)
   {
      compassHeadingStep = 0;
   }
   else if (compassHeadingStep == 0)
   {
      (*mapScreenData).mCompassArrowDirectionIcon = ImageLoader::getAssetBitmapImage(_pickingCompassDirectionImages[compassHeadingStep]);
      mapScreenData.MarkItemModified(ItemKey::MapScreenData::CompassArrowDirectionIconItem);
   }
   else
   {
      // to nothing
   }

   if (currentCompassHeadingStep[Enum_MAP_VIEW_PRIMARY] != compassHeadingStep)
   {
      currentCompassHeadingStep[Enum_MAP_VIEW_PRIMARY] = compassHeadingStep;
      (*mapScreenData).mCompassArrowDirection = compassHeadingStep;
      mapScreenData.MarkItemModified(ItemKey::MapScreenData::CompassArrowDirectionItem);
      (*mapScreenData).mCompassArrowDirectionIcon = ImageLoader::getAssetBitmapImage(_pickingCompassDirectionImages[compassHeadingStep]);
      mapScreenData.MarkItemModified(ItemKey::MapScreenData::CompassArrowDirectionIconItem);
   }
}


void setSecondaryMapScreenDataMapCameraInfos(const navmiddleware::MapCameraInfos& mapCameraInfos, const navmiddleware::NominalScaleInfos& nominalScaleInfos)
{
   DataBindingItem<MapScreenDataDataBindingSource>& mapScreenData = getMapScreenData();
   // update zoom level for secondary map
   if ((*mapScreenData).mSecZoomLevel != mapCameraInfos.getFormattedMapScale().c_str())
   {
      (*mapScreenData).mSecZoomLevel = mapCameraInfos.getFormattedMapScale().c_str();
      mapScreenData.MarkItemModified(ItemKey::MapScreenData::SecZoomLevelItem);
   }

   // update compass for secondary map
   float cameraHeadingDegree = mapCameraInfos.getCameraHeading();
   unsigned int compassHeadingStep = static_cast<unsigned int>((cameraHeadingDegree + compassStepSize / 2.) / compassStepSize);
   if (compassHeadingStep >= maxNumCompassSteps)
   {
      compassHeadingStep = 0;
   }
   if (currentCompassHeadingStep[Enum_MAP_VIEW_SECONDARY] != compassHeadingStep)
   {
      currentCompassHeadingStep[Enum_MAP_VIEW_SECONDARY] = compassHeadingStep;
      (*mapScreenData).mSecMapCompassArrowDirectionIcon = ImageLoader::getAssetBitmapImage(_pickingCompassDirectionImages[compassHeadingStep]);;
      mapScreenData.MarkItemModified(ItemKey::MapScreenData::SecMapCompassArrowDirectionIconItem);
   }
}


void setMapScreenDataArrivalInfos(const ArrivalInfos& arrivalInfos, bool isArrivalTimeModeActive, InfoStoreBase::EstimatedTimeType estimatedTimeType, bool showEtaOnSecMap)
{
   DataBindingItem<MapScreenDataDataBindingSource>& mapScreenData = getMapScreenData();

   // Fix for Bug 917893 : The Language Translation for ETA text is reverted to fit the text in N_Map_ActiveRG scene for all languages.
   (*mapScreenData).mETATextLabelValue = (isArrivalTimeModeActive == true) ? LANGUAGE_STRING(TEXTID_UNAVAILABLE, "ETA") : TimeOnMapStrings[1];
   mapScreenData.MarkItemModified(ItemKey::MapScreenData::ETATextLabelValueItem);

   // show arrival infos
   const ::std::vector<ArrivalInfos::ArrivalInfo>& arrivalInfo = arrivalInfos.getArrivalInfos();
   if ((arrivalInfos.getCurrentDestinationIndex() < arrivalInfo.size()))
      // ToDo(mup6cob): The below condition is commented since the API isRouteCalcCompletedForAllDestinations() is not returning the expected value.
      // This condition check should be uncommented once the bugs 773903, 918943 and 938172 are resolved.
      // && (true == arrivalInfos.isRouteCalcCompletedForAllDestinations()))
   {
      int destinationIndex = (estimatedTimeType == InfoStore::DESTINATION) ? arrivalInfo.size() - 1 : arrivalInfos.getCurrentDestinationIndex();

      std::string distanceToDestination = arrivalInfo[destinationIndex].m_distanceToDestination.c_str();
      if ((*mapScreenData).mDistanceToDestination != distanceToDestination.c_str())
      {
         (*mapScreenData).mDistanceToDestination = distanceToDestination.c_str();
         mapScreenData.MarkItemModified(ItemKey::MapScreenData::DistanceToDestinationItem);
      }

      std::string estimatedTimeofArrival = arrivalInfo[destinationIndex].m_estimatedTimeOfArrival.c_str();
      if (estimatedTimeofArrival.empty())
      {
         estimatedTimeofArrival = "00:00";
      }
      std::string timeToDestination      = arrivalInfo[destinationIndex].m_timeToDestination.c_str();
      std::string arrivalTimeToBeShownOnMap = (true == isArrivalTimeModeActive) ? estimatedTimeofArrival : timeToDestination;
      if ((*mapScreenData).mTimeToDestination != timeToDestination.c_str())
      {
         (*mapScreenData).mTimeToDestination = timeToDestination.c_str();
         mapScreenData.MarkItemModified(ItemKey::MapScreenData::TimeToDestinationItem);
      }
      if ((*mapScreenData).mArrivalTimeToBeShownOnMap != arrivalTimeToBeShownOnMap.c_str())
      {
         (*mapScreenData).mArrivalTimeToBeShownOnMap = arrivalTimeToBeShownOnMap.c_str();
         mapScreenData.MarkItemModified(ItemKey::MapScreenData::ArrivalTimeToBeShownOnMapItem);
      }
      // set total Delay Time to Destination
      std::string DelaytimeToDestination;
      DelaytimeToDestination = arrivalInfo.at(arrivalInfo.size() - 1).m_trafficDelayToDestinationAsString.c_str();

      // ETA of destination and waypoint on mapout screen
      DataBindingItem<MapoutScreenDataDataBindingSource>& mapoutScreenData = getMapoutScreenData();
      destinationIndex = arrivalInfo.size() - 1;
      std::string etaDestination = arrivalInfo[destinationIndex].m_estimatedTimeOfArrival.c_str();
      if (etaDestination.empty())
      {
         etaDestination = "00:00";
      }
      if ((*mapoutScreenData).mETADestination != etaDestination.c_str())
      {
         (*mapoutScreenData).mETADestination = etaDestination.c_str();
         mapoutScreenData.MarkItemModified(ItemKey::MapoutScreenData::ETADestinationItem);
      }

      bool isWaypointPresent = (1 < arrivalInfo.size()) ? true : false;
      if ((*mapoutScreenData).mIsWaypointPresent != isWaypointPresent)
      {
         (*mapoutScreenData).mIsWaypointPresent = isWaypointPresent;
         mapoutScreenData.MarkItemModified(ItemKey::MapoutScreenData::IsWaypointPresentItem);
      }
      if (1 < arrivalInfo.size())
      {
         int currentWaypointIndex = arrivalInfos.getCurrentDestinationIndex();
         std::string etaWaypoint = arrivalInfo[currentWaypointIndex].m_estimatedTimeOfArrival.c_str();
         if (etaWaypoint.empty())
         {
            etaWaypoint = "00:00";
         }
         if ((*mapoutScreenData).mETAWaypoint != etaWaypoint.c_str())
         {
            (*mapoutScreenData).mETAWaypoint = etaWaypoint.c_str();
            mapoutScreenData.MarkItemModified(ItemKey::MapoutScreenData::ETAWaypointItem);
         }
      }
      if ((*mapoutScreenData).mShowEta != showEtaOnSecMap)
      {
         (*mapoutScreenData).mShowEta = showEtaOnSecMap;
         mapoutScreenData.MarkItemModified(ItemKey::MapoutScreenData::ShowEtaItem);
      }
      sendMapoutScreenData();
   }
}


void clearMapScreenDataArrivalInfos()
{
   DataBindingItem<MapScreenDataDataBindingSource>& mapScreenData = getMapScreenData();
   DataBindingItem<MapoutScreenDataDataBindingSource>& mapoutScreenData = getMapoutScreenData();

   (*mapScreenData).mDistanceToDestination = "";
   mapScreenData.MarkItemModified(ItemKey::MapScreenData::DistanceToDestinationItem);
   (*mapScreenData).mTimeToDestination = "";
   mapScreenData.MarkItemModified(ItemKey::MapScreenData::TimeToDestinationItem);
   (*mapScreenData).mArrivalTimeToBeShownOnMap = "";
   mapScreenData.MarkItemModified(ItemKey::MapScreenData::ArrivalTimeToBeShownOnMapItem);

   (*mapoutScreenData).mETADestination = "";
   mapoutScreenData.MarkItemModified(ItemKey::MapoutScreenData::ETADestinationItem);
   (*mapoutScreenData).mETAWaypoint = "";
   mapoutScreenData.MarkItemModified(ItemKey::MapoutScreenData::ETAWaypointItem);
   (*mapoutScreenData).mIsWaypointPresent = false;
   mapoutScreenData.MarkItemModified(ItemKey::MapoutScreenData::IsWaypointPresentItem);
   (*mapoutScreenData).mShowEta = true;
   mapoutScreenData.MarkItemModified(ItemKey::MapoutScreenData::ShowEtaItem);

   sendMapScreenData();
   sendMapoutScreenData();
}


void setMapScreenDataNextManeuverInfos(const navmiddleware::NextManeuverInfos& nextManeuverInfos)
{
   DataBindingItem<ODRDataDataBindingSource>& odrData = getODRData();

   // Turn to street name
   std::string turnToStreetName = nextManeuverInfos.getTurnToStreet();
   if ((*odrData).mTurnToStreet != turnToStreetName.c_str())
   {
      (*odrData).mTurnToStreet = turnToStreetName.c_str();
      odrData.MarkItemModified(ItemKey::ODRData::TurnToStreetItem);
   }

   // set maneuver symbol
   unsigned int maneuverSymbol = nextManeuverInfos.getNextManeuverSymbolInfo().m_symbolIndex;
   if (maneuverSymbol <= 1)
   {
      maneuverSymbol = NO_MANEUVER_SYMBOL_INDEX;   // SceneComposer has no empty symbols for "NO_SYMBOL" and "NO_INFO"
   }

   if ((*odrData).mManeuverSymbolIndex != maneuverSymbol)
   {
      NextManeuverInfos::ManeuverSymbolInfo info = nextManeuverInfos.getNextManeuverSymbolInfo();
      if (MANEUVER_SYMBOL == info.m_maneuverSymbolType)
      {
         (*odrData).mManeuverSymbolIndex = maneuverSymbol;
      }
      else
      {
         (*odrData).mManeuverSymbolIndex = OFF_ROAD_SYMBOL_INDEX + maneuverSymbol;
      }
      odrData.MarkItemModified(ItemKey::ODRData::ManeuverSymbolIndexItem);
   }

   // set distance to maneuver
   if ((*odrData).mDistanceToManeuver != nextManeuverInfos.getDistanceToManeuver().c_str())
   {
      const std::string SPACE = std::string(" ");
      const Candera::UInt8 MIN_DIST_LENGTH = 3; // Unit + Distance
      std::string manuverDistanceStr = nextManeuverInfos.getDistanceToManeuver().c_str();
      int manuverDistanceStrLength = manuverDistanceStr.length();
      if (manuverDistanceStrLength >= MIN_DIST_LENGTH)
      {
         std::size_t found = manuverDistanceStr.find(SPACE);
         (*odrData).mDistanceToManeuver = manuverDistanceStr.substr(0, found).c_str();
         (*odrData).mDistanceToManeuverUnit = manuverDistanceStr.substr(found, manuverDistanceStrLength).c_str();
         odrData.MarkItemModified(ItemKey::ODRData::DistanceToManeuverItem);
         odrData.MarkItemModified(ItemKey::ODRData::DistanceToManeuverUnitItem);
      }
   }

   if (nextManeuverInfos.isBarGraphValid())
   {
      // calculate & send ManeuverBargraphValue update each BARGRAPH_STEP_SIZE amount changed
      const unsigned int oBarGraphValue = nextManeuverInfos.getBarGraphValue();
      const unsigned int BARGRAPH_STEP_SIZE = 100;
      const unsigned int BARGRAPH_STEP_MAX = 1000;
      unsigned int barGraphValue = (static_cast<unsigned int>(((static_cast<float>(oBarGraphValue) / BARGRAPH_STEP_SIZE)) + 0.5)) * BARGRAPH_STEP_SIZE;
      barGraphValue = (barGraphValue > BARGRAPH_STEP_MAX) ? BARGRAPH_STEP_MAX : barGraphValue;
      if ((oBarGraphValue > 0) && (oBarGraphValue < BARGRAPH_STEP_SIZE))
      {
         barGraphValue = BARGRAPH_STEP_SIZE;
      }

      if ((*odrData).mManeuverBargraphValue != static_cast<float>(barGraphValue))
      {
         (*odrData).mManeuverBargraphValue = static_cast<float>(barGraphValue);
         odrData.MarkItemModified(ItemKey::ODRData::ManeuverBargraphValueItem);
      }
   }

   // Turn to street name with distance
   std::string turnToStreetNameWithDistance = (*odrData).mDistanceToManeuver.GetCString();
   turnToStreetNameWithDistance.append((*odrData).mDistanceToManeuverUnit.GetCString());
   if (false == (*odrData).mTurnToStreet.IsEmpty())
   {
      if (0 != turnToStreetNameWithDistance.length())
      {
         turnToStreetNameWithDistance.append(", ");
      }
      turnToStreetNameWithDistance.append((*odrData).mTurnToStreet.GetCString());
   }
   if ((*odrData).mTurnToStreetWithDistance != turnToStreetNameWithDistance.c_str())
   {
      (*odrData).mTurnToStreetWithDistance = turnToStreetNameWithDistance.c_str();
      odrData.MarkItemModified(ItemKey::ODRData::TurnToStreetWithDistanceItem);
   }
}


void clearMapScreenDataManeuverInfos()
{
   DataBindingItem<ODRDataDataBindingSource>& odrData = getODRData();
   (*odrData).mDistanceToManeuver = "";
   (*odrData).mDistanceToManeuverUnit = "";
   (*odrData).mManeuverSymbolIndex = NO_MANEUVER_SYMBOL_INDEX;
   (*odrData).mTurnToStreet = "";
   (*odrData).mTurnToStreetWithDistance = "";
   (*odrData).mManeuverBargraphValue = 0.0f;
   odrData.MarkAllItemsModified();
}


void setMapScreenDataSpeedLimitInfos(const navmiddleware::SpeedInfo& speedInfo, bool speedLimitActiveStatus)
{
   DataBindingItem<MapScreenDataDataBindingSource>& mapScreenData = getMapScreenData();
   // Update isSpeedLimitPresent
   if ((*mapScreenData).misSpeedLimitPresent != speedInfo.isSpeedLimitPresent())
   {
      (*mapScreenData).misSpeedLimitPresent = speedInfo.isSpeedLimitPresent();
      mapScreenData.MarkItemModified(ItemKey::MapScreenData::isSpeedLimitPresentItem);
   }

   // Update speed limit value
   std::stringstream stringStreamSpeedLimit;
   stringStreamSpeedLimit << speedInfo.getSpeedLimit();
   if ((*mapScreenData).mSpeedLimit != stringStreamSpeedLimit.str().c_str())
   {
      (*mapScreenData).mSpeedLimit = stringStreamSpeedLimit.str().c_str();
      mapScreenData.MarkItemModified(ItemKey::MapScreenData::SpeedLimitItem);
   }

   // Update speed limit country code
   unsigned int speedCountryCode = 0;
   if (speedInfo.isSpeedLimitPresent() && speedLimitActiveStatus)
   {
      if (speedInfo.getCountryCode() == "USA")
      {
         speedCountryCode = 1;
      }
      else if (speedInfo.getCountryCode() == "CAN")
      {
         speedCountryCode = 2;
      }
      else
      {
         speedCountryCode = 3;
      }
   }
   if ((*mapScreenData).mSpeedcountryCode != speedCountryCode)
   {
      (*mapScreenData).mSpeedcountryCode = speedCountryCode;
      mapScreenData.MarkItemModified(ItemKey::MapScreenData::SpeedcountryCodeItem);
      (*mapScreenData).mSpeedLimitIconPosition = (3 == speedCountryCode) ? Candera::Vector2(735.0f, 148.0f) : Candera::Vector2(727.0f, 137.0f);
      mapScreenData.MarkItemModified(ItemKey::MapScreenData::SpeedLimitIconPositionItem);
   }
}


void setMapScreenDataOverSpeedInfos(bool overSpeedWarning)
{
   DataBindingItem<MapScreenDataDataBindingSource>& mapScreenData = getMapScreenData();

   //Color of speed limit value is currently not changed in over-speed scenario. Navi FO to confirm if there is any requirement for the same.
   //Candera::Color speedLockTextColor(BlackColor[0], BlackColor[1], BlackColor[2], BlackColor[3]); //Default :Black text
   //if (true == overSpeedStatus)
   //{
   //   overSpeedWarning = true;
   //   speedLockTextColor.Set(RedColor[0], RedColor[1], RedColor[2], RedColor[3]); //changing the text color to RED on overspeed
   //}
   if ((*mapScreenData).misOverSpeedWarning != overSpeedWarning)
   {
      (*mapScreenData).misOverSpeedWarning = overSpeedWarning;
      mapScreenData.MarkItemModified(ItemKey::MapScreenData::isOverSpeedWarningItem);
   }
}


void setNavigationGadgetStatus(NavigationGadgetStatus status, bool isNaviAccessible, unsigned int vehicleProfile)
{
   const int COMPASS_DISABLED_ICON_INDEX = 16;
   const Candera::String DATA_CONTEXT__TEXT_NOGUIDANCE   = LANGUAGE_STRING(TextId_0x0B03, "No guidance");
   const Candera::String DATA_CONTEXT__TEXT_NODATA       = LANGUAGE_STRING(TextId_0x0B04, "No navigation data");
   const Candera::String DATA_CONTEXT__TEXT_INITIALIZING = LANGUAGE_STRING(TextId_0x0ED5, "Navigation initialising");

   DataBindingItem<NavigationGadgetInfoDataBindingSource>& gadgetData = getNavigationGadgetInfoData();
   DataBindingItem<NaviMiddlewareStatusDataBindingSource>& middlewareStatus = getNaviMiddlewareStatusData();
   DataBindingItem<MapScreenDataDataBindingSource>& mapScreenData = getMapScreenData();

   if (NAVI_GADGET_STATUS__NO_GUIDANCE == status)
   {
      (*gadgetData).mIsGuidanceActive = 0;
      (*gadgetData).mHomeTileStreetStr = DATA_CONTEXT__TEXT_NOGUIDANCE;
      (*mapScreenData).mNaviGadgetIcon = ImageLoader::getAssetBitmapImage("AppHmi_NavigationModule#Images#MasterHome#Icon_navi");
   }
   else if (NAVI_GADGET_STATUS__ACTIVE_GUIDANCE == status)
   {
      (*gadgetData).mIsGuidanceActive = 1;
      (*gadgetData).mHomeTileStreetStr = "";
      (*mapScreenData).mNaviGadgetIcon = vehicleProfile ?
                                         ImageLoader::getAssetBitmapImage("AppHmi_NavigationModule#Images#N_NavigationGadget#Icon_stop_navi_carmode") :
                                         ImageLoader::getAssetBitmapImage("AppHmi_NavigationModule#Images#N_NavigationGadget#Icon_stop_navi");
   }
   else
   {
      (*gadgetData).mIsGuidanceActive = 0;
      (*gadgetData).mHomeTileStreetStr = (NAVI_GADGET_STATUS__INITIALIZING == status) ? DATA_CONTEXT__TEXT_INITIALIZING : DATA_CONTEXT__TEXT_NODATA;
      (*mapScreenData).mNaviGadgetIcon = ImageLoader::getAssetBitmapImage("AppHmi_NavigationModule#Images#MasterHome#Icon_navi");

      currentCompassHeadingStep[Enum_MAP_VIEW_PRIMARY] = COMPASS_DISABLED_ICON_INDEX;
      (*mapScreenData).mCompassArrowDirection = COMPASS_DISABLED_ICON_INDEX;
      mapScreenData.MarkItemModified(ItemKey::MapScreenData::CompassArrowDirectionItem);
   }
   gadgetData.MarkAllItemsModified();
   gadgetData.SendUpdate();
   mapScreenData.MarkItemModified(ItemKey::MapScreenData::NaviGadgetIconItem);
   sendMapScreenData();

   if ((*middlewareStatus).mIsNaviMiddlewareActive != isNaviAccessible)
   {
      (*middlewareStatus).mIsNaviMiddlewareActive = isNaviAccessible;
      middlewareStatus.MarkItemModified(ItemKey::NaviMiddlewareStatus::IsNaviMiddlewareActiveItem);
      middlewareStatus.SendUpdate();
   }
}


void computeDurationandETACabinOnly(const uint16_t timeToReachDest, const uint8_t tmMode, const uint64_t timeDate)
{
   std::ostringstream durationCabinOnly;
   int hours = timeToReachDest / 60;
   int minutes = timeToReachDest % 60;
   durationCabinOnly << std::setw(2) << std::setfill('0') << hours << ":" << std::setw(2) << std::setfill('0') << minutes;
   DataBindingItem<MapoutScreenDataDataBindingSource>& mapoutScreenData = getMapoutScreenData();

   // Update the duration in the data binding

   if ((*mapoutScreenData).mDurationCabinOnly != durationCabinOnly.str().c_str())
   {
      (*mapoutScreenData).mDurationCabinOnly = durationCabinOnly.str().c_str();
      mapoutScreenData.MarkItemModified(ItemKey::MapoutScreenData::DurationCabinOnlyItem);
   }

   // Calculate ETA based on the local time
   uint64_t localTimeInSeconds = timeDate / 1000; // Convert milliseconds to seconds
   uint64_t etaInSeconds = localTimeInSeconds + (timeToReachDest * 60); // Add travel time in seconds

   // Convert ETA to hours and minutes
   time_t etaTime = static_cast<time_t>(etaInSeconds);
   struct tm* etaTm = localtime(&etaTime);

   char buffer[TIME_STR_MAX_LENGTH];
   if (tmMode == 12)
   {
      // 12 Hour format
      int hr = etaTm->tm_hour % 12;
      hr = (hr == 0) ? 12 : hr; // Handle midnight and noon
      const char* amPm = (etaTm->tm_hour >= 12) ? "pm" : "am";
      snprintf(buffer, TIME_STR_MAX_LENGTH, "%02d:%02d%s", hr, etaTm->tm_min, amPm);
   }
   else
   {
      // 24 Hour format
      snprintf(buffer, TIME_STR_MAX_LENGTH, "%02d:%02d", etaTm->tm_hour, etaTm->tm_min);
   }

   // Update the ETA in the data binding
   if ((*mapoutScreenData).mETACabinOnly != buffer)
   {
      (*mapoutScreenData).mETACabinOnly = buffer;
      mapoutScreenData.MarkItemModified(ItemKey::MapoutScreenData::ETACabinOnlyItem);
   }
   sendMapoutScreenData();
}


#endif
