/**************************************************************************************
* @file         : MapInteractionHandler.cpp
* @author       :
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) -2018 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include "MapUtils.h"
#include "MapInteractionHandler.h"
#include "AppHmi_NavigationStateMachine.h"
#include "Map/MapScreenDataUtils.h"
#include "hmi_trace_if.h"
#include "CgiExtensions/CourierMessageMapper.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/MapInteractionHandler.cpp.trc.h"
#endif

#ifdef HALL_TO_MDW_COM

const float PINCH_FACTOR_THRESHOLD_FOR_ZOOM_OUT = 0.5f;
const float PINCH_FACTOR_THRESHOLD_FOR_ZOOM_IN = 1.5f;
static const int DEFAULT_ZOOM_SCALE_FOR_LOCATION_OVERVIEW = 200;

using namespace navmiddleware;

MapInteractionHandler::MapInteractionHandler(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore)
   : HMIModelBase(navMiddleware, infoStore)
   , _lastRequestedMapCameraAndViewMode(MAP_CAMERA_AND_VIEW_MODE_MAP_SCROLL)
   , _maxNominalMapScale(INT_MAX)
   , _minNominalMapScale(0)
{
}


MapInteractionHandler::~MapInteractionHandler()
{
}


void MapInteractionHandler::initialize()
{
   ETG_TRACE_USR4(("MapInteractionHandler::initialize()"));
   _navMiddleware.registerMapPropertyUpdateCallback(*this);
   _infoStore.registerDataPropertyUpdateCallback(*this);
}


void MapInteractionHandler::deinitialize()
{
   _navMiddleware.unregisterMapPropertyUpdateCallback(*this);
   _infoStore.unregisterDataPropertyUpdateCallback(*this);
}


bool MapInteractionHandler::onPropertyUpdateMapTouchEvent()
{
   const navmiddleware::TouchEventInfo& touchEventInfo = _navMiddleware.getTouchEventInfo();
   ETG_TRACE_USR4(("MapInteractionHandler::onPropertyUpdateMapTouchEvent(gestureType %d, speedLock %d)", touchEventInfo.m_touchGestureType, _infoStore.getSpeedLockState()));

   if (_infoStore.getSpeedLockState() == InfoStoreBase::SPEEDLOCKSTATE_ON)
   {
      // in speed lock mode process gestures coming from the map and trigger discrete map actions
      switch (touchEventInfo.m_touchGestureType)
      {
         case TOUCH_GESTURE_TAPPED:
            handleTouchGestureTapped();
            break;
         case TOUCH_GESTURE_SWIPE:
            handleTouchGestureSwipeInSpeedLockMode();
            activateMapScrollScreenIfNeeded();
            break;
         case TOUCH_GESTURE_ROTATE:
            handleTouchGestureRotateInSpeedLockMode();
            activateMapScrollScreenIfNeeded();
            break;
         case TOUCH_GESTURE_PINCH:
            handleTouchGesturePinchInSpeedLockMode();
            break;
         case TOUCH_GESTURE_INVALID:
         default:
            break;
      }
   }
   else
   {
      // in normal map mode check tapping gestures and check if we need a transition to map scroll screen
      switch (touchEventInfo.m_touchGestureType)
      {
         case TOUCH_GESTURE_TAPPED:
            //if (touchEventInfo.m_touchGestureTypeDetail.m_touchGestureTap.m_touchGestureTapType != TOUCH_GESTURE_LONG_TAP)
            //{
            if (EXT_IsMapScrollTapEnable)
            {
               handleTouchGestureTapped();
            }
            // }
            break;
         case TOUCH_GESTURE_SWIPE:
         case TOUCH_GESTURE_ROTATE:
         case TOUCH_GESTURE_PINCH:
            activateMapScrollScreenIfNeeded();
            break;
         case TOUCH_GESTURE_INVALID:
         default:
            break;
      }
   }

   return true;
}


void MapInteractionHandler::handleTouchGestureTapped()
{
   const navmiddleware::TouchEventInfo& touchEventInfo = _navMiddleware.getTouchEventInfo();
   navmiddleware::TouchEventInfo::TouchGestureTypeDetail touchGestureTypeDetail = touchEventInfo.m_touchGestureTypeDetail;
   MapCameraModeInfo mapCameraInfo = (MapCameraModeInfo)_navMiddleware.getMapCameraModeInfo();
   const MapCameraMode mapCameraMode = mapCameraInfo.getMapCameraModeInfo();
   navmiddleware::ValidValue<navmiddleware::PickingResultFilter> pickingResultFilter;

   ETG_TRACE_USR4(("MapInteractionHandler::handleTouchGestureTapped(taptype %d)", touchGestureTypeDetail.m_touchGestureTap.m_touchGestureTapType));

   switch (touchGestureTypeDetail.m_touchGestureTap.m_touchGestureTapType)
   {
      case TOUCH_GESTURE_SINGLE_TAP:
      {
         if (mapCameraMode == navmiddleware::MAP_MODE_FREE && ((_lastRequestedMapCameraAndViewMode == MAP_CAMERA_AND_VIEW_MODE_MAP_SCROLL)))
         {
            if (Ext_bIsEnableByMapOKBtn == false)   // Map scroll from main map scene
            {
               POST_MSG((COURIER_MESSAGE_NEW(StartPickingReqMsg)(MAP_PICKING_FOR_POI)));
               POST_MSG((COURIER_MESSAGE_NEW(TriggerMapPickingReqMsg)()));
            }
            else   // "By map" destination input
            {
               _navMiddleware.centerMapToViewCoordinate(MAP_VIEW_ID__PRIMARY, static_cast<uint16_t>(touchGestureTypeDetail.m_touchGestureTap.m_touchEventScreenCoordinate.xPos), static_cast<uint16_t>(touchGestureTypeDetail.m_touchGestureTap.m_touchEventScreenCoordinate.yPos), ::navmiddleware::CameraAnimation__FAST);
               POST_MSG((COURIER_MESSAGE_NEW(SetPickingTimerReqMsg)(true, NAVI_HMI_SM_C_MAP_PICKING_TIMER_DEFAULT)));
               _infoStore.setMoveLocationFlag(true);
            }
         }
         activateMapScrollScreenIfNeeded();
         break;
      }
      case TOUCH_GESTURE_DOUBLE_TAP:
         _navMiddleware.zoomInStep(MAP_VIEW_ID__PRIMARY);
         break;
      case TOUCH_GESTURE_LONG_TAP:
         break;
      default:
         break;
   }
}


void MapInteractionHandler::handleTouchGestureSwipeInSpeedLockMode()
{
   ETG_TRACE_USR4(("MapInteractionHandler::handleTouchGestureSwipeInSpeedLockMode()"));

   _navMiddleware.startScrollingMap(MAP_VIEW_ID__PRIMARY, getScrollDirectionFromSwipeAngle(), SCROLL_SPEED_4, getDataPixelsForDiscreteMoves(_infoStore.getScreenWidth(), _navMiddleware.getMapCameraModeInfo()));
}


void MapInteractionHandler::handleTouchGestureRotateInSpeedLockMode()
{
   ETG_TRACE_USR4(("MapInteractionHandler::handleTouchGestureRotateInSpeedLockMode()"));

   navmiddleware::settings::MapView mapView = _navMiddleware.getChangeMapViewSettings().getMapView();
   navmiddleware::TouchEventInfo touchEventInfo = _navMiddleware.getTouchEventInfo();
   int16_t rotationAngle = touchEventInfo.m_touchGestureTypeDetail.m_touchGestureRotate.m_rotationAngle;

   if (rotationAngle >= 30 && mapView == navmiddleware::settings::MAPVIEW_3D)
   {
      // clockwise rotation by 30 degree in 3D
      ETG_TRACE_USR1(("rotating clockwise with rotationAngle %d", rotationAngle));
      _navMiddleware.startRotatingMap(MAP_VIEW_ID__PRIMARY, ROTATION_TYPE_BOUNDED, ROTATION_SPEED_4, 30);
   }
   else if (rotationAngle <= -30 && mapView == navmiddleware::settings::MAPVIEW_3D)
   {
      // anticlockwise rotation by 30 degree in 3D
      ETG_TRACE_USR1(("rotating anticlockwise with rotationAngle %d", rotationAngle));
      _navMiddleware.startRotatingMap(MAP_VIEW_ID__PRIMARY, ROTATION_TYPE_BOUNDED, ROTATION_SPEED_4, -30);
   }
   else
   {
      ETG_TRACE_USR1(("Invalid rotationAngle %d", rotationAngle));
   }
}


void MapInteractionHandler::handleTouchGesturePinchInSpeedLockMode()
{
   ETG_TRACE_USR4(("MapInteractionHandler::handleTouchGesturePinchInSpeedLockMode()"));

   navmiddleware::TouchEventInfo touchEventInfo = _navMiddleware.getTouchEventInfo();
   float pinchFactor = touchEventInfo.m_touchGestureTypeDetail.m_touchGesturePinch.m_pinchFactor;

   if (pinchFactor <= PINCH_FACTOR_THRESHOLD_FOR_ZOOM_OUT)
   {
      // pinch out if pinchFactor is less than 1.0f
      _navMiddleware.zoomOutStep(MAP_VIEW_ID__PRIMARY, 3);
   }
   else if (pinchFactor >= PINCH_FACTOR_THRESHOLD_FOR_ZOOM_IN)
   {
      // pinch in if pinchFactor is more than 1.0f
      _navMiddleware.zoomInStep(MAP_VIEW_ID__PRIMARY, 3);
   }
   else
   {
      ETG_TRACE_USR1(("Invalid pinchFactor %f", pinchFactor));
   }
}


void MapInteractionHandler::activateMapScrollScreenIfNeeded()
{
   MapCameraModeInfo mapCameraInfo = (MapCameraModeInfo)_navMiddleware.getMapCameraModeInfo();
   const MapCameraMode mapCameraMode = mapCameraInfo.getMapCameraModeInfo();

   if (mapCameraMode != navmiddleware::MAP_MODE_FREE)
   {
      // go to map scroll Map if the camera mode is not free mode
      POST_MSG((COURIER_MESSAGE_NEW(ActivateMapScrollReqMsg)()));
   }
}


navmiddleware::ScrollDirection MapInteractionHandler::getScrollDirectionFromSwipeAngle()
{
   int32_t swipeAngle = _navMiddleware.getTouchEventInfo().m_touchGestureTypeDetail.m_touchGestureSwipe.m_swipeDirection;
   navmiddleware::ScrollDirection scrollDirection = SCROLL_DIRECTION_NORTH;

   if (swipeAngle >= 0 && swipeAngle <= 22.5)
   {
      scrollDirection = navmiddleware::SCROLL_DIRECTION_SOUTH;
   }
   else if (swipeAngle <= 67.5)
   {
      scrollDirection = navmiddleware::SCROLL_DIRECTION_SOUTH_EAST;
   }
   else if (swipeAngle <= 112.5)
   {
      scrollDirection = navmiddleware::SCROLL_DIRECTION_EAST;
   }
   else if (swipeAngle <= 157.5)
   {
      scrollDirection = navmiddleware::SCROLL_DIRECTION_NORTH_EAST;
   }
   else if (swipeAngle <= 180)
   {
      scrollDirection = navmiddleware::SCROLL_DIRECTION_NORTH;
   }
   else if (swipeAngle <= 202.5)
   {
      scrollDirection = navmiddleware::SCROLL_DIRECTION_NORTH;
   }
   else if (swipeAngle <= 247.5)
   {
      scrollDirection = navmiddleware::SCROLL_DIRECTION_NORTH_WEST;
   }
   else if (swipeAngle <= 292.5)
   {
      scrollDirection = navmiddleware::SCROLL_DIRECTION_WEST;
   }
   else if (swipeAngle <= 337.5)
   {
      scrollDirection = navmiddleware::SCROLL_DIRECTION_SOUTH_WEST;
   }
   else if (swipeAngle <= 360)
   {
      scrollDirection = navmiddleware::SCROLL_DIRECTION_SOUTH;
   }
   else
   {
      ETG_TRACE_USR1(("Invalid swipeAngle!"));
   }

   ETG_TRACE_USR1(("MapInteractionHandler::getScrollDirectionFromSwipeAngle(swipe angle %f, scrollDirection %d)", swipeAngle, scrollDirection));

   return scrollDirection;
}


navmiddleware::ScrollDirection MapInteractionHandler::getScrollDirectionFromLongTap()
{
   uint16_t tappedAngle = _navMiddleware.getTouchEventInfo().m_touchGestureTypeDetail.m_touchGestureTap.m_tappedDirection;
   navmiddleware::ScrollDirection tappedDirection = SCROLL_DIRECTION_NORTH;

   if (tappedAngle <= 22.5)
   {
      tappedDirection = navmiddleware::SCROLL_DIRECTION_NORTH;
   }
   else if (tappedAngle <= 67.5)
   {
      tappedDirection = navmiddleware::SCROLL_DIRECTION_NORTH_WEST;
   }
   else if (tappedAngle <= 112.5)
   {
      tappedDirection = navmiddleware::SCROLL_DIRECTION_WEST;
   }
   else if (tappedAngle <= 157.5)
   {
      tappedDirection = navmiddleware::SCROLL_DIRECTION_SOUTH_WEST;
   }
   else if (tappedAngle <= 202.5)
   {
      tappedDirection = navmiddleware::SCROLL_DIRECTION_SOUTH;
   }
   else if (tappedAngle <= 247.5)
   {
      tappedDirection = navmiddleware::SCROLL_DIRECTION_SOUTH_EAST;
   }
   else if (tappedAngle <= 292.5)
   {
      tappedDirection = navmiddleware::SCROLL_DIRECTION_EAST;
   }
   else if (tappedAngle <= 337.5)
   {
      tappedDirection = navmiddleware::SCROLL_DIRECTION_NORTH_EAST;
   }
   else if (tappedAngle <= 360)
   {
      tappedDirection = navmiddleware::SCROLL_DIRECTION_NORTH;
   }
   else
   {
      ETG_TRACE_USR1(("Invalid tappedAngle!"));
   }

   ETG_TRACE_USR1(("MapInteractionHandler::getScrollDirectionFromLongTap(tapped angle %d, tappedDirection %d)", tappedAngle, tappedDirection));

   return tappedDirection;
}


void MapInteractionHandler::onPropertyUpdateMapNominalScalesChanged()
{
   ETG_TRACE_USR1(("MapInteractionHandler::onPropertyUpdateMapNominalScalesChanged()"));
   const navmiddleware::NominalScaleInfos& nominalScaleInfos = _navMiddleware.getNominalScaleInfos();
   const std::vector< navmiddleware::NominalScaleInfos::NominalScaleInfo >& nominalScaleInfoVector = nominalScaleInfos.getNominalScaleInfos();

   if (!nominalScaleInfoVector.empty())
   {
      _infoStore.setDefaultNominalMapScale(findDefaultMapScaleFromNominalScales(nominalScaleInfoVector, DEFAULT_ZOOM_SCALE_FOR_LOCATION_OVERVIEW));
      InfoStoreBase::CoordinatesToBeShownInMap& coordinatesToBeShownInMap = _infoStore.getCoordinatesToBeShownInMap();
      coordinatesToBeShownInMap._scale = _infoStore.getDefaultNominalMapScale();

      _minNominalMapScale = nominalScaleInfoVector.front().getMapScale();
      _maxNominalMapScale = nominalScaleInfoVector.back().getMapScale();
      ETG_TRACE_USR1(("Nominal scales updated, new min %d, new max %d", _minNominalMapScale, _maxNominalMapScale));
   }
}


void MapInteractionHandler::handleMapCameraAndViewMode()
{
   const enMapCameraAndViewModeMode& mapCameraViewMode = _infoStore.getMapCameraViewMode();
   ETG_TRACE_USR4(("MapInteractionHandler::handleMapCameraAndViewMode(), MapCameraViewMode : %d", mapCameraViewMode));

   // check if we have to adapt map interaction possibilities based on the set map camera mode
   switch (mapCameraViewMode)
   {
      case MAP_CAMERA_AND_VIEW_MODE_MAP_MAIN_SCREEN:
      {
         // we disable all touch configurations for MAP_VIEW_ID__PRIMARY and MAP_VIEW_ID__SECONDARY because on touch we make a transition to Map Scroll
         if (isMultiViewActive(_navMiddleware))
         {
            setConfigureTouch(MAP_VIEW_ID__SECONDARY, false, false, false, false, false, false, false, false, false, 0, INT_MAX);
         }
         setConfigureTouch(MAP_VIEW_ID__PRIMARY, false, false, false, false, true, true, true, true, false, 0, INT_MAX);

         break;
      }
      case MAP_CAMERA_AND_VIEW_MODE_MAP_JUNCTION_AND_INTERSECTION_SCREEN:
      {
         setConfigureTouch(MAP_VIEW_ID__PRIMARY, false, false, false, false, true, true, true, true, false, 0, INT_MAX);
         setConfigureTouch(MAP_VIEW_ID__SECONDARY, false, false, false, false, false, false, false, false, false, 0, INT_MAX);
         break;
      }
      case MAP_CAMERA_AND_VIEW_MODE_MAP_SCROLL:
      {
         setConfigureTouch(MAP_VIEW_ID__PRIMARY, true, true, true, true, true, false, false, false, true, 0, INT_MAX);
         break;
      }
      case MAP_CAMERA_AND_VIEW_MODE_LOCATION_DETAILS:
      case MAP_CAMERA_AND_VIEW_MODE_CARSOR_LEFT_OVERVIEW:
      {
         setConfigureTouch(MAP_VIEW_ID__PRIMARY, false, false, false, false, false, false, false, false, false, 0, INT_MAX);
         break;
      }
      case STREAMED_MAP_OVERVIEW_MAP:
      case STREAMED_MAP_2D_MAP:
      case STREAMED_MAP_3D_MAP:
      case STOP_MAP_STREAM:
      {
         break;
      }
      default:
      {
         setConfigureTouch(MAP_VIEW_ID__PRIMARY, true, true, true, true, false, false, false, false, false, 0, INT_MAX);
         break;
      }
   }
   _lastRequestedMapCameraAndViewMode = mapCameraViewMode;
}


void MapInteractionHandler::setConfigureTouch(MapViewId mapViewId, bool moveAllowed, bool zoomAllowed, bool rotateAllowed, bool pitchAllowed, bool touchGestureTapAllowed, bool touchGestureswipeAllowed, bool touchGestureRotateAllowed, bool touchGesturePinchAllowed, bool scrollOnHold, int minScale, int maxScale)
{
   MapScaleRange mapScaleRange;
   TouchActions touchActions;
   PitchRange pitchRange;

   touchActions.setTouchActionMove(moveAllowed);
   touchActions.setTouchActionZoom(zoomAllowed);
   touchActions.setTouchActionRotate(rotateAllowed);
   touchActions.setTouchActionPitch(pitchAllowed);
   touchActions.setTouchActionGestureTap(touchGestureTapAllowed);
   touchActions.setTouchActionGestureSwipe(touchGestureswipeAllowed);
   touchActions.setTouchActionGestureRotate(touchGestureRotateAllowed);
   touchActions.setTouchActionGesturePinch(touchGesturePinchAllowed);
   touchActions.setTouchActionScrollOnHold(scrollOnHold);

   // limit the min scale to min nominal scale to avoid that the user can zoom in more via touch than via (soft) keys
   mapScaleRange.setMinMapScale(minScale < _minNominalMapScale ? _minNominalMapScale : minScale);
   mapScaleRange.setMaxMapScale(maxScale);

   _navMiddleware.configureTouch(mapViewId, touchActions, pitchRange, mapScaleRange);
}


#endif // HALL_TO_MDW_COM
