/**************************************************************************************
* @file         : HMIModelMap.cpp
* @author       : ECG5-Naveen Thangamalayan
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include "HMIModelMap.h"
#include "Common/Util/EnvironmentUtils.h"
#include "MapScreenDataUtils.h"
#include "../Common/Util/CommonUnitsUtility.h"
#include "Common/Util/SurfaceSynchronizationHandler.h"   // For map layer visibility setting
#include "CgiExtensions/ImageLoader.h"
#include "App/Core/DiagnosticsClient/DiagnosticsClientHandler.h"
#include "App/Core/VehicleDataClient/ClusterDataClientHandler.h"
#include "MapUtils.h"
#include "Common/VariantHandling/VariantHandling.h"

#define OSAL_S_IMPORT_INTERFACE_THREADING
#include "osal_if.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/HMIModelMap.cpp.trc.h"
#endif

#ifdef HALL_TO_MDW_COM

static const unsigned int DELAY_TIME_START_OVER_SPEED_BLINKING = 3000;
static const Candera::String EMPTY_STREET_NAME                 = "";
static const Candera::String DATA_CONTEXT__TEXT_FIRSTLINE      = "3D Map view disabled due to";
static const Candera::String DATA_CONTEXT__TEXT_SECONDLINE     = "secondary map activation.";
static const unsigned int DURATION_CABIN_ONLY = 0;

using namespace navmiddleware;

HMIModelMap::HMIModelMap(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore)
   : HMIModelBase(navMiddleware, infoStore)
   , _mapCameraAndViewHandler(navMiddleware, infoStore)
   , _mapInteractionHandler(navMiddleware, infoStore)
   , _isMapActive(false)
   , _wasMapActiveWhenGoingToBackground(false)
   , _currentMapView(navmiddleware::settings::MAPVIEW_2D)
   , _currentMapHMISplitMode(navmiddleware::settings::MAP_HMI_SPLITMODE_FULL_MAP)
   , _isMapViewUpdateNeeded(false)
   , _isHardRestrictionPresent(false)
   , _timeToReachDest(DURATION_CABIN_ONLY)
   , _tmMode(DURATION_CABIN_ONLY)
   , _timeDate(DURATION_CABIN_ONLY)
{
}


HMIModelMap::~HMIModelMap()
{
}


void HMIModelMap::initialize()
{
   ETG_TRACE_USR4(("HMIModelMap::initialize()"));

   _navMiddleware.registerMapPropertyUpdateCallback(*this);
   _navMiddleware.registerPositionPropertyUpdateCallback(*this);
   _navMiddleware.registerRoutePropertyUpdateCallback(*this);
   _navMiddleware.registerGuidancePropertyUpdateCallback(*this);
   _infoStore.registerDataPropertyUpdateCallback(*this);
   _mapCameraAndViewHandler.initialize();
   _mapInteractionHandler.initialize();

   (*getMapScreenData()).mCurrentStreet = LANGUAGE_STRING(TextId_0x1418, "UNKNOWN STREET NAME");
   getMapScreenData().MarkItemModified(ItemKey::MapScreenData::CurrentStreetItem);
   getMapScreenData().SendUpdate();

   _mapCameraAndViewHandler.handleMapCameraAndViewMode(MAP_CAMERA_AND_VIEW_MODE_MAP_MAIN_SCREEN);
   (false == EXT_bIsNaviAppRunning) ? freezeAndHideMap() : unfreezeMap();
}


void HMIModelMap::deinitialize()
{
   _navMiddleware.unregisterMapPropertyUpdateCallback(*this);
   _navMiddleware.unregisterPositionPropertyUpdateCallback(*this);
   _navMiddleware.unregisterRoutePropertyUpdateCallback(*this);
   _navMiddleware.unregisterGuidancePropertyUpdateCallback(*this);
   _infoStore.unregisterDataPropertyUpdateCallback(*this);
   _mapCameraAndViewHandler.deinitialize();
   _mapInteractionHandler.deinitialize();

   // Runtime duration is computed and updated to Diagnostics component if guidance/map streaming is active on shutdown
   computeGuidanceRuntime(false);
   computeMapStreamingRuntime(false);
}


bool HMIModelMap::onPropertyUpdateHmiAppStateChanged()
{
   hmibase::hmiappstates hmiAppState = _infoStore.getHmiAppState();
   ETG_TRACE_USR1(("HMIModelMap::onPropertyUpdateHmiAppStateChanged(%d)", hmiAppState));

   if (hmiAppState == hmibase::TO_BACKGROUND && _isMapActive)
   {
      // apphmi_navigation went to background so freeze map if activated
      _wasMapActiveWhenGoingToBackground = true;
      freezeMap();
   }
   else if (hmiAppState == hmibase::IN_FOREGROUND && _wasMapActiveWhenGoingToBackground)
   {
      // apphmi_navigation went to foreground so unfreeze it
      _wasMapActiveWhenGoingToBackground = false;
      unfreezeMap();
   }
   else
   {
      // to nothing
   }
   return true;
}


bool HMIModelMap::onCourierMessage(const UnfreezeMapReqMsg& oMsg)
{
   COURIER_UNUSED(oMsg);
   ETG_TRACE_USR1(("HMIModelMap::onCourierMessage(UnfreezeMapReqMsg)"));

   if (hmibase::TO_BACKGROUND != _infoStore.getHmiAppState())
   {
      _wasMapActiveWhenGoingToBackground = true;
   }
   unfreezeMap();

   return true;
}


bool HMIModelMap::onCourierMessage(const FreezeMapReqMsg& oMsg)
{
   COURIER_UNUSED(oMsg);
   ETG_TRACE_USR1(("HMIModelMap::onCourierMessage(FreezeMapReqMsg)"));

   if (hmibase::TO_BACKGROUND != _infoStore.getHmiAppState())
   {
      _wasMapActiveWhenGoingToBackground = false;
   }
   freezeMap();

   return true;
}


bool HMIModelMap::onCourierMessage(const FreezeAndHideMapReqMsg& oMsg)
{
   COURIER_UNUSED(oMsg);
   ETG_TRACE_USR1(("HMIModelMap::onCourierMessage(FreezeAndHideMapReqMsg)"));

   freezeAndHideMap();

   return true;
}


bool HMIModelMap::onCourierMessage(const OverSpeedTimerExpiredReqMsg& oMsg)
{
   COURIER_UNUSED(oMsg);
   ETG_TRACE_USR1(("HMIModelMap::onCourierMessage(OverSpeedTimerExpiredReqMsg)"));

   setMapScreenDataSpeedLimitInfos(_navMiddleware.getSpeedInfo(), _navMiddleware.getSafetyGuidanceSettings().isSpeedLimitActive());
   sendMapScreenData();

   return true;
}


bool HMIModelMap::onPropertyUpdateMapCameraModeChanged()
{
   static navmiddleware::MapCameraMode currentMapCameraMode = navmiddleware::MAP_MODE_FREE;

   MapCameraModeInfo mapCameraInfo = (MapCameraModeInfo)_navMiddleware.getMapCameraModeInfo();
   const MapCameraMode newMapCameraMode = mapCameraInfo.getMapCameraModeInfo();

   ETG_TRACE_USR1(("HMIModelMap::onPropertyUpdateMapCameraModeChanged(current %d, new %d)", currentMapCameraMode, newMapCameraMode));

   // check if we need to activate picking timer
   if ((_isMapActive) && (currentMapCameraMode == navmiddleware::MAP_MODE_TOUCH) && (newMapCameraMode == navmiddleware::MAP_MODE_FREE))
   {
      POST_MSG((COURIER_MESSAGE_NEW(SetPickingTimerReqMsg)(true, NAVI_HMI_SM_C_MAP_PICKING_TIMER_DEFAULT)));
      _infoStore.setMoveLocationFlag(true);
   }
   else
   {
      POST_MSG((COURIER_MESSAGE_NEW(SetPickingTimerReqMsg)(false, 0)));
      _infoStore.setMoveLocationFlag(false);
   }

   // save current map camera mode
   currentMapCameraMode = newMapCameraMode;

   return true;
}


bool HMIModelMap::onPropertyUpdateMapCameraInfosChanged() const
{
   MapViewId mapViewId = _navMiddleware.getMapCameraInfos().getMapViewId();
   ETG_TRACE_USR4(("HMIModelMap::onPropertyUpdateMapCameraInfosChanged(), Map view Id : %d", mapViewId));

   if (navmiddleware::MAP_VIEW_ID__PRIMARY == mapViewId)
   {
      setMapScreenDataMapCameraInfos(_navMiddleware.getMapCameraInfos(), _navMiddleware.getNominalScaleInfos());
   }
   else if (navmiddleware::MAP_VIEW_ID__SECONDARY == mapViewId)
   {
      setSecondaryMapScreenDataMapCameraInfos(_navMiddleware.getMapCameraInfos(), _navMiddleware.getNominalScaleInfos());
   }

   return sendMapScreenData();
}


bool HMIModelMap::onPropertyUpdatePositionInfoChanged()
{
   ETG_TRACE_USR4(("HMIModelMap::onPropertyUpdatePositionInfoChanged()"));

   const navmiddleware::PositionInfo& positionInfo = _navMiddleware.getPositionInfo();
   const navmiddleware::PositionStatusInfo& positionStatusInfo = _navMiddleware.getPositionStatusInfo();
   if ((*getMapScreenData()).mCurrentStreet != positionInfo.getCurrentStreet().c_str())
   {
      (*getMapScreenData()).mCurrentStreet = positionInfo.getCurrentStreet().c_str();
      if ((*getMapScreenData()).mCurrentStreet == EMPTY_STREET_NAME)
      {
         // Updating latitude string with direction
         std::string strLatitude = positionStatusInfo.getLatitudeAsString();
         HemisphereLatitude hemisphereLat = positionStatusInfo.getHemisphereLatitude();
         if (HEMISPHERE_LATITUDE__NONE != hemisphereLat)
         {
            strLatitude = (HEMISPHERE_LATITUDE__NORTHERN == hemisphereLat) ? strLatitude.append("N") : strLatitude.append("S");
         }

         // Updating longitude string with direction
         std::string strLongitude = positionStatusInfo.getLongitudeAsString();
         HemisphereLongitude hemisphereLong = positionStatusInfo.getHemisphereLongitude();
         if (HEMISPHERE_LONGITUDE__NONE != hemisphereLong)
         {
            strLongitude = (HEMISPHERE_LONGITUDE__EASTERN == hemisphereLong) ? strLongitude.append("E") : strLongitude.append("W");
         }

         // String with latitude and longitude value separated by forward slash
         std::string strLatLong = strLatitude + "/" + strLongitude;
         (*getMapScreenData()).mCurrentStreet = strLatLong.c_str();
      }
      getMapScreenData().MarkItemModified(ItemKey::MapScreenData::CurrentStreetItem);

      ::App::Core::ClusterDataClientHandler::getInstance()->updateCurrentStreetName((*getMapScreenData()).mCurrentStreet.GetCString());
   }
   if ((*getMapScreenData()).mCurrentCity != positionInfo.getCurrentCity().c_str())
   {
      (*getMapScreenData()).mCurrentCity = positionInfo.getCurrentCity().c_str();
      getMapScreenData().MarkItemModified(ItemKey::MapScreenData::CurrentCityItem);
   }
   ::App::Core::ClusterDataClientHandler::getInstance()->updateVehiclePosition(positionStatusInfo.getLatitude(), positionStatusInfo.getLongitude());

   return sendMapScreenData();
}


void HMIModelMap::onPropertyUpdateArrivalInfosChanged() const
{
   ETG_TRACE_USR4(("HMIModelMap::onPropertyUpdateArrivalInfosChanged()"));
   const ArrivalInfos& arrivalInfos = _navMiddleware.getArrivalInfos();
   if (_navMiddleware.isGuidanceActive() && arrivalInfos.getArrivalInfos().size())
   {
      bool isArrivalTimeModeActive = (_infoStore.getEstimatedTimeMode() == InfoStore::ARRIVAL_TIME) ? true : false;
      setMapScreenDataArrivalInfos(arrivalInfos, isArrivalTimeModeActive, _infoStore.getEstimatedTimeType(), _infoStore.getShowEtaOnSecMap());
      sendMapScreenData();

      if (false == _isHardRestrictionPresent)
      {
         int destinationIndex = arrivalInfos.getCurrentDestinationIndex();
         if (destinationIndex < arrivalInfos.getArrivalInfos().size())
         {
            ArrivalInfos::ArrivalInfo destInfo = arrivalInfos.getArrivalInfos().at(destinationIndex);
            ::App::Core::ClusterDataClientHandler::getInstance()->updateDestinationInfo(
               destInfo.m_distanceToDest, destInfo.m_estimatedTimeOfArrivalHour, destInfo.m_estimatedTimeOfArrivalMinute, destInfo.m_timeToDest);
         }
      }
      else
      {
         ::App::Core::ClusterDataClientHandler::getInstance()->updateDestinationInfo(0, 0, 0, 0);
      }
   }
   else
   {
      clearMapScreenDataArrivalInfos();
      ::App::Core::ClusterDataClientHandler::getInstance()->updateDestinationInfo(0, 0, 0, 0);
   }
}


bool HMIModelMap::onPropertyUpdateEstimatedTimeModeChanged() const
{
   ETG_TRACE_USR4(("HMIModelMap::onPropertyUpdateEstimatedTimeModeChanged()"));

   const ArrivalInfos& arrivalInfos = _navMiddleware.getArrivalInfos();
   bool isArrivalTimeModeActive = _infoStore.getEstimatedTimeMode() == InfoStore::ARRIVAL_TIME ? true : false;

   // check if we can and need to show the Estimated time mode changes info
   if (arrivalInfos.getArrivalInfos().size())
   {
      setMapScreenDataArrivalInfos(arrivalInfos, isArrivalTimeModeActive, _infoStore.getEstimatedTimeType(), _infoStore.getShowEtaOnSecMap());
   }
   else
   {
      DataBindingItem<MapScreenDataDataBindingSource>& mapScreenData = getMapScreenData();
      (*mapScreenData).mETATextLabelValue = (isArrivalTimeModeActive == true) ? LANGUAGE_STRING(TEXTID_UNAVAILABLE, "ETA") : LANGUAGE_STRING(TextId_0x1456, "Duration");
      mapScreenData.MarkItemModified(ItemKey::MapScreenData::ETATextLabelValueItem);
   }
   return sendMapScreenData();
}


bool HMIModelMap::onPropertyUpdateHardRestrictionStatusChanged()
{
   _isHardRestrictionPresent = _infoStore.getIsHardRestrictionPresent();
   enMapCameraAndViewModeMode mapCameraViewMode = _infoStore.getMapCameraViewMode();
   ETG_TRACE_USR4(("HMIModelMap::onPropertyUpdateHardRestrictionStatusChanged, isHardRestrictionPresent : %d, mapCameraViewMode : %d", _isHardRestrictionPresent, mapCameraViewMode));

   if (true == _isHardRestrictionPresent)
   {
      clearMapScreenDataManeuverInfos();
      sendODRData();
      sendMapScreenData();
      ::App::Core::ClusterDataClientHandler::getInstance()->updateTurnToStreetName("");
      ::App::Core::ClusterDataClientHandler::getInstance()->updateDestinationInfo(0, 0, 0, 0);

      // If junction/intersection screen is active when hard restriction status is true, switch to main map screen.
      // setMapCameraViewMode() is invoked for main map screen also so that ODR is disabled.
      if ((MAP_CAMERA_AND_VIEW_MODE_MAP_JUNCTION_AND_INTERSECTION_SCREEN == mapCameraViewMode)
            || (MAP_CAMERA_AND_VIEW_MODE_MAP_MAIN_SCREEN == mapCameraViewMode))
      {
         setMapCameraViewMode(MAP_CAMERA_AND_VIEW_MODE_MAP_MAIN_SCREEN);
      }
   }
   else
   {
      onPropertyUpdateManeuverInfosChanged();
      onPropertyUpdateArrivalInfosChanged();
      // If main map screen is active when vehicle has crossed hard restriction,
      // setMapCameraViewMode() is invoked to enable ODR, guidance arrow and to check if junction/intersection screen should be shown.
      if (MAP_CAMERA_AND_VIEW_MODE_MAP_MAIN_SCREEN == mapCameraViewMode)
      {
         setMapCameraViewMode(MAP_CAMERA_AND_VIEW_MODE_MAP_MAIN_SCREEN);
      }
   }

   // If map streaming is active, setMapCameraViewMode() has to be invoked again for streamed map,
   // so that guidance arrow visibility is enabled/disabled depending upon hard restriction status in ViewModeOptions.
   for (int regionIndex = 0; regionIndex < MAX_REG; regionIndex++)
   {
      if (true == _infoStore.getMapOutStatus(static_cast<REGION>(regionIndex)))
      {
         POST_MSG((COURIER_MESSAGE_NEW(StartStopMapStreamReqMsg)(regionIndex, true, true)));
      }
   }
   return true;
}


bool HMIModelMap::onPropertyUpdateManeuverInfosChanged() const
{
   if (false == _isHardRestrictionPresent)
   {
      ETG_TRACE_USR4(("HMIModelMap::onPropertyUpdateManeuverInfosChanged()"));
      const NextManeuverInfos& nextManeuverInfos = _navMiddleware.getNextManeuverInfos();
      setMapScreenDataNextManeuverInfos(nextManeuverInfos);
      sendODRData();
      sendMapScreenData();
      ::App::Core::ClusterDataClientHandler::getInstance()->updateTurnToStreetName(nextManeuverInfos.getTurnToStreet());
   }
   return true;
}


bool HMIModelMap::onPropertyUpdateSpeedLimitStatusChanged()
{
   ETG_TRACE_USR4(("HMIModelMap::onPropertyUpdateSpeedLimitStatusChanged(), isSpeedLimitPresent : %d", _navMiddleware.getSpeedInfo().isSpeedLimitPresent()));
   setMapScreenDataSpeedLimitInfos(_navMiddleware.getSpeedInfo(), _navMiddleware.getSafetyGuidanceSettings().isSpeedLimitActive());

   return sendMapScreenData();
}


bool HMIModelMap::onPropertyUpdateOverSpeedStatusChanged()
{
   bool overSpeedWarning = _navMiddleware.getSpeedInfo().getOverSpeedWarning();
   ETG_TRACE_USR4(("HMIModelMap::onPropertyUpdateOverSpeedStatusChanged(), isOverSpeedWarning : %d", overSpeedWarning));

   if (_navMiddleware.getSafetyGuidanceSettings().isOverSpeedWarningActive())
   {
      if (overSpeedWarning)
      {
         POST_MSG((COURIER_MESSAGE_NEW(PlayBeepReqMsg)(hmibase::BEEPTYPE_SPEED_WARN)));
         _timer.setTimeout(0, DELAY_TIME_START_OVER_SPEED_BLINKING, COURIER_MESSAGE_NEW(OverSpeedTimerExpiredReqMsg)());
         _timer.start();
      }
      else
      {
         if (_timer.running())
         {
            _timer.stop();
         }
      }
      setMapScreenDataOverSpeedInfos(overSpeedWarning);
      sendMapScreenData();
   }
   return true;
}


void HMIModelMap::onPropertyUpdateGuidanceStateChanged()
{
   bool isGuidanceActive = _navMiddleware.isGuidanceActive();
   ETG_TRACE_USR4(("HMIModelMap::onPropertyUpdateGuidanceStateChanged(%d)", isGuidanceActive));
   static const char* CMN_IMG_GUIDANCE_CARMODE_ICON_NORMAL = "AppHmi_NavigationModule#Images#Common#Icon_Guidance_carmode_normalBmp";
   static const char* CMN_IMG_GUIDANCE_CARMODE_ICON_PRESSED = "AppHmi_NavigationModule#Images#Common#Icon_Guidance_carmode_touchedBmp";
   static const char* CMN_IMG_GUIDANCE_COACH_ICON_NORMAL = "AppHmi_NavigationModule#Images#Common#Icon_Guidance_normalBmp";
   static const char* CMN_IMG_GUIDANCE_COACH_ICON_PRESSED = "AppHmi_NavigationModule#Images#Common#Icon_Guidance_touchedBmp";

   computeGuidanceRuntime(isGuidanceActive);
   if (false == isGuidanceActive)
   {
      clearMapScreenDataArrivalInfos();
      clearMapScreenDataManeuverInfos();
      sendODRData();

      ::App::Core::ClusterDataClientHandler::getInstance()->updateDestinationInfo(0, 0, 0, 0);
      ::App::Core::ClusterDataClientHandler::getInstance()->updateTurnToStreetName("");
   }
   if (isGuidanceActive && _infoStore.getVehicleProfile()) // car mode
   {
      (*getMapScreenData()).mMapActiveRGNaviIconNormal = ImageLoader::getAssetBitmapImage(CMN_IMG_GUIDANCE_CARMODE_ICON_NORMAL);
      (*getMapScreenData()).mMapActiveRGNaviIconPressed = ImageLoader::getAssetBitmapImage(CMN_IMG_GUIDANCE_CARMODE_ICON_PRESSED);
   }
   else
   {
      (*getMapScreenData()).mMapActiveRGNaviIconNormal = ImageLoader::getAssetBitmapImage(CMN_IMG_GUIDANCE_COACH_ICON_NORMAL);
      (*getMapScreenData()).mMapActiveRGNaviIconPressed = ImageLoader::getAssetBitmapImage(CMN_IMG_GUIDANCE_COACH_ICON_PRESSED);
   }
   getMapScreenData().MarkItemModified(ItemKey::MapScreenData::MapActiveRGNaviIconNormalItem);
   getMapScreenData().MarkItemModified(ItemKey::MapScreenData::MapActiveRGNaviIconPressedItem);
   sendMapScreenData();
}


bool HMIModelMap::onCourierMessage(const SetMapCameraModeReqMsg& oMsg)
{
   enMapCameraAndViewModeMode mapCameraViewMode = oMsg.GetMapCameraMode();
   ETG_TRACE_USR4(("HMIModelMap::onCourierMessage(SetMapCameraModeReqMsg), MapCameraViewMode : %d", mapCameraViewMode));

   setMapCameraViewMode(mapCameraViewMode);
   return true;
}


void HMIModelMap::setMapCameraViewMode(enMapCameraAndViewModeMode mapCameraViewMode)
{
   bool updateMapModeInInfostore = true;
   ETG_TRACE_USR1(("HMIModelMap::setMapCameraViewMode(), MapCameraViewMode : %d", mapCameraViewMode));

   if (true == _infoStore.getIsNavMiddlewareStarted())   // we only set a map mode if navmiddleware is initialized
   {
      switch (mapCameraViewMode)
      {
         case MAP_CAMERA_AND_VIEW_MODE_MAP_MAIN_SCREEN:
         {
            // If GUIDANCE_MANEUVERVIEW_CHANGED is received while in any other map mode, then on returning to map main screen,
            // if junction/intersection view is still valid, then the corresponding map mode is displayed.
            if ((false == _isHardRestrictionPresent)
                  &&
                  ((isIntersectionMapActive(_navMiddleware)
                    && (navmiddleware::settings::MAP_HMI_SPLITMODE_INTERSECTION == _navMiddleware.getMapViewSettings().getMapHMISplitMode()))
                   ||
                   isJunctionViewActive(_navMiddleware)
                  )
               )
            {
               mapCameraViewMode = MAP_CAMERA_AND_VIEW_MODE_MAP_JUNCTION_AND_INTERSECTION_SCREEN;
            }
            else if (true == _infoStore.getMultipleRouteCalculationStatus())
            {
               if (navmiddleware::settings::MORE_ROUTES_TYPE_MULTIPLE == _navMiddleware.getRouteOptionSettings().getMoreRoutesType())
               {
                  mapCameraViewMode = MAP_CAMERA_AND_VIEW_MODE_ROUTE_OVERVIEW_MULTIPLE;
               }
               else
               {
                  // If user had changed route type to Single, then the active multiple route overview is removed.
                  POST_MSG(COURIER_MESSAGE_NEW(CancelMultipleRouteReqMsg)());
               }
            }
            break;
         }
         case MAP_CAMERA_AND_VIEW_MODE_MAP_JUNCTION_AND_INTERSECTION_SCREEN:
         {
            // On back transition from any other map scene (request from SM), if previous map mode was intersection/junction view,
            // but if it is no longer active, then map mode is set to main map
            if ((true == _isHardRestrictionPresent)
                  ||
                  (!(isIntersectionMapActive(_navMiddleware)
                     && (navmiddleware::settings::MAP_HMI_SPLITMODE_INTERSECTION == _navMiddleware.getMapViewSettings().getMapHMISplitMode()))
                   && (!isJunctionViewActive(_navMiddleware)))
               )
            {
               mapCameraViewMode = MAP_CAMERA_AND_VIEW_MODE_MAP_MAIN_SCREEN;
            }
            break;
         }
         case STREAMED_MAP_OVERVIEW_MAP:
         case STREAMED_MAP_2D_MAP:
         case STREAMED_MAP_3D_MAP:
         case STOP_MAP_STREAM:
         {
            updateMapModeInInfostore = false;
            break;
         }
         default:
         {
         }
      }

      if (updateMapModeInInfostore)
      {
         _infoStore.setMapCameraViewMode(mapCameraViewMode);
      }

      // inform map camera and view handler about new map camera mode
      _mapCameraAndViewHandler.handleMapCameraAndViewMode(mapCameraViewMode);
   }
}


void HMIModelMap::freezeMap()
{
   ETG_TRACE_USR1(("HMIModelMap::freezeMap(), isMapActive %d", _isMapActive));

   if ((true == _infoStore.getIsNavMiddlewareStarted()) && _isMapActive)
   {
      ETG_TRACE_USR1(("HMIModelMap::freezeMap(), Map freeze requested"));
      _navMiddleware.freezeMap(MAP_VIEW_ID__PRIMARY, navmiddleware::MAP_VIEW_VISIBILITY__VISIBLE);
      _navMiddleware.freezeMap(MAP_VIEW_ID__SECONDARY, navmiddleware::MAP_VIEW_VISIBILITY__VISIBLE);
      _isMapActive = false;
   }
}


void HMIModelMap::freezeAndHideMap()
{
   ETG_TRACE_USR1(("HMIModelMap::freezeAndHideMap(), isMapActive %d", _isMapActive));

   // make the map invisible hide the map when HMISubState is OFF (to avoid flickers)
   _navMiddleware.freezeMap(MAP_VIEW_ID__PRIMARY, navmiddleware::MAP_VIEW_VISIBILITY__HIDDEN);
   _navMiddleware.freezeMap(MAP_VIEW_ID__SECONDARY, navmiddleware::MAP_VIEW_VISIBILITY__HIDDEN);
   _isMapActive = false;
}


void HMIModelMap::unfreezeMap()
{
   ETG_TRACE_USR1(("HMIModelMap::unfreezeMap(), isMapActive %d", _isMapActive));

   if ((true == _infoStore.getIsNavMiddlewareStarted()) && !_isMapActive)
   {
      ETG_TRACE_USR1(("HMIModelMap::unfreezeMap(), Map unfreeze requested"));
      _navMiddleware.unFreezeMap(MAP_VIEW_ID__PRIMARY);
      _navMiddleware.unFreezeMap(MAP_VIEW_ID__SECONDARY);
      _isMapActive = true;
   }
}


bool HMIModelMap::onCourierMessage(const ButtonReactionMsg& oMsg)
{
   const Courier::ViewId MAP_MOVE_SCENE_NAME = Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#N_Map_Move");
   const Courier::ViewId LEFT_MAP_SCALE_SCENE_NAME = Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#N_MapScale");
   const Courier::ViewId MAP_SCENE_NAME      = Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#N_Map");
   const Courier::Identifier  IdZoomWithOutScaleBtn = Courier::Identifier("Layer_MapScale/Layer_MapScaleBtn/Button");
   const Courier::Identifier  IdGotoHomeBtn = Courier::Identifier("Layer_SaveAsHome/ButtonWidget");
   const Courier::Identifier  IdShowMapBtn = Courier::Identifier("Layer_ShowMapBtn/Button");
   const Courier::Identifier  IdRetriggerVoiceBtn = Courier::Identifier("Layer_RetriggerVoiceBtn/Button");
   const Courier::Identifier  IdGuidanceVoiceMuteBtn = Courier::Identifier("Layer_Mute/ButtonWidget2D");
   const Courier::ViewId  MAP_ACTIVE_RG = Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#N_Map_ActiveRG");
   const Courier::ViewId  GUIDANCE_SCENE_NAME = Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#N_Guidance");
   const Courier::Identifier  IdUnmuteBtn = Courier::Identifier("Layer_Unmute/ButtonWidget");
   const Courier::Identifier senderInfo = oMsg.GetSender();
   const Courier::ViewId sceneName = oMsg.GetView();
   const enReaction  btnReaction = oMsg.GetEnReaction();
   bool isMsgConsumed = false;

   if (enRelease == btnReaction)
   {
      if (((sceneName == LEFT_MAP_SCALE_SCENE_NAME) || (sceneName == MAP_MOVE_SCENE_NAME) || (sceneName == MAP_SCENE_NAME)) && (senderInfo == IdZoomWithOutScaleBtn))
      {
         (*_zoomBarInfo).mIsZoomTimerStart = true;
         (*_zoomBarInfo).mIsZoomBarDisplayed = true;
         _zoomBarInfo.MarkAllItemsModified();
         _zoomBarInfo.SendUpdate();
         isMsgConsumed = true;
      }
      else if ((sceneName == MAP_ACTIVE_RG) && (senderInfo == IdShowMapBtn))
      {
         (*getMapScreenData()).mShowMapActiveRGButtons = true;
         (*getMapScreenData()).mStartShowMapActiveRGButtonTimer = true;
         getMapScreenData().MarkItemModified(ItemKey::MapScreenData::ShowMapActiveRGButtonsItem);
         getMapScreenData().MarkItemModified(ItemKey::MapScreenData::StartShowMapActiveRGButtonTimerItem);
         sendMapScreenData();
      }
      else if ((sceneName == MAP_ACTIVE_RG) && (senderInfo == IdRetriggerVoiceBtn))
      {
         ETG_TRACE_USR4(("HMIModelMap::onCourierMessage(ButtonReactionMsg), Retriggering Voice Output"));
         _navMiddleware.retriggerAcousticOutput();

         (*getMapScreenData()).mShowMapActiveRGButtons = true;
         (*getMapScreenData()).mStartShowMapActiveRGButtonTimer = true;
         getMapScreenData().MarkItemModified(ItemKey::MapScreenData::ShowMapActiveRGButtonsItem);
         getMapScreenData().MarkItemModified(ItemKey::MapScreenData::StartShowMapActiveRGButtonTimerItem);
         sendMapScreenData();
      }
      else if (((sceneName == MAP_ACTIVE_RG) && (senderInfo == IdGuidanceVoiceMuteBtn)) || ((sceneName == GUIDANCE_SCENE_NAME) && (senderInfo == IdUnmuteBtn)))
      {
         navmiddleware::settings::GuidanceSettings guidanceSettings = _navMiddleware.getGuidanceSettings();
         guidanceSettings.getVoiceGuidanceMode() ? guidanceSettings.setVoiceGuidanceMode(false) : guidanceSettings.setVoiceGuidanceMode(true);
         _navMiddleware.updateGuidanceSettings(guidanceSettings);
         _navMiddleware.applySettings();

         (*getMapScreenData()).mShowMapActiveRGButtons = true;
         (*getMapScreenData()).mStartShowMapActiveRGButtonTimer = true;
         getMapScreenData().MarkItemModified(ItemKey::MapScreenData::ShowMapActiveRGButtonsItem);
         getMapScreenData().MarkItemModified(ItemKey::MapScreenData::StartShowMapActiveRGButtonTimerItem);
         sendMapScreenData();
      }
      // Below use-case is not executed currently
      else if ((sceneName == MAP_SCENE_NAME) && (senderInfo == IdGotoHomeBtn))
      {
         const navmiddleware::DestinationMemoryStatus& destinationMemoryStatus = _navMiddleware.getDestinationMemoryStatus();
         const int HOME_DETAILEDINFO = 1;
         const int FAVORITE_ICON = 1;
         const int SHOW_DELETE_BUTTON = 1;

         if (destinationMemoryStatus.getNumberOfHomeEntries() == 0)
         {
            POST_MSG((COURIER_MESSAGE_NEW(GotoHomeReqMsg)(false)));
         }
         else
         {
            //Show detailed info view with home details
            POST_MSG((COURIER_MESSAGE_NEW(ShowDetailedAddressMsg)()));
            POST_MSG((COURIER_MESSAGE_NEW(ShowFavEditIcon_Enable_Delete_Guidance_Buttons)(FAVORITE_ICON, SHOW_DELETE_BUTTON, 1, 0, false)));
            POST_MSG((COURIER_MESSAGE_NEW(GotoHomeReqMsg)(true)));
         }
         isMsgConsumed = true;
      }
   }

   return isMsgConsumed;
}


bool HMIModelMap::onCourierMessage(const TimerWidgetExpiryMsg& oMsg)
{
   (*getMapScreenData()).mShowMapActiveRGButtons = false;
   (*getMapScreenData()).mStartShowMapActiveRGButtonTimer = false;
   getMapScreenData().MarkItemModified(ItemKey::MapScreenData::ShowMapActiveRGButtonsItem);
   getMapScreenData().MarkItemModified(ItemKey::MapScreenData::StartShowMapActiveRGButtonTimerItem);
   sendMapScreenData();

   return true;
}


bool HMIModelMap::onCourierMessage(const ZoomInReqMsg& oMsg)
{
   bool isSplitMapActive = oMsg.GetIsSecondaryViewActive();
   if (false == isSplitMapActive)
   {
      _navMiddleware.zoomInStep(MAP_VIEW_ID__PRIMARY);
   }
   else
   {
      _navMiddleware.zoomInStep(MAP_VIEW_ID__SECONDARY);
   }

   (*getMapScreenData()).mShowMapActiveRGButtons = true;
   (*getMapScreenData()).mStartShowMapActiveRGButtonTimer = true;
   getMapScreenData().MarkItemModified(ItemKey::MapScreenData::ShowMapActiveRGButtonsItem);
   getMapScreenData().MarkItemModified(ItemKey::MapScreenData::StartShowMapActiveRGButtonTimerItem);
   sendMapScreenData();
   return true;
}


bool HMIModelMap::onCourierMessage(const ZoomOutReqMsg& oMsg)
{
   bool isSplitMapActive = oMsg.GetIsSecondaryViewActive();
   if (false == isSplitMapActive)
   {
      _navMiddleware.zoomOutStep(MAP_VIEW_ID__PRIMARY);
      (*_zoomBarInfo).mIsZoomTimerStart = true;
      _zoomBarInfo.MarkItemModified(ItemKey::ZoomBarInfo::IsZoomTimerStartItem);
      _zoomBarInfo.SendUpdate();
   }
   else
   {
      _navMiddleware.zoomOutStep(MAP_VIEW_ID__SECONDARY);
   }

   (*getMapScreenData()).mShowMapActiveRGButtons = true;
   (*getMapScreenData()).mStartShowMapActiveRGButtonTimer = true;
   getMapScreenData().MarkItemModified(ItemKey::MapScreenData::ShowMapActiveRGButtonsItem);
   getMapScreenData().MarkItemModified(ItemKey::MapScreenData::StartShowMapActiveRGButtonTimerItem);
   sendMapScreenData();
   return true;
}


bool HMIModelMap::onCourierMessage(const ReCenterMapScrollReqMsg& oMsg)
{
   ETG_TRACE_USR1(("HMIModelMap::onCourierMessage(ReCenterMapScrollReqMsg)"));

   COURIER_UNUSED(oMsg);

   navmiddleware::ValidValue<int32_t> scale;
   if (EXT_bCenterMapToCurrentPosition == 1)   // In this use-case, map has to be centered on CCP. EXT_bCenterMapToCurrentPosition is set to 1 in SM.
   {
      // This condition check is met when re-center button is pressed in N_Map_Move scene.
      _navMiddleware.centerMapToGeoPosition(MAP_VIEW_ID__PRIMARY, navmiddleware::GeoCoordinateDegree(_navMiddleware.getPositionStatusInfo().getLatitude(), _navMiddleware.getPositionStatusInfo().getLongitude()), scale);

      // EXT_bCenterMapToCurrentPosition is reset so that in other use-cases, map is centered on the coordinates updated in infostore.
      EXT_bCenterMapToCurrentPosition = 0;
   }
   else
   {
      if (_infoStore.getCoordinatesToBeShownInMap()._coordinates.size())
      {
         _navMiddleware.centerMapToGeoPosition(MAP_VIEW_ID__PRIMARY, navmiddleware::GeoCoordinateDegree(_infoStore.getCoordinatesToBeShownInMap()._coordinates.front()._latitude, _infoStore.getCoordinatesToBeShownInMap()._coordinates.front()._longitude), scale);
      }
   }
   return true;
}


void HMIModelMap::onPropertyUpdateMapVideoStreamStatusChanged()
{
   MapVideoStreamStatusInfo streamInfo = _navMiddleware.getMapVideoStreamStatusInfo();
   MapVideoStreamStatusInfo::MapVideoStreamStatus status = streamInfo.getMapVideoStreamStatus();
   ETG_TRACE_USR4(("HMIModelMap::onPropertyUpdateMapVideoStreamStatusChanged, Stream status : %d", status));

   InfoStoreBase::MapOutRegionStatus mapOutRegionStatus = _infoStore.getMapOutRequest();

   if (MapVideoStreamStatusInfo::MAP_VIDEO_STREAM_STATUS__AVAILABLE == status)
   {
      // If map streaming is already active in any one of the cabin regions, then it is not required to execute the below code again.
      // The below code block is for disabling split/3D views if already active and for activating map out overlays.
      if (false == _infoStore.isSecondaryMapStreamingActive())
      {
         _currentMapView = _navMiddleware.getChangeMapViewSettings().getMapView();
         _currentMapHMISplitMode = _navMiddleware.getMapViewSettings().getMapHMISplitMode();
         if (!isJunctionViewActive(_navMiddleware))   // Condition check added for Bug fix: RTC-980397
         {
            switchTo2DMapOnStreaming();
         }
         else
         {
            _isMapViewUpdateNeeded = true;
         }

         if ((true == EXT_bIsGuidanceActive) || (_timeToReachDest != DURATION_CABIN_ONLY))
         {
            ETG_TRACE_USR4(("HMIModelMap::onPropertyUpdateMapVideoStreamStatusChanged(), Enable MapOverlay Direct Textures"));
            POST_MSG((COURIER_MESSAGE_NEW(::hmibase::gadget::ActivateImageProviderReqMsg)(MAP_OVERLAY, TEXTURE_ACTIVE)));
         }
      }

      // Map out status is updated in InfoStore and the request is removed from the queue
      _infoStore.setMapOutStatus(mapOutRegionStatus.regionId, true);
      _infoStore.removeMapOutRequest();
   }
   else
   {
      _infoStore.setMapOutStatus(mapOutRegionStatus.regionId, false);
      _infoStore.removeMapOutRequest();

      // Only if map streaming is not active in all the cabin regions, then it is required to execute the below code.
      // The below code block is for enabling split/3D views if previously active and for de-activating map out overlays.
      if (false == _infoStore.isSecondaryMapStreamingActive())
      {
         if (!isJunctionViewActive(_navMiddleware))
         {
            restoreLastMapViewAfterStreaming();
         }
         else
         {
            _isMapViewUpdateNeeded = true;
         }

         if ((true == EXT_bIsGuidanceActive) || (_timeToReachDest != DURATION_CABIN_ONLY))
         {
            ETG_TRACE_USR4(("HMIModelMap::onPropertyUpdateMapVideoStreamStatusChanged(), Disable MapOverlay Direct Textures"));
            POST_MSG((COURIER_MESSAGE_NEW(::hmibase::gadget::ActivateImageProviderReqMsg)(MAP_OVERLAY, TEXTURE_INACTIVE)));
         }
      }
   }

   // For sending information to Master regarding the map out status in cabin region
   POST_MSG((COURIER_MESSAGE_NEW(MapOutStatusUpdMsg)(mapOutRegionStatus.regionId, status)));

   if (true == _infoStore.isMapOutRequestPending())
   {
      mapOutRegionStatus = _infoStore.getMapOutRequest();
      startStopMapStreamRequest(mapOutRegionStatus.regionId, mapOutRegionStatus.mapOutStatus);
   }
}


void HMIModelMap::onPropertyUpdateManeuverViewChanged()
{
   ETG_TRACE_USR4(("HMIModelMap::onPropertyUpdateManeuverViewChanged()"));

   if ((true == _isMapViewUpdateNeeded) && (!isJunctionViewActive(_navMiddleware)))
   {
      _isMapViewUpdateNeeded = false;
      if (true == _infoStore.isSecondaryMapStreamingActive())
      {
         switchTo2DMapOnStreaming();
      }
      else
      {
         restoreLastMapViewAfterStreaming();
      }
   }
}


void HMIModelMap::switchTo2DMapOnStreaming()
{
   ETG_TRACE_USR4(("HMIModelMap::switchTo2DMapOnStreaming()"));

   navmiddleware::settings::ChangeMapViewSettings mapChangeViewSettings = _navMiddleware.getChangeMapViewSettings();
   navmiddleware::settings::MapViewSettings mapViewSettings = _navMiddleware.getMapViewSettings();
   _currentMapView = mapChangeViewSettings.getMapView();
   _currentMapHMISplitMode = mapViewSettings.getMapHMISplitMode();

   bool hasMapSettingChanged = false;
   if (navmiddleware::settings::MAPVIEW_2D != _currentMapView)
   {
      mapChangeViewSettings.setMapView(navmiddleware::settings::MAPVIEW_2D);
      _navMiddleware.updateChangeMapViewSettings(mapChangeViewSettings);
      hasMapSettingChanged = true;

      // To-Do: In the below if condition, the pop-up has to be displayed for MAPVIEW_2D_SPLIT once translation text is corrected.
      if ((navmiddleware::settings::MAPVIEW_3D == _currentMapView) ||
            (navmiddleware::settings::MAPVIEW_3D_SPLIT == _currentMapView))
      {
         ETG_TRACE_USR4(("HMIModelMap::switchTo2DMapOnStreaming(), Display pop-up since the current map view is not 2D"));
         (*_multiLinePopuptext).mTextLine1 = DATA_CONTEXT__TEXT_FIRSTLINE;
         (*_multiLinePopuptext).mTextLine2 = DATA_CONTEXT__TEXT_SECONDLINE;
         _multiLinePopuptext.MarkAllItemsModified();
         _multiLinePopuptext.SendUpdate(true);
         if (!_infoStore.isNonNaviVariant())
         {
            POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_MapOut_Notification"))));
         }
      }
   }

   if (navmiddleware::settings::MAP_HMI_SPLITMODE_FULL_MAP != _currentMapHMISplitMode)
   {
      mapViewSettings.setMapHMISplitMode(navmiddleware::settings::MAP_HMI_SPLITMODE_FULL_MAP);
      _navMiddleware.updateMapViewSettings(mapViewSettings);
      hasMapSettingChanged = true;
   }

   if (true == hasMapSettingChanged)
   {
      applyMapSettingsOnStreaming();
   }
}


void HMIModelMap::restoreLastMapViewAfterStreaming()
{
   ETG_TRACE_USR4(("HMIModelMap::restoreLastMapViewAfterStreaming()"));

   navmiddleware::settings::ChangeMapViewSettings mapChangeViewSettings = _navMiddleware.getChangeMapViewSettings();
   navmiddleware::settings::MapViewSettings mapViewSettings = _navMiddleware.getMapViewSettings();

   bool hasMapSettingChanged = false;
   if (mapChangeViewSettings.getMapView() != _currentMapView)
   {
      mapChangeViewSettings.setMapView(_currentMapView);
      _navMiddleware.updateChangeMapViewSettings(mapChangeViewSettings);
      hasMapSettingChanged = true;
   }

   if (mapViewSettings.getMapHMISplitMode() != _currentMapHMISplitMode)
   {
      mapViewSettings.setMapHMISplitMode(_currentMapHMISplitMode);
      _navMiddleware.updateMapViewSettings(mapViewSettings);
      hasMapSettingChanged = true;
   }

   if (true == hasMapSettingChanged)
   {
      applyMapSettingsOnStreaming();
   }
}


void HMIModelMap::applyMapSettingsOnStreaming()
{
   ETG_TRACE_USR4(("HMIModelMap::applyMapSettingsOnStreaming()"));

   _navMiddleware.applySettings();
   POST_MSG((COURIER_MESSAGE_NEW(UpdateMapSettingsListReqMsg)()));

   const enMapCameraAndViewModeMode& mapCameraViewMode = _infoStore.getMapCameraViewMode();
   if (MAP_CAMERA_AND_VIEW_MODE_MAP_MAIN_SCREEN == mapCameraViewMode)
   {
      // If map is currently displayed scene, then to refresh the map with updated map view settings, the below function is invoked again.
      _mapCameraAndViewHandler.handleMapCameraAndViewMode(mapCameraViewMode);
   }
}


bool HMIModelMap::onCourierMessage(const StartStopMapStreamReqMsg& oMsg)
{
   REGION regionId = static_cast<REGION>(oMsg.GetRegionId());
   bool newMapOutState = oMsg.GetStartMapStream();
   bool forceUpdate = oMsg.GetForceUpdate();
   ETG_TRACE_USR4(("HMIModelMap::onCourierMessage(StartStopMapStreamReqMsg)(Region id : %d, Status : %d, ForceUpdate : %d)", regionId, newMapOutState, forceUpdate));

   if (regionId < MAX_REG)
   {
      bool currentMapOutState = _infoStore.getMapOutStatus(regionId);
      if ((true == forceUpdate) || (currentMapOutState != newMapOutState))
      {
         // If there are pending requests in the queue, newly received request is added to the queue.
         // Else, the newly received request is processed
         bool processRequest = (false == _infoStore.isMapOutRequestPending()) ? true : false;

         InfoStoreBase::MapOutRegionStatus mapOutRegionStatus(regionId, newMapOutState);
         _infoStore.addMapOutRequest(mapOutRegionStatus);

         if (true == processRequest)
         {
            startStopMapStreamRequest(regionId, newMapOutState);
         }
      }
      else
      {
         ETG_TRACE_USR4(("HMIModelMap::onCourierMessage(StartStopMapStreamReqMsg), Current map out state is same as requested state"));
         POST_MSG((COURIER_MESSAGE_NEW(MapOutStatusUpdMsg)(regionId, newMapOutState)));   // For sending information to Master regarding the map out status in cabin region
      }
   }
   else
   {
      // Request with invalid region id is ignored and map out status is updated to Master as false
      POST_MSG((COURIER_MESSAGE_NEW(MapOutStatusUpdMsg)(regionId, false)));
   }
   return true;
}


void HMIModelMap::startStopMapStreamRequest(const REGION& regionId, const bool& status)
{
   ETG_TRACE_USR4(("HMIModelMap::startStopMapStreamRequest(), Region id: %d, Status : %d", regionId, status));

   if (regionId < MAX_REG)
   {
      enMapCameraAndViewModeMode mapCameraAndViewMode = (true == status) ? STREAMED_MAP_2D_MAP : STOP_MAP_STREAM;
      POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(SetMapCameraModeReqMsg)(mapCameraAndViewMode)));
   }
   else
   {
      ETG_TRACE_ERR(("HMIModelMap::startStopMapStreamRequest(), Invalid region id : %d", regionId));
   }
}


bool HMIModelMap::onCourierMessage(const SetMapLayerVisibilityReqMsg& oMsg)
{
   bool isMapLayerVisible = oMsg.GetIsMapLayerVisible();
   ETG_TRACE_USR4(("HMIModelMap::onCourierMessage(SetMapLayerVisibilityReqMsg)(Map layer visibility : %d)", isMapLayerVisible));

   SurfaceSynchronizationHandler::setMapLayerVisibility(isMapLayerVisible);
   return true;
}


bool HMIModelMap::onPropertyUpdateMapStreamStateChanged()
{
   computeMapStreamingRuntime(_infoStore.isSecondaryMapStreamingActive());
   return true;
}


void HMIModelMap::computeGuidanceRuntime(bool guidanceStatus)
{
   static uint32_t guidanceStartTime = 0;
   static uint32_t guidanceStopTime = 0;
   static bool isGuidanceActive = false;

   if (isGuidanceActive != guidanceStatus)
   {
      isGuidanceActive = guidanceStatus;

      if (true == isGuidanceActive)
      {
         guidanceStartTime = OSAL_ClockGetElapsedTime();
         ETG_TRACE_COMP(("HMIModelMap::computeGuidanceRuntime(), Guidance start time : %d ms", guidanceStartTime));
      }
      else
      {
         guidanceStopTime = OSAL_ClockGetElapsedTime();
         ETG_TRACE_COMP(("HMIModelMap::computeGuidanceRuntime(), Guidance stop time : %d ms", guidanceStopTime));

         uint32_t duration = (guidanceStopTime - guidanceStartTime) / 1000;   // Convert milliseconds to seconds
         guidanceStartTime = guidanceStopTime = 0;
         ::App::Core::DiagnosticsClientHandler::getInstance()->updateGuidanceRuntimeInfo(duration);
      }
   }
}


void HMIModelMap::computeMapStreamingRuntime(bool streamStatus)
{
   static uint32_t streamingStartTime = 0;
   static uint32_t streamingStopTime = 0;
   static bool isMapStreamingActive = false;

   if (isMapStreamingActive != streamStatus)
   {
      isMapStreamingActive = streamStatus;

      if (true == isMapStreamingActive)
      {
         streamingStartTime = OSAL_ClockGetElapsedTime();
         ETG_TRACE_COMP(("HMIModelMap::computeMapStreamingRuntime(), Streaming start time : %d ms", streamingStartTime));
      }
      else
      {
         streamingStopTime = OSAL_ClockGetElapsedTime();
         ETG_TRACE_COMP(("HMIModelMap::computeMapStreamingRuntime(), Streaming stop time : %d ms", streamingStopTime));

         uint32_t duration = (streamingStopTime - streamingStartTime) / 1000;
         streamingStartTime = streamingStopTime = 0;
         ::App::Core::DiagnosticsClientHandler::getInstance()->updateMapStreamingRuntimeInfo(duration);
      }
   }
}


void HMIModelMap::sendCANDataToCalculateETACabinOnly(const uint16 timeToReachDest, const uint8 tmMode, const uint64 timeDate)
{
   ETG_TRACE_USR4(("VehicleDataClientHandler::sendCANDataToCalculateETACabinOnly: %d %d %d", timeToReachDest, timeDate, tmMode));
   _timeToReachDest = timeToReachDest;
   _timeDate = timeDate;
   _tmMode = tmMode;
   computeDurationandETACabinOnly(_timeToReachDest, _timeDate, _tmMode);
}


bool HMIModelMap::onCourierMessage(const ETAAvailableUpdateMsg& oMsg)
{
   bool isETAAvailable = oMsg.GetIsETAAvailable();
   ETG_TRACE_USR4(("HMIModelMap::onCourierMessage(ETAAvailableUpdateMsg)(ETAAvailable : %d)", isETAAvailable));
   DataBindingItem<MapoutScreenDataDataBindingSource>& mapoutScreenData = getMapoutScreenData();
   if ((*mapoutScreenData).mShowEtaDurationCabinOnly != isETAAvailable)
   {
      (*mapoutScreenData).mShowEtaDurationCabinOnly = isETAAvailable;
      mapoutScreenData.MarkItemModified(ItemKey::MapoutScreenData::ShowEtaDurationCabinOnlyItem);
   }
   sendMapoutScreenData();
   return true;
}


#endif // HALL_TO_MDW_COM
