/**************************************************************************************
* @file         : SignpostDisplayHandler.cpp
* @author       :
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) 2018-2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include "SignpostDisplayHandler.h"
#include "CgiExtensions/ImageLoader.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/SignpostDisplayHandler.cpp.trc.h"
#endif

#if defined (HALL_TO_MDW_COM)

static const float SIGNPOST_POS_LEFT_X = 72.0f;
static const float SIGNPOST_POS_MIDDLE_X = 108.0f;
static const float SIGNPOST_POS_RIGHT_X = 144.0f;
static const float SIGNPOST_POS_DEFAULT_Y = 56.0f;
static const char* const DATA_CONTEXT_LIST_ROADICONS_ITEM = "RoadIconsListItem";
static const float BACKGROUND_COLOR_SCALE_FACTOR = 255.0f;

SignpostDisplayHandler::SignpostDisplayHandler(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore)
   : HMIModelBase(navMiddleware, infoStore)
{
}


SignpostDisplayHandler::~SignpostDisplayHandler()
{
}


void SignpostDisplayHandler::initialize()
{
   _navMiddleware.registerGuidancePropertyUpdateCallback(*this);
   _infoStore.registerDataPropertyUpdateCallback(*this);
}


void SignpostDisplayHandler::deinitialize()
{
   _navMiddleware.unregisterGuidancePropertyUpdateCallback(*this);
   _infoStore.unregisterDataPropertyUpdateCallback(*this);
}


void SignpostDisplayHandler::onPropertyUpdateGuidanceSignpostInfoUpdate()
{
   ETG_TRACE_USR4(("SignpostDisplayHandler::onPropertyUpdateGuidanceSignpostInfoUpdate()"));
   if(false == _infoStore.getIsHardRestrictionPresent())
   {
      if (_navMiddleware.isGuidanceActive())
      {
         bool isSignpostAvailable = _navMiddleware.getSignPostInfo().isSignPostAvailable();
         ETG_TRACE_USR4(("SignpostDisplayHandler::onPropertyUpdateGuidanceSignpostInfoUpdate(), isSignpostAvailable : %d", isSignpostAvailable));
         setSignpostActive(isSignpostAvailable);
         // if signpost details available
         if (isSignpostAvailable)
         {
            // Display the Exit Number
            setExitNumber();
            // Display line 1 and line 2 signpost info
            setSignpostTextLines();
            // signpost area background color assignment
            setSignpostBackgroundColor();
            // signpost text lines color assignment
            setSignpostTextColor();
            setRoadNumberIconDirection();
            setSignpostPosition();
            sendSignpostTextData();
         }
      }
      else
      {
         setSignpostActive(false);
      }
   }
}


bool SignpostDisplayHandler::onPropertyUpdateHardRestrictionStatusChanged()
{
   bool isHardRestrictionPresent = _infoStore.getIsHardRestrictionPresent();
   ETG_TRACE_USR4(("SignpostDisplayHandler::onPropertyUpdateHardRestrictionStatusChanged(), isHardRestrictionPresent : %d, isSignPostActive : %d", isHardRestrictionPresent, (*_signpostTextData).mSignpostActive));

   if(true == isHardRestrictionPresent)
   {
      if(true == (*_signpostTextData).mSignpostActive)
      {
         setSignpostActive(false);
      }
   }
   else
   {
      onPropertyUpdateGuidanceSignpostInfoUpdate();
   }
   return true;
}


void SignpostDisplayHandler::setSignpostActive(bool isSignpostAvailable)
{
   if ((*_signpostTextData).mSignpostActive != isSignpostAvailable)
   {
      (*_signpostTextData).mSignpostActive = isSignpostAvailable;
      _signpostTextData.MarkItemModified(ItemKey::SignpostText::SignpostActiveItem);
      sendSignpostTextData();
   }
}


void SignpostDisplayHandler::setSignpostTextLines()
{
   const std::vector<std::string>& signpostTextLines = _navMiddleware.getSignPostInfo().getTextLines();

   //clear previous data from lines
   (*_signpostTextData).mSignpostfirstline = "";
   (*_signpostTextData).mSignpostsecondline = "";

   int index = 0;
   if (index < signpostTextLines.size())
   {
      (*_signpostTextData).mSignpostfirstline = signpostTextLines[index].c_str();
      index++;

      if (index < signpostTextLines.size())
      {
         (*_signpostTextData).mSignpostsecondline = signpostTextLines[index].c_str();
      }
   }
   _signpostTextData.MarkItemModified(ItemKey::SignpostText::SignpostfirstlineItem);
   _signpostTextData.MarkItemModified(ItemKey::SignpostText::SignpostsecondlineItem);
}


void SignpostDisplayHandler::setSignpostBackgroundColor()
{
   const navmiddleware::Color& bgColor = _navMiddleware.getSignPostInfo().getBackgroundColor();
   Candera::Color backgroundImageColor = Candera::Color((static_cast<float>(bgColor.m_red) / BACKGROUND_COLOR_SCALE_FACTOR), (static_cast<float>(bgColor.m_green) / BACKGROUND_COLOR_SCALE_FACTOR), (static_cast<float>(bgColor.m_blue) / BACKGROUND_COLOR_SCALE_FACTOR), (static_cast<float>(bgColor.m_alpha) / BACKGROUND_COLOR_SCALE_FACTOR));
   if ((*_signpostTextData).mBackgroundImageColor != backgroundImageColor)
   {
      (*_signpostTextData).mBackgroundImageColor = backgroundImageColor;
      _signpostTextData.MarkItemModified(ItemKey::SignpostText::BackgroundImageColorItem);
   }
}


void SignpostDisplayHandler::setSignpostTextColor()
{
   const navmiddleware::Color& textColor = _navMiddleware.getSignPostInfo().getTextColor();
   Candera::Color textLinesColor((static_cast<float>(textColor.m_red) / BACKGROUND_COLOR_SCALE_FACTOR), (static_cast<float>(textColor.m_green) / BACKGROUND_COLOR_SCALE_FACTOR), (static_cast<float>(textColor.m_blue) / BACKGROUND_COLOR_SCALE_FACTOR), (static_cast<float>(textColor.m_alpha) / BACKGROUND_COLOR_SCALE_FACTOR));

   if ((*_signpostTextData).mTextLinesColor != textLinesColor)
   {
      (*_signpostTextData).mTextLinesColor = textLinesColor;
      _signpostTextData.MarkItemModified(ItemKey::SignpostText::TextLinesColorItem);
   }
}


void SignpostDisplayHandler::sendSignpostTextData()
{
   if (_signpostTextData.HasModifiedItems())
   {
      ETG_TRACE_USR4(("SignpostDisplayHandler::sendSignpostTextData()"));
      if (!(_signpostTextData.SendUpdate()))
      {
         ETG_TRACE_ERR(("Sending update _signpostTextData failed!"));
      }
   }
}


void SignpostDisplayHandler::setRoadNumberIconDirection()
{
   ETG_TRACE_USR4(("SignpostDisplayHandler::setRoadNumberIconDirection()"));

   const std::vector<navmiddleware::RoadNumberInfo>& roadNumberInfo = _navMiddleware.getSignPostInfo().getSignpostRoadNumberInfos();
   if (!roadNumberInfo.empty())
   {
      (*_roadIconData).mRoadIconImage = loadImage((const char*)roadNumberInfo.begin()->m_iconData.getBlob(), roadNumberInfo.begin()->m_iconData.getBlobSize(), false);
      _roadIconData.MarkAllItemsModified();
      _roadIconData.SendUpdate();

      // Currently we show only one road number icon in signpost hence we need the sky direction for only first item in the road number info list
      Candera::String directionName = roadNumberInfo.begin()->m_cardinalDirection.c_str();
      if((*_signpostTextData).mSignpostDirectionName != directionName)
      {
         (*_signpostTextData).mSignpostDirectionName = directionName;
         _signpostTextData.MarkItemModified(ItemKey::SignpostText::SignpostDirectionNameItem);
      }
   }
}


void SignpostDisplayHandler::setExitNumber()
{
   (*_signpostTextData).mExitNumber = "";

   if (false == _navMiddleware.getSignPostInfo().getExitNumber().empty())
   {
      (*_signpostTextData).mExitNumberExists = true;
      (*_signpostTextData).mExitNumber = _navMiddleware.getSignPostInfo().getExitNumber().c_str();

      (*_signpostTextData).mExitNumberPositioning = _infoStore.isNarRegion() ? Candera::Margin(93, 24, 0, 0) : Candera::Margin(9, 8, 0, 0);
      _signpostTextData.MarkItemModified(ItemKey::SignpostText::ExitNumberPositioningItem);

      (*_signpostTextData).mSignpostRegionType = _infoStore.isNarRegion() ? 0 : 1;
      _signpostTextData.MarkItemModified(ItemKey::SignpostText::SignpostRegionTypeItem);
   }
   else
   {
      (*_signpostTextData).mExitNumberExists = false;
   }

   _signpostTextData.MarkItemModified(ItemKey::SignpostText::ExitNumberItem);
   _signpostTextData.MarkItemModified(ItemKey::SignpostText::ExitNumberExistsItem);

   ETG_TRACE_USR4(("SignpostDisplayHandler::setExitNumber(), Exit number : %s", (*_signpostTextData).mExitNumber.GetCString()));
}


void SignpostDisplayHandler::setSignpostPosition()
{
   navmiddleware::SignPostPosition position = _navMiddleware.getSignPostInfo().getSignPostPosition();
   Candera::HorizontalAlignment hAlign = Candera::HorizontalAlignment(Candera::HLeft);
   switch(position)
   {
   case navmiddleware::SignPostPosition_RIGHT:
      hAlign = Candera::HorizontalAlignment(Candera::HRight);
      break;
   case navmiddleware::SignPostPosition_MIDDLE:
      hAlign = Candera::HorizontalAlignment(Candera::HCenter);
      break;
   case navmiddleware::SignPostPosition_LEFT:
      hAlign = Candera::HorizontalAlignment(Candera::HLeft);
      break;
   default:
      break;
   }
   if((*_signpostTextData).mSignpostPosition != hAlign)
   {
      (*_signpostTextData).mSignpostPosition = hAlign;
      _signpostTextData.MarkItemModified(ItemKey::SignpostText::SignpostPositionItem);
   }
}


bool SignpostDisplayHandler::onPropertyUpdateMapCameraViewModeChanged()
{
   const enMapCameraAndViewModeMode& mapCameraViewMode = _infoStore.getMapCameraViewMode();
   ETG_TRACE_USR4(("SignpostDisplayHandler::onPropertyUpdateMapCameraViewModeChanged(), MapCameraViewMode : %d", mapCameraViewMode));

   if (MAP_CAMERA_AND_VIEW_MODE_MAP_JUNCTION_AND_INTERSECTION_SCREEN != mapCameraViewMode)
   {
      setSignpostActive(false);
   }
   return true;
}


#endif // HALL_TO_MDW_COM
