/**************************************************************************************
* @file         : LaneGuidanceHandler.cpp
* @author       : ECG5-Atchuta Sashank Kappagantu
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) 2018-2018 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include "LaneGuidanceHandler.h"
#include "CgiExtensions/ImageLoader.h"
#include "App/Core/VehicleDataClient/ClusterDataClientHandler.h"
//#include "Common/util/ExtensionUtils.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/LaneGuidanceHandler.cpp.trc.h"
#endif

#if defined (HALL_TO_MDW_COM)

// X position for lane guidance in PSD is 229 for full screen and 406 for split screen.
// Since Horizontal (list) composite has an offset of 105 for start of list item entries, the same value is subtracted from the PSD start position
static const float LANE_GUIDANCE_FULL_SCREEN_POS_X = 124.0f;
static const float LANE_GUIDANCE_SPLIT_SCREEN_POS_X = 301.0f;
static const float LANE_GUIDANCE_POS_Y = 70.0f;
static const float LANE_GUIDANCE_IMAGE_WIDTH = 42.0f;
static const int MAXIMUM_TOTAL_LANE = 8;

static const char* const DATA_CONTEXT_LIST_LANEGUIDANCE_ITEM = "LanesListItem";

static ::std::string LaneBackgroundInfo[4] =
{
   "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_bg_ngl",
   "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_bg_ngl",
   "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_bg_ngl",
   "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_bg_gl"
};


static ::std::string StartBoundaryBackgroundInfo[4] =
{
   "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_bg_ngl_start",
   "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_bg_ngl_start",
   "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_bg_ngl_start",
   "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_bg_gl_start"
};


static ::std::string EndBoundaryBackgroundInfo[4] =
{
   "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_bg_ngl_end",
   "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_bg_ngl_end",
   "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_bg_ngl_end",
   "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_bg_gl_end"
};


//check for single lane
static const ::std::string SingleLaneDirectionInfo[10][4] =
{
   //lane status unknown,forbidden,not Recommended,Recommended
   { "", "", "", "" }, // LaneDirectionNONE
   { "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_straight_ngl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_straight_gl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_straight_ngl_hov_s", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_straight_gl_hov_s" }, //  LaneDirectionSTRAIGHT
   { "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_left_ngl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_left_gl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_left_ngl_hov_s", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_left_gl_hov_s" }, //  LaneDirectionLeft
   { "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_right_ngl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_right_gl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_right_ngl_hov_s", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_right_gl_hov_s" },// LaneDirectionRight
   { "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_halfleft_ngl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_halfleft_gl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_halfleft_ngl_hov_s", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_halfleft_gl_hov_s" }, // LaneDirectionHalfLeft
   { "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_halfright_ngl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_halfright_gl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_halfright_ngl_hov_s", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_halfright_gl_hov_s" }, // LaneDirectionHalfRight
   { "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_muchleft_ngl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_muchleft_gl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_muchleft_ngl_hov_s", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_muchleft_gl_hov_s" }, // LaneDirectionMuchLeft
   { "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_muchright_ngl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_muchright_gl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_muchright_ngl_hov_s", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_muchright_gl_hov_s" }, // LaneDirectionMuchRight
   { "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_uturnleft_ngl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_uturnleft_gl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_uturnleft_ngl_hov_s", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_uturnleft_gl_hov_s" }, // LaneDirectionUturnLeft
   { "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_uturnright_ngl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_uturnright_gl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_uturnright_ngl_hov_s", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_uturnright_gl_hov_s" } // LaneDirectionUturnRight
};


static ::std::string CombineLaneDirectionCenterInfo[10][4] =
{
   //lane status unknown,forbidden,not Recommended,Recommended
   { "", "", "", "" }, // LaneDirectionNONE
   { "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_straight_ngl_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_straight_gl_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_straight_ngl_hov_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_straight_gl_hov_c" }, //  LaneDirectionSTRAIGHT
   { "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_left_ngl_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_left_gl_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_left_ngl_hov_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_left_gl_hov_c" }, //  LaneDirectionLeft
   { "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_right_gl_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_right_gl_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_right_ngl_hov_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_right_gl_hov_c" },// LaneDirectionRight
   { "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_halfleft_ngl_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_halfleft_gl_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_halfleft_ngl_hov_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_halfleft_gl_hov_c" }, // LaneDirectionHalfLeft
   { "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_halfright_ngl_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_halfright_gl_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_halfright_ngl_hov_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_halfright_gl_hov_c" }, // LaneDirectionHalfRight
   { "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_muchleft_ngl_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_muchleft_gl_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_muchleft_ngl_hov_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_muchleft_gl_hov_c" }, // LaneDirectionMuchLeft
   { "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_muchright_ngl_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_muchright_gl_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_muchright_ngl_hov_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_muchright_gl_hov_c" }, // LaneDirectionMuchRight
   { "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_uturnleft_ngl_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_uturnleft_gl_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_uturnleft_ngl_hov_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_uturnleft_gl_hov_c" }, // LaneDirectionUturnLeft
   { "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_uturnright_ngl_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_uturnright_gl_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_uturnright_ngl_hov_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_uturnright_gl_hov_c" } // LaneDirectionUturnRight
};


static const ::std::string CombineLaneDirectionRightInfo[10][4] =
{
   //lane status unknown,forbidden,not Recommended,Recommended
   { "", "", "", "" }, // LaneDirectionNONE
   { "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_straight_ngl_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_straight_gl_c", "", "" }, //  LaneDirectionSTRAIGHT
   { "", "", "", "" }, //  LaneDirectionLeft
   { "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_right_ngl_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_right_gl_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_right_ngl_hov_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_right_gl_hov_c" },// LaneDirectionRight
   { "", "", "", "" }, // LaneDirectionHalfLeft
   { "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_halfright_ngl_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_halfright_gl_c", "", "" }, // LaneDirectionHalfRight
   { "", "", "", "" }, // LaneDirectionMuchLeft
   { "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_muchright_ngl_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_muchright_gl_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_muchright_ngl_hov_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_muchright_gl_hov_c" }, // LaneDirectionMuchRight
   { "", "", "", "" }, // LaneDirectionUturnLeft
   { "", "", "", "" } // LaneDirectionUturnRight
};


static const ::std::string CombineLaneDirectionLeftInfo[10][4] =
{
   //lane status unknown,forbidden,not Recommended,Recommended
   { "", "", "", "" }, // LaneDirectionNONE
   { "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_straight_ngl_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_straight_gl_c", "", "" }, //  LaneDirectionSTRAIGHT
   { "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_left_ngl_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_left_gl_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_left_ngl_hov_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_left_gl_hov_c" }, //  LaneDirectionLeft
   { "", "", "", "" },// LaneDirectionRight
   { "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_halfleft_ngl_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_halfleft_gl_c", "", "" }, // LaneDirectionHalfLeft
   { "", "", "", "" }, // LaneDirectionHalfRight
   { "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_muchleft_ngl_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_muchleft_gl_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_muchleft_ngl_hov_c", "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_map_muchleft_gl_hov_c" }, // LaneDirectionMuchLeft
   { "", "", "", "" }, // LaneDirectionMuchRight
   { "", "", "", "" }, // LaneDirectionUturnLeft
   { "", "", "", "" } // LaneDirectionUturnRight
};


// Add for Route
static ::std::string LaneBackgroundInfoRoute[4] =
{
   "AppHmi_NavigationModule#Images#LaneGuidance_Map#LG_Route_bg_ngl",
   "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_bg_ngl",
   "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_bg_ngl",
   "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_bg_gl"
};


static ::std::string StartBoundaryBackgroundInfoRoute[4] =
{
   "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_bg_ngl_start",
   "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_bg_ngl_start",
   "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_bg_ngl_start",
   "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_bg_gl_start"
};


static ::std::string EndBoundaryBackgroundInfoRoute[4] =
{
   "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_bg_ngl_end",
   "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_bg_ngl_end",
   "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_bg_ngl_end",
   "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_bg_gl_end"
};


//check for single lane
static const ::std::string SingleLaneDirectionInfoRoute[10][4] =
{
   //lane status unknown,forbidden,not Recommended,Recommended
   { "", "", "", "" }, // LaneDirectionNONE
   { "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_straight_ngl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_straight_gl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_straight_ngl_hov_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_straight_gl_hov_s" }, //  LaneDirectionSTRAIGHT
   { "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_left_ngl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_left_gl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_left_ngl_hov_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_left_gl_hov_s" }, //  LaneDirectionLeft
   { "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_right_ngl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_right_gl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_right_ngl_hov_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_right_gl_hov_s" },// LaneDirectionRight
   { "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_halfleft_ngl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_halfleft_gl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_halfleft_ngl_hov_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_halfleft_gl_hov_s" }, // LaneDirectionHalfLeft
   { "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_halfright_ngl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_halfright_gl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_halfright_ngl_hov_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_halfright_gl_hov_s" }, // LaneDirectionHalfRight
   { "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_muchleft_ngl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_muchleft_gl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_muchleft_ngl_hov_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_muchleft_gl_hov_s" }, // LaneDirectionMuchLeft
   { "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_muchright_ngl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_muchright_gl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_muchright_ngl_hov_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_muchright_gl_hov_s" }, // LaneDirectionMuchRight
   { "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_uturnleft_ngl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_uturnleft_gl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_uturnleft_ngl_hov_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_uturnleft_gl_hov_s" }, // LaneDirectionUturnLeft
   { "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_uturnright_ngl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_uturnright_gl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_uturnright_ngl_hov_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_uturnright_gl_hov_s" } // LaneDirectionUturnRight
};


static ::std::string CombineLaneDirectionCenterInfoRoute[10][4] =
{
   //lane status unknown,forbidden,not Recommended,Recommended
   { "", "", "", "" }, // LaneDirectionNONE
   { "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_straight_ngl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_straight_gl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_straight_ngl_hov_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_straight_gl_hov_s" }, //  LaneDirectionSTRAIGHT
   { "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_left_ngl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_left_gl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_left_ngl_hov_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_left_gl_hov_s" }, //  LaneDirectionLeft
   { "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_right_gl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_right_gl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_right_ngl_hov_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_right_gl_hov_s" },// LaneDirectionRight
   { "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_halfleft_ngl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_halfleft_gl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_halfleft_ngl_hov_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_halfleft_gl_hov_s" }, // LaneDirectionHalfLeft
   { "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_halfright_ngl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_halfright_gl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_halfright_ngl_hov_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_halfright_gl_hov_s" }, // LaneDirectionHalfRight
   { "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_muchleft_ngl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_muchleft_gl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_muchleft_ngl_hov_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_muchleft_gl_hov_s" }, // LaneDirectionMuchLeft
   { "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_muchright_ngl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_muchright_gl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_muchright_ngl_hov_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_muchright_gl_hov_s" }, // LaneDirectionMuchRight
   { "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_uturnleft_ngl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_uturnleft_gl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_uturnleft_ngl_hov_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_uturnleft_gl_hov_s" }, // LaneDirectionUturnLeft
   { "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_uturnright_ngl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_uturnright_gl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_uturnright_ngl_hov_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_uturnright_gl_hov_s" } // LaneDirectionUturnRight
};


static const ::std::string CombineLaneDirectionRightInfoRoute[10][4] =
{
   //lane status unknown,forbidden,not Recommended,Recommended
   { "", "", "", "" }, // LaneDirectionNONE
   { "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_straight_ngl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_straight_gl_s", "", "" }, //  LaneDirectionSTRAIGHT
   { "", "", "", "" }, //  LaneDirectionLeft
   { "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_right_ngl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_right_gl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_right_ngl_hov_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_right_gl_hov_s" },// LaneDirectionRight
   { "", "", "", "" }, // LaneDirectionHalfLeft
   { "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_halfright_ngl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_halfright_gl_s", "", "" }, // LaneDirectionHalfRight
   { "", "", "", "" }, // LaneDirectionMuchLeft
   { "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_muchright_ngl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_muchright_gl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_muchright_ngl_hov_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_muchright_gl_hov_s" }, // LaneDirectionMuchRight
   { "", "", "", "" }, // LaneDirectionUturnLeft
   { "", "", "", "" } // LaneDirectionUturnRight
};


static const ::std::string CombineLaneDirectionLeftInfoRoute[10][4] =
{
   //lane status unknown,forbidden,not Recommended,Recommended
   { "", "", "", "" }, // LaneDirectionNONE
   { "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_straight_ngl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_straight_gl_s", "", "" }, //  LaneDirectionSTRAIGHT
   { "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_left_ngl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_left_gl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_left_ngl_hov_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_left_gl_hov_s" }, //  LaneDirectionLeft
   { "", "", "", "" },// LaneDirectionRight
   { "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_halfleft_ngl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_halfleft_gl_s", "", "" }, // LaneDirectionHalfLeft
   { "", "", "", "" }, // LaneDirectionHalfRight
   { "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_muchleft_ngl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_muchleft_gl_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_muchleft_ngl_hov_s", "AppHmi_NavigationModule#Images#LaneGuidance_Route#LG_Route_muchleft_gl_hov_s" }, // LaneDirectionMuchLeft
   { "", "", "", "" }, // LaneDirectionMuchRight
   { "", "", "", "" }, // LaneDirectionUturnLeft
   { "", "", "", "" } // LaneDirectionUturnRight
};


LaneGuidanceHandler::LaneGuidanceHandler(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore)
   : HMIModelBase(navMiddleware, infoStore), _laneListIsEmpty(true), _isFromSplitRoute(false)
{
}


LaneGuidanceHandler::~LaneGuidanceHandler()
{
}


void LaneGuidanceHandler::initialize()
{
   _navMiddleware.registerGuidancePropertyUpdateCallback(*this);
   _infoStore.registerDataPropertyUpdateCallback(*this);
}


void LaneGuidanceHandler::deinitialize()
{
   _navMiddleware.unregisterGuidancePropertyUpdateCallback(*this);
   _infoStore.unregisterDataPropertyUpdateCallback(*this);
}


bool LaneGuidanceHandler::onPropertyUpdateLaneDirectionInfoChanged()
{
   bool isGuidanceActive = _navMiddleware.isGuidanceActive();
   bool isHardRestrictionPresent = _infoStore.getIsHardRestrictionPresent();
   ETG_TRACE_USR4(("LaneGuidanceHandler::onPropertyUpdateLaneDirectionInfoChanged(), isGuidanceActive : %d, isHardRestrictionPresent : %d",
                  isGuidanceActive, isHardRestrictionPresent));

   bool prevlaneisEmpty = _laneListIsEmpty;
   tSharedPtrDataProvider dataProvider = getLaneGuidanceListDataProvider();
   if ((!_laneListIsEmpty) || (prevlaneisEmpty != _laneListIsEmpty))
   {
      POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
   }

   if ((true == isGuidanceActive) && (false == isHardRestrictionPresent))
   {
      ::App::Core::ClusterDataClientHandler::getInstance()->updateLaneGuidanceInfo(_navMiddleware.getLaneInfos());
   }
   else
   {
      ::navmiddleware::LaneInfos laneInfos;
      ::App::Core::ClusterDataClientHandler::getInstance()->updateLaneGuidanceInfo(laneInfos);
   }
   return true;
}


int LaneGuidanceHandler::getLaneArrowAlignmentIndex(std::vector <int>& directionIndex, int directions)
{
   // Updates the guided and combined following lanes directions
   int laneAlignment = 0;
   if (navmiddleware::LaneDirection__STRAIGHT & directions)
   {
      directionIndex.push_back(1);
      laneAlignment |= LANE_ALIGN_STRAIGHT;
   }
   if (navmiddleware::LaneDirection__LEFT & directions)
   {
      directionIndex.push_back(2);
      laneAlignment |= LANE_ALIGN_LEFT;
   }
   if (navmiddleware::LaneDirection__RIGHT & directions)
   {
      directionIndex.push_back(3);
      laneAlignment |= LANE_ALIGN_RIGHT;
   }
   if (navmiddleware::LaneDirection__HALF_LEFT & directions)
   {
      directionIndex.push_back(4);
      laneAlignment |= LANE_ALIGN_LEFT;
   }
   if (navmiddleware::LaneDirection__HALF_RIGHT & directions)
   {
      directionIndex.push_back(5);
      laneAlignment |= LANE_ALIGN_RIGHT;
   }
   if (navmiddleware::LaneDirection__SHARP_LEFT & directions)
   {
      directionIndex.push_back(6);
      laneAlignment |= LANE_ALIGN_COMBINED;
   }
   if (navmiddleware::LaneDirection__SHARP_RIGHT & directions)
   {
      directionIndex.push_back(7);
      laneAlignment |= LANE_ALIGN_COMBINED;
   }
   if (navmiddleware::LaneDirection__UTURN_LEFT & directions)
   {
      directionIndex.push_back(8);
      laneAlignment |= LANE_ALIGN_COMBINED;
   }
   if (navmiddleware::LaneDirection__UTURN_RIGHT & directions)
   {
      directionIndex.push_back(9);
      laneAlignment |= LANE_ALIGN_COMBINED;
   }
   return laneAlignment;
}


LaneGuidanceData LaneGuidanceHandler::getLaneGUIElement(const navmiddleware::LaneInfos::LaneInfo& laneInfo)
{
   int combinedDirections = laneInfo.combinedFollowingLaneDirection;
   int guidedLaneDirection = laneInfo.guidedFollowingLaneDirection;
   navmiddleware::LaneStatus currentLaneStatus = laneInfo.laneStatus;

   ETG_TRACE_USR4(("LaneGuidanceListHandler::getLaneGUIElement(%s)", LaneBackgroundInfo[currentLaneStatus].c_str()));
   ETG_TRACE_USR4(("LaneGuidanceListHandler::getLaneGUIElement _isFromSplitRoute %s", (_isFromSplitRoute) ? "Yes" : "No"));

   std::vector <int> directionIndex;
   int laneAlignment = getLaneArrowAlignmentIndex(directionIndex, combinedDirections);
   const ::std::string* pBitmapNameArray = &SingleLaneDirectionInfo[0][0];
   if (_isFromSplitRoute)
   {
      pBitmapNameArray = &SingleLaneDirectionInfoRoute[0][0];
   }
   if (directionIndex.size() > 1)
   {
      ETG_TRACE_USR4(("LaneGuidanceListHandler::getLaneGUIElement directionIndex %d", directionIndex.size()));
      if (_isFromSplitRoute)
      {
         pBitmapNameArray = &SingleLaneDirectionInfoRoute[0][0];
         if (laneAlignment == (LANE_ALIGN_STRAIGHT | LANE_ALIGN_LEFT))
         {
            pBitmapNameArray = &CombineLaneDirectionLeftInfoRoute[0][0];
         }
         else if (laneAlignment == (LANE_ALIGN_STRAIGHT | LANE_ALIGN_RIGHT))
         {
            pBitmapNameArray = &CombineLaneDirectionRightInfoRoute[0][0];
         }
         else
         {
            pBitmapNameArray = &CombineLaneDirectionCenterInfoRoute[0][0];
         }
      }
      else
      {
         if (laneAlignment == (LANE_ALIGN_STRAIGHT | LANE_ALIGN_LEFT))
         {
            pBitmapNameArray = &CombineLaneDirectionLeftInfo[0][0];
         }
         else if (laneAlignment == (LANE_ALIGN_STRAIGHT | LANE_ALIGN_RIGHT))
         {
            pBitmapNameArray = &CombineLaneDirectionRightInfo[0][0];
         }
         else
         {
            pBitmapNameArray = &CombineLaneDirectionCenterInfo[0][0];
         }
      }
   }

   LaneGuidanceData item = getLaneGuidanceData(laneInfo, directionIndex, pBitmapNameArray);

   return item;
}


LaneGuidanceData LaneGuidanceHandler::getLaneGuidanceData(const navmiddleware::LaneInfos::LaneInfo& laneInfo, std::vector <int>& directionIndex, const ::std::string* pBitmapNameArray)
{
   LaneGuidanceData item;
   if (NULL != pBitmapNameArray)
   {
      int guidedLaneDirection = laneInfo.guidedFollowingLaneDirection;
      navmiddleware::LaneStatus currentLaneStatus = laneInfo.laneStatus;
      std::vector< Candera::MemoryManagement::SharedPointer<Candera::Image2D> > imageVector;
      // update lane back ground
      imageVector.push_back(ImageLoader::getAssetBitmapImage(LaneBackgroundInfo[currentLaneStatus].c_str()));

      for (std::vector <int>::iterator itr = directionIndex.begin(); itr != directionIndex.end(); itr++)
      {
         if (!(laneInfo.isHOVLane)) //without HOV
         {
            imageVector.push_back(ImageLoader::getAssetBitmapImage((*(pBitmapNameArray + ((*itr) * 4))).c_str()));
            item.mHovOccupancy = "";
         }
         else //with HOV
         {
            imageVector.push_back(ImageLoader::getAssetBitmapImage((*(pBitmapNameArray + ((*itr) * 4) + 2)).c_str()));
            std::stringstream stringStreamHovoccupancy;
            stringStreamHovoccupancy << laneInfo.laneOccupancy;
            item.mHovOccupancy = stringStreamHovoccupancy.str().c_str();
         }
      }
      directionIndex.clear();

      if (guidedLaneDirection > 0)
      {
         int laneAlignmentGuidedArrow = 0;
         laneAlignmentGuidedArrow = getLaneArrowAlignmentIndex(directionIndex, guidedLaneDirection);
         for (std::vector <int>::iterator itr = directionIndex.begin(); itr != directionIndex.end(); itr++)
         {
            if (!(laneInfo.isHOVLane)) //without HOV
            {
               imageVector.push_back(ImageLoader::getAssetBitmapImage((*(pBitmapNameArray + ((*itr) * 4) + 1)).c_str())); //+1 gets the guided lane direction
               item.mHovOccupancy = "";
            }
            else //with HOV
            {
               imageVector.push_back(ImageLoader::getAssetBitmapImage((*(pBitmapNameArray + ((*itr) * 4) + 3)).c_str()));
               std::stringstream stringStreamHovoccupancy;
               stringStreamHovoccupancy << laneInfo.laneOccupancy;
               item.mHovOccupancy = stringStreamHovoccupancy.str().c_str();
            }
         }
      }
      updateItemBitmapImg(item, imageVector);
   }

   return item;
}


void LaneGuidanceHandler::updateItemBitmapImg(LaneGuidanceData& item, std::vector< Candera::MemoryManagement::SharedPointer<Candera::Image2D> >& imageVector)
{
   switch (imageVector.size())
   {
      case 8:
         item.mBitmapImage8 = imageVector.at(7);
      case 7:
         item.mBitmapImage7 = imageVector.at(6);
      case 6:
         item.mBitmapImage6 = imageVector.at(5);
      case 5:
         item.mBitmapImage5 = imageVector.at(4);
      case 4:
         item.mBitmapImage4 = imageVector.at(3);
      case 3:
         item.mBitmapImage3 = imageVector.at(2);
      case 2:
         item.mBitmapImage2 = imageVector.at(1);
      case 1:
         item.mBitmapImage1 = imageVector.at(0);
      default:
         break;
   }
}


bool LaneGuidanceHandler::onCourierMessage(const ListDateProviderReqMsg& oMsg)
{
   if (oMsg.GetListId() == LIST_ID_LANE_DIRECTION_LIST)
   {
      ETG_TRACE_USR4(("LaneGuidanceListHandler::onCourierMessage(ListDateProviderReqMsg)"));
      tSharedPtrDataProvider dataProvider = getLaneGuidanceListDataProvider();
      POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
      return true;
   }
   return false;
}


tSharedPtrDataProvider LaneGuidanceHandler::getLaneGuidanceListDataProvider()
{
   ETG_TRACE_USR4(("LaneGuidanceHandler::getLaneGuidanceListDataProvider()"));

   ListDataProviderBuilder listBuilder(LIST_ID_LANE_DIRECTION_LIST, DATA_CONTEXT_LIST_LANEGUIDANCE_ITEM);
   _laneListIsEmpty = true;

   if (_navMiddleware.isGuidanceActive())
   {
      const std::vector<navmiddleware::LaneInfos::LaneInfo>& LaneList = _navMiddleware.getLaneInfos().getLaneInfos();
      if (!LaneList.empty())
      {
         setLaneGuidancePositionOnMap(LaneList.size());

         _laneListIsEmpty = false;
         unsigned int idx = 0;
         LaneGuidanceData item;

         // update start boundary background
         item.mBitmapImage1 = ImageLoader::getAssetBitmapImage(StartBoundaryBackgroundInfo[LaneList.front().laneStatus].c_str());
         listBuilder.AddItem(idx++).AddDataBindingUpdater<LaneGuidanceDataBindingSource>(item);

         for (std::vector<navmiddleware::LaneInfos::LaneInfo>::const_iterator itr = LaneList.begin(); itr != LaneList.end(); itr++)
         {
            // update lane info
            listBuilder.AddItem(idx++).AddDataBindingUpdater<LaneGuidanceDataBindingSource>(getLaneGUIElement(*itr));
         }

         // update end boundary background
         item.mBitmapImage1 = ImageLoader::getAssetBitmapImage(EndBoundaryBackgroundInfo[LaneList.back().laneStatus].c_str());
         listBuilder.AddItem(idx++).AddDataBindingUpdater<LaneGuidanceDataBindingSource>(item);
      }
   }
   return listBuilder.CreateDataProvider();
}


void LaneGuidanceHandler::setLaneGuidancePositionOnMap(int numLaneEntries)
{
   float xPosition = _infoStore.isSplitScreenEnabled() ? LANE_GUIDANCE_SPLIT_SCREEN_POS_X : LANE_GUIDANCE_FULL_SCREEN_POS_X;
   if (MAXIMUM_TOTAL_LANE > numLaneEntries)
   {
      xPosition += ((MAXIMUM_TOTAL_LANE - numLaneEntries) / 2) * LANE_GUIDANCE_IMAGE_WIDTH;
      if ((numLaneEntries % 2) != 0)
      {
         xPosition += LANE_GUIDANCE_IMAGE_WIDTH / 2;
      }
   }
   (*_splitMapScreenData).mLaneGuidanceSplitViewPosition = Candera::Vector2(xPosition, LANE_GUIDANCE_POS_Y);
   (*_splitMapScreenData).mEnableSplitMapItems = _infoStore.isSplitScreenEnabled();
   _splitMapScreenData.MarkAllItemsModified();
   _splitMapScreenData.SendUpdate();
}


bool LaneGuidanceHandler::onPropertyUpdateSplitMapStatusChanged()
{
   bool isSplitScreenEnabled = _infoStore.isSplitScreenEnabled();
   ETG_TRACE_USR4(("LaneGuidanceListHandler::onPropertyUpdateSplitMapStatusChanged(), isSplitScreenEnabled : %d", isSplitScreenEnabled));

   Candera::Vector2 laneGuidancePosition = (*_splitMapScreenData).mLaneGuidanceSplitViewPosition;
   if(true == isSplitScreenEnabled)
   {
      laneGuidancePosition.SetX(laneGuidancePosition.GetX() + (LANE_GUIDANCE_SPLIT_SCREEN_POS_X - LANE_GUIDANCE_FULL_SCREEN_POS_X));
   }
   else
   {
      laneGuidancePosition.SetX(laneGuidancePosition.GetX() - (LANE_GUIDANCE_SPLIT_SCREEN_POS_X - LANE_GUIDANCE_FULL_SCREEN_POS_X));
   }
   (*_splitMapScreenData).mLaneGuidanceSplitViewPosition = laneGuidancePosition;
   (*_splitMapScreenData).mEnableSplitMapItems = isSplitScreenEnabled;
   _splitMapScreenData.MarkAllItemsModified();
   _splitMapScreenData.SendUpdate();
   return true;
}


bool LaneGuidanceHandler::onPropertyUpdateHardRestrictionStatusChanged()
{
   bool isHardRestrictionPresent = _infoStore.getIsHardRestrictionPresent();
   ETG_TRACE_USR4(("LaneGuidanceHandler::onPropertyUpdateHardRestrictionStatusChanged(), isHardRestrictionPresent : %d", isHardRestrictionPresent));

   if(true == isHardRestrictionPresent)
   {
      ::navmiddleware::LaneInfos laneInfos;
      ::App::Core::ClusterDataClientHandler::getInstance()->updateLaneGuidanceInfo(laneInfos);
   }
   else
   {
      onPropertyUpdateLaneDirectionInfoChanged();
   }
   return true;
}

#endif // HALL_TO_MDW_COM
