/**************************************************************************************
* @file         : PreviousDestinationListHandler.cpp
* @author       : ECG5-Prieethi Narayanaswamy
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) -2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include "PreviousDestinationListHandler.h"
#include "hmi_trace_if.h"
#include "../DataModel/AddressDetailedInfo/AddressDetailedInfo.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/PreviousDestinationListHandler.cpp.trc.h"
#endif

static const char* const DATA_CONTEXT_LIST_PREV_DESTINATION_ITEM   = "PrevDestListItem";
static const char* const DATA_CONTEXT_EMPTY_LIST_ITEM              = "NoDataListItem";
static const Courier::Identifier  IdDestinationInfoBtn = Courier::Identifier("Layer_DestinationInfo/ButtonWidget");
static const Candera::String NO_ENTRIES = LANGUAGE_STRING(TextId_0x0BC5, "No list entries");
using namespace navmiddleware;

PreviousDestinationListHandler::PreviousDestinationListHandler(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore)
   : HMIModelBase(navMiddleware, infoStore)
   , _listId(0)
   , _startIndex(0)
   , _windowElementSize(0)
   , _lastDestinationElementId(0)
   , _isDestinationInfoBtnPressed(false)
{
}


PreviousDestinationListHandler::~PreviousDestinationListHandler()
{
}


void PreviousDestinationListHandler::initialize()
{
   _navMiddleware.registerDestinationMemoryPropertyUpdateCallback(*this);
   _navMiddleware.registerGuidancePropertyUpdateCallback(*this);
   _navMiddleware.registerRoutePropertyUpdateCallback(*this);
   _navMiddleware.registerLocationPropertyUpdateCallback(*this);
   _infoStore.registerDataPropertyUpdateCallback(*this);
}


void PreviousDestinationListHandler::deinitialize()
{
   _navMiddleware.unregisterDestinationMemoryPropertyUpdateCallback(*this);
   _navMiddleware.unregisterGuidancePropertyUpdateCallback(*this);
   _navMiddleware.unregisterRoutePropertyUpdateCallback(*this);
   _navMiddleware.unregisterLocationPropertyUpdateCallback(*this);
   _infoStore.unregisterDataPropertyUpdateCallback(*this);
}


bool PreviousDestinationListHandler::onCourierMessage(const ListDateProviderReqMsg& oMsg)
{
   bool isMessageConsumed = false;
   if (oMsg.GetListId() == LIST_ID_NAV_PREV_DEST_MEMORY_LIST)
   {
      // we are handling this request
      unsigned int listId = oMsg.GetListId();
      unsigned int startIndex = oMsg.GetStartIndex();
      unsigned int windowElementSize = oMsg.GetWindowElementSize();
      ETG_TRACE_USR4(("PreviousDestinationListHandler::onCourierMessage(ListDateProviderReqMsg(listID %d, startIndex %d, elementSize %d))", listId, startIndex, windowElementSize));
      updatePreviousDestinationsList(listId, startIndex, windowElementSize);
      isMessageConsumed = true;
   }

   return isMessageConsumed;
}


bool PreviousDestinationListHandler::onCourierMessage(const ButtonReactionMsg& oMsg)
{
   bool isMessageProcessed = false;
   const Courier::Identifier senderInfo = oMsg.GetSender();

   ListProviderEventInfo info;
   if ((ListProviderEventInfo::GetItemIdentifierInfo(oMsg.GetSender(), info)))
   {
      if (info.getListId() == LIST_ID_NAV_PREV_DEST_MEMORY_LIST)
      {
         if (oMsg.GetEnReaction() == enRelease)
         {
            // we are handling this request
            ETG_TRACE_USR4(("PreviousDestinationListHandler::onCourierMessage(ButtonReactionMsg(listID %d, row %d, column %d))", info.getListId(), info.getHdlRow(), info.getHdlCol()));

            unsigned int listIdx = (unsigned int)info.getHdlRow();
            const navmiddleware::DestinationMemoryItemList& destinationMemoryItemList = _navMiddleware.getLastDestinationList();
            const ::std::vector<navmiddleware::DestinationMemoryItem>& destinationInfoVector = destinationMemoryItemList.getItemList();

            // use a stream string helper because of missing target support for printing out two separate strings within the ETG trace
            std::stringstream traceOut;

            listIdx = listIdx - _startIndex;

            if (listIdx < destinationInfoVector.size())
            {
               destinationMemoryItem = destinationInfoVector[listIdx];
               _infoStore.setDestinationMemoryStatus(InfoStoreBase::DestinationMemoryStatus(_infoStore.getDestinationMemoryStatus()._category, destinationMemoryItem.getEntry().getId(),
                                                     destinationMemoryItem.getEntry().getDestination().getName()));

               traceOut << destinationMemoryItem.getEntry().getDestination().getName() << " with longitude " << destinationMemoryItem.getEntry().getDestination().getLongitude()
                        << " and latitude " << destinationMemoryItem.getEntry().getDestination().getLatitude();
               ETG_TRACE_USR1(("Start guidance to %s", traceOut.str().c_str()));

               _navMiddleware.setLocationWithDestinationMemoryId(destinationMemoryItem.getEntry().getId());

               std::vector<PosWGS84<double> > positionVector;
               positionVector.push_back(PosWGS84<double>(destinationMemoryItem.getEntry().getDestination().getLongitude(), destinationMemoryItem.getEntry().getDestination().getLatitude()));
               InfoStoreBase::CoordinatesToBeShownInMap& coordinatesToBeShownInMap = _infoStore.getCoordinatesToBeShownInMap();
               coordinatesToBeShownInMap._coordinates = positionVector;

               _infoStore.setLatitude(destinationMemoryItem.getEntry().getDestination().getLatitude());
               _infoStore.setLongitude(destinationMemoryItem.getEntry().getDestination().getLongitude());
               // clearing each id and then spliting the destination details
               _lastDestinationElementId.clear();
               _lastDestinationElementId.push_back(destinationMemoryItem.getIdElement());

               const LocationAttributeInfos& infos = destinationMemoryItem.getEntry().getDestination().getAttributes();

               AddressDetailedInfo* poAddressDetailedInfo = AddressDetailedInfo::getInstance();
               if (NULL != poAddressDetailedInfo)
               {
                  poAddressDetailedInfo->FillAddressDetailedinfo(infos);
               }

               POST_MSG((COURIER_MESSAGE_NEW(ShowDetailedAddressMsg)()));
               POST_MSG((COURIER_MESSAGE_NEW(ShowFavEditIcon_Enable_Delete_Guidance_Buttons)(0, 1, 1, 0, 0)));

               POST_MSG((COURIER_MESSAGE_NEW(LastDestinationsBtnSelectionMsg)()));
            }
            isMessageProcessed = true;
         }
      }
   }
   if (oMsg.GetEnReaction() == enRelease)
   {
      if ((senderInfo == IdDestinationInfoBtn) && (true == EXT_bIsGuidanceActive))
      {
         ETG_TRACE_USR4(("PreviousDestinationListHandler::onCourierMessage(ButtonReactionMsg(IdDestinationInfoBtn))"));

         _isDestinationInfoBtnPressed = true;
         _navMiddleware.requestWaypointList(navmiddleware::WAYPOINT_LIST_MODE__VIEW);
      }
   }

   return isMessageProcessed;
}


void PreviousDestinationListHandler::onPropertyUpdateRouteWaypointInfoChanged()
{
   ETG_TRACE_USR4(("PreviousDestinationListHandler::onPropertyUpdateRouteWaypointInfoChanged()"));

   if (_isDestinationInfoBtnPressed)
   {
      _isDestinationInfoBtnPressed = false;
      const navmiddleware::WaypointInfos& wayPointInfos = _navMiddleware.getWaypointInfos();
      const std::vector< navmiddleware::WaypointInfos::WaypointInfo >& waypointInfoList = wayPointInfos.getWaypointInfos();
      navmiddleware::WaypointInfos::WaypointInfo destinationItem = waypointInfoList.back();
      std::vector<PosWGS84<double> > positionVector;
      positionVector.push_back(PosWGS84<double>(destinationItem.m_coordinate.getLongitude(), destinationItem.m_coordinate.getLatitude()));
      _infoStore.setLatitude(destinationItem.m_coordinate.getLatitude());
      _infoStore.setLongitude(destinationItem.m_coordinate.getLongitude());
      InfoStoreBase::CoordinatesToBeShownInMap& coordinatesToBeShownInMap = _infoStore.getCoordinatesToBeShownInMap();
      coordinatesToBeShownInMap._coordinates = positionVector;
      _navMiddleware.setLocationWithWaypointIndex(waypointInfoList.size() - 1);
      _infoStore.setIsDetailInfoRequested(true, InfoStoreBase::DETAIL_INFO_DESTINATION);
      _navMiddleware.requestLocationAttributes();
   }
}


InfoStoreBase::RouteToBeUsedForE2EGuidance PreviousDestinationListHandler::getRouteToBeUsedForE2EGuidance() const
{
   return _infoStore.getRouteToBeUsedForE2EGuidance();
}


bool PreviousDestinationListHandler::onCourierMessage(const CancelRouteGuidanceReqMsg& oMsg)
{
   //_extLocationVector.clear();

   return false;
}


void PreviousDestinationListHandler::onPropertyUpdateDestinationMemoryLastDestinationListStatusChanged()
{
   ETG_TRACE_USR4(("PreviousDestinationListHandler::onPropertyUpdateDestinationMemoryLastDestinationListStatusChanged"));
   const navmiddleware::DestinationMemoryStatus& destinationMemoryStatus = _navMiddleware.getDestinationMemoryStatus();
   updatePreviousDestCount((uint8_t)destinationMemoryStatus.getNumberOfPreviousDestionationItems(navmiddleware::DestinationMemoryItemList::FILTER_ENTRY_AND_ROUTE_WAYPOINT));
   _navMiddleware.requestLastDeparture();
}


DataBindingItem<PreviousDestCountDataBindingSource>& PreviousDestinationListHandler::getPreviousDestCountInfo()
{
   static DataBindingItem<PreviousDestCountDataBindingSource> sPreviousDestCountInfo;
   return sPreviousDestCountInfo;
}


void PreviousDestinationListHandler::updatePreviousDestCount(const uint8_t count)
{
   if ((*getPreviousDestCountInfo()).mTotalPreviousDestCount != count)
   {
      (*getPreviousDestCountInfo()).mTotalPreviousDestCount = count;
      getPreviousDestCountInfo().MarkItemModified(ItemKey::PreviousDestCount::TotalPreviousDestCountItem);
      if (getPreviousDestCountInfo().SendUpdate() == false)
      {
         ETG_TRACE_ERR(("_previousdestCount update failed!"));
      }
   }
}


// property update of requesting previous destination list requestLastDestinationList()
void PreviousDestinationListHandler::onPropertyUpdateDestinationMemoryLastDestinationListChanged()
{
   // just post a ListDateProviderResMsg with our destination list data provider
   tSharedPtrDataProvider dataProvider;

   ETG_TRACE_USR4(("PreviousDestinationListHandler::onPropertyUpdateDestinationMemoryLastDestinationListChanged: _listId= %d, startIndex = %d, _windowElementSize = %d", \
                   _listId, _startIndex, _windowElementSize));
   const navmiddleware::DestinationMemoryStatus& destinationMemoryStatus = _navMiddleware.getDestinationMemoryStatus();
   updatePreviousDestCount((uint8_t)destinationMemoryStatus.getNumberOfPreviousDestionationItems(navmiddleware::DestinationMemoryItemList::FILTER_ENTRY_AND_ROUTE_WAYPOINT));
   if (_listId == LIST_ID_NAV_PREV_DEST_MEMORY_LIST)
   {
      dataProvider = getPreviousDestinationsListDataProvider(_listId, DATA_CONTEXT_LIST_PREV_DESTINATION_ITEM, _startIndex, _windowElementSize);
      POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
   }
}


void PreviousDestinationListHandler::updatePreviousDestinationsList(unsigned int listId, unsigned int startIndex, unsigned int windowElementSize)
{
   _listId = listId;
   _startIndex = startIndex;
   _windowElementSize = windowElementSize;
   _navMiddleware.requestLastDestinationList(_startIndex, _windowElementSize, navmiddleware::DestinationMemoryEntry::SORTINGALGORITHM_BY_START_GUIDANCE_TIME_NEWEST_FIRST, navmiddleware::DestinationMemoryItemList::FILTER_ENTRY_AND_ROUTE_WAYPOINT);
   _navMiddleware.requestLastDeparture();
}


tSharedPtrDataProvider PreviousDestinationListHandler::getPreviousDestinationsListDataProvider(unsigned int listId, const char* itemId, unsigned int startIndex, unsigned int windowElementSize)
{
   ETG_TRACE_USR4(("PreviousDestinationListHandler::getPreviousDestinationsListDataProvider: _listId= %d, startIndex = %d, _windowElementSize = %d", \
                   _listId, _startIndex, _windowElementSize));

   ListDataProviderBuilder listBuilder(listId, itemId);

   const navmiddleware::DestinationMemoryItemList& destinationMemoryItemList = _navMiddleware.getLastDestinationList();
   const ::std::vector<navmiddleware::DestinationMemoryItem>& infos = destinationMemoryItemList.getItemList();

   unsigned int idx = startIndex;
   unsigned int actDestListSize = (*getPreviousDestCountInfo()).mTotalPreviousDestCount;
   PreviousDestinationValData item;

   if (_listId == LIST_ID_NAV_PREV_DEST_MEMORY_LIST)
   {
      if (actDestListSize == 0)
      {
         listBuilder.AddItem(idx, // - identifies the row
                             0UL, // - unused
                             DATA_CONTEXT_EMPTY_LIST_ITEM).AddData(NO_ENTRIES); // 0 - Name(Id) of the button
         actDestListSize = 1;
      }
      else
      {
         ::std::vector<navmiddleware::DestinationMemoryItem>::const_iterator info = infos.begin();
         // create new vector which include previous Start Point and Previous destination list
         ::std::vector<navmiddleware::DestinationMemoryItem> infos_IncludeStartPoint;
         for (; info != infos.end(); info++)
         {
            infos_IncludeStartPoint.push_back(*info);
         }

         // Build the list based on vector included Previous Start point and Previous Destination
         ::std::vector<navmiddleware::DestinationMemoryItem>::const_iterator infoTotal = infos_IncludeStartPoint.begin();
         for (; (idx < startIndex + windowElementSize) && (infoTotal != infos_IncludeStartPoint.end()); ++idx)
         {
            App::DataModel::util::MultiLangTextBuilder textBuilder;
            if (infoTotal->hasEntry())
            {
               /* Fetch location details */
               std::string locationName, locationAddress;

               const LocationAttributeInfos& infos = infoTotal->getEntry().getDestination().getAttributes();
               const ::std::vector<const LocationAttributeInfos::LocationAttributeInfo*>& locationAttributeInfos = infos.getAttributeInfos();

               ::std::vector<const LocationAttributeInfos::LocationAttributeInfo*>::const_iterator infosIterator = locationAttributeInfos.begin();
               for (; infosIterator != locationAttributeInfos.end(); ++infosIterator)
               {
                  if (NULL != *infosIterator)
                  {
                     switch ((*infosIterator)->m_type)
                     {
                        case LocationAttributeInfos::LocationAttributeInfo::ATTRIBUTE_TYPE__LOCATION_NAME:
                        {
                           // For POI, location name is valid. For address/coordinates, the address details content is seen in location name.
                           if (LocationAttributeInfos::LOCATIONTYPE_POI == infos.getLocationType())
                           {
                              const LocationAttributeInfos::StringAttributeInfo* locationNameInfo = infos.getLocationNameInfo();
                              if (NULL != locationNameInfo)
                              {
                                 locationName = locationNameInfo->m_content;
                                 ETG_TRACE_USR4(("PreviousDestinationListHandler::getPreviousDestinationsListDataProvider(), Location name = %s", locationName.c_str()));
                              }
                           }
                        }
                        break;

                        case LocationAttributeInfos::LocationAttributeInfo::ATTRIBUTE_TYPE__ADDRESS_DETAILS:
                        {
                           const LocationAttributeInfos::AddressDetailInfo* addressDetailInfo = infos.getAddressDetailInfos();
                           if (NULL != addressDetailInfo)
                           {
                              locationAddress = addressDetailInfo->getAddressValue(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__FULL_ADDRESS);
                              ETG_TRACE_USR4(("PreviousDestinationListHandler::getPreviousDestinationsListDataProvider(), Location address = %s", locationAddress.c_str()));
                           }
                        }
                        break;

                        default:
                        {
                           // nothing to do
                        }
                     }     // end of switch case
                  }        // end of if null pointer check
               }           // end of for loop
               /* End of Fetch location details */

               /* Creating list entry by concatenating location name and address */
               if (false == locationName.empty())
               {
                  textBuilder.AddData(locationName);
                  textBuilder.AddData(", ");
               }
               textBuilder.AddData(locationAddress);
               item.mPOIaddress = textBuilder.GetData();
               ETG_TRACE_USR4(("PreviousDestinationListHandler::getPreviousDestinationsListDataProvider(), List entry = %s", item.mPOIaddress.GetCString()));
            }
            else if (infoTotal->hasRoute())
            {
               //In case item has a route type. No need to append the address
               textBuilder.AddData(infoTotal->getRoute().getName().c_str());
            }
            else
            {
               //do nothing
            }
            listBuilder.AddItem(idx, 0UL, DATA_CONTEXT_LIST_PREV_DESTINATION_ITEM).AddDataBindingUpdater<PreviousDestinationValDataBindingSource>(item);
            ++infoTotal;
         } // end of for
      } // end of else
   } //end of if _listId == LIST_ID_NAV_PREV_DEST_MEMORY_LIST

   return listBuilder.CreateDataProvider(startIndex, actDestListSize);
}


bool PreviousDestinationListHandler::onCourierMessage(const DeletePreviousDestinationReqMsg& oMsg)
{
   unsigned int deleteType = oMsg.GetPopupID();

   ETG_TRACE_USR4(("PreviousDestinationListHandler::DeletePreviousDestinationReqMsg: deleteType= %d", deleteType));

   if (deleteType == 0)
   {
      _navMiddleware.removeLastDestinationListItems(_lastDestinationElementId);
   }
   else
   {
      _navMiddleware.removeAllLastDestinationListItems(navmiddleware::DestinationMemoryItemList::FILTER_ENTRY_AND_ROUTE_WAYPOINT);
   }

   return true;
}
