/**************************************************************************************
* @file         : MapPickingHandler.h
* @author       :
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) -2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#ifndef MAP_PICKING_HANDLER_H
#define MAP_PICKING_HANDLER_H

#include "Common/Util/HMIModelIncludes.h"
#if defined(HALL_TO_MDW_COM)

#define ACTIVATE_TIMER_BASED_PICKING

/**
* Handler for map picking
*/
class MapPickingHandler : public HMIModelBase
{
   public:
      MapPickingHandler(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore);
      virtual ~MapPickingHandler();

      virtual void initialize();
      virtual void deinitialize();

      // Incoming property updates from middleware
      PROPERTY_UPDATE_MIDDLEWARE_BEGIN()
      ON_PROPERTY_MIDDLEWARE_UPDATE(navmiddleware::MAP_PICKING_RESULTS_AVAILABLE, onPropertyUpdatePickingResultsAvailable)
      ON_PROPERTY_MIDDLEWARE_UPDATE(navmiddleware::MAP_PICKING_DETAIL_RESULT_AVAILABLE, onPropertyUpdatePickingDetailResultsAvailable)
      ON_PROPERTY_MIDDLEWARE_UPDATE(navmiddleware::MAP_TOUCH_EVENT, onPropertyUpdateMapTouchEvent)
      ON_PROPERTY_MIDDLEWARE_UPDATE(navmiddleware::MAP_PICKING_RESULTS_DISTANCE_UPDATE, onPropertyUpdateMapPickingResultsDistanceUpdate)
      ON_PROPERTY_MIDDLEWARE_UPDATE(navmiddleware::MAP_VIEWSTATUS_CHANGED, onPropertyUpdateMapViewStatusChanged)
      ON_PROPERTY_MIDDLEWARE_UPDATE(navmiddleware::MAP_SET_MAP_CAMERA_AND_VIEW_MODES_DONE, onPropertyUpdateMapCameraAndViewModesDone)
      PROPERTY_UPDATE_MIDDLEWARE_END()

      bool onPropertyUpdatePickingResultsAvailable();

      /**
      * Updates the current street info whenever there is a change in the position
      * \async :: MAP_PICKING_DETAIL_RESULT_AVAILABLE
      */
      void onPropertyUpdatePickingDetailResultsAvailable();

      /**
      * The map information is updatedbased on the type of gesture provided and the timer activity
      * \async :: MAP_TOUCH_EVENT
      */
      bool onPropertyUpdateMapTouchEvent();
      bool onPropertyUpdateMapPickingResultsDistanceUpdate();

      bool onPropertyUpdateMapViewStatusChanged();   // Provides map's state - Freeze/Unfreeze
      bool onPropertyUpdateMapCameraAndViewModesDone();

      // Incoming property updates from info store
      PROPERTY_UPDATE_INFOSTORE_BEGIN()
      ON_PROPERTY_INFOSTORE_UNUSED()
      PROPERTY_UPDATE_INFOSTORE_END()

      // Incoming courier messages from HMI
      COURIER_MSG_MAP_BEGIN(TR_CLASS_APPHMI_NAVIGATION_COURIER_PAYLOAD_MODEL_COMP)
      ON_COURIER_MESSAGE(StartPickingReqMsg)
      ON_COURIER_MESSAGE(TriggerMapPickingReqMsg)
      ON_COURIER_MESSAGE(CompassOrientationReqMsg)
      ON_COURIER_MESSAGE(PickingTimerExpiredReqMsg)
      ON_COURIER_MESSAGE(SetPickingTimerReqMsg)
      ON_COURIER_MESSAGE(ReleasePickInMapReqMsg)
      ON_COURIER_MESSAGE(SetPickInMapLocationReqMsg)
      ON_COURIER_MESSAGE(SavePickInMapCoordinatesReqMsg)
      ON_COURIER_MESSAGE(StoreLocationOnByMapReqMsg)
      COURIER_MSG_MAP_END()

      bool onCourierMessage(const StartPickingReqMsg& oMsg);
      bool onCourierMessage(const TriggerMapPickingReqMsg& oMsg);
      bool onCourierMessage(const CompassOrientationReqMsg& oMsg);
      bool onCourierMessage(const PickingTimerExpiredReqMsg& oMsg);
      bool onCourierMessage(const SetPickingTimerReqMsg& oMsg);
      bool onCourierMessage(const ReleasePickInMapReqMsg& oMsg);
      bool onCourierMessage(const SetPickInMapLocationReqMsg& oMsg);
      bool onCourierMessage(const SavePickInMapCoordinatesReqMsg& oMsg);
      bool onCourierMessage(const StoreLocationOnByMapReqMsg& oMsg);

      /**
      * Checks for the status of map
      */
      void setPickingTimerActive(bool activate, unsigned int duration = NAVI_HMI_SM_C_MAP_PICKING_TIMER_DEFAULT);
      void setOkButtonVisibility(bool Visibility);
      bool sendOkButtonInfoData(void);

      Candera::String updateStormTypeHail(const navmiddleware::PickingResults::PickingResult& pickingResult);
      Candera::String updateStormType(const navmiddleware::PickingResults::PickingResult& pickingResult);
      Candera::String updateDirection(const ::navmiddleware::DirectionDescription& directionDescription);
      Candera::String updateHailInfo(const navmiddleware::PickingResults::PickingResult& pickingResult);

   private:
      enum MapPickingInformationType
      {
         MAP_PICKING_INFO_TYPE_TRAFFIC,
         MAP_PICKING_INFO_TYPE_NON_TRAFFIC
      };

      bool _isMapActive;
      bool _isMapPickTriggerRequired;
      ::navmiddleware::MapCameraModeInfo _mapCamerModeInfo;
      unsigned int _activeMapPickingResultPage;
      unsigned int _numberOfPOIElementsInPickedResult;
      enMapPickingType _mapPickingType;
      Util::Timer _timer;
      bool _isMapChangedBeforePicking;
      bool _isMapIconSelected;
      navmiddleware::settings::MapSettings mapSettings;
      std::vector<PosWGS84<double> > _pickInCoordinates;

      // Databindings used for picking popup in map scroll screen
      DataBindingItem<PickFromMapDataDataBindingSource> _pickFromMapData;
      DataBindingItem<AddressInputValDetailsDataBindingSource> _addressInputValDetailsData;

      void triggerMapPicking();
      void highlightPickingElement(const navmiddleware::PickingResults::PickingResult& pickingResult);
      void unhighlightPickingElement();
      void setDestinationPickingInfoInPickFromMapData();
      void setTrafficPickingInfoInPickFromMapData();
      void setPOIPickingInfoInPickFromMapData();
      void setAddressBookPickingInfoInPickFromMapData();
      void setRoadPickedElementInPickFromMapData();
      void setMapPickingTitleInPickFromMapData();
      void handleMapPickingDetailedResult(const navmiddleware::PickingResults& pickingResults);

      /**
      * Sets the Distance and the Direction for a picked location on the map
      */
      void setDistanceAndDirectionForPickingElement();
      void setPOIPickingInfoDetailsInPickFromMapData();

      /**
      * Ensures that the modified items in PickFromMap are updated
      */
      void sendPickFromMapDataUpdate();
      void sendPickTypeFromMapUpdate();
      void updatePickFromMapPopupData();
      void updatePickFromMapWeatherPopupData();
      void updateHailPopupData(const navmiddleware::PickingResults::PickingResult& pickingResult);
      void updateWeatherAlertData(const navmiddleware::PickingResults::PickingResult& pickingResult);
      void updateStormPopupData(const navmiddleware::PickingResults::PickingResult& pickingResult);
      bool isCarsorPicked(const navmiddleware::PickingResults& pickingResults);
      void resetRoadNameDistanceAndDirectionForPickedElement();
      void setCallButtonBinding();
      void setPhoneNumberForPlaceInfo(const navmiddleware::PickingResults::PickingResult& pickingResult);
      void getDetailedInfoOfSelectedCoOrdinate(const navmiddleware::PickingResults& pickingResults);
      void handlepickingResultType(const navmiddleware::PickingResults::PickingResult& pickingResult);
      void setMapScreenDataPickingResultLatitudeLongitudeInfos(const navmiddleware::PickingResults& pickingResults, bool isLatLongInfoEnable = false);
      void updateInfoStoreCoordinates(const double& latitude, const double& longitude);

      FEATSTD_MAKE_CLASS_UNCOPYABLE(MapPickingHandler);
};


#endif // HALL_TO_MDW_COM
#endif // MAP_PICKING_HANDLER_H
