/**************************************************************************************
* @file         : MapPickingHandler.cpp
* @author       :
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) -2020 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/


#include "gui_std_if.h"
#include "MapPickingHandler.h"
#include "Map/MapScreenDataUtils.h"
#include "Map/MapUtils.h"
#include "Common/Util/ImageUtils.h"
#include "POI/DestinationDataUtils.h"
#include "../Common/Util/CommonUnitsUtility.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/MapPickingHandler.cpp.trc.h"
#endif


#if defined (HALL_TO_MDW_COM)

using namespace navmiddleware;

static const float HEADING_STEP_SIZE = 11.25f;

inline float convertHeadingIndexToHeading(int headingIndex)
{
   return static_cast<float>(headingIndex * HEADING_STEP_SIZE);
}


MapPickingHandler::MapPickingHandler(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore)
   : HMIModelBase(navMiddleware, infoStore)
   , _isMapActive(false)
   , _isMapPickTriggerRequired(false)
   , _mapPickingType(MAP_PICKING_INVALID)
   , _activeMapPickingResultPage(0)
   , _numberOfPOIElementsInPickedResult(0)
   , _isMapIconSelected(false)
   , _isMapChangedBeforePicking(false)
{
}


MapPickingHandler::~MapPickingHandler()
{
}


void MapPickingHandler::initialize()
{
   _navMiddleware.registerMapPropertyUpdateCallback(*this);
   _infoStore.registerDataPropertyUpdateCallback(*this);
}


void MapPickingHandler::deinitialize()
{
   _navMiddleware.unregisterMapPropertyUpdateCallback(*this);
   _infoStore.unregisterDataPropertyUpdateCallback(*this);
}


bool MapPickingHandler::onPropertyUpdatePickingResultsAvailable()
{
   _numberOfPOIElementsInPickedResult = 0;
   ETG_TRACE_USR4(("MapPickingHandler::onPropertyUpdatePickingResultsAvailable(), mapPickingType : %d", _mapPickingType));

   switch (_mapPickingType)
   {
      case MAP_PICKING_FOR_STREET_AUTOPICK:
      case MAP_PICKING_FOR_DESTINATION_AUTOPICK:
      case MAP_PICKING_FOR_DETAILED_INFO_LOCATION_SCROLL:
      case MAP_PICKING_FOR_POI:
      {
         const PickingResults& pickingResults = _navMiddleware.getPickingResults();
         handleMapPickingDetailedResult(pickingResults);
         break;
      }
      default:
         ETG_TRACE_USR4(("MapPickingHandler::onPropertyUpdatePickingResultsAvailable(), Unhandled mapPickingType : %d", _mapPickingType));
         break;
   }

   //update distance and direction when car is static
   setDistanceAndDirectionForPickingElement();
   setOkButtonVisibility(Ext_bIsEnableByMapOKBtn);
   return sendMapScreenData();
}


bool MapPickingHandler::onCourierMessage(const StoreLocationOnByMapReqMsg& oMsg)
{
   ETG_TRACE_USR4(("MapPickingHandler::onCourierMessage(StoreLocationOnByMapReqMsg)"));

   const PickingResults& pickingResults = _navMiddleware.getPickingResults();
   getDetailedInfoOfSelectedCoOrdinate(pickingResults);

   return true;
}


void MapPickingHandler::handleMapPickingDetailedResult(const PickingResults& pickingResults)
{
   ETG_TRACE_USR4(("MapPickingHandler::handleMapPickingDetailedResult()"));
   navmiddleware::settings::MapDetailTriggerSettings mapDetailTriggerSettings = _navMiddleware.getMapDetailTriggerSettings();
   setMapScreenDataPickingResultLatitudeLongitudeInfos(_navMiddleware.getPickingResults(), mapDetailTriggerSettings.isLatLongInfoEnabled());
   if (!pickingResults.getPickingResultList().empty())
   {
      _activeMapPickingResultPage = 1;
      _isMapIconSelected = false;
      const std::vector <navmiddleware::PickingResults::PickingResult>& pickingResult = _navMiddleware.getPickingResults().getPickingResultList();
      for (unsigned int index = 0; (index < pickingResult.size()); ++index)
      {
         handlepickingResultType(pickingResult[index]);
      }
      if ((MAP_PICKING_FOR_POI == _mapPickingType) && (0 == _numberOfPOIElementsInPickedResult) && (false == _isMapIconSelected))
      {
         getDetailedInfoOfSelectedCoOrdinate(pickingResults);
         //Activate Detailed Info When Selected POI is only one
         POST_MSG((COURIER_MESSAGE_NEW(ActivatePOIInfoSelectedOnMapMsg)(false)));
         _mapPickingType = MAP_PICKING_FOR_STREET_AUTOPICK;
      }
      // request for picked element details
      _navMiddleware.requestPickingDetails(MAP_VIEW_ID__PRIMARY, pickingResults.getPickingResultList());
   }
   else
   {
      _mapPickingType = MAP_PICKING_FOR_STREET_AUTOPICK;
   }
   sendPickFromMapDataUpdate();
}


void MapPickingHandler::setMapScreenDataPickingResultLatitudeLongitudeInfos(const navmiddleware::PickingResults& pickingResult, bool isLatLongInfoEnable)
{
   ETG_TRACE_USR4(("MapPickingHandler::setMapScreenDataPickingResultLatitudeLongitudeInfos()"));
   Candera::String pickingResultLatitudeLongitude;
   CommonUnitsUtility commonUtility;
   CommonUnitsUtility::sCoordinateInfo latitude;
   CommonUnitsUtility::sCoordinateInfo longitude;
   std::string strLat;
   std::string strLong;

   strLat = pickingResult.getLatitudeAsString();
   strLong = pickingResult.getLongitudeAsString();
   char direction[4] = { 'N', 'E', 'S', 'W' };

   std::stringstream stringStream;
   latitude = commonUtility.convertCoordinateDisplayFormat(strLat.c_str(), CommonUnitsUtility::EN_LATITUDE, CommonUnitsUtility::EN_SEXAGESIMAL);
   longitude = commonUtility.convertCoordinateDisplayFormat(strLong.c_str(), CommonUnitsUtility::EN_LONGITUDE, CommonUnitsUtility::EN_SEXAGESIMAL);

   // Lat/Long conversion utility function
   stringStream << latitude.degree.c_str() << "°" << latitude.minute.c_str() << "'" << latitude.second.c_str() << "\"" << direction[latitude.direction] << "/"
                << longitude.degree.c_str() << "°" << longitude.minute.c_str() << "'" << longitude.second.c_str() << "\"" << direction[longitude.direction];

   pickingResultLatitudeLongitude = stringStream.str().c_str();

   if ((*_pickFromMapData).mSelectedRoad != pickingResultLatitudeLongitude)
   {
      (*_pickFromMapData).mSelectedRoad = pickingResultLatitudeLongitude;
      _pickFromMapData.MarkItemModified(ItemKey::PickFromMapData::SelectedRoadItem);
   }
}


void MapPickingHandler::handlepickingResultType(const PickingResults::PickingResult& pickingResult)
{
   ETG_TRACE_USR4(("MapPickingHandler::handlepickingResultType()"));
   ETG_TRACE_USR4(("MapPickingHandler::Picking result type : %d)", pickingResult.m_elementType));

   if (pickingResult.m_elementType == PICKING_RESULT_TYPE__ROAD)
   {
      if ((!(pickingResult.m_title.getValue().empty()) && (*_pickFromMapData).mSelectedRoad != pickingResult.m_title.getValue().c_str()))
      {
         (*_pickFromMapData).mSelectedRoad = pickingResult.m_title.getValue().c_str();
         _pickFromMapData.MarkItemModified(ItemKey::PickFromMapData::SelectedRoadItem);
      }
      else
      {
         // In absence of C++11 and to_string()
         std::ostringstream stream;
         stream << double(pickingResult.m_geoCoordinate.getLatitude()) << "," << pickingResult.m_geoCoordinate.getLongitude();
         std::string coordinateStr = stream.str();
         if ((*_pickFromMapData).mSelectedRoad != coordinateStr.c_str())
         {
            (*_pickFromMapData).mSelectedRoad = coordinateStr.c_str();
            _pickFromMapData.MarkItemModified(ItemKey::PickFromMapData::SelectedRoadItem);
         }
      }
      // highlight first picking result
      highlightPickingElement(pickingResult);
   }
   else if (pickingResult.m_elementType == PICKING_RESULT_TYPE__POI)
   {
      _numberOfPOIElementsInPickedResult++;
   }
   else if ((pickingResult.m_elementType == PICKING_RESULT_TYPE__TRAFFIC_ICON)
            || (pickingResult.m_elementType == PICKING_RESULT_TYPE__ADDRESS_BOOK))
   {
      _isMapIconSelected = true;
   }
   else
   {
      //do nothing
   }
}


void MapPickingHandler::getDetailedInfoOfSelectedCoOrdinate(const PickingResults& pickingResults)
{
   ETG_TRACE_USR4(("MapPickingHandler::getDetailedInfoOfSelectedCoOrdinate()"));

   _navMiddleware.setLocationWithCoordinates(pickingResults.getPickedGeoCoordinate().getLatitude(), pickingResults.getPickedGeoCoordinate().getLongitude());
   updateInfoStoreCoordinates(pickingResults.getPickedGeoCoordinate().getLatitude(), pickingResults.getPickedGeoCoordinate().getLongitude());

   _infoStore.setIsDetailInfoRequested(true, InfoStoreBase::DETAIL_INFO_MAPPICK);
   _navMiddleware.requestLocationAttributes();
}


void MapPickingHandler::onPropertyUpdatePickingDetailResultsAvailable()
{
   ETG_TRACE_USR4(("MapPickingHandler::onPropertyUpdatePickingDetailResultsAvailable()"));
   const ::std::vector<navmiddleware::PickingResults::PickingResult>& pickingResultList = _navMiddleware.getPickingResults().getPickingResultList();
   if (0 < pickingResultList.size())
   {
      if ((_mapPickingType == MAP_PICKING_FOR_STREET_AUTOPICK) || (_mapPickingType == MAP_PICKING_FOR_DETAILED_INFO_LOCATION_SCROLL))
      {
         ETG_TRACE_USR4(("MapPickingHandler::onPropertyUpdatePickingDetailResultsAvailable()::MAP_PICKING_FOR_STREET_AUTOPICK"));
         setRoadPickedElementInPickFromMapData();
         sendPickFromMapDataUpdate();
      }
      else if (_mapPickingType == MAP_PICKING_FOR_POI)
      {
         if (((_numberOfPOIElementsInPickedResult == 1)
               || (PICKING_RESULT_TYPE__ADDRESS_BOOK == pickingResultList.at(0).m_elementType)))
         {
            ETG_TRACE_USR4(("MapPickingHandler::onPropertyUpdatePickingDetailResultsAvailable()::MAP_PICKING_FOR_POI, _numberOfPOIElementsInPickedResult = 1"));

            updateInfoStoreCoordinates(pickingResultList.at(0).m_geoCoordinate.getLatitude(), pickingResultList.at(0).m_geoCoordinate.getLongitude());
            _navMiddleware.setLocationWithPickingResult(pickingResultList.at(0));
            setPickedPOIDetailInfo(pickingResultList.at(0), _infoStore);

            //Courier msg has been posted to show POI detailed info with the detailed address details
            POST_MSG((COURIER_MESSAGE_NEW(ShowDetailedAddressMsg)()));
            POST_MSG((COURIER_MESSAGE_NEW(ShowFavEditIcon_Enable_Delete_Guidance_Buttons)(0, false, true, 1, false))); // To show fav icon and to disable delete buttons
            POST_MSG((COURIER_MESSAGE_NEW(POIIconSelectionOnMapMsg)()));
         }
         else if (_numberOfPOIElementsInPickedResult > 1)
         {
            ETG_TRACE_USR4(("MapPickingHandler::onPropertyUpdatePickingDetailResultsAvailable()::MAP_PICKING_FOR_POI, _numberOfPOIElementsInPickedResult > 1"));
            POST_MSG((COURIER_MESSAGE_NEW(ActivatePOIInfoSelectedOnMapMsg)(true)));
         }
         else if (PICKING_RESULT_TYPE__TRAFFIC_ICON == pickingResultList.at(0).m_elementType)
         {
            ETG_TRACE_USR4(("MapPickingHandler::onPropertyUpdatePickingDetailResultsAvailable()::PICKING_RESULT_TYPE__TRAFFIC_ICON"));
            POST_MSG((COURIER_MESSAGE_NEW(ShowTrafficDetailedInfoViewMsg)()));
            setTrafficPickingInfoInPickFromMapData();
         }
         else
         {
            //For do nothing
         }
         _mapPickingType = MAP_PICKING_FOR_STREET_AUTOPICK;
      }
      else
      {
         //For Other Picked Elements
      }
   }
   setOkButtonVisibility(Ext_bIsEnableByMapOKBtn);
}


bool MapPickingHandler::onPropertyUpdateMapTouchEvent()
{
   navmiddleware::TouchEventInfo touchEventInfo = _navMiddleware.getTouchEventInfo();
   ETG_TRACE_USR4(("MapPickingHandler::onPropertyUpdateMapTouchEvent(gestureType %d, speedLock %d)", touchEventInfo.m_touchGestureType, _infoStore.getSpeedLockState()));

   switch (touchEventInfo.m_touchGestureType)
   {
      case TOUCH_GESTURE_TAPPED:
      case TOUCH_GESTURE_SWIPE:
      case TOUCH_GESTURE_ROTATE:
      case TOUCH_GESTURE_PINCH:
         // if a gesture is recognized we deactivate timer based picking
         setPickingTimerActive(false);
         break;
      default:
         break;
   }
   return true;
}


bool MapPickingHandler::onCourierMessage(const CompassOrientationReqMsg& oMsg)
{
   ETG_TRACE_USR4(("MapPickingHandler::onCourierMessage(CompassOrientationReqMsg)"));

   bool isSecondaryMapView = oMsg.GetIsSecondaryViewActive();
   MapRepresentation mapRepresentation;

   if (false == isSecondaryMapView)  // Press on compass in primary map
   {
      navmiddleware::settings::MapView mapView = _navMiddleware.getChangeMapViewSettings().getMapView();
      mapRepresentation.m_mapPitch = ((navmiddleware::settings::MAPVIEW_2D == mapView) || (navmiddleware::settings::MAPVIEW_2D_SPLIT == mapView))
                                     ? navmiddleware::MAP_PITCH_2D : navmiddleware::MAP_PITCH_3D;
      mapRepresentation.m_mapOrientation = _navMiddleware.getMapSettings().getMapOrientation();

      if (mapRepresentation.m_mapPitch == navmiddleware::MAP_PITCH_3D)
      {
         // switch 3D_CAR_HEADING to 2D_NORTH_UP
         mapRepresentation.m_mapPitch = navmiddleware::MAP_PITCH_2D;
         mapRepresentation.m_mapOrientation = navmiddleware::settings::MAP_ORIENTATION_NORTH_UP;
         mapView = (false == isMultiViewActive(_navMiddleware)) ? navmiddleware::settings::MAPVIEW_2D : navmiddleware::settings::MAPVIEW_2D_SPLIT;
      }
      else if (mapRepresentation.m_mapPitch == navmiddleware::MAP_PITCH_2D && mapRepresentation.m_mapOrientation == navmiddleware::settings::MAP_ORIENTATION_NORTH_UP)
      {
         // switch 2D_NORTH_UP to 2D_CAR_HEADING
         mapRepresentation.m_mapOrientation = navmiddleware::settings::MAP_ORIENTATION_HEAD_UP;
      }
      else if ((mapRepresentation.m_mapPitch == navmiddleware::MAP_PITCH_2D && mapRepresentation.m_mapOrientation == navmiddleware::settings::MAP_ORIENTATION_HEAD_UP)
               && (!_infoStore.isNo3D_TypeSelected()) && (!_infoStore.isSecondaryMapStreamingActive()))
      {
         // switch 2D_CAR_HEADING to 3D_CAR_HEADING
         mapRepresentation.m_mapPitch = navmiddleware::MAP_PITCH_3D;
         mapView = (false == isMultiViewActive(_navMiddleware)) ? navmiddleware::settings::MAPVIEW_3D : navmiddleware::settings::MAPVIEW_3D_SPLIT;
      }
      else if ((mapRepresentation.m_mapPitch == navmiddleware::MAP_PITCH_2D && mapRepresentation.m_mapOrientation == navmiddleware::settings::MAP_ORIENTATION_HEAD_UP)
               && (_infoStore.isNo3D_TypeSelected() || _infoStore.isSecondaryMapStreamingActive()))
      {
         mapRepresentation.m_mapOrientation = navmiddleware::settings::MAP_ORIENTATION_NORTH_UP;
      }
      else
      {
         // do nothing
      }
      _navMiddleware.setMapRepresentation(MAP_VIEW_ID__PRIMARY, mapRepresentation, CameraAnimation__FAST);
      if (mapView != _navMiddleware.getChangeMapViewSettings().getMapView())
      {
         navmiddleware::settings::ChangeMapViewSettings mapViewSettings = _navMiddleware.getChangeMapViewSettings();
         mapViewSettings.setMapView(mapView);
         _navMiddleware.updateChangeMapViewSettings(mapViewSettings);
         _navMiddleware.applySettings();
      }
   }
   else   // Press on compass in secondary map
   {
      mapRepresentation.m_mapPitch = navmiddleware::MAP_PITCH_2D;
      navmiddleware::settings::MapOrientation mapOrientation = _navMiddleware.getMapSettings().getMapSecondaryViewOrientation();
      mapRepresentation.m_mapOrientation = (mapOrientation == navmiddleware::settings::MAP_ORIENTATION_NORTH_UP)
                                           ? navmiddleware::settings::MAP_ORIENTATION_HEAD_UP
                                           : navmiddleware::settings::MAP_ORIENTATION_NORTH_UP;
      _navMiddleware.setMapRepresentation(MAP_VIEW_ID__SECONDARY, mapRepresentation, CameraAnimation__FAST);
   }
   return true;
}


bool MapPickingHandler::onPropertyUpdateMapCameraAndViewModesDone()
{
   EXT_bIsMapCameraModeChangeDone = true;
   _mapCamerModeInfo = _navMiddleware.getMapCameraModeInfo();
   ETG_TRACE_USR4(("MapPickingHandler::onPropertyUpdateMapCameraAndViewModesDone(), Map mode : %d", _mapCamerModeInfo.getMapCameraModeInfo()));

   if (navmiddleware::MAP_MODE_FREE == _mapCamerModeInfo.getMapCameraModeInfo())
   {
      if (true == _isMapPickTriggerRequired)
      {
         ETG_TRACE_USR4(("MapPickingHandler::onPropertyUpdateMapCameraAndViewModesDone(), Map picking triggered"));
         triggerMapPicking();
      }
   }
   return true;
}


bool MapPickingHandler::onPropertyUpdateMapViewStatusChanged()
{
   bool newIsMapActive = !(_navMiddleware.isMapFrozen(MAP_VIEW_ID__PRIMARY));
   if (_isMapActive != newIsMapActive)
   {
      _isMapActive = newIsMapActive;
      ETG_TRACE_USR4(("MapPickingHandler::onPropertyUpdateMapViewStatusChanged(), _isMapActive : %d, Map mode : %d", _isMapActive, _mapCamerModeInfo.getMapCameraModeInfo()));

      if (true == _isMapPickTriggerRequired)
      {
         triggerMapPicking();
      }
   }
   return true;
}


bool MapPickingHandler::onCourierMessage(const StartPickingReqMsg& oMsg)
{
   _mapPickingType = oMsg.GetMapPickingType();
   ETG_TRACE_USR4(("MapPickingHandler::onCourierMessage(StartPickingReqMsg(_mapPickingType %d))", _mapPickingType));

   return true;
}


bool MapPickingHandler::onCourierMessage(const TriggerMapPickingReqMsg& oMsg)
{
   ETG_TRACE_USR4(("MapPickingHandler::onCourierMessage(TriggerMapPickingReqMsg)"));

   COURIER_UNUSED(oMsg);
   triggerMapPicking();

   return true;
}


void MapPickingHandler::triggerMapPicking()
{
   ETG_TRACE_USR4(("MapPickingHandler::triggerMapPicking(), _mapPickingType %d, _isMapActive : %d, Map mode : %d", _mapPickingType, _isMapActive, _mapCamerModeInfo.getMapCameraModeInfo()));

   _isMapPickTriggerRequired = false;

   if ((true == _isMapActive) && (true == EXT_bIsMapCameraModeChangeDone) && (navmiddleware::MAP_MODE_FREE == _mapCamerModeInfo.getMapCameraModeInfo()))
   {
      unhighlightPickingElement();

      navmiddleware::ValidValue<PickingResultFilter> pickingResultFilter;

      switch (_mapPickingType)
      {
         case MAP_PICKING_INVALID:
         case MAP_PICKING_FOR_WEATHER_MAP:
            break;

         case MAP_PICKING_FOR_DESTINATION_AUTOPICK:
         case MAP_PICKING_FOR_STREET_AUTOPICK:
         case MAP_PICKING_FOR_SIMULATION_START_POSITION:
         case MAP_PICKING_FOR_AVOID_AREA:
         case MAP_PICKING_FOR_ADDRESS_BOOK_MOVE_LOCATION:
         case MAP_PICKING_FOR_MODIFY_AVOID_AREA:
         {
            _isMapChangedBeforePicking = false;
            _navMiddleware.pickInMap(MAP_VIEW_ID__PRIMARY, _infoStore.getScreenWidth() / 2, _infoStore.getScreenHeight() / 2, pickingResultFilter);
            break;
         }

         case MAP_PICKING_FOR_DETAILED_INFO_LOCATION_SCROLL:
         {
            _isMapChangedBeforePicking = false;
            setOkButtonVisibility(Ext_bIsEnableByMapOKBtn);
            _navMiddleware.pickInMap(MAP_VIEW_ID__PRIMARY, _infoStore.getScreenWidth() / 2, _infoStore.getScreenHeight() / 2, pickingResultFilter);
            break;
         }

         case MAP_PICKING_FOR_POI:
         {
            //POI is selected on Map temporarily by ButtonPress to test in simulation, Will be disabled when touch events are implemented on target
            setPickingTimerActive(false);
            _navMiddleware.pickInMap(MAP_VIEW_ID__PRIMARY, _infoStore.getScreenWidth() / 2, _infoStore.getScreenHeight() / 2, pickingResultFilter);
            break;
         }
         default:
            break;
      }
      sendPickFromMapDataUpdate();
   }
   else
   {
      _isMapPickTriggerRequired = true;
   }
}


void MapPickingHandler::setTrafficPickingInfoInPickFromMapData()
{
   ETG_TRACE_USR4(("MapPickingHandler::setTrafficPickingInfoInPickFromMapData()"));
   const ::std::vector<navmiddleware::PickingResults::PickingResult>& pickingResultList = _navMiddleware.getPickingResults().getPickingResultList();
   if (!(pickingResultList.empty() && pickingResultList.size() >= _activeMapPickingResultPage))
   {
      const navmiddleware::PickingResults::PickingResult& pickingResult = pickingResultList.at(_activeMapPickingResultPage - 1);
      updateInfoStoreCoordinates(pickingResultList.at(0).m_geoCoordinate.getLatitude(), pickingResultList.at(0).m_geoCoordinate.getLongitude());

      Candera::String mapPickingInfo;
      bool mapPickingInfoAvailable = false;
      if (pickingResult.getPickingResultDetailTm()->m_description.isValid())
      {
         ETG_TRACE_USR4(("MapPickingHandler::setTrafficPickingInfoInPickFromMapData() Picking result is valid"));
         // mapPickingInfo = pickingResult.getPickingResultDetailTm()->m_description.getValue().c_str();
         mapPickingInfo = pickingResult.getPickingResultDetailTm()->m_headline.getValue().c_str();
         navmiddleware::ValidValue<int> selectedTrafficMsg;
         selectedTrafficMsg.setValue(pickingResult.getPickingResultDetailTm()->m_tmId.getValue());
         _infoStore.setTmId(selectedTrafficMsg);
         mapPickingInfoAvailable = true;
      }
      else
      {
         mapPickingInfoAvailable = false;
      }
      ETG_TRACE_USR4(("MapPickingHandler::setTrafficPickingInfoInPickFromMapData() mapPickingInfo.GetCString() = %s", mapPickingInfo.GetCString()));
      _infoStore.setAddressInfoDetailsName("");
      _infoStore.setAddressInfoDetailsAddress(mapPickingInfo.GetCString());
      _infoStore.setAddressInfoDetailsPhoneNumber("");
      _infoStore.setIsDetailInfoRequested(false, InfoStoreBase::DETAIL_INFO_TMC);   // Set for disabling favorites button (in AddressDetailedInfo) when TMC detailed info is shown
      POST_MSG((COURIER_MESSAGE_NEW(ShowDetailedAddressMsg)()));
      POST_MSG((COURIER_MESSAGE_NEW(ShowFavEditIcon_Enable_Delete_Guidance_Buttons)(0, 0, 0, 1, 0)));
   }
}


bool MapPickingHandler::onCourierMessage(const PickingTimerExpiredReqMsg& oMsg)
{
   COURIER_UNUSED(oMsg);
   ETG_TRACE_USR4(("MapPickingHandler::onCourierMessage(PickingTimerExpiredReqMsg)"));

   if ((_mapPickingType == MAP_PICKING_FOR_DESTINATION_AUTOPICK) || (_mapPickingType == MAP_PICKING_FOR_STREET_AUTOPICK) || (_mapPickingType == MAP_PICKING_FOR_DETAILED_INFO_LOCATION_SCROLL))
   {
      // start picking at the center of the screen
#if defined(ACTIVATE_TIMER_BASED_PICKING)
      navmiddleware::ValidValue<PickingResultFilter> pickingResultFilter;
      _navMiddleware.pickInMap(MAP_VIEW_ID__PRIMARY, _infoStore.getScreenWidth() / 2, _infoStore.getScreenHeight() / 2, pickingResultFilter);
#endif
   }

   return true;
}


bool MapPickingHandler::onCourierMessage(const ReleasePickInMapReqMsg& oMsg)
{
   COURIER_UNUSED(oMsg);

   ETG_TRACE_USR4(("MapPickingHandler::onCourierMessage(ReleasePickInMapReqMsg()"));
   _navMiddleware.releasePickInMap(MAP_VIEW_ID__PRIMARY);

   resetRoadNameDistanceAndDirectionForPickedElement();
   return true;
}


bool MapPickingHandler::onCourierMessage(const SetPickInMapLocationReqMsg& oMsg)
{
   COURIER_UNUSED(oMsg);
   ETG_TRACE_USR4(("MapPickingHandler::onCourierMessage(SetPickInMapLocationReqMsg), Latitude : %f, Longitude : %f", _pickInCoordinates[0]._latitude, _pickInCoordinates[0]._longitude));

   _navMiddleware.setLocationWithCoordinates(_pickInCoordinates[0]._latitude, _pickInCoordinates[0]._longitude);
   return true;
}


bool MapPickingHandler::onCourierMessage(const SavePickInMapCoordinatesReqMsg& oMsg)
{
   COURIER_UNUSED(oMsg);
   ETG_TRACE_USR4(("MapPickingHandler::onCourierMessage(SavePickInMapCoordinatesReqMsg)"));

   _pickInCoordinates.clear();

   InfoStoreBase::CoordinatesToBeShownInMap& coordinatesToBeShownInMap = _infoStore.getCoordinatesToBeShownInMap();
   _pickInCoordinates = coordinatesToBeShownInMap._coordinates;

   return true;
}


void MapPickingHandler::sendPickFromMapDataUpdate()
{
   if (_pickFromMapData.HasModifiedItems())
   {
      if (!(_pickFromMapData.SendUpdate()))
      {
         ETG_TRACE_ERR(("_pickFromMapData update failed!"));
      }
   }
}


void MapPickingHandler::setRoadPickedElementInPickFromMapData()
{
   ETG_TRACE_USR4(("MapPickingHandler::setRoadPickedElementInPickFromMapData()"));
   const ::std::vector<navmiddleware::PickingResults::PickingResult>& pickingResultList = _navMiddleware.getPickingResults().getPickingResultList();
   if (!(pickingResultList.empty() && pickingResultList.size() >= _activeMapPickingResultPage))
   {
      ::std::vector<navmiddleware::PickingResults::PickingResult>::const_iterator info = pickingResultList.begin();
      //check for road in the list and highlight the road selected road
      for (; (info != pickingResultList.end()); ++info)
      {
         ETG_TRACE_USR4(("MapPickingHandler::setRoadPickedElementInPickFromMapData(), Picking type : %d", info->m_elementType));
         if (info->m_elementType == PICKING_RESULT_TYPE__ROAD)
         {
            if (info->m_title.isValid() && !(info->m_title.getValue().empty()))
            {
               Candera::String selectedRoad = info->m_title.getValue().c_str();
               if ((*_pickFromMapData).mSelectedRoad != selectedRoad)
               {
                  (*_pickFromMapData).mSelectedRoad = selectedRoad;
                  _pickFromMapData.MarkItemModified(ItemKey::PickFromMapData::SelectedRoadItem);
               }
            }
            highlightPickingElement(*info);
            break;
         }
      }
   }
}


void MapPickingHandler::resetRoadNameDistanceAndDirectionForPickedElement()
{
   setOkButtonVisibility(false);

   Candera::String resetItem = Candera::String("");
   if ((*_pickFromMapData).mSelectedRoad != resetItem)
   {
      (*_pickFromMapData).mSelectedRoad = resetItem;
      _pickFromMapData.MarkItemModified(ItemKey::PickFromMapData::SelectedRoadItem);
   }
   if ((*_pickFromMapData).mPickedDistance != resetItem)
   {
      (*_pickFromMapData).mPickedDistance = resetItem;
      _pickFromMapData.MarkItemModified(ItemKey::PickFromMapData::PickedDistanceItem);
   }
   (*_pickFromMapData).mPickedDirection = ImageLoader::getAssetBitmapImage("");
   _pickFromMapData.MarkItemModified(ItemKey::PickFromMapData::PickedDirectionItem);
   sendPickFromMapDataUpdate();
}


void MapPickingHandler::highlightPickingElement(const navmiddleware::PickingResults::PickingResult& pickingResult)
{
   ETG_TRACE_USR4(("MapPickingHandler::highlightPickingElement()"));

   // highlight new element
   ::std::vector<navmiddleware::PickingResults::PickingResult> pickingResultToBeHighlighted;
   pickingResultToBeHighlighted.push_back(pickingResult);
   _navMiddleware.highlightPickingResult(MAP_VIEW_ID__PRIMARY, pickingResultToBeHighlighted);
}


void MapPickingHandler::unhighlightPickingElement()
{
   ETG_TRACE_USR4(("MapPickingHandler::unhighlightPickingElement()"));

   ::std::vector<navmiddleware::PickingResults::PickingResult> pickingResultToBeHighlighted;
   _navMiddleware.highlightPickingResult(MAP_VIEW_ID__PRIMARY, pickingResultToBeHighlighted);
   _numberOfPOIElementsInPickedResult = 0;
}


//update position when car is driving
bool MapPickingHandler::onPropertyUpdateMapPickingResultsDistanceUpdate()
{
   ETG_TRACE_USR4(("MapPickingHandler::onPropertyUpdateMapPickingResultsDistanceUpdate()"));
   setDistanceAndDirectionForPickingElement();
   return true;
}


void MapPickingHandler::setDistanceAndDirectionForPickingElement()
{
   ETG_TRACE_USR4(("MapPickingHandler::setDistanceAndDirectionForPickingElement()"));
   //distance and direction of crosshair wrt to cursor
   const PickingResults& pickingResults = _navMiddleware.getPickingResults();
   Candera::String pickedDistance;
   if (pickingResults.getAirDistance().isValid())
   {
      pickedDistance = pickingResults.getAirDistance().getValue().c_str();
   }
   else
   {
      pickedDistance = " ";
   }

   if (pickingResults.getDirection().isValid())
   {
      unsigned int pickingDirectionIndex = pickingResults.getDirection().getValue();
      if (pickingDirectionIndex < NUM_DIRECTION_IMAGES)
      {
         (*_pickFromMapData).mPickedDirection = ImageLoader::getAssetBitmapImage(_pickingDirectionImages[pickingDirectionIndex]);
      }
      else
      {
         ETG_TRACE_ERR(("Image index not found."));
      }
   }
   else
   {
      (*_pickFromMapData).mPickedDirection = ImageLoader::getAssetBitmapImage("");
   }

   if ((*_pickFromMapData).mPickedDistance != pickedDistance)
   {
      (*_pickFromMapData).mPickedDistance = pickedDistance;
      _pickFromMapData.MarkItemModified(ItemKey::PickFromMapData::PickedDistanceItem);
   }
   else
   {
      // to nothing
   }
   _pickFromMapData.MarkItemModified(ItemKey::PickFromMapData::PickedDirectionItem);
   sendPickFromMapDataUpdate();
}


bool MapPickingHandler::onCourierMessage(const SetPickingTimerReqMsg& oMsg)
{
   COURIER_UNUSED(oMsg);
   ETG_TRACE_USR4(("MapPickingHandler::onCourierMessage(SetPickingTimerReqMsg)"));

   bool activateTimer = oMsg.GetActivateTimer();
   unsigned int timerDuration = oMsg.GetTimerDuration();

   setPickingTimerActive(activateTimer, timerDuration);
   return true;
}


/**
* Provides time for the user to move the cursor and to avoid calling pickInMap()
*/
void MapPickingHandler::setPickingTimerActive(bool activate, unsigned int duration)
{
   ETG_TRACE_USR4(("MapPickingHandler::setPickingTimerActive(), activate : %d, _mapPickingType: %d", activate, _mapPickingType));

   unhighlightPickingElement();
   if (activate && ((_mapPickingType == MAP_PICKING_FOR_DESTINATION_AUTOPICK) || (_mapPickingType == MAP_PICKING_FOR_STREET_AUTOPICK) || (_mapPickingType == MAP_PICKING_FOR_DETAILED_INFO_LOCATION_SCROLL)))
   {
      if (_timer.running())
      {
         _timer.restart();
      }
      else
      {
         _timer.setTimeout(0, duration, COURIER_MESSAGE_NEW(PickingTimerExpiredReqMsg)());
         _timer.start();
      }
      // This variable is used to avoid calling pickInMap(). User is touching the Map to move the cursor to different place.
      _isMapChangedBeforePicking = true;
   }
   else
   {
      if (_timer.running())
      {
         _timer.stop();
      }
      setOkButtonVisibility(false);
   }
}


/**
* The OK Button is made visible whenever there is a change in location for the first time
*/
void MapPickingHandler::setOkButtonVisibility(bool Visibility)
{
   ETG_TRACE_USR4(("MapPickingHandler::setOkButtonVisibility(), Visibility : %d", Visibility));
   static const char* CMN_IMG_GUIDANCE_CARMODE_ICON_NORMAL = "AppHmi_NavigationModule#Images#Common#Icon_Guidance_carmode_normalBmp";
   static const char* CMN_IMG_GUIDANCE_CARMODE_ICON_PRESSED = "AppHmi_NavigationModule#Images#Common#Icon_Guidance_carmode_touchedBmp";
   static const char* CMN_IMG_GUIDANCE_COACH_ICON_NORMAL = "AppHmi_NavigationModule#Images#Common#Icon_Guidance_normalBmp";
   static const char* CMN_IMG_GUIDANCE_COACH_ICON_PRESSED = "AppHmi_NavigationModule#Images#Common#Icon_Guidance_touchedBmp";

   if ((*getOkButtonInfoData()).mIsOKButtonVisible != Visibility)
   {
      (*getOkButtonInfoData()).mIsOKButtonVisible = Visibility;
   }

   if (_infoStore.getVehicleProfile()) // car mode
   {
      (*getOkButtonInfoData()).mNaviIconNormal = ImageLoader::getAssetBitmapImage(CMN_IMG_GUIDANCE_CARMODE_ICON_NORMAL);
      (*getOkButtonInfoData()).mNaviIconPressed = ImageLoader::getAssetBitmapImage(CMN_IMG_GUIDANCE_CARMODE_ICON_PRESSED);
   }
   else
   {
      (*getOkButtonInfoData()).mNaviIconNormal = ImageLoader::getAssetBitmapImage(CMN_IMG_GUIDANCE_COACH_ICON_NORMAL);
      (*getOkButtonInfoData()).mNaviIconPressed = ImageLoader::getAssetBitmapImage(CMN_IMG_GUIDANCE_COACH_ICON_PRESSED);
   }
   if (Ext_bIsEnableByMapOKBtn == false)
   {
      (*getOkButtonInfoData()).mIsCarModeNaviIconVisible = _infoStore.getIsGuidanceActive() && _infoStore.getVehicleProfile();
   }
   else
   {
      (*getOkButtonInfoData()).mIsCarModeNaviIconVisible = false;
   }

   getOkButtonInfoData().MarkAllItemsModified();
   getOkButtonInfoData().SendUpdate();

   sendOkButtonInfoData();
}


/**
* Ensures that the modifications in the visibility of the OK Button are updated successfully
*/
bool MapPickingHandler::sendOkButtonInfoData()
{
   ETG_TRACE_USR4(("MapPickingHandler::sendOkButtonInfoData()"));
   bool retVal = false;
   if (getOkButtonInfoData().HasModifiedItems())
   {
      if (getOkButtonInfoData().SendUpdate() == false)
      {
         ETG_TRACE_ERR(("getOkButtonInfoData() update failed!"));
         retVal = false;
      }
      else
      {
         retVal = true;
      }
   }
   else
   {
      retVal = true;
   }

   return retVal;
}


void MapPickingHandler::updateInfoStoreCoordinates(const double& latitude, const double& longitude)
{
   ETG_TRACE_USR4(("MapPickingHandler::updateInfoStoreCoordinates(), Latitude : %f, Longitude : %f", latitude, longitude));

   std::vector<PosWGS84<double> > coordinatesVector;
   coordinatesVector.push_back(PosWGS84<double>(longitude, latitude));
   InfoStoreBase::CoordinatesToBeShownInMap& coordinatesToBeShownInMap = _infoStore.getCoordinatesToBeShownInMap();
   coordinatesToBeShownInMap._coordinates = coordinatesVector;

   _infoStore.setLatitude(latitude);
   _infoStore.setLongitude(longitude);
}


#endif // HALL_TO_MDW_COM
