/**************************************************************************************
* @file         : DestinationMatchListHandler.cpp
* @author       : ECG5-Naveen Thangamalayan
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) -2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"

#include "util/StrUtf8.h"
#include "hmi_trace_if.h"
#include "CgiExtensions/ImageLoader.h"
#include "CgiExtensions/CourierMessageMapper.h"
#include "AppHmi_NavigationStateMachine.h"
#include "DestinationMatchListHandler.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/DestinationMatchListHandler.cpp.trc.h"
#endif

static const char* const DATA_CONTEXT_DESTINATION_LOADING_LIST_ITEM = "AddressEntryItem";
static const char* const EMPTY_STRING = "";
static const Candera::UInt8 NO_MATCHES_LIST_SIZE = 1;

static const Candera::String INCOMPLETE_MATCHES = LANGUAGE_STRING(TextId_0x1A59, "Re-try with few more characters.");
static const Candera::String NO_MATCHES = LANGUAGE_STRING(TextId_0x171B, "No matches.");
static const Candera::String LOADING = LANGUAGE_STRING(TextId_0x1458, "Loading...");



#ifdef HALL_TO_MDW_COM

DestinationMatchListHandler::DestinationMatchListHandler(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore)
   : HMIModelBase(navMiddleware, infoStore)
   , _listId(0)
   , _startIndex(0)
   , _windowElementSize(0)
{
}


DestinationMatchListHandler::~DestinationMatchListHandler()
{
}


void DestinationMatchListHandler::initialize()
{
   _navMiddleware.registerLocationPropertyUpdateCallback(*this);
}


void DestinationMatchListHandler::deinitialize()
{
   _navMiddleware.unregisterLocationPropertyUpdateCallback(*this);
}


bool DestinationMatchListHandler::onCourierMessage(const ListDateProviderReqMsg& oMsg)
{
   bool isMsgConsumed = false;
   _listId = oMsg.GetListId();
   if (_listId == LIST_ID_NEWDESTINATION_MATCHLIST)
   {
      _startIndex = oMsg.GetStartIndex();
      _windowElementSize = oMsg.GetWindowElementSize();
      ETG_TRACE_USR4(("DestinationMatchListHandler::onCourierMessage(ListDateProviderReqMsg(List ID : %d, Start index : %d, Window size : %d))", _listId, _startIndex, _windowElementSize));

      //Refresh the List
      tSharedPtrDataProvider dataProvider;
      ListDataProviderBuilder listBuilder(LIST_ID_NEWDESTINATION_MATCHLIST, DATA_CONTEXT_DESTINATION_LOADING_LIST_ITEM);
      listBuilder.AddItem(
         0UL, // - identifies the row
         0UL, // - unused
         DATA_CONTEXT_DESTINATION_LOADING_LIST_ITEM).AddData(LOADING);
      dataProvider = listBuilder.CreateDataProvider(_startIndex, NO_MATCHES_LIST_SIZE);
      if (dataProvider->listSize() > 0)
      {
         POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
      }

      _navMiddleware.requestAddressElements(_startIndex, _windowElementSize);
      isMsgConsumed = true;
   }

   return isMsgConsumed;
}


bool DestinationMatchListHandler::onCourierMessage(const ButtonReactionMsg& oMsg)
{
   ListProviderEventInfo info;
   bool isMsgConsumed = false;
   if (ListProviderEventInfo::GetItemIdentifierInfo(oMsg.GetSender(), info) && (info.getListId() == LIST_ID_NEWDESTINATION_MATCHLIST))
   {
      if (oMsg.GetEnReaction() == enRelease)
      {
         isMsgConsumed = true;

         unsigned int listIdx = (unsigned int)info.getHdlRow();

         const navmiddleware::LocationInfos& locationInfos = _navMiddleware.getAddressElements();
         const ::std::vector<navmiddleware::LocationInfos::LocationInfo>& locationInfoVector = locationInfos.getLocationInfos();

         listIdx = listIdx - _startIndex;

         if (listIdx < locationInfoVector.size())
         {
            navmiddleware::LocationInfos::LocationInfo locationInfo = locationInfoVector[listIdx];
            _navMiddleware.selectAddressElement(locationInfo);
            POST_MSG((COURIER_MESSAGE_NEW(AddressInputMatchSelectionDoneMsg)()));
         }
      }
   }

   return isMsgConsumed;
}


void DestinationMatchListHandler::onPropertyUpdateAddressElementChanged()
{
   ETG_TRACE_USR4(("DestinationMatchListHandler::onPropertyUpdateAddressElementChanged()"));

   // Updates the count of match list entries to HMI
   unsigned int totalItems = _navMiddleware.getAddressElements().getCountOfAllLocations();
   std::stringstream strMatchCount;
   strMatchCount << totalItems;

   if (0 != strcmp((*_addressInputNoOfMatches).mNoOfMatches.GetCString(), strMatchCount.str().c_str()))
   {
      (*_addressInputNoOfMatches).mNoOfMatches = strMatchCount.str().c_str();
      _addressInputNoOfMatches.MarkAllItemsModified();
      _addressInputNoOfMatches.SendUpdate(true);
   }

   // Send the match list to HMI
   tSharedPtrDataProvider dataProvider = getHierarchicalListDataProvider(_startIndex, _windowElementSize);
   if (dataProvider->listSize() > 0)
   {
      POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
   }
}


tSharedPtrDataProvider DestinationMatchListHandler::getHierarchicalListDataProvider(unsigned int startIndex, unsigned int windowElementSize) const
{
   ETG_TRACE_USR4(("DestinationMatchListHandler::getHierarchicalListDataProvider(), Start index : %d, Window size : %d", startIndex, windowElementSize));

   ListDataProviderBuilder listBuilder(LIST_ID_NEWDESTINATION_MATCHLIST, DATA_CONTEXT_DESTINATION_LOADING_LIST_ITEM);
   const navmiddleware::LocationInfos& locationInfos = _navMiddleware.getAddressElements();
   const ::std::vector<navmiddleware::LocationInfos::LocationInfo>& infos = locationInfos.getLocationInfos();
   ::std::vector<navmiddleware::LocationInfos::LocationInfo>::const_iterator info = infos.begin();
   unsigned int idx = startIndex;
   unsigned int actDestListSize = locationInfos.getCountOfAllLocations();

   ETG_TRACE_USR4(("DestinationMatchListHandler::getHierarchicalListDataProvider(), Current sort order : %d", locationInfos.getCurrentSortOrder()));

   if (actDestListSize > 0)
   {
      for (; (idx < startIndex + windowElementSize) && (info != infos.end()); ++info, ++idx)
      {
         listBuilder.AddItem(
            idx, // - identifies the row
            0UL, // - unused
            DATA_CONTEXT_DESTINATION_LOADING_LIST_ITEM).AddData(info->m_elementName.c_str());
      }
   }
   else if (0 == actDestListSize)
   {
      if(true == locationInfos.isResultTruncated())
      {
         listBuilder.AddItem(
            idx, // - identifies the row
            0UL, // - unused
            DATA_CONTEXT_DESTINATION_LOADING_LIST_ITEM).AddData(INCOMPLETE_MATCHES);
         actDestListSize = 1;
      }
      else if(false == locationInfos.areResultsOutstanding())
      {
         listBuilder.AddItem(
            idx, // - identifies the row
            0UL, // - unused
            DATA_CONTEXT_DESTINATION_LOADING_LIST_ITEM).AddData(NO_MATCHES);
         actDestListSize = 1;
      }
   }

   return listBuilder.CreateDataProvider(startIndex, actDestListSize);
}


#endif // HALL_TO_MDW_COM
