/**************************************************************************************
* @file         : DestinationLatLongSpellerHandler.cpp
* @author       : ECG-Ramesh Kesavan
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) -2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include "CgiExtensions/ImageLoader.h"
#include "DestinationLatLongSpellerHandler.h"
#include "../Common/Util/StringUtils.h"
#include <iomanip>
#include "../DataModel/AddressDetailedInfo/AddressDetailedInfo.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/DestinationLatLongSpellerHandler.cpp.trc.h"
#endif

using namespace FeatStd;

#if defined (HALL_TO_MDW_COM)

static const std::string DEGREE_SYMBOL = "°";
static const std::string MINUTE_SYMBOL = "'";
static const std::string SECOND_SYMBOL = "\"";
static const std::string DECIMAL_SYMBOL = ".";
static const std::string DELETE_CHAR = "0x232B"; // unicode for deleteChar
static const std::string DELETE_ALLCHARS = "0x2327"; // unicode for deleteAllChar
static const std::string PLUS = "(+)";
static const std::string MINUS = "(-)";


static const Candera::String NORTH = LANGUAGE_STRING(TextId_0x144C, "North");
static const Candera::String SOUTH = LANGUAGE_STRING(TextId_0x144F, "South");
static const Candera::String EAST = LANGUAGE_STRING(TextId_0x144D, "East");
static const Candera::String WEST = LANGUAGE_STRING(TextId_0x1450, "West");

static const char* EMPTYSTR = "";
static const char* IMG_ACTIVE_EDITFIELD = "AppHmi_NavigationModule#Images#N_LatLongInput_DDec#TouchArea_LatitudeDegrees_active";
static const char* IMG_NORMAL_EDITFIELD = "AppHmi_NavigationModule#Images#N_LatLongInput_DDec#TouchArea_LatitudeDegrees_normal";

static const char* IMG_DD_DECIMAL_ACTIVE_EDITFIELD = "AppHmi_NavigationModule#Images#N_LatLongInput_DDec#TouchArea_LatitudeDegreesDec_active";
static const char* IMG_DD_DECIMAL_NORMAL_EDITFIELD = "AppHmi_NavigationModule#Images#N_LatLongInput_DDec#TouchArea_LatitudeDegreesDec_normal";

static const char* IMG_DMS_DECIMAL_ACTIVE_EDITFIELD = "AppHmi_NavigationModule#Images#N_LatLongInput_DDec#TouchArea_LatitudeSecondsDec_active";
static const char* IMG_DMS_DECIMAL_NORMAL_EDITFIELD = "AppHmi_NavigationModule#Images#N_LatLongInput_DDec#TouchArea_LatitudeSecondsDec_normal";

// Degree Decimeal : 000.12345
static const int DD_DEGREES_MAXLENGTH = 3;
static const int DD_DECIMALS_MAXLENGTH = 5;

// Degree Minutes : 000.12.123
static const int DM_DEGREES_MAXLENGTH = 3;
static const int DM_MINUTES_MAXLENGTH = 2;
static const int DM_DECIMALS_MAXLENGTH = 3;

// Degree Minutes : 000.12.12.1
static const int DMS_DEGREES_MAXLENGTH = 3;
static const int DMS_MINUTES_MAXLENGTH = 2;
static const int DMS_SECONDS_MAXLENGTH = 2;
static const int DMS_DECIMALS_MAXLENGTH = 1;

static const int LATITUDE_VALUES = 0;
static const int LONGITUDE_VALUES = 1;

DestinationLatLongSpellerHandler::DestinationLatLongSpellerHandler(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore)
   : HMIModelBase(navMiddleware, infoStore)
   , _activeEditField(DD_LAT_DEG_EDIT_FIELD),
     _previousActiveEditField(DD_LAT_DEG_EDIT_FIELD),
     _latitudeCoordinatesDirection(NORTH_DIRECTION),
     _longitudeCoordinatesDirection(EAST_DIRECTION),
     _coordinatesDisplayType(DEGREES),
     _ignoreTextReceivedMsg(false),
     _ignoreEmptyString(false),
     _isSetLatLongButtonEnabled(false),
     _isGeocoordiantesInvalid(false),
     _currentInputStr(EMPTYSTR),
     _latitudeDegreesStr(EMPTYSTR),
     _longitudeDegreesStr(EMPTYSTR),
     _latitudeMinutesStr(EMPTYSTR),
     _longitudeMinutesStr(EMPTYSTR),
     _latitudeSecondsStr(EMPTYSTR),
     _longitudeSecondsStr(EMPTYSTR),
     _latitudeDecimalsStr(EMPTYSTR),
     _longitudeDecimalsStr(EMPTYSTR)
{
   (*_latlongEditFieldBGImage).mLatitudeDegreesBGBitmap = ImageLoader::getAssetBitmapImage(IMG_ACTIVE_EDITFIELD);
   _latlongEditFieldBGImage.MarkItemModified(ItemKey::LatLongEditFieldBGImageItem::LatitudeDegreesBGBitmapItem);
   _latlongEditFieldBGImage.SendUpdate();
}


DestinationLatLongSpellerHandler::~DestinationLatLongSpellerHandler()
{
}


void DestinationLatLongSpellerHandler::initialize()
{
   ETG_TRACE_USR1(("DestinationLatLongSpellerHandler::initialize"));
}


void DestinationLatLongSpellerHandler::deinitialize()
{
   ETG_TRACE_USR1(("DestinationLatLongSpellerHandler::deinitialize"));
}


void DestinationLatLongSpellerHandler::setLatLongSpellerPressedKeyInfoData(const char* inputStr)
{
   ETG_TRACE_USR1(("DestinationLatLongSpellerHandler::setLatLongSpellerPressedKeyInfoData"));
   (*_latLongSpellerPressedKeyInfoData).mPressedKey_LatLongEditField = inputStr;

   _latLongSpellerPressedKeyInfoData.MarkItemModified(ItemKey::LatLongSpellerPressedKeyInfo::PressedKey_LatLongEditFieldItem);
   _latLongSpellerPressedKeyInfoData.SendUpdate();
}


void DestinationLatLongSpellerHandler::showInvalidCoordPopup()
{
   _isGeocoordiantesInvalid = true; // set to true so that on re-entry into the scene, co-ord values are retained
   (*_enableSetLatLongButton).mIsPopUpVisible = true;
   _enableSetLatLongButton.MarkItemModified(ItemKey::EnableSetLatLongButton::IsPopUpVisibleItem);
   _enableSetLatLongButton.SendUpdate();

   const Candera::String DATA_CONTEXT_INVALID_COORD_LINE1 = LANGUAGE_STRING(TextId_0x0F1C, "The co-ordinates are invalid. Please enter");
   const Candera::String DATA_CONTEXT_INVALID_COORD_LINE2 = LANGUAGE_STRING(TextId_0x0F1D, "valid co-ordinates");
   (*_multiLineSingleBtnPopupText).mTextLine1 = DATA_CONTEXT_INVALID_COORD_LINE1;
   (*_multiLineSingleBtnPopupText).mTextLine2 = DATA_CONTEXT_INVALID_COORD_LINE2;
   _multiLineSingleBtnPopupText.SendUpdate();

   POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_LatLong_Invalid"))));
}


bool DestinationLatLongSpellerHandler::onCourierMessage(const ShowCoordinatesInMapReqMsg& oMsg)
{
   COURIER_UNUSED(oMsg);

   ETG_TRACE_USR1(("DestinationLatLongSpellerHandler::onCourierMessage(ShowCoordinatesInMapReqMsg)"));

   // check for entered latitude and longitude value
   processEnteredCoordinate();

   return true;
}


bool DestinationLatLongSpellerHandler::onCourierBindingItemChange_LatLongSpellerPressedKeyInfoItem(const Courier::Request& request)
{
   ETG_TRACE_USR1(("DestinationLatLongSpellerHandler::onCourierBindingItemChange_LatLongSpellerPressedKeyInfoItem"));
   _latLongSpellerPressedKeyInfoData.SetValue(request.ItemKey(), request.GetItemValue());
   ETG_TRACE_USR4(("Pressed key %s ", (*_latLongSpellerPressedKeyInfoData).mPressedKey_LatLongEditField.GetCString()));
   _latLongSpellerPressedKeyInfoData.MarkAllItemsModified();
   _latLongSpellerPressedKeyInfoData.SendUpdate(true);

   return true;
}


bool DestinationLatLongSpellerHandler::onCourierBindingItemChange_LatLongSpellerTextReceivedItem(const Courier::Request& request)
{
   ETG_TRACE_USR1(("DestinationLatLongSpellerHandler::onCourierBindingItemChange_LatLongSpellerTextReceivedItem"));
   _latLongSpellerTextReceivedInfoData.SetValue(request.ItemKey(), request.GetItemValue());

   _currentInputStr = (*_latLongSpellerTextReceivedInfoData).mTextReceive_LatLongEditField.GetCString();

   /* Below if block is NOT executed when
      1. Text received is a pre-existing visible text which was updated from the code
      2. Text received is an empty string which was updated from the code
      */
   if ((false == _ignoreTextReceivedMsg) && (false == _ignoreEmptyString))
   {
      // while switching active editfields
      switch (_coordinatesDisplayType)
      {
         case DEGREES:
         {
            updateSpellerPressedKeyInfoForDDFormat();
            break;
         }
         case DEGREE_MINS:
         {
            updateSpellerPressedKeyInfoForDMFormat();
            break;
         }
         case DEGREES_MINS_SEC:
         {
            updateSpellerPressedKeyInfoForDMSFormat();
            break;
         }
         default:
            break;
      }
      checkForInputsInEditFields();
   }
   else
   {
      // resetting the ignore flags
      if (true == _ignoreEmptyString)
      {
         _ignoreEmptyString = false;
      }
      else
      {
         if (true == _ignoreTextReceivedMsg)
         {
            _ignoreTextReceivedMsg = false;
         }
      }
   } // End of if-else

   return true;
}


void DestinationLatLongSpellerHandler::updateSpellerPressedKeyInfoForDDFormat()
{
   ETG_TRACE_USR1(("DestinationLatLongSpellerHandler::updateSpellerPressedKeyInfoForDDFormat"));

   bool isStrLengthExceded = false;

   Candera::UInt processing_strlen = _currentInputStr.size();

   switch (_activeEditField)
   {
      case DD_LAT_DEG_EDIT_FIELD:
      {
         if (processing_strlen <= DD_DEGREES_MAXLENGTH)
         {
            (*_latlongDataValues).mLatDegreesInputValues = _currentInputStr.c_str();
            _latlongDataValues.MarkItemModified(ItemKey::LatLongDataValues::LatDegreesInputValuesItem);

            if (DD_DEGREES_MAXLENGTH == processing_strlen)
            {
               // Move focus to next node
               _previousActiveEditField = DD_LAT_DEG_EDIT_FIELD;
               _activeEditField = DD_LAT_DECIMAL_EDIT_FIELD;
               changeActiveEditFieldText();
               changeActiveEditFieldBGImage();
            }
         }
         else
         {
            isStrLengthExceded = true;
         }
         break;
      }
      case DD_LAT_DECIMAL_EDIT_FIELD:
      {
         if (processing_strlen <= DD_DECIMALS_MAXLENGTH)
         {
            (*_latlongDataValues).mLatDecimalsInputValues = _currentInputStr.c_str();
            _latlongDataValues.MarkItemModified(ItemKey::LatLongDataValues::LatDecimalsInputValuesItem);
         }
         else
         {
            isStrLengthExceded = true;
         }
         break;
      }
      case DD_LONG_DEG_EDIT_FIELD:
      {
         if (processing_strlen <= DD_DEGREES_MAXLENGTH)
         {
            (*_latlongDataValues).mLongDegreesInputValues = _currentInputStr.c_str();
            _latlongDataValues.MarkItemModified(ItemKey::LatLongDataValues::LongDegreesInputValuesItem);

            if (DD_DEGREES_MAXLENGTH == processing_strlen)
            {
               // Move focus to next node
               _previousActiveEditField = DD_LONG_DEG_EDIT_FIELD;
               _activeEditField = DD_LONG_DECIMAL_EDIT_FIELD;
               changeActiveEditFieldText();
               changeActiveEditFieldBGImage();
            }
         }
         else
         {
            isStrLengthExceded = true;
         }
         break;
      }
      case DD_LONG_DECIMAL_EDIT_FIELD:
      {
         if (processing_strlen <= DD_DECIMALS_MAXLENGTH)
         {
            (*_latlongDataValues).mLongDecimalsInputValues = _currentInputStr.c_str();
            _latlongDataValues.MarkItemModified(ItemKey::LatLongDataValues::LongDecimalsInputValuesItem);
         }
         else
         {
            isStrLengthExceded = true;
         }
         break;
      }
      default:
         break;
   }

   if (_latlongDataValues.HasModifiedItems())
   {
      if (!(_latlongDataValues.SendUpdate()))
      {
         ETG_TRACE_ERR(("_latlongDataValues update failed!"));
      }
   }

   if (isStrLengthExceded)
   {
      // use case :  The editfield Max character length is 8 character. But, if user starts typing input from last
      //             segment (i.e. which has limited string length say 3). the additional key presses are considerd by the editfield.
      //             Hence, once the max char for this format is reached we are removeing the char in the editfiel widget by sending delete char.
      //             If this segment is removed, only after 5 delete key press, the chracter in the screen will get deleted.
      setLatLongSpellerPressedKeyInfoData(DELETE_CHAR.c_str());

      // set to true to ignore the text update after deleting the character which exceeded the bounds
      _ignoreTextReceivedMsg = true;
   }
}


void DestinationLatLongSpellerHandler::updateSpellerPressedKeyInfoForDMFormat()
{
   ETG_TRACE_USR1(("DestinationLatLongSpellerHandler::updateSpellerPressedKeyInfoForDMFormat"));

   bool isStrLengthExceded = false;
   std::string processString;
   processString = _currentInputStr;
   Candera::UInt processing_strlen = processString.size();

   switch (_activeEditField)
   {
      case DM_LAT_DEG_EDIT_FIELD:
      {
         if (processing_strlen <= DM_DEGREES_MAXLENGTH)
         {
            (*_latlongDataValues).mLatDegreesInputValues = _currentInputStr.c_str();
            _latlongDataValues.MarkItemModified(ItemKey::LatLongDataValues::LatDegreesInputValuesItem);

            if (DM_DEGREES_MAXLENGTH == processing_strlen)
            {
               // Move focus to next node
               _previousActiveEditField = DM_LAT_DEG_EDIT_FIELD;
               _activeEditField = DM_LAT_MINS_EDIT_FIELD;
               changeActiveEditFieldText();
               changeActiveEditFieldBGImage();
            }
         }
         else
         {
            isStrLengthExceded = true;
         }
         break;
      }
      case DM_LAT_MINS_EDIT_FIELD:
      {
         if (processing_strlen <= DM_MINUTES_MAXLENGTH)
         {
            (*_latlongDataValues).mLatMinutesInputValues = _currentInputStr.c_str();
            _latlongDataValues.MarkItemModified(ItemKey::LatLongDataValues::LatMinutesInputValuesItem);

            if (DM_MINUTES_MAXLENGTH == processing_strlen)
            {
               // Move focus to next node
               _previousActiveEditField = DM_LAT_MINS_EDIT_FIELD;
               _activeEditField = DM_LAT_MINS_DECIMAL_EDIT_FIELD;
               changeActiveEditFieldText();
               changeActiveEditFieldBGImage();
            }
         }
         else
         {
            isStrLengthExceded = true;
         }
         break;
      }
      case DM_LAT_MINS_DECIMAL_EDIT_FIELD:
      {
         if (processing_strlen <= DM_DECIMALS_MAXLENGTH)
         {
            (*_latlongDataValues).mLatDecimalsInputValues = _currentInputStr.c_str();
            _latlongDataValues.MarkItemModified(ItemKey::LatLongDataValues::LatDecimalsInputValuesItem);
         }
         else
         {
            isStrLengthExceded = true;
         }
         break;
      }
      case DM_LONG_DEG_EDIT_FIELD:
      {
         if (processing_strlen <= DM_DECIMALS_MAXLENGTH)
         {
            (*_latlongDataValues).mLongDegreesInputValues = _currentInputStr.c_str();
            _latlongDataValues.MarkItemModified(ItemKey::LatLongDataValues::LongDegreesInputValuesItem);

            if (DM_DECIMALS_MAXLENGTH == processing_strlen)
            {
               // Move focus to next node
               _previousActiveEditField = DM_LONG_DEG_EDIT_FIELD;
               _activeEditField = DM_LONG_MINS_EDIT_FIELD;
               changeActiveEditFieldText();
               changeActiveEditFieldBGImage();
            }
         }
         else
         {
            isStrLengthExceded = true;
         }
         break;
      }
      case DM_LONG_MINS_EDIT_FIELD:
      {
         if (processing_strlen <= DM_MINUTES_MAXLENGTH)
         {
            (*_latlongDataValues).mLongMinutesInputValues = _currentInputStr.c_str();
            _latlongDataValues.MarkItemModified(ItemKey::LatLongDataValues::LongMinutesInputValuesItem);

            if (DM_MINUTES_MAXLENGTH == processing_strlen)
            {
               // Move focus to next node
               _previousActiveEditField = DM_LONG_MINS_EDIT_FIELD;
               _activeEditField = DM_LONG_MINS_DECIMAL_EDIT_FIELD;
               changeActiveEditFieldText();
               changeActiveEditFieldBGImage();
            }
         }
         else
         {
            isStrLengthExceded = true;
         }
         break;
      }
      case DM_LONG_MINS_DECIMAL_EDIT_FIELD:
      {
         if (processing_strlen <= DM_DECIMALS_MAXLENGTH)
         {
            (*_latlongDataValues).mLongDecimalsInputValues = _currentInputStr.c_str();
            _latlongDataValues.MarkItemModified(ItemKey::LatLongDataValues::LongDecimalsInputValuesItem);
         }
         else
         {
            isStrLengthExceded = true;
         }
         break;
      }
      default:
         break;
   }

   if (_latlongDataValues.HasModifiedItems())
   {
      if (!(_latlongDataValues.SendUpdate()))
      {
         ETG_TRACE_ERR(("_latlongDataValues update failed!"));
      }
   }

   if (isStrLengthExceded)
   {
      // use case :  The editfield Max character length is 8 character. But, if user starts typing input from last
      //             segment (i.e. which has limited string length say 3). the additional key presses are considerd by the editfield.
      //             Hence, once the max char for this format is reached we are removeing the char in the editfiel widget by sending delete char.
      //             If this segment is removed, only after 5 delete key press, the chracter in the screen will get deleted.
      setLatLongSpellerPressedKeyInfoData(DELETE_CHAR.c_str());

      // set to true to ignore the text update after deleting the character which exceeded the bounds
      _ignoreTextReceivedMsg = true;
   }
}


void DestinationLatLongSpellerHandler::updateSpellerPressedKeyInfoForDMSFormat()
{
   ETG_TRACE_USR1(("DestinationLatLongSpellerHandler::updateSpellerPressedKeyInfoForDMSFormat"));

   bool isStrLengthExceded = false;

   if ((_activeEditField >= DMS_LAT_DEG_EDIT_FIELD) && (_activeEditField <= DMS_LAT_SECS_DECIMAL_EDIT_FIELD))
   {
      isStrLengthExceded = HandleLatitudeValuesForDMSFormat();
   }
   else if ((_activeEditField >= DMS_LONG_DEG_EDIT_FIELD) && (_activeEditField <= DMS_LONG_SECS_DECIMAL_EDIT_FIELD))
   {
      isStrLengthExceded = HandleLongitudeValuesForDMSFormat();
   }

   if (_latlongDataValues.HasModifiedItems())
   {
      if (!(_latlongDataValues.SendUpdate()))
      {
         ETG_TRACE_ERR(("_latlongDataValues update failed!"));
      }
   }

   if (isStrLengthExceded)
   {
      // use case :  The editfield Max character length is 8 character. But, if user starts typing input from last
      //             segment (i.e. which has limited string length say 3). the additional key presses are considerd by the editfield.
      //             Hence, once the max char for this format is reached we are removeing the char in the editfiel widget by sending delete char.
      //             If this segment is removed, only after 5 delete key press, the chracter in the screen will get deleted.
      setLatLongSpellerPressedKeyInfoData(DELETE_CHAR.c_str());

      // set to true to ignore the text update after deleting the character which exceeded the bounds
      _ignoreTextReceivedMsg = true;
   }
}


bool DestinationLatLongSpellerHandler::HandleLatitudeValuesForDMSFormat()
{
   bool isStrLengthExceded = false;
   std::string processString;
   processString = _currentInputStr;
   Candera::UInt processing_strlen = processString.size();

   switch (_activeEditField)
   {
      case DMS_LAT_DEG_EDIT_FIELD:
      {
         if (processing_strlen <= DMS_DEGREES_MAXLENGTH)
         {
            (*_latlongDataValues).mLatDegreesInputValues = _currentInputStr.c_str();
            _latlongDataValues.MarkItemModified(ItemKey::LatLongDataValues::LatDegreesInputValuesItem);

            if (DMS_DEGREES_MAXLENGTH == processing_strlen)
            {
               // Move focus to next node
               _previousActiveEditField = DMS_LAT_DEG_EDIT_FIELD;
               _activeEditField = DMS_LAT_MINS_EDIT_FIELD;
               changeActiveEditFieldText();
               changeActiveEditFieldBGImage();
            }
         }
         else
         {
            isStrLengthExceded = true;
         }
         break;
      }
      case DMS_LAT_MINS_EDIT_FIELD:
      {
         if (processing_strlen <= DMS_MINUTES_MAXLENGTH)
         {
            (*_latlongDataValues).mLatMinutesInputValues = _currentInputStr.c_str();
            _latlongDataValues.MarkItemModified(ItemKey::LatLongDataValues::LatMinutesInputValuesItem);

            if (DMS_MINUTES_MAXLENGTH == processing_strlen)
            {
               // Move focus to next node
               _previousActiveEditField = DMS_LAT_MINS_EDIT_FIELD;
               _activeEditField = DMS_LAT_SECS_EDIT_FIELD;
               changeActiveEditFieldText();
               changeActiveEditFieldBGImage();
            }
         }
         else
         {
            isStrLengthExceded = true;
         }
         break;
      }
      case DMS_LAT_SECS_EDIT_FIELD:
      {
         if (processing_strlen <= DMS_SECONDS_MAXLENGTH)
         {
            (*_latlongDataValues).mLatSecondsInputValues = _currentInputStr.c_str();
            _latlongDataValues.MarkItemModified(ItemKey::LatLongDataValues::LatSecondsInputValuesItem);

            if (DMS_SECONDS_MAXLENGTH == processing_strlen)
            {
               // Move focus to next node
               _previousActiveEditField = DMS_LAT_SECS_EDIT_FIELD;
               _activeEditField = DMS_LAT_SECS_DECIMAL_EDIT_FIELD;
               changeActiveEditFieldText();
               changeActiveEditFieldBGImage();
            }
         }
         else
         {
            isStrLengthExceded = true;
         }
         break;
      }
      case DMS_LAT_SECS_DECIMAL_EDIT_FIELD:
      {
         if (processing_strlen <= DMS_DECIMALS_MAXLENGTH)
         {
            (*_latlongDataValues).mLatDecimalsInputValues = _currentInputStr.c_str();
            _latlongDataValues.MarkItemModified(ItemKey::LatLongDataValues::LatDecimalsInputValuesItem);
         }
         else
         {
            isStrLengthExceded = true;
         }
         break;
      }
      default:
         break;
   }

   return isStrLengthExceded;
}


bool DestinationLatLongSpellerHandler::HandleLongitudeValuesForDMSFormat()
{
   bool isStrLengthExceded = false;
   std::string processString;
   processString = _currentInputStr;
   Candera::UInt processing_strlen = processString.size();

   switch (_activeEditField)
   {
      case DMS_LONG_DEG_EDIT_FIELD:
      {
         if (processing_strlen <= DMS_DEGREES_MAXLENGTH)
         {
            (*_latlongDataValues).mLongDegreesInputValues = _currentInputStr.c_str();
            _latlongDataValues.MarkItemModified(ItemKey::LatLongDataValues::LongDegreesInputValuesItem);

            if (DMS_DEGREES_MAXLENGTH == processing_strlen)
            {
               // Move focus to next node
               _previousActiveEditField = DMS_LONG_DEG_EDIT_FIELD;
               _activeEditField = DMS_LONG_MINS_EDIT_FIELD;
               changeActiveEditFieldText();
               changeActiveEditFieldBGImage();
            }
         }
         else
         {
            isStrLengthExceded = true;
         }
         break;
      }
      case DMS_LONG_MINS_EDIT_FIELD:
      {
         if (processing_strlen <= DMS_MINUTES_MAXLENGTH)
         {
            (*_latlongDataValues).mLongMinutesInputValues = _currentInputStr.c_str();
            _latlongDataValues.MarkItemModified(ItemKey::LatLongDataValues::LongMinutesInputValuesItem);

            if (DMS_MINUTES_MAXLENGTH == processing_strlen)
            {
               // Move focus to next node
               _previousActiveEditField = DMS_LONG_MINS_EDIT_FIELD;
               _activeEditField = DMS_LONG_SECS_EDIT_FIELD;
               changeActiveEditFieldText();
               changeActiveEditFieldBGImage();
            }
         }
         else
         {
            isStrLengthExceded = true;
         }
         break;
      }
      case DMS_LONG_SECS_EDIT_FIELD:
      {
         if (processing_strlen <= DMS_SECONDS_MAXLENGTH)
         {
            (*_latlongDataValues).mLongSecondsInputValues = _currentInputStr.c_str();
            _latlongDataValues.MarkItemModified(ItemKey::LatLongDataValues::LongSecondsInputValuesItem);

            if (DMS_SECONDS_MAXLENGTH == processing_strlen)
            {
               // Move focus to next node
               _previousActiveEditField = DMS_LONG_SECS_EDIT_FIELD;
               _activeEditField = DMS_LONG_SECS_DECIMAL_EDIT_FIELD;
               changeActiveEditFieldText();
               changeActiveEditFieldBGImage();
            }
         }
         else
         {
            isStrLengthExceded = true;
         }
         break;
      }
      case DMS_LONG_SECS_DECIMAL_EDIT_FIELD:
      {
         if (processing_strlen <= DMS_DECIMALS_MAXLENGTH)
         {
            (*_latlongDataValues).mLongDecimalsInputValues = _currentInputStr.c_str();
            _latlongDataValues.MarkItemModified(ItemKey::LatLongDataValues::LongDecimalsInputValuesItem);
         }
         else
         {
            isStrLengthExceded = true;
         }
         break;
      }
      default:
         break;
   } // End of switch-case

   return isStrLengthExceded;
}


bool DestinationLatLongSpellerHandler::onCourierMessage(const ToggleSouthNorthReqMsg& oMsg)
{
   COURIER_UNUSED(oMsg);

   ETG_TRACE_USR1(("DestinationLatLongSpellerHandler::onCourierMessage(ToggleSouthNorthRegMsg)"));

   Candera::String northSouthEditFieldTxt;
   Candera::String northSouthButtonTxt;
   std::string northString = NORTH.GetCString();
   std::string southString = SOUTH.GetCString();
   Candera::String northEditFieldText = (northString + PLUS).c_str();
   Candera::String southEditFieldText = (southString + MINUS).c_str();
   if (SOUTH_DIRECTION == _latitudeCoordinatesDirection)
   {
      northSouthEditFieldTxt = northEditFieldText;
      northSouthButtonTxt = SOUTH;

      _latitudeCoordinatesDirection = NORTH_DIRECTION;
   }
   else
   {
      northSouthEditFieldTxt = southEditFieldText;
      northSouthButtonTxt = NORTH;
      _latitudeCoordinatesDirection = SOUTH_DIRECTION;
   }

   _infoStore.setLatitudeCoordinatesDirection(_latitudeCoordinatesDirection);
   if ((*_latLongCoordinatesDirection).mNorthSouthEditFieldTxt != northSouthEditFieldTxt)
   {
      (*_latLongCoordinatesDirection).mNorthSouthEditFieldTxt = northSouthEditFieldTxt;
      _latLongCoordinatesDirection.MarkItemModified(ItemKey::LatLongCoordinatesDirection::NorthSouthEditFieldTxtItem);
   }

   if ((*_latLongCoordinatesDirection).mNorthSouthButtonTxt != northSouthButtonTxt)
   {
      (*_latLongCoordinatesDirection).mNorthSouthButtonTxt = northSouthButtonTxt;
      _latLongCoordinatesDirection.MarkItemModified(ItemKey::LatLongCoordinatesDirection::NorthSouthButtonTxtItem);
   }

   sendLatLongCoordinatesDirectionUpdate();

   return true;
}


bool DestinationLatLongSpellerHandler::onCourierMessage(const ToggleEastWestReqMsg& oMsg)
{
   COURIER_UNUSED(oMsg);

   ETG_TRACE_USR1(("DestinationLatLongSpellerHandler::onCourierMessage(ToggleEastWestReqMsg)"));

   Candera::String eastWestEditFieldTxt;
   Candera::String eastWestButtonTxt;
   std::string eastString = EAST.GetCString();
   std::string westString = WEST.GetCString();
   Candera::String eastEditFieldText = (eastString + PLUS).c_str();
   Candera::String westEditFieldText = (westString + MINUS).c_str();
   if (EAST_DIRECTION == _longitudeCoordinatesDirection)
   {
      eastWestEditFieldTxt = westEditFieldText;
      eastWestButtonTxt = EAST;
      _longitudeCoordinatesDirection = WEST_DIRECTION;
   }
   else
   {
      eastWestEditFieldTxt = eastEditFieldText;
      eastWestButtonTxt = WEST;
      _longitudeCoordinatesDirection = EAST_DIRECTION;
   }

   _infoStore.setLongitudeCoordinatesDirection(_longitudeCoordinatesDirection);
   if ((*_latLongCoordinatesDirection).mEastWestEditFieldTxt != eastWestEditFieldTxt)
   {
      (*_latLongCoordinatesDirection).mEastWestEditFieldTxt = eastWestEditFieldTxt;
      _latLongCoordinatesDirection.MarkItemModified(ItemKey::LatLongCoordinatesDirection::EastWestEditFieldTxtItem);
   }

   if ((*_latLongCoordinatesDirection).mEastWestButtonTxt != eastWestButtonTxt)
   {
      (*_latLongCoordinatesDirection).mEastWestButtonTxt = eastWestButtonTxt;
      _latLongCoordinatesDirection.MarkItemModified(ItemKey::LatLongCoordinatesDirection::EastWestButtonTxtItem);
   }

   sendLatLongCoordinatesDirectionUpdate();

   return true;
}


void DestinationLatLongSpellerHandler::initializeLatLongSetting()
{
   ETG_TRACE_USR1(("DestinationLatLongSpellerHandler::initializeLatLongSetting(ToggleEastWestReqMsg)"));
   if ((_infoStore.isNarRegion()) || (UK == _infoStore.getRegionType()))
   {
      // North & West
      _latitudeCoordinatesDirection = NORTH_DIRECTION;
      _longitudeCoordinatesDirection = WEST_DIRECTION;
      _infoStore.setLatitudeCoordinatesDirection(_latitudeCoordinatesDirection);
      _infoStore.setLongitudeCoordinatesDirection(_longitudeCoordinatesDirection);
      ETG_TRACE_USR1(("DestinationLatLongSpellerHandler::InitializeLatLongSetting _latitudeCoordinatesDirection(%d)_longitudeCoordinatesDirection(%d)", _latitudeCoordinatesDirection, _longitudeCoordinatesDirection));
   }
   else if (_infoStore.isEurRegion())
   {
      // North & East
      _latitudeCoordinatesDirection = NORTH_DIRECTION;
      _longitudeCoordinatesDirection = EAST_DIRECTION;
      _infoStore.setLatitudeCoordinatesDirection(_latitudeCoordinatesDirection);
      _infoStore.setLongitudeCoordinatesDirection(_longitudeCoordinatesDirection);
      ETG_TRACE_USR1(("DestinationLatLongSpellerHandler::InitializeLatLongSetting _latitudeCoordinatesDirection(%d)_longitudeCoordinatesDirection(%d)", _latitudeCoordinatesDirection, _longitudeCoordinatesDirection));
   }
   else
   {
      ETG_TRACE_USR4(("ERROR - UNKNOW REGION: (%d)", _infoStore.getRegionType()));
   }
}


void DestinationLatLongSpellerHandler::resetLatLongDataValues()
{
   ETG_TRACE_USR1(("DestinationLatLongSpellerHandler::resetLatLongDataValues"));

   if (DEGREES == _coordinatesDisplayType)
   {
      _latitudeDegreesStr = EMPTYSTR;
      _latitudeDecimalsStr = EMPTYSTR;

      _longitudeDegreesStr = EMPTYSTR;
      _longitudeDecimalsStr = EMPTYSTR;

      _activeEditField = DD_LAT_DEG_EDIT_FIELD;
   }
   else if (DEGREE_MINS == _coordinatesDisplayType)
   {
      _latitudeDegreesStr = EMPTYSTR;
      _latitudeMinutesStr = EMPTYSTR;
      _latitudeDecimalsStr = EMPTYSTR;

      _longitudeDegreesStr = EMPTYSTR;
      _longitudeMinutesStr = EMPTYSTR;
      _longitudeDecimalsStr = EMPTYSTR;

      _activeEditField = DM_LAT_DEG_EDIT_FIELD;
   }
   else if (DEGREES_MINS_SEC == _coordinatesDisplayType)
   {
      _latitudeDegreesStr = EMPTYSTR;
      _latitudeMinutesStr = EMPTYSTR;
      _latitudeSecondsStr = EMPTYSTR;
      _latitudeDecimalsStr = EMPTYSTR;

      _longitudeDegreesStr = EMPTYSTR;
      _longitudeMinutesStr = EMPTYSTR;
      _longitudeSecondsStr = EMPTYSTR;
      _longitudeDecimalsStr = EMPTYSTR;

      _activeEditField = DMS_LAT_DEG_EDIT_FIELD;
   }
   else
   {
      // unknown co-ordinate display type
      ETG_TRACE_USR4(("ERROR - INVALID Co-ordiante display format"));
   }

   (*_latlongDataValues).mLatDegreesInputValues = _latitudeDegreesStr.c_str();
   (*_latlongDataValues).mLatMinutesInputValues = _latitudeMinutesStr.c_str();
   (*_latlongDataValues).mLatSecondsInputValues = _latitudeSecondsStr.c_str();
   (*_latlongDataValues).mLatDecimalsInputValues = _latitudeDecimalsStr.c_str();

   (*_latlongDataValues).mLongDegreesInputValues = _longitudeDegreesStr.c_str();
   (*_latlongDataValues).mLongMinutesInputValues = _longitudeMinutesStr.c_str();
   (*_latlongDataValues).mLongSecondsInputValues = _longitudeSecondsStr.c_str();
   (*_latlongDataValues).mLongDecimalsInputValues = _longitudeDecimalsStr.c_str();

   _latlongDataValues.MarkAllItemsModified();
   _latlongDataValues.SendUpdate();
}


bool DestinationLatLongSpellerHandler::onCourierMessage(const InitializeLatLongSpellerEntryReqMsg& oMsg)
{
   COURIER_UNUSED(oMsg);

   ETG_TRACE_USR1(("DestinationLatLongSpellerHandler::onCourierMessage(InitializeLatLongSpellerEntryReqMsg)"));

   if (!_isGeocoordiantesInvalid)
   {
      // initialize the default value to edit field and lat/long directions
      _activeEditField = DD_LAT_DEG_EDIT_FIELD;
      _latitudeCoordinatesDirection = _infoStore.getLatitudeCoordinatesDirection();
      _longitudeCoordinatesDirection = _infoStore.getLongitudeCoordinatesDirection();

      if ((UNDEFINED_LAT_DIRECTION == _latitudeCoordinatesDirection) ||
            (UNDEFINED_LONG_DIRECTION == _longitudeCoordinatesDirection))
      {
         // Need to be initialize base on region
         initializeLatLongSetting();
      }

      // update the text node in the scene
      Candera::String northSouthEditFieldTxt;
      Candera::String northSouthButtonTxt;
      Candera::String eastWestEditFieldTxt;
      Candera::String eastWestButtonTxt;
      std::string northString = NORTH.GetCString();
      std::string southString = SOUTH.GetCString();
      std::string eastString = EAST.GetCString();
      std::string westString = WEST.GetCString();
      Candera::String northEditFieldText = (northString + PLUS).c_str();
      Candera::String southEditFieldText = (southString + MINUS).c_str();
      Candera::String eastEditFieldText  = (eastString + PLUS).c_str();
      Candera::String westEditFieldText  = (westString + MINUS).c_str();

      // update latitude Text labels
      if (NORTH_DIRECTION == _latitudeCoordinatesDirection)
      {
         northSouthEditFieldTxt = northEditFieldText;
         northSouthButtonTxt = SOUTH;
      }
      else
      {
         northSouthEditFieldTxt = southEditFieldText;
         northSouthButtonTxt = NORTH;
      }

      // update longitude text labels
      if (EAST_DIRECTION == _longitudeCoordinatesDirection)
      {
         eastWestEditFieldTxt = eastEditFieldText;
         eastWestButtonTxt = WEST;
      }
      else
      {
         eastWestEditFieldTxt = westEditFieldText;
         eastWestButtonTxt = EAST;
      }

      // Data Binding update for Text label
      (*_latLongCoordinatesDirection).mNorthSouthEditFieldTxt = northSouthEditFieldTxt;
      (*_latLongCoordinatesDirection).mEastWestEditFieldTxt = eastWestEditFieldTxt;

      // Data Binding update for Button label
      (*_latLongCoordinatesDirection).mNorthSouthButtonTxt = northSouthButtonTxt;
      (*_latLongCoordinatesDirection).mEastWestButtonTxt = eastWestButtonTxt;

      _latLongCoordinatesDirection.MarkAllItemsModified();
      _latLongCoordinatesDirection.SendUpdate();

      // update the Co-ordinate Text boxes with predefined Zero
      (*_latLongCoordinatesFormatIndex).mLatLongFormatIndex = static_cast<Candera::UInt8>(_coordinatesDisplayType);

      _latLongCoordinatesFormatIndex.MarkItemModified(ItemKey::LatLongCoordinatesFormat::LatLongFormatIndexItem);
      _latLongCoordinatesFormatIndex.SendUpdate();

      // Fetch the Coordinate Display Type from Infostore
      _coordinatesDisplayType = _infoStore.getCoordinatesDisplayType();

      // Update to switch widget in HMI according to Coordinate Display Type
      (*_latLongCoordinatesFormatIndex).mLatLongFormatIndex = static_cast<Candera::UInt8>(_coordinatesDisplayType);
      _latLongCoordinatesFormatIndex.MarkItemModified(ItemKey::LatLongCoordinatesFormat::LatLongFormatIndexItem);
      _latLongCoordinatesFormatIndex.SendUpdate();

      clearAllEditFields();
      changeActiveEditFieldBGImage(true);
   }
   else
   {
      _isGeocoordiantesInvalid = false;
      /* Below function is invoked to update the text corresponding to the last active field to the widget */
      changeActiveEditFieldText();
   }
   return true;
}


void DestinationLatLongSpellerHandler::sendLatLongCoordinatesDirectionUpdate()
{
   ETG_TRACE_USR1(("DestinationLatLongSpellerHandler::sendLatLongCoordinatesDirectionUpdate"));

   if (_latLongCoordinatesDirection.HasModifiedItems())
   {
      if (!(_latLongCoordinatesDirection.SendUpdate()))
      {
         ETG_TRACE_ERR(("_latLongCoordinatesDirection update failed!"));
      }
   }
}


bool DestinationLatLongSpellerHandler::onCourierMessage(const ChangeCoordinateFormatReqMsg& oMsg)
{
   COURIER_UNUSED(oMsg);
   changeDisplayFormat();
   return true;
}


bool DestinationLatLongSpellerHandler::onCourierMessage(const ChangeActiveEditFieldReqMsg& oMsg)
{
   COURIER_UNUSED(oMsg);

   _previousActiveEditField = _activeEditField;
   enLatLongInputDataType currentDataType = oMsg.GetActiveInputDataType();

   switch (_coordinatesDisplayType)
   {
      case DEGREES:
         if (LAT_DEGREES_DATA == currentDataType)
         {
            _activeEditField = DD_LAT_DEG_EDIT_FIELD;
         }
         else if (LAT_DECIMALS_DATA == currentDataType)
         {
            _activeEditField = DD_LAT_DECIMAL_EDIT_FIELD;
         }
         else if (LONG_DEGREES_DATA == currentDataType)
         {
            _activeEditField = DD_LONG_DEG_EDIT_FIELD;
         }
         else if (LONG_DECIMALS_DATA == currentDataType)
         {
            _activeEditField = DD_LONG_DECIMAL_EDIT_FIELD;
         }
         else
         {
            // Invalid change in active edit field
         }
         break;
      case DEGREE_MINS:
         UpdateActiveEditFieldForDMFormat(currentDataType);
         break;
      case DEGREES_MINS_SEC:
         UpdateActiveEditFieldForDMSFormat(currentDataType);
         break;
      default:
         break;
   } // End of switch-case

   if (_previousActiveEditField != _activeEditField)
   {
      // Update the text corresponding to the newly active editfield
      changeActiveEditFieldText();
      // Update the BG images of the editfield
      changeActiveEditFieldBGImage();
   }

   return true;
}


void DestinationLatLongSpellerHandler::UpdateActiveEditFieldForDMFormat(const enLatLongInputDataType currentDataType)
{
   if (LAT_DEGREES_DATA == currentDataType)
   {
      _activeEditField = DM_LAT_DEG_EDIT_FIELD;
   }
   else if (LAT_MINUTES_DATA == currentDataType)
   {
      _activeEditField = DM_LAT_MINS_EDIT_FIELD;
   }
   else if (LAT_DECIMALS_DATA == currentDataType)
   {
      _activeEditField = DM_LAT_MINS_DECIMAL_EDIT_FIELD;
   }
   else if (LONG_DEGREES_DATA == currentDataType)
   {
      _activeEditField = DM_LONG_DEG_EDIT_FIELD;
   }
   else if (LONG_MINUTES_DATA == currentDataType)
   {
      _activeEditField = DM_LONG_MINS_EDIT_FIELD;
   }
   else if (LONG_DECIMALS_DATA == currentDataType)
   {
      _activeEditField = DM_LONG_MINS_DECIMAL_EDIT_FIELD;
   }
   else
   {
      // Invalid change in active edit field
   }
}


void DestinationLatLongSpellerHandler::UpdateActiveEditFieldForDMSFormat(const enLatLongInputDataType currentDataType)
{
   if (LAT_DEGREES_DATA == currentDataType)
   {
      _activeEditField = DMS_LAT_DEG_EDIT_FIELD;
   }
   else if (LAT_MINUTES_DATA == currentDataType)
   {
      _activeEditField = DMS_LAT_MINS_EDIT_FIELD;
   }
   else if (LAT_SECONDS_DATA == currentDataType)
   {
      _activeEditField = DMS_LAT_SECS_EDIT_FIELD;
   }
   else if (LAT_DECIMALS_DATA == currentDataType)
   {
      _activeEditField = DMS_LAT_SECS_DECIMAL_EDIT_FIELD;
   }
   else if (LONG_DEGREES_DATA == currentDataType)
   {
      _activeEditField = DMS_LONG_DEG_EDIT_FIELD;
   }
   else if (LONG_MINUTES_DATA == currentDataType)
   {
      _activeEditField = DMS_LONG_MINS_EDIT_FIELD;
   }
   else if (LONG_SECONDS_DATA == currentDataType)
   {
      _activeEditField = DMS_LONG_SECS_EDIT_FIELD;
   }
   else if (LONG_DECIMALS_DATA == currentDataType)
   {
      _activeEditField = DMS_LONG_SECS_DECIMAL_EDIT_FIELD;
   }
   else
   {
      // Invalid change in active edit field
   }
}


bool DestinationLatLongSpellerHandler::onCourierMessage(const ResetLatLongPopUpActiveStatusReqMsg& oMsg)
{
   ETG_TRACE_USR1(("DestinationLatLongSpellerHandler::onCourierMessage::ResetLatLongPopUpActiveStatusReqMsg"));

   // this msg trigger on closing Latlong invalid input popup. So setting the databinding value to false
   (*_enableSetLatLongButton).mIsPopUpVisible = false;
   _enableSetLatLongButton.MarkItemModified(ItemKey::EnableSetLatLongButton::IsPopUpVisibleItem);
   _enableSetLatLongButton.SendUpdate();

   return true;
}


void DestinationLatLongSpellerHandler::changeActiveEditFieldText()
{
   ETG_TRACE_USR1(("DestinationLatLongSpellerHandler::changeActiveEditFieldText"));

   Candera::String editFieldText;	// takes the text of the edit field to which active focus was given

   switch (_activeEditField)
   {
      case DD_LAT_DEG_EDIT_FIELD:
      case DM_LAT_DEG_EDIT_FIELD:
      case DMS_LAT_DEG_EDIT_FIELD:
         editFieldText = (*_latlongDataValues).mLatDegreesInputValues;
         break;

      case DM_LAT_MINS_EDIT_FIELD:
      case DMS_LAT_MINS_EDIT_FIELD:
         editFieldText = (*_latlongDataValues).mLatMinutesInputValues;
         break;

      case DMS_LAT_SECS_EDIT_FIELD:
         editFieldText = (*_latlongDataValues).mLatSecondsInputValues;
         break;

      case DD_LAT_DECIMAL_EDIT_FIELD:
      case DM_LAT_MINS_DECIMAL_EDIT_FIELD:
      case DMS_LAT_SECS_DECIMAL_EDIT_FIELD:
         editFieldText = (*_latlongDataValues).mLatDecimalsInputValues;
         break;

      case DD_LONG_DEG_EDIT_FIELD:
      case DM_LONG_DEG_EDIT_FIELD:
      case DMS_LONG_DEG_EDIT_FIELD:
         editFieldText = (*_latlongDataValues).mLongDegreesInputValues;
         break;

      case DM_LONG_MINS_EDIT_FIELD:
      case DMS_LONG_MINS_EDIT_FIELD:
         editFieldText = (*_latlongDataValues).mLongMinutesInputValues;
         break;

      case DMS_LONG_SECS_EDIT_FIELD:
         editFieldText = (*_latlongDataValues).mLongSecondsInputValues;
         break;

      case DD_LONG_DECIMAL_EDIT_FIELD:
      case DM_LONG_MINS_DECIMAL_EDIT_FIELD:
      case DMS_LONG_SECS_DECIMAL_EDIT_FIELD:
         editFieldText = (*_latlongDataValues).mLongDecimalsInputValues;
         break;

      default:
         break;
   }

   /* If the previous active field had no text, then pressed key update with delete all char is not sent */
   if (0 != strcmp(EMPTYSTR, _currentInputStr.c_str()))
   {
      _ignoreEmptyString = true;
      setLatLongSpellerPressedKeyInfoData(DELETE_ALLCHARS.c_str());
   }

   /* If the edit field which received the active focus is not empty, then the corresponding text is updated */
   if (0 != strcmp(EMPTYSTR, editFieldText.GetCString()))
   {
      _ignoreTextReceivedMsg = true;
      setLatLongSpellerPressedKeyInfoData(editFieldText.GetCString());
   }
}


void DestinationLatLongSpellerHandler::changeActiveEditFieldBGImage(bool coordinateFormatChanged)
{
   ETG_TRACE_USR1(("DestinationLatLongSpellerHandler::changeActiveEditFieldBGImage"));

   enActivateEditField currentEditField = _previousActiveEditField;
   const char* currentImage = IMG_NORMAL_EDITFIELD;

   // For co-ordinate change reset all mins,sec and decimals BG images to original Image
   if (true == coordinateFormatChanged)
   {
      Candera::MemoryManagement::SharedPointer<Candera::BitmapImage2D> sharedPtrForNormalBitmapImage = ImageLoader::getAssetBitmapImage(IMG_NORMAL_EDITFIELD);

      // Only Set Latitude Degrees as Active, Longitude degrees will be set as normal
      (*_latlongEditFieldBGImage).mLatitudeDegreesBGBitmap = ImageLoader::getAssetBitmapImage(IMG_ACTIVE_EDITFIELD);
      (*_latlongEditFieldBGImage).mLongitudeDegreesBGBitmap = sharedPtrForNormalBitmapImage;

      // Reset the Bit Maps for the Minutes, seconds and Decimal fields on format change.
      // NOTE : Certain Degree Bitmap can be used across Degrees, Minutes and Seconds not for decimals
      // Minutes
      (*_latlongEditFieldBGImage).mLatitudeMinutesBGBitmap = sharedPtrForNormalBitmapImage;
      (*_latlongEditFieldBGImage).mLongitudeMinutesBGBitmap = sharedPtrForNormalBitmapImage;

      //Seconds
      (*_latlongEditFieldBGImage).mLatitudeSecondsBGBitmap = sharedPtrForNormalBitmapImage;
      (*_latlongEditFieldBGImage).mLongitudeSecondsBGBitmap = sharedPtrForNormalBitmapImage;

      // the Bitmap Image of the decimal field various format to format.
      // Hence, reset all to their original when co-ordinate format is changed
      if (DEGREES == _coordinatesDisplayType)
      {
         // Decimal Images for DD format is bigger
         (*_latlongEditFieldBGImage).mLatitudeDecimalsBGBitmap = ImageLoader::getAssetBitmapImage(IMG_DD_DECIMAL_NORMAL_EDITFIELD);
         (*_latlongEditFieldBGImage).mLongitudeDecimalsBGBitmap = ImageLoader::getAssetBitmapImage(IMG_DD_DECIMAL_NORMAL_EDITFIELD);
      }
      else if (DEGREE_MINS == _coordinatesDisplayType)
      {
         // Decimal Images for DD format is Normal as in Deg, mins and secs
         (*_latlongEditFieldBGImage).mLatitudeDecimalsBGBitmap = sharedPtrForNormalBitmapImage;
         (*_latlongEditFieldBGImage).mLongitudeDecimalsBGBitmap = sharedPtrForNormalBitmapImage;
      }
      else
      {
         // Decimal Images for DD format is smaller
         (*_latlongEditFieldBGImage).mLatitudeDecimalsBGBitmap = ImageLoader::getAssetBitmapImage(IMG_DMS_DECIMAL_NORMAL_EDITFIELD);
         (*_latlongEditFieldBGImage).mLongitudeDecimalsBGBitmap = ImageLoader::getAssetBitmapImage(IMG_DMS_DECIMAL_NORMAL_EDITFIELD);
      }

      _latlongEditFieldBGImage.MarkAllItemsModified();
      _latlongEditFieldBGImage.SendUpdate();
   }  // if (true == coordinateFormatChanged)
   else
   {
      // first iteration to reset previous Editfield with Normal Background Image
      // Second iteration to set current editfield with active Background Image
      for (int iteration = 0; iteration < 2; ++iteration)
      {
         // As the User can randomly switch between Latitude and Longitude edit fields using touch,
         // function calls are used to keep the BG image updated for Both latitude and Longitude.
         // This approach also reduces the cyclomatic complexity of this function

         // function to update the Latitude Edit fields
         UpdateLatitudeEditFieldBackgroundImages(currentEditField, currentImage, iteration);

         // function to update the Longitude Edit fields
         UpdateLongitudeEditFieldBackgroundImages(currentEditField, currentImage, iteration);

         if (_latlongEditFieldBGImage.HasModifiedItems())
         {
            _latlongEditFieldBGImage.SendUpdate();
         }

         // Setting Variables for next iterations
         currentImage = IMG_ACTIVE_EDITFIELD;
         currentEditField = _activeEditField;
      } // End of For loops
   }  // if (true == coordinateFormatChanged)
}


void DestinationLatLongSpellerHandler::UpdateLatitudeEditFieldBackgroundImages(const enActivateEditField currentEditField, const char* currentImage, int currentIteration)
{
   switch (currentEditField)
   {
      case DD_LAT_DEG_EDIT_FIELD:
      case DM_LAT_DEG_EDIT_FIELD:
      case DMS_LAT_DEG_EDIT_FIELD:
      {
         // All Latitude Degree cases share same BG Image
         (*_latlongEditFieldBGImage).mLatitudeDegreesBGBitmap = ImageLoader::getAssetBitmapImage(currentImage);
         _latlongEditFieldBGImage.MarkItemModified(ItemKey::LatLongEditFieldBGImageItem::LatitudeDegreesBGBitmapItem);
         break;
      }
      case DM_LAT_MINS_EDIT_FIELD:
      case DMS_LAT_MINS_EDIT_FIELD:
      {
         // All Latitude Mins cases share same BG Image
         (*_latlongEditFieldBGImage).mLatitudeMinutesBGBitmap = ImageLoader::getAssetBitmapImage(currentImage);
         _latlongEditFieldBGImage.MarkItemModified(ItemKey::LatLongEditFieldBGImageItem::LatitudeMinutesBGBitmapItem);
         break;
      }
      case DMS_LAT_SECS_EDIT_FIELD:
      {
         // All Latitude Seconds cases share same BG Image
         (*_latlongEditFieldBGImage).mLatitudeSecondsBGBitmap = ImageLoader::getAssetBitmapImage(currentImage);
         _latlongEditFieldBGImage.MarkItemModified(ItemKey::LatLongEditFieldBGImageItem::LatitudeSecondsBGBitmapItem);
         break;
      }
      case DD_LAT_DECIMAL_EDIT_FIELD:
      {
         if (0 == currentIteration)
         {
            // Reset the BG Image to normal
            (*_latlongEditFieldBGImage).mLatitudeDecimalsBGBitmap = ImageLoader::getAssetBitmapImage(IMG_DD_DECIMAL_NORMAL_EDITFIELD);
         }
         else
         {
            // Set the BG Image to active
            (*_latlongEditFieldBGImage).mLatitudeDecimalsBGBitmap = ImageLoader::getAssetBitmapImage(IMG_DD_DECIMAL_ACTIVE_EDITFIELD);
         }
         _latlongEditFieldBGImage.MarkItemModified(ItemKey::LatLongEditFieldBGImageItem::LatitudeDecimalsBGBitmapItem);
         break;
      }
      case DM_LAT_MINS_DECIMAL_EDIT_FIELD:
      {
         (*_latlongEditFieldBGImage).mLatitudeDecimalsBGBitmap = ImageLoader::getAssetBitmapImage(currentImage);
         _latlongEditFieldBGImage.MarkItemModified(ItemKey::LatLongEditFieldBGImageItem::LatitudeDecimalsBGBitmapItem);
         break;
      }
      case DMS_LAT_SECS_DECIMAL_EDIT_FIELD:
      {
         if (0 == currentIteration)
         {
            // Reset the BG Image to normal
            (*_latlongEditFieldBGImage).mLatitudeDecimalsBGBitmap = ImageLoader::getAssetBitmapImage(IMG_DMS_DECIMAL_NORMAL_EDITFIELD);
         }
         else
         {
            // Set the BG Image to active
            (*_latlongEditFieldBGImage).mLatitudeDecimalsBGBitmap = ImageLoader::getAssetBitmapImage(IMG_DMS_DECIMAL_ACTIVE_EDITFIELD);
         }
         _latlongEditFieldBGImage.MarkItemModified(ItemKey::LatLongEditFieldBGImageItem::LatitudeDecimalsBGBitmapItem);
         break;
      }
      default:
         break;
   } // End of switch case
}


void DestinationLatLongSpellerHandler::UpdateLongitudeEditFieldBackgroundImages(const enActivateEditField currentEditField, const char* currentImage, int currentIteration)
{
   switch (currentEditField)
   {
      case DD_LONG_DEG_EDIT_FIELD:
      case DM_LONG_DEG_EDIT_FIELD:
      case DMS_LONG_DEG_EDIT_FIELD:
      {
         // All Longitude Degree cases share same BG Image
         (*_latlongEditFieldBGImage).mLongitudeDegreesBGBitmap = ImageLoader::getAssetBitmapImage(currentImage);
         _latlongEditFieldBGImage.MarkItemModified(ItemKey::LatLongEditFieldBGImageItem::LongitudeDegreesBGBitmapItem);
         break;
      }
      case DM_LONG_MINS_EDIT_FIELD:
      case DMS_LONG_MINS_EDIT_FIELD:
      {
         // All Longitude Mins cases share same BG Image
         (*_latlongEditFieldBGImage).mLongitudeMinutesBGBitmap = ImageLoader::getAssetBitmapImage(currentImage);
         _latlongEditFieldBGImage.MarkItemModified(ItemKey::LatLongEditFieldBGImageItem::LongitudeMinutesBGBitmapItem);
         break;
      }
      case DMS_LONG_SECS_EDIT_FIELD:
      {
         // All Longitude Seconds cases share same BG Image
         (*_latlongEditFieldBGImage).mLongitudeSecondsBGBitmap = ImageLoader::getAssetBitmapImage(currentImage);
         _latlongEditFieldBGImage.MarkItemModified(ItemKey::LatLongEditFieldBGImageItem::LongitudeSecondsBGBitmapItem);
         break;
      }
      case DD_LONG_DECIMAL_EDIT_FIELD:
      {
         if (0 == currentIteration)
         {
            // Reset the BG Image to normal
            (*_latlongEditFieldBGImage).mLongitudeDecimalsBGBitmap = ImageLoader::getAssetBitmapImage(IMG_DD_DECIMAL_NORMAL_EDITFIELD);
         }
         else
         {
            // Set the BG Image to active
            (*_latlongEditFieldBGImage).mLongitudeDecimalsBGBitmap = ImageLoader::getAssetBitmapImage(IMG_DD_DECIMAL_ACTIVE_EDITFIELD);
         }
         _latlongEditFieldBGImage.MarkItemModified(ItemKey::LatLongEditFieldBGImageItem::LongitudeDecimalsBGBitmapItem);
         break;
      }
      case DM_LONG_MINS_DECIMAL_EDIT_FIELD:
      {
         (*_latlongEditFieldBGImage).mLongitudeDecimalsBGBitmap = ImageLoader::getAssetBitmapImage(currentImage);
         _latlongEditFieldBGImage.MarkItemModified(ItemKey::LatLongEditFieldBGImageItem::LongitudeDecimalsBGBitmapItem);
         break;
      }
      case DMS_LONG_SECS_DECIMAL_EDIT_FIELD:
      {
         if (0 == currentIteration)
         {
            // Reset the BG Image to normal
            (*_latlongEditFieldBGImage).mLongitudeDecimalsBGBitmap = ImageLoader::getAssetBitmapImage(IMG_DMS_DECIMAL_NORMAL_EDITFIELD);
         }
         else
         {
            // Set the BG Image to active
            (*_latlongEditFieldBGImage).mLongitudeDecimalsBGBitmap = ImageLoader::getAssetBitmapImage(IMG_DMS_DECIMAL_ACTIVE_EDITFIELD);
         }
         _latlongEditFieldBGImage.MarkItemModified(ItemKey::LatLongEditFieldBGImageItem::LongitudeDecimalsBGBitmapItem);
         break;
      }
      default:
         break;
   } // End of switch case
}


void DestinationLatLongSpellerHandler::clearAllEditFields()
{
   ETG_TRACE_USR1(("DestinationLatLongSpellerHandler::clearAllEditFields"));

   // Clear internal members
   _currentInputStr = EMPTYSTR;
   resetLatLongDataValues();

   // Clear Binding data - function call order to be maintained.
   // This call will clear Text Received Data as well
   setLatLongSpellerPressedKeyInfoData(DELETE_ALLCHARS.c_str());
   checkForInputsInEditFields();
}


void DestinationLatLongSpellerHandler::checkForInputsInEditFields()
{
   ETG_TRACE_USR1(("DestinationLatLongSpellerHandler::checkForInputsInEditFields"));

   bool setButtonActive = false;

   _latitudeDegreesStr = (*_latlongDataValues).mLatDegreesInputValues.GetCString();
   _latitudeMinutesStr = (*_latlongDataValues).mLatMinutesInputValues.GetCString();
   _latitudeSecondsStr = (*_latlongDataValues).mLatSecondsInputValues.GetCString();
   _latitudeDecimalsStr = (*_latlongDataValues).mLatDecimalsInputValues.GetCString();

   _longitudeDegreesStr = (*_latlongDataValues).mLongDegreesInputValues.GetCString();
   _longitudeMinutesStr = (*_latlongDataValues).mLongMinutesInputValues.GetCString();
   _longitudeSecondsStr = (*_latlongDataValues).mLongSecondsInputValues.GetCString();
   _longitudeDecimalsStr = (*_latlongDataValues).mLongDecimalsInputValues.GetCString();

   if (DEGREES == _coordinatesDisplayType)
   {
      if (!(_latitudeDegreesStr.empty() || _latitudeDecimalsStr.empty() || _longitudeDegreesStr.empty() || _longitudeDecimalsStr.empty()))
      {
         setButtonActive = true;
      }
   }
   else if (DEGREE_MINS == _coordinatesDisplayType)
   {
      if (!(_latitudeDegreesStr.empty() || _latitudeDecimalsStr.empty() || _latitudeMinutesStr.empty() ||
            _longitudeDegreesStr.empty() || _longitudeDecimalsStr.empty() || _longitudeMinutesStr.empty())
         )
      {
         setButtonActive = true;
      }
   }
   else  // enCoordinatesDisplayType::DEGREES_MINS_SEC
   {
      if (!(_latitudeDegreesStr.empty() || _latitudeDecimalsStr.empty() || _latitudeMinutesStr.empty() || _latitudeSecondsStr.empty() ||
            _longitudeDegreesStr.empty() || _longitudeDecimalsStr.empty() || _longitudeMinutesStr.empty() || _longitudeSecondsStr.empty())
         )
      {
         setButtonActive = true;
      }
   }

   if (setButtonActive != _isSetLatLongButtonEnabled)
   {
      _isSetLatLongButtonEnabled = setButtonActive;
      (*_enableSetLatLongButton).mIsButtonVisible = setButtonActive;
      _enableSetLatLongButton.MarkItemModified(ItemKey::EnableSetLatLongButton::IsButtonVisibleItem);
      _enableSetLatLongButton.SendUpdate();
   }
}


std::string DestinationLatLongSpellerHandler::getCoordinateDecimalFormat(const std::string& inputString, int valueType)
{
   double degreeValue = 0.0f;
   double minuteValue = 0.0f;
   double secondValue = 0.0f;
   double processValue = 0.0f;
   double fractionalValue = 0.0f;

   std::stringstream resultStringStream;
   std::string processString = inputString;

   if (LATITUDE_VALUES == valueType)
   {
      degreeValue = stringToDouble(_latitudeDegreesStr);
      minuteValue = stringToDouble(_latitudeMinutesStr);
      secondValue = stringToDouble(_latitudeSecondsStr);
   }
   else
   {
      degreeValue = stringToDouble(_longitudeDegreesStr);
      minuteValue = stringToDouble(_longitudeMinutesStr);
      secondValue = stringToDouble(_longitudeSecondsStr);
   }

   processValue = degreeValue + ((minuteValue / 60) + (secondValue / 3600));
   fractionalValue = round((processValue - static_cast<int>(processValue)) * 100000);

   resultStringStream << std::setfill('0') << std::setw(2 + valueType) << static_cast<int>(processValue) << '.' << std::setfill('0') << std::setw(5) << static_cast<int>(fractionalValue) << '\0';

   processString = resultStringStream.str();
   return processString;
}


void DestinationLatLongSpellerHandler::changeDisplayFormat()
{
   ETG_TRACE_USR1(("DestinationLatLongSpellerHandler::changeDisplayFormat"));

   // Reset the editfield BG Images. On changing the format
   _previousActiveEditField = _activeEditField;

   Candera::UInt8 procesLatLongFormatIndex;

   if (DEGREES == _coordinatesDisplayType)
   {
      _coordinatesDisplayType = DEGREE_MINS;
      _activeEditField = DM_LAT_DEG_EDIT_FIELD;
   }
   else if (DEGREE_MINS == _coordinatesDisplayType)
   {
      _coordinatesDisplayType = DEGREES_MINS_SEC;
      _activeEditField = DMS_LAT_DEG_EDIT_FIELD;
   }
   else // (DEGREES_MINS_SEC == _coordinatesDisplayType)
   {
      _coordinatesDisplayType = DEGREES;
      _activeEditField = DD_LAT_DEG_EDIT_FIELD;
   }

   _infoStore.setCoordinatesDisplayType(_coordinatesDisplayType);

   // Update to switch widget in HMI
   procesLatLongFormatIndex = static_cast<Candera::UInt8>(_coordinatesDisplayType);
   (*_latLongCoordinatesFormatIndex).mLatLongFormatIndex = procesLatLongFormatIndex;

   _latLongCoordinatesFormatIndex.MarkItemModified(ItemKey::LatLongCoordinatesFormat::LatLongFormatIndexItem);
   _latLongCoordinatesFormatIndex.SendUpdate();

   clearAllEditFields();
   changeActiveEditFieldBGImage(true);
}


void DestinationLatLongSpellerHandler::processEnteredCoordinate()
{
   ETG_TRACE_USR1(("DestinationLatLongSpellerHandler::processEnteredCoordinate"));

   // Convert from string to double
   _coordinate._latitude = getLatitudeValue();
   _coordinate._longitude = getLongitudeValue();

   if (isEnteredCoordinateValid(_coordinate))
   {
      adaptEnteredCoordinateBasedOnSouthWestDirections(_coordinate);
      // Set Coordinate to Map
      setCoordinateToBeShownInMapToInfoStore();
      // Set coordinate as destination to middleware
      _navMiddleware.setLocationWithCoordinates(_coordinate._latitude, _coordinate._longitude);
      _infoStore.setLatitude(_coordinate._latitude);
      _infoStore.setLongitude(_coordinate._longitude);

      // Check for the location data
      _infoStore.setIsDetailInfoRequested(true, InfoStoreBase::DETAIL_INFO_GEOCOORDINATE);
      _navMiddleware.requestLocationAttributes();
      (*_enableSetLatLongButton).mIsButtonVisible = false;
      _enableSetLatLongButton.MarkItemModified(ItemKey::EnableSetLatLongButton::IsButtonVisibleItem);
      _enableSetLatLongButton.SendUpdate();
   }
   else
   {
      // Show debug popup -- for C17 release another text should shown
      //TODO: Along with pop up implementation
      //       showDynamicPopupWithTextID(TextId_IT_00827, "NavigationPopups#Popups#DYNAMIC__MPOP_NAVIGATION");
      showInvalidCoordPopup();
   }
}


double DestinationLatLongSpellerHandler::getLatitudeValue()
{
   ETG_TRACE_USR1(("DestinationLatLongSpellerHandler::getLatitudeValue"));

   std::string processString = EMPTYSTR;
   double latitudeValue;

   if (DEGREES == _coordinatesDisplayType)
   {
      processString = _latitudeDegreesStr + DECIMAL_SYMBOL + _latitudeDecimalsStr;
   }
   else if (DEGREE_MINS == _coordinatesDisplayType)
   {
      _latitudeMinutesStr = _latitudeMinutesStr + DECIMAL_SYMBOL + _latitudeDecimalsStr;
      processString = getCoordinateDecimalFormat(processString.c_str(), LATITUDE_VALUES);
   }
   else // DEGREES_MINS_SEC
   {
      _latitudeSecondsStr = _latitudeSecondsStr + DECIMAL_SYMBOL + _latitudeDecimalsStr;
      processString = getCoordinateDecimalFormat(processString.c_str(), LATITUDE_VALUES);
   }
   latitudeValue = atof(processString.c_str());

   return latitudeValue;
}


double DestinationLatLongSpellerHandler::getLongitudeValue()
{
   ETG_TRACE_USR1(("DestinationLatLongSpellerHandler::getLongitudeValue"));

   std::string processString = EMPTYSTR;
   double longitudeValue;

   if (DEGREES == _coordinatesDisplayType)
   {
      // No conversion Required, the info is stored as Decimal degrees 000.00000 (deg.decimaldegs)
      processString = _longitudeDegreesStr + DECIMAL_SYMBOL + _longitudeDecimalsStr;
   }
   else if (DEGREE_MINS == _coordinatesDisplayType)
   {
      // Conversion is needed, Deg, mins, decimal mins to convereted to deg.decimaldegs
      _longitudeMinutesStr = _longitudeMinutesStr + DECIMAL_SYMBOL + _longitudeDecimalsStr;
      processString = getCoordinateDecimalFormat(processString.c_str(), LONGITUDE_VALUES);
   }
   else // DEGREES_MINS_SEC
   {
      // Conversion is needed, Deg, mins,  Sec,  decimal sec to convereted to deg.decimaldegs
      _longitudeSecondsStr = _longitudeSecondsStr + DECIMAL_SYMBOL + _longitudeDecimalsStr;
      processString = getCoordinateDecimalFormat(processString.c_str(), LONGITUDE_VALUES);
   }
   longitudeValue = atof(processString.c_str());

   return longitudeValue;
}


bool DestinationLatLongSpellerHandler::isEnteredCoordinateValid(const PosWGS84<double>& coordinate)
{
   ETG_TRACE_USR1(("DestinationLatLongSpellerHandler::isEnteredCoordinateValid(_latitude %f, _longitude %f)", coordinate._latitude, coordinate._longitude));

   bool isEnteredLatitudeLongitideValid = false;
   const double MAX_LATITUDE_VALUE = 90.0;
   const double MAX_LONGITUDE_VALUE = 180.0;

   // For the phase 1 implementation, check the validity of latitude and longitude in HMI
   // Wait until middleware interfaces are available
   if ((coordinate._latitude < MAX_LATITUDE_VALUE) && (coordinate._longitude < MAX_LONGITUDE_VALUE))
   {
      isEnteredLatitudeLongitideValid = true;
   }

   return isEnteredLatitudeLongitideValid;
}


void DestinationLatLongSpellerHandler::setCoordinateToBeShownInMapToInfoStore()
{
   ETG_TRACE_USR1(("DestinationLatLongSpellerHandler::setCoordinateToBeShownInMapToInfoStore(_latitude %f, _longitude %f)", _coordinate._latitude, _coordinate._longitude));

   std::vector<PosWGS84<double> > positionVector;
   positionVector.push_back(_coordinate);

   InfoStoreBase::CoordinatesToBeShownInMap& coordinatesToBeShownInMap = _infoStore.getCoordinatesToBeShownInMap();
   coordinatesToBeShownInMap._coordinates = positionVector;
}


void DestinationLatLongSpellerHandler::adaptEnteredCoordinateBasedOnSouthWestDirections(PosWGS84<double>& coordinate)
{
   ETG_TRACE_USR1(("DestinationLatLongSpellerHandler::adaptEnteredCoordinateBasedOnSouthWestDirections(_latitude %f, _longitude %f)", coordinate._latitude, coordinate._longitude));

   if (SOUTH_DIRECTION == _latitudeCoordinatesDirection)
   {
      coordinate._latitude = -coordinate._latitude;
   }
   if (WEST_DIRECTION == _longitudeCoordinatesDirection)
   {
      coordinate._longitude = -coordinate._longitude;
   }
}


#endif // HALL_TO_MDW_COM
