/**************************************************************************************
* @file         : DestinationAddressInputHandlerNarRegion.cpp
* @author       : Preethi Alagappan
* @addtogroup   : AppHmi_Navigation
* @brief        : Handles change of country in hierarchical address input for NAR region
* @copyright    : (c) 2021 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include "DestinationAddressInputHandlerNarRegion.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/DestinationAddressInputHandlerNarRegion.cpp.trc.h"
#endif

const Courier::Identifier IdClickChangeCountryNarRegionBtn = Courier::Identifier("Btn_ChangeCountry_NAR/ButtonWidget");

#if defined (HALL_TO_MDW_COM)

DestinationAddressInputHandlerNarRegion::DestinationAddressInputHandlerNarRegion(navmiddleware::NavMiddleware& _navMiddleware, InfoStore& _infoStore)
   : DestinationAddressInputHandler(_navMiddleware, _infoStore)
   , _isCountryChanged(false)
   , _isCountryDataPresent(false)
   , _currentCountryIndex(0)
{
}


DestinationAddressInputHandlerNarRegion::~DestinationAddressInputHandlerNarRegion()
{
}


void DestinationAddressInputHandlerNarRegion::initialize()
{
   ETG_TRACE_USR4(("DestinationAddressInputHandlerNarRegion::initialize()"));

   _topLevelAddressCategory = STATE;
   DestinationAddressInputHandler::initialize();

   _navMiddleware.startFormBasedAddressInput(navmiddleware::LocationContext(navmiddleware::LocationContext::HMICONTEXT__GUI, navmiddleware::APPLICATIONID__NAVIGATION), true);
}


void DestinationAddressInputHandlerNarRegion::deinitialize()
{
   ETG_TRACE_USR4(("DestinationAddressInputHandlerNarRegion::deinitialize()"));

   DestinationAddressInputHandler::deinitialize();
}


bool DestinationAddressInputHandlerNarRegion::onCourierMessage(const ButtonReactionMsg& oMsg)
{
   bool isMessageProcessed = DestinationAddressInputHandler::onCourierMessage(oMsg);
   if(false == isMessageProcessed)
   {
      const Courier::ViewId sceneName = oMsg.GetView();
      if ((enRelease == oMsg.GetEnReaction())
            && ((sceneName == POI_INPUT_SCENENAME) || (sceneName == DESTINATION_INPUT_SCENENAME))
            && (oMsg.GetSender() == IdClickChangeCountryNarRegionBtn))
      {
         isMessageProcessed = true;
         ETG_TRACE_USR4(("DestinationAddressInputHandlerNarRegion::onCourierMessage(ButtonReactionMsg), Scene name : %s", sceneName.CStr()));

         if(0 < _countryLocationInfos.getLocationInfos().size())
         {
            _isCountryChanged = true;
            _selectedAddressInputCategory = COUNTRY;
            _currentCountryIndex++;
            _currentCountryIndex = _currentCountryIndex % _countryLocationInfos.getLocationInfos().size();
            setCountryIcon(_currentCountryIndex);
            _navMiddleware.selectAddressCategoryForInput(COUNTRY, 0);
            _navMiddleware.setAddressSpellerString("", LIST_WINDOW_SIZE);
            _navMiddleware.requestAddressElements(0, LIST_WINDOW_SIZE);
         }
      }
   }
   return isMessageProcessed;
}


bool DestinationAddressInputHandlerNarRegion::onCourierMessage(const LeaveFormBasedAddressInputReqMsg& oMsg)
{
   ETG_TRACE_USR4(("DestinationAddressInputHandlerNarRegion::onCourierMessage(LeaveFormBasedAddressInputReqMsg())"));

   (*_changeCountryButton).mIsVisible = false;
   (*_changeCountryButton).mCountryIcon = loadImage(NULL, 0, false);
   _changeCountryButton.MarkAllItemsModified();
   _changeCountryButton.SendUpdate(true);

   bool isMessageProcessed = DestinationAddressInputHandler::onCourierMessage(oMsg);
   return isMessageProcessed;
}


void DestinationAddressInputHandlerNarRegion::onPropertyUpdateAddressInputInfoChanged()
{
   ETG_TRACE_USR4(("DestinationAddressInputHandlerNarRegion::onPropertyUpdateAddressInputInfoChanged()"));

   onUpdateCountryIcon();
   DestinationAddressInputHandler::onPropertyUpdateAddressInputInfoChanged();
}


void DestinationAddressInputHandlerNarRegion::onPropertyUpdateAddressElementChanged()
{
   ETG_TRACE_USR4(("DestinationAddressInputHandlerNarRegion::onPropertyUpdateAddressElementChanged(), selectedAddressInputCategory : %d", _selectedAddressInputCategory));
   if(COUNTRY == _selectedAddressInputCategory)
   {
      const navmiddleware::LocationInfos& locationInfos = _navMiddleware.getAddressElements();
      if((false == locationInfos.areResultsOutstanding()) && (0 < locationInfos.getLocationInfos().size()))
      {
         _countryLocationInfos = locationInfos;
         if(true == _isCountryDataPresent)
         {
            if ((true == _isCountryChanged) && (_currentCountryIndex < locationInfos.getLocationInfos().size()))
            {
               ETG_TRACE_USR4(("DestinationAddressInputHandlerNarRegion::onPropertyUpdateAddressElementChanged(), Setting country at index : %d", _currentCountryIndex));
               _navMiddleware.selectAddressElement(locationInfos.getLocationInfos().at(_currentCountryIndex));
               _isCountryChanged = false;
            }
         }
         else
         {
            // On requesting address elements for country, valid icon data is present in the second property update received.
            if(0 != locationInfos.getLocationInfos().at(0).m_image.getBlobSize())
            {
               ETG_TRACE_USR4(("DestinationAddressInputHandlerNarRegion::onPropertyUpdateAddressElementChanged(), Country data updated for NAR region"));
               _isCountryDataPresent = true;
               setCCPCountryIcon();

               if(0 == _addressInputRequestScene)
               {
                  _navMiddleware.leaveFormBasedAddressInput();
               }
               else
               {
                  _navMiddleware.startFormBasedAddressInput(navmiddleware::LocationContext(navmiddleware::LocationContext::HMICONTEXT__GUI, navmiddleware::APPLICATIONID__NAVIGATION), false);
               }
            }
         }
      }
   }
}


void DestinationAddressInputHandlerNarRegion::onUpdateCountryIcon()
{
   ETG_TRACE_USR4(("DestinationAddressInputHandlerNarRegion::onUpdateCountryIcon()"));

   if(true == _isCountryDataPresent)
   {
      if(true == _isCCPAddressInfo)
      {
         _isCCPAddressInfo = false;
         setCCPCountryIcon();
      }
   }
   else
   {
      ETG_TRACE_USR4(("DestinationAddressInputHandlerNarRegion::onPropertyUpdateAddressInputInfoChanged(), Requesting country data for NAR region"));
      const navmiddleware::AddressInputInfo& addresInputInfo = _navMiddleware.getFormBasedAddressInputInfo();
      if(true == addresInputInfo.isCountrySelectable())
      {
         _selectedAddressInputCategory = COUNTRY;
         _navMiddleware.selectAddressCategoryForInput(COUNTRY, 0);
         _navMiddleware.setAddressSpellerString("", LIST_WINDOW_SIZE);
         _navMiddleware.requestAddressElements(0, LIST_WINDOW_SIZE);
      }
   }
}


void DestinationAddressInputHandlerNarRegion::setCCPCountryIcon()
{
   const navmiddleware::AddressInputInfo& addresInputInfo = _navMiddleware.getFormBasedAddressInputInfo();
   const ::std::string& currentCountry = addresInputInfo.getCurrentCountry();
   ETG_TRACE_USR4(("DestinationAddressInputHandlerNarRegion::setCCPCountryIcon(), Country : %s", currentCountry.c_str()));

   ::std::vector<navmiddleware::LocationInfos::LocationInfo> locationInfo = _countryLocationInfos.getLocationInfos();
   ::std::vector<navmiddleware::LocationInfos::LocationInfo>::const_iterator locationIter = locationInfo.begin();
   for( ; locationIter != locationInfo.end(); ++locationIter)
   {
      if(0 == currentCountry.compare(locationIter->m_elementName))
      {
         (*_changeCountryButton).mIsVisible = true;
         (*_changeCountryButton).mCountryIcon = loadImage((const char*)locationIter->m_image.getBlob(), locationIter->m_image.getBlobSize(), true);
         _changeCountryButton.MarkAllItemsModified();
         _changeCountryButton.SendUpdate(true);

         _currentCountryIndex = locationIter->m_index;
         break;
      }
   }
}


void DestinationAddressInputHandlerNarRegion::setCountryIcon(unsigned int countryIndex)
{
   ETG_TRACE_USR4(("DestinationAddressInputHandlerNarRegion::setCountryIcon(), Country index : %d", countryIndex));

   if(countryIndex < _countryLocationInfos.getLocationInfos().size())
   {
      navmiddleware::LocationInfos::LocationInfo locationInfo = _countryLocationInfos.getLocationInfos().at(countryIndex);
      (*_changeCountryButton).mIsVisible = true;
      (*_changeCountryButton).mCountryIcon = loadImage((const char*)locationInfo.m_image.getBlob(), locationInfo.m_image.getBlobSize(), true);
      _changeCountryButton.MarkAllItemsModified();
      _changeCountryButton.SendUpdate(true);
   }
}

#endif
