/**************************************************************************************
* @file         : DestinationAddressInputHandler.cpp
* @author       : ECG5-Naveen Thangamalayan
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) -2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include "DestinationAddressInputHandler.h"
#include "Destination/SpellerHandler.h"   // For fetching text entered in free text speller

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/DestinationAddressInputHandler.cpp.trc.h"
#endif

static const Courier::Identifier  IdClickCountryBtn  = Courier::Identifier("Layer_Country/ButtonWidget");
static const Courier::Identifier  IdClickCityBtn     = Courier::Identifier("Layer_CityPostCode/ButtonWidget");
static const Courier::Identifier  IdClickStreetBtn   = Courier::Identifier("Layer_Street/ButtonWidget");
static const Courier::Identifier  IdClickHouseNumBtn = Courier::Identifier("Layer_HouseNumber/ButtonWidget");
static const Courier::Identifier  IdClickIntersecBtn = Courier::Identifier("Layer_Intersection/ButtonWidget");
static const Courier::Identifier  IdClickMatchesBtn  = Courier::Identifier("Layer_Matches/ButtonWidget");
static const Courier::Identifier  IdClickDetailsBtn  = Courier::Identifier("Layer_Details/ButtonWidget");
static const Candera::String SPELLER_TITLE_ENTER_NAME = LANGUAGE_STRING(TextId_0x04B3, "Enter name");

#if defined (HALL_TO_MDW_COM)

DestinationAddressInputHandler::DestinationAddressInputHandler(navmiddleware::NavMiddleware& _navMiddleware, InfoStore& _infoStore)
   : HMIModelBase(_navMiddleware, _infoStore)
   , _addressInputRequestScene(0)
   , _selectedAddressInputCategory(COUNTRY)
   , _topLevelAddressCategory(COUNTRY)
   , _isCCPAddressInfo(true)
{
   for (int index = 0; index < MAX_ADDRESS_CATEGORY; index++)
   {
      addressBtnState[index] = NULL;
   }
}


DestinationAddressInputHandler::~DestinationAddressInputHandler()
{
}


void DestinationAddressInputHandler::initialize()
{
   ETG_TRACE_USR4(("DestinationAddressInputHandler::initialize()"));

   _navMiddleware.registerLocationPropertyUpdateCallback(*this);
   _infoStore.resetCoordinatesToBeShownInMap();

   (*_newAddressInputValue).mCountry = DEST_ADDRESS_DEFAULT_VALUE[_topLevelAddressCategory];
   (*_newAddressInputValue).mCity = DEST_ADDRESS_DEFAULT_VALUE[CITY];
   (*_newAddressInputValue).mHousenum = DEST_ADDRESS_DEFAULT_VALUE[HOUSE_NUMBER];
   (*_newAddressInputValue).mStreet = DEST_ADDRESS_DEFAULT_VALUE[STREET];
   (*_newAddressInputValue).mIntersection = DEST_ADDRESS_DEFAULT_VALUE[CROSS_STREET];
   (*_newAddressInputValue).mState = DEST_ADDRESS_DEFAULT_VALUE[STATE];
   _newAddressInputValue.MarkAllItemsModified();
   _newAddressInputValue.SendUpdate(true);

   (*_addressInputValPOIData).mCountry = DEST_ADDRESS_DEFAULT_VALUE[_topLevelAddressCategory];
   (*_addressInputValPOIData).mCity = DEST_ADDRESS_DEFAULT_VALUE[CITY];
   _addressInputValPOIData.MarkAllItemsModified();
   _addressInputValPOIData.SendUpdate(true);

   (*_addressEntryBtnState).mCity = false;
   (*_addressEntryBtnState).mStreet = false;
   (*_addressEntryBtnState).mHousenum = false;
   (*_addressEntryBtnState).mInterSection = false;
   (*_addressEntryBtnState).mDetails = false;
   (*_addressEntryBtnState).mGuidance = false;
   _addressEntryBtnState.MarkAllItemsModified();
   _addressEntryBtnState.SendUpdate(true);

   addressBtnState[CITY] = &(*_addressEntryBtnState).mCity;
   addressBtnState[STREET] = &(*_addressEntryBtnState).mStreet;
   addressBtnState[HOUSE_NUMBER] = &(*_addressEntryBtnState).mHousenum;
   addressBtnState[CROSS_STREET] = &(*_addressEntryBtnState).mInterSection;
}


void DestinationAddressInputHandler::deinitialize()
{
   ETG_TRACE_USR4(("DestinationAddressInputHandler::deinitialize()"));
   _navMiddleware.unregisterLocationPropertyUpdateCallback(*this);
}


bool DestinationAddressInputHandler::onCourierMessage(const InitDestinationInputReqMsg& oMsg)
{
   ETG_TRACE_USR4(("DestinationAddressInputHandler::onCourierMessage(InitDestinationInputReqMsg)"));

   _isCCPAddressInfo = !(oMsg.GetRetainDetails()); // If last entered address info should not be retained, then CCP address info is displayed
   _addressInputRequestScene = oMsg.GetRequestScene();

   if(_isCCPAddressInfo == true)
   {
      ETG_TRACE_USR4(("DestinationAddressInputHandler::onCourierMessage(InitDestinationInputReqMsg), First entry"));
      _navMiddleware.startFormBasedAddressInput(navmiddleware::LocationContext(navmiddleware::LocationContext::HMICONTEXT__GUI, navmiddleware::APPLICATIONID__NAVIGATION), true);
   }
   else
   {
      ETG_TRACE_USR4(("DestinationAddressInputHandler::onCourierMessage(InitDestinationInputReqMsg), Second entry"));
      _navMiddleware.startFormBasedAddressInput(navmiddleware::LocationContext(navmiddleware::LocationContext::HMICONTEXT__GUI, navmiddleware::APPLICATIONID__NAVIGATION), false);
   }
   return true;
}


bool DestinationAddressInputHandler::onCourierMessage(const ButtonReactionMsg& oMsg)
{
   bool isMessageProcessed = false;

   const Courier::ViewId SPELLER_SCENE_NAME          = Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#N_SpellerEdit");
   const Courier::ViewId DETAILEDINFO_SCENE_NAME     = Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#N_Address_DetailedInfo");
   const Courier::ViewId FAV_ADD_FAV_SCENE_NAME      = Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#N_Fav_AddFavorites");

   const Courier::Identifier senderInfo = oMsg.GetSender();
   const Courier::ViewId sceneName      = oMsg.GetView();
   const enReaction  reaction           = oMsg.GetEnReaction();

   if (enPress == reaction)
   {
      if ((sceneName == POI_INPUT_SCENENAME) || (sceneName == DESTINATION_INPUT_SCENENAME))
      {
         ETG_TRACE_USR4(("DestinationAddressInputHandler::onCourierMessage(ButtonReactionMsg), Scene name : %s", sceneName.CStr()));
         if (senderInfo != IdClickDetailsBtn)   // Button handler for AddressInputEntry category selection
         {
            isMessageProcessed = true;
            if (senderInfo == IdClickCountryBtn)
            {
               _selectedAddressInputCategory = _topLevelAddressCategory;
            }
            else if (senderInfo == IdClickCityBtn)
            {
               _selectedAddressInputCategory = CITY;
            }
            else if (senderInfo == IdClickStreetBtn)
            {
               _selectedAddressInputCategory = STREET;
            }
            else if (senderInfo == IdClickHouseNumBtn)
            {
               _selectedAddressInputCategory = HOUSE_NUMBER;
            }
            else if (senderInfo == IdClickIntersecBtn)
            {
               _selectedAddressInputCategory = CROSS_STREET;
            }
            else
            {
               isMessageProcessed = false;
            }
            if (true == isMessageProcessed)
            {
               if (false == EXT_bIsSpeedLockActive)
               {
                  (*_addressEntryTitleData).mTitleValue = DEST_ADDRESS_DEFAULT_VALUE[_selectedAddressInputCategory];
                  _addressEntryTitleData.MarkAllItemsModified();
                  _addressEntryTitleData.SendUpdate(true);

                  SpellerHandler::getInstance()->ResetEditFieldTextValues(Candera::String(""));
                  _navMiddleware.selectAddressCategoryForInput(_selectedAddressInputCategory, 0);
                  POST_MSG((COURIER_MESSAGE_NEW(NewAddressInputBtnSelectionMsg)()));
               }
               else
               {
                  POST_MSG((COURIER_MESSAGE_NEW(UpdateSpeedLockPopupTextReqMsg)()));
               }
            }
         }
         else if (senderInfo == IdClickDetailsBtn)  // Handles press of Detailed info button
         {
            bool isLocationSet = _navMiddleware.setLocationWithCurrentAddressInput();
            if (true == isLocationSet)
            {
               setCoordinateToBeShownInMap();
               _infoStore.setIsDetailInfoRequested(true, InfoStoreBase::DETAIL_INFO_HIERARCHICAL_ADDRESS);
               _navMiddleware.requestLocationAttributes();
            }
         }
      }
      else if ((sceneName == SPELLER_SCENE_NAME) && (senderInfo == IdClickMatchesBtn))
      {
         ETG_TRACE_USR4(("DestinationAddressInputHandler::onCourierMessage(ButtonReactionMsg), Scene name : %s", sceneName.CStr()));

         POST_MSG((COURIER_MESSAGE_NEW(EnableSpellerMatchListMsg)()));
      }
      else if ((sceneName == DETAILEDINFO_SCENE_NAME) || (sceneName == FAV_ADD_FAV_SCENE_NAME))
      {
         ETG_TRACE_USR4(("DestinationAddressInputHandler::onCourierMessage(ButtonReactionMsg), Scene name : %s", sceneName.CStr()));

         (*_addressEntryTitleData).mTitleValue = SPELLER_TITLE_ENTER_NAME;
         _addressEntryTitleData.MarkAllItemsModified();
         _addressEntryTitleData.SendUpdate(true);
      }
      else
      {
         // do nothing
      }
   }
   return isMessageProcessed;
}


bool DestinationAddressInputHandler::onCourierMessage(const UpdateSpeedLockPopupTextReqMsg& oMsg)
{
   ETG_TRACE_USR4(("DestinationAddressInputHandler::onCourierMessage(UpdateSpeedLockPopupTextReqMsg)"));

   const Candera::String DATA_CONTEXT__TEXT_FIRSTLINE = LANGUAGE_STRING(TextId_0x0A20, "For your safety, the speller is off ");
   const Candera::String DATA_CONTEXT__TEXT_SECONDLINE = LANGUAGE_STRING(TextId_0x0A21, "while the vehicle is in motion.");

   (*_PopupText).mTextLine1 = DATA_CONTEXT__TEXT_FIRSTLINE;
   (*_PopupText).mTextLine2 = DATA_CONTEXT__TEXT_SECONDLINE;
   _PopupText.MarkAllItemsModified();
   _PopupText.SendUpdate(true);
   POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_MultiLine_Text"))));

   return true;
}


bool DestinationAddressInputHandler::onCourierMessage(const SpellerOKBtnPressedMsg& oMsg)
{
   bool isMessageProcessed = false;
   const Courier::UInt32 sceneId = oMsg.GetSceneType();

   if (sceneId == AppHmi_NavigationModule_NavigationScenes_N_NewDest_AddressInput)
   {
      ETG_TRACE_USR4(("DestinationAddressInputHandler::onCourierMessage(SpellerOKBtnPressedMsg), Hierarchical address input scene"));
      std::string currentString = SpellerHandler::getInstance()->GetEditFieldText().GetCString();
      _navMiddleware.setAddressSpellerString(currentString, LIST_WINDOW_SIZE);
      _navMiddleware.requestAddressElements(0, LIST_WINDOW_SIZE);
      isMessageProcessed = true;
   }

   return isMessageProcessed;
}


bool DestinationAddressInputHandler::onCourierMessage(const SetPOISearchLocationReqMsg& oMsg)
{
   ETG_TRACE_USR4(("DestinationAddressInputHandler::onCourierMessage(SetPOISearchLocationReqMsg())"));

   if (true == (*_addressEntryBtnState).mPOISearch)
   {
      _navMiddleware.setLocationWithCurrentAddressInput();
   }
   return true;
}


bool DestinationAddressInputHandler::onCourierMessage(const LeaveFormBasedAddressInputReqMsg& oMsg)
{
   ETG_TRACE_USR4(("DestinationAddressInputHandler::onCourierMessage(LeaveFormBasedAddressInputReqMsg())"));

   _navMiddleware.leaveFormBasedAddressInput();

   return true;
}


void DestinationAddressInputHandler::onPropertyUpdateAddressInputInfoChanged()
{
   ETG_TRACE_USR4(("DestinationAddressInputHandler::onPropertyUpdateAddressInputInfoChanged()"));

   if (_addressInputRequestScene == AppHmi_NavigationModule_NavigationScenes_N_NewDest_AddressInput)
   {
      onUpdateNewDestinationAddressInput();
   }
   else if (_addressInputRequestScene == AppHmi_NavigationModule_NavigationScenes_N_MyPOIs_AddressInput)
   {
      onUpdatePOISearchAddressInput();
   }
   else
   {
      // do nothing
   }
}


void DestinationAddressInputHandler::onPropertyUpdateAddressElementChanged()
{
   // do nothing
}


void DestinationAddressInputHandler::onUpdateNewDestinationAddressInput()
{
   ETG_TRACE_USR4(("DestinationAddressInputHandler::onUpdateNewDestinationAddressInput()"));

   const navmiddleware::AddressInputInfo& addresInputInfo = _navMiddleware.getFormBasedAddressInputInfo();
   Candera::String currentAddressInfo[MAX_ADDRESS_CATEGORY];        // Current value for each address input from navmiddleware
   Candera::String* previousAddressInfo[MAX_ADDRESS_CATEGORY] = {}; // Address input value last displayed in HMI
   bool destinationAddressButtonSelectable[MAX_ADDRESS_CATEGORY];   // Selectable status for each address category from navmiddleware
   bool isEntryChanged = false;
   bool isNewDestAdrsChanged = false;

   destinationAddressButtonSelectable[COUNTRY] = addresInputInfo.isCountrySelectable();
   destinationAddressButtonSelectable[STATE]   = addresInputInfo.isStateSelectable();
   destinationAddressButtonSelectable[CITY]    = addresInputInfo.isCitySelectable();
   destinationAddressButtonSelectable[STREET]  = addresInputInfo.isStreetSelectable();
   destinationAddressButtonSelectable[HOUSE_NUMBER] = addresInputInfo.isHouseNumberSelectable();
   destinationAddressButtonSelectable[CROSS_STREET] = addresInputInfo.isCrossStreetSelectable();

   currentAddressInfo[COUNTRY] = addresInputInfo.getCurrentCountry().c_str();
   currentAddressInfo[STATE]   = addresInputInfo.getCurrentState().c_str();
   currentAddressInfo[CITY]    = addresInputInfo.getCurrentCity().c_str();
   currentAddressInfo[STREET]  = addresInputInfo.getCurrentStreet().c_str();
   currentAddressInfo[HOUSE_NUMBER] = addresInputInfo.getCurrentHouseNumber().c_str();
   currentAddressInfo[CROSS_STREET] = addresInputInfo.getCurrentCrossStreet().c_str();

   previousAddressInfo[COUNTRY] = &(*_newAddressInputValue).mCountry;
   previousAddressInfo[STATE]   = NULL;   // unused
   previousAddressInfo[CITY]    = &(*_newAddressInputValue).mCity;
   previousAddressInfo[STREET]  = &(*_newAddressInputValue).mStreet;
   previousAddressInfo[HOUSE_NUMBER] = &(*_newAddressInputValue).mHousenum;
   previousAddressInfo[CROSS_STREET] = &(*_newAddressInputValue).mIntersection;

   // Modify Country/State data
   isEntryChanged = changeDestinationInputValue(previousAddressInfo[COUNTRY],
                    currentAddressInfo[_topLevelAddressCategory],
                    _topLevelAddressCategory,
                    destinationAddressButtonSelectable[_topLevelAddressCategory]);
   isNewDestAdrsChanged = isEntryChanged;

   bool enableNextField = (*previousAddressInfo[COUNTRY] != DEST_ADDRESS_DEFAULT_VALUE[_topLevelAddressCategory]) ? true : false;

   // Modify address category values (city, street, house number) based on update from navmiddleware
   int categoryIndex = static_cast<int>(CITY);
   for (; (categoryIndex < CROSS_STREET) && (true == enableNextField); categoryIndex++)
   {
      enableNextField = false;
      isEntryChanged = changeDestinationInputValue(previousAddressInfo[categoryIndex],
                       currentAddressInfo[categoryIndex],
                       static_cast<navmiddleware::AddressCategory>(categoryIndex),
                       destinationAddressButtonSelectable[categoryIndex]);
      isNewDestAdrsChanged = (false == isNewDestAdrsChanged) ? isEntryChanged : true;

      if (*previousAddressInfo[categoryIndex] != DEST_ADDRESS_DEFAULT_VALUE[categoryIndex])
      {
         enableNextField = true;
      }
      else if (false == destinationAddressButtonSelectable[categoryIndex])
      {
         enableNextField = true;
      }
   }

   if (categoryIndex < CROSS_STREET)
   {
      while ((categoryIndex < MAX_ADDRESS_CATEGORY) && (NULL != addressBtnState[categoryIndex]))
      {
         *previousAddressInfo[categoryIndex] = DEST_ADDRESS_DEFAULT_VALUE[categoryIndex];
         *addressBtnState[categoryIndex] = false;
         categoryIndex++;
      }
   }
   else  // Modify address category values for intersection
   {
      isEntryChanged = changeDestinationInputValue(previousAddressInfo[CROSS_STREET],
                       currentAddressInfo[CROSS_STREET],
                       static_cast<navmiddleware::AddressCategory>(CROSS_STREET),
                       destinationAddressButtonSelectable[CROSS_STREET]);
      isNewDestAdrsChanged = (false == isNewDestAdrsChanged) ? isEntryChanged : true;
   }

   if (true == isNewDestAdrsChanged)
   {
      _newAddressInputValue.MarkAllItemsModified();
      _newAddressInputValue.SendUpdate(true);

      // Enable or disable detailed info and guidance button
      if ((true == addresInputInfo.isValidDestination()) && (false == addresInputInfo.isAmbiguityResultionRequired()))
      {
         (*_addressEntryBtnState).mDetails = true;
         (*_addressEntryBtnState).mGuidance = true;
      }
      else
      {
         (*_addressEntryBtnState).mDetails = false;
         (*_addressEntryBtnState).mGuidance = false;
      }
   }
   _addressEntryBtnState.MarkAllItemsModified();
   _addressEntryBtnState.SendUpdate(true);
}


void DestinationAddressInputHandler::onUpdatePOISearchAddressInput()
{
   ETG_TRACE_USR4(("DestinationAddressInputHandler::onPropertyUpdateAddressInputInfoChanged()"));

   const navmiddleware::AddressInputInfo& addresInputInfo = _navMiddleware.getFormBasedAddressInputInfo();
   Candera::String currentAddressInfo[MAX_ADDRESS_CATEGORY_POI_SEARCH];        // Current value for each address input from navmiddleware
   Candera::String* previousAddressInfo[MAX_ADDRESS_CATEGORY_POI_SEARCH] = {}; // Address input value last displayed in HMI
   bool destinationAddressButtonSelectable[MAX_ADDRESS_CATEGORY_POI_SEARCH];   // Selecatability status for each address category from navmiddleware

   bool isAddressInputSelectable = false;
   bool isEntryChanged           = false;
   bool isPoiSearchAdrsChanged   = false;

   destinationAddressButtonSelectable[COUNTRY] = addresInputInfo.isCountrySelectable();
   destinationAddressButtonSelectable[STATE]   = addresInputInfo.isStateSelectable();
   destinationAddressButtonSelectable[CITY]    = addresInputInfo.isCitySelectable();

   currentAddressInfo[COUNTRY] = addresInputInfo.getCurrentCountry().c_str();
   currentAddressInfo[STATE]   = addresInputInfo.getCurrentState().c_str();
   currentAddressInfo[CITY]    = addresInputInfo.getCurrentCity().c_str();

   previousAddressInfo[COUNTRY] = &(*_addressInputValPOIData).mCountry;
   previousAddressInfo[STATE]   = NULL;
   previousAddressInfo[CITY]    = &(*_addressInputValPOIData).mCity;

   // Country/state data
   isEntryChanged = changeDestinationInputValue(previousAddressInfo[COUNTRY],
                    currentAddressInfo[_topLevelAddressCategory],
                    _topLevelAddressCategory,
                    destinationAddressButtonSelectable[_topLevelAddressCategory]);
   isPoiSearchAdrsChanged = isEntryChanged;

   // City data
   isAddressInputSelectable = destinationAddressButtonSelectable[CITY] && (!isEntryChanged);
   isEntryChanged = changeDestinationInputValue(previousAddressInfo[CITY],
                    currentAddressInfo[CITY],
                    CITY,
                    isAddressInputSelectable);
   isPoiSearchAdrsChanged = (false == isPoiSearchAdrsChanged) ? isEntryChanged : true;

   if (true == isPoiSearchAdrsChanged)
   {
      _addressInputValPOIData.MarkAllItemsModified();
      _addressInputValPOIData.SendUpdate(true);

      // Updates button state of City and POI search
      (*_addressEntryBtnState).mCity = destinationAddressButtonSelectable[CITY];
      if ((true == addresInputInfo.isValidDestination()) && (false == addresInputInfo.isAmbiguityResultionRequired()))
      {
         (*_addressEntryBtnState).mPOISearch = true;
      }
      else
      {
         (*_addressEntryBtnState).mPOISearch = false;
      }
      _addressEntryBtnState.MarkItemModified(ItemKey::DEDisableButton::CityItem);
      _addressEntryBtnState.MarkItemModified(ItemKey::DEDisableButton::POISearchItem);
      _addressEntryBtnState.SendUpdate(true);
   }
}


bool DestinationAddressInputHandler::changeDestinationInputValue(Candera::String* addressInputDataBinding,
      Candera::String& addressInputMiddlewareInfo,
      navmiddleware::AddressCategory currentAddressCategory,
      bool isAddressInputSelectable)
{
   ETG_TRACE_USR4(("DestinationAddressInputHandler::changeDestinationInputValue(), Address category : %d, isSelectable : %d", currentAddressCategory, isAddressInputSelectable));
   bool isDataChanged = false;

   if (NULL != addressInputDataBinding)
   {
      ETG_TRACE_USR4(("DestinationAddressInputHandler::changeDestinationInputValue(), Previous address input data : %s", addressInputDataBinding->GetCString()));
      ETG_TRACE_USR4(("DestinationAddressInputHandler::changeDestinationInputValue(), Updated address input data : %s", addressInputMiddlewareInfo.GetCString()));

      if (true == isAddressInputSelectable)
      {
         // Update the latest value of addressInput from middleware
         if (*addressInputDataBinding != addressInputMiddlewareInfo)
         {
            if (false == addressInputMiddlewareInfo.IsEmpty())
            {
               *addressInputDataBinding = addressInputMiddlewareInfo;
            }
            else
            {
               *addressInputDataBinding = DEST_ADDRESS_DEFAULT_VALUE[currentAddressCategory];
            }
            isDataChanged = true;
         }
      }
      else
      {
         if (*addressInputDataBinding != DEST_ADDRESS_DEFAULT_VALUE[currentAddressCategory])
         {
            // Reset the text label of button to default if the button becomes unselectable
            *addressInputDataBinding = DEST_ADDRESS_DEFAULT_VALUE[currentAddressCategory];
            isDataChanged = true;
         }
      }
   }
   if ((CITY <= currentAddressCategory) && (currentAddressCategory < MAX_ADDRESS_CATEGORY))
   {
      if (NULL != addressBtnState[currentAddressCategory])
      {
         *addressBtnState[currentAddressCategory] = isAddressInputSelectable;
      }
   }
   return isDataChanged;
}


void DestinationAddressInputHandler::setCoordinateToBeShownInMap()
{
   ETG_TRACE_USR4(("DestinationAddressInputHandler::setCoordinateToBeShownInMap()"));

   const navmiddleware::AddressInputInfo& addresInputInfo = _navMiddleware.getFormBasedAddressInputInfo();
   if (true == addresInputInfo.getCurrentCoordinates().isValid())
   {
      double coordLongitude = addresInputInfo.getCurrentCoordinates().getValue().getLongitude();
      double coordLatitude  = addresInputInfo.getCurrentCoordinates().getValue().getLatitude();

      std::vector<PosWGS84<double> > positionVector;
      positionVector.push_back(PosWGS84<double>(coordLongitude, coordLatitude));

      InfoStoreBase::CoordinatesToBeShownInMap& coordinatesToBeShownInMap = _infoStore.getCoordinatesToBeShownInMap();
      coordinatesToBeShownInMap._coordinates = positionVector;

      _infoStore.setLongitude(coordLongitude);
      _infoStore.setLatitude(coordLatitude);

      POST_MSG((COURIER_MESSAGE_NEW(SetMapCameraModeReqMsg)(MAP_CAMERA_AND_VIEW_MODE_LOCATION_DETAILS)));
   }
}


#endif
