/**************************************************************************************
* @file         : TraceCommandHandler.cpp
* @author       : ECG5-Atchuta Sashank Kappagantu
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) 2018-2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "App/datapool/NaviDataPoolConfig.h"
#include "TraceCommandHandler.h"

#ifndef _MSC_VER
#include "../../di_trace/components/etg/etg.h"
#include <unistd.h>
#endif

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_NAVIGATION
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_NAVIGATION_"
#define ETG_I_FILE_PREFIX                 TraceCommandHandler::
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/TraceCommandHandler.cpp.trc.h"
#endif

static const int WAYPOINT_DELETE_POPUP                                  = 7;
static const int WAYPOINT_DELETEALL_POPUP                               = 8;

#if defined (HALL_TO_MDW_COM)

const int DEMO_MODE_SPEED_MAX = 1000;
const int DEMO_MODE_SPEED_STEP = 100;
const int DEMO_MODE_SPEED_MIN = 100;

navmiddleware::NavMiddleware* TraceCommandHandler::_sNavMiddleware = NULL;
InfoStore* TraceCommandHandler::_sInfoStore = NULL;
int TraceCommandHandler::_demoModeSpeed = DEMO_MODE_SPEED_MIN;

TraceCommandHandler::TraceCommandHandler()
{
}


TraceCommandHandler::~TraceCommandHandler()
{
}


void TraceCommandHandler::initialize(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore)
{
   ETG_TRACE_COMP(("TraceCommandHandler::initialize() "));

#ifndef _MSC_VER
   ETG_I_REGISTER_FILE();
#endif

   _sNavMiddleware = &navMiddleware;
   _sInfoStore = &infoStore;
}


#ifndef _MSC_VER

ETG_I_CMD_DEFINE((traceCmd_setCoordinates, "SetCoordinates %s", ETG_I_STRING coordinates))
void TraceCommandHandler::traceCmd_setCoordinates(std::string coordinates)
{
   int pos1 = coordinates.find(',');
   int pos2 = coordinates.find(',', (pos1 + 1));
   int length = (pos2 - pos1) - 1;

   double latitude = atof(coordinates.substr(0, pos1).c_str());
   double longitude = atof(coordinates.substr((pos1 + 1), length).c_str());

   ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_setCoordinates() %f, %f", latitude, longitude));
   if (_sNavMiddleware)
   {
      if ((latitude != 0) && (longitude != 0))
      {
         _sNavMiddleware->setLocationWithCoordinates(latitude, longitude);
         ETG_TRACE_COMP(("####traceCmd_setCoordinatesEND####"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("TraceCommandHandler::traceCmd_setCoordinates() _sNavMiddleware not initialized"));
   }
}


ETG_I_CMD_DEFINE((traceCmd_startGuidance, "StartGuidance"))
void TraceCommandHandler::traceCmd_startGuidance()
{
   ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_startGuidance()"));
   if (_sNavMiddleware)
   {
      _sInfoStore->setGuidanceStatefromSDS(true);
      POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(StartRouteGuidanceReqMsg)()));
      ETG_TRACE_COMP(("####traceCmd_startGuidanceEND####"));
   }
   else
   {
      ETG_TRACE_ERR(("_sNavMiddleware not initialized"));
   }
}


ETG_I_CMD_DEFINE((traceCmd_stopGuidance, "StopGuidance"))
void TraceCommandHandler::traceCmd_stopGuidance()
{
   ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_stopGuidance()"));
   if (_sNavMiddleware)
   {
      POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(CancelRouteGuidanceReqMsg)()));
      ETG_TRACE_COMP(("####traceCmd_stopGuidanceEND####"));
   }
   else
   {
      ETG_TRACE_ERR(("_sNavMiddleware not initialized"));
   }
}


ETG_I_CMD_DEFINE((traceCmd_initializeWaypoints, "InitializeWaypoints"))
void TraceCommandHandler::traceCmd_initializeWaypoints()
{
   ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_initializeWaypoints()"));
   if (_sNavMiddleware)
   {
      _sNavMiddleware->requestWaypointList(navmiddleware::WAYPOINT_LIST_MODE__VIEW);
      ETG_TRACE_COMP(("####traceCmd_initializeWaypointsEND####"));
   }
   else
   {
      ETG_TRACE_ERR(("TraceCommandHandler::traceCmd_initializeWaypoints() _sNavMiddleware not initialized"));
   }
}


ETG_I_CMD_DEFINE((traceCmd_clearWaypoints, "ClearWaypoints"))
void TraceCommandHandler::traceCmd_clearWaypoints()
{
   ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_clearWaypoints()"));
   POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(WaypointDeleteReqMsg)(WAYPOINT_DELETEALL_POPUP)));
   ETG_TRACE_COMP(("####traceCmd_stopGuidanceEND####"));
}


/* @TODO: Implementation if required
ETG_I_CMD_DEFINE((traceCmd_applyWaypointsChanges, "ApplyWaypointChanges"))
void TraceCommandHandler::traceCmd_applyWaypointsChanges()
{
   ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_applyWaypointsChanges()"));
   //POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ApplyWaypointsReqMsg)()));
   ETG_TRACE_COMP(("####traceCmd_applyWaypointsChangesEND####"));
}


ETG_I_CMD_DEFINE((traceCmd_insertWaypoint, "InsertWaypoint %u", unsigned int))
void TraceCommandHandler::traceCmd_insertWaypoint(unsigned int index)
{
   ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_insertWaypoint() at index : %u", index));
   if (_sInfoStore)
   {
      _sInfoStore->setWaypointIndex(index);
      //POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(InsertWaypointReqMsg)()));
      ETG_TRACE_COMP(("####traceCmd_insertWaypointEND####"));
   }
   else
   {
      ETG_TRACE_ERR(("TraceCommandHandler::traceCmd_insertWaypoint() _sInfoStore not initialized"));
   }
}*/


ETG_I_CMD_DEFINE((traceCmd_deleteWaypoint, "DeleteWaypoint %u", unsigned int))
void TraceCommandHandler::traceCmd_deleteWaypoint(unsigned int index)
{
   ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_deleteWaypoint() at index : %u", index));
   if (_sInfoStore)
   {
      _sInfoStore->setWaypointListIdx(index);
      POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(WaypointDeleteReqMsg)(WAYPOINT_DELETE_POPUP)));
      ETG_TRACE_COMP(("####traceCmd_deleteWaypointEND####"));
   }
   else
   {
      ETG_TRACE_ERR(("TraceCommandHandler::traceCmd_deleteWaypoint() _sInfoStore not initialized"));
   }
}


ETG_I_CMD_DEFINE((traceCmd_setDemoStartPosition, "SetDemoStartPosition %s", ETG_I_STRING coordinates))
void TraceCommandHandler::traceCmd_setDemoStartPosition(std::string coordinates)
{
   int pos1 = coordinates.find(',');
   int pos2 = coordinates.find(',', (pos1 + 1));
   int length = (pos2 - pos1) - 1;

   double latitude = atof(coordinates.substr(0, pos1).c_str());
   double longitude = atof(coordinates.substr((pos1 + 1), length).c_str());
   double  heading = atof(coordinates.substr(pos2 + 1).c_str());

   ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_setDemoStartPosition() %d %d %f", latitude, longitude, heading));

   if (_sNavMiddleware)
   {
      if ((latitude != 0) && (longitude != 0))
      {
         _sNavMiddleware->setDemoModePosition(latitude, longitude, (float)heading);
      }
      ETG_TRACE_COMP(("####traceCmd_setDemoStartPositionEND####"));
   }
   else
   {
      ETG_TRACE_ERR(("TraceCommandHandler::traceCmd_setDemoStartPosition() _sNavMiddleware not initialized"));
   }
}


ETG_I_CMD_DEFINE((traceCmd_getRGState, "GetRGState"))
void TraceCommandHandler::traceCmd_getRGState()
{
   ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_getRGState()"));
   if (_sNavMiddleware)
   {
      ETG_TRACE_COMP(("GetRGState status %d", _sNavMiddleware->isGuidanceActive()));
      ETG_TRACE_COMP(("####traceCmd_getRGStateEND####"));
   }
   else
   {
      ETG_TRACE_ERR(("_sNavMiddleware not initialized"));
   }
}


ETG_I_CMD_DEFINE((traceCmd_setDemoMode, "SetDemoMode %d", bool))
void TraceCommandHandler::traceCmd_setDemoMode(bool demoModeState)
{
   ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_setDemoMode()"));
   if (_sNavMiddleware)
   {
      if (demoModeState)
      {
         _sNavMiddleware->activateDemoMode();
      }
      else
      {
         _sNavMiddleware->deactivateDemoMode();
      }
      ETG_TRACE_COMP(("####traceCmd_setDemoModeEND####"));
   }
   else
   {
      ETG_TRACE_ERR(("_sNavMiddleware not initialized"));
   }
}


ETG_I_CMD_DEFINE((traceCmd_cancelRouteGuidance, "CancelRouteGuidance"))
void TraceCommandHandler::traceCmd_cancelRouteGuidance()
{
   ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_cancelRouteGuidance()"));
   if (_sNavMiddleware)
   {
      POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(CancelRouteGuidanceReqMsg)()));
      ETG_TRACE_COMP(("####traceCmd_cancelRouteGuidanceEND####"));
   }
   else
   {
      ETG_TRACE_ERR(("_sNavMiddleware not initialized"));
   }
}


ETG_I_CMD_DEFINE((traceCmd_getDemoModeState, "GetDemoModeState "))
void TraceCommandHandler::traceCmd_getDemoModeState()
{
   ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_getDemoModeState()"));
   if (_sNavMiddleware)
   {
      ETG_TRACE_COMP(("GetDemoModeState status %s", _sNavMiddleware->isDemoModeActive() ? "ON" : "OFF"));
      ETG_TRACE_COMP(("####traceCmd_getDemoModeStateEND####"));
   }
   else
   {
      ETG_TRACE_ERR(("_sNavMiddleware not initialized"));
   }
}


ETG_I_CMD_DEFINE((traceCmd_getCurrentCompassHeading, "GetCurrentCompassHeading"))
void TraceCommandHandler::traceCmd_getCurrentCompassHeading()
{
   ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_getCurrentCompassHeading()"));
   if (_sNavMiddleware)
   {
      ETG_TRACE_COMP(("Current Compass Heading: %f", _sNavMiddleware->getPositionStatusInfo().getHeadingInfo()));
      ETG_TRACE_COMP(("####traceCmd_getCurrentCompassHeadingEND####"));
   }
   else
   {
      ETG_TRACE_COMP(("_sNavMiddleware is not initialized"));
   }
}


ETG_I_CMD_DEFINE((traceCmd_getCurrentAvoidAreas, "GetCurrentAvoidAreas"))
void TraceCommandHandler::traceCmd_getCurrentAvoidAreas()
{
   ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_getCurrentAvoidAreas()"));

   const navmiddleware::AvoidAreaInfos& avoidAreaInfos = _sNavMiddleware->getAvoidAreaInfos();
   const ::std::vector<navmiddleware::AvoidAreaInfo>& infos = avoidAreaInfos.getAvoidAreaInfos();
   ::std::vector<navmiddleware::AvoidAreaInfo>::const_iterator index = infos.begin();
   unsigned int numberOfAvoidAreas = 0;

   if (_sNavMiddleware)
   {
      const int NUMBER_OF_SIZE = 5;
      const int NUMBER_OF_OPTION = 2;
      std::string size_string[NUMBER_OF_SIZE] = { "LARGE", "MEDIUM_LARGE", "MEDIUM", "SMALL_MEDIUM", "SMALL" };
      std::string option_string[NUMBER_OF_OPTION] = { "AVOID_AREA_OPTIONS__NONE", "AVOID_AREA_OPTIONS__USE_MOTORWAYS" };
      std::string size;
      std::string option;
      ETG_TRACE_COMP(("Number of Current Avoid Areas: %d\n", std::distance(infos.begin(), infos.end())));
      for (index; index != infos.end(); index++)
      {
         numberOfAvoidAreas++;
         for (int i = 0; i < NUMBER_OF_SIZE; i++)
         {
            if (index->getSize() == i)
            {
               size = size_string[i];
            }
         }
         for (int i = 0; i < NUMBER_OF_OPTION; i++)
         {
            if (index->getAvoidAreaOptions() == i)
            {
               option = option_string[i];
            }
         }
         std::stringstream stream;
         stream << "AvoidArea" << numberOfAvoidAreas
                << ": Name= " << index->getName()
                << ", Size= " << size
                << ", MotorwayOption= " << option
                << ", ID= " << index->getId()
                << ", Scale= " << index->getScale()
                << ", EdgeLength= " << index->getEdgeLength()
                << ", Coordinates= " << index->getCenterGeoCoordinate().toString() << std::endl;
         ETG_TRACE_COMP(("%s", stream.str().c_str()));
      }
      ETG_TRACE_COMP(("####traceCmd_getCurrentAvoidAreasEND####"));
   }
   else
   {
      ETG_TRACE_COMP(("_sNavMiddleware is not initialized"));
   }
}


ETG_I_CMD_DEFINE((traceCmd_setAvoidAreas, "SetAvoidAreas %s", ETG_I_STRING coordinates))
void TraceCommandHandler::traceCmd_setAvoidAreas(std::string coordinates)
{
   //To be implemented
}


ETG_I_CMD_DEFINE((traceCmd_getLaneGuidanceInfo, "GetLaneGuidanceInfo"))
void TraceCommandHandler::traceCmd_getLaneGuidanceInfo()
{
   ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_getLaneGuidanceInfo()"));
   if (_sNavMiddleware)
   {
      const std::vector<navmiddleware::LaneInfos::LaneInfo>& laneList = _sNavMiddleware->getLaneInfos().getLaneInfos();
      ::std::vector<navmiddleware::LaneInfos::LaneInfo>::const_iterator index = laneList.begin();
      int laneIndex = 0;
      std::string laneStatus;
      std::string laneHOV;
      ETG_TRACE_COMP(("Number of Lanes: %d\n", std::distance(laneList.begin(), laneList.end())));
      for (index; index != laneList.end(); index++)
      {
         switch (index->laneStatus)
         {
            case 0:
               laneStatus = "Lane Status: LaneStatus__UNKNOWN";
               break;
            case 1:
               laneStatus = "Lane Status: LaneStatus__FORBIDDEN";
               break;
            case 2:
               laneStatus = "Lane Status: LaneStatus__NOT_RECOMMENDED";
               break;
            case 3:
               laneStatus = "Lane Status: LaneStatus__RECOMMENDED";
               break;
            default:
               break;
         }
         if (index->isHOVLane)
         {
            laneHOV = "TRUE";
         }
         else
         {
            laneHOV = "FALSE";
         }
         std::stringstream stream;
         stream << "LaneIndex=" << laneIndex
                << ", LaneStatus= " << laneStatus
                << ", CombinedFollowingLaneDirection= " << index->combinedFollowingLaneDirection
                << ", GuidedFollowingLaneDirection= " << index->guidedFollowingLaneDirection
                << ", isHOVLane= " << laneHOV
                << ", LaneOccupancy= " << index->laneOccupancy << std::endl;
         ETG_TRACE_COMP(("%s", stream.str().c_str()));
         laneIndex++;
      }
      ETG_TRACE_COMP(("####traceCmd_getLaneGuidanceInfoEND####"));
   }
   else
   {
      ETG_TRACE_COMP(("_sNavMiddleware is not initialized"));
   }
}


ETG_I_CMD_DEFINE((traceCmd_getCurrentAddress, "GetCurrentAddress"))
void TraceCommandHandler::traceCmd_getCurrentAddress()
{
   ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_getCurrentAddress()"));
   if (_sNavMiddleware)
   {
      std::stringstream stream;
      const navmiddleware::PositionInfo& positionInfo = _sNavMiddleware->getPositionInfo();
      stream << "StreetNumber=" << positionInfo.getCurrentStreetNumber()
             << ", Street= " << positionInfo.getCurrentStreet()
             << ", Intersection= " << positionInfo.getCurrentIntersection()
             << ", State= " << positionInfo.getCurrentState()
             << ", City= " << positionInfo.getCurrentCity()
             << ", Country= " << positionInfo.getCurrentCountry() << std::endl;
      ETG_TRACE_COMP(("%s", stream.str().c_str()));
      ETG_TRACE_COMP(("####traceCmd_getCurrentAddressEND####"))
   }
   else
   {
      ETG_TRACE_COMP(("_sNavMiddleware is not initialized"));
   }
}


ETG_I_CMD_DEFINE((traceCmd_getAddressFromGPSCoordinates, "GetAddressFromGPSCoordinates %s", ETG_I_STRING coordinates))
void TraceCommandHandler::traceCmd_getAddressFromGPSCoordinates(std::string coordinates)
{
   using namespace navmiddleware;
   if (_sNavMiddleware)
   {
      int pos = coordinates.find(',');
      double latitude = atof(coordinates.substr(0, pos).c_str());
      double longitude = atof(coordinates.substr(pos + 1).c_str());
      const navmiddleware::GeoCoordinateDegree coordinatesAvoidArea(latitude, longitude);
      _sNavMiddleware->setLocationWithCoordinates(coordinatesAvoidArea.getLatitude(), coordinatesAvoidArea.getLongitude());
      // Check for the location data
      _sNavMiddleware->requestLocationAttributes();
      //Wait for MIDW to return the checked result for coordinates if they exist or not
      usleep(10000);
      if (_sInfoStore->getAddressCoordinateErrorFlag())
      {
         ETG_TRACE_COMP(("Data does not exist"));
         _sInfoStore->setAddressCoordinateErrorFlag(false);
      }
      else
      {
         ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_getAddressFromGPSCoordinates() %f %f", coordinatesAvoidArea.getLatitude(), coordinatesAvoidArea.getLongitude()));
         _sNavMiddleware->requestAddressAttributesForCoordinate(coordinatesAvoidArea);
         //Wait for MIDW to return the address from input coordinates
         sleep(2);

         std::stringstream stream;
         std::string country = "", state = "", city = "", street = "", houseNumber = "", crossStreet = "";

         const navmiddleware::LocationAttributeInfos& locationAttributeInfos = _sNavMiddleware->getAddressAttributesOfCoordinate();
         const ::std::vector<const LocationAttributeInfos::LocationAttributeInfo*>& infos = locationAttributeInfos.getAttributeInfos();
         ::std::vector<const LocationAttributeInfos::LocationAttributeInfo*>::const_iterator infosIterator = infos.begin();
         for (; infosIterator != infos.end(); ++infosIterator)
         {
            if ((NULL != *infosIterator) && (LocationAttributeInfos::LocationAttributeInfo::ATTRIBUTE_TYPE__ADDRESS_DETAILS == (*infosIterator)->m_type))
            {
               const LocationAttributeInfos::AddressDetailInfo* addressDetailInfo = locationAttributeInfos.getAddressDetailInfos();
               if (NULL != addressDetailInfo)
               {
                  country = addressDetailInfo->getAddressValue(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__COUNTRY_NAME);
                  state = addressDetailInfo->getAddressValue(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__STATE_NAME);
                  city = addressDetailInfo->getAddressValue(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__CITY_NAME);
                  street = addressDetailInfo->getAddressValue(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__STREET_NAME);
                  houseNumber = addressDetailInfo->getAddressValue(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__HOUSE_NUMBER);
                  crossStreet = addressDetailInfo->getAddressValue(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__CROSS_STREET_NAME);
               }
            }
         }

         stream << "Country=" << country
                << ", State=" << state
                << ", City=" << city
                << ", Street=" << street
                << ", HouseNumber=" << houseNumber
                << ", CrossStreetName=" << crossStreet << std::endl;
         ETG_TRACE_COMP(("%s", stream.str().c_str()));
         ETG_TRACE_COMP(("####traceCmd_getAddressFromGPSCoordinatesEND####"));
      }
   }
   else
   {
      ETG_TRACE_COMP(("_sNavMiddleware is not initialized"));
   }
}


ETG_I_CMD_DEFINE((traceCmd_getDestDescription, "GetDestDescription"))
void TraceCommandHandler::traceCmd_getDestDescription()
{
   ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_getDestDescription()"));
   if (_sNavMiddleware)
   {
      if (_sNavMiddleware->isGuidanceActive())
      {
         const navmiddleware::DestinationInfos& routeInfoInfos = _sNavMiddleware->getDestinationInfos();
         const std::vector< navmiddleware::DestinationInfos::DestinationInfo>& routeInfoVector = routeInfoInfos.getDestinationInfos();
         std::vector<navmiddleware::DestinationInfos::DestinationInfo>::const_reverse_iterator info = routeInfoVector.rbegin();
         if (routeInfoVector.size() == 1)
         {
            _sNavMiddleware->requestAddressAttributesForCoordinate(navmiddleware::GeoCoordinateDegree(info->m_latitude, info->m_longitude));
            //Wait for MIDW to return the address from input coordinates
            sleep(2);
            //ToDo: getAddressDetailsOfCoordinate() deprecated. getAddressAttributesOfCoordinate() to be used instead
            //ETG_TRACE_COMP(("DestinationPosition: Latitude: %f, Longitude: %f, DestinationName: %s", info->m_latitude, info->m_longitude, _sNavMiddleware->getAddressDetailsOfCoordinate().getLocationDetail(navmiddleware::LocationDetailInfos::LocationDetailInfo::TYPE_ADDRESS).c_str()));
            ETG_TRACE_COMP(("No Waypoints Available"));
         }
         else
         {
            int waypointIndex = 1;
            _sNavMiddleware->requestAddressAttributesForCoordinate(navmiddleware::GeoCoordinateDegree(info->m_latitude, info->m_longitude));
            //Wait for MIDW to return the address from input coordinates
            sleep(2);
            //ToDo: getAddressDetailsOfCoordinate() deprecated. getAddressAttributesOfCoordinate() to be used instead
            //ETG_TRACE_COMP(("DestinationPosition: Latitude: %f, Longitude: %f, DestinationName: %s", info->m_latitude, info->m_longitude, _sNavMiddleware->getAddressDetailsOfCoordinate().getLocationDetail(navmiddleware::LocationDetailInfos::LocationDetailInfo::TYPE_ADDRESS).c_str()));
            std::advance(info, 1);
            for (info; info != routeInfoVector.rend(); info++)
            {
               _sNavMiddleware->requestAddressAttributesForCoordinate(navmiddleware::GeoCoordinateDegree(info->m_latitude, info->m_longitude));
               //Wait for MIDW to return the address from input coordinates
               sleep(2);
               //ToDo: getAddressDetailsOfCoordinate() deprecated. getAddressAttributesOfCoordinate() to be used instead
               //ETG_TRACE_COMP(("Waypoint%dPosition: Latitude: %f, Longitude: %f, Waypoint%dName: %s", waypointIndex, info->m_latitude, info->m_longitude, waypointIndex, _sNavMiddleware->getAddressDetailsOfCoordinate().getLocationDetail(navmiddleware::LocationDetailInfos::LocationDetailInfo::TYPE_ADDRESS).c_str()));
               waypointIndex++;
            }
         }
         ETG_TRACE_COMP(("####traceCmd_getDestDescriptionEND####"));
      }
      else
      {
         ETG_TRACE_COMP(("GetDestinationDescription not possible. RG is not active"));
      }
   }
   else
   {
      ETG_TRACE_COMP(("_sNavMiddleware is not initialized"));
   }
}


ETG_I_CMD_DEFINE((traceCmd_getCurrentSystemLanguage, "GetCurrentSystemLanguage"))
void TraceCommandHandler::traceCmd_getCurrentSystemLanguage()
{
   ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_getCurrentSystemLanguage()"));
   if (_sNavMiddleware)
   {
      ETG_TRACE_COMP(("GetCurrentSystemLanguage status %s", _sInfoStore->getSystemLanguageISOCode().c_str()));
      ETG_TRACE_COMP(("####traceCmd_getCurrentSystemLanguageEND####"));
   }
   else
   {
      ETG_TRACE_ERR(("_sNavMiddleware not initialized"));
   }
}


ETG_I_CMD_DEFINE((traceCmd_getCurrentDistanceUnit, "GetCurrentDistanceUnit"))
void TraceCommandHandler::traceCmd_getCurrentDistanceUnit()
{
   ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_getCurrentDistanceUnit()"));
   std::string distanceUnitSystem = "mi";
   if (_sInfoStore->getDistanceUnitSystem() == InfoStoreBase::DISTANCEUNITSYSTEM_METRIC)
   {
      distanceUnitSystem = "km";
   }
   else
   {
      distanceUnitSystem = "mi";
   }
   if (_sNavMiddleware)
   {
      ETG_TRACE_COMP(("GetCurrentDistanceUnit is %s", distanceUnitSystem.c_str()));
      ETG_TRACE_COMP(("####traceCmd_getCurrentDistanceUnitEND####"));
   }
   else
   {
      ETG_TRACE_ERR(("_sNavMiddleware not initialized"));
   }
}


ETG_I_CMD_DEFINE((traceCmd_getCurrentSpeed, "GetCurrentSpeed"))
void TraceCommandHandler::traceCmd_getCurrentSpeed()
{
   ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_getCurrentSpeed()"));
   if (_sNavMiddleware)
   {
      ETG_TRACE_COMP(("GetCurrentSpeed Value %s", _sNavMiddleware->getPositionStatusInfo().getSpeedInfoAsString().c_str()));
      ETG_TRACE_COMP(("####traceCmd_getCurrentSpeedEND####"));
   }
   else
   {
      ETG_TRACE_ERR(("_sNavMiddleware not initialized"));
   }
}


ETG_I_CMD_DEFINE((traceCmd_getMapOrientation, "GetMapOrientation "))
void TraceCommandHandler::traceCmd_getMapOrientation()
{
   ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_getMapOrientation()"));
   if (_sNavMiddleware)
   {
      ETG_TRACE_COMP(("GetMapOrientation status %s", _sNavMiddleware->getMapSettings().getMapOrientation() ? "Heading UP" : "North UP"));
      ETG_TRACE_COMP(("####traceCmd_getMapOrientationEND####"));
   }
   else
   {
      ETG_TRACE_ERR(("_sNavMiddleware not initialized"));
   }
}


ETG_I_CMD_DEFINE((traceCmd_setMapOrientation, "SetMapOrientation %d %d %d", int, int, int))
void TraceCommandHandler::traceCmd_setMapOrientation(int mapViewId, int mapOrientation, int mapPitch)
{
   ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_SetMapOrientation()"));
   if (_sNavMiddleware)
   {
      navmiddleware::MapRepresentation mapRepresentation((navmiddleware::settings::MapOrientation) mapOrientation, (navmiddleware::MapPitch) mapPitch);
      _sNavMiddleware->setMapRepresentation((navmiddleware::MapViewId)mapViewId, mapRepresentation);
      ETG_TRACE_COMP(("####traceCmd_setMapOrientationEND####"));
   }
   else
   {
      ETG_TRACE_ERR(("TraceCommandHandler::traceCmd_setMapOrientation() _sNavMiddleware not initialized"));
   }
}


ETG_I_CMD_DEFINE((traceCmd_getCurrentOrthoMetricHeight, "GetCurrentOrthometricHeight"))
void TraceCommandHandler::traceCmd_getCurrentOrthoMetricHeight()
{
   ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_getCurrentOrthometricHeight()"));
   if (_sNavMiddleware)
   {
      ETG_TRACE_COMP(("GetCurrentOrthometricHeight status %s", _sNavMiddleware->getGnssDataInfo().getAltitudeAsString().c_str()));
      ETG_TRACE_COMP(("####traceCmd_getCurrentOrthometricHeightEND####"));
   }
   else
   {
      ETG_TRACE_ERR(("_sNavMiddleware not initialized"));
   }
}


ETG_I_CMD_DEFINE((traceCmd_getCurrentETA, "GetCurrentETA"))
void TraceCommandHandler::traceCmd_getCurrentETA()
{
   ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_getCurrentETA()"));
   if (_sNavMiddleware)
   {
      const navmiddleware::ArrivalInfos& arrivalInfos = _sNavMiddleware->getArrivalInfos();
      if (_sNavMiddleware->isGuidanceActive())
      {
         if ((arrivalInfos.getCurrentDestinationIndex() < arrivalInfos.getArrivalInfos().size()))
         {
            if ((_sInfoStore->getEstimatedTimeType() == InfoStoreBase::DESTINATION) || (arrivalInfos.getArrivalInfos().size() == ITEM1))
            {
               ETG_TRACE_COMP(("GetCurrentETA status %s", arrivalInfos.getArrivalInfos().at(arrivalInfos.getArrivalInfos().size() - 1).m_estimatedTimeOfArrival.c_str()));
            }
            else
            {
               ETG_TRACE_COMP(("GetCurrentETA status %s", arrivalInfos.getArrivalInfos()[arrivalInfos.getCurrentDestinationIndex()].m_estimatedTimeOfArrival.c_str()));
            }
         }
         else
         {
            ETG_TRACE_COMP(("Invalid Data"));
         }
      }
      else
      {
         ETG_TRACE_COMP(("Please start Route Guidance to get current ETA"));
      }
      ETG_TRACE_COMP(("####traceCmd_getCurrentETAEND####"));
   }
   else
   {
      ETG_TRACE_ERR(("_sNavMiddleware not initialized"));
   }
}


ETG_I_CMD_DEFINE((traceCmd_getCurrentPosition, "GetCurrentPosition"))
void TraceCommandHandler::traceCmd_getCurrentPosition()
{
   ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_getCurrentPosition()"));
   if (_sNavMiddleware)
   {
      ETG_TRACE_COMP(("GetCurrentPosition status %s", _sNavMiddleware->getPositionInfo().getPositionInfo().c_str()));
      ETG_TRACE_COMP(("Latitude %f,Longitude  %f, Heading %f", _sNavMiddleware->getPositionStatusInfo().getLatitude(), _sNavMiddleware->getPositionStatusInfo().getLongitude(), _sNavMiddleware->getPositionStatusInfo().getHeadingInfo()));
      ETG_TRACE_COMP(("####traceCmd_getCurrentPositionEND####"));
   }
   else
   {
      ETG_TRACE_ERR(("_sNavMiddleware not initialized"));
   }
}


ETG_I_CMD_DEFINE((traceCmd_setZoomIn, "SetZoomInValue %d %d", unsigned int, unsigned int))
void TraceCommandHandler::traceCmd_setZoomIn(unsigned int mapViewId, unsigned int stepcount)
{
   ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_setZoomIn()"));
   if (_sNavMiddleware)
   {
      _sNavMiddleware->zoomInStep((navmiddleware::MapViewId)mapViewId, stepcount);
      ETG_TRACE_COMP(("####traceCmd_setZoomInEND####"));
   }
   else
   {
      ETG_TRACE_ERR(("_sNavMiddleware not initialized"));
   }
}


ETG_I_CMD_DEFINE((traceCmd_setZoomOut, "SetZoomOutValue %d %d", unsigned int, unsigned int))
void TraceCommandHandler::traceCmd_setZoomOut(unsigned int mapViewId, unsigned int stepcount)
{
   ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_setZoomOut()"));
   if (_sNavMiddleware)
   {
      _sNavMiddleware->zoomOutStep((navmiddleware::MapViewId)mapViewId, stepcount);
      ETG_TRACE_COMP(("####traceCmd_setZoomOutEND####"));
   }
   else
   {
      ETG_TRACE_ERR(("_sNavMiddleware not initialized"));
   }
}


ETG_I_CMD_DEFINE((traceCmd_getDistanceToDestination, "GetDistanceToDestination"))
void TraceCommandHandler::traceCmd_getDistanceToDestination()
{
   ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_getDistanceToDestination()"));
   if (_sNavMiddleware->isGuidanceActive())
   {
      const navmiddleware::ArrivalInfos& arrivalInfos = _sNavMiddleware->getArrivalInfos();
      if ((arrivalInfos.getCurrentDestinationIndex() < arrivalInfos.getArrivalInfos().size()))
      {
         ETG_TRACE_COMP(("GetDistanceToDestination to the final destination %s", arrivalInfos.getArrivalInfos().at(arrivalInfos.getArrivalInfos().size() - 1).m_distanceToDestination.c_str()));
         ETG_TRACE_COMP(("GetDistanceToDestination to the immediate Way point %s", arrivalInfos.getArrivalInfos()[arrivalInfos.getCurrentDestinationIndex()].m_distanceToDestination.c_str()));
         ETG_TRACE_COMP(("####traceCmd_getDistanceToDestinationEND####"));
      }
      else
      {
         ETG_TRACE_ERR(("_sNavMiddleware not initialized"));
      }
   }
   else
   {
      ETG_TRACE_COMP(("Please start Route Guidance to get current distance to destination"));
   }
}


ETG_I_CMD_DEFINE((traceCmd_startDemoMode, "StartDemoMode"))
void TraceCommandHandler::traceCmd_startDemoMode()
{
   ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_startDemoMode()"));
   if (_sNavMiddleware)
   {
      if (_sNavMiddleware->isDemoModeActive())
      {
         ETG_TRACE_COMP(("DEMO MODE IS ALREADY ACTIVE"));
      }
      else
      {
         _sNavMiddleware->activateDemoMode();
      }
      ETG_TRACE_COMP(("####traceCmd_startDemoModeEND####"));
   }
   else
   {
      ETG_TRACE_ERR(("_sNavMiddleware not initialized"));
   }
}


ETG_I_CMD_DEFINE((traceCmd_stopDemoMode, "StopDemoMode"))
void TraceCommandHandler::traceCmd_stopDemoMode()
{
   ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_stopDemoMode()"));
   if (_sNavMiddleware)
   {
      if (_sNavMiddleware->isDemoModeActive())
      {
         _sNavMiddleware->deactivateDemoMode();
      }
      else
      {
         ETG_TRACE_COMP(("DEMO MODE IS ALREADY INACTIVE"));
      }
      ETG_TRACE_COMP(("####traceCmd_stopDemoModeEND####"));
   }
   else
   {
      ETG_TRACE_ERR(("_sNavMiddleware not initialized"));
   }
}


ETG_I_CMD_DEFINE((traceCmd_increaseDemoModeSpeed, "IncreaseDemoModeSpeed"))
void TraceCommandHandler::traceCmd_increaseDemoModeSpeed()
{
   const bool guidanceActive = _sNavMiddleware->isGuidanceActive();
   const bool demoModeActive = _sNavMiddleware->isDemoModeActive();

   if (demoModeActive)
   {
      if (_demoModeSpeed < DEMO_MODE_SPEED_MAX)
      {
         _demoModeSpeed += DEMO_MODE_SPEED_STEP;
         _sNavMiddleware->setDemoModeSpeed(_demoModeSpeed);

         if (guidanceActive)
         {
            ETG_TRACE_COMP(("Demo Mode speed was set to: %d", _demoModeSpeed));
         }
         else
         {
            ETG_TRACE_COMP(("RG not active. Demo Mode speed was set to: %d", _demoModeSpeed));
         }
      }
      else
      {
         if (guidanceActive)
         {
            ETG_TRACE_COMP(("Demo mode is at its maximum speed: %d", _demoModeSpeed));
         }
         else
         {
            ETG_TRACE_COMP(("RG not active. Demo mode is at its maximum speed: %d", _demoModeSpeed));
         }
      }
   }
   else
   {
      ETG_TRACE_COMP(("Demo mode is off. Speed is not settable"));
   }
   ETG_TRACE_COMP(("####traceCmd_increaseDemoModeSpeedEND####"));
}


ETG_I_CMD_DEFINE((traceCmd_decreaseDemoModeSpeed, "DecreaseDemoModeSpeed"))
void TraceCommandHandler::traceCmd_decreaseDemoModeSpeed()
{
   const bool guidanceActive = _sNavMiddleware->isGuidanceActive();
   const bool demoModeActive = _sNavMiddleware->isDemoModeActive();

   if (demoModeActive)
   {
      if (_demoModeSpeed >= DEMO_MODE_SPEED_MIN)
      {
         _demoModeSpeed -= DEMO_MODE_SPEED_STEP;
         _sNavMiddleware->setDemoModeSpeed(_demoModeSpeed);

         if (guidanceActive)
         {
            ETG_TRACE_COMP(("Demo Mode speed was set to: %d", _demoModeSpeed));
         }
         else
         {
            ETG_TRACE_COMP(("RG not active. Demo Mode speed was set to: %d", _demoModeSpeed));
         }
      }
      else
      {
         if (guidanceActive)
         {
            ETG_TRACE_COMP(("Demo mode is at its minimum speed: %d", _demoModeSpeed));
         }
         else
         {
            ETG_TRACE_COMP(("RG not active. Demo mode is at its minimum speed: %d", _demoModeSpeed));
         }
      }
   }
   else
   {
      ETG_TRACE_COMP(("Demo mode is off. Speed is not settable"));
   }
   ETG_TRACE_COMP(("####traceCmd_decreaseDemoModeSpeedEND####"));
}


ETG_I_CMD_DEFINE((traceCmd_getDemoModeSpeed, "GetDemoModeSpeed"))
void TraceCommandHandler::traceCmd_getDemoModeSpeed()
{
   const bool demoModeActive = _sNavMiddleware->isDemoModeActive();

   if (demoModeActive)
   {
      ETG_TRACE_COMP(("Demo Mode Speed: %d", _demoModeSpeed))
   }
   else
   {
      ETG_TRACE_COMP(("Demo mode is off. Speed is not gettable"));
   }
   ETG_TRACE_COMP(("####traceCmd_getDemoModeSpeedEND####"));
}


ETG_I_CMD_DEFINE((traceCmd_setDemoModeSpeed, "SetDemoModeSpeed %d", unsigned int))
void TraceCommandHandler::traceCmd_setDemoModeSpeed(unsigned int speed)
{
   const bool demoModeActive = _sNavMiddleware->isDemoModeActive();
   const bool guidanceActive = _sNavMiddleware->isGuidanceActive();

   if (demoModeActive)
   {
      _demoModeSpeed = speed;
      _sNavMiddleware->setDemoModeSpeed(_demoModeSpeed);
      if (guidanceActive)
      {
         ETG_TRACE_COMP(("Demo mode speed was set to: %d", _demoModeSpeed));
      }
      else
      {
         ETG_TRACE_COMP(("RG not active. Demo mode speed was set to: %d", _demoModeSpeed));
      }
   }
   else
   {
      ETG_TRACE_COMP(("Demo mode is off. Speed is not settable"));
   }
}


ETG_I_CMD_DEFINE((traceCmd_setDistanceUnit, "SetDistanceUnit %d", unsigned int))
void TraceCommandHandler::traceCmd_setDistanceUnit(unsigned int distanceUnit)
{
   POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(UnitSettingsStatusUpdateMsg)(distanceUnit, 0, 0)));
}


ETG_I_CMD_DEFINE((traceCmd_setCameraFromCoordinates, "SetCameraFromCoordinates %s", ETG_I_STRING coordinates))
void TraceCommandHandler::traceCmd_setCameraFromCoordinates(std::string coordinates)
{
   int pos1 = coordinates.find(',');
   int pos2 = coordinates.find(',', (pos1 + 1));
   int length = (pos2 - pos1) - 1;

   double latitude = atof(coordinates.substr(0, pos1).c_str());
   double longitude = atof(coordinates.substr((pos1 + 1), length).c_str());
   unsigned int userScale = atof(coordinates.substr(pos2 + 1).c_str());

   navmiddleware::ValidValue<int> scale(userScale);
   //ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_setCameraFromCoordinates() %f, %f, %d", latitude, longitude, scale));
   if (_sNavMiddleware)
   {
      _sNavMiddleware->centerMapToGeoPosition(navmiddleware::MAP_VIEW_ID__PRIMARY, navmiddleware::GeoCoordinateDegree(latitude, longitude), scale);
      ETG_TRACE_COMP(("####traceCmd_setCameraFromCoordinatesEND####"));
   }
   else
   {
      ETG_TRACE_ERR(("TraceCommandHandler::traceCmd_setCameraFromCoordinates() _sNavMiddleware not initialized"));
   }
}


ETG_I_CMD_DEFINE((traceCmd_pickInMap, "PickInMap"))
void TraceCommandHandler::traceCmd_pickInMap()
{
   ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_pickInMap()"));
   if (_sNavMiddleware)
   {
      navmiddleware::ValidValue<navmiddleware::PickingResultFilter> pickingResultFilter;
      _sNavMiddleware->pickInMap(navmiddleware::MAP_VIEW_ID__PRIMARY, _sInfoStore->getScreenWidth() / 2, _sInfoStore->getScreenHeight() / 2, pickingResultFilter);
      ETG_TRACE_COMP(("####traceCmd_pickInMapEND####"));
   }
   else
   {
      ETG_TRACE_ERR(("_sNavMiddleware not initialized"));
   }
}


ETG_I_CMD_DEFINE((traceCmd_getListImageProperties, "GetListImageProperties"))
void TraceCommandHandler::traceCmd_getListImageProperties()
{
   ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_getListImageProperties()"));
   if (_sNavMiddleware)
   {
      const navmiddleware::RouteListInfos& routeListInfos = _sNavMiddleware->getRouteListInfos();
      const std::vector< ::navmiddleware::RouteListInfos::RouteListElementInfo >& routeListElementInfos = routeListInfos.getRouteListElementInfos();
      ::std::vector<navmiddleware::RouteListInfos::RouteListElementInfo>::const_iterator info = routeListElementInfos.begin();

      for (unsigned int idx = 0; (info != routeListElementInfos.end()); ++idx, ++info)
      {
         if (routeListInfos.getRouteListElementInfos().size())
         {
            std::stringstream stream;
            unsigned int roadNumberImageInfo = 0;
            if (!info->m_roadNumberImageBlobs.empty())
            {
               roadNumberImageInfo = info->m_roadNumberImageBlobs[0].second;
            }

            stream
                  << "ListLine=" << idx
                  << ", RoadNumberIconSize=" << roadNumberImageInfo
                  << ", RoadName=" << info->m_roadInfo.c_str()
                  << ", Length=" << info->m_distanceToStartOfElement.c_str();
            ETG_TRACE_COMP(("TurnIcon=%d, %s", ETG_CENUM(TurnListIconType, info->m_turnListIconType), stream.str().c_str()));
         }
      }
      ETG_TRACE_COMP(("####traceCmd_getListImagePropertiesEND####"));
   }
   else
   {
      ETG_TRACE_ERR(("TraceCommandHandler::traceCmd_getListImageProperties() _sNavMiddleware not initialized"));
   }
}


//get current Map View
ETG_I_CMD_DEFINE((traceCmd_getCurrentMapView, "GetCurrentMapView"))
void TraceCommandHandler::traceCmd_getCurrentMapView()
{
   ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_getCurrentMapView()"));
   if (_sNavMiddleware)
   {
      navmiddleware::settings::MapView mapview = _sNavMiddleware->getChangeMapViewSettings().getMapView();
      if (mapview == navmiddleware::settings::MAPVIEW_2D)
      {
         ETG_TRACE_COMP(("MapView : MAPVIEW_2D"));
      }
      else if (mapview == navmiddleware::settings::MAPVIEW_2D_SPLIT)
      {
         ETG_TRACE_COMP(("MapView : MAPVIEW_2D_SPLIT"));
      }
      else if (mapview == navmiddleware::settings::MAPVIEW_3D)
      {
         ETG_TRACE_COMP(("MapView : MAPVIEW_3D"));
      }
      else if (mapview == navmiddleware::settings::MAPVIEW_3D_SPLIT)
      {
         ETG_TRACE_COMP(("MapView : MAPVIEW_3D_SPLIT"));
      }
      else
      {
         // do nothing
      }
      ETG_TRACE_COMP(("####traceCmd_getCurrentMapViewEND####"));
   }
   else
   {
      ETG_TRACE_ERR(("_sNavMiddleware not initialized"));
   }
}


//getcurrentCameraPosition
ETG_I_CMD_DEFINE((traceCmd_getCurrentCameraPosition, "GetCurrentCameraPosition"))
void TraceCommandHandler::traceCmd_getCurrentCameraPosition()
{
   ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_getCurrentCameraPosition()"));
   if (_sNavMiddleware)
   {
      std::stringstream stream;
      navmiddleware::settings::MapView mapview = _sNavMiddleware->getChangeMapViewSettings().getMapView();
      if (mapview == navmiddleware::settings::MAPVIEW_2D)
      {
         stream << "[Camera_1] Lat/lon: " << _sNavMiddleware->getPositionStatusInfo().getLatitude()
                << ", " << _sNavMiddleware->getPositionStatusInfo().getLongitude()
                << "; Heading: " << _sNavMiddleware->getPositionStatusInfo().getHeadingInfo()
                << "; CameraMode: 2D; Scale: " << _sNavMiddleware->getMapCameraInfos().getFormattedMapScale().c_str() << std::endl;
         ETG_TRACE_COMP(("%s", stream.str().c_str()));
         ETG_TRACE_COMP(("####traceCmd_getCurrentCameraPositionEND####"));
      }
      else if (mapview == navmiddleware::settings::MAPVIEW_3D)
      {
         stream << "[Camera_1] Lat/lon: " << _sNavMiddleware->getPositionStatusInfo().getLatitude()
                << ", " << _sNavMiddleware->getPositionStatusInfo().getLongitude()
                << "; Heading: " << _sNavMiddleware->getPositionStatusInfo().getHeadingInfo()
                << "; CameraMode: 3D; Scale: " << _sNavMiddleware->getMapCameraInfos().getFormattedMapScale().c_str() << std::endl;
         ETG_TRACE_COMP(("%s", stream.str().c_str()));
         ETG_TRACE_COMP(("####traceCmd_getCurrentCameraPositionEND####"));
      }
      else if (mapview == navmiddleware::settings::MAPVIEW_2D_SPLIT)
      {
         stream << "[Camera_1] Lat/lon: " << _sNavMiddleware->getPositionStatusInfo().getLatitude()
                << ", " << _sNavMiddleware->getPositionStatusInfo().getLongitude()
                << "; Heading: " << _sNavMiddleware->getPositionStatusInfo().getHeadingInfo()
                << "; CameraMode: 2D; Scale: " << _sNavMiddleware->getMapCameraInfos().getFormattedMapScale().c_str()
                << "; [Camera_2] Lat/lon: " << _sNavMiddleware->getPositionStatusInfo().getLatitude()
                << ", " << _sNavMiddleware->getPositionStatusInfo().getLongitude()
                << "; Heading: " << _sNavMiddleware->getPositionStatusInfo().getHeadingInfo()
                << "; CameraMode: 2D; Scale: " << _sNavMiddleware->getMapCameraInfos().getFormattedMapScale().c_str() << std::endl;
         ETG_TRACE_COMP(("%s", stream.str().c_str()));
         ETG_TRACE_COMP(("####traceCmd_getCurrentCameraPositionEND####"));
      }
      else
      {
         stream << "[Camera_1] Lat/lon: " << _sNavMiddleware->getPositionStatusInfo().getLatitude()
                << ", " << _sNavMiddleware->getPositionStatusInfo().getLongitude()
                << "; Heading: " << _sNavMiddleware->getPositionStatusInfo().getHeadingInfo()
                << "; CameraMode: 3D; Scale: " << _sNavMiddleware->getMapCameraInfos().getFormattedMapScale().c_str()
                << "; [Camera_2] Lat/lon: " << _sNavMiddleware->getPositionStatusInfo().getLatitude() << ", " << _sNavMiddleware->getPositionStatusInfo().getLongitude()
                << "; Heading: " << _sNavMiddleware->getPositionStatusInfo().getHeadingInfo()
                << "; CameraMode: 3D; Scale: " << _sNavMiddleware->getMapCameraInfos().getFormattedMapScale().c_str() << std::endl;
         ETG_TRACE_COMP(("%s", stream.str().c_str()));
         ETG_TRACE_COMP(("####traceCmd_getCurrentCameraPositionEND####"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("_sNavMiddleware not initialized"));
   }
}


ETG_I_CMD_DEFINE((traceCmd_setSpeedLockState, "SetSpeedLockState %d", bool))
void TraceCommandHandler::traceCmd_setSpeedLockState(bool speedLockState)
{
   ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_setSpeedLockState()"));

   EXT_bIsSpeedLockActive = speedLockState;
   if (EXT_bIsSpeedLockActive)
   {
      POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(DisplaySpeedLockPopuponSpellerReqMsg)()));
   }
   else
   {
      POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(HideSpeedLockPopuponSpellerReqMsg)()));
   }
}


ETG_I_CMD_DEFINE((traceCmd_setMapStreamState, "SetMapStreamState %d", bool))
void TraceCommandHandler::traceCmd_setMapStreamState(bool mapStreamState)
{
   ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_setMapStreamState()"));
   POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(StartStopMapStreamReqMsg)(REG_A, mapStreamState, false)));
}


ETG_I_CMD_DEFINE((traceCmd_setAutoZoomEnabled, "SetAutoZoomEnabled %d", bool))
void TraceCommandHandler::traceCmd_setAutoZoomEnabled(bool autoZoomEnabled)
{
   if (_sNavMiddleware)
   {
      navmiddleware::settings::MapSettings mapSettings = _sNavMiddleware->getMapSettings();
      mapSettings.setAutoZoomEnabled(autoZoomEnabled);
      ETG_TRACE_COMP(("TraceCommandHandler::traceCmd_setAutoZoomEnabled()"));
   }
   else
   {
      ETG_TRACE_ERR(("_sNavMiddleware not initialized"));
   }
}


#endif

#endif // HALL_TO_MDW_COM
