/**************************************************************************************
* @file         : SurfaceSynchronizationHandler.cpp
* @author       :
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) -2018 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include "HMIModelIncludes.h"
#include "EnvironmentUtils.h"
#include "SurfaceSynchronizationHandler.h"

#if !defined(_MSC_VER)
#include "../../di_trace/components/etg/etg.h"
#endif

#if defined(GEN3ARM) || defined(GEN3X86)
#define ILM_AVAILABLE
#endif

#if defined(ILM_AVAILABLE)
#include <ilm/ilm_control.h>
#endif

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#define ETG_I_TRACE_CHANNEL TR_TTFIS_APPHMI_NAVIGATION
#define ETG_I_TTFIS_CMD_PREFIX "APPHMI_NAVIGATION_"
#define ETG_I_FILE_PREFIX SurfaceSynchronizationHandler::
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/SurfaceSynchronizationHandler.cpp.trc.h"
#endif

SurfaceSynchronizationHandler* gSurfaceSynchronizationHandler = NULL;

const unsigned int SURFACEID_SYNC_SURFACE_NAVIGATION = 103;
const unsigned int MAP_LAYER_ID = 1000;

SurfaceSynchronizationHandler::SurfaceSynchronizationHandler()
   : _surfaceSynchronizationState(SURFACE_SYNCHRONIZATION_DEACTIVATED)
   , _numLayerResponsesReceived(0)
   , _manualSurfaceSync(false)
{
   _surfaceSynchronizationState = (surfaceSynchronizationState)getEnvVarAsInt(ENV_HMI_CFG_SURFACE_SYNCHRONIZATION_ENABLED, (int)SURFACE_SYNCHRONIZATION_WITHOUT_LAYER);

   gSurfaceSynchronizationHandler = this;

#if !defined(_MSC_VER)
   ETG_I_REGISTER_FILE();
#endif
}


SurfaceSynchronizationHandler::~SurfaceSynchronizationHandler()
{
}


bool SurfaceSynchronizationHandler::onCourierMessage(const Courier::LayerResMsg& oMsg)
{
   if (_surfaceSynchronizationState != SURFACE_SYNCHRONIZATION_DEACTIVATED)
   {
      if (!_surfaces.size())
      {
         return false;
      }
      ETG_TRACE_USR1(("SurfaceSynchronizationHandler::onCourierMessage(LayerResMsg(%d, %d, %d))", oMsg.GetSuccess(), _numLayerResponsesReceived + 1, _layers.size()));

      if (oMsg.GetSuccess())
      {
         ++_numLayerResponsesReceived;
         if (_numLayerResponsesReceived == _layers.size())
         {
            setSynchronizedSurfaces(_surfaces);

            // call client code to modify surfaces now
            ETG_TRACE_USR1(("Calling onPrepareSurfaces()"));
            onPrepareSurfaces();
         }
      }
      else
      {
         ETG_TRACE_ERR(("LayerResMsg returned with error! Calling surfacesPrepared() to at least unlock the screen again!"));
         surfacesPrepared();
      }
   }

   return true;
}


bool SurfaceSynchronizationHandler::startSurfaceSyncronization(bool manualSurfaceSync)
{
   _manualSurfaceSync = manualSurfaceSync;
   bool success = false;

   switch (_surfaceSynchronizationState)
   {
      case SurfaceSynchronizationHandler::SURFACE_SYNCHRONIZATION_WITH_LAYER:
      {
         std::vector<unsigned int> surfaces;
         surfaces.push_back(Enum_SURFACEID_MAIN_SURFACE_NAVIGATION);
         std::vector<unsigned int> layers;
         layers.push_back(LAYERID_APPHMI_NAVIGATION);
         success = startSurfaceSyncronization(surfaces, layers);
      }
      break;
      case SurfaceSynchronizationHandler::SURFACE_SYNCHRONIZATION_WITHOUT_LAYER:
      {
         std::vector<unsigned int> surfaces;
         surfaces.push_back(SURFACEID_SYNC_SURFACE_NAVIGATION);
         std::vector<unsigned int> layers;
         success = startSurfaceSyncronization(surfaces, layers);
      }
      break;
      case SurfaceSynchronizationHandler::SURFACE_SYNCHRONIZATION_DEACTIVATED:
         onPrepareSurfaces();
         break;
      default:
         break;
   }

   return success;
}


bool SurfaceSynchronizationHandler::startSurfaceSyncronization(const std::vector<unsigned int>& surfaces, const std::vector<unsigned int>& layers)
{
   if (_surfaceSynchronizationState != SURFACE_SYNCHRONIZATION_DEACTIVATED)
   {
      ETG_TRACE_USR1(("SurfaceSynchronizationHandler::startSurfaceSyncronization(surface %d, count %d)", surfaces[0], surfaces.size()));

      clear();
      _surfaces = surfaces;
      _layers = layers;

      if (_surfaces.size() && !_layers.size())
      {
         setSynchronizedSurfaces(_surfaces);

         if (!_manualSurfaceSync)
         {
            // call client code to modify surfaces now
            ETG_TRACE_USR1(("Calling onPrepareSurfaces()"));
            onPrepareSurfaces();
         }
      }
      else
      {
         // deactivate rendering for deactivated surfaces and wait for responses
         sendLayerRequestMessages(false, true);
      }
   }
   else
   {
      onPrepareSurfaces();
   }

   _manualSurfaceSync = false;

   return true;
}


void SurfaceSynchronizationHandler::surfacesPrepared()
{
   if (_surfaceSynchronizationState != SURFACE_SYNCHRONIZATION_DEACTIVATED)
   {
      if (!_surfaces.size())
      {
         return;
      }
      ETG_TRACE_USR1(("SurfaceSynchronizationHandler::surfacesPrepared()"));

      // activate rendering for deactivated surfaces
      sendLayerRequestMessages(true, false);

      // remove surface sync again
      removeSynchronizedSurfaces(_surfaces);

      clear();
   }
}


void SurfaceSynchronizationHandler::sendLayerRequestMessages(bool activate, bool subscribe)
{
   if (!_layers.size())
   {
      return;
   }
   ETG_TRACE_USR1(("SurfaceSynchronizationHandler::sendLayerRequestMessages(activate %d subscribe %d)", activate, subscribe));

   if (subscribe)
   {
      Courier::LayerResMsg::Subscribe(static_cast<Courier::ComponentId>(Courier::ComponentType::Model));
   }
   else
   {
      Courier::LayerResMsg::Unsubscribe(static_cast<Courier::ComponentId>(Courier::ComponentType::Model));
   }

   if (activate)
   {
      for (unsigned int i = 0; i < _layers.size(); ++i)
      {
         POST_MSG((COURIER_MESSAGE_NEW(Courier::LayerReqMsg)(_layers[i], true)));
      }
   }
   else
   {
      for (unsigned int i = 0; i < _layers.size(); ++i)
      {
         POST_MSG((COURIER_MESSAGE_NEW(Courier::LayerReqMsg)(_layers[i], false)));
      }
   }
}


void SurfaceSynchronizationHandler::clear()
{
   _surfaces.clear();
   _layers.clear();
   _numLayerResponsesReceived = 0;
}


bool SurfaceSynchronizationHandler::setSynchronizedSurfaces(std::vector<unsigned int>& surfaces)
{
   if (!surfaces.size())
   {
      return false;
   }
   ETG_TRACE_USR1(("SurfaceSynchronizationHandler::setSynchronizedSurfaces(surface %d, count %d)", surfaces[0], surfaces.size()));

#if defined(ILM_AVAILABLE)
   ilmErrorTypes err = ilm_setSynchronizedSurfaces(&surfaces[0], surfaces.size());
   if (err == ILM_SUCCESS)
   {
      err = ilm_commitChanges();
      if (err == ILM_SUCCESS)
      {
         return true;
      }
   }
   ETG_TRACE_ERR(("ilm_setSynchronizedSurfaces() failed with err %d", err));
#else
   PARAM_UNUSED(surfaces);
#endif
   return false;
}


bool SurfaceSynchronizationHandler::removeSynchronizedSurfaces(std::vector<unsigned int>& surfaces)
{
   if (!surfaces.size())
   {
      return false;
   }
   ETG_TRACE_USR1(("SurfaceSynchronizationHandler::removeSynchronizedSurfaces(surface %d, count %d)", surfaces[0], surfaces.size()));

#if defined(ILM_AVAILABLE)
   ilmErrorTypes err = ilm_removeSynchronizedSurfaces(&surfaces[0], surfaces.size());
   if (err == ILM_SUCCESS)
   {
      err = ilm_commitChanges();
      if (err == ILM_SUCCESS)
      {
         return true;
      }
   }
   ETG_TRACE_ERR(("ilm_removeSynchronizedSurfaces() failed with err %d", err));
#else
   PARAM_UNUSED(surfaces);
#endif
   return false;
}


#if !defined(_MSC_VER)

ETG_I_CMD_DEFINE((traceCmd_startSurfaceSynchronization, "StartSurfaceSynchronization"))
void SurfaceSynchronizationHandler::traceCmd_startSurfaceSynchronization()
{
   ETG_TRACE_COMP(("SurfaceSynchronizationHandler::traceCmd_startSurfaceSynchronization()"));

   if (gSurfaceSynchronizationHandler)
   {
      // do surface synchronization
      std::vector<unsigned int> surfaces;
      surfaces.push_back(Enum_SURFACEID_MAIN_SURFACE_NAVIGATION);
      std::vector<unsigned int> layers;
      layers.push_back(LAYERID_APPHMI_NAVIGATION);
      gSurfaceSynchronizationHandler->startSurfaceSyncronization(surfaces, layers);
   }
   else
   {
      ETG_TRACE_ERR(("gSurfaceSynchronizationHandler not initialized"));
   }
}


ETG_I_CMD_DEFINE((traceCmd_surfacesPrepared, "SurfacesPrepared"))
void SurfaceSynchronizationHandler::traceCmd_surfacesPrepared()
{
   ETG_TRACE_COMP(("SurfaceSynchronizationHandler::traceCmd_surfacesPrepared()"));

   if (gSurfaceSynchronizationHandler)
   {
      gSurfaceSynchronizationHandler->surfacesPrepared();
   }
   else
   {
      ETG_TRACE_ERR(("gSurfaceSynchronizationHandler not initialized"));
   }
}


void SurfaceSynchronizationHandler::setMapLayerVisibility(const bool& isVisible)
{
   ETG_TRACE_USR4(("SurfaceSynchronizationHandler::setMapLayerVisibility(), isVisible : %d", isVisible));

   ilmErrorTypes err = ilm_layerSetVisibility(MAP_LAYER_ID, isVisible);
   if (err == ILM_SUCCESS)
   {
      err = ilm_commitChanges();
      if (err != ILM_SUCCESS)
      {
         ETG_TRACE_ERR(("ilm_commitChanges() failed with err %d", err));
      }
   }
   else
   {
      ETG_TRACE_ERR(("ilm_layerSetVisibility() failed with err %d", err));
   }
}


#endif
