/**************************************************************************************
* @file         : PosWGS84.h
* @author       :
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) -2018 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#ifndef POSITIONWGS84_H
#define POSITIONWGS84_H 1

#include <vector>
#include <sstream>
#include <math.h>
#include "NavMiddleware.h"

/**
* Simple base data class for wgs84 coordinates
*/
template <typename T>
class PosWGS84
{
   public:
      PosWGS84()
         : _longitude(0)
         , _latitude(0) {}

      PosWGS84(T longitude, T latitude)
         : _longitude(longitude)
         , _latitude(latitude) {}

      PosWGS84(const PosWGS84<T>& Pos)
         : _longitude(Pos._longitude)
         , _latitude(Pos._latitude) {}

      PosWGS84(const std::string& string)
      {
         std::stringstream ss(string);

         ss >> _longitude >> _latitude;
      }

      std::string toString() const
      {
         std::stringstream ss;

         ss << _longitude << " " << _latitude << " ";

         return ss.str();
      }

      void fromStringStream(std::stringstream& ss)
      {
         ss >> _longitude >> _latitude;
      }

      void reset()
      {
         _longitude = static_cast<T>(0.0);
         _latitude = static_cast<T>(0.0);
      }

      T _longitude;
      T _latitude;

      float length();
      bool operator == (const PosWGS84<T>& rhs) const;
      bool operator != (const PosWGS84<T>& rhs) const;
      PosWGS84<T>& operator = (const PosWGS84<T>& rhs);
      PosWGS84<T> operator - (const PosWGS84<T>& rhs) const;
      PosWGS84<T>& operator -= (const PosWGS84<T>& rhs);
};


typedef std::vector< PosWGS84<int> > PositionWGS84Vector;
typedef std::vector< PosWGS84<float> > PositionWGS84fVector;
typedef std::vector< PosWGS84<double> > PositionWGS84dVector;

template <typename T>
float PosWGS84<T>::length()
{
   return sqrt((float)(_longitude * _longitude) + (float)(_latitude * _latitude));
}


template <typename T>
bool PosWGS84<T>::operator == (const PosWGS84<T>& rhs) const
{
   return ((rhs._latitude == _latitude) && (rhs._longitude == _longitude));
}


template <typename T>
bool PosWGS84<T>::operator != (const PosWGS84<T>& rhs) const
{
   return !this->operator==(rhs);
}


template <typename T>
PosWGS84<T>& PosWGS84<T>::operator = (const PosWGS84<T>& rhs)
{
   if (this != &rhs)
   {
      _longitude = rhs._longitude;
      _latitude = rhs._latitude;
   }
   return *this;
}


template <typename T>
PosWGS84<T> PosWGS84<T>::operator - (const PosWGS84<T>& rhs) const
{
   PosWGS84<T> temp;
   temp._longitude = _longitude - rhs._longitude;
   temp._latitude = _latitude - rhs._latitude;
   return temp;
}


template <typename T>
PosWGS84<T>& PosWGS84<T>::operator -= (const PosWGS84<T>& rhs)
{
   _longitude -= rhs._longitude;
   _latitude -= rhs._latitude;
   return *this;
}


/**
* Simple base data class for wgs84 coordinates and heading in radians
*/
template <typename T>
class PosHeadingWGS84
{
   public:
      PosHeadingWGS84() : _heading(0.f) {}
      PosWGS84<T> _pos;
      float _heading;

      bool operator == (const PosHeadingWGS84<T>& rhs) const;
      bool operator != (const PosHeadingWGS84<T>& rhs) const;
};


template <typename T>
bool PosHeadingWGS84<T>::operator == (const PosHeadingWGS84<T>& rhs) const
{
   return ((rhs._pos == _pos) && (rhs._heading == _heading));
}


template <typename T>
bool PosHeadingWGS84<T>::operator != (const PosHeadingWGS84<T>& rhs) const
{
   return !this->operator==(rhs);
}


/**
* Simple base data class for wgs84 coordinates with a name string
*/
class NamedPosition
{
   public:
      NamedPosition() {}
      NamedPosition(const PosWGS84<double>& position, const std::string name)
         : _position(position)
         , _name(name) {}

      PosWGS84<double> _position;
      std::string _name;
};


typedef std::vector< NamedPosition > NamedPositionVector;

std::vector<navmiddleware::GeoCoordinateDegree> convert(const std::vector<PosWGS84<double> >& positionVector);

double calculateDistanceInMeter(double lat1d, double lon1d, double lat2d, double lon2d);

#endif // POSITIONWGS84_H
