/**************************************************************************************
* @file         : InfoStoreBase.h
* @author       : ECG5-Atchuta Sashank Kappagantu
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) -2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#ifndef INFO_STORE_BASE_H
#define INFO_STORE_BASE_H

#include <queue>
#include "ProjectBaseExchangingTextures.h"
#include "HMIModelDefines.h"
#include "InfoStoreCallbackManager.h"
#include "PosWGS84.h"
#include "AppHmi_NavigationTypes.h"
#include "CgiExtensions/DataBindingItem.hpp"
#include "AppHmi_NavigationStateMachine.h"

class InfoStoreBase : public InfoStoreCallbackManager
{
   public:
      InfoStoreBase();
      virtual ~InfoStoreBase();

      COURIER_MSG_MAP_BEGIN(TR_CLASS_APPHMI_NAVIGATION_COURIER_PAYLOAD_MODEL_COMP)
      COURIER_CASE_DUMMY_ENTRY()
      COURIER_MSG_MAP_END()

      enum TcuState
      {
         NAVI_TCU_STATE_UNKNOWN = 0,
         NAVI_TCU_STATE_CONNECTED,
         NAVI_SPI_STATE_NOT_AVAILABLE
      };

      enum DisplayAspectRatio
      {
         DISPLAY_ASPECT_RATIO_NOT_CONFIGURED = 0,
         DISPLAY_ASPECT_RATIO_15_BY_9 = 1,
         DISPLAY_ASPECT_RATIO_17_BY_9 = 2
      };

      enum DistanceUnitSystem
      {
         DISTANCEUNITSYSTEM_METRIC = 0,
         DISTANCEUNITSYSTEM_IMPERIAL,
         DISTANCEUNITSYSTEM_BRITISH_IMPERIAL
      };

      /*  enum TemperatureUnit
        {
           TEMPERATUREUNIT_UNKNOWN = 0,
           TEMPERATUREUNIT_DEG_F,
           TEMPERATUREUNIT_DEG_C
        };

        enum FuelUnit
        {
           FUELUNIT_UNKNOWN = 0,
           FUELUNIT_KM_PER_L,
           FUELUNIT_L_PER_100KM,
           FUELUNIT_US_PER_MPG,
           FUELUNIT_UK_PER_MPG
        };

        enum HandBrakeState
        {
           HANDBRAKESTATE_UNKNOWN = 0,
           HANDBRAKESTATE_OFF,
           HANDBRAKESTATE_ON
        };

        enum ReverseGearState
        {
           REVERSEGEARSTATE_UNKNOWN = 0,
           REVERSEGEARSTATE_OFF,
           REVERSEGEARSTATE_ON
        };*/

      enum SpeedLockState
      {
         SPEEDLOCKSTATE_OFF = 0,
         SPEEDLOCKSTATE_ON
      };

      enum EstimatedTimeMode
      {
         TRAVEL_TIME = 0,
         ARRIVAL_TIME
      };

      enum EstimatedTimeType
      {
         DESTINATION = 0,
         WAYPOINT
      };

      enum enOptionListElement
      {
         OPTION_LIST_FIRST_ELEMENT = 0,
         OPTION_LIST_SECOND_ELEMENT,
         OPTION_LIST_THIRD_ELEMENT,
         OPTION_LIST_FOURTH_ELEMENT,
         OPTION_LIST_FIFTH_ELEMENT,
         OPTION_LIST_MAX_ELEMENT
      };

      enum enAddressDetailInfoType
      {
         DETAIL_INFO_DEFAULT = 0,
         DETAIL_INFO_WAYPOINT = 1,
         DETAIL_INFO_HIERARCHICAL_ADDRESS = 2,
         DETAIL_INFO_GEOCOORDINATE = 3,
         DETAIL_INFO_MAPPICK = 4,
         DETAIL_INFO_TMC = 5,
         DETAIL_INFO_DESTINATION = 6,
         DETAIL_INFO_FAVORITES = 7
      };

      class DestinationMemoryStatus
      {
         public:
            DestinationMemoryStatus()
               : _category(navmiddleware::DESTINATION_MEMORY_ENTRY_CATEGORY__HOME)
               , _id()
               , _name() {}

            DestinationMemoryStatus(const navmiddleware::DestinationMemoryEntryCategory& category, const navmiddleware::DestinationMemoryEntry::Id& id, const std::string& name)
               : _category(category)
               , _id(id)
               , _name(name) {}

            bool operator==(const DestinationMemoryStatus& rhs) const
            {
               if (this->_category == rhs._category && this->_id == rhs._id && this->_name == rhs._name)
               {
                  return true;
               }
               return false;
            }

            navmiddleware::DestinationMemoryEntryCategory _category;
            navmiddleware::DestinationMemoryEntry::Id _id;
            std::string _name;
      };

      class DestinationMemoryHomeWorkInfo
      {
         public:
            DestinationMemoryHomeWorkInfo()
               : _status()
               , _id()
               , _destinationMemoryAttributesInfo()
               , _coordinates() {}

            DestinationMemoryHomeWorkInfo(bool status, const navmiddleware::DestinationMemoryEntry::Id& id,
                                          const navmiddleware::DestinationMemoryEntryAttributes& destinationMemoryAttributesInfo, std::vector<PosWGS84<double> > coordinates)
               : _status(status)
               , _id(id)
               , _destinationMemoryAttributesInfo(destinationMemoryAttributesInfo)
               , _coordinates(coordinates) {}

            bool _status;
            navmiddleware::DestinationMemoryEntry::Id _id;
            navmiddleware::DestinationMemoryEntryAttributes _destinationMemoryAttributesInfo;
            std::vector<PosWGS84<double> > _coordinates;
      };

      class CoordinatesToBeShownInMap
      {
         public:
            CoordinatesToBeShownInMap()
               : _scale(200)
               , _mapIconInfo()
               , _coordinates()
               , _visibleListInfo()
               , _locationInputType(navmiddleware::LOCATIONINPUTTYPE_ADDRESS) {}

            CoordinatesToBeShownInMap(std::vector<PosWGS84<double> > coordinates)
               : _scale(200)
               , _mapIconInfo()
               , _coordinates(coordinates)
               , _visibleListInfo()
               , _locationInputType(navmiddleware::LOCATIONINPUTTYPE_ADDRESS) {}

            CoordinatesToBeShownInMap(const navmiddleware::MapIconInfo& mapIconInfo, std::vector<PosWGS84<double> > coordinates)
               : _scale(200)
               , _mapIconInfo(mapIconInfo)
               , _coordinates(coordinates)
               , _visibleListInfo()
               , _locationInputType(navmiddleware::LOCATIONINPUTTYPE_ADDRESS) {}

            CoordinatesToBeShownInMap(const navmiddleware::VisibleListInfo& visibleListInfo, navmiddleware::LocationInputType locationInputType)
               : _scale(200)
               , _mapIconInfo()
               , _coordinates()
               , _visibleListInfo(visibleListInfo)
               , _locationInputType(locationInputType) {}

            int _scale;
            navmiddleware::MapIconInfo _mapIconInfo;
            std::vector<PosWGS84<double> > _coordinates;
            navmiddleware::VisibleListInfo _visibleListInfo;
            navmiddleware::LocationInputType _locationInputType;
      };

      /** Helper Class for representing route data - destination coordinates, route type and route options */
      class RouteToBeUsedForE2EGuidance
      {
         public:
            RouteToBeUsedForE2EGuidance()
               : _coordinates(),
                 _externalLocationVector(),
                 _routeType(),
                 _routeOptions() {}
            //Setter
            void setExternalLocationToBeUsedForE2EGuidance(const std::vector< navmiddleware::ExternalLocation >& extLocationVector)
            {
               _externalLocationVector = extLocationVector;
            }
            void setCoordinatesToBeUsedForE2EGuidance(const std::vector< PosWGS84<double> >& coordinatesToBeUsed)
            {
               _coordinates = coordinatesToBeUsed;
            }
            void setRouteCriteria(const navmiddleware::settings::RouteCriterion& routeCriteria)
            {
               _routeType.setRouteCriterion(routeCriteria);
            }
            void setRouteOptionsSettings(const navmiddleware::settings::RouteOptionsSettings& routeOptionsSettings);

            //getter
            std::vector< navmiddleware::ExternalLocation> getExternalLocationToBeUsedForE2EGuidance() const
            {
               return _externalLocationVector;
            }
            std::vector< PosWGS84<double> > getCoordinatesToBeUsedForE2EGuidance() const
            {
               return _coordinates;
            }
            navmiddleware::settings::RouteCriterion getRouteCriteria() const
            {
               return _routeType.getRouteCriterion();
            }
            navmiddleware::settings::RouteOptionsSettings getRouteOptionsSettings() const
            {
               return _routeOptions;
            }

         protected:
            std::vector< PosWGS84<double> > _coordinates;
            std::vector< navmiddleware::ExternalLocation> _externalLocationVector;
            navmiddleware::settings::RouteCriteriaSettings _routeType;
            navmiddleware::settings::RouteOptionsSettings _routeOptions;
      };

      //Structure to Hold Option List Item element details. For now, only enable state which is changing is added.
      //Later, if the case gets extended to dynamic text updation, even that can be added to the structure and extension will be simple.
      typedef struct _stOptionsListElement
      {
         bool OptionItemEnableState;

         //Constructor
         _stOptionsListElement()
         {
            OptionItemEnableState = 0;
         }
         ~_stOptionsListElement()
         {
            /*No Action*/
         }
         _stOptionsListElement(const _stOptionsListElement& oStruct)
         {
            //Call overloaded assignment operator
            *this = oStruct;
         }

         bool operator==(const _stOptionsListElement& rhs) const
         {
            if (this->OptionItemEnableState == rhs.OptionItemEnableState)
            {
               return true;
            }
            return false;
         }

         _stOptionsListElement& operator=(const _stOptionsListElement& oStruct)
         {
            if (&oStruct != this)
            {
               OptionItemEnableState = oStruct.OptionItemEnableState;
            }
            return *this;
         }

         void clear(void)
         {
            OptionItemEnableState = false;
         }
      } stOptionsListElement;

      //Class that stores Option List Item details Info.
      class StoredOptionsListItemInfo
      {
         public:
            StoredOptionsListItemInfo() :
               _OptionContextID(0) {}
            StoredOptionsListItemInfo(const StoredOptionsListItemInfo& oStruct)
            {
               //Call overloaded assignment operator
               *this = oStruct;
            }

            StoredOptionsListItemInfo& operator=(const StoredOptionsListItemInfo& oStruct)
            {
               if (&oStruct != this)
               {
                  _OptionContextID = oStruct._OptionContextID;
                  for (uint8_t count = 0; count < static_cast<int8_t>(OPTION_LIST_MAX_ELEMENT); ++count)
                  {
                     _OptionsListItem[count] = oStruct._OptionsListItem[count];
                  }
               }
               return *this;
            }

            bool operator==(const StoredOptionsListItemInfo& rhs) const
            {
               if ((this->_OptionContextID == rhs._OptionContextID) &&
                     (this->_OptionsListItem[OPTION_LIST_FIRST_ELEMENT] == rhs._OptionsListItem[OPTION_LIST_FIRST_ELEMENT]) &&
                     (this->_OptionsListItem[OPTION_LIST_SECOND_ELEMENT] == rhs._OptionsListItem[OPTION_LIST_SECOND_ELEMENT]) &&
                     (this->_OptionsListItem[OPTION_LIST_THIRD_ELEMENT] == rhs._OptionsListItem[OPTION_LIST_THIRD_ELEMENT]) &&
                     (this->_OptionsListItem[OPTION_LIST_FOURTH_ELEMENT] == rhs._OptionsListItem[OPTION_LIST_FOURTH_ELEMENT]) &&
                     (this->_OptionsListItem[OPTION_LIST_FIFTH_ELEMENT] == rhs._OptionsListItem[OPTION_LIST_FIFTH_ELEMENT]))
               {
                  return true;
               }
               return false;
            }

            //Clears all the contents of the Options List Item Structure.
            void clearOptionsListItems(void)
            {
               _OptionContextID = 0;
               for (uint8_t count = 0; count < static_cast<int8_t>(OPTION_LIST_MAX_ELEMENT); ++count)
               {
                  _OptionsListItem[count].clear();
               }
            }

            //Set Function to set details of individual element.
            void setOptionsListItemEnableInfo(const enOptionListElement enListElement, const bool enableStatusToSet);

            //Set function that sets the Context ID information.
            void setOptionsLisItemContextIDInfo(const Courier::UInt32 optionsContextID);

            //Get function to get individual element details.
            bool getOptionsListItemEnableInfo(const enOptionListElement enListElement) const;

            //Get function to get Context ID information.
            Courier::UInt32 getOptionsListItemContextInfo(void) const;

         private:
            Courier::UInt32 _OptionContextID; //Context ID info
            stOptionsListElement _OptionsListItem[OPTION_LIST_MAX_ELEMENT]; //Individual Options List Item Information.
      };

      class StoredRouteInfo
      {
         public:
            StoredRouteInfo()
               : _id()
               , _name() {}

            StoredRouteInfo(const navmiddleware::DestinationMemoryRoute::Id& id,
                            const ::std::string& name)
               : _id(id)
               , _name(name) {}

            bool operator==(const StoredRouteInfo& rhs) const
            {
               if (this->_id == rhs._id && this->_name == rhs._name)
               {
                  return true;
               }
               return false;
            }

            navmiddleware::DestinationMemoryRoute::Id _id;
            std::string _name;
      };

      struct MapOutRegionStatus
      {
         REGION regionId;
         bool mapOutStatus;
         MapOutRegionStatus()
         {
            regionId = MAX_REG;
            mapOutStatus = false;
         }
         MapOutRegionStatus(const REGION& id, const bool& status)
         {
            regionId = id;
            mapOutStatus = status;
         }
      };
      TcuState getTcuState() const
      {
         return _tcuState;
      }

      void setSortOrder(navmiddleware::LocationSortOrder sortOrder, bool forceUpdate = false);
      navmiddleware::LocationSortOrder getSortOrder() const
      {
         return _sortOrder;
      }

      unsigned char getSystemLanguageIndex() const
      {
         return _languageIndexSystem;
      }

      std::string getSystemLanguageISOCode() const
      {
         return _languageISOCodeSystem;
      }

      unsigned int getCountryCode() const
      {
         return _countryCode;
      }
      unsigned char getSDSLanguageIndex() const
      {
         return _languageIndexSds;
      }

      std::string getSDSLanguageISOCode() const
      {
         return _languageISOCodeSds;
      }

      unsigned char getClusterLanguageIndex() const
      {
         return _languageIndexCluster;
      }

      std::string getClusterLanguageISOCode() const
      {
         return _languageISOCodeCluster;
      }

      unsigned char getTrafficTextLanguageIndex() const
      {
         return _languageIndexTrafficText;
      }

      std::string getTrafficTextISOCode() const
      {
         return _languageISOCodeTrafficText;
      }

      DistanceUnitSystem getDistanceUnitSystem() const
      {
         return _distanceUnitSystem;
      }

      enLongitudeCoordinatesDirection getLongitudeCoordinatesDirection() const
      {
         return _longitudeCoordinatesDirection;
      }

      enLatitudeCoordinatesDirection getLatitudeCoordinatesDirection() const
      {
         return _latitudeCoordinatesDirection;
      }

      enCoordinatesDisplayType getCoordinatesDisplayType() const
      {
         return _coordinatesDisplayType;
      }

      //FuelUnit getFuelUnit() const
      //{
      //   return _fuelUnit;
      //}

      //TemperatureUnit getTemperatureUnit() const
      //{
      //   return _temperatureUnit;
      //}

      //HandBrakeState getHandBrakeState() const
      //{
      //   return _handBrakeState;
      //}

      //ReverseGearState getReverseGearState() const
      //{
      //   return _reverseGearState;
      //}

      SpeedLockState getSpeedLockState() const
      {
         return _speedLockState;
      }

      bool getIsDetailInfoRequested() const
      {
         return _isDetailInfoRequested;
      }

      enAddressDetailInfoType getAddressDetailInfoType() const
      {
         return _addressDetailInfoType;
      }

      enRegionType getRegionType() const
      {
         return _regionType;
      }

      const PosHeadingWGS84<float>& getDefaultPosition() const
      {
         return _defaultPosition;
      }

      const std::string& getNavigationUnitId() const
      {
         return _navigationUnitId;
      }

      const std::string& getAddressInfoDetailsName() const
      {
         return _addressInfoDetailsName;
      }

      const std::string& getAddressInfoDetailsAddress() const
      {
         return _addressInfoDetailsAddress;
      }

      const std::string& getAddressInfoDetailsPhoneNumber() const
      {
         return _addressInfoDetailsPhoneNumber;
      }

      enLightMode getLightMode() const
      {
         return _lightMode;
      }

      bool getBTActiveDeviceAvailable() const
      {
         return _isAbleToCall;
      }

      bool getBTConnectedDeviceAvailable() const
      {
         return _isBTDeviceConnected;
      }

      int getGuidanceVolume() const
      {
         return _guidanceVolume;
      }

      bool getMeterConnectionStatus() const
      {
         return _meterConnected;
      }

      int getScreenWidth() const
      {
         return _screenWidth;
      }

      int getScreenHeight() const
      {
         return _screenHeight;
      }

      int getSpeedLimit() const
      {
         return _speedLimit;
      }

      int getSpeedLimitDisplay() const
      {
         return _speedLimitDisplay;
      }

      enSpeedUnit getSpeedLimitUnit() const
      {
         return _speedLimitUnit;
      }

      navmiddleware::DateFormat getDateFormat() const
      {
         return _dateFormat;
      }

      enTimeFormat getTimeFormat() const
      {
         return _timeFormat;
      }

      bool isE2EEnabled() const
      {
         return _isE2EEnabled;
      }
      void setE2EEnabled(bool isE2EEnabled, bool forceUpdate = false);

      enVehicleType getVehicleType() const
      {
         return _vehicleType;
      }

      navmiddleware::settings::VehicleProfile getDefaultVehicleProfile() const
      {
         return _defaultVehicleProfile;
      }

      void setEstimatedTimeMode(EstimatedTimeMode estimatedTimeMode, bool forceUpdate = false);
      EstimatedTimeMode getEstimatedTimeMode() const
      {
         return _estimatedTimeMode;
      }

      void setEstimatedTimeType(EstimatedTimeType estimatedTimeType, bool forceUpdate = false);
      EstimatedTimeType getEstimatedTimeType() const
      {
         return _estimatedTimeType;
      }

      void setShowEtaOnSecMap(bool showEtaOnSecMap, bool forceUpdate = false);
      bool getShowEtaOnSecMap() const
      {
         return _showETAOnSecondaryMap;
      }

      void setWaypointIndex(unsigned int waypointIndex, bool forceUpdate = false);
      unsigned int getWaypointIndex() const
      {
         return _waypointIndex;
      }

      void setWaypointListIdx(unsigned int waypointListIdx, bool forceUpdate = false);
      unsigned int getWaypointListIdx() const
      {
         return _waypointListIdx;
      }

      void setWaypointListSize(unsigned int waypointSize);
      unsigned int getWaypointListSize() const
      {
         return _waypointListSize;
      }

      void setIsWaypointListModified(bool isWaypointListModified);
      bool getIsWaypointListModified() const
      {
         return _isWaypointListModified;
      }

      void setWaypointListRequestNeeded(bool waypointListRequestNeeded, bool forceUpdate = false);
      bool getWaypointListRequestNeeded() const
      {
         return _waypointListRequestNeeded;
      }

      void setAddressCoordinateErrorFlag(bool errorFlag)
      {
         _addressCoordinateErrorFlag = errorFlag;
      }
      bool getAddressCoordinateErrorFlag() const
      {
         return _addressCoordinateErrorFlag;
      }

      void setTmId(navmiddleware::ValidValue<int> tmId, bool forceUpdate = false);
      navmiddleware::ValidValue<int> getTmId() const
      {
         return _tmIdToBeShownInMap;
      }

      void setDestinationMemoryStatus(const DestinationMemoryStatus& destinationMemoryStatus, bool forceUpdate = false);
      DestinationMemoryStatus getDestinationMemoryStatus() const
      {
         return _destinationMemoryStatus;
      }
      void setRouteToBeUsedForE2EGuidance(const InfoStoreBase::RouteToBeUsedForE2EGuidance& routeToBeUsedForE2EGuidance);
      void setIsGuidanceActive(const bool isGuidanceActive, bool forceUpdate = false);
      void setIsHardRestrictionPresent(const bool isHardRestrictionPresent, bool forceUpdate = false);
      void setIsTourEnabled(const bool isTourActive);
      void setIsAddDestinationSelected(const bool isAddDestinationSelected);
      void setIsNavMiddlewareStarted(const bool isNavMiddlewareStarted);
      bool getIsNavMiddlewareStarted() const
      {
         return _isNavMiddlewareStarted;
      }
      bool getIsGuidanceActive() const
      {
         return _isGuidanceActive;
      }
      bool getIsHardRestrictionPresent() const
      {
         return _isHardRestrictionPresent;
      }
      bool getIsTourEnabled() const
      {
         return _isTourEnabled;
      }

      bool getIsAddDestinationSelected() const
      {
         return _isAddDestinationSelected;
      }

      void setNaviServiceRequestedData(const bool isNaviServiceRequestedData, bool forceUpdate = false);
      bool getNaviServiceRequestedData() const
      {
         return _isNaviServiceRequestedData;
      }
      void setDestinationAddress(const std::string name, bool forceUpdate = false);
      ::std::string  getDestinationAddress() const
      {
         return _destinationName;
      }
      void setIsDisclaimerAccepted(bool isDisclaimerAccepted);
      bool getIsDisclaimerAccepted() const
      {
         return _isDisclaimerAccepted;
      }

      bool getIsStateEntry() const
      {
         return _isStateEntry;
      }

      bool getRefineSpellerStatus() const
      {
         return _refineSpellerStatus;
      }

      unsigned int getVehicleProfile() const
      {
         return _vehicleProfile;
      }
      void setVehicleProfile(const unsigned int vehicleProfile, bool forceUpdate = false);

      unsigned int getProfileAndGuidanceStateLastWrittenLineNo() const
      {
         return _profileAndGuidanceStateLastWrittenLineNo;
      }
      void setProfileAndGuidanceStateLastWrittenLineNo(const unsigned int profileAndGuidanceStateLastWrittenLineNo, bool forceUpdate = false);

      void addSelectedTurnInfo(const int selectedIndex);
      void clearSelectedTurnInfo();
      const ::std::vector<int>& getSelectedTurnInfo() const
      {
         return _avoidTurnList;
      }

      void setCurrentlySelectedTurnIndex(const int selectedHmiTurnIndex, const int selectedMWTurnIndex);
      void setNearbyPoi(const navmiddleware::ValidValue< navmiddleware::RouteListInfos::RouteListElementInfo::ExitPOIInfo >& routePOIInfo);
      navmiddleware::ValidValue< navmiddleware::RouteListInfos::RouteListElementInfo::ExitPOIInfo > getNearbyPoi() const;
      const int getSelectedHmiTurnIndex() const
      {
         return _selectedHmiTurnIndex;
      }

      const int getSelectedMWTurnIndex() const
      {
         return _selectedMWTurnIndex;
      }

      const InfoStoreBase::CoordinatesToBeShownInMap& getCoordinatesToBeShownInMap() const;
      InfoStoreBase::CoordinatesToBeShownInMap& getCoordinatesToBeShownInMap();
      RouteToBeUsedForE2EGuidance& getRouteToBeUsedForE2EGuidance();
      void resetCoordinatesToBeShownInMap();

      void setTourInformation(navmiddleware::Tour& tour);
      navmiddleware::Tour getTourInformation() const
      {
         return _tour;
      }

      void setTrafficReceptionState(navmiddleware::TrafficReceptionState trafficReceptionState);
      navmiddleware::TrafficReceptionState getTrafficReceptionState() const
      {
         return _trafficReceptionState;
      }

      void setRouteCalculationStatus(bool isRouteCalculated, bool forceUpdate = false);
      bool getRouteCalculationStatus() const   // Returns TRUE if route calculation is in progress
      {
         return _routeCalculationStatus;
      }

      void setGuidanceStatefromSDS(bool isStartGuidance, bool forceUpdate = false);
      bool getGuidanceStatefromSDS() const
      {
         return _speechGuidanceStatus;
      }

      void setMultipleRouteCalculationStatus(bool isMulipleRouteCalculated, bool forceUpdate = false);
      bool getMultipleRouteCalculationStatus() const   // Returns TRUE if route calculation is in progress
      {
         return _isMulipleRouteCalculated;
      }

      void setSXMSubscriptionStatus(bool status, bool forceUpdate = false);
      bool getSXMSubscriptionStatus() const
      {
         return _sxmSubscriptionStatus;
      }

      void setGoStraightIndicationStatus(bool isGoStraightIndicationActive);
      bool getGoStraightIndicationStatus() const
      {
         return _isGoStraightIndicationActive;
      }

      void setAutoPlayCoordinates(std::vector<navmiddleware::GeoCoordinateDegree>& autoPlayCoordinates);
      std::vector<navmiddleware::GeoCoordinateDegree> getAutoPlayCoordinates() const
      {
         return _autoPlayCoordinates;
      }
      //void setWeatherDataInfo(const navmiddleware::WeatherDataInfos& weatherData)
      //{
      //   _weatherDataInfos = weatherData;
      //   notify(NAVI_PICK_FROM_MAP_WEATHER_DATA_CHANGED);
      //}
      //navmiddleware::WeatherDataInfos& getWeatherDataInfo()
      //{
      //   return _weatherDataInfos;

      void triggerMapMainScreenStateChanged();

      void setAddressBookSortByCategory(navmiddleware::DestinationMemoryEntry::SortingAlgorithm _addressBookSortByCategory, bool forceUpdate = false);
      navmiddleware::DestinationMemoryEntry::SortingAlgorithm getAddressBookSortByCategory() const;
      void setDestinationAttributesInfoName(const ::std::string& name, bool forceUpdate = false);
      void setDestinationAttributesInfoSoundId(const navmiddleware::DestinationMemoryEntryAttributes::Sound& soundId, bool forceUpdate = false);
      void setDestinationAttributesInfoVoiceTagId(const uint64_t& voiceTag, bool forceUpdate = false);
      void setDestinationAttributesInfoInvalidateVoiceTagId();
      void setDestinationAttributesInfoIconId(const uint64_t& iconId, bool forceUpdate = false);
      void setDestinationAttributesInfoDistance(const navmiddleware::DestinationMemoryEntryAttributes::Distance& distance, bool forceUpdate = false);
      void setDestinationAttributesInfoDirection(const uint16_t& direction, bool forceUpdate = false);
      void setDestinationAttributesInfoInvalidateDirection();
      void setDestinationAttributesInfoShowOnMap(const bool& showOnMap, bool forceUpdate = false);
      void setDestinationAttributesInfoPhoneNumber(const ::std::vector< ::std::string >& phoneNumberList, bool forceUpdate = false);
      void setDestinationAttributesInfoGroupList(const ::std::vector<navmiddleware::DestinationMemoryAttributesGroup>& groupList, bool forceUpdate = false);
      void setDestinationAttributesInfoInternalMapIconId(const uint64_t& internalMapIconId, bool forceUpdate = false);
      void setDestinationAttributesInfo(const navmiddleware::DestinationMemoryEntryAttributes& destinationMemoryEntryAttributes);
      void setRouteListElementIndexToBeDeleted(unsigned int routeListElementIndexToBeDeleted, bool forceUpdate = false);
      void setNumberOfAddressBookEntries(unsigned int numberOfAddressBookEntries, bool forceUpdate = false);
      void setNumberOfStoredRouteEntries(unsigned int numberOfStoredRouteEntries, bool forceUpdate = false);
      void setAddressBookSortByGroupListStartIndex(unsigned int addressBookSortByGroupIndex, bool forceUpdate = false);
      void setAddressBookSortByGroupType(navmiddleware::DestinationMemoryAttributesGroup _addressBookSortByGroupType, bool forceUpdate = false);
      unsigned int getNumberOfAddressBookEntries() const;
      unsigned int getNumberOfStoredRouteEntries() const;
      unsigned int getRouteListElementIndexToBeDeleted() const;
      void setListFocusIndex(unsigned int focusIndex);
      void setListStartIndex(unsigned int startIndex);
      unsigned int getListStartIndex() const;
      void setLongitude(double longitude);
      void setLatitude(double latitude);
      double getLongitude() const;
      double getLatitude() const;
      unsigned int getListFocusIndex() const;
      inline unsigned int getCurrentWeekDay() const
      {
         return _weekDay;
      }
      inline unsigned int getCurrentHour() const
      {
         return _clockHour;
      }
      inline unsigned int getCurrentMinute() const
      {
         return _clockMinutes;
      }
      void setLongitudeCoordinatesDirection(enLongitudeCoordinatesDirection longitudeCoordinatesDirection, bool forceUpdate = false);
      void setLatitudeCoordinatesDirection(enLatitudeCoordinatesDirection latitudeCoordinatesDirection, bool forceUpdate = false);
      void setCoordinatesDisplayType(enCoordinatesDisplayType coordinatesDisplayType, bool forceUpdate = false);

      void setStoredRouteInfo(const StoredRouteInfo& storedRouteInfo, bool forceUpdate = false);
      StoredRouteInfo getStoredRouteInfo() const
      {
         return _storedRouteInfo;
      };

      void setTcuConfig(unsigned int tcuConfig);
      unsigned int getTcuConfig()
      {
         return _tcuConfig;
      }

      void setDisplayAspectRatio(unsigned int displayAspectRatio);
      DisplayAspectRatio getDisplayAspectRatio()
      {
         return _displayAspectRatio;
      }

      const navmiddleware::DestinationMemoryEntryAttributes& getDestinationAttributesInfo() const
      {
         return _destinationAttributesInfo;
      }

      void setAvoidAreaInfo(const navmiddleware::AvoidAreaInfo& avoidAreaInfo);
      void setAvoidAreaInfoSize(navmiddleware::AvoidAreaInfo::AvoidAreaSize avoidAreaSize, bool forceUpdate = false);
      void setAvoidAreaInfoName(const ::std::string& avoidAreaName, bool forceUpdate = false);
      void setAvoidAreaInfoScale(const uint32_t scale, bool forceUpdate = false);
      void setAvoidAreaInfoEdgelength(const uint32_t edgelength, bool forceUpdate = false);
      void setAvoidAreaInfoCenterGeoCoordinate(const navmiddleware::GeoCoordinateDegree& geoCoordinateDegree, bool forceUpdate = false);
      void setAvoidAreaInfoAvoidAreaOptions(const navmiddleware::AvoidAreaInfo::AvoidAreaOptions avoidAreaOption, bool forceUpdate = false);
      void setAvoidAreaAddNewFlowStatus(bool flowStatus, bool forceUpdate = false);
      const navmiddleware::AvoidAreaInfo& getAvoidAreaInfo() const
      {
         return _avoidAreaInfo;
      }
      bool getAvoidAreaAddNewFlowStatus()
      {
         return _avoidAreaAddNewFlow;
      }
      void setAvoidAreaDeleteOptionEnableStatus(bool status, bool forceUpdate = false);
      bool getAvoidAreaDeleteOptionEnableStatus() const
      {
         return _avoidAreaDeleteOptionEnable;
      }
      navmiddleware::AvoidAreaPreviewStyle getAvoidAreaPreviewStyle() const
      {
         return _avoidAreaPreviewStyle;
      }
      void setAvoidAreaPreviewStyle();
      void setDestinationMemoryHomeStatus(const bool& status, bool forceUpdate = false);
      void setDestinationMemoryWorkStatus(const bool& status, bool forceUpdate = false);
      void setDestinationMemoryHomeId(const navmiddleware::DestinationMemoryEntry::Id& id, bool forceUpdate = false);
      void setDestinationMemoryWorkId(const navmiddleware::DestinationMemoryEntry::Id& id, bool forceUpdate = false);
      void setDestinationHomeAttribute(const navmiddleware::DestinationMemoryEntryAttributes& destinationHomeAttributes);
      void setDestinationWorkAttribute(const navmiddleware::DestinationMemoryEntryAttributes& destinationWorkAttributes);
      void setDestinationHomeCoordinates(const std::vector< PosWGS84<double> >& destinationHomeCoordinates, bool forceUpdate = false);
      void setDestinationWorkCoordinates(const std::vector< PosWGS84<double> >& destinationWorkCoordinates, bool forceUpdate = false);
      void setMoveLocationFlag(const bool& editMoveLocationFlag, bool forceUpdate = false);
      void setStateEntry(bool stateEntry);
      void setRefineSpellerStatus(bool refineSpellerStatus);
      void setCoachWeight(const float coachWeight, bool forceUpdate = false);
      void setCoachLoadPerAxle(const float coachLoadPerAxle, bool forceUpdate = false);

      float getCoachWeight() const
      {
         return _coachWeight;
      }

      float getCoachLoadPerAxle() const
      {
         return _coachLoadPerAxle;
      }
      bool getMoveLocationFlag() const
      {
         return _moveLocationFlag;
      }

      void setStoredLocationPosition(const uint64_t& storedLocationPosition, bool forceUpdate = false);
      uint64_t getStoredLocationPosition() const
      {
         return _storedLocationPosition;
      }

      const DestinationMemoryHomeWorkInfo& getDestinationMemoryHomeInfo() const
      {
         return _destinationMemoryHomeInfo;
      }

      const DestinationMemoryHomeWorkInfo& getDestinationMemoryWorkInfo() const
      {
         return _destinationMemoryWorkInfo;
      }

      unsigned int getAddressBookSortByGroupIndex() const
      {
         return _addressBookSortByGroupIndex;
      }

      navmiddleware::DestinationMemoryAttributesGroup getAddressBookSortByGroupType() const
      {
         return _addressBookSortByGroupType;
      }

      //void setMediaDeviceConnectedStatus(enMediaDeviceConnected mediaDeviceConnected, bool forceUpdate = false);
      //enMediaDeviceConnected getMediaDeviceConnectedStatus() const
      //{
      //   return _mediaDeviceConnected;
      //};

      //inline void triggerStoredOptionsListItemInfoChanged(const Courier::UInt32 optionsContextID)
      //{
      //   _storedOptionsListItemInfo.setOptionsLisItemContextIDInfo(optionsContextID);
      //   notify(NAVI_OPTIONS_LIST_ITEM_INFO_CHANGES);
      //}

      //inline void setStoredOptionsListItemInfo(const StoredOptionsListItemInfo& OptionsListItemInfo)
      //{
      //   _storedOptionsListItemInfo = OptionsListItemInfo;
      //   triggerStoredOptionsListItemInfoChanged(_storedOptionsListItemInfo.getOptionsListItemContextInfo());
      //}

      inline const StoredOptionsListItemInfo getStoredOptionsListItemInfo(void) const
      {
         return _storedOptionsListItemInfo;
      }

      inline void clearOptionsListItemInfo(void)
      {
         _storedOptionsListItemInfo.clearOptionsListItems();
      }

      //Set Function to set details of individual element.
      inline void setOptionsListItemEnableInfo(const enOptionListElement enListElement, const bool enableStatusToSet)
      {
         _storedOptionsListItemInfo.setOptionsListItemEnableInfo(enListElement, enableStatusToSet);
      }

      //Get function to get individual element details.
      inline bool getOptionsListItemEnableInfo(const enOptionListElement enListElement) const
      {
         return _storedOptionsListItemInfo.getOptionsListItemEnableInfo(enListElement);
      }

      //Get function to get Context ID information.
      inline Courier::UInt32 getOptionsListItemContextInfo(void) const
      {
         return _storedOptionsListItemInfo.getOptionsListItemContextInfo();
      }

      int32_t getLocalTimeOffset()
      {
         return _localTimeOffset;
      }

      bool isClockAutoMode()
      {
         return _isClockAutoMode;
      }

      inline void setMapScale(int mapScale)
      {
         _mapScale = mapScale;
      }

      inline int getMapScale()
      {
         return _mapScale;
      }

      void setE2EVerificationCode(const std::string& code);
      std::string getE2EVerificationCode()
      {
         return _E2EVerificationCode;
      }

      int getUserSelectedRouteAlternative() const
      {
         return _userSelectedRouteAlternative;
      }
      void setUserSelectedRouteAlternative(int userSelectedRouteAlternative);

      int getDefaultNominalMapScale() const
      {
         return _defaultNominalMapScale;
      }
      void setDefaultNominalMapScale(int32_t defaultNominalMapScale);

      bool isSecondaryMapStreamingActive();

      void setSplitScreenEnabled(bool splitScreenEnabled = false);
      bool isSplitScreenEnabled();

      void addMapOutRequest(const MapOutRegionStatus& mapOutRequestInfo);
      const MapOutRegionStatus getMapOutRequest() const;
      void removeMapOutRequest();
      bool isMapOutRequestPending() const;
      void setMapOutStatus(const REGION& regionId, const bool& mapOutStatus);
      bool getMapOutStatus(const REGION& regionId) const;

      bool isSWUpdating() const
      {
         return _isSWUpdating;
      }
      void setNavDataUpdateRunning(bool isRunning);
      bool isNavDataUpdateRunning() const
      {
         return _isNavDataUpdateRunning;
      }

      void setNavDataUpdateRecovery(bool isRecovery);
      bool isNavDataUpdateRecovery() const
      {
         return _isNavDataUpdateRecovery;
      }

      void setSpmSystemState(enSpmSystemState systemState, bool forceUpdate = false);
      enSpmSystemState getSpmSystemState() const
      {
         return _spmSystemState;
      }

      void setHmiAppState(hmibase::hmiappstates hmiAppState);
      hmibase::hmiappstates getHmiAppState() const
      {
         return _hmiAppState;
      }

      void setAddressInfoDetailsName(const ::std::string& addressInfoDetailsName);
      void setAddressInfoDetailsAddress(const ::std::string& addressInfoDetailsAddress);
      void setAddressInfoDetailsPhoneNumber(const ::std::string& addressInfoDetailsPhoneNumber);

      void setIsDetailInfoRequested(bool isDetailInfoRequested, enAddressDetailInfoType detailInfo = DETAIL_INFO_DEFAULT);

      const uint8_t& getSatelliteSource() const
      {
         return _satelliteSource;
      }
      void setSatelliteSource(uint8_t satelliteSource);

      const bool& getIsBlackListValid() const
      {
         return _isBlackListValid;
      }
      const std::vector<std::string>& getBlackListCountries() const
      {
         return _blackListCountries;
      }

      const enMapCameraAndViewModeMode& getMapCameraViewMode() const
      {
         return _mapCameraViewMode;
      }
      void setMapCameraViewMode(enMapCameraAndViewModeMode mapCameraViewMode);

   protected:
      void setTcuState(TcuState tcuState, bool forceUpdate = false);
      void setLanguageIndexAndISOCodeRelevantForNavigation(
         unsigned char languageIndexSystem, const std::string& languageISOCodeSystem,
         unsigned char languageIndexSds, const std::string& languageISOCodeSds,
         unsigned char languageIndexCluster, const std::string& languageISOCodeCluster,
         bool forceUpdate = false);
      void setTrafficTextLanguageIndexAndISOCode(
         unsigned char languageIndexTrafficText, const std::string& languageISOCodeTrafficText, bool forceUpdate = false);
      void setCountryCode(unsigned int countryCode, bool forceUpdate = false);
      void setDistanceUnitSystem(DistanceUnitSystem distanceUnitSystem, bool forceUpdate = false);
      void setSpeedLockState(SpeedLockState speedLockState, bool hmiDataServiceSpeedLockStatus, bool forceUpdate = false);
      void setRegionType(enRegionType regionType, bool forceUpdate = false);
      void setDefaultPosition(float latitude, float longitude, float heading);
      void setNavigationUnitId(const ::std::string& navigationUnitId);

      void setLightMode(enLightMode lightMode, bool forceUpdate = false);
      void setBTActiveDeviceAvailable(bool isAbleToCall, bool forceUpdate = false);
      void setBTConnectedDeviceAvailable(bool isDeviceConnected, bool forceUpdate = false);
      void setGuidanceVolume(int value, bool forceUpdate = false);
      void setMeterConnectionStatus(bool meterConnected, bool forceUpdate = false);
      void setSpeedLimitStatus(int speedLimit, int speedLimitDisplay, enSpeedUnit speedLimitUnit, bool freeSpeed, bool overSpeed, bool forceUpdate = false);
      void setDateFormat(navmiddleware::DateFormat dateFormat, bool forceUpdate = false);
      void setTimeFormat(enTimeFormat timeFormat, bool forceUpdate = false);
      void sendRegionTypeDataBinding();
      void sendDisplayOrientationStatusDataBinding(bool isPotrait = false);
      void sendSelectedRoutelistElementIndexDataBinding();
      void setLocalTimeOffsetInfo(int32_t offsetValue, bool isClockAutoMode, bool forceUpdate = false);
      void setMapDownloadLockState(bool isSWUpdating, bool forceUpdate = false);
      void setCurrentWeekDay(uint32_t weekDay, bool forceUpdate = false);
      void setCurrentHour(uint32_t clockHour, bool forceUpdate = false);
      void setCurrentMinute(uint32_t clockMinute, bool forceUpdate = false);
      void setUserProfileIndex(uint8_t userProfileIndex, bool forceUpdate = false);

      /** if it's set, InfoStore is written in one step to DataPool, single value writing will be suppressed */
      bool _writeLazyToInfoStore;

      /** This variable tells if we need to request waypoint list again from presentation layer */
      bool _waypointListRequestNeeded;
      enRegionType _regionType;
      /** Current waypoint index */
      unsigned int _waypointIndex;

      /** Current waypoint List Selected Index */
      unsigned int _waypointListIdx;

      /** Current waypoint list Size */
      unsigned int _waypointListSize;

      bool _isWaypointListModified;   // Set to true if waypoint insert/delete operation is in progress

      /** Current tcu connectivity state */
      TcuState _tcuState;

      /** Current language index and matching ISO code */
      unsigned char _languageIndexSystem;
      std::string _languageISOCodeSystem;
      unsigned int _countryCode;

      unsigned char _languageIndexSds;
      std::string _languageISOCodeSds;

      unsigned char _languageIndexCluster;
      std::string _languageISOCodeCluster;

      unsigned char _languageIndexTrafficText;
      std::string _languageISOCodeTrafficText;

      /** Current distance unit system */
      DistanceUnitSystem _distanceUnitSystem;

      /** Longitude Coordinates Direction */
      enLatitudeCoordinatesDirection _latitudeCoordinatesDirection;

      /** Longitude Coordinates Direction */
      enLongitudeCoordinatesDirection _longitudeCoordinatesDirection;

      /** Display Coordinates format */
      enCoordinatesDisplayType _coordinatesDisplayType;

      /** Current SpeedLock state */
      SpeedLockState _speedLockState;

      /** Current HMI Data Service SpeedLock status */
      bool _hmiDataServiceSpeedLockStatus;

      bool _isDetailInfoRequested;
      enAddressDetailInfoType _addressDetailInfoType;

      /** Current estimated time mode */
      EstimatedTimeMode _estimatedTimeMode;

      /** Current estimated time type mode */
      EstimatedTimeType _estimatedTimeType;
      bool _showETAOnSecondaryMap;

      PosHeadingWGS84<float> _defaultPosition;

      /*navigation unit identifier unique for every device*/
      ::std::string _navigationUnitId;

      /*Address Detailed info Name*/
      ::std::string _addressInfoDetailsName;

      /*Address Detailed info Address*/
      ::std::string _addressInfoDetailsAddress;

      /*Address Detailed info PhoneNumber*/
      ::std::string _addressInfoDetailsPhoneNumber;

      /** Current dimming light mode */
      enLightMode _lightMode;

      /** Current date format */
      navmiddleware::DateFormat _dateFormat;

      /** Current time format */
      enTimeFormat _timeFormat;

      /** BTPhone Dial Availability */
      bool _isAbleToCall;
      bool _isBTDeviceConnected;

      /** POI and FTS search order */
      navmiddleware::LocationSortOrder _sortOrder;

      /** Guidance volume */
      int _guidanceVolume;

      /** Meter connection status */
      bool _meterConnected;

      /** E2E activation state */
      bool _isE2EEnabled;

      /** Screen width in pixel */
      int _screenWidth;

      /** Screen height in pixel */
      int _screenHeight;

      /** Speed Limit from external camera */
      int _speedLimit;
      int _speedLimitDisplay;
      enSpeedUnit _speedLimitUnit;
      bool _freeSpeed;
      bool _overSpeed;

      /** Coach weight and load per axle */
      float _coachWeight;
      float _coachLoadPerAxle;

      /** Tour information */
      navmiddleware::Tour _tour;

      /** Auto play coordinates information */
      std::vector<navmiddleware::GeoCoordinateDegree> _autoPlayCoordinates;

      /** Traffic reception status */
      navmiddleware::TrafficReceptionState _trafficReceptionState;

      /** SXM subscription status */
      bool _sxmSubscriptionStatus;

      /** Current selected traffic message to be shown in map */
      navmiddleware::ValidValue<int> _tmIdToBeShownInMap;

      /** Destination Memory status for currently selected item */
      DestinationMemoryStatus _destinationMemoryStatus;

      /*Destination Attribute for currently selected item*/
      navmiddleware::DestinationMemoryEntryAttributes _destinationAttributesInfo;

      DestinationMemoryHomeWorkInfo _destinationMemoryHomeInfo;
      DestinationMemoryHomeWorkInfo _destinationMemoryWorkInfo;
      bool _moveLocationFlag;
      bool _addressCoordinateErrorFlag;

      /** Position used in "Editing Entry Number" for Addressbook */
      uint64_t _storedLocationPosition;

      /** Databinding for region type */
      //DataBindingItem<RegionTypeDataBindingSource> _regionTypeDataBinding;

      ///** Databinding for edit route */
      //DataBindingItem<EditRouteDataBindingSource> _editRouteDataBinding;

      ///** Databinding for Meter connection status */
      //DataBindingItem<FCMeterConnectionCheckDataBindingSource> _meterConnectionDataBinding;

      ///** Databinding for Display orientation update **/
      //DataBindingItem<DisplayOrientationDataBindingSource> _displayOrientation;

      ///** Databinding for Meter connection status */
      //DataBindingItem<SpeedLockPropertiesDataBindingSource> _speedLockPropertiesDataBinding;

      ///** Databinding for vehicle profile type */
      DataBindingItem<VehicleProfileTypeDataBindingSource> _vehicleProfileTypeDataBinding;

      /** State of Guidance */
      bool _isGuidanceActive;

      bool _isHardRestrictionPresent;

      bool _isNavMiddlewareStarted;

      /** State of Tour */
      bool _isTourEnabled;

      bool _isAddDestinationSelected;

      /** Navigataion service requested for MW data */
      bool _isNaviServiceRequestedData;

      /*destination name*/
      ::std::string _destinationName;

      /** Disclaimer acceptance state */
      bool _isDisclaimerAccepted;

      /** stateEntry flag */
      bool _isStateEntry;

      /** refineSpellerStatus flag */
      bool _refineSpellerStatus;

      /** Cureently Selected Turn */
      int _selectedHmiTurnIndex;

      /** List Start Index */
      int _selectedMWTurnIndex;

      /** Route Exit Point Info */
      navmiddleware::ValidValue< navmiddleware::RouteListInfos::RouteListElementInfo::ExitPOIInfo > _routeExitPoiInfo;

      /**Turnlist Info For Avoid Road*/
      ::std::vector<int> _avoidTurnList;

      /*No Of AddressBook Entries*/
      unsigned int _numberOfAddressBookEntries;

      /*No Of Stored Route Entries*/
      unsigned int _numberOfStoredRouteEntries;

      /** route list index which was selected to be deleted */
      unsigned int _routeListElementIndexToBeDeleted;

      /** List of geo coordinates to be shown in an overview map */
      CoordinatesToBeShownInMap _coordinatesToBeShownInMap;

      /* Route received from E2E client app */
      RouteToBeUsedForE2EGuidance _routeToBeUsedForE2EGuidance;

      /** AvoidAreaInfo object */
      navmiddleware::AvoidAreaInfo _avoidAreaInfo;

      /*  Avoid Area Delete button option*/
      bool _avoidAreaDeleteOptionEnable;
      bool _avoidAreaAddNewFlow;

      /*Sorting AddressBook based on user selection category*/
      navmiddleware::DestinationMemoryEntry::SortingAlgorithm _addressBookSortByCategory;
      /* Avoid Area Preview style*/
      navmiddleware::AvoidAreaPreviewStyle _avoidAreaPreviewStyle;

      /*Sorting AddressBook Based on the Group Type Selection*/
      navmiddleware::DestinationMemoryAttributesGroup _addressBookSortByGroupType;

      /*Starting index of the Address Book Sort By Group Selection*/
      unsigned int _addressBookSortByGroupIndex;

      StoredRouteInfo _storedRouteInfo;

      /*  Status of route calculation status*/
      bool _routeCalculationStatus;

      /*  Status of Speech route calculation status*/
      bool _speechGuidanceStatus;

      /*  Status of Multiple route calculation*/
      bool _isMulipleRouteCalculated;

      /*Current Media Device Connected status*/
      //enMediaDeviceConnected _mediaDeviceConnected;

      /* To understand if there's no follow-up intersection or junction*/
      bool _isGoStraightIndicationActive;

      enVehicleType _vehicleType;
      navmiddleware::settings::VehicleProfile _defaultVehicleProfile;

      enVariantSelection _variantSelectionType;

      unsigned int _vehicleTypeEV;
      unsigned int _tcuConfig;

      /* used display 15:9 or 17:9 */
      /* used display 15:9 or 17:9 */
      DisplayAspectRatio _displayAspectRatio;

      /* Offset between Final display time and system time */
      int32_t _localTimeOffset;

      /* Indicate if clock is set in Auto Mode (display time depends on vehicle's position) */
      bool _isClockAutoMode;

      /*Eco Score Drive Report status availability*/
      //bool _isEcoDriveReportAvailable;

      int _userSelectedRouteAlternative;

      int _mapScale;

      /** Address book start index for show icons the map. */
      unsigned int _listStartIndex;

      double _longitude;
      double _latitude;

      /** Address book focus index value to show icons in map*/
      unsigned int _listFocusIndex;

      int _defaultNominalMapScale;

      /** variable to hold the current active user profile. */

      /** Verification code for E2E TLS communication */
      std::string _E2EVerificationCode;

      //bool _isPivi;
      /** Display type variables -for p-ivi */
      //bool _isDualdisplay;

      /* current weekday */
      uint32_t _weekDay;

      /* current Clock Hour */
      uint32_t _clockHour;

      /* current Clock Minutes */
      uint32_t _clockMinutes;
      /** Status of MapDownloadLockState partition if it is locked by SW update */
      bool _isSWUpdating;
      /** Indicate if MDW's LifeCycle is in NavDataUpdate */
      bool _isNavDataUpdateRunning;
      enSpmSystemState _spmSystemState;
      hmibase::hmiappstates _hmiAppState;
      /** Indicate if MDW's LifeCycle is in Recovery */
      bool _isNavDataUpdateRecovery;
      bool _isSplitScreenEnabled;

      uint8_t _satelliteSource;

      bool                     _isBlackListValid;
      std::vector<std::string> _blackListCountries;

      enMapCameraAndViewModeMode _mapCameraViewMode;
      unsigned int _vehicleProfile;
      unsigned int _profileAndGuidanceStateLastWrittenLineNo;

   private:
      StoredOptionsListItemInfo _storedOptionsListItemInfo;

      ::std::queue<MapOutRegionStatus> _mapOutRequestQueue;
      MapOutRegionStatus _mapOutStatus[MAX_REG];
};


#endif // INFO_STORE_BASE_H
