/**************************************************************************************
* @file         : InfoStore.cpp
* @author       : ECG5-Atchuta Sashank Kappagantu
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) -2020 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include <fstream>
#include <boost/algorithm/string/trim.hpp>
#include "gui_std_if.h"
#include "App/datapool/NaviDataPoolConfig.h"
#include "EnvironmentUtils.h"
#include "Settings/VehicleSettingsHelper.h"
#include "InfoStore.h"
using namespace std;

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/InfoStore.cpp.trc.h"
#endif

InfoStore::InfoStore()
   : tripCount(0)
   , carsorHeadingIndex(0)
{
}


InfoStore::~InfoStore()
{
}


void InfoStore::initialize()
{
   loadPersistencyFromDataPool();
   _screenWidth = 800;
   _screenHeight = 480;
}


void InfoStore::deinitialize()
{
   ETG_TRACE_COMP(("InfoStore::deinitialize()"));

   savePersistencyToDataPool();
}


void InfoStore::loadPersistencyFromDataPool()
{
   ETG_TRACE_USR1(("InfoStore::loadPersistencyFromDataPool()"));

   class NaviDataPoolConfig* dataPoolInstance = NaviDataPoolConfig::getInstance();
   if (NULL != dataPoolInstance)
   {
      //initialize region type from datapool
      setRegionType((enRegionType)dataPoolInstance->getDpRegionType());
      ETG_TRACE_USR1(("setRegionType getRegionType() %d", (enRegionType)dataPoolInstance->getDpRegionType()));
      //setTcuConfig(dataPoolInstance->getTcuConfig());
      //sendRegionTypeDataBinding();

      //set languages for navmiddleware startup
      _languageIndexSystem        = dataPoolInstance->getDpLanguageIndexSystem();
      _languageISOCodeSystem      = dataPoolInstance->getDpLanguageISOCodeSystem();
      _languageIndexSds           = dataPoolInstance->getDpLanguageIndexSds();
      _languageISOCodeSds         = dataPoolInstance->getDpLanguageISOCodeSds();
      _languageIndexCluster       = dataPoolInstance->getDpLanguageIndexCluster();
      _languageISOCodeCluster     = dataPoolInstance->getDpLanguageISOCodeCluster();
      _languageIndexTrafficText   = dataPoolInstance->getDpLanguageIndexTrafficText();
      _languageISOCodeTrafficText = dataPoolInstance->getDpLanguageISOCodeTrafficText();
      //_countryCode                = dataPoolInstance->getDpCountryCode();

      //POST_MSG((COURIER_MESSAGE_NEW(LocalizationReqMsg)(dataPoolInstance->getDpLanguageISOCodeSystem().c_str())));

      //init Distance Unit System from Navi's datapool
      _distanceUnitSystem = static_cast<InfoStoreBase::DistanceUnitSystem>(dataPoolInstance->getDpDistanceUnitSystem());

      // lat and long Direction
      _latitudeCoordinatesDirection    = static_cast<enLatitudeCoordinatesDirection>(dataPoolInstance->getDpLatitudeDirection());
      _longitudeCoordinatesDirection   = static_cast<enLongitudeCoordinatesDirection>(dataPoolInstance->getDpLongitudeDirection());
      _coordinatesDisplayType          = static_cast<enCoordinatesDisplayType>(dataPoolInstance->getDpCoordinatiateDisplayFormat());

      //initialize startup position
      setDefaultPosition(
         NaviDataPoolConfig::getInstance()->getDpDefaultPositionLatitude(),
         NaviDataPoolConfig::getInstance()->getDpDefaultPositionLongitude(),
         NaviDataPoolConfig::getInstance()->getDpDefaultPositionHeading());

      //set device identifier
      //setNavigationUnitId(NaviDataPoolConfig::getInstance()->getDpNavigationUnitId());

      //initialize Guidance state from datapool
      bool isGuidanceActive = dataPoolInstance->getDpIsGuidanceActive();
      setIsGuidanceActive(isGuidanceActive);
      ETG_TRACE_USR1(("The guidance state is %d", isGuidanceActive));
      EXT_bIsGuidanceActive = isGuidanceActive;

      _variantSelectionType = (enVariantSelection)NaviDataPoolConfig::getInstance()->getVariantSelectionType();
      ETG_TRACE_USR4(("Vaiant Selection Type is _variantSelectionType %d", _variantSelectionType));

      //localTimeOffset info
      _localTimeOffset = NaviDataPoolConfig::getInstance()->getDpLocalTimeOffset();
      _isClockAutoMode = NaviDataPoolConfig::getInstance()->getDpClockAutoModeActivated();

      _estimatedTimeMode = static_cast<EstimatedTimeMode>(dataPoolInstance->getDpEstimatedTimeMode());
      _estimatedTimeType = static_cast<EstimatedTimeType>(dataPoolInstance->getDpEstimatedTimeType());

      setCoachWeight(dataPoolInstance->getDpCoachWeight());
      setCoachLoadPerAxle(dataPoolInstance->getDpCoachLoadPerAxle());
      _satelliteSource = NaviDataPoolConfig::getInstance()->getKDSSatelliteSource();
      bool isETASettingsEnabled = dataPoolInstance->getDpShowETAOnSecMap();
      setShowEtaOnSecMap(isETASettingsEnabled);
      updateDefaultVehicleProfile();   // Initialize default vehicle settings value from KDS
      setVehicleProfile(isGuidanceActive ? (dataPoolInstance->getDpVehicleProfile()) : COACH_PROFILE_1);
      setProfileAndGuidanceStateLastWrittenLineNo(dataPoolInstance->getDpProfileAndGuidanceStateLastWrittenLineNo());

      if ((OTH_EUR == _regionType) || (UK == _regionType))
      {
         updateBlackListCountries();      // Read black list countries from text file and update in InfoStoreBase
      }
   }
}


void InfoStore::savePersistencyToDataPool() const
{
   ETG_TRACE_USR1(("InfoStore::savePersistencyToDataPool()"));

   NaviDataPoolConfig* dpConfig = NaviDataPoolConfig::getInstance();
   if (NULL != dpConfig)
   {
      // copy navigation languages from info store to datapool to be permanently stored
      dpConfig->setDpLanguageIndexSystem(_languageIndexSystem);
      dpConfig->setDpLanguageISOCodeSystem(_languageISOCodeSystem);
      dpConfig->setDpLanguageIndexSds(_languageIndexSds);
      dpConfig->setDpLanguageISOCodeSds(_languageISOCodeSds);
      dpConfig->setDpLanguageIndexCluster(_languageIndexCluster);
      dpConfig->setDpLanguageISOCodeCluster(_languageISOCodeCluster);
      dpConfig->setDpLanguageIndexTrafficText(_languageIndexTrafficText);
      dpConfig->setDpLanguageISOCodeTrafficText(_languageISOCodeTrafficText);

      dpConfig->setDpIsGuidanceActive(_isGuidanceActive);
      _isGuidanceActive ? dpConfig->setDpVehicleProfile(_vehicleProfile) : dpConfig->setDpVehicleProfile(COACH_PROFILE_1);
      dpConfig->setDpProfileAndGuidanceStateLastWrittenLineNo(_profileAndGuidanceStateLastWrittenLineNo);
      dpConfig->setDpShowETAOnSecMap(_showETAOnSecondaryMap);
   }
}


void InfoStore::notifyDataUpdatesFromDataPool()
{
   ETG_TRACE_USR1(("InfoStore::notifyDataUpdatesFromDataPool()"));

   //update all datamodels regardless of data changes
   bool forceNotification = true;

   // initialize navigation languages from datapool
   setTrafficTextLanguageIndexAndISOCode(_languageIndexTrafficText, _languageISOCodeTrafficText); // do not forceUpdate
   // NAVI_LANGUAGE_CHANGED will be notified in below setLanguageIndexAndISOCodeRelevantForNavigation()
   setLanguageIndexAndISOCodeRelevantForNavigation(
      _languageIndexSystem, _languageISOCodeSystem,
      _languageIndexSds, _languageISOCodeSds,
      _languageIndexCluster, _languageISOCodeCluster,
      forceNotification
   );
   // NAVI_COUNTRYCODE_CHANGED will be notified in below setCountryCode()
   setCountryCode(_countryCode);

   // initialize distance unit system
   setDistanceUnitSystem(_distanceUnitSystem, forceNotification);

   setLightMode(NaviDataPoolConfig::getInstance()->getDpLightMode(), forceNotification);
   setDateFormat(static_cast<navmiddleware::DateFormat>(NaviDataPoolConfig::getInstance()->getDpDateFormat()), forceNotification);
   setTimeFormat(NaviDataPoolConfig::getInstance()->getDpTimeFormat(), forceNotification);
   setLocalTimeOffsetInfo(_localTimeOffset, _isClockAutoMode, forceNotification);
   setEstimatedTimeMode(static_cast<EstimatedTimeMode>(NaviDataPoolConfig::getInstance()->getDpEstimatedTimeMode()), forceNotification);
   setEstimatedTimeType(static_cast<EstimatedTimeType>(NaviDataPoolConfig::getInstance()->getDpEstimatedTimeType()), forceNotification);
   setSpmSystemState(_spmSystemState, forceNotification);
   setIsGuidanceActive(NaviDataPoolConfig::getInstance()->getDpIsGuidanceActive(), forceNotification);
}


void InfoStore::notifyDataUpdates()
{
   ETG_TRACE_USR1(("InfoStore::notifyDataUpdates()"));

   //update all datamodels regardless of data changes
   const bool forceNotification = true;
   setSpmSystemState(_spmSystemState, forceNotification);
}


void InfoStore::notifyDataUpdatesRecovery()
{
   ETG_TRACE_USR1(("InfoStore::notifyDataUpdatesRecovery()"));
   const bool forceNotification = true;
   setSpmSystemState(_spmSystemState, forceNotification);
}


unsigned int InfoStore::getCarsorHeadingIndex() const
{
   return carsorHeadingIndex;
}


void InfoStore::setCarsorHeadingIndex(unsigned int headingIndex)
{
   carsorHeadingIndex = headingIndex;
}


bool InfoStore::onCourierMessage(const LanguageRelevantForNavigationUpdateMsg& oMsg)
{
   ETG_TRACE_COMP(("InfoStore::onCourierMessage(LanguageRelevantForNavigationUpdateMsg())"));

   setLanguageIndexAndISOCodeRelevantForNavigation(
      oMsg.GetSystemLanguageIndex(), oMsg.GetSystemLanguageISOCode().GetCString(),
      oMsg.GetSdsLanguageIndex(), oMsg.GetSdsLanguageISOCode().GetCString(),
      oMsg.GetClusterLanguageIndex(), oMsg.GetClusterLanguageISOCode().GetCString());

   // as long as there is no source given for the TrafficTextLanguage it will be equal to the SystemLanguage
   setTrafficTextLanguageIndexAndISOCode(
      oMsg.GetSystemLanguageIndex(), oMsg.GetSystemLanguageISOCode().GetCString());

   return true;
}


bool InfoStore::onCourierMessage(const GuidanceVolumeUpdateMsg& oMsg)
{
   ETG_TRACE_USR1(("InfoStore::onCourierMessage(GuidanceVolumeUpdateMsg(%d))", oMsg.GetValue()));

   setGuidanceVolume(oMsg.GetValue());

   return true;
}


bool InfoStore::onCourierMessage(const SpeedLockStatusUpdateMsg& oMsg)
{
   ETG_TRACE_USR1(("InfoStore::onCourierMessage(SpeedLockStatusUpdateMsg(%d, %d))", oMsg.GetCurrentState(), oMsg.GetHMIDataServiceSpeedLockStatus()));
   EXT_bIsSpeedLockActive = (true == oMsg.GetHMIDataServiceSpeedLockStatus()) ? oMsg.GetCurrentState() : false;
   setSpeedLockState(oMsg.GetCurrentState() ? InfoStore::SPEEDLOCKSTATE_ON : InfoStore::SPEEDLOCKSTATE_OFF, oMsg.GetHMIDataServiceSpeedLockStatus());
   return true;
}


bool InfoStore::onCourierMessage(const DateFormatChangeReqMsg& oMsg)
{
   ETG_TRACE_USR1(("InfoStore::onCourierMessage(DateFormatChangeReqMsg(%d))", oMsg.GetDateFormat()));
   setDateFormat(static_cast<navmiddleware::DateFormat>(oMsg.GetDateFormat()));
   return true;
}


bool InfoStore::onCourierMessage(const TimeFormatChangeReqMsg& oMsg)
{
   ETG_TRACE_USR1(("InfoStore::onCourierMessage(TimeFormatChangeReqMsg(%d))", oMsg.GetTimeFormat()));
   setTimeFormat(oMsg.GetTimeFormat());
   return true;
}


bool InfoStore::onCourierMessage(const BTActiveDeviceUpdateMsg& oMsg)
{
   ETG_TRACE_USR1(("InfoStore::onCourierMessage(BTActiveDeviceUpdateMsg(%d))", oMsg.GetIsActive()));
   setBTActiveDeviceAvailable(oMsg.GetIsActive());
   return true;
}


bool InfoStore::onCourierMessage(const BTConnectedDeviceUpdateMsg& oMsg)
{
   ETG_TRACE_USR1(("InfoStore::onCourierMessage(BTConnectedDeviceUpdateMsg(%d))", oMsg.GetIsConnected()));
   setBTConnectedDeviceAvailable(oMsg.GetIsConnected());
   return true;
}


bool InfoStore::onCourierMessage(const LocalTimeOffsetInfoUpdMsg& oMsg)
{
   ETG_TRACE_USR1(("InfoStore::onCourierMessage(LocalTimeOffsetInfoUpdMsg())"));
   setLocalTimeOffsetInfo(oMsg.GetOffsetValue(), oMsg.GetIsAutoModeOn());
   return true;
}


bool InfoStore::onCourierMessage(const LocalTimeChangeReqMsg& oMsg)
{
   setCurrentWeekDay(oMsg.GetWeekDay());
   setCurrentHour(oMsg.GetHour());
   setCurrentMinute(oMsg.GetMinutes());
   return true;
}


bool InfoStore::onCourierMessage(const MapDownloadLockStateChangedUpdMsg& oMsg)
{
   ETG_TRACE_USR1(("InfoStore::onCourierMessage(MapDownloadLockStateChangedUpdMsg(%d))", oMsg.GetIsSWUpdating()));

   setMapDownloadLockState(oMsg.GetIsSWUpdating());

   return true;
}


bool InfoStore::onCourierMessage(const SpeedLimitStatusUpdateMsg& oMsg)
{
   ETG_TRACE_USR1(("InfoStore::onCourierMessage(SpeedLimitStatusUpdateMsg())"));

   setSpeedLimitStatus(oMsg.GetSpeedLimit(), oMsg.GetSpeedLimitDisplay(), oMsg.GetSpeedLimitUnit(), oMsg.GetFreeSpeed(), oMsg.GetOverSpeed());

   return true;
}


bool InfoStore::onCourierMessage(const CountryCodeUpdateMsg& oMsg)
{
   ETG_TRACE_USR4(("InfoStore::onCourierMessage(CountryCodeUpdateMsg())"));

   setCountryCode(oMsg.GetCountryCode());

   return true;
}


const bool InfoStore::isNarRegion() const
{
   bool returnValue = false;
   if (_regionType == USA || _regionType == CAN || _regionType == MEX)
   {
      returnValue = true;
   }
   return returnValue;
}


const bool InfoStore::isEurRegion() const
{
   ETG_TRACE_USR1(("isEurRegion getRegionType() %d", getRegionType()));
   const InfoStoreBase* baseptr = this;
   bool returnValue = false;
   if (_regionType == OTH_EUR)
   {
      returnValue = true;
   }

   return returnValue;
}


const bool InfoStore::isTruckType() const
{
   ETG_TRACE_USR1(("isTruckType _vehicleType %d", _vehicleType));
   bool returnValue = false;
   if (_vehicleType == TRUCK)
   {
      returnValue = true;
   }

   return returnValue;
}


const bool InfoStore::isNo3D_TypeSelected() const
{
   ETG_TRACE_USR1(("isNo3D_TypeSelected _variantSelectionType %d", _variantSelectionType));
   bool returnValue = false;
   if (_variantSelectionType == NO_3D)
   {
      returnValue = true;
   }

   return returnValue;
}


const bool InfoStore::isNonNaviVariant() const
{
   ETG_TRACE_USR1(("isNonNaviVariant _variantSelectionType %d", _variantSelectionType));
   bool returnValue = false;
   if (_variantSelectionType == DA)
   {
      returnValue = true;
   }

   return returnValue;
}


void InfoStore::updateDefaultVehicleProfile()
{
   ETG_TRACE_USR1(("InfoStore::updateDefaultVehicleProfile()"));

   class NaviDataPoolConfig* dataPoolInstance = NaviDataPoolConfig::getInstance();
   if (NULL != dataPoolInstance)
   {
      _defaultVehicleProfile.setUnitSystem(static_cast<navmiddleware::DistanceUnitSystem>(_distanceUnitSystem));

      // Vehicle related information from KDS
      _vehicleType = (enVehicleType)dataPoolInstance->getKDSVehicleType();
      ETG_TRACE_USR1(("InfoStore::updateDefaultVehicleProfile(), KDS Vehicle type is %d", _vehicleType));

      navmiddleware::settings::VehicleTypeValue enVehicleTypeValue = (TRUCK == _vehicleType) ? navmiddleware::settings::USE_VEHICLETYPE_TRUCK : navmiddleware::settings::USE_VEHICLETYPE_LIGHT_TRUCK;
      _defaultVehicleProfile.setVehicleType(enVehicleTypeValue);

      unsigned int height = dataPoolInstance->getKDSHeightOfVehicle();
      height = VehicleSettingsHelper::resetDimensionsToBound(VEHICLE_SETTINGS_HEIGHT, height, static_cast<navmiddleware::DistanceUnitSystem>(_distanceUnitSystem));
      ETG_TRACE_USR4(("InfoStore::updateDefaultVehicleProfile(), Default height : %d", height));
      _defaultVehicleProfile.setHeightOfVehicle(height);

      unsigned int length = dataPoolInstance->getKDSLengthOfVehicle();
      length = VehicleSettingsHelper::resetDimensionsToBound(VEHICLE_SETTINGS_LENGTH, length, static_cast<navmiddleware::DistanceUnitSystem>(_distanceUnitSystem));
      ETG_TRACE_USR4(("InfoStore::updateDefaultVehicleProfile(), Default length : %d", length));
      _defaultVehicleProfile.setLengthOfVehicle(length);

      unsigned int width = dataPoolInstance->getKDSWidthOfVehicle();
      width = VehicleSettingsHelper::resetDimensionsToBound(VEHICLE_SETTINGS_WIDTH, width, static_cast<navmiddleware::DistanceUnitSystem>(_distanceUnitSystem));
      ETG_TRACE_USR4(("InfoStore::updateDefaultVehicleProfile(), Default width : %d", width));
      _defaultVehicleProfile.setWidthOfVehicle(width);

      unsigned int maxWeight = dataPoolInstance->getKDSWeightOfVehicle();
      maxWeight = VehicleSettingsHelper::resetDimensionsToBound(VEHICLE_SETTINGS_TOTAL_WEIGHT, maxWeight, static_cast<navmiddleware::DistanceUnitSystem>(_distanceUnitSystem));
      ETG_TRACE_USR4(("InfoStore::updateDefaultVehicleProfile(), Default maximum weight of vehicle : %d", maxWeight));
      _defaultVehicleProfile.setMaxWeightOfVehicle(maxWeight);

      unsigned int trailerType = dataPoolInstance->getKDSTrailerType();
      trailerType = VehicleSettingsHelper::resetDimensionsToBound(VEHICLE_SETTINGS_TRAILER_TYPE, trailerType, static_cast<navmiddleware::DistanceUnitSystem>(_distanceUnitSystem));
      ETG_TRACE_USR4(("InfoStore::updateDefaultVehicleProfile(), Default trailer type : %d", trailerType));
      _defaultVehicleProfile.setTrailer(static_cast<navmiddleware::settings::TrailerValue>(trailerType));

      unsigned int maxSpeed = (DISTANCEUNITSYSTEM_METRIC == _distanceUnitSystem) ? NaviDataPoolConsts::DEFAULT_MAX_SPEED_METRICS : NaviDataPoolConsts::DEFAULT_MAX_SPEED_IMPERIAL;
      ETG_TRACE_USR4(("InfoStore::updateDefaultVehicleProfile(), Default max speed : %d", maxSpeed));
      _defaultVehicleProfile.setMaxSpeed(maxSpeed);

      unsigned int loadPerAxle = dataPoolInstance->getKDSMaximumLoadPerAxle();
      loadPerAxle = VehicleSettingsHelper::resetDimensionsToBound(VEHICLE_SETTINGS_LOAD_PER_AXLE, loadPerAxle, static_cast<navmiddleware::DistanceUnitSystem>(_distanceUnitSystem));
      ETG_TRACE_USR4(("InfoStore::updateDefaultVehicleProfile(), Default load per axle : %d", loadPerAxle));
      _defaultVehicleProfile.setMaxLoadPerAxleOfVehicle(loadPerAxle);

      ::std::vector<navmiddleware::settings::CargoType> cargoTypes;
      cargoTypes.push_back(static_cast<navmiddleware::settings::CargoType>(NaviDataPoolConsts::DEFAULT_GOODS_TYPE_OF_VEHICLE));
      _defaultVehicleProfile.setCargoTypes(cargoTypes);
   }
}


void InfoStore::updateBlackListCountries()
{
   ETG_TRACE_USR4(("InfoStore::updateBlackListCountries()"));

   const char COMMENT = '#';
   const uint8_t COUNTRY_CODE_LENGTH = 3;
   const char* blackListPath = "/opt/bosch/navigation/registry/BlackListCountriesForResidentialOrLocalMode.txt";

   ifstream reader(blackListPath);
   if (!reader)
   {
      _isBlackListValid = false;
      ETG_TRACE_ERR(("InfoStore::updateBlackListCountries(), Error in reading BlackList file. ResidentialOrLocalMode feature deactivated"));
   }
   else
   {
      _isBlackListValid = true;
      while (!reader.eof())
      {
         string blackListCountry;
         getline(reader, blackListCountry);
         boost::algorithm::trim(blackListCountry); // Remove leading/trailing spaces/carriage returns
         if ((!blackListCountry.empty()) && (COMMENT != blackListCountry.at(0)))
         {
            // Validate black list country code read from text file
            bool isCountryCodeValid = false;
            if (COUNTRY_CODE_LENGTH == blackListCountry.length())
            {
               int index = 0;
               for (; index < COUNTRY_CODE_LENGTH; index++)
               {
                  if ((('A' <= blackListCountry.at(index)) && ('Z' >= blackListCountry.at(index)))
                        || (('a' <= blackListCountry.at(index)) && ('z' >= blackListCountry.at(index))))
                  {
                     blackListCountry.at(index) = toupper(blackListCountry.at(index));
                  }
                  else
                  {
                     break;
                  }
               }
               if (index == COUNTRY_CODE_LENGTH)
               {
                  isCountryCodeValid = true;
               }
            }
            if (true == isCountryCodeValid)
            {
               // Country codes added to the list should be unique
               std::vector<std::string>::iterator iter = std::find(_blackListCountries.begin(), _blackListCountries.end(), blackListCountry);
               if (iter == _blackListCountries.end())
               {
                  _blackListCountries.push_back(blackListCountry);
                  ETG_TRACE_USR4(("InfoStore::updateBlackListCountries(), Country code : %s", blackListCountry.c_str()));
               }
            }
         }
      }
      reader.close();
   }
}


bool InfoStore::onCourierMessage(const UnitSettingsStatusUpdateMsg& oMsg)
{
   ETG_TRACE_USR1(("InfoStore::onCourierMessage(UnitSettingsStatusUpdateMsg(%d))", oMsg.GetDistanceUnit()));

   const unsigned char DISTANCE_UNIT_MILES_GENERIC = 1;
   InfoStore::DistanceUnitSystem distanceUnitSystem = InfoStore::DISTANCEUNITSYSTEM_METRIC;
   if (DISTANCE_UNIT_MILES_GENERIC == oMsg.GetDistanceUnit())
   {
      if ((UK == _regionType) || (OTH_EUR == _regionType))
      {
         distanceUnitSystem = InfoStore::DISTANCEUNITSYSTEM_BRITISH_IMPERIAL;
      }
      else
      {
         distanceUnitSystem = InfoStore::DISTANCEUNITSYSTEM_IMPERIAL;
      }
   }

   setDistanceUnitSystem(distanceUnitSystem);
   //setFuelUnit((InfoStore::FuelUnit)oMsg.GetFuelUnit());
   //setTemperatureUnit((InfoStore::TemperatureUnit)oMsg.GetTemperatureUnit());

   return true;
}


bool InfoStore::onCourierMessage(const DimmingModeStatusUpdateMsg& oMsg)
{
   ETG_TRACE_USR1(("InfoStore::onCourierMessage(DimmingModeStatusUpdateMsg(%d))", oMsg.GetLightMode()));

   setLightMode(oMsg.GetLightMode());

   return true;
}


bool InfoStore::onCourierMessage(const ResetFactoryReqMsg& oMsg)
{
   ETG_TRACE_USR1(("InfoStore::onCourierMessage(ResetFactoryReqMsg(%d))", oMsg.GetResetFactoryType()));
   if (oMsg.GetResetFactoryType() == RESET_FACTORY_DIA_SYSSET_TYPE_HMI)
   {
      notify(NAVI_RESET_FACTORY_CHANGED);
   }
   else if (oMsg.GetResetFactoryType() == RESET_FACTORY_DIA_SYSSET_TYPE_PRODUCTION)
   {
      notify(NAVI_RESET_FACTORY_PRODUCTION_CHANGED);
   }

   return true;
}


bool InfoStore::onCourierMessage(const SpmSystemStateChangedUpdMsg& oMsg)
{
   setSpmSystemState(oMsg.GetSystemState());
   return true;
}
