/**************************************************************************************
* @file         : ImageUtils.h
* @author       :
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) -2018 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#ifndef IMAGEUTILS_H
#define IMAGEUTILS_H 1

#include "HMIModelDefines.h"
#include "Candera/System/MemoryManagement/SharedPointer.h"
#include "Candera/Engine2D/Core/BitmapImage2D.h"
#include "Courier/Platform/CriticalSection.h"
#include "CgiExtensions/ImageLoader.h"

Candera::MemoryManagement::SharedPointer<Candera::BitmapImage2D> loadImage(const char* data, size_t dataSize, bool cacheImage);
void dumpImageToDisk(const char* data, size_t dataSize);

class IconNames
{
   public:
      const char* _activeBitmap;
      const char* _activeFocussedBitmap;
      const char* _normalBitmap;
      const char* _focusedBitmap;
      const char* _pressedBitmap;
};


/**
* Helper class to cache decoded images and to avoid decoding images multiple times (e.g. list with identical icons in different rows)
*/
class ImageCache
{
   public:
      /** Checks if the image with the given key and size is available in the cache. If yes, the image is being set */
      bool findImage(int key, size_t size, Candera::MemoryManagement::SharedPointer<Candera::BitmapImage2D>& image)
      {
         ScopedCriticalSection scopedCriticalSection;
         for (std::list<ImageCacheElement>::iterator it = _imageCache.begin(); it != _imageCache.end(); ++it)
         {
            if (it->_key == key && it->_size == size)
            {
               image = it->_image;
               return true;
            }
         }
         return false;
      }

      /** Adds the given image to the cache */
      void addImage(int key, size_t size, Candera::MemoryManagement::SharedPointer<Candera::BitmapImage2D> image)
      {
         ScopedCriticalSection scopedCriticalSection;
         if (_imageCache.size() > CACHE_SIZE)
         {
            _imageCache.pop_back();
         }
         _imageCache.push_front(ImageCacheElement(key, size, image));
      }

   private:
      class ImageCacheElement
      {
         public:
            ImageCacheElement(int key, size_t size, Candera::MemoryManagement::SharedPointer<Candera::BitmapImage2D> image)
               : _key(key), _size(size), _image(image) {}
            int _key;
            size_t _size;
            Candera::MemoryManagement::SharedPointer<Candera::BitmapImage2D> _image;
      };

      class ScopedCriticalSection
      {
         public:
            ScopedCriticalSection()
            {
               getCriticalSection().Obtain();
            }
            ~ScopedCriticalSection()
            {
               getCriticalSection().Release();
            }
         private:
            static Courier::Platform::CriticalSection& getCriticalSection();
            FEATSTD_MAKE_CLASS_UNCOPYABLE(ScopedCriticalSection);
      };

      std::list<ImageCacheElement> _imageCache;

      static const int CACHE_SIZE = 20;
};


#endif // IMAGEUTILS_H
