/**************************************************************************************
* @file         : HMIModelNavigationDataUpdateService.cpp
* @author       :
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) 2018-2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include <hmibase/gui_std_if.h>
#include "HMIModelNavigationDataUpdateService.h"
#include "NavigationServiceUtils.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#include "trcGenProj/Header/HMIModelNavigationDataUpdateService.cpp.trc.h"
#endif

#if defined(HALL_TO_MDW_COM)
#if defined(HALL_NAVIGATION_DATAUPDATE_SERVICE)

using namespace ::org::bosch::cm::navigation::NavigationDataUpdateService;
using namespace navmiddleware;


HMIModelNavigationDataUpdateService::HMIModelNavigationDataUpdateService(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore)
   : HMIModelBase(navMiddleware, infoStore)
   , NavigationDataUpdateServiceStub("navigationDataUpdatePort")
   , _diagPerformFullUpdateViaUsbAct(0)
   , _PerformFullUpdateViaUsbAct(0)
   , _getAvailableUpdatesAct(0)
   , _deleteUpdateComponentAct(0)
   , _setFavUpdateRegionAct(0)
   , _clearFavUpdateRegionAct(0)
   , _PerformMapExportAct(0)
{
}


HMIModelNavigationDataUpdateService::~HMIModelNavigationDataUpdateService()
{
}


void HMIModelNavigationDataUpdateService::initialize()
{
   static bool initialized = false;

   if (!initialized)
   {
      _navMiddleware.registerNavDataUpdatePropertyUpdateCallback(*this);
   }
   else
   {
      ETG_TRACE_USR1(("HMIModelNavigationDataUpdateService was initialized before!"));
   }
   _navMiddleware.registerOnNavDataContextChanges();
}


void HMIModelNavigationDataUpdateService::deinitialize()
{
   _navMiddleware.unregisterNavDataUpdatePropertyUpdateCallback(*this);
}


void HMIModelNavigationDataUpdateService::recoverNavData()
{
   ETG_TRACE_USR1(("HMIModelNavigationDataUpdateService::recoverNavData()"));
   initialize();
}


void HMIModelNavigationDataUpdateService::onGetAvailableUpdatesRequest(const ::boost::shared_ptr< GetAvailableUpdatesRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationDataUpdateService:: onGetAvailableUpdatesRequest ()"));
   uint32_t deviceId = 0;
   bool isValidDeviceId = false;
   const NavDataDeviceInfos& navDataDeviceInfos = _navMiddleware.getNavDataDeviceInfos();

   for (::std::vector<NavDataDeviceInfo>:: const_iterator iter = navDataDeviceInfos.getNavDataDeviceInfos().begin(); iter != navDataDeviceInfos.getNavDataDeviceInfos().end(); ++iter)
   {
      if ((*iter).getDeviceType() == DEVICETYPE__SOTA_HMI)
      {
         deviceId = (*iter).getDeviceId();
         isValidDeviceId = true;
         break;
      }
   }
   if (isValidDeviceId)
   {
      if (_getAvailableUpdatesAct == 0)
      {
         _getAvailableUpdatesAct = request->getAct();
         if (!_infoStore.isNavDataUpdateRunning())
         {
            _navMiddleware.requestAvailableUpdates(deviceId, " ", AVAILABLE_UPDATES_REQUEST_TYPE__INITIAL_REQUEST);
         }
         else
         {
            ETG_TRACE_USR4(("HMIModelNavigationDataUpdateService:: onGetAvailableUpdatesRequest ()::Update is Already Going On"));
         }
      }
      else
      {
         AvailableUpdates availableUpdates;
         availableUpdates.setResult(GetAvailableUpdatesResult__FAILED_INSTALLER_BUSY);
         sendGetAvailableUpdatesResponse(availableUpdates, request->getAct());
      }
   }
   else
   {
      sendGetAvailableUpdatesError(DBUS_ERROR_FAILED, "no SOTA device is available", request->getAct());
   }
}


void HMIModelNavigationDataUpdateService::onDeleteUpdateComponentRequest(const ::boost::shared_ptr< DeleteUpdateComponentRequest >& request)
{
   //ETG_TRACE_USR4(("HMIModelNavigationDataUpdateService:: onDeleteUpdateComponentRequest( productId %d, supplierId %d, RegionId %d)", request->getProductId(), request->getSupplierId(), request->getRegionId()));

   _deleteUpdateComponentAct = request->getAct();
   _navMiddleware.requestScomoDeinstallComponent(request->getProductId(), request->getSupplierId(), request->getRegionId());
}


void HMIModelNavigationDataUpdateService::onStartFullUpdateRequest(const ::boost::shared_ptr< StartFullUpdateRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationDataUpdateService:: onStartFullUpdateRequest ()"));

   _PerformFullUpdateViaUsbAct = request->getAct();
   if (_deviceInfo._isValidDeviceType)
   {
      if (_deviceInfo._isValidFullUpdate)
      {
         //Context Switch will be done by the HMI when user presses Yes after RequestAvailableUpdateFinished
         //POST_MSG((COURIER_MESSAGE_NEW(ContextSwitchOutReqMsg)(NAVIGATION_CONTEXT_MAP_UPDATE, 0, APPID_APPHMI_NAVIGATION)));
         POST_MSG((COURIER_MESSAGE_NEW(NAVDATAUPDATE_PerformFullUpdateViaUSBReqMsg)()));
      }
      else
      {
         sendStartFullUpdateResponse(StartFullUpdateResult__DECLINED__NO_FULL_UPDATE_AVAILABLE, _PerformFullUpdateViaUsbAct);
         _PerformFullUpdateViaUsbAct = 0;
      }
   }
   else
   {
      sendStartFullUpdateResponse(StartFullUpdateResult__DECLINED, _PerformFullUpdateViaUsbAct);
      _PerformFullUpdateViaUsbAct = 0;
   }
}


void HMIModelNavigationDataUpdateService::onDiagPerformFullUpdateViaUSBRequest(const ::boost::shared_ptr< DiagPerformFullUpdateViaUSBRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationDataUpdateService::onDiagPerformFullUpdateViaUSBRequest()"));
   _diagPerformFullUpdateViaUsbAct = request->getAct();
   POST_MSG((COURIER_MESSAGE_NEW(NAVDATAUPDATE_DiagPerformFullUpdateViaUSBReqMsg)()));
}


void HMIModelNavigationDataUpdateService::onPropertyNavDataUpdateStatusChanged()
{
   const NavDataUpdateStatus& navDataUpdateStatus = _navMiddleware.getUpdateStatus();
   ETG_TRACE_USR4(("HMIModelNavigationDataUpdateService::onPropertyNavDataUpdateStatusChanged(status %d, progress %d, remainingTime %u)",
                   navDataUpdateStatus.getDataUpdateStatus(), navDataUpdateStatus.getProgress(), navDataUpdateStatus.getRemainingTimeInSeconds().getValue()));
   setNavigationDataUpdateStatusInfo(
      NavigationDataUpdateStatusInfo(
         getMappedNavigationDataUpdateStatus(navDataUpdateStatus.getDataUpdateStatus()),
         navDataUpdateStatus.getProgress(),
         navDataUpdateStatus.getRemainingTimeInSeconds().getValue(),
         navDataUpdateStatus.getUpdateProductName(),
         navDataUpdateStatus.getUpdateRegionName()));

   UpdateStep updateStep(UpdateStep__UPDATE_INSTALLATION_UNKNOWN);

   switch (navDataUpdateStatus.getDataUpdateStatus())
   {
      case NavDataUpdateStatus::DATAUPDATESTATUS__IDLE:
         updateStep = UpdateStep__UPDATE_INACTIVE;
         break;
      case NavDataUpdateStatus::DATAUPDATESTATUS__INSTALLATION_FOREGROUND_STARTED:
         updateStep = UpdateStep__UPDATE_FOREGROUND_INSTALLATION_STARTED;
         break;
      case NavDataUpdateStatus::DATAUPDATESTATUS__INSTALLATION_FOREGROUND_FINISHED:
         updateStep = UpdateStep__UPDATE_FOREGROUND_INSTALLATION_STOPPED;
         break;
      case NavDataUpdateStatus::DATAUPDATESTATUS__INSTALLATION_BACKGROUND_STARTED:
         updateStep =  UpdateStep__UPDATE_BACKGROUND_INSTALLATION_STARTED;
         break;
      case NavDataUpdateStatus::DATAUPDATESTATUS__INSTALLATION_BACKGROUND_FINISHED:
         updateStep = UpdateStep__UPDATE_BACKGROUND_INSTALLATION_STOPPED;
         break;
      case NavDataUpdateStatus::DATAUPDATESTATUS__INSTALLATION_IN_PROGRESS:
         updateStep = UpdateStep__UPDATE_INSTALLATION_IN_PROGRESS;
         break;
      case NavDataUpdateStatus::DATAUPDATESTATUS__INSTALLATION_FINISHED:
         updateStep = UpdateStep__UPDATE_INSTALLATION_FINISHED;
         break;
      case NavDataUpdateStatus::DATAUPDATESTATUS__UPDATE_SOURCE_ERROR:
         updateStep = UpdateStep__UPDATE_INSTALLATION_ABORTED_BY_ERROR;
         break;
      case NavDataUpdateStatus::DATAUPDATESTATUS__INSTALLATION_ABORTED_BY_USER:
         updateStep = UpdateStep__UPDATE_INSTALLATION_ABORTED_BY_USER;
         break;
      default:
         ETG_TRACE_USR4(("No DATAUPDATESTATUS mapping available"));
         break;
   }

   if (UpdateStep__UPDATE_INSTALLATION_UNKNOWN != updateStep)
   {
      setInstallationStatus(
         InstallationStatus(
            navDataUpdateStatus.getProductId(), navDataUpdateStatus.getSupplierId(),
            navDataUpdateStatus.getRegionId(), updateStep,
            navDataUpdateStatus.getProgress(), navDataUpdateStatus.isProductUpdate(),
            navDataUpdateStatus.getUpdateProductName(), navDataUpdateStatus.getUpdateRegionName()));
   }
}


bool HMIModelNavigationDataUpdateService::onCourierMessage(const NAVDATAUPDATE_DiagFullUpdateViaUSBAckUpdMsg& oMsg)
{
   ETG_TRACE_USR4(("HMIModelNavigationDataUpdateService::onCourierMessage(NAVDATAUPDATE_DiagFullUpdateViaUSBAckUpdMsg(act %d, ack %d))", _diagPerformFullUpdateViaUsbAct, oMsg.GetAck()));
   if (0 != _diagPerformFullUpdateViaUsbAct)
   {
      if (oMsg.GetAck() == NAVDATAUPDATE__DIAG_RESULT__ACCEPTED)
      {
         sendDiagPerformFullUpdateViaUSBResponse(NavigationDataUpdateServiceAck__ACCEPTED, _diagPerformFullUpdateViaUsbAct);
      }
      else if (oMsg.GetAck() == NAVDATAUPDATE__DIAG_RESULT__DECLINED__UPDATE_IN_PROGRESS)
      {
         sendDiagPerformFullUpdateViaUSBResponse(NavigationDataUpdateServiceAck__DECLINED__REQUEST_ACTIVE, _diagPerformFullUpdateViaUsbAct);
      }
      else
      {
         sendDiagPerformFullUpdateViaUSBResponse(NavigationDataUpdateServiceAck__DECLINED__CANNOT_PROCESS, _diagPerformFullUpdateViaUsbAct);
      }
      _diagPerformFullUpdateViaUsbAct = 0;
   }
   else
   {
      ETG_TRACE_ERR(("Diag did not request for full update!"));
   }

   return true;
}


bool HMIModelNavigationDataUpdateService::onCourierMessage(const NAVDATAUPDATE_DiagFullUpdateViaUSBResultUpdMsg& oMsg)
{
   ETG_TRACE_USR4(("HMIModelNavigationDataUpdateService::onCourierMessage(NAVDATAUPDATE_DiagFullUpdateViaUSBResultUpdMsg(act %d, result %d))", _diagPerformFullUpdateViaUsbAct, oMsg.GetResult()));
   // results on failing only
   if (getMappedNavigationDataUpdateStatus(oMsg.GetResult()) != NavigationDataUpdateStatus__UNKNOWN)
   {
      setNavigationDataUpdateStatusInfo(
         NavigationDataUpdateStatusInfo(
            getMappedNavigationDataUpdateStatus(oMsg.GetResult()),
            0,
            0,
            "",
            ""));
   }

   return true;
}


void HMIModelNavigationDataUpdateService::onPropertyNavigationDataStatusChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationDataUpdateService::onPropertyNavigationDataStatusChanged(NavDataStatus %d)", _navMiddleware.getNavDataStatus()));

   switch (_navMiddleware.getNavDataStatus())
   {
      case NAVDATA_STATUS__OK:
         setNavDataStatus(NavigationDataStatus__NAVDATA_OK);
         break;
      case NAVDATA_STATUS__INCONSISTENT:
         setNavDataStatus(NavigationDataStatus__NAVDATA_INCONSISTENT);
         break;
      case NAVDATA_STATUS__UPDATE_IN_PROGRESS:
         setNavDataStatus(NavigationDataStatus__NAVDATA_DELTA_UPDATE_IN_PROGRESS);
         break;
      case NAVDATA_STATUS__FULL_UPDATE_IN_PROGRESS:
         setNavDataStatus(NavigationDataStatus__NAVDATA_FULL_UPDATE_IN_PROGRESS);
         break;
      case NAVDATA_STATUS__UNKNOWN:
         setNavDataStatus(NavigationDataStatus__NAVDATA_UNKNOWN);
         break;
      default:
         ETG_TRACE_USR4(("No NAVDATA_STATUS mapping available"));
         break;
   }
}


void HMIModelNavigationDataUpdateService::onPropertyNavDataContextChanged()
{
   const ::navmiddleware::NavDataContext& navDataContext = _navMiddleware.getNavDataContext();
   ETG_TRACE_USR4(("HMIModelNavigationDataUpdateService::onPropertyNavDataContextChanged()"));

   const ::std::vector< ::navmiddleware::NavDataContext::ProductInfo >& productInfo = navDataContext.getProducts();
   ::std::vector< Product > productList;
   ::std::string navigationSWVersion = navDataContext.getNaviVersionString();
   ::std::string packageId = navDataContext.getPackageId();

   for (::std::vector< ::navmiddleware::NavDataContext::ProductInfo >::const_iterator iter = productInfo.begin(); iter != productInfo.end(); ++iter)
   {
      Product product;

      product.setComponentId((*iter).getComponentId());
      product.setProductId((*iter).getProductId());
      product.setSupplierId((*iter).getSupplierId());
      product.setBaselineMapId((*iter).getBaselineMapId());
      product.setProductGroupId((*iter).getProductGroupId());
      product.setProductVersionId((*iter).getProductVersion().getVersionId());
      product.setProductVersionDate((*iter).getProductVersion().getVersionDate());
      product.setProductVersionIsDirty((*iter).isProductVersionIsDirty());
      product.setProductName((*iter).getProductName());
      product.setProductTypeMask((*iter).getProductTypeMask());

      const ::std::vector< ::navmiddleware::NavDataContext::RegionSetInfo >& regionSetInfo = (*iter).getRegionSets();
      ::std::vector< RegionSet > regionSetList;

      for (::std::vector< ::navmiddleware::NavDataContext::RegionSetInfo >::const_iterator regionIter = regionSetInfo.begin(); regionIter != regionSetInfo.end(); ++regionIter)
      {
         RegionSet regionSet;

         regionSet.setRegionSetId((*regionIter).getRegionSetId());
         regionSet.setRegionSetName((*regionIter).getRegionSetName());

         std::vector< uint32 > updateRegionId((*regionIter).getUpdateRegionIds().begin(), (*regionIter).getUpdateRegionIds().end());
         regionSet.setUpdateRegionIds(updateRegionId);

         regionSetList.push_back(regionSet);
      }
      product.setRegionSet(regionSetList);

      const ::std::vector< ::navmiddleware::NavDataContext::UpdateRegionInfo >& updateRegions = (*iter).getUpdateRegions();
      ::std::vector< UpdateRegion > updateRegionList;

      for (::std::vector< ::navmiddleware::NavDataContext::UpdateRegionInfo >::const_iterator updateRegionIter = updateRegions.begin(); updateRegionIter != updateRegions.end(); ++updateRegionIter)
      {
         UpdateRegion updateRegion;

         updateRegion.setComponentId((*updateRegionIter).getComponentId());
         updateRegion.setRegionId((*updateRegionIter).getRegionId());
         updateRegion.setRegionName((*updateRegionIter).getRegionName());
         updateRegion.setCurrentSizeInBytes((*updateRegionIter).getCurrentSizeInBytes());
         updateRegion.setIsRemovable((*updateRegionIter).isRemovable());
         updateRegion.setIsInstalled((*updateRegionIter).isInstalled());
         updateRegion.setCurrentVersionId((*updateRegionIter).getRegionVersion().getVersionId());
         updateRegion.setCurrentVersionDate((*updateRegionIter).getRegionVersion().getVersionDate());
         updateRegion.setIsSubscriptionActive((*updateRegionIter).isSubscriptionActive());
         updateRegion.setIsFavorite((*updateRegionIter).isFavorite());
         updateRegionList.push_back(updateRegion);
      }

      product.setUpdateRegions(updateRegionList);
      productList.push_back(product);
   }

   StorageSpace storagespace;
   storagespace.setSpaceTotalInBytes(navDataContext.getStorageSpace().getSpaceTotalInBytes());
   storagespace.setSpaceLeftInBytes(navDataContext.getStorageSpace().getSpaceLeftInBytes());

   setNavDataContext(::org::bosch::cm::navigation::NavigationDataUpdateService::NavDataContext(navigationSWVersion, packageId, productList, storagespace));
}


void HMIModelNavigationDataUpdateService::onPropertNavDataUpdateFavoriteRegionSettingsChanged()
{
   const ::navmiddleware::settings::NavDataUpdateFavoriteRegionSettings navDataUpdateFavRegionSettings = _navMiddleware.getNavDataUpdateFavoriteRegionSettings();
   ETG_TRACE_USR4(("HMIModelNavigationDataUpdateService::onPropertNavDataUpdateFavoriteRegionSettingsChanged(productId %d, supplierId %d, BaselineMapId %d)", navDataUpdateFavRegionSettings.getProductIdentifier().getProductId(), navDataUpdateFavRegionSettings.getProductIdentifier().getSupplierId(), navDataUpdateFavRegionSettings.getProductIdentifier().getBaselineMapId()));

   ::std::set<uint32_t> updateRegionIdsSet = navDataUpdateFavRegionSettings.getUpdateRegionIds();
   ::std::vector<uint32_t> updateRegionIdsVector(updateRegionIdsSet.begin(), updateRegionIdsSet.end());

   ::org::bosch::cm::navigation::NavigationDataUpdateService::FavoriteUpdateRegionStatus favoriteUpdateRegionStatus;

   switch (navDataUpdateFavRegionSettings.getFavoriteUpdateRegionStatus())
   {
      case ::navmiddleware::settings::FAVORITE_UPDATE_REGION_STATUS__UKNOWN:
         favoriteUpdateRegionStatus = FavoriteUpdateRegionStatus__UKNOWN_STATUS;
         break;
      case ::navmiddleware::settings::FAVORITE_UPDATE_REGION_STATUS__ACTIVATED_DEFAULT_SETTINGS:
         favoriteUpdateRegionStatus = FavoriteUpdateRegionStatus__ACTIVATED_DEFAULT_SETTINGS;
         break;
      case ::navmiddleware::settings::FAVORITE_UPDATE_REGION_STATUS__SERVER_UPDATED_SETTINGS:
         favoriteUpdateRegionStatus = FavoriteUpdateRegionStatus__SERVER_UPDATED_SETTINGS;
         break;
      case ::navmiddleware::settings::FAVORITE_UPDATE_REGION_STATUS__STORED_SETTINGS:
         favoriteUpdateRegionStatus = FavoriteUpdateRegionStatus__STORED_SETTINGS;
         break;
      case ::navmiddleware::settings::FAVORITE_UPDATE_REGION_STATUS__SUGGESTED_SETTINGS:
         favoriteUpdateRegionStatus = FavoriteUpdateRegionStatus__SUGGESTED_SETTINGS;
         break;
      case ::navmiddleware::settings::FAVORITE_UPDATE_REGION_STATUS__USER_UPDATED_SETTINGS:
         favoriteUpdateRegionStatus = FavoriteUpdateRegionStatus__USER_UPDATED_SETTINGS;
         break;
      default:
         favoriteUpdateRegionStatus = FavoriteUpdateRegionStatus__UKNOWN_STATUS;
         ETG_TRACE_ERR(("FavoriteUpdateRegionStatus mapping not available, so, updated to default"));
         break;
   }

   setFavoriteUpdateRegions
   (
      FavoriteUpdateRegions
      (
         favoriteUpdateRegionStatus,
         FavoriteUpdateRegionData
         (
            navDataUpdateFavRegionSettings.getProductIdentifier().getProductId(),
            navDataUpdateFavRegionSettings.getProductIdentifier().getSupplierId(),
            navDataUpdateFavRegionSettings.getProductIdentifier().getBaselineMapId(),
            updateRegionIdsVector
         )
      )
   );
}


void HMIModelNavigationDataUpdateService::onPropertyNavDataUpdateComponentDeinstallResultChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationDataUpdateService:: onPropertyNavDataUpdateComponentDeinstallResultChanged(act %d)", _deleteUpdateComponentAct));

   if (0 != _deleteUpdateComponentAct)
   {
      DeleteUpdateComponentResult deinstallResult(DeleteUpdateComponentResult__UNKNOWN);
      const DeinstallScomoComponentResult& deinstallScomoComponentResult = _navMiddleware.getScomoDeinstallComponentResult();
      switch (deinstallScomoComponentResult.getResult())
      {
         case DEINSTALL_RESULT__SUCCESS:
            deinstallResult = DeleteUpdateComponentResult__SUCCESS;
            break;
         case DEINSTALL_RESULT__DEINSTALL_PACKAGE_MISSING:
            deinstallResult = DeleteUpdateComponentResult__FAILED_DEINSTALL_MISSING;
            break;
         case DEINSTALL_RESULT__BUSY:
            deinstallResult = DeleteUpdateComponentResult__FAILED_INSTALLER_BUSY;
            break;
         case DEINSTALL_RESULT__REGION_NOT_INSTALLED:
            deinstallResult = DeleteUpdateComponentResult__FAILED_REGION_NOT_INSTALLED;
            break;
         case DEINSTALL_RESULT__FAILED:
            deinstallResult = DeleteUpdateComponentResult__FAILED;
            break;
         default:
            ETG_TRACE_USR4(("No DeinstallScomoComponentResult mapping available"));
            break;
      }
      sendDeleteUpdateComponentResponse(deinstallResult, _deleteUpdateComponentAct);
      _deleteUpdateComponentAct = 0;
   }
}


void HMIModelNavigationDataUpdateService::onPropertyNavDataUpdateFavoriteRegionSettingsResultChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationDataUpdateService:: onPropertyNavDataUpdateFavoriteRegionSettingsResultChanged"));
   if (0 != _setFavUpdateRegionAct)
   {
      ::org::bosch::cm::navigation::NavigationDataUpdateService::SetFavoriteUpdateRegionsResult favoriteUpdateRegionsResult;
      switch (_navMiddleware.getUpdateNavDataUpdateFavoriteRegionSettingsResult())
      {
         case SET_FAVORITE_UPDATE_REGION_RESULT__SUCCESS:
            favoriteUpdateRegionsResult = SetFavoriteUpdateRegionsResult__OK; //0
            break;
         case SET_FAVORITE_UPDATE_REGION_RESULT__FAILED:
         case SET_FAVORITE_UPDATE_REGION_RESULT__FAILED_NOT_ALLOWED:
         case SET_FAVORITE_UPDATE_REGION_RESULT__UNKNOWN:
            favoriteUpdateRegionsResult = SetFavoriteUpdateRegionsResult__FAILED; //2
            break;
         case SET_FAVORITE_UPDATE_REGION_RESULT__FAILED_UPDATE_REGION_NOT_AVAILABLE:
            favoriteUpdateRegionsResult = SetFavoriteUpdateRegionsResult__FAILED_UPDATE_REGION_NOT_AVAILABLE; //1
            break;
         default:
            favoriteUpdateRegionsResult = SetFavoriteUpdateRegionsResult__FAILED; //2
            ETG_TRACE_ERR(("SetFavoriteUpdateRegionsResult mapping not available, so, updated to default"));
            break;
      }
      sendSetFavoriteUpdateRegionsResponse(favoriteUpdateRegionsResult, _setFavUpdateRegionAct);
      _setFavUpdateRegionAct = 0;
   }
   if (0 != _clearFavUpdateRegionAct)
   {
      ::org::bosch::cm::navigation::NavigationDataUpdateService::ClearFavoriteUpdateRegionsResult clearFavoriteUpdateRegionsResult;
      switch (_navMiddleware.getUpdateNavDataUpdateFavoriteRegionSettingsResult())
      {
         case SET_FAVORITE_UPDATE_REGION_RESULT__UNKNOWN:
         case SET_FAVORITE_UPDATE_REGION_RESULT__SUCCESS:
            clearFavoriteUpdateRegionsResult = ClearFavoriteUpdateRegionsResult__OK; //0
            break;
         case SET_FAVORITE_UPDATE_REGION_RESULT__FAILED:
         case SET_FAVORITE_UPDATE_REGION_RESULT__FAILED_UPDATE_REGION_NOT_AVAILABLE:
            clearFavoriteUpdateRegionsResult = ClearFavoriteUpdateRegionsResult__FAILED; //2
            break;
         case SET_FAVORITE_UPDATE_REGION_RESULT__FAILED_NOT_ALLOWED:
            clearFavoriteUpdateRegionsResult = ClearFavoriteUpdateRegionsResult__FAILED_NOT_ALLOWED; //1
            break;
         default:
            clearFavoriteUpdateRegionsResult = ClearFavoriteUpdateRegionsResult__OK; //0
            ETG_TRACE_ERR(("ClearFavoriteUpdateRegionsResult mapping not available, so, updated to default"));
            break;
      }
      sendClearFavoriteUpdateRegionsResponse(clearFavoriteUpdateRegionsResult, _clearFavUpdateRegionAct);
      _clearFavUpdateRegionAct = 0;
   }
}


void HMIModelNavigationDataUpdateService::onSetFavoriteUpdateRegionsRequest(const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationDataUpdateService::SetFavoriteUpdateRegionsRequest >& request)
{
   uint32_t productId = request->getProductId();
   uint32_t supplierId = request->getSupplierId();
   uint32_t baselineMapId = request->getBaselineMapId();
   ::std::vector< uint32_t > updateRegionIdList = request->getUpdateRegionIds();

   ETG_TRACE_USR4(("HMIModelNavigationDataUpdateService:: onSetFavoriteUpdateRegionsRequest(productId %d, supplierId %d, BaselineMapId %d)", productId, supplierId, baselineMapId));

   navmiddleware::settings::NavDataUpdateFavoriteRegionSettings navDataUpdateFavoriteRegionSettings;
   const navmiddleware::settings::NavDataUpdateFavoriteRegionSettings::ProductIdentifier& productIdentifier = navmiddleware::settings::NavDataUpdateFavoriteRegionSettings::ProductIdentifier(productId, supplierId, baselineMapId);

   navDataUpdateFavoriteRegionSettings.setProductIdentifier(productIdentifier);

   for (::std::vector< uint32_t >::const_iterator itr = updateRegionIdList.begin(); itr != updateRegionIdList.end(); ++itr)
   {
      navDataUpdateFavoriteRegionSettings.getUpdateRegionIdsMutable().insert(*itr);
   }

   _setFavUpdateRegionAct = request->getAct();
   _navMiddleware.updateNavDataUpdateFavoriteRegionSettings(navDataUpdateFavoriteRegionSettings);
}


void HMIModelNavigationDataUpdateService::onClearFavoriteUpdateRegionsRequest(const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationDataUpdateService::ClearFavoriteUpdateRegionsRequest >&  request)
{
   ETG_TRACE_USR4(("HMIModelNavigationDataUpdateService:: onClearFavoriteUpdateRegionsRequest"));
   navmiddleware::settings::NavDataUpdateFavoriteRegionSettings navDataUpdateFavoriteRegionSettings;

   const navmiddleware::settings::NavDataUpdateFavoriteRegionSettings::ProductIdentifier& productIdentifier = navmiddleware::settings::NavDataUpdateFavoriteRegionSettings::ProductIdentifier(0, 0, 0);

   navDataUpdateFavoriteRegionSettings.setProductIdentifier(productIdentifier);
   navDataUpdateFavoriteRegionSettings.setFavoriteUpdateRegionStatus(::navmiddleware::settings::FAVORITE_UPDATE_REGION_STATUS__UKNOWN);
   navDataUpdateFavoriteRegionSettings.getUpdateRegionIdsMutable().clear();

   _clearFavUpdateRegionAct = request->getAct();
   _navMiddleware.updateNavDataUpdateFavoriteRegionSettings(navDataUpdateFavoriteRegionSettings);
}


void HMIModelNavigationDataUpdateService::onPropertyNavDataUpdateDeviceStatusChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationDataUpdateService:: onPropertyNavDataUpdateDeviceStatusChanged()"));
   const NavDataDeviceInfos& navDataDeviceInfos = _navMiddleware.getNavDataDeviceInfos();
   _deviceInfo._isValidDeviceType = false;
   _deviceInfo._isValidFullUpdate = false;
   ::org::bosch::cm::navigation::NavigationDataUpdateService::MapupdateExportStatus mapupdateExportStatus;
   ::std::vector< org::bosch::cm::navigation::NavigationDataUpdateService::DeviceInfo > deviceInfos;
   for (::std::vector<NavDataDeviceInfo>::const_iterator iter = navDataDeviceInfos.getNavDataDeviceInfos().begin(); iter != navDataDeviceInfos.getNavDataDeviceInfos().end(); ++iter)
   {
      org::bosch::cm::navigation::NavigationDataUpdateService::DeviceInfo deviceInfo;
      deviceInfo.setDeviceId((*iter).getDeviceId());
      deviceInfo.setDeviceType(static_cast<Devicetype>((*iter).getDeviceType()));
      deviceInfo.setDeviceStatus(static_cast<Devicestatus>((*iter).getDeviceStatus()));
      deviceInfos.push_back(deviceInfo);
      if ((*iter).getDeviceType() == DEVICETYPE__USB)
      {
         _deviceInfo._isValidDeviceType = true;
         _deviceInfo._validDeviceId = (*iter).getDeviceId();
         ETG_TRACE_USR4(("HMIModelNavigationDataUpdateService::deviceId = %d", _deviceInfo._validDeviceId));
         if ((*iter).getDeviceStatus() == DEVICESTATUS__FULL_UPDATE)
         {
            _deviceInfo._isValidFullUpdate = true;
         }
      }
   }
   setDeviceInfos(deviceInfos);
}


void HMIModelNavigationDataUpdateService::onPropertyNavDataUpdateCatalogueStatusChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationDataUpdateService:: onPropertyNavDataUpdateCatalogueStatusChanged()"));

   const ::navmiddleware::CatalogueStatus& catalogueStatusInfo = _navMiddleware.getCatalogueStatus();
   ::org::bosch::cm::navigation::NavigationDataUpdateService::CatalogueStatus catalogueStatus;

   switch (catalogueStatusInfo)
   {
      case CATALOGUE_STATUS__UNKNOWN:
         setCatalogueStatus(CatalogueStatus__UNKNOWN);
         break;
      case CATALOGUE_STATUS__AVAILABLE:
         setCatalogueStatus(CatalogueStatus__AVAILABLE);
         break;
      case CATALOGUE_STATUS__NOT_AVAILABLE:
         setCatalogueStatus(CatalogueStatus__NOT_AVAILABLE);
         break;
      case CATALOGUE_STATUS__OUTDATED:
         setCatalogueStatus(CatalogueStatus__OUTDATED);
         break;
      case CATALOGUE_STATUS__UPDATING:
         setCatalogueStatus(CatalogueStatus__UPDATING);
         break;
      default:
         ETG_TRACE_ERR(("No CATALOGUE_STATUS mapping available"));
         break;
   }
}


void HMIModelNavigationDataUpdateService::onStartMapDataExportRequest(const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationDataUpdateService::StartMapDataExportRequest >& request)
{
   uint32_t deviceId = request->getDeviceId();
   _PerformMapExportAct = request->getAct();
   ETG_TRACE_USR4(("HMIModelNavDataUpdateService::onStartMapDataExportRequest"))
   if (static_cast<Devicetype>(DEVICETYPE__USB) == request->getDeviceType())
   {
      if (deviceId == _deviceInfo._validDeviceId)
      {
         _navMiddleware.requestPrepareUSBDevice(deviceId);
         sendStartMapDataExportResponse(MapExportStartStatus__EXPORT_TRIGGER_SUCCESSFULLY, _PerformMapExportAct);
      }
      else
      {
         sendStartMapDataExportResponse(MapExportStartStatus__DEVICEID_INVALID, _PerformMapExportAct);
      }
      _PerformMapExportAct = 0;
   }
   else
   {
      ETG_TRACE_ERR(("HMIModelNavDataUpdateService::onStartMapDataExportRequest:USB device is not available!"));
      sendStartMapDataExportResponse(MapExportStartStatus__DEVICE_TYPE_INVALID, _PerformMapExportAct);
      _PerformMapExportAct = 0;
   }
}


void  HMIModelNavigationDataUpdateService::onPropertyUpdatePreparationStatusChanged()
{
   const NavDataUpdateUSBDeviceStatus deviceStatus = _navMiddleware.getPreparedUSBUpdateDeviceResult();
   ETG_TRACE_USR4(("HMIModelNavigationDataUpdateService::onPropertyUpdatePreparationStatusChanged(%d)", deviceStatus));
   ::org::bosch::cm::navigation::NavigationDataUpdateService::MapupdateExportStatus mapupdateExportStatus;
   switch (deviceStatus)
   {
      case NAVDATAUPDATE_USB_DEVICE_STATUS__SUCCESS:
         setMapupdateExportStatus(MapupdateExportStatus__USB_DEVICE_STATUS__SUCCESS);
         break;
      case NAVDATAUPDATE_USB_DEVICE_STATUS__FAILURE:
         setMapupdateExportStatus(MapupdateExportStatus__USB_DEVICE_STATUS__FAILURE);
         break;
      case NAVDATAUPDATE_USB_DEVICE_STATUS__OUT_OF_SPACE_ERROR:
         setMapupdateExportStatus(MapupdateExportStatus__USB_DEVICE_STATUS__OUT_OF_SPACE_ERROR);
         break;
      case NAVDATAUPDATE_USB_DEVICE_STATUS__REMOVED_ERROR:
         setMapupdateExportStatus(MapupdateExportStatus__USB_DEVICE_STATUS__REMOVED_ERROR);
         break;
      case NAVDATAUPDATE_USB_DEVICE_STATUS__SUBSCRIPTION_EXPIRED:
         setMapupdateExportStatus(MapupdateExportStatus__USB_DEVICE_STATUS__SUBSCRIPTION_EXPIRED);
         break;
      default:
         break;
   }
}


#endif // HALL_NAVIGATION_DATAUPDATE_SERVICE
#endif // HALL_TO_MDW_COM
